/*
 *   IRC - Internet Relay Chat, common/dbuf.c
 *   Copyright (C) 1990 Markku Savela
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __GNUG__
#pragma implementation
#endif
#include "sys.h"
#include <stdio.h>
#include "h.h"
#include "dbuf.h"

RCSTAG_CC("$Id$")

#if !defined(VALLOC) && !defined(valloc)
#define	valloc malloc
#endif

/*
 *  dbuf is a collection of functions which can be used to
 *  maintain a dynamic buffering of a byte stream.
 *  Functions allocate and release memory dynamically as
 *  required [Actually, there is nothing that prevents
 *  this package maintaining the buffer on disk, either]
 */

int dbufalloc = 0, dbufblocks = 0;
static dbufbuf *freelist = NULL;

/* This is a dangerous define because a broken compiler will set DBUFSIZ
 *  to 4, which will work but will be very inefficient. However, there
 *  are other places where the code breaks badly if this is screwed
 *  up, so... -- Wumpus
 */

#define DBUFSIZ sizeof(((dbufbuf *)0)->data)

/*
 *  dbuf_alloc - allocates a dbufbuf structure either from freelist or
 *  creates a new one.
 */
static dbufbuf *dbuf_alloc(void)
{
#if defined(VALLOC) && !defined(DEBUGMODE)
  Reg1 dbufbuf *dbptr, *db2ptr;
  Reg2 int num;
#else
  Reg1 dbufbuf *dbptr;
#endif

  dbufalloc++;
  if ((dbptr = freelist))
  {
    freelist = freelist->next;
    return dbptr;
  }
  if (dbufalloc * DBUFSIZ > BUFFERPOOL)
  {
    dbufalloc--;
    return NULL;
  }

#if defined(VALLOC) && !defined(DEBUGMODE)
#if defined(SOL2) || defined(_SC_PAGESIZE)
  num = sysconf(_SC_PAGESIZE) / sizeof(dbufbuf);
#else
  num = getpagesize() / sizeof(dbufbuf);
#endif
  if (num < 0)
    num = 1;

  dbufblocks += num;

  dbptr = (dbufbuf *) valloc(num * sizeof(dbufbuf));
  if (!dbptr)
    return (dbufbuf *) NULL;

  num--;
  for (db2ptr = dbptr; num; num--)
  {
    db2ptr = (dbufbuf *) ((char *)db2ptr + sizeof(dbufbuf));
    db2ptr->next = freelist;
    freelist = db2ptr;
  }
  return dbptr;
#else
  dbufblocks++;
  return (dbufbuf *) RunMalloc(sizeof(dbufbuf));
#endif
}

/*
 * dbuf_free - return a dbufbuf structure to the freelist
 */
static void dbuf_free(dbufbuf * ptr)
{
  dbufalloc--;
  ptr->next = freelist;
  freelist = ptr;
}

/*
 * This is called when malloc fails. Scrap the whole content
 * of dynamic buffer and return -1. (malloc errors are FATAL,
 * there is no reason to continue this buffer...). After this
 * the "dbuf" has consistent EMPTY status... ;)
 */
static int dbuf_malloc_error(dbuf * dyn)
{
  dbufbuf *p;

  dyn->length = 0;
  dyn->offset = 0;
  while ((p = dyn->head) != NULL)
  {
    dyn->head = p->next;
    dbuf_free(p);
  }
  dyn->tail = dyn->head;
  return -1;
}

/*
 *  dbuf_put
 *    Append the number of bytes to the buffer, allocating more
 *    memory as needed. Bytes are copied into internal buffers
 *    from users buffer.
 * 
 *    returns > 0, if operation successfull
 *            < 0, if failed (due memory allocation problem)
 *
 * dyn:		Dynamic buffer header
 * buf:		Pointer to data to be stored
 * length:	Number of bytes to store
 */
int dbuf_put(dbuf * dyn, char *buf, int length)
{
  Reg1 dbufbuf **h, *d;
  Reg2 int off;
  Reg3 int chunk;

  off = (dyn->offset + dyn->length) % DBUFSIZ;
  /*
   *  Locate the last non-empty buffer. If the last buffer is
   *  full, the loop will terminate with 'd==NULL'. This loop
   *  assumes that the 'dyn->length' field is correctly
   *  maintained, as it should--no other check really needed.
   */
  if (!dyn->length)
    h = &(dyn->head);
  else
  {
    if (off)
      h = &(dyn->tail);
    else
      h = &(dyn->tail->next);
  }
  /*
   *  Append users data to buffer, allocating buffers as needed
   */
  chunk = DBUFSIZ - off;
  dyn->length += length;
  for (; length > 0; h = &(d->next))
  {
    if ((d = *h) == NULL)
    {
      if ((d = (dbufbuf *) dbuf_alloc()) == NULL)
	return dbuf_malloc_error(dyn);
      dyn->tail = d;
      *h = d;
      d->next = NULL;
    }
    if (chunk > length)
      chunk = length;
    memcpy(d->data + off, buf, chunk);
    length -= chunk;
    buf += chunk;
    off = 0;
    chunk = DBUFSIZ;
  }
  return 1;
}

/*
 *  dbuf_map, dbuf_delete
 *    These functions are meant to be used in pairs and offer
 *    a more efficient way of emptying the buffer than the
 *    normal 'dbuf_get' would allow--less copying needed.
 * 
 *    map     returns a pointer to a largest contiguous section
 *            of bytes in front of the buffer, the length of the
 *            section is placed into the indicated "long int"
 *            variable. Returns NULL *and* zero length, if the
 *            buffer is empty.
 * 
 *    delete  removes the specified number of bytes from the
 *            front of the buffer releasing any memory used for them.
 * 
 *    Example use (ignoring empty condition here ;)
 * 
 *            buf = dbuf_map(&dyn, &count);
 *            <process N bytes (N <= count) of data pointed by 'buf'>
 *            dbuf_delete(&dyn, N);
 * 
 *    Note:   delete can be used alone, there is no real binding
 *            between map and delete functions...
 *
 * dyn:		Dynamic buffer header
 * length:	Return number of bytes accessible
 */
char *dbuf_map(dbuf * dyn, int *length)
{
  if (dyn->head == NULL)
  {
    dyn->tail = NULL;
    *length = 0;
    return NULL;
  }
  *length = DBUFSIZ - dyn->offset;
  if (*length > dyn->length)
    *length = dyn->length;
  return (dyn->head->data + dyn->offset);
}

/*
 * dyn:		Dynamic buffer header
 * length:	Number of bytes to delete
 */
int dbuf_delete(dbuf * dyn, int length)
{
  dbufbuf *d;
  int chunk;

  if (length > dyn->length)
    length = dyn->length;
  chunk = DBUFSIZ - dyn->offset;
  while (length > 0)
  {
    if (chunk > length)
      chunk = length;
    length -= chunk;
    dyn->offset += chunk;
    dyn->length -= chunk;
    if (dyn->offset == DBUFSIZ || dyn->length == 0)
    {
      d = dyn->head;
      dyn->head = d->next;
      dyn->offset = 0;
      dbuf_free(d);
    }
    chunk = DBUFSIZ;
  }
  if (dyn->head == (dbufbuf *) NULL)
  {
    dyn->length = 0;
    dyn->tail = 0;
  }
  return 0;
}

/*
 *  dbuf_get
 *    Remove number of bytes from the buffer, releasing dynamic
 *    memory, if applicaple. Bytes are copied from internal buffers
 *    to users buffer.
 * 
 *    returns the number of bytes actually copied to users buffer,
 *            if >= 0, any value less than the size of the users
 *            buffer indicates the dbuf became empty by this operation.
 * 
 *            Return 0 indicates that buffer was already empty.
 * 
 *            Negative return values indicate some unspecified
 *            error condition, rather fatal...
 *
 * dyn:		Dynamic buffer header
 * buf:		Pointer to buffer to receive the data
 * length:	Max amount of bytes that can be received
 */
int dbuf_get(dbuf * dyn, char *buf, int length)
{
  int moved = 0;
  int chunk;
  char *b;

  while (length > 0 && (b = dbuf_map(dyn, &chunk)) != NULL)
  {
    if (chunk > length)
      chunk = length;
    memcpy(buf, b, chunk);
    dbuf_delete(dyn, chunk);
    buf += chunk;
    length -= chunk;
    moved += chunk;
  }
  return moved;
}

/*
 *  dbuf_getmsg
 * 
 *  Check the buffers to see if there is a string which is terminted with
 *  either a \r or \n prsent.  If so, copy as much as possible (determined by
 *  length) into buf and return the amount copied - else return 0.
 */
int dbuf_getmsg(dbuf * dyn, char *buf, int length)
{
  dbufbuf *d;
  register char *s;
  register int dlen;
  register int i;
  int copy;

getmsg_init:
  d = dyn->head;
  dlen = dyn->length;
  i = DBUFSIZ - dyn->offset;
  if (i <= 0)
    return -1;
  copy = 0;
  if (d && dlen)
    s = dyn->offset + d->data;
  else
    return 0;

  if (i > dlen)
    i = dlen;
  while (length > 0 && dlen > 0)
  {
    dlen--;
    if (*s == '\n' || *s == '\r')
    {
      copy = dyn->length - dlen;
      /*
       *  Shortcut this case here to save time elsewhere.
       *  -avalon
       */
      if (copy == 1)
      {
	dbuf_delete(dyn, 1);
	goto getmsg_init;
      }
      break;
    }
    length--;
    if (!--i)
    {
      if ((d = d->next))
      {
	s = d->data;
	i = MIN(DBUFSIZ, dlen);
      }
    }
    else
      s++;
  }

  if (copy <= 0)
    return 0;

  /*
   *  copy as much of the message as wanted into parse buffer
   */
  i = dbuf_get(dyn, buf, MIN(copy, length));
  /*
   *  and delete the rest of it!
   */
  if (copy - i > 0)
    dbuf_delete(dyn, copy - i);
  if (i >= 0)
    *(buf + i) = '\0';		/* mark end of messsage */

  return i;
}
