/*
 *   IRC - Internet Relay Chat, common/parse.c
 *   Copyright (C) 1990 Jarkko Oikarinen and
 *                      University of Oulu, Computing Center
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __GNUG__
#pragma implementation
#endif
#include "sys.h"
#include "h.h"
#include "struct.h"
#include "s_serv.h"
#include "send.h"
#include "parse.h"
#include "match.h"
#include "s_bsd.h"
#include "msg.h"
#include "s_user.h"
#include "s_serv.h"
#include "channel.h"
#include "whowas.h"
#include "s_ping.h"
#include "s_conf.h"
#include "res.h"
#include "map.h"
#include "hash.h"
#include "numeric.h"
#include "ircd.h"
#include "s_misc.h"
#include "common.h"
#include "s_numeric.h"

RCSTAG_CC("$Id$")

aMessage msgtab[] = {
    {MSG_PRIVATE, TOK_PRIVATE, m_private, 0, MAXPARA, 1, 0L},
    {MSG_NICK, TOK_NICK, m_nick, 0, MAXPARA, 1, 0L},
    {MSG_NOTICE, TOK_NOTICE, m_notice, 0, MAXPARA, 1, 0L},
    {MSG_JOIN, TOK_JOIN, m_join, 0, MAXPARA, 1, 0L},
    {MSG_MODE, TOK_MODE, m_mode, 0, MAXPARA, 1, 0L},
    {MSG_BURST, TOK_BURST, m_burst, 0, MAXPARA, 1, 0L},
    {MSG_CREATE, TOK_CREATE, m_create, 0, MAXPARA, 1, 0L},
    {MSG_QUIT, TOK_QUIT, m_quit, 0, MAXPARA, 1, 0L},
    {MSG_PART, TOK_PART, m_part, 0, MAXPARA, 1, 0L},
    {MSG_TOPIC, TOK_TOPIC, m_topic, 0, MAXPARA, 1, 0L},
    {MSG_INVITE, TOK_INVITE, m_invite, 0, MAXPARA, 1, 0L},
    {MSG_KICK, TOK_KICK, m_kick, 0, MAXPARA, 1, 0L},
    {MSG_WALLOPS, TOK_WALLOPS, m_wallops, 0, MAXPARA, 1, 0L},
    {MSG_PING, TOK_PING, m_ping, 0, MAXPARA, 1, 0L},
    {MSG_PONG, TOK_PONG, m_pong, 0, MAXPARA, 1, 0L},
    {MSG_ERROR, TOK_ERROR, m_error, 0, MAXPARA, 1, 0L},
    {MSG_KILL, TOK_KILL, m_kill, 0, MAXPARA, 1, 0L},
    {MSG_USER, TOK_USER, m_user, 0, MAXPARA, 1, 0L},
    {MSG_AWAY, TOK_AWAY, m_away, 0, MAXPARA, 1, 0L},
    {MSG_ISON, TOK_ISON, m_ison, 0, 1, 1, 0L},
    {MSG_SERVER, TOK_SERVER, m_server, 0, MAXPARA, 1, 0L},
    {MSG_SQUIT, TOK_SQUIT, m_squit, 0, MAXPARA, 1, 0L},
    {MSG_WHOIS, TOK_WHOIS, m_whois, 0, MAXPARA, 1, 0L},
    {MSG_WHO, TOK_WHO, m_who, 0, MAXPARA, 1, 0L},
    {MSG_WHOWAS, TOK_WHOWAS, m_whowas, 0, MAXPARA, 1, 0L},
    {MSG_LIST, TOK_LIST, m_list, 0, MAXPARA, 1, 0L},
    {MSG_NAMES, TOK_NAMES, m_names, 0, MAXPARA, 1, 0L},
    {MSG_USERHOST, TOK_USERHOST, m_userhost, 0, 1, 1, 0L},
    {MSG_TRACE, TOK_TRACE, m_trace, 0, MAXPARA, 1, 0L},
    {MSG_PASS, TOK_PASS, m_pass, 0, MAXPARA, 1, 0L},
    {MSG_LUSERS, TOK_LUSERS, m_lusers, 0, MAXPARA, 1, 0L},
    {MSG_TIME, TOK_TIME, m_time, 0, MAXPARA, 1, 0L},
    {MSG_SETTIME, TOK_SETTIME, m_settime, 0, MAXPARA, 1, 0L},
    {MSG_RPING, TOK_RPING, m_rping, 0, MAXPARA, 1, 0L},
    {MSG_RPONG, TOK_RPONG, m_rpong, 0, MAXPARA, 1, 0L},
    {MSG_OPER, TOK_OPER, m_oper, 0, MAXPARA, 1, 0L},
    {MSG_CONNECT, TOK_CONNECT, m_connect, 0, MAXPARA, 1, 0L},
    {MSG_UPING, TOK_UPING, m_uping, 0, MAXPARA, 1, 0L},
    {MSG_MAP, TOK_MAP, m_map, 0, MAXPARA, 1, 0L},
    {MSG_VERSION, TOK_VERSION, m_version, 0, MAXPARA, 1, 0L},
    {MSG_STATS, TOK_STATS, m_stats, 0, MAXPARA, 1, 0L},
    {MSG_LINKS, TOK_LINKS, m_links, 0, MAXPARA, 1, 0L},
    {MSG_ADMIN, TOK_ADMIN, m_admin, 0, MAXPARA, 1, 0L},
    {MSG_HELP, TOK_HELP, m_help, 0, MAXPARA, 1, 0L},
    {MSG_INFO, TOK_INFO, m_info, 0, MAXPARA, 1, 0L},
    {MSG_MOTD, TOK_MOTD, m_motd, 0, MAXPARA, 1, 0L},
    {MSG_CLOSE, TOK_CLOSE, m_close, 0, MAXPARA, 1, 0L},
    {MSG_SILENCE, TOK_SILENCE, m_silence, 0, MAXPARA, 1, 0L},
    {MSG_GLINE, TOK_GLINE, m_gline, 0, MAXPARA, 1, 0L},
    {MSG_END_OF_BURST, TOK_END_OF_BURST, m_end_of_burst, 0, MAXPARA, 1, 0L},
    {MSG_END_OF_BOUNCE, TOK_END_OF_BOUNCE, m_end_of_bounce, 0, MAXPARA, 1, 0L},
    {MSG_NOTE, TOK_NOTE, m_note, 0, 1, 1, 0L},
    {MSG_HASH, TOK_HASH, m_hash, 0, MAXPARA, 1, 0L},
    {MSG_DNS, TOK_DNS, m_dns, 0, MAXPARA, 1, 0L},
#if defined(OPER_REHASH) || defined(LOCOP_REHASH)
    {MSG_REHASH, TOK_REHASH, m_rehash, 0, MAXPARA, 1, 0L},
#endif
#if defined(OPER_RESTART) || defined(LOCOP_RESTART)
    {MSG_RESTART, TOK_RESTART, m_restart, 0, MAXPARA, 1, 0L},
#endif
#if defined(OPER_DIE) || defined(LOCOP_DIE)
    {MSG_DIE, TOK_DIE, m_die, 0, MAXPARA, 1, 0L},
#endif
    {(char *)0, (char *)0, (int (*)())0, 0, 0, 0, 0L}
};

/*
 * NOTE: parse() should not be called recursively by other functions!
 */
static char *para[MAXPARA + 1];
static char sender[HOSTLEN + 1];

/*
 *   Find a client (server or user) by name.
 * 
 *   *Note*
 *    Semantics of this function has been changed from
 *    the old. 'name' is now assumed to be a null terminated
 *    string and the search is the for server and user.
 */
aClient *find_client(char *name, aClient * cptr)
{
  if (name)
    cptr = hash_find_client(name, cptr);

  return cptr;
}

aClient *find_nickserv(char *name, aClient * cptr)
{
  if (name)
    cptr = hash_find_nickserver(name, cptr);

  return cptr;
}

/*
 *   Find a user@host (server or user).
 * 
 *   *Note*
 *    Semantics of this function has been changed from
 *    the old. 'name' is now assumed to be a null terminated
 *    string and the search is the for server and user.
 */
aClient *find_userhost(char *user, char *host, aClient * cptr, int *count)
{
  Reg1 aClient *c2ptr;
  Reg2 aClient *res = cptr;

  *count = 0;
  if (collapse(user))
    for (c2ptr = client; c2ptr; c2ptr = c2ptr->next)
    {
      if (!MyClient(c2ptr))	/* implies mine and a user */
	continue;
      if ((!host || !match(host, c2ptr->user->host)) &&
	  mycmp(user, c2ptr->user->username) == 0)
      {
	(*count)++;
	res = c2ptr;
      }
    }
  return res;
}

/*
 *   Find server by name.
 * 
 *    This implementation assumes that server and user names
 *    are unique, no user can have a server name and vice versa.
 *    One should maintain separate lists for users and servers,
 *    if this restriction is removed.
 * 
 *   *Note*
 *    Semantics of this function has been changed from
 *    the old. 'name' is now assumed to be a null terminated
 *    string.
 */
aClient *find_server(char *name, aClient * cptr)
{
  if (name)
    cptr = hash_find_server(name, cptr);
  return cptr;
}

aClient *find_match_server(char *mask)
{
  aClient *acptr;

  if (BadPtr(mask))
    return NULL;
  for (acptr = client, collapse(mask); acptr; acptr = acptr->next)
  {
    if (!IsServer(acptr) && !IsMe(acptr))
      continue;
    if (!match(mask, acptr->name))
      break;
    continue;
  }
  return acptr;
}

aClient *find_name(char *name, aClient * cptr)
{
  Reg1 aClient *c2ptr = cptr;

  if (!collapse(name))
    return c2ptr;

  if ((c2ptr = hash_find_server(name, cptr)))
    return (c2ptr);
  if (!strchr(name, '*'))
    return c2ptr;
  for (c2ptr = client; c2ptr; c2ptr = c2ptr->next)
  {
    if (!IsServer(c2ptr) && !IsMe(c2ptr))
      continue;
    if (match(name, c2ptr->name) == 0)
      break;
    if (strchr(c2ptr->name, '*'))
      if (match(c2ptr->name, name) == 0)
	break;
  }
  return (c2ptr ? c2ptr : cptr);
}

/*
 *   Find person by (nick)name.
 */
aClient *find_person(char *name, aClient * cptr)
{
  Reg1 aClient *c2ptr = cptr;

  c2ptr = find_client(name, c2ptr);

  if (c2ptr && IsClient(c2ptr) && c2ptr->user)
    return c2ptr;
  else
    return cptr;
}

/*
 * parse a buffer.
 *
 * NOTE: parse() should not be called recusively by any other fucntions!
 */
int parse(aClient * cptr, char *buffer, char *bufend, aMessage *mptr)
{
  Reg1 aClient *from = cptr;
  Reg2 char *ch, *s;
  Reg3 int len, i, numeric = 0, paramcount, noprefix = 0;

  Debug((DEBUG_DEBUG, "Parsing: %s", buffer));
  if (IsDead(cptr))
    return 0;

  s = sender;
  *s = '\0';
  for (ch = buffer; *ch == ' '; ch++)
    ;
  para[0] = from->name;
  if (*ch == ':')
  {
    /*
     *  Copy the prefix to 'sender' assuming it terminates
     *  with SPACE (or NULL, which is an error, though).
     */
    for (++ch, i = 0; *ch && *ch != ' '; ++ch)
      if (s < (sender + sizeof(sender) - 1))
	*s++ = *ch;		/* leave room for NULL */
    *s = '\0';

    /*
     *  Actually, only messages coming from servers can have
     *  the prefix--prefix silently ignored, if coming from
     *  a user client...
     * 
     *  ...sigh, the current release "v2.2PL1" generates also
     *  null prefixes, at least to NOTIFY messages (e.g. it
     *  puts "sptr->nickname" as prefix from server structures
     *  where it's null--the following will handle this case
     *  as "no prefix" at all --msa  (": NOTICE nick ...")
     */
    if (*sender && IsServer(cptr))
    {
      from = find_client(sender, (aClient *) NULL);
      if (!from || match(from->name, sender))
	from = find_server(sender, (aClient *) NULL);
      else if (!from && strchr(sender, '@'))
	from = find_nickserv(sender, (aClient *) NULL);

      para[0] = sender;

      /* If the client corresponding to the
       * prefix is not found. We must ignore it,
       * it is simply a lagged message travelling
       * upstream a SQUIT that removed the client
       * --Run
       */
      if (!from)
      {
	Debug((DEBUG_NOTICE,
		"Unknown prefix (%s)(%s) from (%s)",
		sender, buffer, cptr->name));
	ircstp->is_unpf++;
	while (*ch == ' ')
	  ch++;
	/*
	 *  However, the only thing that MUST be
	 *  allowed to travel upstream against an
	 *  squit, is an SQUIT itself (the timestamp
	 *  protects us from being used wrong)
	 */
	if (strncmp(ch, "SQUIT ", 6) == 0)
	{
	  strcpy(sender, cptr->name);
	  from = cptr;
	}
	else
	  return 0;
      }
      else if (from->from != cptr)
      {
	ircstp->is_wrdi++;
	Debug((DEBUG_NOTICE,
		"Fake direction: Message (%s) coming from (%s)",
		buffer, cptr->name));
	return 0;
      }
    }
    while (*ch == ' ')
      ch++;
  }
  else
    noprefix = 1;
  if (*ch == '\0')
  {
    ircstp->is_empt++;
    Debug((DEBUG_NOTICE, "Empty message from host %s:%s",
	    cptr->name, from->name));
    return (-1);
  }
  /*
   *  Extract the command code from the packet.  Point s to the end
   *  of the command code and calculate the length using pointer
   *  arithmetic.  Note: only need length for numerics and *all*
   *  numerics must have paramters and thus a space after the command
   *  code. -avalon
   */
  s = strchr(ch, ' ');		/* s -> End of the command code */
  len = (s) ? (s - ch) : 0;
  if (len == 3 && isdigit(*ch) && isdigit(*(ch + 1)) && isdigit(*(ch + 2)))
  {
    numeric = (*ch - '0') * 100 + (*(ch + 1) - '0') * 10 + (*(ch + 2) - '0');
    paramcount = MAXPARA;
    ircstp->is_num++;
  }
  else
  {
    if (s)
      *s++ = '\0';

    if (IsServer(cptr))
    {
      /* Version      Receive         Send
       * 2.9                Long            Long
       * 2.10.0       Tkn/Long        Long
       * 2.10.10      Tkn/Long        Tkn
       * 2.10.20      Tkn             Tkn
       *
       * Clients/unreg servers always receive/
       * send long commands   -record
       */

#if 0				/* for 2.10.20 */
      /* This is a server. Check the token command
       * list only. -record!jegelhof@cloud9.net
       */

      for (; mptr->cmd; mptr++)
	if (mycmp(mptr->tok, ch) == 0)
	  break;
#else /* for 2.10.0/2.10.10 */
      /* This code supports 2.9 sending long commands,
       * 2.10.0 servers sending long commands, and
       * 2.10.10 servers sending tokenized commands.
       * It makes more calls to mycmp() than the above
       * so it will be somewhat slower.
       */

      for (; mptr->cmd; mptr++)
	if ((mycmp(mptr->tok, ch) == 0) ||
	    (mycmp(mptr->cmd, ch) == 0))
	  break;
#endif
    }
    else
      /* This is a client/unregistered entity.
       * Check long command list only.
       */
      for (; mptr->cmd; mptr++)
	if (mycmp(mptr->cmd, ch) == 0)
	  break;

    if (!mptr->cmd)
    {
      /*
       *  Note: Give error message *only* to recognized
       *  persons. It's a nightmare situation to have
       *  two programs sending "Unknown command"'s or
       *  equivalent to each other at full blast....
       *  If it has got to person state, it at least
       *  seems to be well behaving. Perhaps this message
       *  should never be generated, though...  --msa
       *  Hm, when is the buffer empty -- if a command
       *  code has been found ?? -Armin
       */
      if (buffer[0] != '\0')
      {
	if (IsPerson(from))
	  sendto_one(from,
	      ":%s %d %s %s :Unknown command",
	      me.name, ERR_UNKNOWNCOMMAND,
	      from->name, ch);
	Debug((DEBUG_ERROR, "Unknown (%s) from %s",
		ch, get_client_name(cptr, TRUE)));
      }
      ircstp->is_unco++;
      return (-1);
    }
    paramcount = mptr->parameters;
    i = bufend - ((s) ? s : ch);
    mptr->bytes += i;
    if ((mptr->flags & 1) && !(IsServer(cptr) || IsService(cptr)))
      cptr->since += (2 + i / 120);
    /* Allow only 1 msg per 2 seconds
     * (on average) to prevent dumping.
     * to keep the response rate up,
     * bursts of up to 5 msgs are allowed
     * -SRB
     */
  }
  /*
   *  Must the following loop really be so devious? On
   *  surface it splits the message to parameters from
   *  blank spaces. But, if paramcount has been reached,
   *  the rest of the message goes into this last parameter
   *  (about same effect as ":" has...) --msa
   */

  /* Note initially true: s==NULL || *(s-1) == '\0' !! */

  i = 0;
  if (s)
  {
    if (paramcount > MAXPARA)
      paramcount = MAXPARA;
    for (;;)
    {
      /*
       *  Never "FRANCE " again!! ;-) Clean
       *  out *all* blanks.. --msa
       */
      while (*s == ' ')
	*s++ = '\0';

      if (*s == '\0')
	break;
      if (*s == ':')
      {
	/*
	 *  The rest is single parameter--can
	 *  include blanks also.
	 */
	para[++i] = s + 1;
	break;
      }
      para[++i] = s;
      if (i >= paramcount)
	break;
      for (; *s != ' ' && *s; s++)
	;
    }
  }
  para[++i] = NULL;
  if (numeric)
    return (do_numeric(numeric, cptr, from, i, para));
  mptr->count++;
  if (!IsRegistered(cptr) && (
  /* patch to avoid server flooding from unregistered connects :  --dl */
	  (mptr->func != m_user)
	  && (mptr->func != m_quit)
	  && (mptr->func != m_admin)
	  && (mptr->func != m_version)
	  && (mptr->func != m_server)
	  && (mptr->func != m_pass)
	  && (mptr->func != m_nick)
	  && (mptr->func != m_pong)
	  && (mptr->func != m_error)))
  {
    sendto_one(from,
	":%s %d %s %s :Register first.",
	me.name, ERR_NOTREGISTERED,
	from->name, ch);
    return -1;
  }
  if (IsRegisteredUser(cptr) &&
#ifdef	IDLE_FROM_MSG
      mptr->func == m_private)
#else
      mptr->func != m_ping && mptr->func != m_pong)
#endif
      from->user->last = now;

  return (*mptr->func) (cptr, from, i, para);
}

/*
 * field breakup for ircd.conf file.
 */
char *getfield(char *newline)
{
  static char *line = NULL;
  char *end, *field;

  if (newline)
    line = newline;
  if (line == NULL)
    return (NULL);

  field = line;
  if ((end = strchr(line, ':')) == NULL)
  {
    line = NULL;
    if ((end = strchr(field, '\n')) == NULL)
      end = field + strlen(field);
  }
  else
    line = end + 1;
  *end = '\0';
  return (field);
}
