/*
 *   IRC - Internet Relay Chat, common/send.c
 *   Copyright (C) 1990 Jarkko Oikarinen and
 *		      University of Oulu, Computing Center
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __GNUG__
#pragma implementation
#endif
#include "sys.h"
#include <stdio.h>
#include "h.h"
#include "struct.h"
#include "s_bsd.h"
#include "s_serv.h"
#include "send.h"
#include "s_misc.h"
#include "match.h"
#include "s_bsd.h"
#include "list.h"
#include "ircd.h"
#include "channel.h"
#include "bsd.h"
#include "class.h"
#include "s_user.h"

RCSTAG_CC("$Id$")

char sendbuf[2048];
static int sentalong[MAXCONNECTIONS];

/*
 *  dead_link
 *    An error has been detected. The link *must* be closed,
 *    but *cannot* call ExitClient (m_bye) from here.
 *    Instead, mark it with FLAGS_DEADSOCKET. This should
 *    generate ExitClient from the main loop.
 * 
 *    If 'notice' is not NULL, it is assumed to be a format
 *    for a message to local opers. I can contain only one
 *    '%s', which will be replaced by the sockhost field of
 *    the failing link.
 * 
 *    Also, the notice is skipped for "uninteresting" cases,
 *    like Persons and yet unknown connections...
 */
char *last_dead_comment = NULL;

static void dead_link(aClient * to, char *notice)
{
  char dead_comment_buf[256];	/* SHOULD be big enough */

  to->flags |= FLAGS_DEADSOCKET;
  /*
   * If because of BUFFERPOOL problem then clean dbuf's now so that
   * notices don't hurt operators below.
   */
  DBufClear(&to->recvQ);
  DBufClear(&to->sendQ);

  /* Keep a copy of the last comment, for later use... */
  sprintf(dead_comment_buf, notice, get_client_name(to, FALSE));
  if (last_dead_comment)
    RunFree(last_dead_comment);
  last_dead_comment = (char *)RunMalloc(strlen(dead_comment_buf) + 1);
  strcpy(last_dead_comment, dead_comment_buf);

  if (!IsPerson(to) && !IsUnknown(to) && !(to->flags & FLAGS_CLOSING))
    sendto_ops(last_dead_comment);
  Debug((DEBUG_ERROR, last_dead_comment));
}

/*
 *  flush_connections
 *    Used to empty all output buffers for all connections. Should only
 *    be called once per scan of connections. There should be a select in
 *    here perhaps but that means either forcing a timeout or doing a poll.
 *    When flushing, all we do is empty the obuffer array for each local
 *    client and try to send it. if we cant send it, it goes into the sendQ
 *    -avalon
 */
void flush_connections(int fd)
{
  Reg1 int i;
  Reg2 aClient *cptr;

  if (fd == me.fd)
  {
    for (i = highest_fd; i >= 0; i--)
      if ((cptr = local[i]) && DBufLength(&cptr->sendQ) > 0)
	send_queued(cptr);
  }
  else if (fd >= 0 && (cptr = local[fd]) && DBufLength(&cptr->sendQ) > 0)
    send_queued(cptr);
}

/*
 *  send_queued
 *    This function is called from the main select-loop (or whatever)
 *    when there is a chance that some output would be possible. This
 *    attempts to empty the send queue as far as possible...
 */
void send_queued(aClient * to)
{
#ifndef pyr
  if (to->flags & FLAGS_BLOCKED)
    return;			/* Don't bother */
#endif
  /*
   *  Once socket is marked dead, we cannot start writing to it,
   *  even if the error is removed...
   */
  if (IsDead(to))
  {
    /*
     *  Actually, we should *NEVER* get here--something is
     *  not working correct if send_queued is called for a
     *  dead socket... --msa
     */
    return;
  }
  while (DBufLength(&to->sendQ) > 0)
  {
    char *msg;
    int len, rlen;

    msg = dbuf_map(&to->sendQ, &len);
    /* Returns always len > 0 */
    if ((rlen = deliver_it(to, msg, len)) < 0)
    {
      dead_link(to, "Write error to %s, closing link");
      return;
    }
    dbuf_delete(&to->sendQ, rlen);
    to->lastsq = DBufLength(&to->sendQ) / 1024;
    if (rlen < len)
    {
      to->flags |= FLAGS_BLOCKED;	/* Wait till select() says we can write again */
      break;
    }
  }

  return;
}

/*
 *  send message to single client
 */
void sendto_one(aClient * to, char *pattern,...)
{
  va_list vl;
  va_start(vl, pattern);
  vsendto_one(to, pattern, vl);
  va_end(vl);
}

void vsendto_one(aClient * to, char *pattern, va_list vl)
{
  vsprintf(sendbuf, pattern, vl);
  sendbufto_one(to);
}

void sendbufto_one(aClient * to)
{
  int len;

  Debug((DEBUG_SEND, "Sending [%s] to %s", sendbuf, to->name));

  if (to->from)
    to = to->from;
  if (IsDead(to))
    return;			/* This socket has already been marked as dead */
  if (to->fd < 0)
  {
    /* This is normal when 'to' was being closed (via exit_client
     *  and close_connection) --Run
     *  Print the debug message anyway...
     */
    Debug((DEBUG_ERROR,
	    "Local socket %s with negative fd %d... AARGH!",
	    to->name, to->fd));
    return;
  }

  len = strlen(sendbuf);
  if (sendbuf[len - 1] != '\n')
  {
#ifndef	IRCII_KLUDGE
    if (len > 510)
      len = 510;
    sendbuf[len++] = '\r';
#else
    if (len > 511)
      len = 511;
#endif
    sendbuf[len++] = '\n';
    sendbuf[len] = '\0';
  }

  if (IsMe(to))
  {
    char tmp_sendbuf[sizeof(sendbuf)];

    strcpy(tmp_sendbuf, sendbuf);
    sendto_ops("Trying to send [%s] to myself!", tmp_sendbuf);
    return;
  }

  if (DBufLength(&to->sendQ) > get_sendq(to))
  {
    if (IsServer(to))
      sendto_ops("Max SendQ limit exceeded for %s: %d > %d",
	  get_client_name(to, FALSE),
	  DBufLength(&to->sendQ), get_sendq(to));
    dead_link(to, "Max Sendq exceeded");
    return;
  }

  else if (dbuf_put(&to->sendQ, sendbuf, len) < 0)
  {
    dead_link(to, "Buffer allocation error for %s");
    return;
  }
  /*
   *  Update statistics. The following is slightly incorrect
   *  because it counts messages even if queued, but bytes
   *  only really sent. Queued bytes get updated in SendQueued.
   */
  to->sendM += 1;
  me.sendM += 1;
  if (to->acpt != &me)
    to->acpt->sendM += 1;
  /*
   *  This little bit is to stop the sendQ from growing too large when
   *  there is no need for it to. Thus we call send_queued() every time
   *  2k has been added to the queue since the last non-fatal write.
   *  Also stops us from deliberately building a large sendQ and then
   *  trying to flood that link with data (possible during the net
   *  relinking done by servers with a large load).
   */
  if (DBufLength(&to->sendQ) / 1024 > to->lastsq)
    send_queued(to);
}

static void vsendto_prefix_one(to, from, pattern, vl)
Reg1 aClient *to;
Reg2 aClient *from;
char *pattern;
va_list vl;
{
  static char sender[HOSTLEN + NICKLEN + USERLEN + 5];
  Reg3 anUser *user;
  char *par;
  int flag = 0;

  par = va_arg(vl, char *);
  if (to && from && MyClient(to) && IsPerson(from) &&
      !mycmp(par, from->name))
  {
    user = from->user;
    strcpy(sender, from->name);
    if (user)
    {
      if (*user->username)
      {
	strcat(sender, "!");
	strcat(sender, user->username);
      }
      if (*user->host && !MyConnect(from))
      {
	strcat(sender, "@");
	strcat(sender, user->host);
	flag = 1;
      }
    }
    /*
     *  flag is used instead of strchr(sender, '@') for speed and
     *  also since username/nick may have had a '@' in them. -avalon
     */
    if (!flag && MyConnect(from) && *user->host)
    {
      strcat(sender, "@");
      if (IsUnixSocket(from))
	strcat(sender, user->host);
      else
	strcat(sender, from->sockhost);
    }
    par = sender;
  }
  /* Assuming 'pattern' always starts with ":%s ..." */
  sprintf(sendbuf, ":%s", par);
  vsprintf(sendbuf + strlen(sendbuf), &pattern[3], vl);
  sendbufto_one(to);
}

void sendto_channel_butone(aClient * one, aClient * from, aChannel * chptr, char *pattern,...)
{
  va_list vl;
  Reg1 Link *lp;
  Reg2 aClient *acptr;
  Reg3 int i;

  va_start(vl, pattern);

  for (i = 0; i < MAXCONNECTIONS; i++)
    sentalong[i] = 0;
  for (lp = chptr->members; lp; lp = lp->next)
  {
    acptr = lp->value.cptr;
    if (acptr->from == one ||	/* ...was the one I should skip */
	(lp->flags & CHFL_ZOMBIE) || IsDeaf(acptr))
      continue;
    i = acptr->from->fd;
    if (MyConnect(acptr) && IsRegisteredUser(acptr))
    {
      vsendto_prefix_one(acptr, from, pattern, vl);
      sentalong[i] = 1;
    }
    else
    {
      /* Now check whether a message has been sent to this
       * remote link already */
      if (sentalong[i] == 0)
      {
	vsendto_prefix_one(acptr, from, pattern, vl);
	sentalong[i] = 1;
      }
    }
  }
  va_end(vl);
  return;
}

/*
 * sendto_server_butone
 *
 * Send a message to all connected servers except the client 'one'.
 */
void sendto_serv_butone(aClient * one, char *pattern,...)
{
  va_list vl;
  Reg1 Dlink *lp;

  va_start(vl, pattern);
  vsprintf(sendbuf, pattern, vl);
  va_end(vl);

  for (lp = me.serv->down; lp; lp = lp->next)
  {
    if (one && lp->value.cptr == one->from)
      continue;
    sendbufto_one(lp->value.cptr);
  }
}

/*
 * sendto_common_channels()
 *
 * Sends a message to all people (inclusing user) on local server who are
 * in same channel with user.
 */
void sendto_common_channels(aClient * user, char *pattern,...)
{
  va_list vl;
  Reg1 int i;
  Reg2 aClient *cptr;
  Reg3 Link *lp;

  va_start(vl, pattern);

  for (i = 0; i <= highest_fd; i++)
  {
    if (!(cptr = local[i]) || IsServer(cptr) ||
	user == cptr || !user->user)
      continue;
    for (lp = user->user->channel; lp; lp = lp->next)
      if (IsMember(user, lp->value.chptr) &&
	  IsMember(cptr, lp->value.chptr))
      {
	vsendto_prefix_one(cptr, user, pattern, vl);
	break;
      }
  }
  if (MyConnect(user))
    vsendto_prefix_one(user, user, pattern, vl);
  va_end(vl);
  return;
}

/*
 * sendto_channel_butserv
 *
 * Send a message to all members of a channel that are connected to this
 * server.
 */
void sendto_channel_butserv(aChannel * chptr, aClient * from, char *pattern,...)
{
  va_list vl;
  Reg1 Link *lp;
  Reg2 aClient *acptr;

  for (va_start(vl, pattern), lp = chptr->members; lp; lp = lp->next)
    if (MyConnect(acptr = lp->value.cptr) &&
	!(lp->flags & CHFL_ZOMBIE))
      vsendto_prefix_one(acptr, from, pattern, vl);
  va_end(vl);
  return;
}

/*
 *  send a msg to all ppl on servers/hosts that match a specified mask
 *  (used for enhanced PRIVMSGs)
 * 
 *  addition -- Armin, 8jun90 (gruner@informatik.tu-muenchen.de)
 */

static int match_it(aClient * one, char *mask, int what)
{
  switch (what)
  {
    case MATCH_HOST:
      return (match(mask, one->user->host) == 0);
    case MATCH_SERVER:
    default:
      return (match(mask, one->user->server->name) == 0);
  }
}

/*
 * sendto_match_servs
 *
 * send to all servers which match the mask at the end of a channel name
 * (if there is a mask present) or to all if no mask.
 */
void sendto_match_servs(aChannel * chptr, aClient * from, char *format,...)
{
  va_list vl;
  Reg1 Dlink *lp;
  Reg2 aClient *cptr;
  char *mask;

  va_start(vl, format);
  if (chptr)
  {
    if (*chptr->chname == '&')
      return;
    if ((mask = strrchr(chptr->chname, ':')))
      mask++;
  }
  else
    mask = NULL;

  vsprintf(sendbuf, format, vl);
  va_end(vl);
  for (lp = me.serv->down; lp; lp = lp->next)
  {
    if ((cptr = lp->value.cptr) == from)
      continue;
    if (!BadPtr(mask) && match(mask, cptr->name))
      continue;
    sendbufto_one(cptr);
  }
}

/*
 * sendto_match_butone
 *
 * Send to all clients which match the mask in a way defined on 'what';
 * either by user hostname or user servername.
 */
void sendto_match_butone(aClient * one, aClient * from, char *mask, int what, char *pattern,...)
{
  va_list vl;
  Reg1 int i;
  Reg2 aClient *cptr, *acptr;

  va_start(vl, pattern);
  for (i = 0; i <= highest_fd; i++)
  {
    if (!(cptr = local[i]))
      continue;			/* that clients are not mine */
    if (cptr == one)		/* must skip the origin !! */
      continue;
    if (IsServer(cptr))
    {
      for (acptr = client; acptr; acptr = acptr->next)
	if (IsRegisteredUser(acptr)
	    && match_it(acptr, mask, what)
	    && acptr->from == cptr)
	  break;
      /* a person on that server matches the mask, so we
       *  send *one* msg to that server ...
       */
      if (acptr == NULL)
	continue;
      /* ... but only if there *IS* a matching person */
    }
    /* my client, does he match ? */
    else if (!(IsRegisteredUser(cptr) &&
	    match_it(cptr, mask, what)))
      continue;
    vsendto_prefix_one(cptr, from, pattern, vl);
  }
  va_end(vl);

  return;
}

/*
 * sendto_lops_butone
 *
 *      Send to *local* ops but one.
 */
void sendto_lops_butone(aClient * one, char *pattern,...)
{
  va_list vl;
  Reg1 aClient *cptr;
  Reg2 Dlink *lp;
  char nbuf[1024];

  sprintf(nbuf, ":%s NOTICE %%s :*** Notice -- ", me.name);
  va_start(vl, pattern);
  vsprintf(nbuf + strlen(nbuf), pattern, vl);
  va_end(vl);
  for (lp = me.serv->client; lp; lp = lp->next)
    if ((cptr = lp->value.cptr) != one && SendServNotice(cptr))
    {
      sprintf(sendbuf, nbuf, cptr->name);
      sendbufto_one(cptr);
    }
  return;
}

/*
 * sendto_ops
 *
 *      Send to *local* ops only.
 */
void vsendto_ops(const char *pattern, va_list vl)
{
  Reg1 aClient *cptr;
  Reg2 int i;
  char fmt[1024];
  char *fmt_target;

  sprintf(fmt, ":%s NOTICE ", me.name);
  fmt_target = &fmt[strlen(fmt)];

  for (i = 0; i <= highest_fd; i++)
    if ((cptr = local[i]) && !IsServer(cptr) && !IsMe(cptr) &&
	SendServNotice(cptr))
    {
      strcpy(fmt_target, cptr->name);
      strcat(fmt_target, " :*** Notice -- ");
      strcat(fmt_target, pattern);
      vsendto_one(cptr, fmt, vl);
    }
#ifdef USE_SERVICES
    else if (cptr && IsService(cptr) &&
	(cptr->service->wanted & SERVICE_WANT_SERVNOTE))
    {
      strcpy(fmt_target, cptr->name);
      strcat(fmt, " :*** Notice -- ");
      strcat(fmt, pattern);
      vsendto_one(cptr, fmt, vl);
    }
#endif

  return;
}

void sendto_ops(const char *pattern,...)
{
  va_list vl;
  va_start(vl, pattern);
  vsendto_ops(pattern, vl);
  va_end(vl);
}

/*
 *  sendto_ops_butone
 *    Send message to all operators.
 *  one - client not to send message to
 *  from- client which message is from *NEVER* NULL!!
 */
void sendto_ops_butone(aClient * one, aClient * from, char *pattern,...)
{
  va_list vl;
  Reg1 int i;
  Reg2 aClient *cptr;

  va_start(vl, pattern);
  for (i = 0; i <= highest_fd; i++)
    sentalong[i] = 0;
  for (cptr = client; cptr; cptr = cptr->next)
  {
    if (!SendWallops(cptr))
      continue;
    i = cptr->from->fd;		/* find connection oper is on */
    if (sentalong[i])		/* sent message along it already ? */
      continue;
    if (cptr->from == one)
      continue;			/* ...was the one I should skip */
    sentalong[i] = 1;
    vsendto_prefix_one(cptr->from, from, pattern, vl);
  }
  va_end(vl);

  return;
}

/*
 * to - destination client
 * from - client which message is from
 *
 * NOTE: NEITHER OF THESE SHOULD *EVER* BE NULL!!
 * -avalon
 */
void sendto_prefix_one(Reg1 aClient * to, Reg2 aClient * from, char *pattern,...)
{
  va_list vl;
  va_start(vl, pattern);
  vsendto_prefix_one(to, from, pattern, vl);
  va_end(vl);
}

/*
 * sendto_realops
 *
 *      Send to *local* ops only but NOT +s nonopers.
 */
void sendto_realops(char *pattern,...)
{
  va_list vl;
  Reg1 aClient *cptr;
  Reg2 int i;
  char fmt[1024];
  Reg3 char *fmt_target;

  va_start(vl, pattern);

  sprintf(fmt, ":%s NOTICE ", me.name);
  fmt_target = &fmt[strlen(fmt)];

  for (i = 0; i <= highest_fd; i++)
    if ((cptr = local[i]) && IsOper(cptr))
    {
      strcpy(fmt_target, cptr->name);
      strcat(fmt_target, " :*** Notice -- ");
      strcat(fmt_target, pattern);
      vsendto_one(cptr, fmt, vl);
    }
  va_end(vl);
  return;
}

/*
 *  send message to all servers of protocol 'p' and lower.
 */
void sendto_lowprot_butone(aClient * cptr, int p, char *pattern,...)
{
  va_list vl;
  Dlink *lp;
  va_start(vl, pattern);
  for (lp = me.serv->down; lp; lp = lp->next)
    if (lp->value.cptr != cptr && Protocol(lp->value.cptr) <= p)
      vsendto_one(lp->value.cptr, pattern, vl);
  va_end(vl);
}

/*
 *  send message to all servers of protocol 'p' and higher.
 */
void sendto_highprot_butone(aClient * cptr, int p, char *pattern,...)
{
  va_list vl;
  Dlink *lp;
  va_start(vl, pattern);
  for (lp = me.serv->down; lp; lp = lp->next)
    if (lp->value.cptr != cptr && Protocol(lp->value.cptr) >= p)
      vsendto_one(lp->value.cptr, pattern, vl);
  va_end(vl);
}
