/*
 *   IRC - Internet Relay Chat, common/support.c
 *   Copyright (C) 1990, 1991 Armin Gruner
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __GNUG__
#pragma implementation
#endif
#include "sys.h"
#ifdef DYNIXPTX
#include <sys/timers.h>
#include <stddef.h>
#endif
#include <unistd.h>
#include <stdarg.h>
#include <signal.h>
#include "h.h"
#include "send.h"
#include "ircd.h"
#include "s_bsd.h"
#include "support.h"

RCSTAG_CC("$Id$")

#ifdef NEED_STRTOKEN
/*
 *   strtoken.c --   walk through a string of tokens, using a set
 *                   of separators
 *                   argv 9/90
 */

char *strtoken(char **save, char *str, char *fs)
{
  char *pos = *save;		/* keep last position across calls */
  Reg1 char *tmp;

  if (str)
    pos = str;			/* new string scan */

  while (pos && *pos && strchr(fs, *pos) != NULL)
    pos++;			/* skip leading separators */

  if (!pos || !*pos)
    return (pos = *save = NULL);	/* string contains only sep's */

  tmp = pos;			/* now, keep position of the token */

  while (*pos && strchr(fs, *pos) == NULL)
    pos++;			/* skip content of the token */

  if (*pos)
    *pos++ = '\0';		/* remove first sep after the token */
  else
    pos = NULL;			/* end of string */

  *save = pos;
  return (tmp);
}
#endif /* NEED_STRTOKEN */

#ifdef	NEED_STRTOK
/*
 *  NOT encouraged to use!
 */

char *strtok(char *str, char *fs)
{
  static char *pos;

  return strtoken(&pos, str, fs);
}

#endif /* NEED_STRTOK */

#ifdef NEED_STRERROR
/*
 *   strerror - return an appropriate system error string to a given errno
 *
 *              argv 11/90
 */

char *strerror(int err_no)
{
  static char buff[40];
  char *errp;

  errp = (err_no > sys_nerr ? (char *)NULL : sys_errlist[err_no]);

  if (errp == (char *)NULL)
  {
    errp = buff;
    sprintf(errp, "Unknown Error %d", err_no);
  }
  return errp;
}

#endif /* NEED_STRERROR */

/*
 *    inetntoa  --    changed name to remove collision possibility and
 *                    so behaviour is gaurunteed to take a pointer arg.
 *                    -avalon 23/11/92
 *    inet_ntoa --    returned the dotted notation of a given
 *                    internet number (some ULTRIX don't have this)
 *                    argv 11/90).
 *    inet_ntoa --    its broken on some Ultrix/Dynix too. -avalon
 *    $Id: support.c,v 1.1.1.1 1996/05/05 22:25:22 klmitch Exp $
 */

char *inetntoa(char *in)
{
  static char buf[16];
  Reg1 u_char *s = (u_char *) in;
  Reg2 int a, b, c, d;

  a = (int)*s++;
  b = (int)*s++;
  c = (int)*s++;
  d = (int)*s++;
  sprintf(buf, "%d.%d.%d.%d", a, b, c, d);

  return buf;
}

#ifdef NEED_INET_NETOF
/*
 *    inet_netof --   return the net portion of an internet number
 *                    argv 11/90
 *    $Id: support.c,v 1.1.1.1 1996/05/05 22:25:22 klmitch Exp $
 * 
 */

int inet_netof(struct in_addr in)
{
  int addr = in.s_net;

  if (addr & 0x80 == 0)
    return ((int)in.s_net);

  if (addr & 0x40 == 0)
    return ((int)in.s_net * 256 + in.s_host);

  return ((int)in.s_net * 256 + in.s_host * 256 + in.s_lh);
}
#endif /* NEED_INET_NETOF */

#ifdef DYNIXPTX
/* This is copied from ircu3.0.0 (with permission), not vica versa. */
int gettimeofday(struct timeval *tv, void * /*UNUSED */ )
{
  register int ret;
  static struct timespec tp;

  if ((ret = getclock(TIMEOFDAY, &tp)))
    return ret;
  tv->tv_sec = (long)tp.tv_sec;
  tv->tv_usec = (tp.tv_nsec + 500) / 1000;
  return 0;
}
#endif /* DYNIXPTX */

#ifdef DEBUGMODE

void dumpcore(const char *pattern,...)
{
  va_list vl;
  static time_t lastd = 0;
  static int dumps = 0;
  char corename[12];
  time_t now;
  int p;

  va_start(vl, pattern);

  now = time(NULL);

  if (!lastd)
    lastd = now;
  else if (now - lastd < 60 && dumps > 2)
    s_die();
  if (now - lastd > 60)
  {
    lastd = now;
    dumps = 1;
  }
  else
    dumps++;
  p = getpid();
  if (fork() > 0)
  {
    kill(p, 3);
    kill(p, 9);
  }
  write_pidfile();
  sprintf(corename, "core.%d", p);
  rename("core", corename);
  Debug((DEBUG_FATAL, "Dumped core : core.%d", p));
  sendto_ops("Dumped core : core.%d", p);
  vdebug(DEBUG_FATAL, pattern, vl);
  vsendto_ops(pattern, vl);
  s_die();

  va_end(vl);
}
#endif

/*
 *  read a string terminated by \r or \n in from a fd
 * 
 *  Created: Sat Dec 12 06:29:58 EST 1992 by avalon
 *  Returns:
 *    0 - EOF
 *    -1 - error on read
 *      >0 - number of bytes returned (<=num)
 *  After opening a fd, it is necessary to init dgets() by calling it as
 *    dgets(x,y,0);
 *  to mark the buffer as being empty.
 */
int dgets(int fd, char *buf, int num)
{
  static char dgbuf[8192];
  static char *head = dgbuf, *tail = dgbuf;
  register char *s, *t;
  register int n, nr;

  /*
   *  Sanity checks.
   */
  if (head == tail)
    *head = '\0';
  if (!num)
  {
    head = tail = dgbuf;
    *head = '\0';
    return 0;
  }
  if (num > sizeof(dgbuf) - 1)
    num = sizeof(dgbuf) - 1;
dgetsagain:
  if (head > dgbuf)
  {
    for (nr = tail - head, s = head, t = dgbuf; nr > 0; nr--)
      *t++ = *s++;
    tail = t;
    head = dgbuf;
  }
  /*
   *  check input buffer for EOL and if present return string.
   */
  if (head < tail &&
      ((s = strchr(head, '\n')) || (s = strchr(head, '\r'))) && s < tail)
  {
    n = MIN(s - head + 1, num);	/* at least 1 byte */
  dgetsreturnbuf:
    memcpy(buf, head, n);
    head += n;
    if (head == tail)
      head = tail = dgbuf;
    return n;
  }

  if (tail - head >= num)	/* dgets buf is big enough */
  {
    n = num;
    goto dgetsreturnbuf;
  }

  n = sizeof(dgbuf) - (tail - dgbuf) - 1;
  nr = read(fd, tail, n);
  if (nr == -1)
  {
    head = tail = dgbuf;
    return -1;
  }
  if (!nr)
  {
    if (head < tail)
    {
      n = MIN(tail - head, num);
      goto dgetsreturnbuf;
    }
    head = tail = dgbuf;
    return 0;
  }
  tail += nr;
  *tail = '\0';
  for (t = head; (s = strchr(t, '\n'));)
  {
    if ((s > head) && (s > dgbuf))
    {
      t = s - 1;
      for (nr = 0; *t == '\\'; nr++)
	t--;
      if (nr & 1)
      {
	t = s + 1;
	s--;
	nr = tail - t;
	while (nr--)
	  *s++ = *t++;
	tail -= 2;
	*tail = '\0';
      }
      else
	s++;
    }
    else
      s++;
    t = s;
  }
  *tail = '\0';
  goto dgetsagain;
}
