/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * config.h: A header for the configuration functions.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: config.h,v 1.66.2.5 2004/07/25 06:44:02 musirc Exp $
 */

#ifndef INCLUDED_config_h
#define INCLUDED_config_h
#include "fileio.h"
#include "defs.h"
#include "motd.h"
#include "client.h"

struct Client;
struct DNSReply;

struct ip_value
{
  struct irc_ssaddr ip;
  int ip_mask;
  int type;
};

extern FBFILE *conf_fbfile_in;
extern char conf_line_in[256];

typedef enum {  
  CONF_TYPE,
  CLASS_TYPE,
  OPER_TYPE,
  CLIENT_TYPE,
  SERVER_TYPE,
  HUB_TYPE,
  LEAF_TYPE,
  KLINE_TYPE,
  DLINE_TYPE,
  EXEMPTDLINE_TYPE,
  XLINE_TYPE,    
  ULINE_TYPE,
  CJUPE_TYPE,     
  NJUPE_TYPE
} ConfType;

struct ConfItem
{
  char *name;/* Primary key */
  dlink_node node;/* link into known ConfItems of this type */
  ConfType type;
};

/* MatchItem - used for XLINE and ULINE types */
struct MatchItem
{
  char *user;		/* Used for ULINE only */
  char *host;		/* Used for ULINE only */
  char *reason;
  char *oper_reason;
  int action;		/* used for xline and uline */
  int ref_count; /* How many times is this matchitem in use */
  int illegal; /* Should it be deleted when possible? */
};

struct AccessItem
{
  dlink_node node;
  unsigned int status;   /* If CONF_ILLEGAL, delete when no clients */
  unsigned int flags;
  int clients;  /* Number of *LOCAL* clients using this */
  struct irc_ssaddr my_ipnum; /* ip to bind to for outgoing connect */
  struct irc_ssaddr ipnum;	/* ip to connect to */
  char *host;     /* host part of user@host */
  char *passwd;
  char *spasswd;  /* Password to send. */
  char *reason;
  char *oper_reason;
  char *user;     /* user part of user@host */
  int port;
  char *fakename;   /* Mask name */
  time_t hold;     /* Hold action until this time (calendar time) */
  struct ConfItem *class_ptr;  /* Class of connection */
  struct DNSQuery* dns_query;
  int aftype;
};

struct ClassItem
{
  long max_sendq;
  int con_freq;
  int ping_freq;
  int max_total;
  int max_local;
  int max_global;
  int max_ident;
  int max_perip;
  int curr_user_count;
};

#define ConFreq(x)	((x)->con_freq)
#define PingFreq(x)	((x)->ping_freq)
#define MaxTotal(x)	((x)->max_total)
#define MaxGlobal(x)	((x)->max_global)
#define MaxLocal(x)	((x)->max_local)
#define MaxIdent(x)	((x)->max_ident)
#define MaxPerIp(x)	((x)->max_perip)
#define MaxSendq(x)	((x)->max_sendq)
#define CurrUserCount(x) ((x)->curr_user_count)
#define ClassPtr(x)      ((x)->class_ptr)

#define CONF_ILLEGAL            0x80000000
#define CONF_RESERVED           0x00000001
#define CONF_CLIENT             0x00000002
#define CONF_SERVER             0x00000004
#define CONF_OPERATOR           0x00000008
#define CONF_KLINE		0x00000010
#define CONF_CLASS              0x00000020
#define CONF_LEAF               0x00000040
#define CONF_LISTEN_PORT        0x00000080
#define CONF_HUB                0x00000100
#define CONF_EXEMPTKLINE        0x00000200
#define CONF_NOLIMIT            0x00000400
#define CONF_DLINE              0x00000800
#define CONF_XLINE              0x00001000
#define CONF_ULINE              0x00002000
#define CONF_EXEMPTDLINE        0x00004000

#define CONF_SERVER_MASK       CONF_SERVER
#define CONF_CLIENT_MASK       (CONF_CLIENT | CONF_OPERATOR | CONF_SERVER_MASK)

#define CONF_CJUPE	        0x80000001
#define CONF_NJUPE	        0x80000002

#define IsConfIllegal(x)	((x)->status & CONF_ILLEGAL)
#define SetConfIllegal(x)	((x)->status |= CONF_ILLEGAL)
#define IsConfServer(x)		((x)->status == CONF_SERVER)
#define SetConfServer(x)	((x)->status = CONF_SERVER)
#define IsConfOperator(x)	((x)->status & CONF_OPERATOR)
#define IsConfHub(x)		((x)->status == CONF_HUB)
#define SetConfHub(x)		((x)->status = CONF_HUB)
#define IsConfLeaf(x)		((x)->status == CONF_LEAF)
#define SetConfLeaf(x)		((x)->status = CONF_LEAF)
#define IsConfHubOrLeaf(x)	((x)->status & (CONF_HUB|CONF_LEAF))
#define IsConfKill(x)		((x)->status == CONF_KLINE)
#define IsConfClient(x)		((x)->status & CONF_CLIENT)
#define IsConfTypeOfClient(x)	((x)->status & CONF_CLIENT_MASK)
#define IsConfUline(x)		((x)->status & CONF_ULINE)
#define IsConfXline(x)		((x)->status & CONF_XLINE)

#define CONF_FLAGS_DO_IDENTD            0x00000001
#define CONF_FLAGS_LIMIT_IP             0x00000002
#define CONF_FLAGS_NO_TILDE             0x00000004
#define CONF_FLAGS_NEED_IDENTD          0x00000008
#define CONF_FLAGS_PASS_IDENTD          0x00000010
#define CONF_FLAGS_NOMATCH_IP           0x00000020
#define CONF_FLAGS_EXEMPTKLINE          0x00000040
#define CONF_FLAGS_NOLIMIT              0x00000080
#define CONF_FLAGS_IDLE_LINED           0x00000100
#define CONF_FLAGS_SPOOF_IP             0x00000200
#define CONF_FLAGS_SPOOF_NOTICE         0x00000400
#define CONF_FLAGS_RESTRICTED           0x00000800
#define CONF_FLAGS_CAN_FLOOD            0x00001000
#define CONF_FLAGS_NEED_PASSWORD	0x00002000
/* server flags */
#define CONF_FLAGS_ALLOW_AUTO_CONN      0x00004000
#define CONF_FLAGS_ENCRYPTED            0x00008000
#define CONF_FLAGS_TEMPORARY            0x00010000

/* Macros for struct AccessItem */
#define IsLimitIp(x)            ((x)->flags & CONF_FLAGS_LIMIT_IP)
#define IsNoTilde(x)            ((x)->flags & CONF_FLAGS_NO_TILDE)
#define IsConfCanFlood(x)       ((x)->flags & CONF_FLAGS_CAN_FLOOD)
/* see below (default is 0 and need_password = yes) */
#define IsNeedPassword(x)	(!((x)->flags & CONF_FLAGS_NEED_PASSWORD))
#define IsNeedIdentd(x)         ((x)->flags & CONF_FLAGS_NEED_IDENTD)
#define IsPassIdentd(x)         ((x)->flags & CONF_FLAGS_PASS_IDENTD)
#define IsNoMatchIp(x)          ((x)->flags & CONF_FLAGS_NOMATCH_IP)
#define IsConfExemptKline(x)    ((x)->flags & CONF_FLAGS_EXEMPTKLINE)
#define IsConfExemptLimits(x)   ((x)->flags & CONF_FLAGS_NOLIMIT)
#define IsConfIdlelined(x)      ((x)->flags & CONF_FLAGS_IDLE_LINED)
#define IsConfDoIdentd(x)       ((x)->flags & CONF_FLAGS_DO_IDENTD)
#define IsConfDoSpoofIp(x)      ((x)->flags & CONF_FLAGS_SPOOF_IP)
#define IsConfSpoofNotice(x)    ((x)->flags & CONF_FLAGS_SPOOF_NOTICE)
#define IsConfRestricted(x)     ((x)->flags & CONF_FLAGS_RESTRICTED)
#define IsConfEncrypted(x)      ((x)->flags & CONF_FLAGS_ENCRYPTED)
#define IsConfAllowAutoConn(x)  ((x)->flags & CONF_FLAGS_ALLOW_AUTO_CONN)
#define SetConfAllowAutoConn(x)	((x)->flags |= CONF_FLAGS_ALLOW_AUTO_CONN)
#define ClearConfAllowAutoConn(x) ((x)->flags &= ~CONF_FLAGS_ALLOW_AUTO_CONN)
#define SetConfTemporary(x)	((x)->flags |= CONF_FLAGS_TEMPORARY)

/* shared server entry types */
#define SHARED_KLINE	0x01
#define SHARED_XLINE	0x02
#define SHARED_JUPE	0x04
#define SHARED_ALL	(SHARED_KLINE | SHARED_XLINE | SHARED_JUPE)

struct config_file_entry
{
  const char *dpath;          /* DPATH if set from command line */
  const char *configfile;
  const char *klinefile;
  const char *xlinefile;
  const char *dlinefile;
  const char *cjupefile;
  const char *njupefile;
  char *logpath;
  char *operlog;
  MessageFile motd;
  MessageFile opermotd;
  MessageFile linksfile;
  int dots_in_ident;
  int anti_nick_flood;
  int anti_spam_exit_message_time;
  int max_accept;
  int max_nick_time;
  int max_nick_changes;
  int ts_max_delta;
  int ts_warn_delta;
  int hide_ban_reason;
  int stats_o_oper_only;
  int stats_k_oper_only;
  int stats_i_oper_only;
  int stats_P_oper_only;
  int pace_wait;
  int pace_wait_simple;
  int no_oper_flood;
  int true_no_oper_flood;
  int oper_pass_jupes;
  int drone_detect;
  int autodline_drones;
  int drone_dline_time;
  int cjupe_dline_time;
  int hide_spoof_ips;
  int idletime;
  int maximum_links;
  int oper_umodes;
  int crypt_oper_password;
  int max_targets;
  int ssignore_wait;
  int min_nonwildcard;
  int min_nonwildcard_simple;
  int kill_chase_time_limit;
  int default_floodcount;
  int client_flood;
  int throttle_time;
  int disable_auth;
  int disable_remote;
  int use_services;
};

struct config_channel_entry
{
  int cjupe_dline;
  int disable_local_channels;
  int knock_delay;
  int knock_delay_channel;
  unsigned int max_bans;
  unsigned int max_chans_per_user;
  int no_create_on_split;
  int no_join_on_split;
  int quiet_on_ban;
  int default_split_server_count;
  int default_split_user_count;
};

struct config_server_hide
{
  int flatten_links;
  int hide_servers;
  int links_delay;
  int links_disabled;
  int hidden;
  int disable_hidden;
  int hide_server_ips;
};

struct server_info
{
  char *name;
  char *description;
  char *network_name;
  int hub;
  struct irc_ssaddr ip;
  struct irc_ssaddr ip6;
  int max_clients;
  int specific_ipv4_vhost;
  int specific_ipv6_vhost;
  struct sockaddr_in dns_host;
  int can_use_v6;
};

struct admin_info
{
  char *name;
  char *description;
  char *email;
};

struct logging_entry
{
  unsigned int use_logging;
  char operlog[MAXPATHLEN + 1];
  char userlog[MAXPATHLEN + 1];
  char failed_operlog[MAXPATHLEN + 1];
};

extern unsigned int scount;
extern int ypass;
extern int specific_ipv4_vhost; /* used in bsd.c */
extern int specific_ipv6_vhost;
extern dlink_list class_items;
extern dlink_list server_items;
extern dlink_list hub_items;
extern dlink_list leaf_items;
extern dlink_list temporary_klines;
extern dlink_list temporary_dlines;
extern struct logging_entry ConfigLoggingEntry;
extern struct config_file_entry ConfigFileEntry;/* defined in ircd.c*/
extern struct config_channel_entry ConfigChannel;/* defined in channel.c*/
extern struct config_server_hide ConfigServerHide; /* defined in config.c */
extern struct server_info ServerInfo;       /* defined in ircd.c */
extern struct admin_info AdminInfo;        /* defined in ircd.c */
/* End GLOBAL section */

extern unsigned long get_sendq(struct Client *);
extern int get_con_freq(struct ClassItem *);
extern const char *get_client_class(struct Client *);
extern int get_client_ping(struct Client *);
extern void check_class(void);
extern void init_class(void);
extern struct ConfItem *find_class(const char *);
extern void init_ip_hash_table(void);
extern void count_ip_hash(int *, unsigned long *);
extern void remove_one_ip(struct irc_ssaddr *);
extern struct ConfItem *make_conf_item(ConfType);
extern void free_access_item(struct AccessItem *);
extern void read_conf_files(int);
extern int attach_conf(struct Client *, struct ConfItem *);
extern int attach_connect_block(struct Client *, const char *, const char *);
extern int check_client(struct Client *, struct Client *, const char *);

extern int detach_conf(struct Client *, ConfType);
extern void detach_all_confs(struct Client *);

extern struct ConfItem *find_conf_name(dlink_list *, const char *, ConfType);
extern struct ConfItem *find_conf_exact(ConfType, const char *, const char *, const char *);
extern struct AccessItem *find_kill(struct Client *);
extern int conf_connect_allowed(struct irc_ssaddr *, int);
extern char *oper_privs_as_string(const unsigned int);
extern void split_user_host(char *, char **, char **);
extern struct ConfItem *find_matching_name_conf(ConfType, const char *,
                                                const char *, const char *, int);
extern struct ConfItem *find_exact_name_conf(ConfType, const char *,
                                             const char *, const char *);
extern void delete_conf_item(struct ConfItem *);
extern void get_printable_conf(struct ConfItem *, char **, char **,
                               char **, int *, char **, char **);
extern void report_confitem_types(struct Client *, ConfType);
extern void yyerror(const char *);
extern int conf_yy_fatal_error(void);
extern int conf_fbgets(char *, unsigned int, FBFILE *);
extern void write_conf_line(const struct Client *, struct ConfItem *,
                            const char *, time_t);
extern int remove_conf_line(ConfType, struct Client *, const char *,
                            const char *);
extern void add_tkline(struct AccessItem *);
extern void add_tdline(struct AccessItem *);
extern void cleanup_tklines(void *);
extern const char *get_conf_name(ConfType);
extern int rehash(int);
extern int conf_add_server(struct ConfItem *, unsigned int, const char *);
extern void conf_add_class_to_conf(struct ConfItem *, const char *);
extern void conf_add_d_conf(struct AccessItem *);
extern void parse_csv_file(FBFILE *, ConfType);
extern char *get_oper_name(const struct Client *);
extern void *map_to_conf(struct ConfItem *);
struct ConfItem *unmap_conf_item(void *);

extern int yylex(void);

#define NOT_AUTHORIZED    (-1)
#define I_LINE_FULL       (-2)
#define TOO_MANY          (-3)
#define BANNED_CLIENT	  (-4)
#define TOO_FAST	  (-5)
#define CLEANUP_TKLINES_TIME 60
#endif
