/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_channel.c,v 1.22.2.15 2004/07/18 21:43:54 musirc Exp $
 */

#include "handlers.h"
#include "channel.h"
#include "channel_mode.h"
#include "client.h"
#include "hash.h"
#include "jupe.h"
#include "istring.h"
#include "sprintf.h"
#include "numeric.h"
#include "send.h"
#include "server.h"
#include "config.h"
#include "packet.h"
#include "user.h"
#include "modules.h"
#include "log.h"

struct entity
{
  struct Channel *chptr;
  char *key;
  int flags;
};

static struct entity targets[512];
static int ntargets, pargs;
static int build_target_list(struct Client *, char *, char *);
static int is_target(struct Channel *);
static void m_part(struct Client *, struct Client *, int, char **);
static void m_join(struct Client *, struct Client *, int, char **);
static void ms_join(struct Client *, struct Client *, int, char **);
static void part_one_client(struct Client *, struct Client *, char *, char *);
static void set_final_mode(struct Mode *, struct Mode *);
static void remove_our_modes(struct Channel *, struct Client *);
static void remove_a_mode(struct Channel *, struct Client *, int, char);
static void ms_sjoin(struct Client *, struct Client *, int, char **);
static void m_mode(struct Client *, struct Client *, int, char **);
static void mo_omode(struct Client *, struct Client *, int, char **);
static void m_kick(struct Client *, struct Client *, int, char **);
static char modebuf[BUFSIZE], parabuf[BUFSIZE], sendbuf[BUFSIZE], *mbuf;
extern BlockHeap *ban_heap;

struct Message join_msgtab = {
  "JOIN", 2, MFLG_SLOW,
  {m_unregistered, m_join, ms_join, m_join}
};
struct Message part_msgtab = {
  "PART", 2, MFLG_SLOW,
  {m_unregistered, m_part, m_part, m_part}
};
struct Message sjoin_msgtab = {
  "SJOIN", 5, MFLG_SLOW,
  {m_unregistered, m_ignore, ms_sjoin, m_ignore}
};
struct Message mode_msgtab = {
  "MODE", 2, MFLG_SLOW,
  {m_unregistered, m_mode, m_mode, m_mode}
};
struct Message omode_msgtab = {
  "OMODE", 2, MFLG_SLOW,
  {m_ignore, m_ignore, m_ignore, mo_omode}
};
struct Message kick_msgtab = {
  "KICK", 3, MFLG_SLOW,
  {m_unregistered, m_kick, m_kick, m_kick}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&join_msgtab);
  mod_add_cmd(&part_msgtab);
  mod_add_cmd(&sjoin_msgtab);
  mod_add_cmd(&mode_msgtab);
  mod_add_cmd(&omode_msgtab);
  mod_add_cmd(&kick_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&join_msgtab);
  mod_del_cmd(&part_msgtab);
  mod_del_cmd(&sjoin_msgtab);
  mod_del_cmd(&mode_msgtab);
  mod_del_cmd(&omode_msgtab);
  mod_del_cmd(&kick_msgtab);
}
const char *_version = "$Revision: 1.22.2.15 $";
#endif

/* parv[0] = sender prefix
 * parv[1] = channel
 * parv[2] = channel password (key)
 */
static void
m_join(struct Client *client_p, struct Client *source_p,
       int parc, char *parv[])
{
  struct Channel *chptr = NULL;
  char *key = NULL;
  int i, a = 0, successful_join_count = 0;
  unsigned int flags = 0;

  if (build_target_list(source_p, parv[1], ((parc > 2) ? parv[2] : NULL)) == -2)
  {
    exit_client(client_p, source_p, &me, "Joining cjuped channel");
    return;
  }

  for (;a<ntargets;++a)
  {
    chptr = targets[a].chptr;
    key = targets[a].key;
    flags = targets[a].flags;

    if (IsMember(source_p, chptr))
      continue;

    /* can_join checks for +i key, bans. */
    if ((i = can_join(source_p, chptr, key)))
    {
      sendto_one(source_p, form_str(i), me.name, source_p->name, chptr->name);
      continue;
    }
    ++successful_join_count;
    add_user_to_channel(chptr, source_p, flags);

    /*  Set timestamp if appropriate, and propagate */
    if (flags & CHFL_CHANOP)
    {
      chptr->channelts = CurrentTime;
      chptr->mode.mode |= MODE_TOPICLIMIT;
      chptr->mode.mode |= MODE_NOPRIVMSGS;

      sendto_server(client_p, chptr, ":%s SJOIN %lu %s +nt :@%s",
                    me.name, (unsigned long)chptr->channelts,
                    chptr->name, source_p->name);
      /* notify all other users on the new channel */
      sendto_channel_local(ALL_MEMBERS, chptr, ":%s!%s@%s JOIN :%s",
                           source_p->name, source_p->username,
                           source_p->host, chptr->name);
      sendto_channel_local(ALL_MEMBERS, chptr, ":%s MODE %s +nt",
                           me.name, chptr->name);
    }
    else
    {
      sendto_server(client_p, chptr, ":%s SJOIN %lu %s + :%s",
                    me.name, (unsigned long)chptr->channelts,
                    chptr->name, source_p->name);

      sendto_channel_local(ALL_MEMBERS, chptr, ":%s!%s@%s JOIN :%s",
                           source_p->name, source_p->username,
                           source_p->host, chptr->name);
    }
    del_invite(chptr, source_p);

    if (chptr->topic != NULL)
    {
      sendto_one(source_p, form_str(RPL_TOPIC), me.name,
                 source_p->name, chptr->name, chptr->topic);

      sendto_one(source_p, form_str(RPL_TOPICWHOTIME),
                 me.name, source_p->name, chptr->name,
                 chptr->topic_info, chptr->topic_time);
    }
    channel_member_names(source_p, chptr, 1, 0);
  }
}

/* inputs       - parv[0] = uid
 *                parv[1] = ts
 *                parv[2] = channel name
 *                parv[3] = modes
 * side effects - handles remote JOIN's sent by servers. In TSora
 *                remote clients are joined using SJOIN, hence a
 *                JOIN sent by a server on behalf of a client is an error.
 *                here, the initial code is in to take an extra parameter
 *                and use it for the TimeStamp on a new channel.
 */
static void
ms_join(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  struct Channel *chptr;
  time_t newts, oldts;
  static struct Mode mode, *oldmode;
  int args = 0, keep_our_modes = 1, keep_new_modes = 1, isnew;
  char *s;
  const char *servername;

  if (parc < 4)
    return;

  if (*parv[2] == '&')
    return;

  if (!check_channel_name(parv[2]))
    return;

  mbuf = modebuf;
  mode.mode = mode.limit = 0;
  mode.key[0] = '\0';

  s = parv[3];
  while (*s)
  {
    switch (*(s++))
    {
      case 'C':
        mode.mode |= MODE_NOCTCP;
        break;
      case 'c':
	mode.mode |= MODE_NOCONTROL;
	break;
      case 'i':
        mode.mode |= MODE_INVITEONLY;
        break;
      case 'm':
        mode.mode |= MODE_MODERATED;
        break;
      case 'n':
        mode.mode |= MODE_NOPRIVMSGS;
        break;
      case 'p':
        mode.mode |= MODE_PRIVATE;
        break;
      case 's':
        mode.mode |= MODE_SECRET;
        break;
      case 't':
        mode.mode |= MODE_TOPICLIMIT;
        break;
      case 'k':
        if (parc < 5+args)
          return;

        strlcpy(mode.key, parv[4 + args], sizeof(mode.key));
        args++;
        break;
      case 'l':
        if (parc < 5+args)
          return;

        mode.limit = atoi(parv[4 + args]);
        args++;
        break;
    }
  }

  if ((chptr = get_or_create_channel(parv[2], &isnew)) == NULL)
    return; /* channel name too long? */

  newts = atol(parv[1]);
  oldts = chptr->channelts;
  oldmode = &chptr->mode;

  if (!newts && !isnew && oldts)
  {
    sendto_channel_local(ALL_MEMBERS, chptr,
                         ":%s NOTICE %s :*** TS for %s changed from %lu to 0",
                         me.name, chptr->name, chptr->name, (unsigned long)oldts);
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "Server %s changing TS on %s from %lu to 0",
                         source_p->name, chptr->name, (unsigned long)oldts);
  }

  if (isnew)
    chptr->channelts = newts;
  else if (newts == 0 || oldts == 0)
    chptr->channelts = 0;
  else if (newts == oldts)
    ;
  else if (newts < oldts)
  {
    keep_our_modes = 0;
    chptr->channelts = newts;
  }
  else
    keep_new_modes = 0;

  if (!keep_new_modes)
    mode = *oldmode;
  else if (keep_our_modes)
  {
    mode.mode |= oldmode->mode;
    if (oldmode->limit > mode.limit)
      mode.limit = oldmode->limit;
    if (strcmp(mode.key, oldmode->key) < 0)
      strcpy(mode.key, oldmode->key);
  }

  set_final_mode(&mode, oldmode);
  chptr->mode = mode;
  /* Lost the TS, other side wins, so remove modes on this side */
  if (!keep_our_modes)
  {
    remove_our_modes(chptr, source_p);
    sendto_channel_local(ALL_MEMBERS, chptr,
                         ":%s NOTICE %s :*** TS for %s changed from %lu to %lu",
                          me.name, chptr->name, chptr->name,
                         (unsigned long)oldts, (unsigned long)newts);
  }

  if (*modebuf != '\0')
  {
    servername = (ConfigServerHide.hide_servers || IsHidden(source_p)) ?
                  me.name : source_p->name;
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s MODE %s %s %s",
                         servername, chptr->name, modebuf, parabuf);
  }

  if (!IsMember(source_p, chptr))
  {
    add_user_to_channel(chptr, source_p, 0);
    sendto_channel_local(ALL_MEMBERS,chptr, ":%s!%s@%s JOIN :%s",
                         source_p->name, source_p->username,
                         source_p->host, chptr->name);
  }
  sendto_server(client_p, chptr, ":%s SJOIN %lu %s + :%s",
                source_p->user->server->name, (unsigned long)chptr->channelts,
                chptr->name, source_p->name);
}

/* inputs       - pointer to given client_p (server)
 *              - pointer to given source (oper/client etc.)
 *              - pointer to list of channels
 *              - pointer to list of keys
 * output       - number of valid entities
 * side effects - targets list is modified to contain a list of
 *                pointers to channels. display whatever errors
 *                that result from a join attempt to the user.
 */
static int
build_target_list(struct Client *source_p, char *channels, char *keys)
{
  int error_reported, flags = 0;
  char *p, *p2, *chan, *key = keys;
  struct Channel *chptr = NULL;

  ntargets = error_reported = 0;

  for (chan = strtoken(&p, channels, ","); chan;
       key = (key) ? strtoken(&p2, keys, ",") : NULL,
       chan = strtoken(&p, NULL, ","))
  {
    if (!check_channel_name(chan))
    {
      sendto_one(source_p, form_str(ERR_BADCHANNAME),
                 me.name, source_p->name, chan);
      continue;
    }

    if (!IsChanPrefix(*chan))
    {
      sendto_one(source_p, form_str(ERR_NOSUCHCHANNEL),
                 me.name, source_p->name, chan);
      continue;
    }

    if (ConfigChannel.disable_local_channels && (*chan == '&'))
    {
      sendto_one(source_p, form_str(ERR_NOSUCHCHANNEL),
                 me.name, source_p->name, chan);
      continue;
    }

    if (strlen(chan) > CHANNELLEN)
    {
      sendto_one(source_p, form_str(ERR_BADCHANNAME),
                 me.name, source_p->name, chan);
      continue;
    }

    if (find_channel_jupe(chan) &&
        (!IsOper(source_p) && ConfigFileEntry.oper_pass_jupes))
    {
      sendto_one(source_p, form_str(ERR_BADCHANNAME),
                 me.name, source_p->name, chan);
      sendto_realops_flags(UMODE_JUPES, L_ALL,
                           "User %s %s@%s joining cjuped channel %s",
                           source_p->name, source_p->username, source_p->host, chan);

      if (ConfigChannel.cjupe_dline)
      {
	struct ConfItem *conf;
	struct AccessItem *aconf;

	conf = make_conf_item(DLINE_TYPE);
	aconf = (struct AccessItem *)map_to_conf(conf);
	DupString(aconf->host, source_p->localClient->sockhost);
	DupString(aconf->reason, "Joining cjuped channel");
	DupString(aconf->oper_reason, "Auto D-Line");
	aconf->hold = CurrentTime + ConfigFileEntry.cjupe_dline_time;
	add_tdline(aconf);
	sendto_realops_flags(UMODE_ALL, L_ALL, "AUTODLINE: %d min. D-Line %s",
                             ConfigFileEntry.cjupe_dline_time/60,
                             source_p->localClient->sockhost);
	ilog(TRACE, "AUTODLINE: %d min. D-Line %s for joining cjuped channel",
        ConfigFileEntry.cjupe_dline_time/60, source_p->localClient->sockhost);
        return(CLIENT_EXITED);
      }
      else
        continue;
    }

    if ((dlink_list_length(&source_p->user->channel) >= ConfigChannel.max_chans_per_user) &&
        (!IsOper(source_p) || (dlink_list_length(&source_p->user->channel) >=
                               ConfigChannel.max_chans_per_user * 3)))
    {
      if (!error_reported++)
        sendto_one(source_p, form_str(ERR_TOOMANYCHANNELS),
                   me.name, source_p->name, chan);
      continue;
    }

    if ((chptr = hash_find_channel(chan)) != NULL)
    {
      if (splitmode && !IsOper(source_p) && (*chan != '&') &&
          ConfigChannel.no_join_on_split)
      {
        sendto_one(source_p, form_str(ERR_UNAVAILRESOURCE),
                   me.name, source_p->name, chan);
        continue;
      }

      if (dlink_list_length(&chptr->members) == 0)
        flags = CHFL_CHANOP;
      else
        flags = 0;
    }
    else
    {
      if (splitmode && !IsOper(source_p) && (*chan != '&') &&
          (ConfigChannel.no_create_on_split || ConfigChannel.no_join_on_split))
      {
        sendto_one(source_p, form_str(ERR_UNAVAILRESOURCE),
                   me.name, source_p->name, chan);
        continue;
      }

      flags = CHFL_CHANOP;

      if ((chptr = get_or_create_channel(chan, NULL)) == NULL)
      {
        sendto_one(source_p, form_str(ERR_UNAVAILRESOURCE),
                   me.name, source_p->name, chan);
        continue;
      }
    }

    if (is_target(chptr))
      continue;

    targets[ntargets].chptr = chptr;
    targets[ntargets].key = key;
    targets[ntargets++].flags = flags;
  }
  return ((ntargets) ? 1 : 0);
}

/* inputs       - channel to check
 * output       - YES if duplicate pointer in table, NO if not.
 *                note, this does the canonize using pointers
 */
static int
is_target(struct Channel *chptr)
{
  int i;

  for (i = ntargets-1; i >=0; i--)
  {
    if (targets[i].chptr == chptr)
      return i;
  }

  return 0;
}

/* parv[0] = sender prefix
 * parv[1] = channel
 * parv[2] = reason
 */
static void
m_part(struct Client *client_p, struct Client *source_p,
       int parc, char *parv[])
{
  char *p, *name, reason[TOPICLEN + 1];

  if (IsServer(source_p))
    return;

  reason[0] = '\0';

  if (parc > 2)
    strlcpy(reason, parv[2], sizeof(reason));

  name = strtoken(&p, parv[1], ",");

  if (MyClient(source_p) && !IsFloodDone(source_p))
    flood_endgrace(source_p);

  while (name)
  {
    part_one_client(client_p, source_p, name, reason);
    name = strtoken(&p, NULL, ",");
  }
}

/* inputs - pointer to server
 * - pointer to source client to remove
 * - char pointer of name of channel to remove from
 * side effects- remove ONE client given the channel name
 */
static void
part_one_client(struct Client *client_p, struct Client *source_p,
                char *name, char *reason)
{
  struct Channel *chptr;
  struct Membership *ms;

  if ((chptr = hash_find_channel(name)) == NULL)
  {
    sendto_one(source_p, form_str(ERR_NOSUCHCHANNEL),
    	       me.name, source_p->name, name);
    return;
  }

  if ((ms = find_channel_link(source_p, chptr)) == NULL)
  {
    sendto_one(source_p, form_str(ERR_NOTONCHANNEL),
               me.name, source_p->name, name);
    return;
  }

  /* Remove user from the old channel (if any)
   * only allow /part reasons in -m chans
   */
   if (reason[0] && (!MyConnect(source_p) ||
     ((can_send_part(ms, chptr, source_p, reason) > 0 &&
     (source_p->firsttime + ConfigFileEntry.anti_spam_exit_message_time)
      < CurrentTime))))
  {
    sendto_server(client_p, chptr, ":%s PART %s :%s",
		  source_p->name, chptr->name, reason);
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s!%s@%s PART %s :%s",
                         source_p->name, source_p->username,
                         source_p->host, chptr->name, reason);
  }
  else
  {
    sendto_server(client_p, chptr, ":%s PART %s", source_p->name, chptr->name);
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s!%s@%s PART %s",
                         source_p->name, source_p->username,
                         source_p->host, chptr->name);
  }
  remove_user_from_channel(ms);
}

/* inputs       - pointer to mode to setup
 *              - pointer to old mode
 */
static const struct mode_letter
{
  unsigned int mode;
  unsigned char letter;
} flags[] = {
  { MODE_NOCTCP,     'C' },
  { MODE_NOCONTROL,  'c' },
  { MODE_INVITEONLY, 'i' },
  { MODE_MODERATED,  'm' },
  { MODE_NOPRIVMSGS, 'n' },
  { MODE_PRIVATE,    'p' },
  { MODE_SECRET,     's' },
  { MODE_TOPICLIMIT, 't' },
  { 0, '\0' }
};

static void
set_final_mode(struct Mode *mode, struct Mode *oldmode)
{
  char *pbuf = parabuf;
  int what = 0, len, i;

  for (i = 0; flags[i].letter; i++)
  {
    if ((flags[i].mode & mode->mode) &&
       !(flags[i].mode & oldmode->mode))
    {
      if (what != 1)
      {
         *mbuf++ = '+';
         what = 1;
      }
      *mbuf++ = flags[i].letter;
    }
  }
  for (i = 0; flags[i].letter; i++)
  {
    if ((flags[i].mode & oldmode->mode) &&
       !(flags[i].mode & mode->mode))
    {
      if (what != -1)
      {
        *mbuf++ = '-';
        what = -1;
      }
      *mbuf++ = flags[i].letter;
    }
  }
  if (oldmode->limit != 0 && mode->limit == 0)
  {
    if (what != -1)
    {
      *mbuf++ = '-';
      what = -1;
    }
    *mbuf++ = 'l';
  }
  if (oldmode->key[0] && !mode->key[0])
  {
    if (what != -1)
    {
      *mbuf++ = '-';
      what = -1;
    }
    *mbuf++ = 'k';
    len = ircsprintf(pbuf,"%s ", oldmode->key);
    pbuf += len;
    pargs++;
  }
  if (mode->limit != 0 && oldmode->limit != mode->limit)
  {
    if (what != 1)
    {
      *mbuf++ = '+';
      what = 1;
    }
    *mbuf++ = 'l';
    len = ircsprintf(pbuf, "%d ", mode->limit);
    pbuf += len;
    pargs++;
  }
  if (mode->key[0] && strcmp(oldmode->key, mode->key))
  {
    if (what != 1)
    {
      *mbuf++ = '+';
      what = 1;
    }
    *mbuf++ = 'k';
    len = ircsprintf(pbuf, "%s ", mode->key);
    pbuf += len;
    pargs++;
  }
  *mbuf = '\0';
}

/* inputs       - pointer to channel to remove modes from
 *              - client pointer
 * side effects - Go through the local members, remove all their
 *                chanop modes etc., this side lost the TS.
 */
static void
remove_our_modes(struct Channel *chptr, struct Client *source_p)
{
  remove_a_mode(chptr, source_p, CHFL_CHANOP, 'o');
  remove_a_mode(chptr, source_p, CHFL_HALFOP, 'h');
  remove_a_mode(chptr, source_p, CHFL_VOICE, 'v');
}

/* remove_a_mode()
 * side effects - remove ONE mode from a channel
 */
static void
remove_a_mode(struct Channel *chptr, struct Client *source_p,
              int mask, char flag)
{
  dlink_node *ptr;
  struct Membership *ms;
  char lmodebuf[BUFSIZE], *lpara[MAXSERVERMODES];
  int count = 0, lcount;

  mbuf = lmodebuf;
  *mbuf++ = '-';

  for(lcount = 0; lcount < MAXSERVERMODES; lcount++)
    lpara[lcount] = "";
  sendbuf[0] = '\0';

  DLINK_FOREACH(ptr, chptr->members.head)
  {
    ms = ptr->data;
    if ((ms->flags & mask) == 0)
      continue;

    if (IsService(ms->client_p))
      continue;

    ms->flags &= ~mask;

    lpara[count++] = ms->client_p->name;

    *mbuf++ = flag;

    if (count >= MAXSERVERMODES)
    {
      *mbuf = '\0';
      for(lcount = 0; lcount < MAXSERVERMODES; lcount++)
      {
        if (*lpara[lcount] == '\0')
          break;

        strlcat(sendbuf, " ", sizeof(sendbuf));
        strlcat(sendbuf, lpara[lcount], sizeof(sendbuf));
        lpara[lcount] = "";
      }
      sendto_channel_local(ALL_MEMBERS, chptr,
                           ":%s MODE %s %s%s",
                           (IsHidden(source_p) ||
                           ConfigServerHide.hide_servers) ?
                           me.name : source_p->name,
                           chptr->name, lmodebuf, sendbuf);
      mbuf = lmodebuf;
      *mbuf++ = '-';
      count = 0;
      sendbuf[0] = '\0';
    }
  }

  if (count != 0)
  {
    *mbuf = '\0';
    for(lcount = 0; lcount < MAXSERVERMODES; lcount++)
    {
      if (*lpara[lcount] == '\0')
        break;

      strlcat(sendbuf, " ", sizeof(sendbuf));
      strlcat(sendbuf, lpara[lcount], sizeof(sendbuf));
    }
    sendto_channel_local(ALL_MEMBERS, chptr,
                         ":%s MODE %s %s%s",
                        (IsHidden(source_p) || ConfigServerHide.hide_servers) ?
                         me.name : source_p->name,
                         chptr->name, lmodebuf, sendbuf);
  }
}

/* parv[0] - sender
 * parv[1] - TS
 * parv[2] - channel
 * parv[3] - modes + n arguments (key and/or limit)
 * parv[4+n] - flags+nick list (all in one parameter)
 * process a SJOIN, taking the TS's into account to either ignore the
 * incoming modes or undo the existing ones or merge them, and JOIN
 * all the specified users while sending JOIN/MODEs to local clients
 */
static void
ms_sjoin(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  struct Channel *chptr;
  struct Client *target_p;
  time_t newts, oldts, tstosend;
  static struct Mode mode, *oldmode;
  int args = 0, keep_our_modes = 1, keep_new_modes = 1, isnew, i, people = 0, num_prefix = 0,
      buflen = 0, lcount, slen;
  unsigned int fl;
  char *s, *p, *sptr, *nick_ptr;
  static char nick_buf[2*BUFSIZE], *para[MAXSERVERMODES];
  dlink_node *m;
  const char *servername = (ConfigServerHide.hide_servers || IsHidden(source_p)) ?
                            me.name : source_p->name;

  if (IsClient(source_p))
    return;

  /* SJOIN's for local channels can't happen. */
  if (*parv[2] != '#')
    return;

  if (!check_channel_name(parv[2]))
    return;

  modebuf[0] = '\0';
  mbuf = modebuf;
  pargs = 0;
  newts = atol(parv[1]);
  mode.mode = mode.limit = 0;
  mode.key[0] = '\0';
  s = parv[3];

  while (*s)
  {
    switch (*(s++))
    {
      case 'C':
        mode.mode |= MODE_NOCTCP;
        break;
      case 'c':
        mode.mode |= MODE_NOCONTROL;
        break;
      case 'i':
        mode.mode |= MODE_INVITEONLY;
        break;
      case 'k':
        strlcpy(mode.key, parv[4 + args], sizeof(mode.key));
        args++;
        if (parc < 5+args)
          return;
        break;
      case 'l':
        mode.limit = atoi(parv[4 + args]);
        args++;
        if (parc < 5+args)
          return;
        break;
      case 'm':
        mode.mode |= MODE_MODERATED;
        break;
      case 'n':
        mode.mode |= MODE_NOPRIVMSGS;
        break;
      case 'p':
        mode.mode |= MODE_PRIVATE;
        break;
      case 's':
        mode.mode |= MODE_SECRET;
        break;
      case 't':
        mode.mode |= MODE_TOPICLIMIT;
        break;
    }
  }
  parabuf[0] = '\0';

  if ((chptr = get_or_create_channel(parv[2], &isnew)) == NULL)
    return; /* channel name too long? */

  oldts = chptr->channelts;
  oldmode = &chptr->mode;

  if (!newts && !isnew && oldts)
  {
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s NOTICE %s :*** TS for %s changed from %lu to 0",
                         me.name, chptr->name, chptr->name, (unsigned long)oldts);
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "Server %s changing TS on %s from %lu to 0",
                         source_p->name, chptr->name, (unsigned long)oldts);
  }
  if (isnew)
    chptr->channelts = tstosend = newts;
  else if (newts == 0 || oldts == 0)
    chptr->channelts = tstosend = 0;
  else if (newts == oldts)
    tstosend = oldts;
  else if (newts < oldts)
  {
    keep_our_modes = 0;
    chptr->channelts = tstosend = newts;
  }
  else
  {
    keep_new_modes = 0;
    tstosend = oldts;
  }

  if (!keep_new_modes)
    mode = *oldmode;
  else if (keep_our_modes)
  {
    mode.mode |= oldmode->mode;
    if (oldmode->limit > mode.limit)
      mode.limit = oldmode->limit;
    if (strcmp(mode.key, oldmode->key) < 0)
      strcpy(mode.key, oldmode->key);
  }
  set_final_mode(&mode, oldmode);
  chptr->mode = mode;

  /* Lost the TS, other side wins, so remove modes on this side */
  if (!keep_our_modes)
  {
    remove_our_modes(chptr, source_p);
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s NOTICE %s :*** TS for %s changed from %lu to %lu",
                         me.name, chptr->name, chptr->name,
                         (unsigned long)oldts, (unsigned long)newts);
  }

  if (*modebuf != '\0')
  {
    /* This _SHOULD_ be to ALL_MEMBERS
     * It contains only +cCimnpstlk, etc */
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s MODE %s %s %s",
                         servername, chptr->name, modebuf, parabuf);
  }

  modebuf[0] = parabuf[0] = '\0';
  if (parv[3][0] != '0' && keep_new_modes)
  {
    channel_modes(chptr, source_p, modebuf, parabuf);
  }
  else
  {
    modebuf[0] = '0';
    modebuf[1] = '\0';
  }

  buflen = ircsprintf(nick_buf, ":%s SJOIN %lu %s %s %s:",
                      source_p->name, (unsigned long)tstosend,
                      chptr->name, modebuf, parabuf);
  nick_ptr = nick_buf + buflen;
  /* check we can fit a nick on the end, as well as \r\n and a prefix "
   * @+".
   */
  if (buflen >= (BUFSIZE - 2 - NICKLEN))
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "Long SJOIN from server: %s(via %s) (ignored)",
                         source_p->name, client_p->name);
    return;
  }

  mbuf = modebuf;
  for(lcount = 0; lcount < MAXSERVERMODES; lcount++)
    para[lcount] = "";
  sendbuf[0] = '\0';
  pargs = 0;

  *mbuf++ = '+';

  s = parv[args + 4];

  /* remove any leading spaces */
  while (*s == ' ')
    s++;

  /* if theres a space, theres going to be more than one nick, change the
   * first space to \0, so s is just the first nick, and point p to the
   * second nick
   */
  if ((p = strchr(s, ' ')) != NULL)
    *p++ = '\0';

  while (s != NULL)
  {
    fl = 0;
    num_prefix = 0;

    for (i = 0; i < 3; i++)
    {
      if (*s == '@')
      {
        fl |= CHFL_CHANOP;
        s++;
      }
      else if (*s == '%')
      {
        fl |= CHFL_HALFOP;
        s++;
      }
      else if (*s == '+')
      {
        fl |= CHFL_VOICE;
        s++;
      }
    }

    target_p = find_person(s);

    /* if the client doesnt exist, or if its fake direction/server, skip. */
    if (target_p == NULL || target_p->from != client_p || !IsPerson(target_p))
      goto nextnick;

    if (keep_new_modes)
    {
      if (fl & CHFL_CHANOP)
        *nick_ptr++ = '@';
      if (fl & CHFL_HALFOP)
        *nick_ptr++ = '%';
      if (fl & CHFL_VOICE)
        *nick_ptr++ = '+';
    }

    /* copy the nick to the two buffers */
    nick_ptr += ircsprintf(nick_ptr, "%s ", target_p->name);

    if (!keep_new_modes)
    {
      if ((fl & (CHFL_CHANOP|CHFL_HALFOP)) && !IsService(target_p))
        fl = CHFL_DEOPPED;
      else
        fl = 0;
    }
    people++;

    if (!IsMember(target_p, chptr))
    {
      add_user_to_channel(chptr, target_p, fl);
      sendto_channel_local(ALL_MEMBERS, chptr, ":%s!%s@%s JOIN :%s",
                           target_p->name, target_p->username,
                           target_p->host, chptr->name);
    }

    if (fl & CHFL_CHANOP)
    {
      *mbuf++ = 'o';
      para[pargs++] = target_p->name;

      if (pargs >= MAXSERVERMODES)
      {
        sptr = sendbuf;
        *mbuf = '\0';
        for(lcount = 0; lcount < MAXSERVERMODES; lcount++)
        {
          slen = ircsprintf(sptr, " %s", para[lcount]);
          sptr += slen;
        }
        sendto_channel_local(ALL_MEMBERS, chptr, ":%s MODE %s %s%s",
                             servername, chptr->name, modebuf, sendbuf);
        mbuf = modebuf;
        *mbuf++ = '+';

        sendbuf[0] = '\0';
        pargs = 0;
      }
    }
    if (fl & CHFL_HALFOP)
    {
      *mbuf++ = 'h';
      para[pargs++] = target_p->name;

      if (pargs >= MAXSERVERMODES)
      {
        sptr = sendbuf;
        *mbuf = '\0';
        for(lcount = 0; lcount < MAXSERVERMODES; lcount++)
        {
          slen = ircsprintf(sptr, " %s", para[lcount]);
          sptr += slen;
        }
        sendto_channel_local(ALL_MEMBERS, chptr, ":%s MODE %s %s%s",
                             servername, chptr->name, modebuf, sendbuf);
        mbuf = modebuf;
        *mbuf++ = '+';

        sendbuf[0] = '\0';
        pargs = 0;
      }
    }
    if (fl & CHFL_VOICE)
    {
      *mbuf++ = 'v';
      para[pargs++] = target_p->name;

      if (pargs >= MAXSERVERMODES)
      {
        *mbuf = '\0';
        for(lcount = 0; lcount < MAXSERVERMODES; lcount++)
        {
          if (*para[lcount] == '\0')
            break;

          strlcat(sendbuf, " ", sizeof(sendbuf));
          strlcat(sendbuf, para[lcount], sizeof(sendbuf));
          para[lcount] = "";
        }
        sendto_channel_local(ALL_MEMBERS, chptr, ":%s MODE %s %s%s",
                             servername, chptr->name, modebuf, sendbuf);
        mbuf = modebuf;
        *mbuf++ = '+';

        sendbuf[0] = '\0';
        pargs = 0;
      }
    }

nextnick:
    /* p points to the next nick */
    s = p;

    /* if there was a trailing space and p was pointing to it, then we
     * need to exit.. this has the side effect of breaking double spaces
     * in an sjoin.. but that shouldnt happen anyway
     */
    if (s && (*s == '\0'))
      s = p = NULL;

    /* if p was NULL due to no spaces, s wont exist due to the above, so
     * we cant check it for spaces.. if there are no spaces, then when
     * we next get here, s will be NULL
     */
    if (s != NULL && ((p = strchr(s, ' ')) != NULL))
      *p++ = '\0';
  }

  *mbuf = '\0';
  *(nick_ptr - 1) = '\0';

  if (pargs != 0)
  {
    sptr = sendbuf;
    for(lcount = 0; lcount < pargs; lcount++)
    {
      slen = ircsprintf(sptr, " %s", para[lcount]);
      sptr += slen;
    }
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s MODE %s %s%s",
                         servername, chptr->name, modebuf, sendbuf);
  }
  if ((people == 0) && isnew)
  {
    destroy_channel(chptr);
    return;
  }

  if (parv[4 + args][0] == '\0')
    return;

  /* relay the SJOIN to other servers */
  DLINK_FOREACH(m, serv_list.head)
  {
    target_p = m->data;

    if (target_p == client_p)
      continue;

    sendto_one(target_p, "%s", nick_buf);
  }
}

static void
m_mode(struct Client *client_p, struct Client *source_p,
       int parc, char *parv[])
{
  struct Channel *chptr = NULL;
  struct Membership *member;
  static char mmodebuf[MODEBUFLEN], mparabuf[MODEBUFLEN];

  if (parv[1][0] == '\0')
  {
    sendto_one(source_p, form_str(ERR_NEEDMOREPARAMS),
               me.name, source_p->name, "MODE");
    return;
  }

  /* Now, try to find the channel in question */
  if (!IsChanPrefix(parv[1][0]))
  {
    set_user_mode(client_p, source_p, parc, parv);
    return;
  }

  if (!check_channel_name(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_BADCHANNAME),
               me.name, source_p->name, parv[1]);
    return;
  }

  chptr = hash_find_channel(parv[1]);

  if (chptr == NULL)
  {
    sendto_one(source_p, form_str(ERR_NOSUCHCHANNEL),
               me.name, source_p->name, parv[1]);
    return;
  }
   
  /* Now known the channel exists */
  if (parc < 3)
  {
    channel_modes(chptr, source_p, mmodebuf, mparabuf);
    sendto_one(source_p, form_str(RPL_CHANNELMODEIS),
               me.name, source_p->name, chptr->name, mmodebuf, mparabuf);
    sendto_one(source_p, form_str(RPL_CREATIONTIME),
               me.name, source_p->name, chptr->name, chptr->channelts);
  }
  else if (IsServer(source_p))
  {
    set_channel_mode(client_p, source_p, chptr, NULL, parc - 2, parv + 2,
                     chptr->name);
  }
  else
  {
    member = find_channel_link(source_p, chptr);
  
    if (!has_member_flags(member, CHFL_DEOPPED))
    {
      if (MyClient(source_p) && !IsFloodDone(source_p))
      {
        if (!((parc == 3) && (parv[2][0] == 'b') && (parv[2][1] == '\0')))
          flood_endgrace(source_p); 
      }
      set_channel_mode(client_p, source_p, chptr, member, parc - 2, parv + 2,
                       chptr->name);
    }
  }
}
   
/* parv[0] - sender
 * parv[1] - channel
 */
static void
mo_omode(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  int status = 0;
  static char omodebuf[MODEBUFLEN], oparabuf[MODEBUFLEN];
  struct Channel *chptr = NULL;
      
  if (!IsChanPrefix(parv[1][0]) || !check_channel_name(parv[1]))
  {
    sendto_one(client_p, form_str(ERR_BADCHANNAME),
               me.name, client_p->name, parv[1]);
    return;
  }
      
  if ((chptr = hash_find_channel(parv[1])) == NULL)
  {
    sendto_one(client_p, form_str(ERR_NOSUCHCHANNEL),
               me.name, client_p->name, parv[1]);
    return;
  }
  status = client_p->status;
  client_p->status = STAT_SERVER;
  channel_modes(chptr, client_p, omodebuf, oparabuf);
  client_p->status = status;

  sendto_one(client_p, form_str(RPL_CHANNELMODEIS),
             me.name, source_p->name, chptr->name, omodebuf, oparabuf);
  sendto_one(client_p, form_str(RPL_CREATIONTIME),
             me.name, source_p->name, chptr->name, chptr->channelts);
  sendto_realops_flags(UMODE_SPY, L_ALL, "OMODE '%s' by %s (%s@%s)",
                       parv[1], source_p->name, source_p->username, source_p->host);
}

/* parv[0] = sender prefix
 * parv[1] = channel
 * parv[2] = client to kick
 * parv[3] = kick comment
 */
static void
m_kick(struct Client *client_p, struct Client *source_p,
       int parc, char *parv[])
{
  struct Client *who;  
  struct Channel *chptr;
  int chasing = 0;
  char *comment, *name, *user, def_reason[] = "No Reason", *p = NULL;
  struct Membership *ms = NULL, *target_p;

  if (!IsClient(source_p))
    return;

  if (MyClient(source_p) && !IsFloodDone(source_p))
    flood_endgrace(source_p);

  comment = (EmptyString(parv[3])) ? def_reason : parv[3];
  
  if (strlen(comment) > (size_t) TOPICLEN)
    comment[TOPICLEN] = '\0';

  name = parv[1];
  while (*name == ',')
    name++;
  if ((p = strchr(name,',')) != NULL)
    *p = '\0';
  if (!*name)
    return;

  if ((chptr = hash_find_channel(name)) == NULL)
  {
    sendto_one(source_p, form_str(ERR_NOSUCHCHANNEL),
               me.name, source_p->name, name);  
    return;
  }
  
  if ((ms = find_channel_link(source_p, chptr)) == NULL)
  {
    if (MyConnect(source_p))
    {
      sendto_one(source_p, form_str(ERR_NOTONCHANNEL),
                 me.name, source_p->name, name);   
      return;
    }
  }
  if (!has_member_flags(ms, CHFL_CHANOP|CHFL_HALFOP))
  {
    if (MyConnect(source_p) || (chptr->channelts == 0)) 
    {
      sendto_one(source_p, form_str(ERR_CHANOPRIVSNEEDED),
                 me.name, source_p->name, name);
      return;
    }
  }
  user = parv[2];
  
  while (*user == ',')
    user++;
    
  if ((p = strchr(user,',')) != NULL)
    *p = '\0';
                 
  if (!*user)
    return;
   
  if ((who = find_chasing(source_p, user, &chasing)) == NULL)
    return;
    
  if ((target_p = find_channel_link(who, chptr)) != NULL)
  {
    /* half ops cannot kick other halfops on private channels */
    if (has_member_flags(ms, CHFL_HALFOP) && !has_member_flags(ms, CHFL_CHANOP))
    {
      if (((chptr->mode.mode & MODE_PRIVATE) && has_member_flags(target_p, CHFL_CHANOP|CHFL_HALFOP)) ||
            has_member_flags(target_p, CHFL_CHANOP))
      {
        sendto_one(source_p, form_str(ERR_CHANOPRIVSNEEDED),
                   me.name, source_p->name, name);
        return;
      }
    }
    if (IsService(who))
    {
      sendto_one(source_p, form_str(ERR_ISSERVICE),
                 me.name, source_p->name, who->name, "kick");
      return;
    }
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s!%s@%s KICK %s %s :%s",
                         source_p->name, source_p->username,
                         source_p->host, chptr->name, who->name, comment);
    sendto_server(client_p, chptr, ":%s KICK %s %s :%s", source_p->name, chptr->name,
                  who->name, comment);
    remove_user_from_channel(target_p);
  }
  else
    sendto_one(source_p, form_str(ERR_USERNOTINCHANNEL),
               me.name, source_p->name, user, name);
}  
