/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * m_nick.c: Sets a users nick.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_nick.c,v 1.109.2.3 2004/07/18 21:43:54 musirc Exp $
 */

#include "handlers.h"
#include "client.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "numeric.h"
#include "config.h"
#include "user.h"
#include "hash.h"
#include "whowas.h"
#include "server.h"
#include "send.h"
#include "jupe.h"
#include "modules.h"
#include "packet.h"

static void m_nick(struct Client *, struct Client *, int, char **);
static void mr_nick(struct Client *, struct Client *, int, char **);
static void ms_nick(struct Client *, struct Client *, int, char **);
static int nick_from_server(struct Client *, struct Client *, int, char **,
                            time_t, char *, char *);
static int check_clean_nick(struct Client *, struct Client *, char *, char *);
static int check_clean_user(struct Client *, char *, char *);
static int check_clean_host(struct Client *, char *, char *);
static int clean_nick_name(char *);
static void perform_nick_collides(struct Client *, struct Client *, struct Client *,
				  int, char **, time_t, char *, char *);
                            
struct Message nick_msgtab = {
  "NICK", 1, MFLG_SLOW,
  {mr_nick, m_nick, ms_nick, m_nick}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&nick_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&nick_msgtab);
}

const char *_version = "$Revision: 1.109.2.3 $";
#endif

/* parv[0] = sender prefix
 * parv[1] = nickname
 */
static void
mr_nick(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  struct Client *target_p;
  char nick[NICKLEN], *s;
   
  if (parc < 2 || EmptyString(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_NONICKNAMEGIVEN),
        me.name, EmptyString(parv[0]) ? "*" : parv[0]);
    return;
  }

  /* Terminate the nick at the first ~ */
  if ((s = strchr(parv[1], '~')) != NULL)
    *s = '\0';
                               
  /* copy the nick and terminate it */
  strlcpy(nick, parv[1], sizeof(nick));

  /* check the nickname is ok */
  if(!clean_nick_name(nick))
  {
    sendto_one(source_p, form_str(ERR_ERRONEUSNICKNAME),
	       me.name, EmptyString(parv[0]) ? "*" : parv[0], parv[1]);
    return;
  }

  /* check if the nick is njuped */
  if(find_matching_name_conf(NJUPE_TYPE, nick, NULL, NULL, 0))
  {
    sendto_one(source_p, form_str(ERR_ERRONEUSNICKNAME),
	       me.name, EmptyString(parv[0]) ? "*" : parv[0], nick);
    sendto_realops_flags(UMODE_JUPES, L_ALL,
                         "User %s@%s using njuped nick %s",
                          source_p->username, source_p->host, nick);
    return;
  }

  if ((target_p = find_client(nick)) == NULL)
  {
    set_initial_nick(client_p, source_p, nick);
    return;
  }
  else if(source_p == target_p)
  {
    strcpy(source_p->name, nick);
    return;
  }
  else
  {
    sendto_one(source_p, form_str(ERR_NICKNAMEINUSE), me.name, "*", nick);
  }
}

/* parv[0] = sender prefix
 * parv[1] = nickname
 */
static void
m_nick(struct Client *client_p, struct Client *source_p,
       int parc, char *parv[])
{
  char nick[NICKLEN];
  struct Client *target_p;

  if (parc < 2 || EmptyString(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_NONICKNAMEGIVEN),
               me.name, parv[0]);
    return;
  }

  /* mark end of grace period, to prevent nickflooding */
  if (!IsFloodDone(source_p))
    flood_endgrace(source_p);

  /* terminate nick to NICKLEN */
  strlcpy(nick, parv[1], sizeof(nick));

  /* check the nickname is ok */
  if (!clean_nick_name(nick))
  {
    sendto_one(source_p, form_str(ERR_ERRONEUSNICKNAME),
               me.name, source_p->name, nick);
    return;
  }

  if (find_matching_name_conf(NJUPE_TYPE, nick, NULL, NULL, 0) &&
     !(IsOper(source_p) && ConfigFileEntry.oper_pass_jupes) && !IsService(source_p))
  {
    sendto_one(source_p, form_str(ERR_ERRONEUSNICKNAME),
               me.name, source_p->name, nick);
    sendto_realops_flags(UMODE_JUPES, L_ALL,
                         "User %s@%s using njuped nick %s",
                          source_p->username, source_p->host, nick);
    return;
  }

  if ((target_p = find_client(nick)))
  {
    /* If(target_p == source_p) the client is changing nicks between
     * equivalent nicknames ie: [nick] -> {nick}
     */
    if(target_p == source_p)
    {
      /* check the nick isnt exactly the same */
      if(strcmp(target_p->name, nick))
      {
        change_local_nick(client_p, source_p, nick);
        return;
      }
      else
      {
        /* client is doing :old NICK old
         * ignore it..
         */
        return;
      }
    }

    /* if the client that has the nick isnt registered yet (nick but no
     * user) then drop the unregged client
     */
    if (IsUnknown(target_p))
    {
      exit_client(NULL, target_p, &me, "Nick collision");
      change_local_nick(client_p, source_p, nick);
      return;
    }
    else
    {
      sendto_one(source_p, form_str(ERR_NICKNAMEINUSE), me.name,
                 source_p->name, nick);
      return;
    }
  }
  else
  {
    change_local_nick(client_p, source_p, nick);
    return;
  }
}

/* server -> server nick change
 *    parv[0] = sender prefix
 *    parv[1] = nickname
 *    parv[2] = TS when nick change
 * server introducing new nick
 *    parv[0] = sender prefix
 *    parv[1] = nickname
 *    parv[2] = hop count
 *    parv[3] = TS
 *    parv[4] = umode
 *    parv[5] = username
 *    parv[6] = hostname
 *    parv[7] = server
 *    parv[8] = ircname
 */
static void
ms_nick(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  struct Client *target_p;
  char nick[NICKLEN], *nnick = parv[1], *nhop = parv[2], *nts = parv[3],
       *nusername = parv[5], *nhost = parv[6], *nserver = parv[7], ngecos[REALLEN];
  time_t newts = 0;

  if (parc < 2 || EmptyString(nnick))
    return;

  strlcpy(nick, nnick, sizeof(nick));

  if (parc == 9)
  {
    strlcpy(ngecos, parv[8], sizeof(ngecos));
    if (find_server(nserver) == NULL)
    {
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Invalid server %s from %s for NICK %s",
                           nserver, source_p->name, nick);
      sendto_one(client_p, ":%s KILL %s :%s (Server doesn't exist!)",
                 me.name, nick, me.name);
      return;
    }
    if (check_clean_nick(client_p, source_p, nick, nnick) ||
        check_clean_user(client_p, nick, nusername) ||
        check_clean_host(client_p, nick, nhost))
      return;

    if (strlen(parv[8]) > REALLEN)
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Long realname for %s from %s",
                           nnick, nserver);

    if (IsServer(source_p))
      newts = atol(nts);
  }
  else if (parc == 3)
  {
    if (IsServer(source_p))
      return;

    if (check_clean_nick(client_p, source_p, nick, nnick))
      return;

    newts = atol(nhop);
  }

  /* if the nick doesnt exist, allow it and process like normal */
  if (!(target_p = find_client(nick)))
  {
    nick_from_server(client_p, source_p, parc, parv, newts, nick, ngecos);
    return;
  }

  /* we're not living in the past anymore, an unknown client is local only. */
  if (IsUnknown(target_p))
  {
    exit_client(NULL, target_p, &me, "Overridden");
    nick_from_server(client_p, source_p, parc, parv, newts, nick, ngecos);
    return;
  }

  if (target_p == source_p)
  {
    if (strcmp(target_p->name, nick))
    {
      /* client changing case of nick */
      nick_from_server(client_p, source_p, parc, parv, newts, nick, ngecos);
      return;
    }
    else
      /* client not changing nicks at all */
      return;
  }
  perform_nick_collides(source_p, client_p, target_p, parc, parv, newts, nick, ngecos);
}

/* input        - pointer to source
 *              - nickname
 *              - truncated nickname
 *              - origin of client
 *              - pointer to server nick is coming from
 * side effects - if nickname is erroneous, or a different length to
 *                truncated nickname, return 1
 */
static int
check_clean_nick(struct Client *client_p, struct Client *source_p, char *nick, char *newnick)
{
  if (!clean_nick_name(nick) || strcmp(nick, newnick))
  {
    sendto_one(client_p, ":%s KILL %s :%s (Bad Nickname)",
               me.name, newnick, me.name);

    if (source_p != client_p)
    {
      SetKilled(source_p);
      exit_client(client_p, source_p, &me, "Bad Nickname");
    }
    return (1);
  }
  return (0);
}

/* input        - pointer to client sending data
 *              - nickname
 *              - username to check
 *              - origin of NICK
 * side effects - if username is erroneous, return 1
 */
static int
check_clean_user(struct Client *client_p, char *nick, char *user)
{
  if (strlen(user) > USERLEN)
  {
    sendto_one(client_p, ":%s KILL %s :%s (Bad Username)",
               me.name, nick, me.name);
    return (1);
  }
  return (0);
}

/* input        - pointer to client sending us data
 *              - nickname
 *              - hostname to check
 *              - source name
 * side effects - if hostname is erroneous, return 1
 */
static int
check_clean_host(struct Client *client_p, char *nick, char *host)
{
  if (strlen(host) > HOSTLEN)
  {
    sendto_one(client_p, ":%s KILL %s :%s (Bad Hostname)",
               me.name, nick, me.name);
    return (1);
  }
  return (0);
}

/* input        - nickname
 * side effects - walks through the nickname, returning 0 if erroneous
 */
static int
clean_nick_name(char *nick)
{
  if (nick == NULL)
    return (0);

  /* nicks cant start with a digit or - or be 0 length */
  if (*nick == '-' || IsDigit(*nick) || *nick == '\0')
    return (0);

  for(; *nick; nick++)
  {
    if (!IsNickChar(*nick))
      return (0);
  }
  return (1);
}

static int
nick_from_server(struct Client *client_p, struct Client *source_p, int parc,
                 char *parv[], time_t newts, char *nick, char *ngecos)
{
  if (IsServer(source_p))
  {
    /* A server introducing a new client, change source */
    source_p = make_client(client_p);
    dlinkAdd(source_p, &source_p->node, &global_client_list);

    if (parc > 2)
      source_p->hopcount = atoi(parv[2]);
    if (newts)
      source_p->tsinfo = newts;
    else
    {
      newts = source_p->tsinfo = CurrentTime;
      ts_warn("Remote nick %s (%s) introduced without a TS", nick, parv[0]);
    }

    /* copy the nick in place */
    strcpy(source_p->name, nick);
    hash_add_client(source_p);

    if (parc > 8)
    {
      unsigned int flag;
      char *m;
      /* parse usermodes */
      m = &parv[4][1];
      while (*m)
      {
        flag = user_modes_from_c_to_bitmask[(unsigned char)*m];
        if (!(source_p->umodes & UMODE_OPER) && (flag & UMODE_OPER))
          Count.oper++;
        if (!(source_p->umodes & UMODE_SERVICE) && (flag & UMODE_SERVICE))
          Count.service++;

        source_p->umodes |= flag & SEND_UMODES;
        m++;
      }
      return(register_remote_user(client_p, source_p, parv[5], parv[6],
                                  parv[7], ngecos));
    }
  }
  else if(source_p->name[0])
  {
    /* client changing their nick */
    if (irccmp(parv[0], nick))
      source_p->tsinfo = newts ? newts : CurrentTime;

    sendto_common_channels_local(source_p, 1, ":%s!%s@%s NICK :%s",
                                 source_p->name,source_p->username,
                                 source_p->host, nick);

    if (source_p->user != NULL)
    {
      add_history(source_p, 1);
      sendto_server(client_p, NULL, ":%s NICK %s :%lu",
      parv[0], nick, (unsigned long)source_p->tsinfo);
    }
  }

  /* set the new nick name */
  if (source_p->name[0])
    hash_del_client(source_p);

  strcpy(source_p->name, nick);
  hash_add_client(source_p);

  /* remove all accepts pointing to the client */
  del_all_accepts(source_p);

  return(0);
}

static void
perform_nick_collides(struct Client *source_p, struct Client *client_p,
                      struct Client *target_p, int parc, char *parv[],
                      time_t newts, char *nick, char *ngecos)
{
  int sameuser;
  char buffer[TOPICLEN];

  /* server introducing new nick */
  if (IsServer(source_p))
  {
    /* if we dont have a ts, or their TS's are the same, kill both */
    if (!newts || !target_p->tsinfo || (newts == target_p->tsinfo))
    {
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Nick collision on %s(%s <- %s)(both killed)",
                           target_p->name, target_p->from->name,
                           client_p->name);

      sendto_one(target_p, form_str(ERR_NICKCOLLISION),
                 me.name, target_p->name, target_p->name);

      SetKilled(target_p);
      if (!ConfigServerHide.hide_servers)
        ircsprintf(buffer, "Nick collision (%s <- %s)",
		   target_p->from->name, client_p->name);
      else
        ircsprintf(buffer, "Nick collision");
      exit_client(client_p, target_p, &me, buffer);
      return;
    }
    /* the timestamps are different */
    else
    {
      sameuser = (target_p->user) && !irccmp(target_p->username, parv[5])
                 && !irccmp(target_p->host, parv[6]);

      /* if the users are the same (loaded a client on a different server)
       * and the new users ts is older, or the users are different and the
       * new users ts is newer, ignore the new client and let it do the kill
       */
      if ((sameuser && newts < target_p->tsinfo) ||
         (!sameuser && newts > target_p->tsinfo))
      {
        return;
      }
      else
      {
        if(sameuser)
          sendto_realops_flags(UMODE_ALL, L_ALL,
                          "Nick collision on %s(%s <- %s)(older killed)",
                          target_p->name, target_p->from->name,
                          client_p->name);
        else
          sendto_realops_flags(UMODE_ALL, L_ALL,
                          "Nick collision on %s(%s <- %s)(newer killed)",
                          target_p->name, target_p->from->name,
                          client_p->name);

        sendto_one(target_p, form_str(ERR_NICKCOLLISION),
                   me.name, target_p->name, target_p->name);

        SetKilled(target_p);
        if (!ConfigServerHide.hide_servers)
          ircsprintf(buffer, "Nick collision (%s <- %s)",
                     target_p->from->name, client_p->name);
        else
          ircsprintf(buffer, "Nick collision");
        exit_client(client_p, target_p, &me, buffer);

        nick_from_server(client_p, source_p, parc, parv, newts, nick, ngecos);
        return;
      }
    }
  }

  /* its a client changing nick and causing a collide */
  if (!newts || !target_p->tsinfo || (newts == target_p->tsinfo) ||
      !source_p->user)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,
               "Nick change collision from %s to %s(%s <- %s)(both killed)",
               source_p->name, target_p->name, target_p->from->name,
               client_p->name);

    sendto_one(target_p, form_str(ERR_NICKCOLLISION),
               me.name, target_p->name, target_p->name);

    SetKilled(target_p);
    if (!ConfigServerHide.hide_servers)
      ircsprintf(buffer, "Nick change collision (%s <- %s)",
                 target_p->from->name, client_p->name);
    else
      ircsprintf(buffer, "Nick change collision");
    exit_client(NULL, target_p, &me, buffer);
    SetKilled(source_p);
    if (!ConfigServerHide.hide_servers)
      ircsprintf(buffer, "Nick change collision (%s <- %s)",
                 target_p->from->name, client_p->name);
    else
      ircsprintf(buffer, "Nick change collision");
    exit_client(client_p, source_p, &me, buffer);
    return;
  }
  else
  {
    sameuser = !irccmp(target_p->username, source_p->username) &&
               !irccmp(target_p->host, source_p->host);

    if ((sameuser && newts < target_p->tsinfo) ||
        (!sameuser && newts > target_p->tsinfo))
    {
      if(sameuser)
        sendto_realops_flags(UMODE_ALL, L_ALL,
             "Nick change collision from %s to %s(%s <- %s)(older killed)",
             source_p->name, target_p->name, target_p->from->name,
             client_p->name);
      else
        sendto_realops_flags(UMODE_ALL, L_ALL,
             "Nick change collision from %s to %s(%s <- %s)(newer killed)",
             source_p->name, target_p->name, target_p->from->name,
             client_p->name);

      SetKilled(source_p);
      if(sameuser)
      {
        if (!ConfigServerHide.hide_servers)
          ircsprintf(buffer, "Nick collision (%s <- %s) (older killed)",
                     target_p->from->name, client_p->name);
        else
          ircsprintf(buffer, "Nick collision (older killed)");
        exit_client(client_p, source_p, &me, buffer);
      }
      else
      {
        if (!ConfigServerHide.hide_servers)
          ircsprintf(buffer, "Nick collision (%s <- %s) (newer killed)",
                     target_p->from->name, client_p->name);
        else
          ircsprintf(buffer, "Nick collision (newer killed)");
          exit_client(client_p, source_p, &me, buffer);
      }
      return;
   }
   else
   {
     if (sameuser)
       sendto_realops_flags(UMODE_ALL, L_ALL,
                            "Nick collision on %s(%s <- %s)(older killed)",
                            target_p->name, target_p->from->name,
                            client_p->name);
     else
       sendto_realops_flags(UMODE_ALL, L_ALL,
                            "Nick collision on %s(%s <- %s)(newer killed)",
                            target_p->name, target_p->from->name,
                            client_p->name);

     sendto_one(target_p, form_str(ERR_NICKCOLLISION),
                me.name, target_p->name, target_p->name);

     SetKilled(target_p);
     if (!ConfigServerHide.hide_servers)
       ircsprintf(buffer, "Nick collision (%s <- %s)",
                  target_p->from->name, client_p->name);
     else
       ircsprintf(buffer, "Nick collision");
     exit_client(client_p, target_p, &me, buffer);
   }
 }

 /* we should only ever call nick_from_server() here, as
  * this is a client changing nick, not a new client
  */
  nick_from_server(client_p, source_p, parc, parv, newts, nick, ngecos);
}
