/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * m_server.c: Server management.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_server.c,v 1.217.2.18 2004/07/25 06:54:13 musirc Exp $
 */

#include "stdinc.h"
#include "handlers.h"
#include "ircd.h"
#include "numeric.h"
#include "log.h"
#include "whowas.h"
#include "server.h"
#include "send.h"
#include "istring.h"
#include "sprintf.h"
#include "hash.h"
#include "list.h"
#include "restart.h"
#include "modules.h"
#include "config.h"
#include "packet.h"
#include "user.h"
#include "misc.h"
#include "jupe.h"

static char buf[BUFSIZE];

static void mo_die(struct Client*, struct Client*, int, char**);
static void mo_rehash(struct Client*, struct Client*, int, char**);
static void mo_restart(struct Client *, struct Client *, int, char **);
static void m_lusers(struct Client*, struct Client*, int, char**);
static void ms_lusers(struct Client*, struct Client*, int, char**);
static void m_time(struct Client*, struct Client*, int, char**);
static void mo_time(struct Client*, struct Client*, int, char**);
static void ms_svinfo(struct Client*, struct Client*, int, char**);
static void mr_server(struct Client*, struct Client*, int, char **);
static void ms_server(struct Client*, struct Client*, int, char **);
static void ms_squit(struct Client*, struct Client*, int, char**);
static void mo_squit(struct Client*, struct Client*, int, char**);
static void ms_kill(struct Client*, struct Client*, int, char**);
static void mo_kill(struct Client*, struct Client*, int, char**);
static void relay_kill(struct Client *, struct Client *, struct Client *,
                       const char *, const char *);
static void set_server_gecos(struct Client *, char *);
static void mo_sjupe(struct Client *, struct Client *, int, char **);
static void mo_unsjupe(struct Client *, struct Client *, int, char **);
static void ms_unsjupe(struct Client *, struct Client *, int, char **);
static void mr_pass(struct Client*, struct Client*, int, char**);
static void m_version(struct Client*, struct Client*, int, char**);
static void ms_version(struct Client*, struct Client*, int, char**);
static void mo_version(struct Client*, struct Client*, int, char**);
static void mr_capab(struct Client*, struct Client*, int, char**);
static void mo_connect(struct Client *, struct Client *, int, char **);
static void ms_connect(struct Client *, struct Client *, int, char **);
static void m_quit(struct Client *, struct Client *, int, char **);
static void ms_quit(struct Client *, struct Client *, int, char **);
static void m_error(struct Client *, struct Client *, int, char **);
static void ms_error(struct Client *, struct Client *, int, char **);
static int bogus_host(char *);
static int bogus_reason(const char *);

struct Message die_msgtab = {
  "DIE", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_die}
};
struct Message rehash_msgtab = {
  "REHASH", 0, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_rehash}
};
struct Message restart_msgtab = {
  "RESTART", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_restart}
};
struct Message lusers_msgtab = {
  "LUSERS", 0, MFLG_SLOW,
  {m_unregistered, m_lusers, ms_lusers, ms_lusers}
};
struct Message time_msgtab = {
  "TIME", 0, MFLG_SLOW,
  {m_unregistered, m_time, mo_time, mo_time}
};
struct Message svinfo_msgtab = {
  "SVINFO", 4, MFLG_SLOW,
  {m_unregistered, m_ignore, ms_svinfo, m_ignore}
};
struct Message server_msgtab = {
  "SERVER", 4, MFLG_SLOW | MFLG_UNREG,
  {mr_server, m_registered, ms_server, m_registered}
};
struct Message squit_msgtab = {
  "SQUIT", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, ms_squit, mo_squit}
};
struct Message kill_msgtab = {
  "KILL", 3, MFLG_SLOW,
  {m_unregistered, m_not_oper, ms_kill, mo_kill}
};
struct Message sjupe_msgtab = {
  "SJUPE", 3, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_sjupe}
};
struct Message unsjupe_msgtab = {
  "UNSJUPE", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, ms_unsjupe, mo_unsjupe}
};
struct Message pass_msgtab = {
  "PASS", 2, MFLG_SLOW | MFLG_UNREG,
  {mr_pass, m_registered, m_ignore, m_registered}
};
struct Message version_msgtab = {
  "VERSION", 0, MFLG_SLOW,
  {m_unregistered, m_version, ms_version, mo_version}
};
struct Message capab_msgtab = {
  "CAPAB", 0, MFLG_SLOW | MFLG_UNREG,
  {mr_capab, m_ignore, m_ignore, m_ignore}
};
struct Message connect_msgtab = {
  "CONNECT", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, ms_connect, mo_connect}
};
struct Message quit_msgtab = {
  "QUIT", 0, MFLG_SLOW | MFLG_UNREG,
  {m_quit, m_quit, ms_quit, m_quit}
};
struct Message error_msgtab = {
 "ERROR", 1, MFLG_SLOW | MFLG_UNREG,
  {m_error, m_ignore, ms_error, m_ignore}
};

#ifndef STATIC_MODULES
void 
_modinit(void)
{
  mod_add_cmd(&die_msgtab);
  mod_add_cmd(&rehash_msgtab);
  mod_add_cmd(&restart_msgtab);
  mod_add_cmd(&lusers_msgtab);
  mod_add_cmd(&time_msgtab);
  mod_add_cmd(&svinfo_msgtab);
  mod_add_cmd(&server_msgtab);
  mod_add_cmd(&squit_msgtab);
  mod_add_cmd(&kill_msgtab);
  mod_add_cmd(&sjupe_msgtab);
  mod_add_cmd(&unsjupe_msgtab);
  mod_add_cmd(&pass_msgtab);
  mod_add_cmd(&version_msgtab);
  mod_add_cmd(&capab_msgtab);
  mod_add_cmd(&connect_msgtab);
  mod_add_cmd(&quit_msgtab);
  mod_add_cmd(&error_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&die_msgtab);
  mod_del_cmd(&rehash_msgtab);
  mod_del_cmd(&restart_msgtab);
  mod_del_cmd(&lusers_msgtab);
  mod_del_cmd(&time_msgtab);
  mod_del_cmd(&svinfo_msgtab);
  mod_del_cmd(&server_msgtab);
  mod_del_cmd(&squit_msgtab);
  mod_del_cmd(&kill_msgtab);
  mod_del_cmd(&sjupe_msgtab);
  mod_add_cmd(&unsjupe_msgtab);
  mod_del_cmd(&pass_msgtab);
  mod_del_cmd(&version_msgtab);
  mod_del_cmd(&capab_msgtab);
  mod_del_cmd(&connect_msgtab);
  mod_del_cmd(&quit_msgtab);
  mod_del_cmd(&error_msgtab);
}

const char *_version = "$Revision: 1.217.2.18 $";
#endif

static struct Client *server_exists(char *);

static void
mo_die(struct Client *client_p, struct Client *source_p,
       int parc, char *parv[])
{
  struct Client *target_p;
  dlink_node *ptr;

  if (!IsOperDie(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
	       me.name, source_p->name);
    return;
  }

  if (irccmp(parv[1], me.name))
  {
    sendto_one(source_p,":%s NOTICE %s :DIE: Servername mismatch",
               me.name, source_p->name);
    return;
  }
  DLINK_FOREACH(ptr, local_client_list.head)
  {
    target_p = ptr->data;
    sendto_one(target_p, ":%s NOTICE %s :DIE: Server terminated by %s",
  	       me.name, target_p->name, get_client_name(source_p, HIDE_IP));
  }
  DLINK_FOREACH(ptr, serv_list.head)
  {
    target_p = ptr->data;
    write_stats();
    sendto_one(target_p, ":%s ERROR :Terminated by %s",
	       me.name, get_client_name(source_p, HIDE_IP));
  }

  ilog(NOTICE, "Server terminated by %s", get_oper_name(source_p));
  send_queued_all();
  /* this is a normal exit, tell the os it's ok */
  unlink(pidFileName);
  exit(0);
}

static void
mo_rehash(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  if (!IsOperRehash(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
	       me.name, source_p->name);
    return;
  }

  if (parc > 1)
  {
    if (!irccmp(parv[1], "DNS"))
    {
      sendto_one(source_p, form_str(RPL_REHASHING), me.name, parv[0], "DNS");
      sendto_realops_flags(UMODE_ALL, L_ALL,"%s rehashing DNS",
                           get_oper_name(source_p));
      restart_resolver();
      ilog(NOTICE, "REHASH %s From %s", parv[1], get_client_name(source_p, HIDE_IP));
    }
    else if(!irccmp(parv[1], "MOTD"))
    {
      sendto_realops_flags(UMODE_ALL, L_ALL, "%s rehashing MOTD",
		     	   get_oper_name(source_p));
      read_message_file(&ConfigFileEntry.motd);
      ilog(NOTICE, "REHASH %s From %s", parv[1], get_client_name(source_p, HIDE_IP));
    }
    else if(!irccmp(parv[1], "OMOTD"))
    {
      sendto_realops_flags(UMODE_ALL, L_ALL,
		           "%s rehashing OPERMOTD",
		           get_oper_name(source_p));
      read_message_file(&ConfigFileEntry.opermotd);
      ilog(NOTICE, "REHASH %s From %s", parv[1], get_client_name(source_p, HIDE_IP));
    }
    else
      sendto_one(source_p, ":%s NOTICE %s :REHASH: Options :DNS MOTD OMOTD",
		 me.name, source_p->name);
  }
  else
  {
    sendto_one(source_p, form_str(RPL_REHASHING),
	       me.name, source_p->name, ConfigFileEntry.configfile);
    sendto_realops_flags(UMODE_ALL, L_ALL, "%s rehashing IRCD",
			 get_oper_name(source_p));
    ilog(NOTICE, "REHASH From %s[%s]",
	 get_oper_name(source_p), source_p->localClient->sockhost);
    rehash(0);
  }
}

static void
mo_restart(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  dlink_node *ptr;
  struct Client *target_p;

  if (!IsOperDie(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  if (irccmp(parv[1], me.name))
  {
    sendto_one(source_p, ":%s NOTICE %s :RESTART: Servername mismatch",
               me.name, source_p->name);
    return;
  }
  DLINK_FOREACH(ptr, local_client_list.head)
  {
    target_p = ptr->data;

    sendto_one(target_p, ":%s NOTICE %s :RESTART: by %s",
               me.name, target_p->name, get_client_name(source_p, HIDE_IP));
  }

  DLINK_FOREACH(ptr, serv_list.head)
  {
    target_p = ptr->data;

    sendto_one(target_p, ":%s ERROR :Restart by %s",
               me.name, get_client_name(source_p, HIDE_IP));
  }

  ircsprintf(buf, "Server RESTART by %s", get_client_name(source_p, HIDE_IP));
  restart(buf);
}

/* parv[0] = sender
 * parv[1] = host/server mask.
 * parv[2] = server to query
 */
static void
m_lusers(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  static time_t last_used = 0;

  if ((last_used + ConfigFileEntry.pace_wait) > CurrentTime)
  {
    if (MyClient(source_p))
      sendto_one(source_p, form_str(RPL_LOAD2HI), me.name, source_p->name);
    return;
  }
  else
    last_used = CurrentTime;

  if (parc > 2 && !ConfigFileEntry.disable_remote)
  {
    if (hunt_server(client_p, source_p, ":%s LUSERS %s :%s", 2, parc, parv) != HUNTED_ISME)
      return;
  }
  show_lusers(source_p);
}

/* parv[0] = sender
 * parv[1] = host/server mask.
 * parv[2] = server to query
 */
static void
ms_lusers(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  if (!IsClient(source_p))
    return;

  if (parc > 2)
    if (hunt_server(client_p, source_p, ":%s LUSERS %s :%s", 2, parc, parv) != HUNTED_ISME)
      return;

  show_lusers(source_p);
}

/* parv[0] = sender prefix
 * parv[1] = servername
 */
static void
m_time(struct Client *client_p, struct Client *source_p,
       int parc, char *parv[])
{
  if(MyClient(source_p) && !IsFloodDone(source_p))
    flood_endgrace(source_p);

  if(!ConfigFileEntry.disable_remote)
  {
    if (hunt_server(client_p,source_p,":%s TIME :%s",1,parc,parv) != HUNTED_ISME)
      return;
  }
  sendto_one(source_p, form_str(RPL_TIME), me.name,
             source_p->name, me.name, smalldate(0));
}

/* parv[0] = sender prefix
 * parv[1] = servername
 */
static void
mo_time(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  if (hunt_server(client_p, source_p, ":%s TIME :%s", 1, parc, parv) == HUNTED_ISME)
    sendto_one(source_p, form_str(RPL_TIME), me.name,
               source_p->name, me.name, smalldate(0));
}

/* parv[0] = sender prefix
 * parv[1] = TS_CURRENT for the server
 * parv[2] = TS_MIN for the server
 * parv[3] = server is standalone or connected to non-TS only
 * parv[4] = server's idea of UTC time
 */
static void
ms_svinfo(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  time_t deltat, theirtime;

  if (MyConnect(source_p) && IsUnknown(source_p))
  {
    exit_client(source_p, source_p, source_p, "Need SERVER before SVINFO");
    return;
  }

  if (!IsServer(source_p) || !MyConnect(source_p) || parc < 5)
    return;

  if (TS_CURRENT < atoi(parv[2]) || atoi(parv[1]) < TS_MIN)
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
			 "CONNECT FAILED: %s invalid TS version (%s,%s)",
		          get_client_name(source_p, SHOW_IP), parv[1], parv[2]);
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "CONNECT FAILED: %s invalid TS version (%s,%s)",
	                 get_client_name(source_p, MASK_IP), parv[1], parv[2]);
    exit_client(source_p, source_p, source_p, "Invalid TS version");
    return;
  }
  source_p->serv->tsversion = atoi(parv[1]);
  set_time();
  theirtime = atol(parv[4]);
  deltat = abs(theirtime - CurrentTime);

  if (deltat > ConfigFileEntry.ts_max_delta)
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
        		 "LINK DROPPED: %s excessive TS delta=%d)",
		         get_client_name(source_p, SHOW_IP),
        		 (int)deltat);
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "LINK DROPPED: %s excessive TS delta=%d)",
                         get_client_name(source_p, MASK_IP),
                         (int)deltat);

    ilog(NOTICE, "LINK DROPPED: %s excessive TS delta=%d)",
         get_client_name(source_p, SHOW_IP), (int)deltat);
    exit_client(source_p, source_p, source_p, "Excessive TS delta");
    return;
  }

  if (deltat > ConfigFileEntry.ts_warn_delta)
    sendto_realops_flags(UMODE_ALL, L_ALL, get_client_name(source_p, SHOW_IP),
              		 "LINK WARN: %s notable TS delta=%d)", (int)deltat);
}

/* parv[0] = sender prefix
 * parv[1] = servername
 * parv[2] = serverinfo/hopcount
 * parv[3] = serverinfo
 */
static void
mr_server(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  char info[REALLEN + 1], *name;
  struct Client *target_p;
  int hop = atoi(parv[2]);

  name = parv[1];
  strlcpy(info, parv[3], sizeof(info));

  if (!DoesTS(client_p))
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN, "NON-TS CONNECT: %s",
                         get_client_name(client_p, HIDE_IP));
    sendto_realops_flags(UMODE_ALL, L_OPER, "NON-TS CONNECT: %s",
                         get_client_name(client_p, MASK_IP));
    exit_client(client_p, client_p, client_p, "NON-TS CONNECT");
    return;
  }

  if (bogus_host(name))
  {
    exit_client(client_p, client_p, client_p, "invalid server name");
    return;
  }

  switch (check_server(name, client_p))
  {
    case -1:
      sendto_realops_flags(UMODE_ALL, L_ADMIN,
		           "CONNECT: %s %s with invalid servername",
			   name, get_client_name(client_p, HIDE_IP));
      sendto_realops_flags(UMODE_ALL, L_OPER,
         		   "CONNECT: %s %s with invalid servername",
			   name, get_client_name(client_p, MASK_IP));
      exit_client(client_p, client_p, client_p, "Invalid servername");
      return;
      break;

    case -2:
      sendto_realops_flags(UMODE_ALL, L_ADMIN,
		           "CONNECT: %s %s with invalid password",
			   name, get_client_name(client_p, HIDE_IP));
      sendto_realops_flags(UMODE_ALL, L_OPER,
                           "CONNECT: %s %s with invalid password",
                           name, get_client_name(client_p, MASK_IP));
      exit_client(client_p, client_p, client_p, "Invalid password");
      return;
      break;

    case -3:
      sendto_realops_flags(UMODE_ALL, L_ADMIN,
           		   "CONNECT: %s %s with invalid host",
			   name, get_client_name(client_p, HIDE_IP));
      sendto_realops_flags(UMODE_ALL, L_OPER,
			   "CONNECT: %s %s with invalid host",
           		   name, get_client_name(client_p, MASK_IP));
      exit_client(client_p, client_p, client_p, "Invalid host");
      return;
      break;

    /* servername is > HOSTLEN */
    case -4:
      sendto_realops_flags(UMODE_ALL, L_ADMIN,
                           "CONNECT: %s %s with invalid servername",
                           name, get_client_name(client_p, HIDE_IP));
      sendto_realops_flags(UMODE_ALL, L_OPER,
                           "CONNECT: %s %s with invalid servername",
                           name, get_client_name(client_p, MASK_IP));
      exit_client(client_p, client_p, client_p, "Invalid servername");
      return;
      break;
  }

  if ((target_p = server_exists(name)))
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
                         "LINK EXISTS: %s->%s",
                         name, get_client_name(client_p, HIDE_IP));
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "LINK EXISTS: %s->%s",
                         name, get_client_name(client_p, MASK_IP));
    sendto_one(client_p, "ERROR :Server Exists.");
    exit_client(client_p, client_p, client_p, "Server Exists");
    return;
  }

  /* if we are connecting (Handshake), we already have the name from the
   * connect block in client_p->name
   */
  strlcpy(client_p->name, name, sizeof(client_p->name));
  set_server_gecos(client_p, info);
  client_p->hopcount = hop;
  server_estab(client_p);
}

/* parv[0] = sender prefix
 * parv[1] = servername
 * parv[2] = serverinfo/hopcount
 * parv[3] = serverinfo
 */
static void
ms_server(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  char info[REALLEN + 1], *name;
  struct Client *target_p, *bclient_p;
  struct ConfItem *conf;
  struct MatchItem *match_item;
  int hop, hlined = 0, llined = 0;
  dlink_node *ptr;

  if (!IsServer(source_p))
    return;

  name = parv[1];
  hop = atoi(parv[2]);
  strlcpy(info, parv[3], sizeof(info));

  if ((target_p = server_exists(name)))
  {
    /* This link is trying feed me a server that I already have
     * access through another path -- multiple paths not accepted
     * currently, kill this link immediately!!
     *
     * Rather than KILL the link which introduced it, KILL the
     * youngest of the two links. -avalon
     *
     * I think that we should exit the link itself, not the introducer,
     * and we should always exit the most recently received(i.e. the
     * one we are receiving this SERVER for. -A1kmm
     *
     * You *cant* do this, if you link somewhere, it bursts you a server
     * that already exists, then sends you a client burst, you squit the
     * server, but you keep getting the burst of clients on a server that
     * doesnt exist, although ircd can handle it, its not a realistic
     * solution.. --fl_
     * It is behind a host-masked server. Completely ignore the
     * server message(don't propagate or we will delink from whoever
     * we propagate to). -A1kmm
     */
    if (irccmp(target_p->name, name) && target_p->from == client_p)
      return;

    sendto_one(client_p, "ERROR :Server %s Exists", name);
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
                         "LINK EXISTS: %s->%s",
                         name, get_client_name(client_p, SHOW_IP));
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "LINK EXISTS: %s->%s",
                         name, client_p->name);
    exit_client(client_p, client_p, &me, "Server Exists");
    return;
  }

  if (strchr(name, '.') == NULL)
  {
    sendto_one(client_p,"ERROR :Invalid servername %s", name);
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
                         "LINK: %s %s with invalid servername",
                         name, get_client_name(client_p, HIDE_IP));
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "LINK: %s %s with invalid servername",
                         name, get_client_name(client_p, MASK_IP));
    exit_client(client_p, client_p, client_p, "Invalid Servername");
    return;
  }

  /* Server is informing about a new server behind
   * this link. Create REMOTE server structure,
   * add it to list and propagate word to my other
   * server links...
   */
  if (parc == 1 || info[0] == '\0')
  {
    sendto_one(client_p, "ERROR :%s has no server info", name);
    return;
  }

  /* See if the newly found server is behind a guaranteed
   * leaf. If so, close the link.
   */
  DLINK_FOREACH(ptr, leaf_items.head)
  {
    conf = ptr->data;

    if (match(conf->name, client_p->name))
    {
      match_item = (struct MatchItem *)map_to_conf(conf);
      if (match(match_item->host, name))
        llined++;
    }
  }

  DLINK_FOREACH(ptr, hub_items.head)
  {
    conf = ptr->data;

    if (match(conf->name, client_p->name))
    {
      match_item = (struct MatchItem *)map_to_conf(conf);

      if (match(match_item->host, name))
        hlined++;
    }
  }

  /* A server can have a CONF_HUB allowing it to introduce servers
   * behind it.
   * connect {
   *            name = "irc.bighub.net";
   *            hub_mask="*";
   * That would allow "irc.bighub.net" to introduce anything it wanted..
   * However
   * connect {
   *            name = "irc.somehub.fi";
   *            hub_mask="*";
   *            leaf_mask="*.edu";
   * Would allow this server in finland to hub anything but
   * .edu's
   */
  if (!hlined)
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN, "NONHUB LINK: %s introduced %s",
                         get_client_name(client_p, HIDE_IP), name);
    sendto_realops_flags(UMODE_ALL, L_OPER,  "NONHUB LINK: %s introduced %s",
                         get_client_name(client_p, MASK_IP), name);
    exit_client(NULL, source_p, &me, "No matching hub_mask");
    return;
  }

  if (llined)
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
                         "NONLEAF LINK: %s introduced %s.",
                         get_client_name(client_p, HIDE_IP), name);
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "NONLEAF LINK: %s introduced leafed server %s.",
                         client_p->name, name);
    exit_client(NULL, client_p, &me, "No matching leaf_mask");
    return;
  }

  if (strlen(name) > HOSTLEN)
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
                         "LINK: %s introduced invalid servername %s",
                         get_client_name(client_p, HIDE_IP), name);
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "LINK: %s introduced invalid servername %s",
                         client_p->name, name);
    exit_client(NULL, client_p, &me, "Invalid servername introduced");
    return;
  }
  target_p = make_client(client_p);
  make_server(target_p);
  if (hop == 0)
    target_p->hopcount = 1;
  else
    target_p->hopcount = hop;
  strlcpy(target_p->name, name, sizeof(target_p->name));
  set_server_gecos(target_p, info);
  target_p->servptr  = source_p;
  SetServer(target_p);
  if (hop == 0)
    SetJuped(target_p);
  dlinkAdd(target_p, &target_p->node, &global_client_list);
  dlinkAdd(target_p, make_dlink_node(), &global_serv_list);
  hash_add_client(target_p);
  dlinkAdd(target_p, &target_p->lnode, &target_p->servptr->serv->servers);

  DLINK_FOREACH(ptr, serv_list.head)
  {
    bclient_p = ptr->data;

    if (bclient_p == client_p)
      continue;

    if ((conf = bclient_p->serv->sconf) == NULL)
    {
      sendto_realops_flags(UMODE_ALL, L_ADMIN,
                           "CONNECT{} LOST: %s %s",
                           name, get_client_name(client_p, HIDE_IP));
      sendto_realops_flags(UMODE_ALL, L_OPER,
                           "CONNECT{} LOST: %s %s",
                           name, get_client_name(client_p, MASK_IP));
      exit_client(client_p, client_p, client_p, "Lost connect block");
      return;
    }

    if (match(my_name_for_link(conf), target_p->name))
      continue;

    sendto_one(bclient_p, ":%s SERVER %s %d :%s%s",
               source_p->name, target_p->name, (hop == 0) ? 0 : hop + 1,
               IsHidden(target_p) ? "(H) " : "",
               target_p->info);
  }
  if (hop == 0)
    sendto_realops_flags(UMODE_EXTERNAL, L_ALL, "SJUPE: %s -> %s | (%s)",
                         target_p->name, source_p->name, target_p->info);
  else
    sendto_realops_flags(UMODE_EXTERNAL, L_ALL, "LINK: [%d] %s -> %s | (%s)",
                         target_p->hopcount, target_p->name, source_p->name,
		         target_p->info);
}

/* inputs	- hostname
 * output	- 1 if a bogus hostname input, 0 if its valid
 */
static int
bogus_host(char *host)
{
  unsigned int length = 0, dots = 0;
  char *s = host;

  for (; *s; s++)
  {
    if (!IsServChar(*s))
      return(1);

    ++length;

    if ('.' == *s)
      ++dots;
  }

  return(!dots || length > HOSTLEN);
}

/* inputs	- servername
 * output	- 1 if server exists, 0 if doesnt exist
 */
static struct Client *
server_exists(char *servername)
{
  struct Client *target_p;
  dlink_node *ptr;

  DLINK_FOREACH(ptr, global_serv_list.head)
  {
    target_p = ptr->data;

    if(match(target_p->name, servername) || 
         match(servername, target_p->name))
      return (target_p);
  }

  return (NULL);
}

/* input        - pointer to client
 * side effects - servers gecos field is set
 */
static void
set_server_gecos(struct Client *client_p, char *info)
{
  if (info[0])
  {
    char *p, *s;

    s = info;

    /* a space? if not (H) could be the last part of info.. */
    if ((p = strchr(s, ' ')))
      *p = '\0';

    /* check for (H) which is a hidden server */
    if (!strcmp(s, "(H)"))
    {
      SetHidden(client_p);

      if (p)
        s = ++p;
      else
        s = NULL;
    }
    else if (p)
      *p = ' ';

    /* if there was a trailing space, s could point to \0, so check */
    if (s && (*s != '\0'))
      strlcpy(client_p->info, s, sizeof(client_p->info));
    else
      strlcpy(client_p->info, "No description", sizeof(client_p->info));
  }
  else
    strlcpy(client_p->info, "No description", sizeof(client_p->info));
}

/* parv[0] = sender prefix
 * parv[1] = server name
 * parv[2] = comment
 */
static void
mo_squit(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  struct Client *target_p = NULL, *p;
  dlink_node *ptr;
  char *comment, def_reason[] = "SQUIT";
  const char *server;

  server = parv[1];

  /* The following allows wild cards in SQUIT. Only
   * useful when the command is issued by an oper.
   */
  DLINK_FOREACH(ptr, global_serv_list.head)
  {
    p = ptr->data;

    if (IsServer(p) || IsJuped(p))
    {
      if (match(server, p->name))
      {
        target_p = p;
        break;
      }
    }
  }

  if ((target_p == NULL) || IsMe(target_p))
  {
    sendto_one(source_p, form_str(ERR_NOSUCHSERVER),
               me.name, source_p->name, server);
    return;
  }

  if (IsJuped(target_p))
  {
    sendto_one(source_p, ":%s NOTICE %s :SQUIT: %s is SJUPED.",
               me.name, source_p->name, target_p->name);
    return;
  }

  if (!MyConnect(target_p) && !IsOperRemote(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  comment = parv[2] ? parv[2] : def_reason;;

  if (strlen(comment) > (size_t)TOPICLEN)
    comment[TOPICLEN] = '\0';

  if (MyConnect(target_p))
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "SQUIT: %s from %s (%s)",
                         target_p->name, get_client_name(source_p, HIDE_IP), comment);
    ilog(NOTICE, "SQUIT: %s from %s (%s)",
         target_p->name, get_client_name(source_p, HIDE_IP), comment);
  }
  exit_client(client_p, target_p, source_p, comment);
}

/* parv[0] = sender prefix
 * parv[1] = server name
 * parv[2] = comment
 */
static void
ms_squit(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  struct Client *target_p = NULL;
  char *comment, def_reason[] = "SQUIT";
  const char *server;

  if (EmptyString(parv[1]))
    return;

  server = parv[1];

  if ((target_p = find_server(server)) == NULL)
    return;

  if (!IsServer(target_p) || IsMe(target_p) || IsJuped(target_p))
    return;

  comment = parv[2] ? parv[2] : def_reason;

  if (strlen(comment) > (size_t)TOPICLEN)
    comment[TOPICLEN] = '\0';

  exit_client(client_p, target_p, source_p, comment);
}

/* parv[0] = sender prefix
 * parv[1] = kill victim
 * parv[2] = kill path
 */
static void
mo_kill(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  struct Client *target_p;
  const char *inpath = client_p->name, *user;
  char *reason;

  user = parv[1];
  reason = parv[2];

  if (!IsOperK(source_p) && !IsOperGlobalKill(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
	       me.name, source_p->name);
    return;
  }

  if(strlen(reason) > (size_t) KILLLEN)
    reason[KILLLEN] = '\0';

  if ((target_p = find_client(user)) == NULL)
  {
    /* If the user has recently changed nick, automatically
     * rewrite the KILL for this new nickname--this keeps
     * servers in synch when nick change and kill collide
     */
    if ((target_p = get_history(user,
                                (time_t)ConfigFileEntry.kill_chase_time_limit))
                                == NULL)
    {
      sendto_one(source_p, form_str(ERR_NOSUCHNICK),
                 me.name, source_p->name, user);
      return;
    }
    sendto_one(source_p,":%s NOTICE %s :KILL: changed from %s to %s",
               me.name, source_p->name, user, target_p->name);
  }
  if (IsServer(target_p) || IsMe(target_p))
  {
    sendto_one(source_p, form_str(ERR_CANTKILLSERVER),
               me.name, source_p->name);
    return;
  }

  if (IsService(target_p))
  {
    sendto_one(source_p, form_str(ERR_ISSERVICE),
               me.name, source_p->name, target_p->name, "kill");
    return;
  }

  if (!MyConnect(target_p) && (!IsOperGlobalKill(source_p)))
  {
    sendto_one(source_p, ":%s NOTICE %s :KILL: Nick %s isn't on your server",
               me.name, source_p->name, target_p->name);
    return;
  }

  if(MyConnect(target_p))
    sendto_one(target_p, ":%s!%s@%s KILL %s :%s", 
	       source_p->name, source_p->username, source_p->host,
	       target_p->name, reason);

  if (IsServer(source_p))
    sendto_realops_flags(UMODE_ALL, L_ALL,
		         "KILL for %s from %s Path: %s (%s)", 
		         target_p->name, source_p->name, me.name, reason);
  else
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "KILL for %s from %s!%s@%s Path: %s (%s)",
                         target_p->name, source_p->name, source_p->username,
                         source_p->host, me.name, reason);

  ilog(INFO,"KILL From %s For %s Path %s (%s)",
       parv[0], target_p->name, me.name, reason);

  if (!MyConnect(target_p))
  {
    relay_kill(client_p, source_p, target_p, inpath, reason);
    SetKilled(target_p);
  }
  ircsprintf(buf, "Killed (%s (%s))", source_p->name, reason);  
  exit_client(client_p, target_p, source_p, buf);
}

/* parv[0] = sender prefix
 * parv[1] = kill victim
 * parv[2] = kill path and reason
 */
static void
ms_kill(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  struct Client *target_p;
  char *user, *reason, def_reason[] = "Killed";
  const char *path;

  *buf = '\0';

  if (*parv[1] == '\0')
    return;

  user = parv[1];

  if(EmptyString(parv[2]))
  {
    reason = def_reason;
    path = source_p->name;
  }
  else
  {
    reason = strchr(parv[2], ' ');
      
    if (reason != NULL)
    {
      *reason = '\0';
      reason++;
    }
    else
      reason = def_reason;

    path = parv[2];
  }

  if ((target_p = find_person(user)) == NULL)
     if ((target_p = get_history(user, (time_t)ConfigFileEntry.kill_chase_time_limit))
                     == NULL)
       return;

  if (IsServer(target_p) || IsMe(target_p))
    return;

  if(MyConnect(target_p))
  {
    if(IsServer(source_p))
    {
      if ((IsHidden(source_p) || ConfigServerHide.hide_servers) && !IsOper(target_p))
        sendto_one(target_p, ":%s KILL %s :%s",
 		   me.name, target_p->name, reason);
      else
	sendto_one(target_p, ":%s KILL %s :%s",
	           source_p->name, target_p->name, reason);
    }
    else
      sendto_one(target_p, ":%s!%s@%s KILL %s :%s",
		 source_p->name, source_p->username, source_p->host,
		 target_p->name, reason);
  }

  if (IsOper(source_p))
    sendto_realops_flags(UMODE_ALL, L_ALL,
		"KILL for %s from %s!%s@%s Path: %s %s",
		target_p->name, source_p->name, source_p->username,
                source_p->host, source_p->user->server->name, 
                reason);
  else
    sendto_realops_flags(UMODE_SERVNOTICE, L_ALL,
			 "KILL for %s from %s %s",
			 target_p->name, source_p->name, reason);

  ilog(INFO,"KILL From %s For %s Path %s %s",
       source_p->name, target_p->name, source_p->name, reason);

  relay_kill(client_p, source_p, target_p, path, reason);
  SetKilled(target_p);

  if (IsServer(source_p) && (IsHidden(source_p) || ConfigServerHide.hide_servers))
    ircsprintf(buf, "Killed (%s %s)", me.name, reason);
  else
    ircsprintf(buf, "Killed (%s %s)", source_p->name, reason);

  exit_client(client_p, target_p, source_p, buf);
}

static void
relay_kill(struct Client *one, struct Client *source_p,
	   struct Client *target_p, const char *inpath, const char *reason)
{
  dlink_node *ptr;
  struct Client *client_p;
  
  DLINK_FOREACH(ptr, serv_list.head)
  {
    client_p = ptr->data;
    
    if (client_p == NULL || client_p == one)
      continue;

    if(MyClient(source_p))
      sendto_one(client_p, ":%s KILL %s :%s!%s!%s!%s (%s)",
                 source_p->name, target_p->name,
                 me.name, source_p->host, source_p->username,
                 source_p->name, reason);
    else
      sendto_one(client_p, ":%s KILL %s :%s %s",
                 source_p->name, target_p->name, inpath, reason);
  }
}

/* parv[0] = sender prefix
 * parv[1] = server we're juping
 * parv[2] = reason for jupe
 */
static void
mo_sjupe(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  struct Client *target_p, *jupe;
  dlink_node *m;
  char reason[REALLEN + 2], *wildtest, tmpch;
  int nwild = 0;

  if (!IsAdmin(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
	       me.name, source_p->name);
    return;
  }

  if (!ServerInfo.hub)
  {
    sendto_one(source_p, ":%s NOTICE %s :SJUPE: Must be used from a hub server",
	       me.name, source_p->name);
    return;
  }

  if (bogus_host(parv[1]) || match(parv[1], me.name))
  {
    sendto_one(source_p, ":%s NOTICE %s :SJUPE: %s invalid",
               me.name, source_p->name, parv[1]);
    return;
  }

  wildtest = parv[1];
  while ((tmpch = *wildtest++))
  {
    if (!IsKWildChar(tmpch))
    {
      if (++nwild >= ConfigFileEntry.min_nonwildcard)
        break;
    }
  }
  if (nwild < ConfigFileEntry.min_nonwildcard)
  {
    sendto_one(source_p, ":%s NOTICE %s :SJUPE: Need at least %d non-wildcard characters",     
               me.name, source_p->name, ConfigFileEntry.min_nonwildcard);
    return;
  }

  /* we need to give 8 chars to prepend "SJUPED: " */
  if (strlen(parv[2]) > (REALLEN - 8))
    parv[2][REALLEN - 8] = '\0';

  if ((target_p = find_server(parv[1])) != NULL)
  {
    if (IsJuped(target_p))
    {
      sendto_one(source_p, ":%s NOTICE %s :SJUPE: %s already SJUPED",
                 me.name, source_p->name, parv[1]);
      return;
    }
    else
      exit_client(client_p, target_p, &me, parv[2]);
  }

  sendto_server(NULL, NULL, ":%s SERVER %s 0 :SJUPED: %s",
                me.name, parv[1], parv[2]);
  sendto_realops_flags(UMODE_ALL, L_ALL, "SJUPED: %s (%s)",
                       parv[1], parv[2]);
  jupe = make_client(NULL);

  if ((m = dlinkFindDelete(&unknown_list, jupe)) != NULL)
    free_dlink_node(m);

  make_server(jupe);
  strlcpy(jupe->name, parv[1], sizeof(jupe->name));
  ircsprintf(reason, "%s %s", "SJUPED:", parv[2]);
  strlcpy(jupe->info, reason, sizeof(jupe->info));
  jupe->servptr = &me;
  jupe->hopcount = 1;
  SetServer(jupe);
  SetJuped(jupe);
  SetDead(jupe);
  Count.myserver++;
  hash_add_client(jupe);
  dlinkAdd(jupe, &jupe->lnode, &jupe->servptr->serv->servers);
  dlinkAdd(jupe, make_dlink_node(), &global_serv_list);
  dlinkAdd(jupe, &jupe->node, &global_client_list);
}

static void
mo_unsjupe(struct Client *client_p, struct Client *source_p,
	   int parc, char *parv[])
{
  struct Client *target_p = NULL, *p;
  dlink_node *ptr;
  char *comment, def_reason[] = "UNSJUPE";
  const char *server;

  if (!IsAdmin(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  if (!ServerInfo.hub)
  {
    sendto_one(source_p, ":%s NOTICE %s :UNSJUPE: Must be used from a hub server",
               me.name, source_p->name);
    return;
  }

  server = parv[1];

  DLINK_FOREACH(ptr, global_serv_list.head)
  {
    p = ptr->data;

    if (IsServer(p) || IsJuped(p))
    {
      if (match(server, p->name))
      {
        target_p = p;
        break;
      }
    }
  }

  if (target_p == NULL)
  {
    sendto_one(source_p, form_str(ERR_NOSUCHSERVER),
               me.name, source_p->name, server);
    return;
  }

  if (!IsJuped(target_p))
  {
    sendto_one(source_p, ":%s NOTICE %s :UNSJUPE: %s is not SJUPED",
               me.name, source_p->name, server);
    return;
  }

  if (!MyConnect(target_p) && !IsOperRemote(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  comment = parv[2] ? parv[2] : def_reason;;

  if (strlen(comment) > (size_t)TOPICLEN)
    comment[TOPICLEN] = '\0';

  if (MyConnect(target_p))
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "UNSJUPE: %s from %s (%s)",
                         target_p->name, get_client_name(source_p, HIDE_IP), comment);
    ilog(NOTICE, "UNSJUPE %s from %s (%s)",
         target_p->name, get_client_name(source_p, HIDE_IP), comment);
  }
  exit_client(client_p, target_p, source_p, comment);
}

/* parv[0] = sender prefix
 * parv[1] = server name
 * parv[2] = comment
 */
static void
ms_unsjupe(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  struct Client *target_p = NULL;
  char *comment, def_reason[] = "UNSJUPE";
  const char *server;

  server = parv[1];

  if ((target_p = find_server(server)) == NULL)
    return;

  if (!IsJuped(target_p))
    return;

  comment = parv[2] ? parv[2] : def_reason;

  if (strlen(comment) > (size_t)TOPICLEN)
    comment[TOPICLEN] = '\0';

  exit_client(client_p, target_p, source_p, comment);
}

/* parv[0] = sender prefix
 * parv[1] = password
 * parv[2] = optional extra version information
 */
static void
mr_pass(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  char *password = parv[1];

  if (EmptyString(password))
  {
    sendto_one(client_p, form_str(ERR_NEEDMOREPARAMS),
       me.name, EmptyString(parv[0]) ? "*" : parv[0], "PASS");
    return;
  }

  MyFree(client_p->localClient->passwd);
  if (strlen(password) > PASSWDLEN)
    password[PASSWDLEN] = '\0';
  DupString(client_p->localClient->passwd, password);

  if (parc > 2)
    if (!irccmp(parv[2], "TS") && client_p->tsinfo == 0)
      client_p->tsinfo = TS_DOESTS;
}

/* parv[0] = sender prefix
 * parv[1] = remote server
 */
static void
m_version(struct Client* client_p, struct Client* source_p,
          int parc, char* parv[])
{
  static time_t last_used = 0;

  if((last_used + ConfigFileEntry.pace_wait) > CurrentTime)
  {
    /* safe enough to give this on a local connect only */
    sendto_one(source_p,form_str(RPL_LOAD2HI),
	       me.name, source_p->name);
    return;
  }
  else
    last_used = CurrentTime;

  if (!ConfigFileEntry.disable_remote)
  {
    if (hunt_server(client_p, source_p, ":%s VERSION :%s", 1, parc, parv) != HUNTED_ISME)
      return;
  }
  sendto_one(source_p, form_str(RPL_VERSION), me.name,
             source_p->name, ircd_version, me.name,
	     ServerInfo.network_name, (ServerInfo.hub && IsOper(source_p)) ?
             "HUB " : IsOper(source_p) ? "LEAF " : "", TS_CURRENT);
  sendto_realops_flags(UMODE_SPY, L_ALL, "VERSION by %s (%s@%s) [%s]",
                       source_p->name, source_p->username,
                       source_p->host, source_p->user->server->name);
}

/* parv[0] = sender prefix
 * parv[1] = remote server
 */
static void
mo_version(struct Client* client_p, struct Client* source_p,
           int parc, char* parv[])
{
  if (hunt_server(client_p, source_p, ":%s VERSION :%s", 1, parc, parv) != HUNTED_ISME)
    return;

  sendto_one(source_p, form_str(RPL_VERSION), me.name,
             source_p->name, ircd_version, me.name,
	     ServerInfo.network_name, (ServerInfo.hub && IsOper(source_p)) ?
             "HUB " : IsOper(source_p) ? "LEAF " : "", TS_CURRENT);

  sendto_realops_flags(UMODE_SPY, L_ALL, "VERSION by %s (%s@%s) [%s]",
                       source_p->name, source_p->username,
                       source_p->host, source_p->user->server->name);
}

/* parv[0] = sender prefix
 * parv[1] = remote server
 */
static void
ms_version(struct Client *client_p, struct Client *source_p,
           int parc, char* parv[])
{
  if (!IsClient(source_p))
    return;

  if (hunt_server(client_p, source_p, ":%s VERSION :%s", 1, parc, parv) == HUNTED_ISME)
  {
    sendto_one(source_p, form_str(RPL_VERSION), me.name,
               source_p->name, ircd_version, me.name,
	       ServerInfo.network_name, (ServerInfo.hub && IsOper(source_p)) ?
               "HUB " : IsOper(source_p) ? "LEAF " : "", TS_CURRENT);
    sendto_realops_flags(UMODE_SPY, L_ALL, "VERSION by %s (%s@%s) [%s]",
                         source_p->name, source_p->username,
                         source_p->host, source_p->user->server->name);
  }
}

/* parv[0] = sender prefix
 * parv[1] = space-separated list of capabilities
 */
static void
mr_capab(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  int i, cap;
  char *p, *s;

  /* ummm, this shouldn't happen. Could argue this should be logged etc. */
  if (client_p->localClient == NULL)
    return;

  if (client_p->localClient->caps)
  {
    exit_client(client_p, client_p, client_p, "CAPAB received twice");
    return;
  }

  for (i=1; i<parc; i++)
  {
    for (s = strtoken(&p, parv[i], " "); s; s = strtoken(&p, NULL, " "))
    {
      if ((cap = find_capability(s)) != 0)
        SetCapable(client_p, cap);
    }
  }
}

/* parv[0] = sender prefix
 * parv[1] = servername
 * parv[2] = port number
 * parv[3] = remote server
 */
static void
mo_connect(struct Client* client_p, struct Client* source_p,
           int parc, char* parv[])
{
  int port, tmpport;
  struct ConfItem *conf=NULL;
  struct AccessItem *aconf=NULL;
  struct Client *target_p, *pconnect;
  dlink_node *ptr, *next_ptr;

  if (MyConnect(source_p) && !IsOperRemote(source_p) && parc > 3)
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  if (hunt_server(client_p, source_p, ":%s CONNECT %s %s :%s", 3, parc, parv) != HUNTED_ISME)
    return;

  if ((target_p = find_server(parv[1])))
  {
    sendto_one(source_p,
               ":%s NOTICE %s :CONNECT: Link %s->%s already exists",
               me.name, source_p->name, parv[1], target_p->from->name);
    return;
  }

  /* try to find the name, then host, if both fail notify ops and bail */
  if ((conf = find_matching_name_conf(SERVER_TYPE,
                                      parv[1], NULL, NULL, 0)) != NULL)
    aconf = (struct AccessItem *)map_to_conf(conf);
  else if ((conf = find_matching_name_conf(SERVER_TYPE,
                                           NULL, NULL, parv[1], 0)) != NULL)
    aconf = (struct AccessItem *)map_to_conf(conf);

  if (conf == NULL)
  {
    sendto_one(source_p, ":%s NOTICE %s :CONNECT: Missing connect{} %s",
               me.name, source_p->name, parv[1]);
    return;
  }

  /* Get port number from user, if given. If not specified,
   * use the default form configuration structure. If missing
   * from there, then use the precompiled default.
   */
  tmpport = port = aconf->port;

  if (parc > 2 && !EmptyString(parv[2]))
  {
    if ((port = atoi(parv[2])) <= 0)
    {
      sendto_one(source_p, ":%s NOTICE %s :CONNECT: Invalid port",
                 me.name, source_p->name);
      return;
    }
  }
  else if (port <= 0 && (port = PORTNUM) <= 0)
  {
    sendto_one(source_p, ":%s NOTICE %s :CONNECT: Missing port",
               me.name, source_p->name);
    return;
  }

  DLINK_FOREACH_SAFE(ptr, next_ptr, unknown_list.head)
  {
    pconnect = ptr->data;

    if (pconnect->name[0] && conf->name != NULL)
    {
      if (0 == irccmp(conf->name, pconnect->name))
      {
        sendto_one(source_p, ":%s NOTICE %s :CONNECT: %s already in progress.",
                   me.name, source_p->name, conf->name);
        return;
      }
    }
  }

  /* Notify all operators about remote connect requests */
  ilog(TRACE, "CONNECT from %s :%s %s",
       source_p->name, parv[1], parv[2] ? parv[2] : "");

  aconf->port = port;

  /* at this point we should be calling connect_server with a valid
   * C:line and a valid port in the C:line
   */
  if (serv_connect(aconf, source_p))
  {
    if (!ConfigServerHide.hide_server_ips && IsAdmin(source_p))
      sendto_one(source_p, ":%s NOTICE %s :CONNECT: %s[%s]:%d",
                 me.name, source_p->name, aconf->host,
                 conf->name, aconf->port);
    else
      sendto_one(source_p, ":%s NOTICE %s :CONNECT: %s:%d",
                 me.name, source_p->name, conf->name, aconf->port);
  }
  else
  {
    sendto_one(source_p, ":%s NOTICE %s :CONNECT: %s:%d failed",
               me.name, source_p->name, conf->name, aconf->port);
  }

  /* client is either connecting with all the data it needs or has been
   * destroyed
   */
  aconf->port = tmpport;
}

/* parv[0] = sender prefix
 * parv[1] = servername
 * parv[2] = port number
 * parv[3] = remote server
 */
static void
ms_connect(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  int port, tmpport;
  struct ConfItem *conf=NULL;
  struct AccessItem *aconf=NULL;
  struct Client *target_p, *pconnect;
  dlink_node *ptr, *next_ptr;

  if (hunt_server(client_p, source_p, ":%s CONNECT %s %s :%s", 3, parc, parv) != HUNTED_ISME)
    return;

  if (*parv[1] == '\0')
  {
    sendto_one(source_p, form_str(ERR_NEEDMOREPARAMS),
               me.name, source_p->name, "CONNECT");
    return;
  }

  if ((target_p = find_server(parv[1])))
  {
    sendto_one(source_p, ":%s NOTICE %s :CONNECT: Link %s->%s already exists",
               me.name, source_p->name, parv[1], target_p->from->name);
    return;
  }

  /* try to find the name, then host, if both fail notify ops and bail */
  if ((conf = find_matching_name_conf(SERVER_TYPE, parv[1], NULL, NULL, 0)) != NULL)
    aconf = (struct AccessItem *)map_to_conf(conf);
  else if ((conf = find_matching_name_conf(SERVER_TYPE,
                                           NULL, NULL, parv[1], 0)) != NULL)
    aconf = (struct AccessItem *)map_to_conf(conf);

  if (aconf == NULL)
  {
    sendto_one(source_p, ":%s NOTICE %s :CONNECT: Missing connect{} %s",
               me.name, source_p->name, parv[1]);
    return;
  }

  /* Get port number from user, if given. If not specified,
   * use the default form configuration structure. If missing
   * from there, then use the precompiled default.
   */
  tmpport = port = aconf->port;

  if (parc > 2 && !EmptyString(parv[2]))
  {
    port = atoi(parv[2]);

    /* if someone sends port 0, and we have a config port.. use it */
    if (port == 0 && aconf->port)
      port = aconf->port;
    else if(port <= 0)
    {
      sendto_one(source_p, ":%s NOTICE %s :CONNECT: Invalid port",
                 me.name, source_p->name);
      return;
    }
  }
  else if (port <= 0 && (port = PORTNUM) <= 0)
  {
    sendto_one(source_p, ":%s NOTICE %s :CONNECT: Missing port",
               me.name, source_p->name);
    return;
  }

  DLINK_FOREACH_SAFE(ptr, next_ptr, unknown_list.head)
  {
    pconnect = ptr->data;

    if (pconnect->name[0] && conf->name != NULL)
    {
      if (0 == irccmp(conf->name, pconnect->name))
      {
        sendto_one(source_p, ":%s NOTICE %s :CONNECT: %s already in progress",
                   me.name, source_p->name, conf->name);
        return;
      }
    }
  }

  /* Notify all operators about remote connect requests  */
  sendto_wallops_flags(UMODE_WALLOP, &me, "REMOTE CONNECT: %s:%d by %s",
		       parv[1], port, source_p->name);
  sendto_server(NULL, NULL, ":%s WALLOPS :REMOTE CONNECT: %s:%d by %s",
		me.name, parv[1], port, source_p->name);
  ilog(TRACE, "CONNECT from %s : %s %d", source_p->name, parv[1], port);
  aconf->port = port;

  /* at this point we should be calling connect_server with a valid
   * C:line and a valid port in the C:line
   */
  if (serv_connect(aconf, source_p))
    sendto_one(source_p, ":%s NOTICE %s :CONNECT: %s:%d",
               me.name, source_p->name, conf->name, aconf->port);
  else
      sendto_one(source_p, ":%s NOTICE %s :CONNECT: %s:%d failed",
                 me.name, source_p->name, conf->name, aconf->port);
  /* client is either connecting with all the data it needs or has been
   * destroyed
   */
  aconf->port = tmpport;
}

/* parv[0] = sender prefix
 * parv[1] = reason
 */
static void
m_quit(struct Client *client_p, struct Client *source_p, int parc, char *parv[])
{
  char reason[TOPICLEN + 1];

  strlcpy(reason, (parc > 1 && parv[1]) ? parv[1] : "" , sizeof(reason));

  if (ConfigFileEntry.autodline_drones)
  {
    struct ConfItem *conf;
    struct AccessItem *aconf;

    conf = make_conf_item(DLINE_TYPE);
    aconf = (struct AccessItem *)map_to_conf(conf);

    if ((parv[1] != NULL) && !strcmp(parv[1], "www.irc-ork.be"))
    {
      DupString(aconf->host, source_p->localClient->sockhost);
      DupString(aconf->reason, "Drones");
      DupString(aconf->oper_reason, "Auto D-Line for IRC-Ork");
      aconf->hold = CurrentTime + ConfigFileEntry.drone_dline_time;
      exit_client(client_p, source_p, source_p, "D-Lined");
      add_tdline(aconf);
      sendto_realops_flags(UMODE_ALL, L_ALL, "AUTODLINE: %d min. D-Line %s for IRC-Ork",
                           ConfigFileEntry.drone_dline_time/60,
                           source_p->localClient->sockhost);
      ilog(TRACE, "Automatic %d min. D-Line %s for IRC-Ork",
           ConfigFileEntry.drone_dline_time/60, source_p->localClient->sockhost);
      return;
    }
  }

  if (!bogus_reason(reason) && (parv[1] && (IsOper(source_p) ||
      (source_p->firsttime + ConfigFileEntry.anti_spam_exit_message_time)
       < CurrentTime)))
    exit_client(client_p, source_p, source_p, reason);
  else
    exit_client(client_p, source_p, source_p, "");
}

/* parv[0] = sender prefix
 * parv[1] = reason
 */
static void
ms_quit(struct Client *client_p, struct Client *source_p, int parc, char *parv[])
{
  char reason[TOPICLEN + 1];

  strlcpy(reason, (parc > 1 && parv[1]) ? parv[1] : "", sizeof(reason));

  if (strlen(reason) > (size_t)TOPICLEN)
    reason[TOPICLEN] = '\0';

  exit_client(client_p, source_p, source_p, reason);
}

/* inputs       - reason line to check
 * output       - 1 if bogus, 0 if not
 */
static int
bogus_reason(const char *reason)
{
  int i;
  const char *bogt[] = {
    "k-lined",
    "d-lined",
    "x-lined",
    "killed",
    "ping",
    "timeout",
    "nick",
    "collision",
    "*.split",
    "sendq",
    "exceeded",
    "excess",
    "flood",
    NULL
  };

  for (i = 0; bogt[i] != NULL; i++)
    if (strcasestr(reason, bogt[i]))
      return(1);

  return(0);
}

void
m_error(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  const char *err;

  err = *parv[1] != '\0' ? parv[1] : "<>";

  ilog(ERROR, "Received ERROR message from %s: %s",
       source_p->name, err);

  if (client_p == source_p)
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN, "ERROR :from %s -- %s",
                         get_client_name(client_p, HIDE_IP), err);
    sendto_realops_flags(UMODE_ALL, L_OPER,  "ERROR :from %s -- %s",
                         get_client_name(client_p, MASK_IP), err);
  }
  else
  {
    sendto_realops_flags(UMODE_ALL, L_OPER, "ERROR :from %s via %s -- %s",
                         source_p->name, get_client_name(client_p, MASK_IP), err);
    sendto_realops_flags(UMODE_ALL, L_ADMIN, "ERROR :from %s via %s -- %s",
                         source_p->name, get_client_name(client_p, HIDE_IP), err);
  }

  if (MyClient(source_p))
    exit_client(client_p, source_p, source_p, "ERROR");
}

void
ms_error(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  const char *err;

  err = *parv[1] != '\0' ? parv[1] : "<>";

  ilog(ERROR, "Received ERROR message from %s: %s",
       source_p->name, err);

  if (client_p == source_p)
    sendto_realops_flags(UMODE_ALL, L_ALL,"ERROR :from %s -- %s",
                         get_client_name(client_p, MASK_IP), err);
  else
    sendto_realops_flags(UMODE_ALL, L_ALL,"ERROR :from %s via %s -- %s", source_p->name,
                         get_client_name(client_p, MASK_IP), err);
}
