/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * m_bans.c: ban management.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_bans.c,v 1.20.2.14 2004/07/25 06:44:04 musirc Exp $
 */

#include "channel.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "hostmask.h"
#include "numeric.h"
#include "config.h"
#include "log.h"
#include "misc.h"
#include "send.h"
#include "hash.h"
#include "handlers.h"
#include "modules.h"
#include "server.h"
#include "fileio.h"
#include "list.h"
#include "jupe.h"

static void mo_kline(struct Client *, struct Client *, int, char **);
static void ms_kline(struct Client *, struct Client *, int, char **);
static void mo_dline(struct Client *, struct Client *, int, char **);
#ifndef IPV6
static char *make_cidr(char *, struct Client *);
#endif
static void mo_testline(struct Client *, struct Client *, int, char **);
static void mo_unkline(struct Client *, struct Client *, int, char **);
static void ms_unkline(struct Client *, struct Client *, int, char **);
static void mo_undline(struct Client *, struct Client *, int, char **);
static int remove_tkline_match(const char *, const char *);
static int remove_tdline_match(const char *);
static void mo_close(struct Client *, struct Client *, int, char **);
static void mo_xline(struct Client *, struct Client *, int, char **);
static void ms_xline(struct Client *, struct Client *, int, char **);
static void mo_unxline(struct Client *, struct Client *, int, char **);
static void ms_unxline(struct Client *, struct Client *, int, char **);
static int valid_xline(struct Client *, char *, char *);
static void write_xline(struct Client *, char *, char *);
static void mo_cjupe(struct Client *, struct Client *, int, char **);
static void mo_njupe(struct Client *, struct Client *, int, char **);
static void mo_uncjupe(struct Client *, struct Client *, int, char **);
static void mo_unnjupe(struct Client *, struct Client *, int, char **);
static void ms_cjupe(struct Client *, struct Client *, int, char **);
static void ms_uncjupe(struct Client *, struct Client *, int, char **);
static void ms_njupe(struct Client *, struct Client *, int, char **);
static void ms_unnjupe(struct Client *, struct Client *, int, char **);
static void parse_jupe(struct Client *, char *, char *);
static void remove_jupe(struct Client *, char *);

struct Message kline_msgtab = {
  "KLINE", 4, MFLG_SLOW,
  {m_unregistered, m_not_oper, ms_kline, mo_kline}
};
struct Message dline_msgtab = {
  "DLINE", 4, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_dline}
};
struct Message testline_msgtab = {
  "TESTLINE", 0, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_testline}
};
struct Message unkline_msgtab = {
  "UNKLINE", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, ms_unkline, mo_unkline}
};
struct Message undline_msgtab = {
  "UNDLINE", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_undline}
};
struct Message close_msgtab = {
  "CLOSE", 0, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_close}
};
struct Message xline_msgtab = {
  "XLINE", 3, MFLG_SLOW,
  {m_unregistered, m_not_oper, ms_xline, mo_xline}
};
struct Message unxline_msgtab = {
  "UNXLINE", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, ms_unxline, mo_unxline}
};
struct Message cjupe_msgtab = {
  "CJUPE", 3, MFLG_SLOW,
  {m_ignore, m_not_oper, ms_cjupe, mo_cjupe}
};
struct Message njupe_msgtab = {
  "NJUPE", 3, MFLG_SLOW,
  {m_ignore, m_not_oper, ms_njupe, mo_njupe}
};
struct Message uncjupe_msgtab = {
  "UNCJUPE", 2, MFLG_SLOW,
  {m_ignore, m_not_oper, ms_uncjupe, mo_uncjupe}
};
struct Message unnjupe_msgtab = {
  "UNNJUPE", 2, MFLG_SLOW,
  {m_ignore, m_not_oper, ms_unnjupe, mo_unnjupe}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&kline_msgtab);
  mod_add_cmd(&dline_msgtab);
  mod_add_cmd(&testline_msgtab);
  mod_add_cmd(&unkline_msgtab);
  mod_add_cmd(&undline_msgtab);
  mod_add_cmd(&close_msgtab);
  mod_add_cmd(&xline_msgtab);
  mod_add_cmd(&unxline_msgtab);
  mod_add_cmd(&cjupe_msgtab);
  mod_add_cmd(&njupe_msgtab);
  mod_add_cmd(&uncjupe_msgtab);
  mod_add_cmd(&unnjupe_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&kline_msgtab);
  mod_del_cmd(&dline_msgtab);
  mod_del_cmd(&testline_msgtab);
  mod_del_cmd(&unkline_msgtab);
  mod_del_cmd(&undline_msgtab);
  mod_del_cmd(&close_msgtab);
  mod_del_cmd(&xline_msgtab);
  mod_del_cmd(&unxline_msgtab);
  mod_del_cmd(&cjupe_msgtab);
  mod_del_cmd(&njupe_msgtab);
  mod_del_cmd(&uncjupe_msgtab);
  mod_del_cmd(&unnjupe_msgtab);
}
const char *_version = "$Revision: 1.20.2.14 $";
#endif

static time_t valid_tkline(char *, int);
static char *mkline(char *);
static int find_user_host(struct Client *, char *, char *, char *);
static int valid_comment(struct Client *, char *);
static int valid_user_host(struct Client *, char *, char *);
static int valid_wild_card(struct Client *, char *, char *);
static int already_placed_kline(struct Client *, const char *, const char *);
static void apply_kline(struct Client *, struct ConfItem *, const char *, time_t);
static void apply_tkline(struct Client *, struct ConfItem *, int);

/* inputs       - pointer to server
 *              - pointer to client
 *              - parameter count
 *              - parameter list
 * side effects - k line is added
 */
static void
mo_kline(struct Client *client_p, struct Client *source_p,
         int parc, char **parv)
{
  char *reason, *oper_reason, user[USERLEN+2], host[HOSTLEN+2];
  const char *current_date, *target_server = NULL;
  struct ConfItem *conf;
  struct AccessItem *aconf;
  time_t tkline_time = 0;
  static char buffer[BUFSIZE];

  if (!IsOperK(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  if ((tkline_time = valid_tkline(parv[1], 1)) < 0)
  {
    sendto_one(source_p, ":%s NOTICE %s :KLINE: Invalid time",
               me.name, source_p->name);
    return;
  }

  if (!find_user_host(source_p, parv[2], user, host))
    return;

  if (!irccmp(parv[3], "ON"))
  {
    if (parc < 6)
    {
      sendto_one(source_p, form_str(ERR_NEEDMOREPARAMS),
                 me.name, source_p->name, "KLINE");
      return;
    }
    target_server = parv[4];
    reason = parv[5];
  }
  else
    reason = parv[3];

  if (!valid_user_host(source_p, user, host) || !valid_wild_card(source_p, user, host) ||
      !valid_comment(source_p, reason))
    return;

  if (target_server != NULL)
  {
    sendto_server(NULL, NULL, ":%s KLINE %s %lu %s %s :%s",
                  source_p->name, target_server, (unsigned long)tkline_time,
                  user, host, reason);
    if (!match(target_server, me.name))
      return;
  }

  if (already_placed_kline(source_p, user, host))
    return;

  if ((oper_reason = strchr(reason, '|')) != NULL)
  {
    *oper_reason = '\0';
    oper_reason++;
  }

  set_time();
  current_date = smalldate(CurrentTime);
  conf = make_conf_item(KLINE_TYPE);
  aconf = (struct AccessItem *)map_to_conf(conf);
  DupString(aconf->host, host);
  DupString(aconf->user, user);
  aconf->port = 0;
  ircsprintf(buffer, "%s (%s)", reason, current_date);
  DupString(aconf->reason, buffer);

  if (tkline_time != 0)
  {
    if (oper_reason != NULL)
    {
      ircsprintf(buffer, "%d min. K-Line - %s", (int)(tkline_time/60),
		 oper_reason);
      DupString(aconf->oper_reason, buffer);
    }
    else
    {
      ircsprintf(buffer, "%d min. K-Line", (int)(tkline_time/60));
      DupString(aconf->oper_reason, buffer);
    }
    apply_tkline(source_p, conf, tkline_time);
  }
  else
  {
    if (oper_reason != NULL)
      DupString(aconf->oper_reason, oper_reason);
    apply_kline(source_p, conf, current_date, CurrentTime);
  }
}

static void
ms_kline(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  struct ConfItem *conf = NULL;
  struct AccessItem *aconf = NULL;
  int tkline_time;
  const char* current_date;
  char *kuser, *khost, *kreason, *oper_reason;
  static char buffer[BUFSIZE];

  if (parc != 6)
    return;

  /* parv[0]  parv[1]        parv[2]      parv[3]  parv[4]  parv[5]
   * oper     target_server  tkline_time  user     host     reason */
  sendto_match_servs(source_p, parv[1], "KLINE %s %s %s %s :%s",
                     parv[1], parv[2], parv[3], parv[4], parv[5]);

  if (!match(parv[1], me.name))
    return;

  if ((tkline_time = valid_tkline(parv[2], 0)) < 0)
    return;

  kuser = parv[3];
  khost = parv[4];
  kreason = parv[5];
  set_time();
  current_date = smalldate(CurrentTime);

  if ((oper_reason = strchr(kreason, '|')) != NULL)
  {
    *oper_reason = '\0';
    oper_reason++;
  }

  if (find_matching_name_conf(ULINE_TYPE, source_p->user->server->name,
                              source_p->username, source_p->host, SHARED_KLINE))
  {
    if (!valid_wild_card(source_p, kuser, khost) ||
        !valid_user_host(source_p, kuser, khost) ||
        !valid_comment(source_p, kreason) || !IsPerson(source_p) ||
        already_placed_kline(source_p, kuser, khost))
      return;

    conf = make_conf_item(KLINE_TYPE);
    aconf = (struct AccessItem *)map_to_conf(conf);
    DupString(aconf->host, khost);
    DupString(aconf->user, kuser);
    ircsprintf(buffer, "%s (%s)", kreason, current_date);
    DupString(aconf->reason, buffer);

    if (tkline_time != 0)
    {
      if (oper_reason != NULL)
      {
        ircsprintf(buffer, "%d min. K-Line - %s", (int)(tkline_time/60),
                   oper_reason);
        DupString(aconf->oper_reason, buffer);
      }
      else
      {
        ircsprintf(buffer, "%d min. K-Line", (int)(tkline_time/60));
        DupString(aconf->oper_reason, buffer);
      }
      apply_tkline(source_p, conf, tkline_time);
    }
    else
    {
      if (oper_reason != NULL)
        DupString(aconf->oper_reason, oper_reason);
      apply_kline(source_p, conf, current_date, CurrentTime);
    }
  }
}

/* side effects - kline as given, is added to the hashtable
 *                and conf file
 */
static void
apply_kline(struct Client *source_p, struct ConfItem *conf,
            const char *current_date, time_t cur_time)
{
  struct AccessItem *aconf;

  aconf = (struct AccessItem *)map_to_conf(conf);
  add_conf_by_address(CONF_KLINE, aconf);
  write_conf_line(source_p, conf, current_date, cur_time);
  /* Now, activate kline against current online clients */
  rehashed_klines = 1;
}

static void
apply_tkline(struct Client *source_p, struct ConfItem *conf,
             int tkline_time)
{
  struct AccessItem *aconf;

  aconf = (struct AccessItem *)map_to_conf(conf);
  aconf->hold = CurrentTime + tkline_time;
  add_tkline(aconf);
  sendto_realops_flags(UMODE_ALL, L_ALL,
                       "%s added %d min. K-Line %s@%s [%s]",
                       get_oper_name(source_p), tkline_time/60,
                       aconf->user, aconf->host, aconf->reason);
  sendto_one(source_p, ":%s NOTICE %s :KLINE: %d min. %s@%s [%s]",
	     me.name, source_p->name, tkline_time/60, aconf->user, aconf->host,
	     aconf->reason);
  ilog(TRACE, "%s added %d min. K-Line %s@%s [%s]",
       source_p->name, tkline_time/60,
       aconf->user, aconf->host, aconf->reason);
  rehashed_klines = 1;
}

static void
apply_tdline(struct Client *source_p, struct ConfItem *conf, int tkline_time)
{
  struct AccessItem *aconf;

  aconf = (struct AccessItem *)map_to_conf(conf);
  aconf->hold = CurrentTime + tkline_time;
  add_tdline(aconf);
  sendto_realops_flags(UMODE_ALL, L_ALL,
                       "%s added %d min. D-Line %s [%s]",
                       get_oper_name(source_p), tkline_time/60,
                       aconf->host, aconf->reason);
  sendto_one(source_p, ":%s NOTICE %s :DLINE: %d min. %s [%s]",
             me.name, source_p->name, tkline_time/60, aconf->host, aconf->reason);
  ilog(TRACE, "%s added %d min. D-Line %s [%s]",
       source_p->name, tkline_time/60, aconf->host, aconf->reason);
  rehashed_klines = 1;
}

/* inputs       - pointer to ascii string to check
 *              - whether the specified time is in seconds or minutes
 * output       - -1 not enough parameters
 *              - 0 if not an integer number, else the number
 */
static time_t
valid_tkline(char *p, int mins)
{
  time_t result = 0;

  while (*p)
  {
    if (IsDigit(*p))
    {
      result *= 10;
      result += ((*p) & 0xF);
      p++;
    }
    else
      return(-1);
  }

  if (result <= 0)
    return(0);

  if (!mins)
    result = (time_t)result / (time_t)60;

  if (result > MAX_TDKLINE_TIME)
    result = MAX_TDKLINE_TIME;

  result = (time_t)result * (time_t)60;  /* turn it into seconds */

  return(result);
}

/* inputs       - pointer to a hostname
 * output       - pointer to a static of the hostname masked
 *                for use in a kline.
 */
static char *
mkline(char *hostname)
{
  static char result[HOSTLEN + 1];      /* result to return */
  char temphost[HOSTLEN + 1], *ipp, *host_mask, *zap_point = NULL, *tld;
  int is_ip_number, number_of_dots;

  if (hostname == NULL)
    return(NULL);

  /* If a '@' is found in the hostname, this is bogus
   * and must have been introduced by server that doesn't
   * check for bogus domains (dns spoof) very well. *sigh* just return it...
   * I could also legitimately return (char *)NULL as above.
   */
  if (strchr(hostname, '@'))
  {
    strlcpy(result, hostname, sizeof(result));
    return(result);
  }
  strlcpy(temphost, hostname, sizeof(temphost));
  is_ip_number = 1;   /* assume its an IP# */
  ipp = temphost;
  number_of_dots = 0;

  while (*ipp)
  {
    if (*ipp == '.')
    {
      number_of_dots++;

      if(number_of_dots == 3)
        zap_point = ipp;
      ipp++;
    }
    else if (!IsDigit(*ipp))
    {
      is_ip_number = 0;
      break;
    }
    ipp++;
  }

  if (is_ip_number && (number_of_dots == 3))
  {
    zap_point++;
    *zap_point++ = '*';               /* turn 111.222.333.444 into */
    *zap_point = '\0';                /*      111.222.333.*        */
    strlcpy(result, temphost, sizeof(result));
    return(result);
  }
  else
  {
    tld = strrchr(temphost, '.');
    if (tld)
    {
      number_of_dots = 2;
      if (tld[3])
        number_of_dots = 1;

      if(tld != temphost)
        host_mask = tld - 1;
      else
        host_mask = tld;

      while (host_mask != temphost)
      {
        if(*host_mask == '.')
          number_of_dots--;
        if(number_of_dots == 0)
        {
          result[0] = '*';
          strlcpy(result + 1, host_mask, sizeof(result) - 1);
          return(result);
        }
        host_mask--;
      }
      result[0] = '*';                      /* foo.com => *foo.com */
      strlcpy(result + 1, temphost, sizeof(result) - 1);
    }
    else      /* no tld found oops. just return it as is */
    {
      strlcpy(result, temphost, sizeof(result));
      return(result);
    }
  }

  return(result);
}

/* inputs       - pointer to server
 *              - pointer to client
 *              - parameter count
 *              - parameter list
 * side effects - D line is added
 */
static void
mo_dline(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  static char buffer[BUFSIZE];
  char *dlhost, *oper_reason, *reason;
  const char *creason, *current_date;
#ifndef IPV6
  struct Client *target_p;
#endif
  struct irc_ssaddr daddr;
  struct ConfItem *conf = NULL;
  struct AccessItem *aconf = NULL;
  time_t tkline_time = 0;
  int bits, t;

  if (!IsOperDline(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }
  if ((tkline_time = valid_tkline(parv[1], 1)) < 0)
  {
    sendto_one(source_p, ":%s NOTICE %s :DLINE: Invalid time",
               me.name, source_p->name);
    return;
  }
  dlhost = parv[2];

  if ((t=parse_netmask(dlhost, NULL, &bits)) == HM_HOST)
  {
#ifdef IPV6
    sendto_one(source_p, ":%s NOTICE %s :DLINE: Invalid IP",
               me.name, source_p->name);
    return;
#else
    if ((target_p = find_chasing(source_p, dlhost, NULL)) == NULL)
      return;

    if(target_p->user == NULL)
      return;

    t = HM_IPV4;

    if (IsServer(target_p))
    {
      sendto_one(source_p, ":%s NOTICE %s :DLINE: Can't DLINE servers",
                 me.name, source_p->name);
      return;
    }

    if (!MyConnect(target_p))
    {
      sendto_one(source_p, ":%s NOTICE %s :DLINE: Can't DLINE nick on other servers",
                 me.name, source_p->name);
      return;
    }

    if (IsExemptKline(target_p))
    {
      sendto_one(source_p, ":%s NOTICE %s :DLINE: %s is E-Lined", me.name,
                 source_p->name, target_p->name);
      return;
    }

    if ((dlhost = make_cidr(dlhost, target_p)) == NULL)
      return;

    bits = 0xFFFFFF00UL;
#endif
  }
  if (!valid_comment(source_p, parv[3]))
    return;

  reason = parv[3];

  if (bits < 8)
  {
    sendto_one(source_p, ":%s NOTICE %s :DLINE: Bitmasks less than 8 require conf access.",
               me.name, source_p->name);
    return;
  }

#ifdef IPV6
  if (t == HM_IPV6)
    t = AF_INET6;
  else
#endif
  t = AF_INET;

  parse_netmask(dlhost, &daddr, NULL);

  if ((aconf = find_dline_conf(&daddr, t)) != NULL)
  {
    creason = aconf->reason ? aconf->reason : "D-Lined";
    if (IsConfExemptKline(aconf))
      sendto_one(source_p, ":%s NOTICE %s :DLINE: %s is E/D-Lined %s [%s]",
                 me.name, source_p->name, dlhost, aconf->host, creason);
    else
      sendto_one(source_p, ":%s NOTICE %s :DLINE: %s already D-Lined by %s [%s]",
                 me.name, source_p->name, dlhost, aconf->host, creason);
    return;
  }
  set_time();
  current_date = smalldate(CurrentTime);

  if ((oper_reason = strchr(reason, '|')) != NULL)
  {
    *oper_reason = '\0';
    oper_reason++;
  }

  conf = make_conf_item(DLINE_TYPE);
  aconf = (struct AccessItem *)map_to_conf(conf);
  DupString(aconf->host, dlhost);
  ircsprintf(buffer, "%s (%s)", reason, current_date);
  DupString(aconf->reason, buffer);

  if (tkline_time != 0)
  {
    if (oper_reason != NULL)
    {
      ircsprintf(buffer, "%d min. D-Line - %s", (int)(tkline_time/60),
                 oper_reason);
      DupString(aconf->oper_reason, buffer);
    }
    else
    {
      ircsprintf(buffer, "%d min. D-Line", (int)(tkline_time/60));
      DupString(aconf->oper_reason, buffer);
    }
    apply_tdline(source_p, conf, tkline_time);
  }
  else
  {
    if (oper_reason != NULL)
      DupString(aconf->oper_reason, oper_reason);
    add_conf_by_address(CONF_DLINE, aconf);
    write_conf_line(source_p, conf, current_date, CurrentTime);
  }
  rehashed_klines = 1;
}

/* inputs       - pointer to client placing kline
 *              - pointer to user_host_or_nick
 *              - pointer to user buffer
 *              - pointer to host buffer
 * output       - 0 if not ok to kline, 1 to kline i.e. if valid user host
 */
static int
find_user_host(struct Client *source_p, char *user_host_or_nick,
	       char *luser, char *lhost)
{
  struct Client *target_p;
  char *hostp;

  if ((hostp = strchr(user_host_or_nick, '@')) || *user_host_or_nick == '*')
  {
    /* Explicit user@host mask given */

    if (hostp != NULL)                            /* I'm a little user@host */
    {
      *(hostp++) = '\0';                       /* short and squat */
      if (*user_host_or_nick)
        strlcpy(luser,user_host_or_nick,USERLEN + 1); /* here is my user */
      else
        strcpy(luser, "*");
      if (*hostp)
        strlcpy(lhost, hostp, HOSTLEN + 1);    /* here is my host */
      else
        strcpy(lhost, "*");
    }
    else
    {
      luser[0] = '*';             /* no @ found, assume its *@somehost */
      luser[1] = '\0';
      strlcpy(lhost, user_host_or_nick, HOSTLEN + 1);
    }

    return(1);
  }
  else
  {
    /* Try to find user@host mask from nick */
    if (!(target_p = find_chasing(source_p, user_host_or_nick, NULL)))
      return(0);

    if (target_p->user == NULL)
      return(0);

    if (IsServer(target_p))
    {
      sendto_one(source_p, ":%s NOTICE %s :KLINE: Can't KLINE servers",
                 me.name, source_p->name);
      return(0);
    }

    if (IsExemptKline(target_p))
    {
      if(!IsServer(source_p))
        sendto_one(source_p, ":%s NOTICE %s :KLINE: %s is E-Lined",
                   me.name, source_p->name, target_p->name);
      return(0);
    }

    /* turn the "user" bit into "*user", blow away '~'
     * if found in original user name (non-idented)
     */
    strlcpy(luser, target_p->username, USERLEN + 1);
    if (*target_p->username == '~')
      luser[0] = '*';

    strlcpy(lhost, mkline(target_p->host), HOSTLEN + 1);
  }
  return(1);
}

/* inputs       - pointer to source
 *              - pointer to user buffer
 *              - pointer to host buffer
 * output       - 1 if valid user or host, 0 if invalid
 */
static int
valid_user_host(struct Client *source_p, char *luser, char *lhost)
{
  /* Check for # in user@host */
  if(strchr(lhost, '#') || strchr(luser, '#'))
  {
    sendto_one(source_p, ":%s NOTICE %s :KLINE: Invalid character '#'",
               me.name, source_p->name);
    return(0);
  }

  /* Dont let people kline *!ident@host, as the ! is invalid.. */
  if(strchr(luser, '!'))
  {
    sendto_one(source_p, ":%s NOTICE %s :KLINE: Invalid character '!'",
               me.name, source_p->name);
    return(0);
  }
  return(1);
}

/* input        - pointer to client
 *              - pointer to user to check
 *              - pointer to host to check
 * output       - 0 if not valid, 1 if valid
 */
static int
valid_wild_card(struct Client *source_p, char *luser, char *lhost)
{
  char *p, tmpch;
  int nonwild;

  nonwild = 0;
  p = luser;
  while ((tmpch = *p++))
  {
    if (!IsKWildChar(tmpch))
    {
      /* If we find enough non-wild characters, we can
       * break - no point in searching further.
       */
      if (++nonwild >= ConfigFileEntry.min_nonwildcard)
        break;
    }
  }

  if (nonwild < ConfigFileEntry.min_nonwildcard)
  {
    /* The user portion did not contain enough non-wild
     * characters, try the host.
     */
    p = lhost;
    while ((tmpch = *p++))
    {
      if (!IsKWildChar(tmpch))
        if (++nonwild >= ConfigFileEntry.min_nonwildcard)
          break;
    }
  }

  if (nonwild < ConfigFileEntry.min_nonwildcard)
  {
    sendto_one(source_p, ":%s NOTICE %s :KLINE: Less than %d non-wildcard characters",
               me.name, source_p->name, ConfigFileEntry.min_nonwildcard);
    return(0);
  }
  else
    return(1);
}

/* valid_comment()
 * inputs       - pointer to client
 *              - pointer to comment
 * output       - 0 if no valid comment,
 *              - 1 if valid
 * side effects - truncates reason where necessary
 */
static int
valid_comment(struct Client *source_p, char *comment)
{
  if (strchr(comment, '"'))
  {
    sendto_one(source_p, ":%s NOTICE %s :*** Invalid character '\"'",
               me.name, source_p->name);
    return(0);
  }
  if (strlen(comment) > REASONLEN)
    comment[REASONLEN-1] = '\0';

  return(1);
}

/* inputs       - user to complain to, username & host to check for
 * outputs      - returns 1 on existing K-Line, 0 if doesn't exist
 * side effects - notifies source_p if the K-Line already exists
 * Note: This currently works if the new K-Line is a special case of an
 *       existing K-Line, but not the other way round. To do that we would
 *       have to walk the hash and check every existing K-Line. -A1kmm.
 */
static int
already_placed_kline(struct Client *source_p, const char *luser, const char *lhost)
{
  const char *reason;
  struct irc_ssaddr iphost, *piphost;
  struct AccessItem *aconf;
  int t;

  if ((t=parse_netmask(lhost, &iphost, &t)) != HM_HOST)
  {
#ifdef IPV6
    if (t == HM_IPV6)
      t = AF_INET6;
    else
#endif
      t = AF_INET;
    piphost = &iphost;
  }
  else
  {
    t = 0;
    piphost = NULL;
  }

  if ((aconf = find_conf_by_address(lhost, piphost, CONF_KLINE, t, luser, NULL)))
  {
    reason = aconf->reason ? aconf->reason : "K-Lined";
    sendto_one(source_p,
               ":%s NOTICE %s :KLINE: %s@%s already K-Lined by %s@%s [%s]",
               me.name, source_p->name, luser, lhost, aconf->user,
               aconf->host, reason);
    return(1);
  }
  return(0);
}

#ifndef IPV6
static char *
make_cidr(char *dlhost, struct Client *target_p)
{
  static char cidr_form_host[HOSTLEN + 2];
  char *p;

  strlcpy(cidr_form_host, inetntoa((const char *)&target_p->localClient->ip),
	  sizeof(cidr_form_host));
  if ((p = strchr(cidr_form_host,'.')) == NULL)
    return(NULL);
  
  /* 192. <- p */
   p++;
   if ((p = strchr(p,'.')) == NULL)
     return(NULL);

   /* 192.168. <- p */
   p++;
   if ((p = strchr(p,'.')) == NULL)
     return(NULL);

   /* 192.168.0. <- p */
   p++;
   *p++ = '0';
   *p++ = '/';
   *p++ = '2';
   *p++ = '4';
   *p++ = '\0';

   return(cidr_form_host);
}
#endif

/* inputs       - pointer to physical connection request is coming from
 *              - pointer to source connection request is comming from
 *              - parc arg count
 *              - parv actual arguments   
 * output       - always 0
 * side effects - command to test I/K lines on server
 * i.e. /quote testline user@host,ip [password]
 */  
static void
mo_testline(struct Client *client_p, struct Client *source_p,
            int parc, char *parv[])
{
  struct AccessItem *aconf;
  struct irc_ssaddr ip;
  int host_mask, port, t;
  char *host, *pass, *user, *classname, *given_host, *given_name, *p, *oreason;

  if (parc < 2)
  {
    sendto_one(source_p, ":%s NOTICE %s :TESTLINE: Usage: user@host|ip [password]",
               me.name, source_p->name);
    return;
  }
  given_name = parv[1];

  if ((p = (char *) strchr(given_name, '@')) == NULL)
  {
    if ((t = parse_netmask(given_name, &ip, &host_mask)) != HM_HOST)
    {
      aconf = find_dline_conf(&ip,
#ifdef IPV6
                              (t == HM_IPV6) ? AF_INET6 : AF_INET
#else
                              AF_INET
#endif
                              );
      if (aconf != NULL)
      {
        struct ConfItem *conf;
        conf = unmap_conf_item(aconf);

        get_printable_conf(conf, &host, &pass, &user, &port,
                           &classname, &oreason);
        if (aconf->status & CONF_EXEMPTDLINE)
          sendto_one(source_p,
                     ":%s NOTICE %s :TESTLINE: Exempt D-Line: %s [%s]",
                     me.name, source_p->name, host, pass);
        else
          sendto_one(source_p,
                     ":%s NOTICE %s :TESTLINE: D-Line: %s %s [%s]", me.name,
                     source_p->name, host, pass, oreason);
      }
      else
	sendto_one(source_p, ":%s NOTICE %s :TESTLINE: D-Line not found",
                   me.name, source_p->name);
    }
    else sendto_one(source_p, ":%s NOTICE %s :TESTLINE: Usage: user@host|ip [password]",
                    me.name, source_p->name);
    return;
  }

  *p++ = '\0';
  given_host = p;

  if ((t = parse_netmask(given_host, &ip, &host_mask)) != HM_HOST)
    aconf = find_address_conf(given_host, given_name, &ip,
#ifdef IPV6
                              (t == HM_IPV6) ? AF_INET6 : AF_INET,
#else
                              AF_INET,
#endif
                              parv[2]);
  else
    aconf = find_address_conf(given_host, given_name, NULL, 0, parv[2]);

  if (aconf != NULL)
  {
    struct ConfItem *conf;
    conf = unmap_conf_item(aconf);
    get_printable_conf(conf, &host, &pass, &user, &port, &classname, &oreason);
    if (aconf->status & CONF_KLINE)
      sendto_one(source_p,
                 ":%s NOTICE %s :TESTLINE: %c-Line %s %s %s %s",
                 me.name, source_p->name,
                 (aconf->flags & CONF_FLAGS_TEMPORARY) ? 'k' : 'K',
                 user, host, pass, oreason);
    else if (aconf->status & CONF_CLIENT)
      sendto_one(source_p, ":%s NOTICE %s :TESTLINE: I-Line * %s %s %s %d %s",
                 me.name, source_p->name, show_iline_prefix(source_p, aconf, user),
		 user, host, port, classname);
  }
  else sendto_one(source_p, ":%s NOTICE %s :TESTLINE: No matches found",
		  me.name, source_p->name);
}

/* parv[0] = sender
 * parv[1] = address to remove
 */
static void
mo_unkline(struct Client *client_p,struct Client *source_p,
           int parc,char *parv[])
{
  char star[] = "*", *user, *host;

  if (!IsOperK(source_p))
  {
     sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
	        me.name, source_p->name);
     return;
  }

  if ((host = strchr(parv[1], '@')) || *parv[1] == '*')
  {
    /* Explicit user@host mask given */
    if (host)                  /* Found user@host */
    {
      user = parv[1];       /* here is user part */
      *(host++) = '\0';     /* and now here is host */
    }
    else
    {
      user = star;           /* no @ found, assume its *@somehost */
      host = parv[1];
    }
  }
  else
  {
    sendto_one(source_p, ":%s NOTICE %s :KLINE: %s not found",
               me.name, source_p->name, parv[1]);
    return;
  }

  /* UNKLINE bill@mu.org ON irc.mu.org */
  if ((parc > 3) && (!irccmp(parv[2], "ON")))
  {
    sendto_match_servs(source_p, parv[3], "UNKLINE %s %s %s",
                       parv[3], user, host);

    if (!match(parv[3], me.name))
      return;
  }

  if (remove_tkline_match(host, user))
  {
    sendto_one(source_p, ":%s NOTICE %s :UNKLINE: %s@%s",
               me.name, source_p->name, user, host);
    sendto_realops_flags(UMODE_ALL, L_ALL,
			 "%s removed %s@%s",
                         get_oper_name(source_p), user, host);
    ilog(NOTICE, "%s removed K-Line %s@%s", source_p->name, user, host);
    return;
  }

  if (remove_conf_line(KLINE_TYPE, source_p, user, host) > 0)
  {
    sendto_one(source_p, ":%s NOTICE %s :UNKLINE: %s@%s",
               me.name, source_p->name, user,host);
    sendto_realops_flags(UMODE_ALL, L_ALL, "%s removed K-Line %s@%s",
                         get_oper_name(source_p), user, host);
    ilog(NOTICE, "%s removed K-Line %s@%s", source_p->name, user, host);
    rehashed_klines = 1;
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :UNKLINE: %s@%s not found",
               me.name, source_p->name, user, host);
}

/* inputs       - server
 *              - client
 *              - parc
 *              - parv
 * side effects - if server is authorized, kline is removed
 */
static void
ms_unkline(struct Client *client_p, struct Client *source_p,
	   int parc, char *parv[])
{
  const char *kuser, *khost;

  if (parc != 4)
    return;

  kuser = parv[2];
  khost = parv[3];

  sendto_match_servs(source_p, parv[1], "UNKLINE %s %s %s",
                     parv[1], kuser, khost);

  if (!match(parv[1], me.name))
    return;

  if (!IsPerson(source_p))
    return;

  if (find_matching_name_conf(ULINE_TYPE,
		   source_p->user->server->name,
                   source_p->username, source_p->host,
                   SHARED_KLINE))
  {
    if (remove_tkline_match(khost, kuser))
    {
      sendto_one(source_p, ":%s NOTICE %s :UNKLINE: %s@%s",
                 me.name, source_p->name, kuser, khost);
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "%s removed K-Line %s@%s",
                           get_oper_name(source_p), kuser, khost);
      ilog(NOTICE, "%s removed K-Line %s@%s",
           source_p->name, kuser, khost);
      return;
    }

    if (remove_conf_line(KLINE_TYPE, source_p, kuser, khost) > 0)
    {
      sendto_one(source_p, ":%s NOTICE %s :UNKLINE: %s@%s",
                 me.name, source_p->name, kuser, khost);
      sendto_realops_flags(UMODE_ALL, L_ALL, "%s removed K-Line %s@%s",
                           get_oper_name(source_p), kuser, khost);
      ilog(NOTICE, "%s removed K-Line %s@%s", source_p->name, kuser, khost);
    }
    else
      sendto_one(source_p, ":%s NOTICE %s :UNKLINE: %s@%s not found",
                 me.name, source_p->name, kuser, khost);
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :UNKLINE: %s@%s not found",
               me.name, source_p->name, kuser, khost);
}

/* Input: A hostname, a username to unkline.
 * Output: returns YES on success, NO if no tkline removed.
 * Side effects: Any matching tklines are removed.
 */
static int
remove_tkline_match(const char *host, const char *user)
{
  struct AccessItem *tk_c;
  dlink_node *tk_n;
  struct irc_ssaddr addr, caddr;
  int nm_t, cnm_t, bits, cbits;

  nm_t = parse_netmask(host, &addr, &bits);

  DLINK_FOREACH(tk_n, temporary_klines.head)
  {
    tk_c = (struct AccessItem*)tk_n->data;
    cnm_t = parse_netmask(tk_c->host, &caddr, &cbits);
    if (cnm_t != nm_t || irccmp(user, tk_c->user))
      continue;
    if ((nm_t==HM_HOST && !irccmp(tk_c->host, host)) ||
        (nm_t==HM_IPV4 && bits==cbits && match_ipv4(&addr, &caddr, bits))
#ifdef IPV6
        || (nm_t==HM_IPV6 && bits==cbits && match_ipv6(&addr, &caddr, bits))
#endif
        )
    {
      dlinkDelete(tk_n, &temporary_klines);
      free_dlink_node(tk_n);
      delete_one_address_conf(tk_c->host, tk_c);
      return(1);
    }
  }
  return(0);
}

static int
remove_tdline_match(const char *cidr)
{
  struct AccessItem *td_conf;
  dlink_node *td_node;
  struct irc_ssaddr addr, caddr;
  int nm_t, cnm_t, bits, cbits;

  nm_t = parse_netmask(cidr, &addr, &bits);

  DLINK_FOREACH(td_node, temporary_dlines.head)
  {
    td_conf = (struct AccessItem *)td_node->data;
    cnm_t = parse_netmask(td_conf->host, &caddr, &cbits);

    if (cnm_t != nm_t)
      continue;

    if ((nm_t==HM_HOST && !irccmp(td_conf->host, cidr)) ||
       (nm_t==HM_IPV4 && bits==cbits && match_ipv4(&addr, &caddr, bits))
#ifdef IPV6
       || (nm_t==HM_IPV6 && bits==cbits && match_ipv6(&addr, &caddr, bits))
#endif
      )
    {
      dlinkDelete(td_node, &temporary_dlines);
      free_dlink_node(td_node);
      delete_one_address_conf(td_conf->host, td_conf);
      return(1);
    }
  }
  return(0);
}

/* parv[0] = sender nick
 * parv[1] = dline to remove
 */
static void
mo_undline(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  const char *cidr;

  if (!IsOperDline(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
	       me.name, source_p->name);
    return;
  }

  cidr = parv[1];

  if (remove_tdline_match(cidr))
  {
    sendto_one(source_p, ":%s NOTICE %s :UNDLINE: %s",
              me.name, source_p->name, cidr);
    sendto_realops_flags(UMODE_ALL, L_ALL, "%s removed D-Line %s",
                         get_oper_name(source_p), cidr);
    ilog(NOTICE, "%s removed D-Line %s", source_p->name, cidr);
    return;
  }

  if (remove_conf_line(DLINE_TYPE, source_p, cidr, NULL) > 0)
  {
    sendto_one(source_p, ":%s NOTICE %s :UNDLINE: %s",
               me.name, source_p->name, cidr);
    sendto_realops_flags(UMODE_ALL, L_ALL, "%s removed D-Line %s",
                         get_oper_name(source_p), cidr);
    ilog(NOTICE, "%s removed D-Line %s", source_p->name, cidr);
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :UNDLINE: %s not found",
               me.name, source_p->name, cidr);
}

static void
mo_close(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  struct Client *target_p;
  dlink_node *ptr, *ptr_next;
  unsigned int closed = 0;

  DLINK_FOREACH_SAFE(ptr, ptr_next, unknown_list.head)
  {
    target_p = ptr->data;

    sendto_one(source_p, form_str(RPL_CLOSING), me.name, parv[0],
               get_client_name(target_p, SHOW_IP), target_p->status);
    /* exit here is safe, because it is guaranteed not to be source_p
     * because it is unregistered and source_p is an oper.
     */
    exit_client(target_p, target_p, target_p, "Connection Closed");
    closed++;
  }
  sendto_one(source_p, form_str(RPL_CLOSEEND), 
	     me.name, source_p->name, closed);
  sendto_realops_flags(UMODE_ALL, L_ALL,
		       "%s issued CLOSE, %d connection%s closed",
		       source_p->name, closed, (closed == 1) ? "" : "s");
}

/* inputs       - pointer to server
 *              - pointer to client
 *              - parameter count
 *              - parameter list
 * side effects - x line is added
 */
static void
mo_xline(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  struct ConfItem *conf;
  struct MatchItem *match_item;
  char *reason, *target_server = NULL;

  reason = '\0';

  if (!IsOperX(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  /* XLINE <gecos> ON <server> :reason */
  if (!irccmp(parv[2], "ON"))
  {
    if (parc < 5)
    {
      sendto_one(source_p, form_str(ERR_NEEDMOREPARAMS),
                 me.name, source_p->name, "XLINE");
      return;
    }
    else
    {
      target_server = parv[3];
      reason = parv[4];
    }
  }
  else
    reason = parv[2];

  if (!valid_xline(source_p, parv[1], reason))
    return;

  if (target_server != NULL)
  {
    sendto_match_servs(source_p, target_server, "XLINE %s %s :%s",
                       target_server, parv[1], reason);

    if (!match(target_server, me.name))
      return;
  }

  if ((conf = find_matching_name_conf(XLINE_TYPE, parv[1], NULL, NULL, 0)) != NULL)
  {
    match_item = (struct MatchItem *)map_to_conf(conf);

    sendto_one(source_p, ":%s NOTICE %s :XLINE: %s already X-Lined by %s [%s]",
               me.name, source_p->name, parv[1],
               conf->name, match_item->reason);
    return;
  }
  write_xline(source_p, parv[1], reason);
}

/* inputs       - oper, target server, xline, type, reason
 * side effects - propagates xline, applies it if we are a target
 */
static void
ms_xline(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  struct ConfItem *conf;
  struct MatchItem *match_item;

  if (parc != 4)
    return;

  if (!IsPerson(source_p))
    return;

  if (!valid_xline(source_p, parv[2], parv[3]))
    return;

  sendto_match_servs(source_p, parv[1], "XLINE %s %s :%s",
                     parv[1], parv[2], parv[3]);

  if (!match(parv[1], me.name))
    return;

  if (find_matching_name_conf(ULINE_TYPE, source_p->user->server->name,
                   source_p->username, source_p->host, SHARED_XLINE))
  {
    if ((conf = find_matching_name_conf(XLINE_TYPE, parv[2],
					NULL, NULL, 0)) != NULL)
    {
      match_item = (struct MatchItem *)map_to_conf(conf);
      sendto_one(source_p, ":%s NOTICE %s :XLINE: %s already X-Lined by %s [%s]",
                 me.name, source_p->name, parv[2],
		 conf->name, match_item->reason);
      return;
    }
    write_xline(source_p, parv[2], parv[3]);
  }
}

/* inputs       - pointer to server
 *              - pointer to client
 *              - parameter count
 *              - parameter list
 * side effects - removes a xline
 */
static void
mo_unxline(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  if (!IsOperX(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  /* UNXLINE bill ON irc.server.com */
  if ((parc > 3) && (!irccmp(parv[2], "ON")))
  {
    sendto_match_servs(source_p, parv[3], "UNXLINE %s %s",
                       parv[3], parv[1]);

    if (!match(parv[3], me.name))
      return;
  }

  if (remove_conf_line(XLINE_TYPE, source_p, parv[1], NULL) > 0)
  {
    sendto_one(source_p, ":%s NOTICE %s :UNXLINE: %s",   
               me.name, source_p->name, parv[1]);
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "%s removed X-Line %s", get_oper_name(source_p), parv[1]);
    ilog(NOTICE, "%s removed X-Line %s", source_p->name, parv[1]);
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :XLINE: %s not found",
               me.name, source_p->name, parv[1]);
}

/* inputs       - oper, target server, gecos
 * side effects - propagates unxline, applies it if we are a target
 */
static void
ms_unxline(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  if (parc != 3)
    return;

  if (EmptyString(parv[2]))
    return;

  sendto_match_servs(source_p, parv[1], "UNXLINE %s %s",
                     parv[1], parv[2]);

  if (!match(parv[1], me.name) || !IsPerson(source_p))
    return;

  if (find_matching_name_conf(ULINE_TYPE, source_p->user->server->name,
                   source_p->username, source_p->host, SHARED_XLINE))
  {
    if (remove_conf_line(XLINE_TYPE, source_p, parv[2], NULL) > 0)
    {
      sendto_one(source_p, ":%s NOTICE %s :UNXLINE: %s",
                 me.name, source_p->name, parv[2]);
      sendto_realops_flags(UMODE_ALL, L_ALL, "%s removed X-Line %s",
                           get_oper_name(source_p), parv[2]);
      ilog(NOTICE, "%s removed X-Line %s", source_p->name,
           parv[2]);
    }
    else
      sendto_one(source_p, ":%s NOTICE %s :UNXLINE: %s not found",
                 me.name, source_p->name, parv[2]);
  }
}

/* inputs       - client to complain to, gecos, reason, whether to complain
 * outputs      - 1 for valid, else 0
 */
static int
valid_xline(struct Client *source_p, char *gecos, char *reason)
{
  if (strchr(reason, ':') != NULL)
  {
    sendto_one(source_p, ":%s NOTICE %s :XLINE: Invalid character ':'",
               me.name, source_p->name);
    return(0);
  }

  if (!valid_wild_card_simple(gecos))
  {
    sendto_one(source_p, ":%s NOTICE %s :XLINE: Less than %d non-wildcard characters",
               me.name, source_p->name, ConfigFileEntry.min_nonwildcard_simple);

    return(0);
  }
  return(1);
}

/* inputs       - client taking credit for xline, gecos, reason, xline type
 * side effects - when succesful, adds an xline to the conf
 */
static void
write_xline(struct Client *source_p, char *gecos, char *reason)
{
  struct ConfItem *conf;
  struct MatchItem *match_item;
  const char *current_date;

  conf = make_conf_item(XLINE_TYPE);
  match_item = (struct MatchItem *)map_to_conf(conf);
  collapse(gecos);
  DupString(conf->name, gecos);
  DupString(match_item->reason, reason);
  DupString(match_item->oper_reason, "");
  set_time();
  current_date = smalldate(CurrentTime);
  write_conf_line(source_p, conf, current_date, CurrentTime);
  rehashed_xlines = 1;
}

/* parv[0] = sender prefix
 * parv[1] = channel to jupe
 */
static void
mo_cjupe(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  char *reason;

  if (!IsChanPrefix(*parv[1]))
  {
    sendto_one(source_p, ":%s NOTICE %s :CJUPE: Invalid channel %s",
               me.name, source_p->name, parv[1]);
    return;
  }

  /* CJUPE #channel ON irc.server.com :abuse */
  if ((parc > 3) && (!irccmp(parv[2], "ON")))
  {
    if (parc < 5)
    {
      if (!irccmp(parv[2], "ON"))
      sendto_one(source_p, form_str(ERR_NEEDMOREPARAMS),
                 me.name, source_p->name, "CJUPE");
      return;
    }
    reason = parv[4];
    sendto_match_servs(source_p, parv[3], "CJUPE %s %s :%s", parv[3], parv[1], reason);

    if (!match(parv[3], me.name))
      return;

    parse_jupe(source_p, parv[1], reason);
  }
  else
  {
    /* CJUPE #channel :abuse */
    reason = parv[2];
    parse_jupe(source_p, parv[1], reason);
  }
}

/* parv[0] = sender prefix
 * parv[1] = nick to jupe
 */
static void
mo_njupe(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  char *reason;

  if (!clean_jupe_nick(parv[1]))
  {
    sendto_one(source_p, ":%s NOTICE %s :NJUPE: Invalid nick %s",
               me.name, source_p->name, parv[1]);
    return;
  }

  /* NJUPE nick ON irc.server.com :abuse */
  if ((parc > 3) && (!irccmp(parv[2], "ON")))
  {
    if (parc < 5)
    {
      if (!irccmp(parv[2], "ON"))
      sendto_one(source_p, form_str(ERR_NEEDMOREPARAMS),
                 me.name, source_p->name, "NJUPE");
      return;
    }
    reason = parv[4];
    sendto_match_servs(source_p, parv[3], "NJUPE %s %s :%s", parv[3], parv[1], reason);

    if (!match(parv[3], me.name))
      return;

    parse_jupe(source_p, parv[1], reason);
  }
  else
  {
    /* NJUPE nick :abuse */
    reason = parv[2];
    parse_jupe(source_p, parv[1], reason);
  }
}

/* parv[0] = sender prefix
 * parv[1] = target server
 * parv[2] = channel to jupe
 * parv[3] = reason
 */
static void
ms_cjupe(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  if ((parc != 4) || EmptyString(parv[3]) || !IsPerson(source_p) ||
      !IsChanPrefix(*parv[2]))
    return;

  sendto_match_servs(source_p, parv[1], "CJUPE %s %s :%s",
                     parv[1], parv[2], parv[3]);

  if (!match(parv[1], me.name))
    return;

  if (find_matching_name_conf(ULINE_TYPE, source_p->user->server->name,
                   source_p->username, source_p->host, SHARED_JUPE))
    parse_jupe(source_p, parv[2], parv[3]);
}

/* parv[0] = sender prefix
 * parv[1] = target server
 * parv[2] = nick to jupe
 * parv[3] = reason
 */
static void
ms_njupe(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  if ((parc != 4) || EmptyString(parv[3]) || !IsPerson(source_p) || 
      !clean_jupe_nick(parv[2]))
    return;

  sendto_match_servs(source_p, parv[1], "NJUPE %s %s :%s",
                     parv[1], parv[2], parv[3]);

  if (!match(parv[1], me.name))
    return;

  if (find_matching_name_conf(ULINE_TYPE, source_p->user->server->name,
                   source_p->username, source_p->host, SHARED_JUPE))
    parse_jupe(source_p, parv[2], parv[3]);
}

/* parv[0] = sender prefix
 * parv[1] = channel to unjupe
 */
static void
mo_uncjupe(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  if (!IsChanPrefix(*parv[1]))
  {
    sendto_one(source_p, ":%s NOTICE %s :UNCJUPE: Invalid channel %s",
               me.name, source_p->name, parv[1]);
    return;
  }

  /* UNCJUPE #channel ON irc.server.com */
  if ((parc > 3) && (!irccmp(parv[2], "ON")))
  {
    sendto_match_servs(source_p, parv[3], "UNCJUPE %s %s",
                       parv[3], parv[1]);
    if (!match(parv[3], me.name))
      return;
  }
  remove_jupe(source_p, parv[1]);
}

/* parv[0] = sender prefix
 * parv[1] = nick to unjupe
 */
static void
mo_unnjupe(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  if (!clean_jupe_nick(parv[1]))
  {
    sendto_one(source_p, ":%s NOTICE %s :UNNJUPE: Invalid nick %s",
               me.name, source_p->name, parv[1]);
    return;
  }

  /* UNNJUPE nick ON irc.server.com */
  if ((parc > 3) && (!irccmp(parv[2], "ON")))
  {
    sendto_match_servs(source_p, parv[3], "UNNJUPE %s %s",
                       parv[3], parv[1]);
    if (!match(parv[3], me.name))
      return;
  }
  remove_jupe(source_p, parv[1]);
}

/* parv[0] = sender prefix
 * parv[1] = target server
 * parv[2] = jupe to remove
 */
static void
ms_uncjupe(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  if (parc != 3 || EmptyString(parv[2]) || !IsPerson(source_p) ||
      !IsChanPrefix(*parv[2]))
    return;

  sendto_match_servs(source_p, parv[1], "UNCJUPE %s %s",
                     parv[1], parv[2]);

  if (!match(parv[1], me.name))
    return;

  if (find_matching_name_conf(ULINE_TYPE, source_p->user->server->name,
                   source_p->username, source_p->host, SHARED_JUPE))
    remove_jupe(source_p, parv[2]);
}

/* parv[0] = sender prefix
 * parv[1] = target server
 * parv[2] = jupe to remove
 */
static void
ms_unnjupe(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  if (parc != 3 || EmptyString(parv[2]) || !IsPerson(source_p) ||
      !clean_jupe_nick(parv[2]))
    return;

  sendto_match_servs(source_p, parv[1], "UNNJUPE %s %s",
                     parv[1], parv[2]);

  if (!match(parv[1], me.name))
    return;

  if (find_matching_name_conf(ULINE_TYPE, source_p->user->server->name,
                   source_p->username, source_p->host, SHARED_JUPE))
    remove_jupe(source_p, parv[2]);
}


/* inputs       - source_p, NULL supported
 *              - thing to jupe
 *              - reason  
 * side effects - parse jupe, create if valid
 */
static void
parse_jupe(struct Client *source_p, char *name, char *reason)
{
  struct ConfItem *conf;

  if (IsChanPrefix(*name))
  {
    struct JupeChannel *jupe_p;

    if ((conf = create_channel_jupe(name, reason, 0)) == NULL)
    {
      sendto_one(source_p, ":%s NOTICE %s :*** %s already juped",
                 me.name, source_p->name, name);
      return;
    }
    jupe_p = (struct JupeChannel *)map_to_conf(conf);
    sendto_one(source_p, ":%s NOTICE %s :*** Juped %s [%s]",
               me.name, source_p->name, jupe_p->name, jupe_p->reason);
    sendto_realops_flags(UMODE_ALL, L_ALL, "%s juped %s [%s]",
                         get_oper_name(source_p), jupe_p->name, jupe_p->reason);
    ilog(TRACE, "%s juped %s [%s]", source_p->name, jupe_p->name, jupe_p->reason);
    write_conf_line(source_p, conf, NULL, 0);
  }
  else if (clean_jupe_nick(name))
  {
    struct MatchItem *jupe_p;

    if ((strchr(name, '*') || strchr(name, '?')) && !IsAdmin(source_p))
    {
      sendto_one(source_p, ":%s NOTICE %s :*** You need admin = yes;",
                 me.name, source_p->name);
      return;
    }

    if ((conf = create_nick_jupe(name, reason, 0)) == NULL)
    {
      sendto_one(source_p, ":%s NOTICE %s :*** %s already juped",
                 me.name, source_p->name, name);
      return;
    }
    jupe_p = (struct MatchItem *)map_to_conf(conf);
    sendto_one(source_p, ":%s NOTICE %s :*** Juped %s [%s]",
               me.name, source_p->name, conf->name, jupe_p->reason);
    sendto_realops_flags(UMODE_ALL, L_ALL, "%s juped %s [%s]",
                         get_oper_name(source_p), conf->name, jupe_p->reason);
    ilog(TRACE, "%s juped %s [%s]",
         source_p->name, conf->name, jupe_p->reason);
    write_conf_line(source_p, conf, NULL, 0);
  }
  else 
    sendto_one(source_p, ":%s NOTICE %s :*** Invalid jupe %s",
               me.name, source_p->name, name);
}

static void
remove_jupe(struct Client *source_p, char *name)
{
  struct ConfItem *conf;
                 
  if (IsChanPrefix(*name))
  {  
    struct JupeChannel *jupe_p;
    
    if (jupe_channel_list.head == NULL || !(jupe_p = hash_find_jupe(name)))
    {
      sendto_one(source_p, ":%s NOTICE %s :UNCJUPE: %s not found",
                 me.name, source_p->name, name);
      return;
    }
    else if (jupe_p->conf)
    {
      sendto_one(source_p, ":%s NOTICE %s :UNCJUPE: %s must be removed from ircd.conf.",
                 me.name, source_p->name, name);
      return;
    }
    else
    {
      delete_channel_jupe(jupe_p);
      remove_conf_line(CJUPE_TYPE, source_p, name, NULL);
      sendto_one(source_p, ":%s NOTICE %s :UNCJUPE: %s",
                 me.name, source_p->name, name);
      sendto_realops_flags(UMODE_ALL, L_ALL, "%s UNCJUPE %s",
                           get_oper_name(source_p), name);
      ilog(NOTICE, "%s UNCJUPE %s", source_p->name, name);

    }
  }
  else if (clean_jupe_nick(name))
  {
    struct MatchItem *jupe_p;
    conf = find_matching_name_conf(NJUPE_TYPE, name, NULL, NULL, 0);
                 
    if (conf == NULL)
    {
      sendto_one(source_p, ":%s NOTICE %s :UNNJUPE: %s not found",
                 me.name, source_p->name, name);
      return;
    }
    jupe_p = (struct MatchItem *)map_to_conf(conf);
    if (jupe_p->action)
    {
      sendto_one(source_p, ":%s NOTICE %s :UNNJUPE %s must be removed from ircd.conf",
                 me.name, source_p->name, name);
      return;
    }
    else
    {
      delete_conf_item(conf);
      remove_conf_line(NJUPE_TYPE, source_p, name, NULL);

      sendto_one(source_p, ":%s NOTICE %s :UNNJUPE: %s",
                 me.name, source_p->name, name);
      sendto_realops_flags(UMODE_ALL, L_ALL, "%s UNNJUPE %s",
                           get_oper_name(source_p), name);
      ilog(NOTICE, "%s UNNJUPE %s", source_p->name, name);
    }
  }
}
