/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * m_knock.c: Requests to be invited to a channel.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_knock.c,v 1.34 2004/02/08 06:46:04 musirc Exp $
 */

#include "handlers.h"
#include "client.h"
#include "channel.h"
#include "channel_mode.h"
#include "hash.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "numeric.h"
#include "send.h"
#include "config.h"
#include "modules.h"
#include "server.h"

static void m_knock(struct Client *, struct Client *, int, char **);
static void ms_knock(struct Client *, struct Client *, int, char **);
static void send_knock(struct Client *, struct Client *,
                       struct Channel *, char *, char *, int);
static int is_banned_knock(struct Channel *, struct Client *, char *);
static int check_banned_knock(struct Channel *, char *, char *);

struct Message knock_msgtab = {
  "KNOCK", 2, MFLG_SLOW,
  {m_unregistered, m_knock, ms_knock, m_knock}
};

#ifndef STATIC_MODULES

void
_modinit(void)
{
  mod_add_cmd(&knock_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&knock_msgtab);
}

const char *_version = "$Revision: 1.34 $";
#endif

/* parv[0] = sender prefix
 * parv[1] = channel
 */
static void
m_knock(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  char *sockhost = NULL;
  struct Channel *chptr;
  char *p, *name, *key;

  if (!IsClient(source_p))
    return;

  if (!MyConnect(source_p))
  {
    if(!ServerInfo.hub)
      return;
    else
    {
      sockhost = parv[2];
      
      if (parc > 3)
      {
        parv[2] = parv[3];
        parv[3] = NULL;
      }
      else
        parv[2] = NULL;

      parc--;
    }
  }
  name = parv[1];
  key = (parc > 2) ? parv[2] : NULL;

  if ((p = strchr(name,',')) != NULL)
    *p = '\0';

  if (!IsChanPrefix(*name))
  {
    sendto_one(source_p, form_str(ERR_NOSUCHCHANNEL),
               me.name, source_p->name, name);
    return;
  }

  if ((chptr = hash_find_channel(name)) == NULL)
  {
    sendto_one(source_p, form_str(ERR_NOSUCHCHANNEL),
               me.name, source_p->name, name);
    return;
  }
  /* Normal channel, just be sure they aren't on it */
  if (IsMember(source_p, chptr))
  {
    sendto_one(source_p, form_str(ERR_KNOCKONCHAN),
               me.name, source_p->name, name);
    return;
  }

  if (!((chptr->mode.mode & MODE_INVITEONLY) || (*chptr->mode.key) ||
     (chptr->mode.limit && dlink_list_length(&chptr->members) >=
      chptr->mode.limit)))
  {
    sendto_one(source_p, form_str(ERR_CHANOPEN),
               me.name, source_p->name, name);
    return;
  }

  /* don't allow a knock if the user is banned, or the channel is secret */
  if ((chptr->mode.mode & MODE_PRIVATE) ||
      (sockhost && is_banned_knock(chptr, source_p, sockhost)) ||
      (!sockhost && is_banned(chptr, source_p)))
  {
    sendto_one(source_p, form_str(ERR_CANNOTSENDTOCHAN),
               me.name, source_p->name, name);
    return;
  }

  if (MyClient(source_p) &&
    (source_p->localClient->last_knock + ConfigChannel.knock_delay) >
    CurrentTime)
  {
    sendto_one(source_p, form_str(ERR_TOOMANYKNOCK),
               me.name, source_p->name, parv[1], "user");
    return;
  }
  else if (chptr->last_knock + ConfigChannel.knock_delay_channel > CurrentTime)
  {
    sendto_one(source_p, form_str(ERR_TOOMANYKNOCK),
               me.name, source_p->name, parv[1], "channel");
    return;
  }
  send_knock(client_p, source_p, chptr, name, key,
             MyClient(source_p) ? 0 : 1);    
}

/* parv[0] = sender prefix
 * parv[1] = channel
 */
static void
ms_knock(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  struct Channel *chptr;
  char *p, *name, *key;

  if (!IsClient(source_p))
    return;

  name = parv[1];
  key = parv[2];

  if ((p = strchr(name,',')) != NULL)
    *p = '\0';

  if (!IsChanPrefix(*name) || (chptr = hash_find_channel(name)) == NULL)
    return;

  if (IsMember(source_p, chptr))
    return;

  if (!((chptr->mode.mode & MODE_INVITEONLY) || (*chptr->mode.key) ||
     (chptr->mode.limit && dlink_list_length(&chptr->members) >=
      chptr->mode.limit)))
    return;

  if (chptr)
    send_knock(client_p, source_p, chptr, name, key, 0);
}

/* input        - pointer to physical struct client_p
 *              - pointer to source struct source_p
 *              - pointer to channel struct chptr
 *              - pointer to base channel name
 * side effects - knock is sent locally (if enabled) and propagated
 */
static void
send_knock(struct Client *client_p, struct Client *source_p,
	   struct Channel *chptr, char *name, char *key, int llclient)
{
  chptr->last_knock = CurrentTime;

  if (MyClient(source_p))
  {
    source_p->localClient->last_knock = CurrentTime;

    sendto_one(source_p, form_str(RPL_KNOCKDLVR),
               me.name, source_p->name, name);
  }
  else if (llclient == 1)
    sendto_one(source_p, form_str(RPL_KNOCKDLVR),
               me.name, source_p->name, name);

  if (source_p->user != NULL)
  {
    sendto_channel_local(CHFL_CHANOP|CHFL_HALFOP,
			 chptr, form_str(RPL_KNOCK),
			 me.name, name, name,
			 source_p->name, source_p->username,
			 source_p->host);
      
    sendto_server(client_p, chptr, ":%s KNOCK %s %s",
	          source_p->name, name, key != NULL ? key : "");
  }
}

/* input	- pointer to channel
 *		- pointer to client
 *		- clients sockhost
 * side effects - return check_banned_knock()
 */
static int
is_banned_knock(struct Channel *chptr, struct Client *who, char *sockhost)
{
  char src_host[NICKLEN + USERLEN + HOSTLEN + 6];
  char src_iphost[NICKLEN + USERLEN + HOSTLEN + 6];

  if (!IsPerson(who))
    return(0);

  ircsprintf(src_host, "%s!%s@%s", who->name, who->username, who->host);
  ircsprintf(src_iphost, "%s!%s@%s", who->name, who->username, sockhost);

  return(check_banned_knock(chptr, src_host, src_iphost));
}

/* input	- pointer to channel
 * 		- pointer to client
 *		- preformed nick!user@host
 *		- preformed nick!user@ip
 * side effects - return CHFL_EXCEPTION, CHFL_BAN or 0
 */
static int
check_banned_knock(struct Channel *chptr, char *s, char *s2)
{
  dlink_node *ban, *except;
  struct Ban *actualBan = NULL, *actualExcept = NULL;

  DLINK_FOREACH(ban, chptr->banlist.head)
  {
    actualBan = ban->data;

    if (match(actualBan->banstr, s) || match(actualBan->banstr, s2))
      break;
    else
      actualBan = NULL;
  }

  if (actualBan != NULL)
  {
    DLINK_FOREACH(except, chptr->exceptlist.head)
    {
      actualExcept = except->data;

      if (match(actualExcept->banstr, s) || match(actualExcept->banstr, s2))
	return(CHFL_EXCEPTION);
    }
  }

  return((actualBan ? CHFL_BAN : 0));
}
