/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * m_oper.c: Makes a user an IRC Operator.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_oper.c,v 1.49.2.2 2004/07/22 20:31:25 musirc Exp $
 */

#include "handlers.h"
#include "client.h"
#include "istring.h"
#include "ircd.h"
#include "numeric.h"
#include "config.h"
#include "log.h"
#include "user.h"
#include "send.h"
#include "modules.h"
#include "packet.h"
#include "server.h"

static struct ConfItem *find_password_conf(const char *, struct Client *);
static int match_oper_password(const char *, struct AccessItem *);
static void failed_oper_notice(struct Client *, const char *, const char *);
static void m_oper(struct Client *, struct Client *, int, char **);
static void mo_oper(struct Client *, struct Client *, int, char **);

struct Message oper_msgtab = {
  "OPER", 3, MFLG_SLOW,
  {m_unregistered, m_oper, m_ignore, mo_oper}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&oper_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&oper_msgtab);
}

const char *_version = "$Revision: 1.49.2.2 $";
#endif

/* parv[0] = sender prefix
 * parv[1] = oper name
 * parv[2] = oper password
 */
static void
m_oper(struct Client *client_p, struct Client *source_p, int parc, char *parv[])
{
  struct ConfItem *conf;
  struct AccessItem *aconf=NULL;
  const char *name = parv[1], *password = parv[2];

  if (!IsFloodDone(source_p))
    flood_endgrace(source_p);

  if ((conf = find_password_conf(name,source_p)) == NULL)
  {
    sendto_one(source_p, form_str(ERR_NOOPERHOST), me.name, source_p->name);
    conf = find_exact_name_conf(OPER_TYPE, name, NULL, NULL);
    failed_oper_notice(source_p, name, (conf != NULL) ?
                       "host mismatch" : "no oper block");
    log_failed_oper(source_p, name);
    return;
  }
  aconf = (struct AccessItem *)map_to_conf(conf);

  if (match_oper_password(password, aconf))
  {
    if (attach_conf(source_p, conf) != 0)
    {
      sendto_one(source_p, ":%s NOTICE %s :OPER: Can't attach conf!",
                 me.name, source_p->name);
      failed_oper_notice(source_p, name, "Can't attach conf!");
      log_failed_oper(source_p, name);
      return;
    }
    sendto_server(client_p, NULL, ":%s WALLOPS :Oper %s by %s (%s@%s)", me.name,
                  name, source_p->name, source_p->username, source_p->host);
    oper_up(source_p);
    ilog(TRACE, "OPER %s by %s!%s@%s",
         name, source_p->name, source_p->username, source_p->host);
    log_oper(source_p, name);
  }
  else
  {
    sendto_one(source_p, form_str(ERR_PASSWDMISMATCH), me.name, parv[0]);
    failed_oper_notice(source_p, name, "password mismatch");
    log_failed_oper(source_p, name);
  }
}

static void
mo_oper(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  sendto_one(source_p, form_str(RPL_YOUREOPER), me.name, source_p->name);
}

static struct ConfItem *
find_password_conf(const char *name, struct Client *source_p)
{
  struct ConfItem *conf;

  if ((conf = find_exact_name_conf(OPER_TYPE,
                                   name, source_p->username, source_p->host
                                   )) != NULL)
    return(conf);
  else
  if ((conf = find_exact_name_conf(OPER_TYPE,
                                   name, source_p->username,
                                   source_p->localClient->sockhost)) != NULL)
    return(conf);

  return(NULL);
}

/* inputs       - pointer to given password
 *              - pointer to Conf 
 * output       - YES or NO if match
 */
static int
match_oper_password(const char *password, struct AccessItem *aconf)
{
  const char *encr = NULL;

  if (!IsConfOperator(aconf))
    return(0);

  if (ConfigFileEntry.crypt_oper_password)
  {
    /* use first two chars of the password they send in as salt
     * If the password in the conf is MD5, and ircd is linked
     * to scrypt on FreeBSD, or the standard crypt library on
     * glibc Linux, then this code will work fine on generating
     * the proper encrypted hash for comparison.
     * passwd may be NULL pointer. Head it off at the pass... */
    if (aconf->passwd == NULL)
      return(0);

    if (password && *aconf->passwd)
      encr = crypt(password, aconf->passwd);
    else
      encr = "";
  }
  else
    encr = password;

  if (!strcmp(encr, aconf->passwd))
    return(1);
  else
    return(0);
}

/* inputs       - pointer to client doing /oper ...
 *              - pointer to nick they tried to oper as
 *              - pointer to reason they have failed
 * side effects - notices all opers of the failed oper attempt if enabled
 */
static void
failed_oper_notice(struct Client *source_p, const char *name, const char *reason)
{
  sendto_realops_flags(UMODE_ALL, L_ALL, "Failed OPER attempt as %s "
                       "by %s (%s@%s) - %s", name, source_p->name,
                       source_p->username, source_p->host, reason);
}
