/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * m_set.c: Sets a server parameter.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_set.c,v 1.32.2.1 2004/07/22 20:31:25 musirc Exp $
 */

#include "handlers.h"
#include "client.h"
#include "event.h"
#include "istring.h"
#include "ircd.h"
#include "fdlist.h"
#include "numeric.h"
#include "server.h"
#include "send.h"
#include "channel.h"
#include "log.h"
#include "config.h"
#include "modules.h"

static void mo_set(struct Client *, struct Client *, int, char **);

struct Message set_msgtab = {
  "SET", 0, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_set}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&set_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&set_msgtab);
}

const char *_version = "$Revision: 1.32.2.1 $";
#endif

struct SetStruct
{
  const char *name;
  void (*handler)();
  int wants_char; /* 1 if it expects (char *, [int]) */
  int wants_int;  /* 1 if it expects ([char *], int) */
  /* eg:  0, 1 == only an int arg
   * eg:  1, 1 == char and int args */
};

static void autoconn(struct Client *, const char *, int);
static void autoconnall(struct Client *, int);
static void floodcount(struct Client *, int);
static void identtimeout(struct Client *, int);
static void idletime(struct Client *, int);
static void setlog(struct Client *, int);
static void max(struct Client *, int);
static void setlocale(struct Client *, char *);
static void setsplitmode(struct Client *, char *);
static void splitnum(struct Client *, int);
static void splitusers(struct Client *, int);
static void list_commands(struct Client *);

static struct SetStruct set_cmd_table[] =
{
  /* name		function   string arg  int arg */
  { "AUTOCONN",		autoconn,	1,	1 },
  { "AUTOCONNALL",	autoconnall,	0,	1 },
  { "FLOODCOUNT",	floodcount,	0,	1 },
  { "IDENTTIMEOUT",	identtimeout,	0,	1 },
  { "IDLETIME",		idletime,	0,	1 },
  { "LOG",		setlog,		0,	1 },
  { "MAX",		max,		0,	1 },
  { "LOCALE",		setlocale,	1,	0 },
  { "SPLITMODE",	setsplitmode,	1,	0 },
  { "SPLITNUM",		splitnum,	0,	1 },
  { "SPLITUSERS",	splitusers,	0,	1 },
  { (char *)0,		(void(*)()) 0,	0,	0 }
};

static void
list_commands(struct Client *source_p)
{
  int i, j = 0;
  const char *names[4];

  names[0] = names[1] = names[2] = names[3] = "";

  for (i = 0; set_cmd_table[i].handler; i++)
  {
    names[j++] = set_cmd_table[i].name;

    if (j > 3)
    {
      sendto_one(source_p, ":%s NOTICE %s :SET: Options: %s %s %s %s",
                 me.name, source_p->name, names[0], names[1], 
                 names[2], names[3]);
      j = 0;
      names[0] = names[1] = names[2] = names[3] = "";
    }

  }
  if (j)
    sendto_one(source_p, ":%s NOTICE %s :SET: Options: %s %s %s %s",
               me.name, source_p->name, names[0], names[1], 
               names[2], names[3]);
}

static void
autoconn(struct Client *source_p, const char *arg, int newval)
{
  set_autoconn(source_p, arg, newval);
}

static void
autoconnall(struct Client *source_p, int newval)
{
  if (newval >= 0)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "%s has changed AUTOCONNALL to %i",
                         source_p->name, newval);

    GlobalSetOptions.autoconn = newval;
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :AUTOCONNALL: currently %i",
               me.name, source_p->name, GlobalSetOptions.autoconn);
}

static void
floodcount( struct Client *source_p, int newval)
{
  if(newval >= 0)
  {
    GlobalSetOptions.floodcount = newval;
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "%s has changed FLOODCOUNT to %i", source_p->name,
                         GlobalSetOptions.floodcount);
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :FLOODCOUNT: currently %i",
               me.name, source_p->name, GlobalSetOptions.floodcount);
}

static void
identtimeout(struct Client *source_p, int newval)
{
  if (!IsAdmin(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
	       me.name, source_p->name);
    return;
  }

  if(newval > 0)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,
		         "%s has changed IDENTTIMEOUT to %d",
			 get_oper_name(source_p), newval);
    GlobalSetOptions.ident_timeout = newval;
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :IDENTTIMEOUT: currently %d",
	       me.name, source_p->name, GlobalSetOptions.ident_timeout);
}

static void
idletime(struct Client *source_p, int newval)
{
  if (newval >= 0)
  {
    if (newval == 0)
    {
      sendto_realops_flags(UMODE_ALL, L_ALL, "%s has disabled idletime checking",
                           source_p->name);
      GlobalSetOptions.idletime = 0;
    }
    else
    {
      sendto_realops_flags(UMODE_ALL, L_ALL, "%s has changed IDLETIME to %i",
                           source_p->name, newval);
      GlobalSetOptions.idletime = (newval*60);
    }
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :IDLETIME: currently %i",
               me.name, source_p->name, GlobalSetOptions.idletime/60);
}

static void
setlog( struct Client *source_p, int newval )
{
  const char *log_level_as_string;

  if (newval >= 0)
  {
    if (newval < WARN)
    {
      sendto_one(source_p, ":%s NOTICE %s :LOG: must be > %d (WARN)",
                 me.name, source_p->name, WARN);
      return;
    }

    if (newval > INFO)
      newval = INFO;

    set_log_level(newval);
    log_level_as_string = get_log_level_as_string(newval);
    sendto_realops_flags(UMODE_ALL, L_ALL,"%s has changed LOG level to %i (%s)",
                         source_p->name, newval, log_level_as_string);
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :LOGLEVEL: currently %i (%s)",
               me.name, source_p->name, get_log_level(),
               get_log_level_as_string(get_log_level()));
}

static void
max(struct Client *source_p, int newval)
{
  if (newval > 0)
  {
    if (newval > MASTER_MAX)
    {
      sendto_one(source_p,
	":%s NOTICE %s :SET: Can't set MAXCLIENTS to > MASTER_MAX (%d)",
	me.name, source_p->name, MASTER_MAX);
      return;
    }

    if (newval < 32)
    {
      sendto_one(source_p,
	":%s NOTICE %s :SET: Can't set MAXCLIENTS to < 32 (%d:%d)",
	me.name, source_p->name, GlobalSetOptions.maxclients, highest_fd);
      return;
    }

    GlobalSetOptions.maxclients = newval;
    sendto_realops_flags(UMODE_ALL, L_ALL,
	"%s!%s@%s set new MAXCLIENTS to %d (%d current)",
	source_p->name, source_p->username, source_p->host,
	GlobalSetOptions.maxclients, Count.local);
    return;
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :MAXCLIENTS: currently %d (%d)",
        me.name, source_p->name, GlobalSetOptions.maxclients, Count.local);
}

static void
setlocale(struct Client *source_p, char *locale)
{
  if (locale != NULL)
  {
    set_locale(locale);
    sendto_one(source_p, ":%s NOTICE %s :LOCALE: '%s'",
               me.name, source_p->name, get_locale());
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :LOCALE: currently '%s'",
               me.name, source_p->name, get_locale());
}

/* this table is what splitmode may be set to */
static const char *splitmode_values[] =
{
  "OFF",
  "ON",
  "AUTO",
  NULL
};

static const char *splitmode_status[] =
{
  "OFF",
  "AUTO (OFF)",
  "ON",
  "AUTO (ON)",
  NULL
};

static void
setsplitmode(struct Client *source_p, char *charval)
{
  if (charval)
  {
    int newval;

    for (newval = 0; splitmode_values[newval]; newval++)
    {
      if (!irccmp(splitmode_values[newval], charval))
        break;
    }

    if (newval == 0)
    {
      sendto_realops_flags(UMODE_ALL, L_ALL, 
                           "%s is disabling splitmode",
                           get_oper_name(source_p));

      splitmode = splitchecking = 0;
      eventDelete(check_splitmode, NULL);
    }
    else if (newval == 1)
    {
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "%s is enabling and activating splitmode",
	                   get_oper_name(source_p));		 
      splitmode = 1;
      splitchecking = 0;

      /* we might be deactivating an automatic splitmode, so pull the event */
      eventDelete(check_splitmode, NULL);
    }
    /* AUTO */
    else if (newval == 2)
    {
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "%s is enabling automatic splitmode",
			   get_oper_name(source_p));

      splitchecking = 1;
      check_splitmode(NULL);
    }
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :SPLITMODE: currently %s", 
               me.name, source_p->name, 
	       splitmode_status[(splitchecking + (splitmode*2))]);
}

static void
splitnum(struct Client *source_p, int newval)
{
  if (newval >= 0)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "%s has changed SPLITNUM to %i", 
			 source_p->name, newval);
    split_servers = newval;

    if (splitchecking)
      check_splitmode(NULL);
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :SPLITNUM: currently %i", 
               me.name, source_p->name, split_servers);
}

static void
splitusers(struct Client *source_p, int newval)
{
  if (newval >= 0)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "%s has changed SPLITUSERS to %i", 
			 source_p->name, newval);
    split_users = newval;

    if (splitchecking)
      check_splitmode(NULL);
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :SPLITUSERS: currently %i", 
               me.name, source_p->name, split_users);
}

static void
mo_set(struct Client *client_p, struct Client *source_p,
       int parc, char *parv[])
{
  int i, n, newval;
  const char *arg = NULL, *intarg = NULL;

  if (parc > 1)
  {
    /* Go through all the commands in set_cmd_table, until one is
     * matched.  I realize strcmp() is more intensive than a numeric
     * lookup, but at least it's better than a big-ass switch/case
     * statement.
     */
    for (i = 0; set_cmd_table[i].handler; i++)
    {
      if (!irccmp(set_cmd_table[i].name, parv[1]))
      {
        /* Command found; now execute the code */
        n = 2;

        if (set_cmd_table[i].wants_char)
          arg = parv[n++];

        if (set_cmd_table[i].wants_int)
          intarg = parv[n++];

        if ((n - 1) > parc)
        {
          if (parc > 2)
            sendto_one(source_p,
                       ":%s NOTICE %s :SET: %s expects (\"%s%s\") args",
                       me.name, source_p->name, set_cmd_table[i].name,
                       (set_cmd_table[i].wants_char ? "string, " : ""),
                       (set_cmd_table[i].wants_char ? "int" : "")
                      );
        }

        if(parc <= 2)
        {
          arg = NULL;
          intarg = NULL;
        }

        if (!strcmp(set_cmd_table[i].name, "AUTOCONN") && (parc < 4))
        {
          sendto_one(source_p, form_str(ERR_NEEDMOREPARAMS),
                     me.name, source_p->name, "SET");
          return;
        }

        if (set_cmd_table[i].wants_int && (parc > 2))
        {
          if (intarg)
          {
            if (!irccmp(intarg, "yes") || !irccmp(intarg, "on"))
              newval = 1;
            else if (!irccmp(intarg, "no") || !irccmp(intarg, "off") )
              newval = 0;
            else
              newval = atoi(intarg);
          }
          else
            newval = -1;

          if (newval < 0)
          {
            sendto_one(source_p,
                       ":%s NOTICE %s :SET: Value less than 0 invalid for %s",
                       me.name, source_p->name,
                       set_cmd_table[i].name);

            return;
          }
        }
        else
          newval = -1;

        if (set_cmd_table[i].wants_char)
        {
          if (set_cmd_table[i].wants_int)
            set_cmd_table[i].handler(source_p, arg, newval);
          else
            set_cmd_table[i].handler(source_p, arg);
          return;
        }
        else
        {
          if (set_cmd_table[i].wants_int)
            set_cmd_table[i].handler(source_p, newval);
          else
            /* Just in case someone actually wants a
             * set function that takes no args.. *shrug* */
            set_cmd_table[i].handler(source_p);
          return;
        }
      }
    }

    /* Code here will be executed when a /QUOTE SET command is not
     * found within set_cmd_table.
     */
    sendto_one(source_p, ":%s NOTICE %s :SET: Option not found",
	       me.name, source_p->name);
    return;
  }
  list_commands(source_p);
}
