/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * m_stats.c: Sends the user statistics or config information.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_stats.c,v 1.79.2.6 2004/07/25 06:44:04 musirc Exp $
 */

#include "handlers.h"
#include "client.h"
#include "istring.h"
#include "ircd.h"
#include "hostmask.h"
#include "listener.h"
#include "numeric.h"
#include "send.h"
#include "config.h"
#include "misc.h"
#include "server.h"
#include "user.h"
#include "modules.h"
#include "jupe.h"

static void m_stats(struct Client *, struct Client *, int, char **);
static void mo_stats(struct Client *, struct Client *, int, char **);
static void ms_stats(struct Client *, struct Client *, int, char **);

struct Message stats_msgtab = {
  "STATS", 2, MFLG_SLOW,
  {m_unregistered, m_stats, ms_stats, mo_stats}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&stats_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&stats_msgtab);
}

const char *_version = "$Revision: 1.79.2.6 $";
#endif

static char *parse_args(int, char **, int *, int *);
static void L(struct Client *, char *, int, int, char);
static void L_list(struct Client *s, char *, int, int, dlink_list *, char);
static void dns_servers(struct Client *);
static void sconnect(struct Client *);
static void deny(struct Client *);
static void tdeny(struct Client *);
static void exempt(struct Client *);
static void hubleaf(struct Client *);
static void auth(struct Client *);
static void tklines(struct Client *);
static void klines(struct Client *);
static void oper(struct Client *);
static void operedup(struct Client *);
static void operlist(struct Client *);
static void ports(struct Client *);
static void jupe(struct Client *);
static void suptime(struct Client *);
static void shared(struct Client *);
static void servers(struct Client *);
static void xline(struct Client *);
static void class(struct Client *);
static void links(struct Client *);
static void sltrace(struct Client *, int, char**);

static const struct StatsStruct
{
  const unsigned char letter;
  void (*handler)();
  const unsigned int need_oper;
  const unsigned int need_admin;
} cmd_table[] = {
  /* letter     function     need_oper need_admin */
  { 'a',	dns_servers,	1,	1,	},
  { 'A',	dns_servers,	1,	1,	},
  { 'c',	sconnect,	1,	0,	},
  { 'C',	sconnect,	1,	0,	},
  { 'd',	tdeny,		1,	0,	},
  { 'D',	deny,		1,	0,	},
  { 'e',        exempt,         1,      0,      },
  { 'E', 	exempt,		1,	0,	},
  { 'h',	hubleaf,	1,	0,	},
  { 'H',	hubleaf,	1,	0,	},
  { 'i',	auth,		0,	0,	},
  { 'I',	auth,		0,	0,	},
  { 'k',	tklines,	0,	0,	},
  { 'K',	klines,		0,	0,	},
  { 'l',	sltrace,	1,	0,	},
  { 'L',	sltrace,	1,	0,	},
  { 'o',	operlist,	0,	0,	},
  { 'O',	oper,		0,	0,	},
  { 'p',	operedup,	0,	0,	},
  { 'P',	ports,		0,	0,	},
  { 'q',	jupe,		1,	0,	},
  { 'Q',	jupe,		1,	0,	},
  { 'u',	suptime,	0,	0,	},
  { 'U',	shared,		1,	0,	},
  { 'v',	servers,	1,	0,	},
  { 'x',	xline,		1,	0,	},
  { 'X',	xline,		1,	0,	},
  { 'y',	class,		1,	0,	},
  { 'Y',	class,		1,	0,	},
  { '?',	links,		0,	0,	},
  { '\0',	(void(*)())0,	0,	0,	}
};

/* parv[0] = sender prefix
 * parv[1] = stat letter/command
 * parv[2] = (if present) server/mask in stats L
 */
static void
m_stats(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  int i;
  char statchar;
  static time_t last_used = 0;

  /* Check the user is actually allowed to do /stats, and isnt flooding */
  if ((last_used + ConfigFileEntry.pace_wait) > CurrentTime)
  {
    sendto_one(source_p,form_str(RPL_LOAD2HI),
	       me.name, source_p->name);
    return;
  }
  else
    last_used = CurrentTime;

  /* Is the stats meant for us? */
  if (!ConfigFileEntry.disable_remote)
    if (hunt_server(client_p,source_p,":%s STATS %s :%s",2,parc,parv) != HUNTED_ISME)
      return;

  statchar = parv[1][0];

  if (statchar == '\0')
  {
    sendto_one(source_p, form_str(RPL_ENDOFSTATS),
               me.name, source_p->name, '*');
    return;
  }

  for (i = 0; cmd_table[i].handler; i++)
  {
    if (cmd_table[i].letter == statchar)
    {
      /* The stats table says what privs are needed, so check --fl_ */
      if (cmd_table[i].need_oper || cmd_table[i].need_admin)
      {
        sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
		   me.name, source_p->name);
        break;
      }

      /* Blah, stats L needs the parameters, none of the others do.. */
      if (statchar == 'L' || statchar == 'l')
        cmd_table[i].handler(source_p, parc, parv);
      else
        cmd_table[i].handler(source_p);
      break;
    }
  }
  sendto_one(source_p, form_str(RPL_ENDOFSTATS),
	     me.name, source_p->name, statchar);

  if((statchar != 'L') && (statchar != 'l'))
    sendto_realops_flags(UMODE_SPY, L_ALL, "STATS %c by %s (%s@%s) [%s]",
                        statchar, source_p->name, source_p->username,
                        source_p->host, source_p->user->server->name);
}

/* parv[0] = sender prefix
 * parv[1] = stat letter/command
 * parv[2] = (if present) server/mask in stats L, or target
 */
static void
mo_stats(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  int i;
  char statchar;

  if (hunt_server(client_p,source_p,":%s STATS %s :%s", 2, parc, parv) != HUNTED_ISME)
    return;

  statchar = parv[1][0];

  if (statchar == '\0')
  {
    sendto_one(source_p, form_str(RPL_ENDOFSTATS),
               me.name, source_p->name, '*');
    return;
  }

  for (i = 0; cmd_table[i].handler; i++)
  {
    if (cmd_table[i].letter == statchar)
    {
      /* The stats table says what privs are needed, so check --fl_ */
      /* Called for remote clients and for local opers, so check need_admin
       * and need_oper
       */
      if ((cmd_table[i].need_admin && !IsAdmin(source_p)) ||
         (cmd_table[i].need_oper && !IsOper(source_p)))
      {
        sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
   	           me.name, source_p->name);
        break;
      }
      if (statchar == 'L' || statchar == 'l')
        cmd_table[i].handler(source_p, parc, parv, statchar);
      else
        cmd_table[i].handler(source_p);

      break;
    }
  }
  sendto_one(source_p, form_str(RPL_ENDOFSTATS),
	     me.name, source_p->name, statchar);

  if ((statchar != 'L') && (statchar != 'l'))
    sendto_realops_flags(UMODE_SPY, L_ALL, "STATS %c by %s (%s@%s) [%s]",
                        statchar, source_p->name, source_p->username,
                        source_p->host, source_p->user->server->name);
}

static void
dns_servers(struct Client *source_p)
{
  report_dns_servers(source_p);
}

static void
sconnect(struct Client *source_p)
{
  report_confitem_types(source_p, SERVER_TYPE);
}

/* input	- client to report to
 * side effects - client is given dline list.
 */
static void
deny(struct Client *source_p)
{
  char *host, *pass, *user, *classname, *oreason;
  struct AddressRec *arec;
  struct ConfItem *conf;
  struct AccessItem *aconf;
  int i, port;

  for (i = 0; i < ATABLE_SIZE; i++)
  {
    for (arec = atable[i]; arec; arec=arec->next)
    {
      if (arec->type == CONF_DLINE)
      {
        aconf = arec->aconf;

        /* dont report a tdline as a dline */
        if (aconf->flags & CONF_FLAGS_TEMPORARY)
          continue;

        conf = unmap_conf_item(aconf);

        get_printable_conf(conf, &host, &pass, &user, &port, &classname, &oreason);
        sendto_one(source_p, form_str(RPL_STATSDLINE),
      		   me.name, source_p->name, 'D', host, pass, oreason);
      }
    }
  }
}

/* input        - client to report to
 * side effects - client is given dline list.
 */
static void
tdeny(struct Client *source_p)
{
  char *host, *pass, *user, *classname, *oreason;
  struct AddressRec *arec;
  struct ConfItem *conf;
  struct AccessItem *aconf;
  int i, port;

  for (i = 0; i < ATABLE_SIZE; i++)
  {
    for (arec = atable[i]; arec; arec=arec->next)
    {
      if (arec->type == CONF_DLINE)
      {
        aconf = arec->aconf;

        /* dont report a permanent dline as a tdline */
        if (!(aconf->flags & CONF_FLAGS_TEMPORARY))
          continue;

        conf = unmap_conf_item(aconf);
        get_printable_conf(conf, &host, &pass, &user, &port, &classname, &oreason);
        sendto_one(source_p, form_str(RPL_STATSDLINE),
		   me.name, source_p->name, 'd', host, pass, oreason);
      }
    }
  }
}

/* input	- client to report to
 * side effects - client is given list of exempt blocks
 */
static void
exempt(struct Client *source_p)
{
  char *host, *pass, *user, *classname, *oreason;
  struct AddressRec *arec;
  struct ConfItem *conf;
  struct AccessItem *aconf;
  int i, port;

  for (i = 0; i < ATABLE_SIZE; i++)
  {
    for (arec = atable[i]; arec; arec=arec->next)
    {
      if (arec->type == CONF_EXEMPTDLINE)
      {
        aconf = arec->aconf;

        conf = unmap_conf_item(aconf);
        get_printable_conf(conf, &host, &pass, &user, &port, &classname, &oreason);
	sendto_one(source_p, form_str(RPL_STATSDLINE),
		   me.name, source_p->name, 'E', host, pass, oreason);
      }
    }
  }
}

static void
hubleaf(struct Client *source_p)
{
  report_confitem_types(source_p, HUB_TYPE);
  report_confitem_types(source_p, LEAF_TYPE);
}

static void
auth(struct Client *source_p)
{
  /* Oper only, if unopered, return ERR_NOPRIVS */
  if ((ConfigFileEntry.stats_i_oper_only == 2) && !IsOper(source_p))
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);

  /* If unopered, Only return matching auth blocks */
  else if ((ConfigFileEntry.stats_i_oper_only == 1) && !IsOper(source_p))
  {
    struct ConfItem *conf;
    struct AccessItem *aconf;
    char *host, *pass, *user, *classname, *oreason;
    int port;

    if(MyConnect(source_p))
      aconf = find_conf_by_address(source_p->host, &source_p->localClient->ip,
				   CONF_CLIENT, source_p->localClient->aftype,
				   source_p->username, source_p->localClient->passwd);
    else
      aconf = find_conf_by_address(source_p->host, NULL, CONF_CLIENT,
				   0, source_p->username, NULL);

    if (aconf == NULL)
      return;
    
    conf = unmap_conf_item(aconf);
    get_printable_conf(conf, &host, &pass, &user, &port, &classname, &oreason);

    sendto_one(source_p, form_str(RPL_STATSILINE), me.name,
               source_p->name, (IsConfRestricted(aconf)) ? 'i' : 'I',
	       "*", show_iline_prefix(source_p, aconf, user), host,
	       port, classname);
  }
  /* They are opered, or allowed to see all auth blocks */
  else
    report_auth(source_p);
}

static void
tklines(struct Client *source_p)
{
  struct ConfItem *conf;
  /* Oper only, if unopered, return ERR_NOPRIVS */
  if ((ConfigFileEntry.stats_k_oper_only == 2) && !IsOper(source_p))
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);

  /* If unopered, Only return matching klines */
  else if ((ConfigFileEntry.stats_k_oper_only == 1) && !IsOper(source_p))
  {
    struct AccessItem *aconf;
    char *host, *pass, *user, *classname, *oreason;
    int port;

    if(MyConnect(source_p))
      aconf = find_conf_by_address(source_p->host, &source_p->localClient->ip,
				   CONF_KLINE, source_p->localClient->aftype,
				   source_p->username, NULL);
    else
      aconf = find_conf_by_address(source_p->host, NULL, CONF_KLINE,
                                   0, source_p->username, NULL);

    if(aconf == NULL)
      return;

    /* dont report a permanent kline as a tkline */
    if (!(aconf->flags & CONF_FLAGS_TEMPORARY))
      return;

    conf = unmap_conf_item(aconf);
    get_printable_conf(conf, &host, &pass, &user, &port, &classname, &oreason);
    sendto_one(source_p, form_str(RPL_STATSKLINE), me.name,
               source_p->name, 'k', host, user, pass, oreason);
  }
  /* Theyre opered, or allowed to see all klines */
  else
    report_Klines(source_p, 1);
}

static void
klines(struct Client *source_p)
{
  struct ConfItem *conf = NULL;

  if((ConfigFileEntry.stats_k_oper_only == 2) && !IsOper(source_p))
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
  else if((ConfigFileEntry.stats_k_oper_only == 1) && !IsOper(source_p))
  {
    struct AccessItem *aconf;
    char *host, *pass, *user, *classname, *oreason;
    int port;

    /* search for a kline */
    if(MyConnect(source_p))
      aconf = find_conf_by_address(source_p->host, &source_p->localClient->ip,
				   CONF_KLINE, source_p->localClient->aftype,
				   source_p->username, NULL);
    else
      aconf = find_conf_by_address(source_p->host, NULL, CONF_KLINE,
                                   0, source_p->username, NULL);

    if (aconf == NULL)
      return;

    /* dont report a tkline as a kline */
    if(aconf->flags & CONF_FLAGS_TEMPORARY)
      return;
      
    get_printable_conf(conf, &host, &pass, &user, &port, &classname, &oreason);
    sendto_one(source_p, form_str(RPL_STATSKLINE), me.name,
               source_p->name, 'K', host, user, pass, "");
  }
  /* Theyre opered, or allowed to see all klines */
  else
    report_Klines(source_p, 0);
}

static void
oper(struct Client *source_p)
{
  if (!IsOper(source_p) && ConfigFileEntry.stats_o_oper_only)
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
 	       me.name, source_p->name);
  else
    report_confitem_types(source_p, OPER_TYPE);
}

/* input	- client pointer
 * side effects - client is shown a list of active opers
 */
static void
operedup(struct Client *source_p)
{
  dlink_node *ptr;

  DLINK_FOREACH(ptr, oper_list.head)
  {
    struct Client *target_p = ptr->data;

    if (MyClient(source_p) && IsOper(source_p))
    {
      sendto_one(source_p, ":%s %d %s p :[%c][%s] %s (%s@%s) Idle: %d",
                 me.name, RPL_STATSDEBUG, source_p->name,
		 IsAdmin(target_p) ? 'A': 'O',
                 oper_privs_as_string(target_p->localClient->operflags),
		 target_p->name, target_p->username, target_p->host,
		 (int)(CurrentTime - target_p->user->last));
    }
    else
    {
      sendto_one(source_p, ":%s %d %s p :[%c] %s (%s@%s) Idle: %d",
                 me.name, RPL_STATSDEBUG, source_p->name,
		 IsAdmin(target_p) ? 'A' : 'O',
		 target_p->name, target_p->username, target_p->host,
		 (int)(CurrentTime - target_p->user->last));
    }
  }
  sendto_one(source_p, ":%s %d %s p :%lu OPER(s)",
             me.name, RPL_STATSDEBUG, source_p->name, dlink_list_length(&oper_list));
}

static void
operlist(struct Client *source_p)
{
  struct Client *target_p;
  dlink_node *oper_ptr, *ptr;
  int count = 0;

  DLINK_FOREACH(oper_ptr, global_client_list.head)
  {
    target_p = oper_ptr->data;
    if(IsOper(target_p))
    {
      count++;
      if (IsOper(source_p) && MyClient(source_p) && MyClient(target_p))
      {
        ptr = target_p->localClient->confs.head;
        sendto_one(source_p, ":%s %d %s o :[%c][%s] %s (%s@%s) on %s Idle: %d",
                   me.name, RPL_STATSDEBUG, source_p->name,
                   IsAdmin(target_p) ? 'A' : 'O',
                   oper_privs_as_string(target_p->localClient->operflags),
                   target_p->name, target_p->username, target_p->host,
		   (ConfigServerHide.hide_servers && !IsOper(source_p)) ?
		   me.name : target_p->user->server->name,
		   (int)(CurrentTime - target_p->user->last));
      }
      else if (MyClient(target_p))
      {
        ptr = target_p->localClient->confs.head;
        sendto_one(source_p, ":%s %d %s o :[%c] %s (%s@%s) on %s Idle: %d",
                   me.name, RPL_STATSDEBUG, source_p->name,
                   IsAdmin(target_p) ? 'A' : 'O',
                   target_p->name, target_p->username, target_p->host,
                   (ConfigServerHide.hide_servers && !IsOper(source_p)) ?
                   me.name : target_p->user->server->name,
                   (int)(CurrentTime - target_p->user->last));
      }
      else
      {
        sendto_one(source_p, ":%s %d %s o :[%c] %s (%s@%s) on %s",
                   me.name, RPL_STATSDEBUG, source_p->name,
                   IsAdmin(target_p) ? 'A' : 'O',
                   target_p->name, target_p->username, target_p->host,
                   (ConfigServerHide.hide_servers && !IsOper(source_p)) ?
                   me.name : target_p->user->server->name);                   
      }
    }
  }
  sendto_one(source_p, ":%s %d %s o :%d OPER(s)",
             me.name, RPL_STATSDEBUG, source_p->name, count);
}

static void
ports(struct Client *source_p)
{
  if (!IsOper(source_p) && ConfigFileEntry.stats_P_oper_only)
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
 	       me.name, source_p->name);
  else
    show_ports(source_p);
}

static void
jupe(struct Client *source_p)
{
  report_jupe(source_p);
}

static void
suptime(struct Client *source_p)
{
  time_t now = CurrentTime - me.since;

  sendto_one(source_p, form_str(RPL_STATSUPTIME), me.name, source_p->name,
             now/86400, (now/3600)%24, (now/60)%60, now%60);
}

static void
shared(struct Client *source_p)
{
  report_confitem_types(source_p, ULINE_TYPE);
}

/* input	- client pointer
 * side effects - client is shown lists of who connected servers
 */
static void
servers(struct Client *source_p)
{
  struct Client *target_p;
  dlink_node *ptr;
  int j = 0;
  
  DLINK_FOREACH(ptr, serv_list.head)
  {
    target_p = ptr->data;

    j++;

    sendto_one(source_p, ":%s %d %s v :%s (%s) Idle: %d",
               me.name, RPL_STATSDEBUG, source_p->name,
	       target_p->name,
	       (target_p->serv->by[0] ? target_p->serv->by : "Remote"),
	       (int)(CurrentTime - target_p->lasttime));
  }
  sendto_one(source_p, ":%s %d %s v :%d Server(s)",
	     me.name, RPL_STATSDEBUG, source_p->name, j);
}

static void
xline(struct Client *source_p)
{
  report_confitem_types(source_p, XLINE_TYPE);
}

static void
class(struct Client *source_p)
{
  report_confitem_types(source_p, CLASS_TYPE);
}

static void
links(struct Client *source_p)
{
  long uptime, sendK, receiveK;
  struct Client *target_p;
  dlink_node *ptr;
  int j = 0;

  if (ConfigServerHide.flatten_links && !IsOper(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
	       me.name, source_p->name);
    return;
  }

  sendK = receiveK = 0;

  DLINK_FOREACH(ptr, serv_list.head)
  {
    target_p = ptr->data;

    j++;
    sendK += target_p->localClient->sendK;
    receiveK += target_p->localClient->receiveK;

    sendto_one(source_p, form_str(RPL_STATSLINKINFO),
               me.name, source_p->name,
               IsAdmin(source_p) ? get_client_name(target_p, SHOW_IP)
               : get_client_name(target_p, MASK_IP),
               (int)dbuf_length(&target_p->localClient->buf_sendq),
               (int)target_p->localClient->sendM,
               (int)target_p->localClient->sendK,
               (int)target_p->localClient->receiveM,
               (int)target_p->localClient->receiveK,
               (unsigned)(CurrentTime - target_p->firsttime),
               (CurrentTime > target_p->since) ? (unsigned)(CurrentTime - target_p->since): 0,
               IsOper(source_p) ? show_capabilities(target_p) : "TS");
  }
  
  sendto_one(source_p, ":%s %d %s ? :%u total server(s)",
             me.name, RPL_STATSDEBUG, source_p->name, j);
  sendto_one(source_p, ":%s %d %s ? :Sent total : %7.2f %s",
             me.name, RPL_STATSDEBUG, source_p->name,
	     _GMKv(sendK), _GMKs(sendK));
  sendto_one(source_p, ":%s %d %s ? :Recv total : %7.2f %s",
             me.name, RPL_STATSDEBUG, source_p->name,
	     _GMKv(receiveK), _GMKs(receiveK));

  uptime = (CurrentTime - me.since);

  sendto_one(source_p, ":%s %d %s ? :Server send: %7.2f %s (%4.1f K/s)",
             me.name, RPL_STATSDEBUG, source_p->name,
	     _GMKv(me.localClient->sendK), _GMKs(me.localClient->sendK),
	     (float)((float)me.localClient->sendK / (float)uptime));
  sendto_one(source_p, ":%s %d %s ? :Server recv: %7.2f %s (%4.1f K/s)",
             me.name, RPL_STATSDEBUG, source_p->name,
	     _GMKv(me.localClient->receiveK),
	     _GMKs(me.localClient->receiveK),
	     (float)((float)me.localClient->receiveK / (float)uptime));
}

static void
sltrace(struct Client *source_p, int parc, char *parv[])
{
  int doall = 0, wilds = 0;
  char *name = NULL, statchar;
  
  if ((name = parse_args(parc,parv,&doall,&wilds)) != NULL)
  {
    statchar=parv[1][0];

    L(source_p, name, doall, wilds, statchar);

    sendto_realops_flags(UMODE_SPY, L_ALL, "STATS %c by %s (%s@%s) [%s] on %s",
                         statchar, source_p->name, source_p->username,
                         source_p->host, source_p->user->server->name, name);
  }
  else
    sendto_one(source_p, form_str(ERR_NEEDMOREPARAMS),
               me.name, source_p->name, "STATS");
}

/* parv[0] = sender prefix
 * parv[1] = statistics selector (defaults to Message frequency)
 * parv[2] = server name (current server defaulted, if omitted)
 */
static void
ms_stats(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  if (hunt_server(client_p,source_p,":%s STATS %s :%s",2,parc,parv)!=HUNTED_ISME)
    return;

  if (IsClient(source_p))
    mo_stats(client_p, source_p, parc, parv);
}

/* inputs	- pointer to client to report to
 *		- doall flag
 *		- wild card or not
 */
static void
L(struct Client *source_p,char *name,int doall,
        int wilds,char statchar)
{
  L_list(source_p, name, doall, wilds, &unknown_list, statchar);
  L_list(source_p, name, doall, wilds, &local_client_list, statchar);
  L_list(source_p, name, doall, wilds, &serv_list, statchar);
}

static void
L_list(struct Client *source_p,char *name, int doall, int wilds,
             dlink_list *list,char statchar)
{
  dlink_node *ptr;
  struct Client *target_p;

  /* send info about connections which match, or all if the
   * mask matches from.  Only restrictions are on those who
   * are invisible not being visible to 'foreigners' who use
   * a wild card based search to list it.
   */
  DLINK_FOREACH(ptr, list->head)
  {
    target_p = ptr->data;

    if (IsInvisible(target_p) && (doall || wilds) &&
        !(MyConnect(source_p) && IsOper(source_p)) &&
        !IsOper(target_p) && (target_p != source_p))
      continue;
    if (!doall && wilds && !match(name, target_p->name))
      continue;
    if (!(doall || wilds) && irccmp(name, target_p->name))
      continue;

    /* This basically shows ips for our opers if its not a server/admin, or
     * its one of our admins.  */
    if(MyClient(source_p) && IsOper(source_p) &&
       (IsAdmin(source_p) ||
       (!IsServer(target_p) && !IsAdmin(target_p) &&
       !IsHandshake(target_p) && !IsConnecting(target_p))))
    {
      sendto_one(source_p, form_str(RPL_STATSLINKINFO),
                 me.name, source_p->name,
                 (IsUpper(statchar)) ?
                 get_client_name(target_p, SHOW_IP) :
                 get_client_name(target_p, HIDE_IP),
                 (int)dbuf_length(&target_p->localClient->buf_sendq),
                 (int)target_p->localClient->sendM,
                 (int)target_p->localClient->sendK,
                 (int)target_p->localClient->receiveM,
                 (int)target_p->localClient->receiveK,
                 (unsigned)(CurrentTime - target_p->firsttime),
                 (CurrentTime > target_p->since) ? (unsigned)(CurrentTime - target_p->since):0,
                 IsServer(target_p) ? show_capabilities(target_p) : "-");
    }
    else
    {
      /* If its a hidden ip, an admin, or a server, mask the real IP */
      if(IsIPSpoof(target_p) || IsServer(target_p) || IsAdmin(target_p)
         || IsHandshake(target_p) || IsConnecting(target_p))
        sendto_one(source_p, form_str(RPL_STATSLINKINFO),
                   me.name, source_p->name,
                   get_client_name(target_p, MASK_IP),
                   (int)dbuf_length(&target_p->localClient->buf_sendq),
                   (int)target_p->localClient->sendM,
                   (int)target_p->localClient->sendK,
                   (int)target_p->localClient->receiveM,
                   (int)target_p->localClient->receiveK,
                   (unsigned)(CurrentTime - target_p->firsttime),
                   (CurrentTime > target_p->since) ? (unsigned)(CurrentTime - target_p->since):0,
                   IsServer(target_p) ? show_capabilities(target_p) : "-");
      else /* show the real IP */
        sendto_one(source_p, form_str(RPL_STATSLINKINFO),
                   me.name, source_p->name,
                   (IsUpper(statchar)) ?
                   get_client_name(target_p, SHOW_IP) :
                   get_client_name(target_p, HIDE_IP),
                   (int)dbuf_length(&target_p->localClient->buf_sendq),
                   (int)target_p->localClient->sendM,
                   (int)target_p->localClient->sendK,
                   (int)target_p->localClient->receiveM,
                   (int)target_p->localClient->receiveK,
                   (unsigned)(CurrentTime - target_p->firsttime),
                   (CurrentTime > target_p->since) ? (unsigned)(CurrentTime - target_p->since):0,
                   IsServer(target_p) ? show_capabilities(target_p) : "-");
    }
  }
}

/* inputs	- arg count
 *		- args
 *		- doall flag
 *		- wild card or not
 * output	- pointer to name to use
 */
static char *
parse_args(int parc, char *parv[], int *doall, int *wilds)
{
  char *name;

  if (parc > 2)
  {
    name = parv[2];
    if (!irccmp(name, me.name))
      *doall = 2;
    else if (match(name, me.name))
      *doall = 1;
    if (strchr(name, '*') ||
      strchr(name, '?'))
      *wilds = 1;

    return(name);
  }
  else
    return(NULL);
}
