/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * m_whois.c: Shows who a user is.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: m_whois.c,v 1.109.2.2 2004/07/22 20:31:25 musirc Exp $
 */
 
#include "whowas.h"
#include "handlers.h"
#include "hash.h"
#include "istring.h"
#include "sprintf.h"
#include "numeric.h"
#include "ircd.h"
#include "server.h"
#include "config.h"
#include "send.h"
#include "modules.h"
#include "channel.h"
#include "channel_mode.h"

static void do_whois(struct Client *, int, char **);
static int single_whois(struct Client *, struct Client *, int, int);
static void whois_person(struct Client *, struct Client *, int);
static int global_whois(struct Client *, const char *, int, int);
static void m_whois(struct Client *, struct Client *, int, char **);
static void mo_whois(struct Client *, struct Client *, int, char **);
static void ms_whois(struct Client *, struct Client *, int, char **);
static void mo_owhois(struct Client *, struct Client *, int, char **);
static void m_whowas(struct Client*, struct Client*, int, char**);
static void mo_whowas(struct Client*, struct Client*, int, char**);
static void whowas_do(struct Client *, struct Client *, int, char **);

struct Message whois_msgtab = {
  "WHOIS", 0, MFLG_SLOW,
  {m_unregistered, m_whois, ms_whois, mo_whois}
};
struct Message owhois_msgtab = {
  "OWHOIS", 0, MFLG_SLOW,
  {m_ignore, m_ignore, m_ignore, mo_owhois}
};
struct Message whowas_msgtab = {
  "WHOWAS", 0, MFLG_SLOW,
  {m_unregistered, m_whowas, m_ignore, mo_whowas}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&whois_msgtab);
  mod_add_cmd(&owhois_msgtab);
  mod_add_cmd(&whowas_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&whois_msgtab);
  mod_del_cmd(&owhois_msgtab);
  mod_del_cmd(&whowas_msgtab);
}

const char *_version = "$Revision: 1.109.2.2 $";
#endif

/* parv[0] = sender prefix
 * parv[1] = nickname masklist
 */
static void
m_whois(struct Client *client_p, struct Client *source_p, int parc, char *parv[])
{
  static time_t last_used = 0;
  
  if (parc < 2 || EmptyString(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_NONICKNAMEGIVEN),
               me.name, source_p->name);
    return;
  }

  /* seeing as this is going across servers, we should limit it */
  if ((last_used + ConfigFileEntry.pace_wait_simple) > CurrentTime)
  {
    sendto_one(source_p, form_str(RPL_LOAD2HI),
	       me.name, source_p->name);
    return;
  }
  else
    last_used = CurrentTime;

  /* if we have serverhide enabled, they can either ask the clients
   * server, or our server.. I dont see why they would need to ask
   * anything else for info about the client.. --fl_
   */
  if (parc > 2)
  {
    if (hunt_server(client_p,source_p,":%s WHOIS %s :%s", 1, parc, parv) != HUNTED_ISME)
      return;

    parv[1] = parv[2];
  }
  do_whois(source_p, parc, parv);
}

/* parv[0] = sender prefix
 * parv[1] = nickname masklist
 */
static void
mo_whois(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  if (parc < 2 || EmptyString(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_NONICKNAMEGIVEN),
               me.name, source_p->name);
    return;
  }
  if (parc > 2)
  {
    if (hunt_server(client_p,source_p,":%s WHOIS %s :%s", 1, parc, parv) != HUNTED_ISME)
      return;

    parv[1] = parv[2];
  }
  do_whois(source_p, parc, parv);
}

/* parv[0] = sender prefix
 * parv[1] = nickname masklist
 */
static void
mo_owhois(struct Client *client_p, struct Client *source_p, int parc, char *parv[])
{
  char buf[BUFSIZE], *t;
  int cur_len = 0, mlen, tlen, reply_to_send = 0;
  dlink_node *lp;
  struct Channel *chptr;
  struct Client *target_p = NULL, *server_p;

  if (parc < 2 || EmptyString(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_NONICKNAMEGIVEN),
               me.name, source_p->name);
    return;
  }

  if ((target_p = find_client(parv[1])) == NULL || !IsClient(target_p))
  {
    sendto_one(client_p, form_str(ERR_NOSUCHNICK),
	       me.name, client_p->name, parv[1]);
    return;
  }
  server_p = target_p->user->server;

  sendto_one(client_p, form_str(RPL_WHOISUSER), me.name,
             client_p->name, target_p->name, target_p->username,
             target_p->host, target_p->info);
  if (!IsIPSpoof(target_p) && ConfigFileEntry.hide_spoof_ips &&
      MyConnect(target_p) && ((MyClient(source_p) && (IsOper(source_p) ||
      !ConfigServerHide.hide_servers)) || (source_p == target_p)))
     if ((IsOper(source_p) && IsOper(target_p)) ||
        !IsOper(target_p))
  sendto_one(source_p, form_str(RPL_WHOISACTUALLY),
             me.name, source_p->name, target_p->name,
             target_p->localClient->sockhost);

  ircsprintf(buf, form_str(RPL_WHOISCHANNELS), me.name,
             client_p->name, target_p->name, "");

  mlen = strlen(buf);
  cur_len = mlen;
  t = buf + mlen;

  DLINK_FOREACH(lp, target_p->user->channel.head)
  {
    chptr = ((struct Membership *)lp->data)->chptr;
    if ((cur_len + strlen(chptr->name) + 2) > (BUFSIZE - 4))
    {
      sendto_one(client_p, "%s", buf);
      cur_len = mlen;
      t = buf + mlen;
    }

    ircsprintf(t, "%s%s%s ",
               ShowChannel(client_p, chptr) ? "" : "^",
	       get_member_status((struct Membership *) lp->data, 1),
               chptr->name);
    tlen = strlen(t);
    t += tlen;
    cur_len += tlen;
    reply_to_send = 1;
  }

  if (reply_to_send)
    sendto_one(client_p, "%s", buf);

   sendto_one(client_p, form_str(RPL_WHOISSERVER), me.name,
	      client_p->name, target_p->name, server_p->name,
              server_p->info);

  if (IsOper(target_p))
    sendto_one(client_p, form_str(RPL_WHOISOPERATOR),
	       me.name, client_p->name, target_p->name);

  if (IsService(target_p))
  {
    sendto_one(source_p, form_str(RPL_WHOISSERVICE),
               me.name, source_p->name, target_p->name);
    sendto_one(client_p, form_str(RPL_ENDOFWHOIS),
               me.name, client_p->name, target_p->name);

    sendto_realops_flags(UMODE_SPY, L_ALL, "OWHOIS '%s' by %s (%s@%s) [%s]",
                         target_p->name, source_p->name, source_p->username,
                         source_p->host, source_p->user->server->name);
    return;
  }

  if (MyConnect(target_p))
    sendto_one(client_p, form_str(RPL_WHOISIDLE), me.name,
               client_p->name, target_p->name, CurrentTime - target_p->user->last,
               target_p->firsttime);
  sendto_one(client_p, form_str(RPL_ENDOFWHOIS),
	     me.name, client_p->name, target_p->name);

  sendto_realops_flags(UMODE_SPY, L_ALL, "OWHOIS '%s' by %s (%s@%s) [%s]",
                       target_p->name, source_p->name, source_p->username,
                       source_p->host, source_p->user->server->name);
}

static void
do_whois(struct Client *source_p, int parc, char *parv[])
{
  struct Client *target_p;
  char *nick, *p = NULL;
  int found = 0, wilds, glob = 0;

  /* This lets us make all "whois nick" queries look the same, and all
   * "whois nick nick" queries look the same.  We have to pass it all
   * the way down to whois_person() though -- fl */
  if (parc > 2)
    glob = 1;

  nick = parv[1];
  while (*nick == ',')
    nick++;
  if ((p = strchr(nick,',')) != NULL)
    *p = '\0';

  if (*nick == '\0')
    return;

  collapse(nick);
  wilds = (strchr(nick, '?') || strchr(nick, '*'));

  if (!wilds)
  {
    if ((target_p = find_client(nick)) != NULL)
    {
      /* im being asked to reply to a client that isnt mine..
       * I cant answer authoritively, so better make it non-detailed
       */
      if (!MyClient(target_p))
         glob = 0;

      if (IsPerson(target_p))
      {
        single_whois(source_p, target_p, wilds, glob);
        found = 1;
      }
    }
  }
  else /* wilds is true */
  {
    if (MyClient(source_p))
      found = global_whois(source_p, nick, wilds, glob);
  }

  if (!found)
    sendto_one(source_p, form_str(ERR_NOSUCHNICK),
	       me.name, source_p->name, parv[1]);
  sendto_one(source_p, form_str(RPL_ENDOFWHOIS),
	     me.name, source_p->name, parv[1]);
}

/* Inputs	- source_p client to report to
 *		- target_p client to report on
 *		- wilds whether wildchar char or not
 * Output	- if found return 1
 * Side Effects	- do a single whois on given client
 * 		  writing results to source_p
 */
static int
global_whois(struct Client *source_p, const char *nick, int wilds, int glob)
{
  dlink_node *ptr;
  struct Client *target_p;
  int found = 0;

  DLINK_FOREACH(ptr, global_client_list.head)
  {
    target_p = ptr->data;

    if (!IsPerson(target_p))
      continue;
 
    if (!match(nick, target_p->name))
      continue;

    /* 'Rules' established for sending a WHOIS reply:
     * - if wildcards are being used dont send a reply if
     *   the querier isnt any common channels and the
     *   client in question is invisible and wildcards are
     *   in use (allow exact matches only);
     *
     * - only send replies about common or public channels
     *   the target user(s) are on;
     */
    if (single_whois(source_p, target_p, wilds, glob))
      found = 1;
  }
  return(found);
}

/* Inputs       - source_p client to report to
 *              - target_p client to report on
 *              - wilds whether wildchar char or not
 * Output       - if found return 1
 * Side Effects - do a single whois on given client
 *                writing results to source_p
 */
static int
single_whois(struct Client *source_p, struct Client *target_p,
             int wilds, int glob)
{
  dlink_node *ptr;
  struct Channel *chptr;
  const char *name;
  int invis, member, showperson;

 if (target_p->name[0] == '\0')
    name = "?";
  else
    name = target_p->name;

  invis = IsInvisible(target_p);
  member = (target_p->user->channel.head != NULL) ? 1 : 0;
  showperson = (wilds && !invis && !member) || !wilds;

  DLINK_FOREACH(ptr, target_p->user->channel.head)
  {
    chptr = ((struct Membership *)ptr->data)->chptr;
    member = IsMember(source_p, chptr);
    if (invis && !member)
      continue;
    if (member || (!invis && PubChannel(chptr)))
    {
      showperson = 1;
      break;
    }
    if (!invis && HiddenChannel(chptr) && !SecretChannel(chptr))
    {
      showperson = 1;
      break;
    }
  }
  if (showperson)
    whois_person(source_p, target_p, glob);
  return(1);
}

/* inputs       - source_p client to report to
 *              - target_p client to report on
 */
static void
whois_person(struct Client *source_p,struct Client *target_p, int glob)
{
  char buf[BUFSIZE], *t;
  dlink_node *lp;
  struct Client *server_p;
  struct Channel *chptr;
  struct Membership *ms;
  int cur_len = 0, mlen, tlen, reply_to_send = 0;

  server_p = target_p->user->server;

  sendto_one(source_p, form_str(RPL_WHOISUSER), me.name,
         source_p->name, target_p->name,
         target_p->username, target_p->host, target_p->info);
  if (!IsIPSpoof(target_p) && ConfigFileEntry.hide_spoof_ips &&
      MyConnect(target_p) && ((glob) || (MyClient(source_p) && (IsOper(source_p) ||
      !ConfigServerHide.hide_servers)) || (source_p == target_p)))
     if ((IsOper(source_p) && IsOper(target_p)) ||
	!IsOper(target_p))
        sendto_one(source_p, form_str(RPL_WHOISACTUALLY),
                   me.name, source_p->name, target_p->name,
                   target_p->localClient->sockhost);

  ircsprintf(buf, form_str(RPL_WHOISCHANNELS),
             me.name, source_p->name, target_p->name, "");

  mlen = strlen(buf);
  cur_len = mlen;
  t = buf + mlen;

  DLINK_FOREACH(lp, target_p->user->channel.head)
  {
    if (IsService(target_p))
      break;

    ms = lp->data;
    chptr = ms->chptr;

    if (ShowChannel(source_p, chptr))
    {
      if ((cur_len + strlen(chptr->name) + 2) > (BUFSIZE - 4))
      {
        sendto_one(source_p, "%s", buf);
        cur_len = mlen;
        t = buf + mlen;
      }
      ircsprintf(t, "%s%s ", get_member_status(ms, 1), chptr->name);

      tlen = strlen(t);
      t += tlen;
      cur_len += tlen;
      reply_to_send = 1;
    }
  }

  if (reply_to_send)
    sendto_one(source_p, "%s", buf);

  if ((IsOper(source_p) || !ConfigServerHide.hide_servers) || target_p == source_p)
    sendto_one(source_p, form_str(RPL_WHOISSERVER),
               me.name, source_p->name, target_p->name, server_p->name,
	       server_p->info);
  else
    sendto_one(source_p, form_str(RPL_WHOISSERVER),
               me.name, source_p->name, target_p->name,
               ServerInfo.network_name, server_p->info);

  if (target_p->user->away)
    sendto_one(source_p, form_str(RPL_AWAY), me.name,
               source_p->name, target_p->name, target_p->user->away);

  if (IsOper(target_p))
    sendto_one(source_p, form_str(RPL_WHOISOPERATOR),
               me.name, source_p->name, target_p->name);

  if (IsService(target_p))
  {
    sendto_one(source_p, form_str(RPL_WHOISSERVICE),
               me.name, source_p->name, target_p->name);
    return;
  }

  if (MyConnect(target_p) && IsOper(target_p) && (target_p != source_p)
      && target_p->umodes & UMODE_SPY)
     sendto_one(target_p, ":%s NOTICE %s :WHOIS: by %s (%s@%s) [%s]",
                me.name, source_p->name, source_p->name, source_p->username,
                source_p->host, source_p->user->server->name);

  if (MyConnect(target_p) && ((glob) || (MyClient(source_p) && (IsOper(source_p) ||
     !ConfigServerHide.hide_servers)) || (source_p == target_p)))
    sendto_one(source_p, form_str(RPL_WHOISIDLE),
               me.name, source_p->name, target_p->name,
               CurrentTime - target_p->user->last,
               target_p->firsttime);
}

/* parv[0] = sender prefix
 * parv[1] = nickname masklist
 */
static void
ms_whois(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  if (!IsClient(source_p))
    return;

  if (parc < 2 || EmptyString(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_NONICKNAMEGIVEN),
               me.name, source_p->name);
    return;
  }
  if (parc > 2)
  {
    struct Client *target_p;

    /* check if parv[1] is a person.. (most common) */
    if ((target_p = find_person(parv[1])) == NULL)
    {
      /* ok, parv[1] isnt a client, is it a server? */
      if ((target_p = find_server(parv[1])) == NULL)
      {
        /* its not a server either.. theyve sent a bad whois */
        sendto_one(source_p, form_str(ERR_NOSUCHSERVER), me.name,
                   parv[0], parv[1]);
        return;
      }
    }
    else
      target_p = target_p->servptr;

    if (!IsMe(target_p))
    {
      sendto_one(target_p->from, ":%s WHOIS %s :%s",
                 source_p->name, target_p->name, parv[2]);
      return;
    }
    /* ok, the target is either us, or a client on our server, so perform the whois
     * but first, parv[1] == server to perform the whois on, parv[2] == person
     * to whois, so make parv[1] = parv[2] so do_whois is ok -- fl_
     */
    parv[1] = parv[2];
    do_whois(source_p, parc, parv);
    return;
  }
  do_whois(source_p, parc, parv);
}

/* parv[0] = sender prefix
 * parv[1] = nickname queried
 */
static void
m_whowas(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  static time_t last_used = 0;

  if (parc < 2 || EmptyString(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_NONICKNAMEGIVEN),
               me.name, source_p->name);
    return;
  }

  if((last_used + ConfigFileEntry.pace_wait_simple) > CurrentTime)
  {
    sendto_one(source_p,form_str(RPL_LOAD2HI),
 	       me.name,source_p->name);
    return;
  }
  else
    last_used = CurrentTime;
  whowas_do(client_p,source_p,parc,parv);
}

static void
mo_whowas(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  if (parc < 2 || EmptyString(parv[1]))
  {
    sendto_one(source_p, form_str(ERR_NONICKNAMEGIVEN),
               me.name, source_p->name);
    return;
  }
  whowas_do(client_p,source_p,parc,parv);
}

static void
whowas_do(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  struct Whowas *temp;
  int cur = 0, max = -1, found = 0;
  char *p, *nick;

  if (parc > 2)
    max = atoi(parv[2]);
  if (parc > 3)
    if (hunt_server(client_p,source_p,":%s WHOWAS %s %s :%s", 3,parc,parv))
      return;

  nick = parv[1];
  while (*nick == ',')
    nick++;
  if((p = strchr(nick,',')) != NULL)
    *p = '\0';
  if (!*nick)
    return;

  temp = WHOWASHASH[hash_whowas_name(nick)];

  for(;temp;temp=temp->next)
  {
    if (!irccmp(nick, temp->name))
    {
      sendto_one(source_p, form_str(RPL_WHOWASUSER),
                 me.name, source_p->name, temp->name,
                 temp->username, temp->hostname, temp->realname);

       if (ConfigServerHide.hide_servers && !IsOper(source_p))
         sendto_one(source_p, form_str(RPL_WHOISSERVER),
                    me.name, source_p->name, temp->name,
                    ServerInfo.network_name, myctime(temp->logoff));
       else
	 sendto_one(source_p, form_str(RPL_WHOISSERVER),
                    me.name, source_p->name, temp->name,
                    temp->servername, myctime(temp->logoff));
       cur++;
       found++;
     }
     if (max > 0 && cur >= max)
       break;
  }
  if (!found)
    sendto_one(source_p, form_str(ERR_WASNOSUCHNICK),
               me.name, source_p->name, parv[1]);

  sendto_one(source_p, form_str(RPL_ENDOFWHOWAS), me.name, source_p->name, parv[1]);
}
