/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * config.c: Configuration file functions.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: config.c,v 1.40.2.9 2004/07/25 06:48:43 musirc Exp $
 */

#include "tools.h"
#include "config.h"
#include "server.h"
#include "jupe.h"
#include "channel.h"
#include "istring.h"
#include "sprintf.h"
#include "bsd.h"
#include "ircd.h"
#include "hostmask.h"
#include "list.h"
#include "modules.h"
#include "numeric.h"
#include "log.h"
#include "send.h"
#include "res.h"
#include "userhost.h"
#include "getinfo.h"

struct config_server_hide ConfigServerHide;
dlink_list server_items = { NULL, NULL, 0 };
dlink_list hub_items = { NULL, NULL, 0 };
dlink_list leaf_items  = { NULL, NULL, 0 };
dlink_list oconf_items = { NULL, NULL, 0 };
dlink_list uconf_items = { NULL, NULL, 0 };
dlink_list xconf_items = { NULL, NULL, 0 };
dlink_list njupe_items = { NULL, NULL, 0 };
dlink_list class_items = { NULL, NULL, 0 };
dlink_list temporary_klines = { NULL, NULL, 0 };
dlink_list temporary_dlines = { NULL, NULL, 0 };

extern int lineno;
extern char linebuf[];
extern char conffilebuf[BUFSIZE];
extern int yyparse(void);
unsigned int scount = 0;
int ypass = 1;

static void lookup_confhost(struct ConfItem *);
static void set_default_conf(void);
static void validate_conf(void);
static void read_conf(FBFILE *);
static void clear_out_old_conf(void);
static void flush_deleted_I_P(void);
static void expire_tklines(dlink_list *);
static void garbage_collect_ip_entries(void);
static int hash_ip(struct irc_ssaddr *);
static int verify_access(struct Client *, const char *);
static int attach_iline(struct Client *, struct ConfItem *);
static struct ip_entry *find_or_add_ip(struct irc_ssaddr *);
static void parse_conf_file(int, int);
static dlink_list *map_to_list(ConfType);

FBFILE *conf_fbfile_in;
extern char yytext[];

static struct ConfItem *class_default;

#define IP_HASH_SIZE 0x1000

struct ip_entry
{
  struct irc_ssaddr ip;
  int count;
  time_t last_attempt;
  struct ip_entry *next;
};
static struct ip_entry *ip_hash_table[IP_HASH_SIZE];
static BlockHeap *ip_entry_heap = NULL;
static int ip_entries_count = 0;

inline void *
map_to_conf(struct ConfItem *aconf)
{
  void *conf;
  conf = (void *)((unsigned long)aconf +
  (unsigned long)sizeof(struct ConfItem));
  return(conf);
}

inline struct ConfItem *
unmap_conf_item(void *aconf)
{
  struct ConfItem *conf;

  conf = (struct ConfItem *)((unsigned long)aconf -
     (unsigned long)sizeof(struct ConfItem));
  return(conf);
}

/* inputs	- pointer to struct AccessItem
 *		- pointer to DNSReply reply
 * side effects	- called when resolver query finishes
 * if the query resulted in a successful search, hp will contain
 * a non-null pointer, otherwise hp will be null.
 * if successful save hp in the conf item it was called with
 */
static void
conf_dns_callback(void *vptr, struct DNSReply *reply)
{
  struct AccessItem *aconf = (struct AccessItem *)vptr;
  struct ConfItem *conf;

  MyFree(aconf->dns_query);
  aconf->dns_query = NULL;

  if (reply != NULL)
    memcpy(&aconf->ipnum, &reply->addr, sizeof(reply->addr));
  else
  {
    ilog(NOTICE, "Host not found: %s, ignoring connect block",
         aconf->host);
    conf = unmap_conf_item(aconf);
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "Ignoring connect block for %s - host not found",
			 conf->name);
    delete_conf_item(conf);
  }
}

/* do a nameserver lookup of the conf host
 * if the conf entry is currently doing a ns lookup do nothing, otherwise
 * allocate a dns_query and start ns lookup.
 */
static void
conf_dns_lookup(struct AccessItem *aconf)
{
  if (aconf->dns_query == NULL)
  {
    aconf->dns_query = MyMalloc(sizeof(struct DNSQuery));
    aconf->dns_query->ptr = aconf;
    aconf->dns_query->callback = conf_dns_callback;
    gethost_byname(aconf->host, aconf->dns_query);
  }
}

/* inputs       - type of item
 * output       - pointer to new conf entry
 */
struct ConfItem *
make_conf_item(ConfType type)
{
  struct ConfItem *conf;
  struct AccessItem *aconf;
  struct ClassItem *aclass;
  int status = 0;

  switch (type)
  {
  case DLINE_TYPE:
  case EXEMPTDLINE_TYPE:
  case KLINE_TYPE:
  case CLIENT_TYPE:
  case OPER_TYPE:
  case SERVER_TYPE:
    conf = (struct ConfItem *)MyMalloc(sizeof(struct ConfItem) +
                                         sizeof(struct AccessItem));
    aconf = (struct AccessItem *)map_to_conf(conf);
    aconf->aftype = AF_INET;

    /* Yes, sigh. switch on type again */
    switch (type)
    {
    case EXEMPTDLINE_TYPE:
      status = CONF_EXEMPTDLINE;
      break;

    case DLINE_TYPE:
      status = CONF_DLINE;
      break;

    case KLINE_TYPE:
      status = CONF_KLINE;
      break;

    case CLIENT_TYPE:
      status = CONF_CLIENT;
      break;

    case OPER_TYPE:
      status = CONF_OPERATOR;
      dlinkAdd(conf, &conf->node, &oconf_items);
      break;

    case SERVER_TYPE:
      status = CONF_SERVER;
      dlinkAdd(conf, &conf->node, &server_items);
      break;

    default:
      break;
    }
    aconf->status = status;
    break;

  case LEAF_TYPE:
    conf = (struct ConfItem *)MyMalloc(sizeof(struct ConfItem) +
					 sizeof(struct MatchItem));
    dlinkAdd(conf, &conf->node, &leaf_items);
    break;

  case HUB_TYPE:
    conf = (struct ConfItem *)MyMalloc(sizeof(struct ConfItem) +
					 sizeof(struct MatchItem));
    dlinkAdd(conf, &conf->node, &hub_items);
    break;

  case ULINE_TYPE:
    conf = (struct ConfItem *)MyMalloc(sizeof(struct ConfItem) +
                                         sizeof(struct MatchItem));
    dlinkAdd(conf, &conf->node, &uconf_items);
    break;

  case XLINE_TYPE:
    conf = (struct ConfItem *)MyMalloc(sizeof(struct ConfItem) +
                                         sizeof(struct MatchItem));
    dlinkAdd(conf, &conf->node, &xconf_items);
    break;

  case CJUPE_TYPE:
    conf = (struct ConfItem *)MyMalloc(sizeof(struct ConfItem) +
                                         sizeof(struct JupeChannel));
    break;

  case NJUPE_TYPE:
    conf = (struct ConfItem *)MyMalloc(sizeof(struct ConfItem) +
                                         sizeof(struct MatchItem));
    dlinkAdd(conf, &conf->node, &njupe_items);
    break;

  case CLASS_TYPE:
    conf = (struct ConfItem *)MyMalloc(sizeof(struct ConfItem) +
                                         sizeof(struct ClassItem));
    dlinkAdd(conf, &conf->node, &class_items);
    aclass = (struct ClassItem *)map_to_conf(conf);
    ConFreq(aclass)  = DEFAULT_CONNECTFREQUENCY;
    PingFreq(aclass) = DEFAULT_PINGFREQUENCY;
    MaxTotal(aclass) = ConfigFileEntry.maximum_links;
    MaxSendq(aclass) = DEFAULT_SENDQ;
    CurrUserCount(aclass) = 0;
    break;

  default:
    conf = NULL;
    break;
  }
  conf->type = type;

  return(conf);
}

void
delete_conf_item(struct ConfItem *conf)
{
  struct MatchItem *match_item;
  struct AccessItem *aconf;
  ConfType type = conf->type;

  MyFree(conf->name);

  switch(type)
  {
  case DLINE_TYPE:
  case EXEMPTDLINE_TYPE:
  case KLINE_TYPE:
  case CLIENT_TYPE:
  case OPER_TYPE:
  case SERVER_TYPE:
    aconf = (struct AccessItem *)map_to_conf(conf);
    if (aconf->dns_query != NULL)
    {
      delete_resolver_queries(aconf->dns_query);
      MyFree(aconf->dns_query);
    }
    if (aconf->passwd != NULL)
      memset(aconf->passwd, 0, strlen(aconf->passwd));
    if (aconf->spasswd != NULL)
      memset(aconf->spasswd, 0, strlen(aconf->spasswd));
    aconf->class_ptr = NULL;

    MyFree(aconf->passwd);
    MyFree(aconf->spasswd);
    MyFree(aconf->reason);
    MyFree(aconf->oper_reason);
    MyFree(aconf->user);
    MyFree(aconf->host);
    MyFree(aconf->fakename);

    switch(type)
    {
    case EXEMPTDLINE_TYPE:
    case DLINE_TYPE:
    case KLINE_TYPE:
    case CLIENT_TYPE:
      MyFree(conf);
      break;

    case OPER_TYPE:
      aconf = (struct AccessItem *)map_to_conf(conf);
      if (!IsConfIllegal(aconf))
      dlinkDelete(&conf->node, &oconf_items);
      MyFree(conf);
      break;

    case SERVER_TYPE:
      aconf = (struct AccessItem *)map_to_conf(conf);
      if (!IsConfIllegal(aconf))
      dlinkDelete(&conf->node, &server_items);
      MyFree(conf);
      break;

    default:
      break;
    }
    break;

  case HUB_TYPE:
    match_item = (struct MatchItem *)map_to_conf(conf);
    MyFree(match_item->user);
    MyFree(match_item->host);
    MyFree(match_item->reason);
    MyFree(match_item->oper_reason);
    /* If marked illegal, its already been pulled off of the hub_items list */
    if (!match_item->illegal)
      dlinkDelete(&conf->node, &hub_items);
    MyFree(conf);
    break;

  case LEAF_TYPE:
    match_item = (struct MatchItem *)map_to_conf(conf);
    MyFree(match_item->user);
    MyFree(match_item->host);
    MyFree(match_item->reason);
    MyFree(match_item->oper_reason);
    /* If marked illegal, its already been pulled off of the leaf_items list */
    if (!match_item->illegal)
      dlinkDelete(&conf->node, &leaf_items);
    MyFree(conf);
    break;

  case ULINE_TYPE:
    match_item = (struct MatchItem *)map_to_conf(conf);
    MyFree(match_item->user);
    MyFree(match_item->host);
    MyFree(match_item->reason);
    MyFree(match_item->oper_reason);
    dlinkDelete(&conf->node, &uconf_items);
    MyFree(conf);
    break;

  case XLINE_TYPE:
    match_item = (struct MatchItem *)map_to_conf(conf);
    MyFree(match_item->user);
    MyFree(match_item->host);
    MyFree(match_item->reason);
    MyFree(match_item->oper_reason);
    dlinkDelete(&conf->node, &xconf_items);
    MyFree(conf);
    break;

  case NJUPE_TYPE:
    match_item = (struct MatchItem *)map_to_conf(conf);
    MyFree(match_item->user);
    MyFree(match_item->host);
    MyFree(match_item->reason);
    MyFree(match_item->oper_reason);
    dlinkDelete(&conf->node, &njupe_items);
    MyFree(conf);
    break;

  case CJUPE_TYPE:
    MyFree(conf);
    break;

  case CLASS_TYPE:
    dlinkDelete(&conf->node, &class_items);
    MyFree(conf);
    break;

  default:
    break;
  }
}

/* inputs       - pointer to conf to free
 * side effects - crucial password fields are zeroed, conf is freed
 */
void
free_access_item(struct AccessItem *aconf)
{
  struct ConfItem *conf;

  if (aconf == NULL)
    return;

  conf = unmap_conf_item(aconf);
  delete_conf_item(conf);
}

/* inputs       - pointer to client requesting confitem report
 *              - ConfType to report
 */
void
report_confitem_types(struct Client *source_p, ConfType type)
{
  dlink_node *ptr;
  struct ConfItem *conf = NULL;
  struct AccessItem *aconf;
  struct MatchItem *matchitem;
  struct ClassItem *classitem;
  char *host, *reason, *user, *classname, *oreason, buf[4], *p = buf;
  int port;

  switch (type)
  {
  case XLINE_TYPE:
    DLINK_FOREACH(ptr, xconf_items.head)
    {
      conf = ptr->data;
      matchitem = (struct MatchItem *)map_to_conf(conf);
      sendto_one(source_p, form_str(RPL_STATSXLINE),
                 me.name, source_p->name, conf->name, matchitem->reason);
    }
    break;

  case ULINE_TYPE:
    DLINK_FOREACH(ptr, uconf_items.head)
    {
      conf = ptr->data;
      matchitem = (struct MatchItem *)map_to_conf(conf);

      buf[0] = '\0';
      p = buf;

      if (matchitem->action & SHARED_KLINE)
        *p++ = 'K';
      else
        *p++ = 'k';

      if (matchitem->action & SHARED_JUPE)
        *p++ = 'R';
      else
        *p++ = 'r';

      if (matchitem->action & SHARED_XLINE)
        *p++ = 'X';
      else
	*p++ = 'x';

      *p++ = '\0';
      sendto_one(source_p, form_str(RPL_STATSULINE),
                 me.name, source_p->name, conf->name,
                 matchitem->user, matchitem->host, buf);
    }
    break;

  case OPER_TYPE:
    DLINK_FOREACH(ptr, oconf_items.head)
    {
      conf = ptr->data;
      aconf = (struct AccessItem *)map_to_conf(conf);
      get_printable_conf(conf, &host, &reason, &user, &port, &classname, &oreason);

      /* Don't allow non opers to see oper privs */
      if (IsOper(source_p))
        sendto_one(source_p, form_str(RPL_STATSOLINE),
                   me.name, source_p->name, 'O', user, host,
	           conf->name, oper_privs_as_string(port), classname);
      else
        sendto_one(source_p, form_str(RPL_STATSOLINE),
                   me.name, source_p->name, 'O', user, host,
	           conf->name, "0", classname);
    }
    break;

  case CLASS_TYPE:
    DLINK_FOREACH(ptr, class_items.head)
    {
      conf = ptr->data;
      classitem = (struct ClassItem *)map_to_conf(conf);
      sendto_one(source_p, form_str(RPL_STATSYLINE),
                 me.name, source_p->name, 'Y',
	         conf->name, PingFreq(classitem),
                 ConFreq(classitem),
                 MaxTotal(classitem), MaxSendq(classitem));
    }
    break;

  case CONF_TYPE:
  case CLIENT_TYPE:
    break;

  case SERVER_TYPE:
    DLINK_FOREACH(ptr, server_items.head)
    {
      char sbuf[4];
      char *s = sbuf;

      conf = ptr->data;
      aconf = (struct AccessItem *)map_to_conf(conf);
      get_printable_conf(conf, &host, &reason, &user, &port, &classname, &oreason);

      sbuf[0] = '\0';

      if (IsConfAllowAutoConn(aconf))
        *s++ = 'A';
      if (aconf->fakename)
        *s++ = 'M';
      if (sbuf[0] == '\0')
        *s++ = '*';

      *s++ = '\0';

      /* Allow admins to see actual ips
       * unless hide_server_ips is enabled
       */
      if (!ConfigServerHide.hide_server_ips && IsAdmin(source_p))
        sendto_one(source_p, form_str(RPL_STATSCLINE),
                   me.name, source_p->name, 'C', host,
	           sbuf, conf->name, port, classname);
        else
          sendto_one(source_p, form_str(RPL_STATSCLINE),
                     me.name, source_p->name, 'C',
	             "*@masked", sbuf, conf->name, port, classname);
    }
    break;

  case HUB_TYPE:
    DLINK_FOREACH(ptr, hub_items.head)
    {
      conf = ptr->data;
      matchitem = (struct MatchItem *)map_to_conf(conf);
      sendto_one(source_p, form_str(RPL_STATSHLINE), me.name,
		 source_p->name, 'H', matchitem->host, conf->name, 0, "*");
    }
    break;

  case LEAF_TYPE:
    DLINK_FOREACH(ptr, leaf_items.head)
    {
      conf = ptr->data;
      matchitem = (struct MatchItem *)map_to_conf(conf);
      sendto_one(source_p, form_str(RPL_STATSLLINE), me.name,
		 source_p->name, 'L', matchitem->host, conf->name, 0, "*");
    }
    break;

  case KLINE_TYPE:
  case DLINE_TYPE:
  case EXEMPTDLINE_TYPE:
  case CJUPE_TYPE:
  case NJUPE_TYPE:
    break;
  }
}

/* inputs	- pointer to client
 * output	- 0 = Success
 * 		  NOT_AUTHORIZED (-1) = Access denied (no I line match)
 * 		  I_LINE_FULL    (-2) = I-Line full
 *		  TOO_MANY       (-3) = Too many connections from hostname
 * 		  BANNED_CLIENT  (-4) = K-Lined
 *		  TOO_FAST	 (-5) = Connecting too fast
 * side effects - Ordinary client access check.
 *		  Look for conf lines which have the same
 * 		  status as the flags passed.
 */
int
check_client(struct Client *client_p, struct Client *source_p, const char *username)
{
  int i;
 
  if ((i = verify_access(source_p, username)))
    ilog(INFO, "Access denied: %s[%s]",
         source_p->name, source_p->localClient->sockhost);

  switch (i)
  {
    case TOO_MANY:
      sendto_realops_flags(UMODE_CCONN, L_ALL, 
                           "Too many on IP %s (%s).",
			   get_client_name(source_p, SHOW_IP),
			   source_p->localClient->sockhost);			   
      ilog(INFO,"Too many connections from IP %s.",
	   get_client_name(source_p, SHOW_IP));
      exit_client(client_p, source_p, &me, "Too many connections from IP");
      break;

    case I_LINE_FULL:
      sendto_realops_flags(UMODE_CCONN, L_ALL,
                           "I-Line full for %s (%s).",
			   get_client_name(source_p, SHOW_IP),
			   source_p->localClient->sockhost);			   
      ilog(INFO,"Too many connections from %s.",
	   get_client_name(source_p, SHOW_IP));
      exit_client(client_p, source_p, &me, "Connection class full");
      break;

    case NOT_AUTHORIZED:
    {
      static char ipaddr[HOSTIPLEN];

      irc_getnameinfo((struct sockaddr*)&source_p->localClient->ip,
		      source_p->localClient->ip.ss_len, ipaddr, HOSTIPLEN, NULL, 0,
		      NI_NUMERICHOST);
      sendto_realops_flags(UMODE_UNAUTH, L_ALL,
			   "Unauthorized client connection from %s %s on %s/%u.",
			   get_client_name(source_p, SHOW_IP),
			   ipaddr, source_p->localClient->listener->name,
			   source_p->localClient->listener->port);
      ilog(INFO, "Unauthorized client connection from %s on %s/%u.",
	   get_client_name(source_p, SHOW_IP),
	   source_p->localClient->listener->name,
	   source_p->localClient->listener->port);
	  
      exit_client(client_p, source_p, &me, "Unauthorized connection");
      break;
    }
    case BANNED_CLIENT:
      exit_client(client_p, source_p, &me, "K-Lined");
      break;
    case 0:
    default:
      break;
  }
  return(i);
}

/* inputs	- pointer to client to verify
 *		- pointer to proposed username
 * output	- 0 if success -'ve if not
 * side effect	- find the first (best) I line to attach.
 */
static int
verify_access(struct Client *client_p, const char *username)
{
  struct AccessItem *aconf;
  struct ConfItem *conf;
  char non_ident[USERLEN + 1];

  if (IsGotId(client_p))
  {
    aconf = find_address_conf(client_p->host,client_p->username,
                              &client_p->localClient->ip,
			      client_p->localClient->aftype,
			      client_p->localClient->passwd);
  }
  else
  {
    strlcpy(non_ident, "~", sizeof(non_ident));
    strlcat(non_ident, username, sizeof(non_ident));
    aconf = find_address_conf(client_p->host,non_ident,
                              &client_p->localClient->ip,
                              client_p->localClient->aftype,
			      client_p->localClient->passwd);
  }

  if (aconf != NULL)
  {
    if (IsConfClient(aconf))
    {
      conf = unmap_conf_item(aconf);
      if (IsConfDoIdentd(aconf))
	SetNeedId(client_p);

      if (IsConfRestricted(aconf))
	SetRestricted(client_p);

      if (IsConfDoSpoofIp(aconf))
      {
        conf = unmap_conf_item(aconf);
        if (!ConfigFileEntry.hide_spoof_ips && IsConfSpoofNotice(aconf))
          sendto_realops_flags(UMODE_ALL, L_ADMIN, "%s spoofing: %s as %s",
			       client_p->name, client_p->host, conf->name);
	strlcpy(client_p->host, conf->name, sizeof(client_p->host));
	SetIPSpoof(client_p);
      }
      return(attach_iline(client_p, conf));
    }
    else if (IsConfKill(aconf))
      return (BANNED_CLIENT);
  }
  return(NOT_AUTHORIZED);
}

/* inputs	- client pointer
 *		- conf pointer
 * side effects	- do actual attach
 */
static int
attach_iline(struct Client *client_p, struct ConfItem *conf)
{
  struct AccessItem *aconf;
  struct ClassItem *aclass;
  struct ip_entry *ip_found;
  int a_limit_reached = 0;
  int max_limit_reached = 0;
  int local;
  int global;
  int ident;

  ip_found = find_or_add_ip(&client_p->localClient->ip);

  SetIpHash(client_p);
  ip_found->count++;

  aconf = (struct AccessItem *)map_to_conf(conf);
  /* only check it if its non zero */
  if (aconf->class_ptr != NULL)
  {
    aclass = (struct ClassItem *)map_to_conf(aconf->class_ptr);
    count_user_host(client_p->username, client_p->host,
                    &global, &local, &ident);

    if ((MaxTotal(aclass) != 0) &&
        (CurrUserCount(aclass) > MaxTotal(aclass)))
    {
      max_limit_reached = 1;
      a_limit_reached = 1;
    }
    if ((MaxPerIp(aclass) != 0) && (ip_found->count > MaxPerIp(aclass)))
      a_limit_reached = 1;
    else if ((MaxLocal(aclass) != 0) && (local > MaxLocal(aclass)))
      a_limit_reached = 1;
    else if ((MaxGlobal(aclass) != 0) && (global > MaxGlobal(aclass)))
      a_limit_reached = 1;

    if ((MaxIdent(aclass) != 0) && ((*client_p->username != '~') &&
				    ident < MaxIdent(aclass)) &&
        !max_limit_reached)
      a_limit_reached = 0;

    if (a_limit_reached)
    {
      if (!IsConfExemptLimits(aconf))
        return(TOO_MANY); /* Already at maximum allowed */
      else
      {
        sendto_one(client_p,
                   ":%s NOTICE %s :*** Your connection class is full, "
                   "but you have exceed_limit = yes;", me.name, client_p->name);
      }
    }
  }
  else
    return(NOT_AUTHORIZED);

  return(attach_conf(client_p, conf));
}

/* side effects         - allocate memory for ip_entry(s)
 *                      - clear the ip hash table
 */
void
init_ip_hash_table(void)
{
  ip_entry_heap = BlockHeapCreate(sizeof(struct ip_entry), 2*HARD_FDLIMIT);
  memset((void *)ip_hash_table, 0, sizeof(ip_hash_table));
}

/* inputs	- pointer to struct irc_ssaddr
 * output       - pointer to a struct ip_entry
 * If the ip # was not found, a new struct ip_entry is created, and the ip
 * count set to 0.
 */
static struct ip_entry *
find_or_add_ip(struct irc_ssaddr *ip_in)
{
  struct ip_entry *ptr, *newptr;
  int hash_index = hash_ip(ip_in), res;
  struct sockaddr_in *v4 = (struct sockaddr_in *)ip_in, *ptr_v4;
#ifdef IPV6
  struct sockaddr_in6 *v6 = (struct sockaddr_in6 *)ip_in, *ptr_v6;
#endif

  for (ptr = ip_hash_table[hash_index]; ptr; ptr = ptr->next)
  {
#ifdef IPV6
    if (ptr->ip.ss.ss_family != ip_in->ss.ss_family)
      continue;
    if (ip_in->ss.ss_family == AF_INET6)
    {
      ptr_v6 = (struct sockaddr_in6 *)&ptr->ip;
      res = memcmp(&v6->sin6_addr, &ptr_v6->sin6_addr, sizeof(struct in6_addr));
    }
    else
#endif
    {
      ptr_v4 = (struct sockaddr_in *)&ptr->ip;
      res = memcmp(&v4->sin_addr, &ptr_v4->sin_addr, sizeof(struct in_addr));
    }
    if (res == 0)
    {
      /* Found entry already in hash, return it. */
      return(ptr);
    }
  }

  if (ip_entries_count >= (2*HARD_FDLIMIT))
    garbage_collect_ip_entries();

  newptr = BlockHeapAlloc(ip_entry_heap);
  ip_entries_count++;
  memcpy(&newptr->ip, ip_in, sizeof(struct irc_ssaddr));
  newptr->count = 0;
  newptr->last_attempt = 0;

  if ((ptr = ip_hash_table[hash_index]) != NULL)
    newptr->next = ptr;
  else
    newptr->next = NULL;

  ip_hash_table[hash_index] = newptr;
  return(newptr);
}

/* inputs        - unsigned long IP address value
 * side effects  - The ip address given, is looked up in ip hash table
 *                 and number of ip#'s for that ip decremented.
 *                 If ip # count reaches 0 and has expired,
 *                 the struct ip_entry is returned to the ip_entry_heap
 */
void
remove_one_ip(struct irc_ssaddr *ip_in)
{
  struct ip_entry *ptr, *last_ptr = NULL;
  int hash_index = hash_ip(ip_in), res;
  struct sockaddr_in *v4 = (struct sockaddr_in *)ip_in, *ptr_v4;
#ifdef IPV6
  struct sockaddr_in6 *v6 = (struct sockaddr_in6 *)ip_in, *ptr_v6;
#endif

  for (ptr = ip_hash_table[hash_index]; ptr; ptr = ptr->next)
  {
#ifdef IPV6
    if (ptr->ip.ss.ss_family != ip_in->ss.ss_family)
      continue;
    if (ip_in->ss.ss_family == AF_INET6)
    {
      ptr_v6 = (struct sockaddr_in6 *)&ptr->ip;
      res = memcmp(&v6->sin6_addr, &ptr_v6->sin6_addr, sizeof(struct in6_addr));
    }
    else
#endif
    {
      ptr_v4 = (struct sockaddr_in *)&ptr->ip;
      res = memcmp(&v4->sin_addr, &ptr_v4->sin_addr, sizeof(struct in_addr));
    }
    if (res)
      continue;
    if (ptr->count > 0)
      ptr->count--;
    if (ptr->count == 0 &&
        (CurrentTime-ptr->last_attempt) >= ConfigFileEntry.throttle_time)
    {
      if (last_ptr != NULL)
        last_ptr->next = ptr->next;
      else
        ip_hash_table[hash_index] = ptr->next;

      BlockHeapFree(ip_entry_heap, ptr);
      ip_entries_count--;
      return;
    }
    last_ptr = ptr;
  }
}

/* input        - pointer to an irc_inaddr
 * output       - integer value used as index into hash table
 */
static int
hash_ip(struct irc_ssaddr *addr)
{
  if (addr->ss.ss_family == AF_INET)
  {
    struct sockaddr_in *v4 = (struct sockaddr_in *)addr;
    int hash;
    u_int32_t ip;

    ip   = ntohl(v4->sin_addr.s_addr);
    hash = ((ip >> 12) + ip) & (IP_HASH_SIZE-1);
    return(hash);
  }
#ifdef IPV6
  else
  {
    int hash;
    struct sockaddr_in6 *v6 = (struct sockaddr_in6 *)addr;
    u_int32_t *ip = (u_int32_t *)&v6->sin6_addr.s6_addr;

    hash  = ip[0] ^ ip[3];
    hash ^= hash >> 16;
    hash ^= hash >> 8;
    hash  = hash & (IP_HASH_SIZE - 1);
    return(hash);
  }
#else
  return(0);
#endif
}

/* inputs        - pointer to counter of number of ips hashed
 *               - pointer to memory used for ip hash
 * output        - returned via pointers input
 * number of hashed ip #'s is counted up, plus the amount of memory
 * used in the hash.
 */
void
count_ip_hash(int *number_ips_stored, unsigned long *mem_ips_stored)
{
  struct ip_entry *ptr;
  int i;

  *number_ips_stored = 0;
  *mem_ips_stored    = 0;

  for (i = 0; i < IP_HASH_SIZE; i++)
  {
    for (ptr = ip_hash_table[i]; ptr; ptr = ptr->next)
    {
      *number_ips_stored += 1;
      *mem_ips_stored += sizeof(struct ip_entry);
    }
  }
}

/* side effects - free up all ip entries with no connections */
static void
garbage_collect_ip_entries(void)
{
  struct ip_entry *ptr, *last_ptr, *next_ptr;
  int i;

  for (i = 0; i < IP_HASH_SIZE; i++)
  {
    last_ptr = NULL;

    for (ptr = ip_hash_table[i]; ptr; ptr = next_ptr)
    {
      next_ptr = ptr->next;

      if (ptr->count == 0)
      {
        if (last_ptr != NULL)
          last_ptr->next = ptr->next;
        else
          ip_hash_table[i] = ptr->next;
        BlockHeapFree(ip_entry_heap, ptr);
        ip_entries_count--;
      }
      else
        last_ptr = ptr;
    }
  }
}

/* inputs       - pointer to client to detach
 *              - type of conf to detach
 * output       - 0 for success, -1 for failure
 * side effects - Disassociate configuration from the client.
 *                Also removes a class from the list if marked for deleting.
 */
int
detach_conf(struct Client *client_p, ConfType type)
{
  dlink_node *ptr;
  struct ConfItem *conf;
  struct ClassItem *aclass;
  struct AccessItem *aconf;
  struct ConfItem *aclass_conf;
  struct MatchItem *match_item;

  DLINK_FOREACH(ptr, client_p->localClient->confs.head)
  {
    conf = ptr->data;
    if (conf->type == type)
    {
      dlinkDelete(ptr, &client_p->localClient->confs);
      free_dlink_node(ptr);

      switch (conf->type)
      {
        case CLIENT_TYPE:
        case OPER_TYPE:
        case SERVER_TYPE:
          aconf = (struct AccessItem *)map_to_conf(conf);
          if ((aclass_conf = ClassPtr(aconf)) != NULL)
          {
             aclass = (struct ClassItem *)map_to_conf(aclass_conf);
             if (CurrUserCount(aclass) > 0)
               aclass->curr_user_count--;

             if (MaxTotal(aclass) < 0 && CurrUserCount(aclass) <= 0)
	       delete_conf_item(aclass_conf);
           }

	/* Please, no ioccc entries - Dianora */
	if (aconf->clients > 0)
	  --aconf->clients;
	if (aconf->clients == 0 && IsConfIllegal(aconf))
	  delete_conf_item(conf);
	break;
	      case LEAF_TYPE:
	      case HUB_TYPE:
	match_item = (struct MatchItem *)map_to_conf(conf);
	if ((match_item->ref_count == 0) && (match_item->illegal))
	  delete_conf_item(conf);
	break;
      default:
	break;
      }
      return(0);
    }
  }
  return(-1);
}

/* inputs       - pointer to client to detach
 * side effects - Disassociate all configuration from the client.
 *                Also removes a class from the list if marked for deleting.
 */
void
detach_all_confs(struct Client *client_p)
{
  dlink_node *ptr;
  dlink_node *next_ptr;
  struct ConfItem *conf;
  struct ClassItem *aclass;
  struct AccessItem *aconf;
  struct ConfItem *aclass_conf;
  struct MatchItem *match_item;

  DLINK_FOREACH_SAFE(ptr, next_ptr, client_p->localClient->confs.head)
  {
    conf = ptr->data;

    dlinkDelete(ptr, &client_p->localClient->confs);
    free_dlink_node(ptr);

    switch(conf->type)
    {
    case CLIENT_TYPE:
    case OPER_TYPE:
    case SERVER_TYPE:
      aconf = (struct AccessItem *)map_to_conf(conf);

      if ((aclass_conf = ClassPtr(aconf)) != NULL)
      {
        aclass = (struct ClassItem *)map_to_conf(aclass_conf);
        if (CurrUserCount(aclass) > 0)
          aclass->curr_user_count--;

        if (MaxTotal(aclass) < 0 && CurrUserCount(aclass) <= 0)
        {
          delete_conf_item(aclass_conf);
        }
      }

      /* Please, no ioccc entries - Dianora */
      if (aconf->clients > 0)
        --aconf->clients;
      if (aconf->clients == 0 && IsConfIllegal(aconf))
        delete_conf_item(conf);
      break;
    case LEAF_TYPE:
    case HUB_TYPE:
      match_item = (struct MatchItem *)map_to_conf(conf);
      if ((match_item->ref_count == 0) && (match_item->illegal))
        delete_conf_item(conf);
      break;
    default:
      break;
    }
  }
}

/* inputs       - client pointer
 *              - conf pointer
 * side effects - Associate a specific configuration entry to a *local*
 *                client (this is the one which used in accepting the
 *                connection). Note, that this automatically changes the
 *                attachment if there was an old one...
 */
int
attach_conf(struct Client *client_p, struct ConfItem *conf)
{
  struct AccessItem *aconf;
  struct MatchItem *match_item;

  if (dlinkFind(&client_p->localClient->confs, conf) != NULL)
    return(1);

  if ((conf->type == CLIENT_TYPE) ||
     (conf->type == SERVER_TYPE) ||
     (conf->type == OPER_TYPE))
  {
    aconf = (struct AccessItem *)map_to_conf(conf);

    if (IsConfIllegal(aconf))
      return(NOT_AUTHORIZED);

    if (conf->type == CLIENT_TYPE)
    {
      struct ClassItem *aclass;
      aclass = (struct ClassItem *)map_to_conf(aconf->class_ptr);

      if (MaxTotal(aclass) > 0)
      {
        if (CurrUserCount(aclass) >= MaxTotal(aclass))
        {
          if (!IsConfExemptLimits(aconf))
            return(I_LINE_FULL);
          else
          {
            sendto_one(client_p, ":%s NOTICE %s :*** Your connection class is "
                       "full, but you have exceed_limit = yes;",
                       me.name, client_p->name);
            SetExemptLimits(client_p);
          }
        }
        CurrUserCount(aclass)++;
      }
      else
        return(NOT_AUTHORIZED);
    }
    aconf->clients++;
  }
  else if ((conf->type == HUB_TYPE) || (conf->type == LEAF_TYPE))
  {
    match_item = (struct MatchItem *)map_to_conf(conf);
    match_item->ref_count++;
  }
  dlinkAdd(conf, make_dlink_node(), &client_p->localClient->confs);

  return(0);
}

/* inputs       - pointer to server to attach
 *              - name of server
 *              - hostname of server
 * output       - true (1) if both are found, otherwise return false (0)
 * side effects - find connect block and attach them to connecting client
 */
int
attach_connect_block(struct Client *client_p, const char *name,
                     const char *host)
{
  dlink_node *ptr;
  struct ConfItem *conf;
  struct AccessItem *aconf;

  if (client_p == NULL || host == NULL)
    return(0);

  DLINK_FOREACH(ptr, server_items.head)
  {
    conf = ptr->data;
    aconf = (struct AccessItem *)map_to_conf(conf);

    if (!match(conf->name, name) || !match(aconf->host, host))
      continue;

    attach_conf(client_p, conf);
    return(-1);
  }

  return(0);
}

/* inputs       - type of ConfItem
 *              - pointer to name to find
 *              - pointer to username to find
 *              - pointer to host to find
 * output       - NULL or pointer to conf found
 * side effects - find a conf entry which matches the hostname
 *                and has the same name.
 */
struct ConfItem *
find_conf_exact(ConfType type, const char *name, const char *user,
                const char *host)
{
  dlink_node *ptr;
  dlink_list *list_p;
  struct ConfItem *conf=NULL;
  struct AccessItem *aconf;

  /* Only valid for OPER_TYPE and ...? */
  list_p = map_to_list(type);

  DLINK_FOREACH(ptr, (*list_p).head)
  {
    conf = ptr->data;

    if (conf->name == NULL)
      continue;
    aconf = (struct AccessItem *)map_to_conf(conf);
    if (aconf->host == NULL)
      continue;
    if (irccmp(conf->name, name) != 0)
      continue;

    /* Accept if the *real* hostname (usually sockethost)
     * socket host) matches *either* host or name field
     * of the configuration.
     */
    if (!match(aconf->host, host) || !match(aconf->user,user)
        || irccmp(conf->name, name) )
      continue;
    if (type == OPER_TYPE)
    {
      struct ClassItem *aclass;

      aclass = (struct ClassItem *)aconf->class_ptr;
      if (aconf->clients < MaxTotal(aclass))
        return(conf);
      else
        continue;
    }
    else
      return(conf);
  }
  return(NULL);
}

/* inputs       - pointer to conf link list to search
 *              - pointer to name to find
 *              - int mask of type of conf to find
 * output       - NULL or pointer to conf found
 * side effects - find a conf entry which matches the name
 *                and has the given mask.
 */
struct ConfItem *
find_conf_name(dlink_list *list, const char *name, ConfType type)
{
  dlink_node *ptr;
  struct ConfItem* conf;

  DLINK_FOREACH(ptr, list->head)
  {
    conf = ptr->data;

    if (conf->type == type)
    {
      if (conf->name && (!irccmp(conf->name, name) ||
  	  match(conf->name, name)))
      return(conf);
    }
  }

  return(NULL);
}

/* inputs       - ConfType conf
 * output       - pointer to dlink_list to use
 */
static dlink_list *
map_to_list(ConfType type)
{
  switch (type)
  {
  case XLINE_TYPE:
    return(&xconf_items);
    break;
  case ULINE_TYPE:
    return(&uconf_items);
    break;
  case NJUPE_TYPE:
    return(&njupe_items);
    break;
  case OPER_TYPE:
    return(&oconf_items);
    break;
  case CLASS_TYPE:
    return(&class_items);
    break;
  case SERVER_TYPE:
    return(&server_items);
    break;
  case CONF_TYPE:
  case KLINE_TYPE:
  case DLINE_TYPE:
  case CJUPE_TYPE:
  default:
    return(NULL);
  }
}

/* inputs       - type of link list to look in
 *              - pointer to name string to find
 *              - pointer to user
 *              - pointer to host
 *              - optional action to match on as well
 * output       - NULL or pointer to found struct MatchItem
 * side effects - looks for a match on name field
 */
struct ConfItem *
find_matching_name_conf(ConfType type, const char *name, const char *user,
                        const char *host, int action)
{
  dlink_node *ptr=NULL;
  struct ConfItem *conf=NULL;
  struct AccessItem *aconf=NULL;
  struct MatchItem *match_item=NULL;
  dlink_list *list_p = map_to_list(type);

  switch (type)
  {
    case XLINE_TYPE:
    case ULINE_TYPE:
    case NJUPE_TYPE:

    DLINK_FOREACH(ptr, (*list_p).head)
    {
      conf = ptr->data;

      match_item = map_to_conf(conf);
      if (EmptyString(conf->name))
        continue;
      if ((name != NULL) && match_esc(conf->name, name))
      {
        if ((user == NULL && (host == NULL)))
          return (conf);
        if ((match_item->action & action) != action)
          continue;
        if (EmptyString(match_item->user) || EmptyString(match_item->host))
          return (conf);
        if (match(match_item->user, user) && match(match_item->host, host))
          return (conf);
      }
    }
    break;

  case OPER_TYPE:
    DLINK_FOREACH(ptr, (*list_p).head)
    {
      conf = ptr->data;

      if ((name != NULL) && (!irccmp(name, conf->name)))
      {
        if ((user == NULL && (host == NULL)))
          return (conf);

        aconf = map_to_conf(conf);
        if (EmptyString(aconf->user) || EmptyString(aconf->host))
          return (conf);
        if (match(aconf->user, user) && match(aconf->host, host))
          return (conf);
      }
    }
    break;

    case SERVER_TYPE:
      DLINK_FOREACH(ptr, (*list_p).head)
      {
        conf = ptr->data;
        aconf = map_to_conf(conf);

        if ((name != NULL) && match_esc(name, conf->name))
          return(conf);
        else if ((host != NULL) && match_esc(host, aconf->host))
          return(conf);
      }
      break;
  default:
    break;
  }
  return(NULL);
}

/* inputs       - type of link list to look in
 *              - pointer to name string to find
 *              - pointer to user
 *              - pointer to host
 * output       - NULL or pointer to found struct MatchItem
 * side effects - looks for an exact match on name field
 */
struct ConfItem *
find_exact_name_conf(ConfType type, const char *name,
                     const char *user, const char *host)
{
  dlink_node *ptr = NULL;
  struct ConfItem *conf;
  struct AccessItem *aconf;
  struct MatchItem *match_item;
  struct ClassItem *aclass;
  dlink_list *list_p;

  list_p = map_to_list(type);

  switch(type)
  {
  case XLINE_TYPE:
  case ULINE_TYPE:
  case NJUPE_TYPE:

    DLINK_FOREACH(ptr, (*list_p).head)
    {
      conf = ptr->data;
      match_item = (struct MatchItem *)map_to_conf(conf);
      if (EmptyString(conf->name))
        continue;

      if (!irccmp(conf->name, name))
      {
        if ((user == NULL && (host == NULL)))
          return (conf);
        if (EmptyString(match_item->user) || EmptyString(match_item->host))
          return (conf);
        if (match(match_item->user, user) && match(match_item->host, host))
          return (conf);
      }
    }
    break;

  case OPER_TYPE:
    DLINK_FOREACH(ptr, (*list_p).head)
    {
      conf = ptr->data;
      aconf = (struct AccessItem *)map_to_conf(conf);
      if (EmptyString(conf->name))
        continue;

      if (!irccmp(conf->name, name))
      {
        if ((user == NULL && (host == NULL)))
          return (conf);
        if (EmptyString(aconf->user) || EmptyString(aconf->host))
          return (conf);
        if (match(aconf->user, user) && match(aconf->host, host))
          return (conf);
      }
    }
    break;

  case SERVER_TYPE:
    DLINK_FOREACH(ptr, (*list_p).head)
    {
      conf = ptr->data;
      aconf = (struct AccessItem *)map_to_conf(conf);
      if (EmptyString(conf->name))
        continue;

      if (name == NULL)
      {
        if (EmptyString(aconf->host))
          continue;
        if (!irccmp(aconf->host, host))
          return(conf);
      }
      else if (!irccmp(conf->name, name))
      {
          return (conf);
      }
    }
    break;

  case CLASS_TYPE:
    DLINK_FOREACH(ptr, (*list_p).head)
    {
      conf = ptr->data;
      aclass = (struct ClassItem *)map_to_conf(conf);
      if (EmptyString(conf->name))
        continue;

      if (!irccmp(conf->name, name))
        return (conf);
    }
    break;

  default:
    break;
  }
  return(NULL);
}

/* Actual REHASH service routine. Called with sig == 0 if it has been called
 * as a result of an operator issuing this command, else assume it has been
 * called as a result of the server receiving a HUP signal.
 */
int
rehash(int sig)
{
  if (sig != 0)
    sendto_realops_flags(UMODE_ALL, L_ALL, "SIGHUP caught, rehashing ircd.conf");
  restart_resolver();
  check_can_use_v6();
  read_conf_files(0);

  if (ServerInfo.description != NULL)
    strlcpy(me.info, ServerInfo.description, sizeof(me.info));

  flush_deleted_I_P();
  rehashed_klines = 1;
  rehashed_xlines = 1;

  if (ConfigLoggingEntry.use_logging)
    reopen_log(logFileName);

  return(0);
}

/* side effects - Set default values here.
 *                This is called **PRIOR** to parsing the
 *                configuration file.  If you want to do some validation
 *                of values later, put them in validate_conf().
 */
static void
set_default_conf(void)
{
  ServerInfo.description = NULL;
  DupString(ServerInfo.network_name, NETWORK_NAME_DEFAULT);

  memset(&ServerInfo.ip, 0, sizeof(ServerInfo.ip));
  ServerInfo.specific_ipv4_vhost = 0;
  memset(&ServerInfo.ip6, 0, sizeof(ServerInfo.ip6));
  ServerInfo.specific_ipv6_vhost = 0;

  ServerInfo.max_clients = MAXCONN;
  ServerInfo.dns_host.sin_addr.s_addr = 0;
  ServerInfo.dns_host.sin_port = 0;
  AdminInfo.name = NULL;
  AdminInfo.email = NULL;
  AdminInfo.description = NULL;

  set_log_level(NOTICE);
  ConfigLoggingEntry.use_logging = 1;
  ConfigLoggingEntry.operlog[0] = '\0';
  ConfigLoggingEntry.userlog[0] = '\0';
  ConfigLoggingEntry.failed_operlog[0] = '\0';

  ConfigChannel.cjupe_dline = 1;
  ConfigChannel.disable_local_channels = 0;
  ConfigChannel.knock_delay = 300;
  ConfigChannel.knock_delay_channel = 60;
  ConfigChannel.max_chans_per_user = 25;
  ConfigChannel.quiet_on_ban = 1;
  ConfigChannel.max_bans = 100;
  ConfigChannel.default_split_user_count = 0;
  ConfigChannel.default_split_server_count = 0;
  ConfigChannel.no_join_on_split = 0;
  ConfigChannel.no_create_on_split = 0;

  ConfigServerHide.flatten_links = 0;
  ConfigServerHide.links_delay = 300;
  ConfigServerHide.hidden = 0;
  ConfigServerHide.disable_hidden = 0;
  ConfigServerHide.hide_servers = 0;
  ConfigServerHide.hide_server_ips = 0;

  ConfigFileEntry.use_services = 0;
  ConfigFileEntry.hide_spoof_ips = 1;
  ConfigFileEntry.disable_auth = 0;
  ConfigFileEntry.disable_remote = 1;
  ConfigFileEntry.kill_chase_time_limit = 90;
  ConfigFileEntry.default_floodcount = 8;
  ConfigFileEntry.dots_in_ident = 0;
  ConfigFileEntry.min_nonwildcard = 4;
  ConfigFileEntry.min_nonwildcard_simple = 3;
  ConfigFileEntry.max_accept = 20;
  ConfigFileEntry.anti_nick_flood = 0;
  ConfigFileEntry.max_nick_time = 20;
  ConfigFileEntry.max_nick_changes = 5;
  ConfigFileEntry.anti_spam_exit_message_time = 0;
  ConfigFileEntry.ts_warn_delta = TS_WARN_DELTA_DEFAULT;
  ConfigFileEntry.ts_max_delta = TS_MAX_DELTA_DEFAULT;
  ConfigFileEntry.hide_ban_reason = 0;
  ConfigFileEntry.stats_o_oper_only = 1;
  ConfigFileEntry.stats_k_oper_only = 1;
  ConfigFileEntry.stats_i_oper_only = 1;
  ConfigFileEntry.stats_P_oper_only = 0;
  ConfigFileEntry.ssignore_wait = 60;
  ConfigFileEntry.pace_wait = 10;
  ConfigFileEntry.pace_wait_simple = 1;
  ConfigFileEntry.no_oper_flood = 0;
  ConfigFileEntry.true_no_oper_flood = 0;
  ConfigFileEntry.oper_pass_jupes = 1;
  ConfigFileEntry.drone_detect = 1;
  ConfigFileEntry.autodline_drones = 1;
  ConfigFileEntry.drone_dline_time = 7200;
  ConfigFileEntry.cjupe_dline_time = 7200;
  ConfigFileEntry.idletime = 0;
  ConfigFileEntry.maximum_links = MAXIMUM_LINKS_DEFAULT;
  ConfigFileEntry.max_targets = MAX_TARGETS_DEFAULT;
  ConfigFileEntry.client_flood = CLIENT_FLOOD_DEFAULT;
  ConfigFileEntry.oper_umodes = UMODE_LOCOPS | UMODE_SERVNOTICE | UMODE_WALLOP;
  ConfigFileEntry.crypt_oper_password = 1;
  ConfigFileEntry.throttle_time = 10;
}

/* inputs       - file descriptor pointing to config file to use
 * side effects - Read configuration file.
 */
static void
read_conf(FBFILE *file)
{
  scount = lineno = 0;

  set_default_conf(); /* Set default values prior to conf parsing */
  ypass = 1;
  yyparse();          /* pick up the classes first */
  (void)fbrewind(file);
  ypass = 2;
  yyparse();          /* Load the values from the conf */
  validate_conf();    /* Check to make sure some values are still okay. */
                      /* Some global values are also loaded here. */
  check_class();      /* Make sure classes are valid */
}

static void
validate_conf(void)
{
  if (ConfigFileEntry.ts_warn_delta < TS_WARN_DELTA_MIN)
    ConfigFileEntry.ts_warn_delta = TS_WARN_DELTA_DEFAULT;

  if (ConfigFileEntry.ts_max_delta < TS_MAX_DELTA_MIN)
    ConfigFileEntry.ts_max_delta = TS_MAX_DELTA_DEFAULT;

  if (ServerInfo.network_name == NULL)
    DupString(ServerInfo.network_name,NETWORK_NAME_DEFAULT);

  if ((ConfigFileEntry.client_flood < CLIENT_FLOOD_MIN) ||
      (ConfigFileEntry.client_flood > CLIENT_FLOOD_MAX))
     ConfigFileEntry.client_flood = CLIENT_FLOOD_MAX;

  GlobalSetOptions.idletime = (ConfigFileEntry.idletime * 60);
}

/* inputs       - pointer to original string of form "user@host"
 *              - pointer to new user part
 *              - pointer to new host part
 * side effects - splits user@host found in a name field of conf given
 *                stuff the user into ->user and the host into ->host
 */
void
split_user_host(char *user_host, char **user_p, char **host_p)
{
  char *p;
  char *new_user;
  char *new_host;

  if ((p = strchr(user_host, '@')) != NULL)
  {
    *p = '\0';
    DupString(new_user, user_host);
    p++;
    DupString(new_host, p);
    MyFree(user_host);
    *user_p = new_user;
    *host_p = new_host;
  }
  else
  {
    DupString(*user_p, "*");
  }
}

/* start DNS lookups of all hostnames in the conf
 * line and convert an IP addresses in a.b.c.d number for to IP#s.
 */
static void
lookup_confhost(struct ConfItem *conf)
{
  struct AccessItem *aconf;
  struct addrinfo hints, *res;
  int ret;

  aconf = map_to_conf(conf);

  if (EmptyString(aconf->host) ||
      EmptyString(aconf->user))
  {
    ilog(ERROR, "Host/server name error: (%s) (%s)",
         aconf->host, conf->name);
    return;
  }

  if (strchr(aconf->host, '*') ||
      strchr(aconf->host, '?'))
    return;

  /* Do name lookup now on hostnames given and store the
   * ip numbers in conf structure.
   */
  memset(&hints, 0, sizeof(hints));

  hints.ai_family   = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;

  /* Get us ready for a bind() and don't bother doing dns lookup */
  hints.ai_flags = AI_PASSIVE | AI_NUMERICHOST;

  if ((ret = irc_getaddrinfo(aconf->host, NULL, &hints, &res)))
  {
    conf_dns_lookup(aconf);
    return;
  }

  memcpy(&aconf->ipnum, res->ai_addr, res->ai_addrlen);
  aconf->ipnum.ss_len = res->ai_addrlen;
  aconf->ipnum.ss.ss_family = res->ai_family;
  irc_freeaddrinfo(res);
}

/* inputs       - pointer to inaddr
 *              - int type ipv4 or ipv6
 * output       - BANNED or accepted
 */
int
conf_connect_allowed(struct irc_ssaddr *addr, int aftype)
{
  struct ip_entry *ip_found;
  struct AccessItem *aconf = find_dline_conf(addr, aftype);

  /* DLINE exempt also gets you out of static limits/pacing... */
  if (aconf && (aconf->status & CONF_EXEMPTDLINE))
    return(0);

  if (aconf != NULL)
    return(BANNED_CLIENT);

  ip_found = find_or_add_ip(addr);

  if ((CurrentTime - ip_found->last_attempt) <
      ConfigFileEntry.throttle_time)
  {
    ip_found->last_attempt = CurrentTime;
    return(TOO_FAST);
  }

  ip_found->last_attempt = CurrentTime;
  return(0);
}

/* inputs       - pointer to client structure
 * output       - pointer to struct AccessItem if found
 * side effects - See if this user is klined already,
 *                and if so, return struct AccessItem pointer
 */
struct AccessItem *
find_kill(struct Client *client_p)
{
  struct AccessItem *aconf;

  if (client_p == NULL)
    return(NULL);

  aconf = find_kline_conf(client_p->host, client_p->username,
                          &client_p->localClient->ip,
                          client_p->localClient->aftype);
  if (aconf == NULL)
    return(NULL);
  if (aconf->status & CONF_KLINE)
    return(aconf);

  return(NULL);
}

/* inputs        - pointer to struct AccessItem
 * Side effects  - links in given struct AccessItem into
 *                 temporary kline link list
 */
void
add_tkline(struct AccessItem *aconf)
{
  dlinkAdd(aconf, make_dlink_node(), &temporary_klines);
  SetConfTemporary(aconf);
  add_conf_by_address(CONF_KLINE, aconf);
}

/* inputs        - pointer to struct AccessItem
 * Side effects  - links in given struct AccessItem into
 *                 temporary dline link list
 */
void
add_tdline(struct AccessItem *aconf)
{
  dlinkAdd(aconf, make_dlink_node(), &temporary_dlines);
  SetConfTemporary(aconf);
  MyFree(aconf->user);
  aconf->user = NULL;
  add_conf_by_address(CONF_DLINE, aconf);
}

void
cleanup_tklines(void *n)
{
  expire_tklines(&temporary_klines);
  expire_tklines(&temporary_dlines);
}

static void
expire_tklines(dlink_list *tklist)
{
  dlink_node *kill_node;
  dlink_node *next_node;
  struct AccessItem *kill_ptr;

  DLINK_FOREACH_SAFE(kill_node, next_node, tklist->head)
  {
    kill_ptr = kill_node->data;

    if (kill_ptr->hold <= CurrentTime)
    {
      if (kill_ptr->status & CONF_KLINE)
        sendto_realops_flags(UMODE_ALL, L_ALL,
                             "KLINE: %s@%s expired",
                             (kill_ptr->user) ? kill_ptr->user : "*",
                             (kill_ptr->host) ? kill_ptr->host : "*");
      else
        sendto_realops_flags(UMODE_ALL, L_ALL,
                             "DLINE: %s expired",
                             (kill_ptr->host) ? kill_ptr->host : "*");
      delete_one_address_conf(kill_ptr->host, kill_ptr);
      dlinkDelete(kill_node, tklist);
      free_dlink_node(kill_node);
    }
  }
}

/* inputs        - pointer to client_p or NULL
 * output        - pointer to static string showing oper privs
 * side effects  - return as string, the oper privs as derived from port
 */
static const struct oper_privs
{
  const unsigned int oprivs;
  const unsigned int hidden;
  const unsigned char c;
} flag_list[] = {
  { OPER_FLAG_ADMIN,       0, 'A' },
  { OPER_FLAG_DIE,         0, 'D' },
  { OPER_FLAG_REHASH,      0, 'H' },
  { OPER_FLAG_K,           0, 'K' },
  { OPER_FLAG_DLINE,       0, 'L' },
  { OPER_FLAG_N,           0, 'N' },
  { OPER_FLAG_GLOBAL_KILL, 0, 'O' },
  { OPER_FLAG_REMOTE,      0, 'R' },
  { OPER_FLAG_X,           0, 'X' },
  { 0, 0, '\0' }
};

char *
oper_privs_as_string(const unsigned int port)
{
  static char privs_out[14];
  char *privs_ptr = privs_out;
  unsigned int i;

  for (i = 0; flag_list[i].oprivs; i++)
  {
    if ((port & flag_list[i].oprivs) &&
        (port & flag_list[i].hidden) == 0)
      *privs_ptr++ = flag_list[i].c;
    else
      *privs_ptr++ = ToLowerTab[flag_list[i].c];
  }
  *privs_ptr = '\0';

  return(privs_out);
}

/* Input: A client to find the active oper{} name for.
 * Output: The nick!user@host{oper} of the oper.
 *         "oper" is server name for remote opers
 */
char *
get_oper_name(const struct Client *client_p)
{
  dlink_node *cnode;
  struct ConfItem *conf;
  struct AccessItem *aconf;

  /* +5 for !,@,{,} and null */
  static char buffer[NICKLEN+USERLEN+HOSTLEN+HOSTLEN+5];

  if (MyConnect(client_p))
  {
    DLINK_FOREACH(cnode, client_p->localClient->confs.head)
    {
      conf = cnode->data;
      aconf = (struct AccessItem *)map_to_conf(conf);

      if (IsConfOperator(aconf))
      {
        ircsprintf(buffer, "%s!%s@%s{%s}", client_p->name,
                   client_p->username, client_p->host,
		   conf->name);
        return(buffer);
      }
    }
  }
  ircsprintf(buffer, "%s!%s@%s{%s}", client_p->name,
             client_p->username, client_p->host, client_p->servptr->name);
  return(buffer);
}

/* inputs        - struct ConfItem
 * output         - host
 *                - reason
 *                - user
 *                - port
 * Examine the struct ConfItem *conf, setting the values
 * of host, pass, user to values either
 * in aconf, or "<NULL>" port is set to aconf->port in all cases.
 */
void
get_printable_conf(struct ConfItem *conf, char **host,
                   char **reason, char **user, int *port, char **classname, char **oreason)
{
  struct AccessItem *aconf;
  static char null[] = "<NULL>";
  static char zero[] = "default";

  aconf = (struct AccessItem *)map_to_conf(conf);
  *host = EmptyString(aconf->host) ? null : aconf->host;
  *reason = EmptyString(aconf->reason) ? "No Reason" : aconf->reason;
  *oreason = EmptyString(aconf->oper_reason) ? "" : aconf->oper_reason;
  *user = EmptyString(aconf->user) ? null : aconf->user;
  *classname = aconf->class_ptr == NULL ? zero : aconf->class_ptr->name;
  *port = (int)aconf->port;
}

/* inputs       - cold start YES or NO
 * side effects - read all conf files needed, ircd.conf kline.conf etc.
 */
void
read_conf_files(int cold)
{
  const char *filename;

  conf_fbfile_in = NULL;
  filename = get_conf_name(CONF_TYPE);
  strlcpy(conffilebuf, filename, sizeof(conffilebuf));

  if ((conf_fbfile_in = fbopen(filename, "r")) == NULL)
  {
    if (cold)
    {
      ilog(CRIT, "Failed in reading configuration file %s", filename);
      exit(-1);
    }
    else
    {
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Can't open file '%s' - aborting rehash!",
                           filename );
      return;
    }
  }

  if (!cold)
    clear_out_old_conf();

  read_conf(conf_fbfile_in);
  fbclose(conf_fbfile_in);

  parse_conf_file(KLINE_TYPE, cold);
  parse_conf_file(DLINE_TYPE, cold);
  parse_conf_file(XLINE_TYPE, cold);
  parse_conf_file(NJUPE_TYPE, cold);
  parse_conf_file(CJUPE_TYPE, cold);
}

/* parse_conf_file()
 * inputs       - type of conf file to parse
 * side effects - conf file for givenconf type is opened and read then parsed
 */
static void
parse_conf_file(int type, int cold)
{
  const char *filename;
  FBFILE *file;
  filename = get_conf_name(type);

  if ((file = fbopen(filename, "r")) == NULL)
  {
    if (cold)
      ilog(ERROR, "Failed reading file %s", filename);
    else
      sendto_realops_flags(UMODE_ALL, L_ALL, "Can't open %s file ", filename);
  }
  else
  {
    parse_csv_file(file, type);
    fbclose(file);
  }
}

/* side effects - Clear out the old configuration */
static void
clear_out_old_conf(void)
{
  dlink_node *ptr;
  dlink_node *next_ptr;
  struct ConfItem *conf;
  struct AccessItem *aconf;
  struct ClassItem *cltmp;
  struct MatchItem *match_item;
  dlink_list * free_items [] = {
    &server_items, &oconf_items, &hub_items, &leaf_items, &uconf_items,
    &xconf_items, &njupe_items, NULL
  };
  dlink_list ** iterator = free_items; /* C is dumb */

  /* We only need to free anything allocated by yyparse() here.
   * Resetting structs, etc, is taken care of by set_default_conf().
   */
  for (; *iterator != NULL; iterator++)
  {
    DLINK_FOREACH_SAFE(ptr, next_ptr, (*iterator)->head)
    {
      conf = ptr->data;
      if (conf->type == SERVER_TYPE)
      {
        aconf = (struct AccessItem *)map_to_conf(conf);
        if (aconf->clients != 0)
        {
          SetConfIllegal(aconf);
          dlinkDelete(&conf->node, &server_items);
        }
        else
        {
          delete_conf_item(conf);
        }
      }
      else if (conf->type == OPER_TYPE)
      {
        aconf = (struct AccessItem *)map_to_conf(conf);
        if (aconf->clients != 0)
        {
          SetConfIllegal(aconf);
          dlinkDelete(&conf->node, &oconf_items);
        }
        else
        {
          delete_conf_item(conf);
        }
      }
      else if (conf->type == CLIENT_TYPE)
      {
        aconf = (struct AccessItem *)map_to_conf(conf);
        if (aconf->clients != 0)
          SetConfIllegal(aconf);
        else
          delete_conf_item(conf);
      }
      else
      {
        if ((conf->type == LEAF_TYPE) || (conf->type == HUB_TYPE))
        {
          match_item = (struct MatchItem *)map_to_conf(conf);
          if ((match_item->ref_count <= 0))
            delete_conf_item(conf);
          else
          {
            match_item->illegal = 1;
            dlinkDelete(&conf->node, *iterator);
          }
        }
        else
          delete_conf_item(conf);
      }
    }
  }

  /* don't delete the class table, rather mark all entries
   * for deletion. The table is cleaned up by check_class. - avalon
   */
  DLINK_FOREACH(ptr, class_items.head)
  {
    conf = ptr->data;
    cltmp = (struct ClassItem *)map_to_conf(conf);
    if (ptr != class_items.tail)  /* never mark the "default" class */
      MaxTotal(cltmp) = -1;
  }
  clear_out_address_conf();

  /* clean out module paths */
#ifndef STATIC_MODULES
  mod_clear_paths();
#endif

  /* clean out ServerInfo */
  MyFree(ServerInfo.description);
  ServerInfo.description = NULL;
  MyFree(ServerInfo.network_name);
  ServerInfo.network_name = NULL;

  clear_conf_jupe();

  /* clean out AdminInfo */
  MyFree(AdminInfo.name);
  AdminInfo.name = NULL;
  MyFree(AdminInfo.email);
  AdminInfo.email = NULL;
  MyFree(AdminInfo.description);
  AdminInfo.description = NULL;

  /* operator{} and class blocks are freed above
   * clean out listeners
   */
  close_listeners();
}

/* side effects - This function removes I/P conf items */
static void
flush_deleted_I_P(void)
{
  dlink_node *ptr;
  dlink_node *next_ptr;
  struct ConfItem *conf;
  struct AccessItem *aconf;
  dlink_list * free_items [] = {
    &server_items, &oconf_items, &hub_items, &leaf_items, NULL
  };
  dlink_list ** iterator = free_items; /* C is dumb */

  /* flush out deleted I and P lines
   * although still in use.
   */
  for (; *iterator != NULL; iterator++)
  {
    DLINK_FOREACH_SAFE(ptr, next_ptr, (*iterator)->head)
    {
      conf = ptr->data;
      aconf = (struct AccessItem *)map_to_conf(conf);

      if (IsConfIllegal(aconf))
      {
        dlinkDelete(ptr, *iterator);

        if (aconf->clients == 0)
          delete_conf_item(conf);
      }
    }
  }
}

/* inputs       - type of conf file to return name of file for
 * output       - pointer to filename for type of conf
 */
const char *
get_conf_name(ConfType type)
{
  switch (type)
  {
    case CONF_TYPE:
      return(ConfigFileEntry.configfile);
      break;
    case KLINE_TYPE:
      return(ConfigFileEntry.klinefile);
      break;
    case DLINE_TYPE:
      return(ConfigFileEntry.dlinefile);
      break;
    case XLINE_TYPE:
      return(ConfigFileEntry.xlinefile);
      break;
    case CJUPE_TYPE:
      return(ConfigFileEntry.cjupefile);
      break;
    case NJUPE_TYPE:
      return(ConfigFileEntry.njupefile);
      break;
    default:
      return(NULL); /* This should NEVER HAPPEN since we call this function
                       only with the above values, this will cause us to core
                       at some point if this happens so we know where it was */
      break;
  }
}

/* inputs       - pointer to struct AccessItem
 * output       - ping frequency
 */
static int
get_conf_ping(struct ConfItem *conf)
{
  struct ClassItem *aclass;
  struct AccessItem *aconf;

  if (conf != NULL)
  {
    aconf = (struct AccessItem *)map_to_conf(conf);
    if (aconf->class_ptr != NULL)
    {
      aclass = (struct ClassItem *)map_to_conf(aconf->class_ptr);
      return(PingFreq(aclass));
    }
  }
  return(-1);
}

/* inputs       - pointer to client struct
 * output       - pointer to name of class
 */
const char *
get_client_class(struct Client *target_p)
{
  dlink_node *ptr;
  struct ConfItem *conf;
  struct AccessItem *aconf;

  if (target_p != NULL && !IsMe(target_p) &&
      target_p->localClient->confs.head != NULL)
  {
    DLINK_FOREACH(ptr, target_p->localClient->confs.head)
    {
      conf = ptr->data;

      if (conf->type == CLIENT_TYPE || conf->type == SERVER_TYPE ||
          conf->type == OPER_TYPE)
      {
        aconf = (struct AccessItem *) map_to_conf(conf);
        if (aconf->class_ptr != NULL)
          return aconf->class_ptr->name;
      }
    }
  }

 return "default";
}

/* inputs       - pointer to client struct
 * output       - ping frequency
 */
int
get_client_ping(struct Client *target_p)
{
  int ping;
  struct ConfItem *conf;
  dlink_node *nlink;

  if (target_p->localClient->confs.head != NULL)
    DLINK_FOREACH(nlink, target_p->localClient->confs.head)
    {
      conf = nlink->data;

      if ((conf->type == CLIENT_TYPE) || (conf->type == SERVER_TYPE) ||
          (conf->type == OPER_TYPE))
      {
         ping = get_conf_ping(conf);
         if (ping > 0)
           return ping;
      }
    }
  return DEFAULT_PINGFREQUENCY;
}

/* inputs       - pointer to class struct
 * output       - connection frequency
 */
int
get_con_freq(struct ClassItem *clptr)
{
  if (clptr != NULL)
    return(ConFreq(clptr));

  return(DEFAULT_CONNECTFREQUENCY);
}

/* inputs       - string name of class
 * output       - corresponding Class pointer
 */
struct ConfItem *
find_class(const char *classname)
{
  struct ConfItem *conf;

  if ((conf = find_exact_name_conf(CLASS_TYPE, classname, NULL, NULL)) != NULL)
    return(conf);
  return class_default;
}

void
check_class(void)
{
  dlink_node *ptr;
  dlink_node *next_ptr;
  struct ConfItem *conf;
  struct ClassItem *aclass;

  DLINK_FOREACH_SAFE(ptr, next_ptr, class_items.head)
  {
    conf = ptr->data;
    aclass = (struct ClassItem *)map_to_conf(conf);

    if (MaxTotal(aclass) < 0)
    {
      if (CurrUserCount(aclass) > 0)
        dlinkDelete(&conf->node, &class_items);
      else
        delete_conf_item(conf);
    }
  }
}

void
init_class(void)
{
  struct ClassItem *aclass;

  class_default = make_conf_item(CLASS_TYPE);
  aclass = (struct ClassItem *)map_to_conf(class_default);
  DupString(class_default->name, "default");
  ConFreq(aclass)  = DEFAULT_CONNECTFREQUENCY;
  PingFreq(aclass) = DEFAULT_PINGFREQUENCY;
  MaxTotal(aclass) = ConfigFileEntry.maximum_links;
  MaxSendq(aclass) = DEFAULT_SENDQ;
  CurrUserCount(aclass) = 0;
}

/* inputs       - pointer to client
 * output       - sendq for this client as found from its class
 */
unsigned long
get_sendq(struct Client *client_p)
{
  unsigned long sendq = DEFAULT_SENDQ;
  dlink_node *ptr;
  struct ConfItem *conf;
  struct ConfItem *class_conf;
  struct ClassItem *aclass;
  struct AccessItem *aconf;

  if (client_p && !IsMe(client_p) && (client_p->localClient->confs.head))
  {
    DLINK_FOREACH(ptr, client_p->localClient->confs.head)
    {
      conf = ptr->data;
      if ((conf->type == SERVER_TYPE) || (conf->type == OPER_TYPE)
          || (conf->type == CLIENT_TYPE))
      {
        aconf = (struct AccessItem *)map_to_conf(conf);
        if ((class_conf = aconf->class_ptr) == NULL)
          continue;
        aclass = (struct ClassItem *)map_to_conf(class_conf);
        sendq = MaxSendq(aclass);
        return(sendq);
      }
    }
  }
  return(DEFAULT_SENDQ);
}

/* inputs       - pointer to config item
 * side effects - Add a class pointer to a conf
 */
void
conf_add_class_to_conf(struct ConfItem *conf, const char *class_name)
{
  struct AccessItem *aconf;
  struct ClassItem *aclass;

  aconf = (struct AccessItem *)map_to_conf(conf);

  if (class_name == NULL)
  {
    aconf->class_ptr = class_default;
    if (conf->type == CLIENT_TYPE)
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Warning *** Defaulting to default class for %s@%s",
                           aconf->user, aconf->host);
    else
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Warning *** Defaulting to default class for %s",
                           conf->name);
  }
  else
  {
    aconf->class_ptr = find_class(class_name);
  }

  if (aconf->class_ptr == NULL)
  {
    if (conf->type == CLIENT_TYPE)
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Warning *** Defaulting to default class for %s@%s",
                           aconf->user, aconf->host);
    else
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Warning *** Defaulting to default class for %s",
                           conf->name);
    aconf->class_ptr = class_default;
  }
  else
  {
    aclass = (struct ClassItem *)map_to_conf(aconf->class_ptr);
    if (MaxTotal(aclass) < 0)
      aconf->class_ptr = class_default;
  }
}

/* inputs       - pointer to config item
 *              - pointer to link count already on this conf
 * side effects - Add a connect block
 */
int
conf_add_server(struct ConfItem *conf, unsigned int lcount, const char *class_name)
{
  struct AccessItem *aconf;

  aconf = map_to_conf(conf);

  conf_add_class_to_conf(conf, class_name);

  if (lcount > 150 || !aconf->host || !conf->name)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "Bad connect block");
    ilog(WARN, "Bad connect block");
    return(-1);
  }

  split_user_host(aconf->host, &aconf->user, &aconf->host);
  lookup_confhost(conf);

  return(0);
}

/* inputs       - pointer to config item
 * side effects - Add a d/D line
 */
void
conf_add_d_conf(struct AccessItem *aconf)
{
  if (aconf->host == NULL)
    return;

  aconf->user = NULL;

  if (parse_netmask(aconf->host, NULL, NULL) == HM_HOST)
  {
    ilog(WARN, "Invalid D-Line %s ignored", aconf->host);
    free_access_item(aconf);
  }
  else
  {
    MyFree(aconf->user);
    aconf->user = NULL;
    add_conf_by_address(CONF_DLINE, aconf);
  }
}

/* inputs       - message from parser
 * side effects - message to opers and log file entry is made
 */
void
yyerror(const char *msg)
{
  char newlinebuf[BUFSIZE];

  if (ypass != 1)
    return;

  strip_tabs(newlinebuf, (const unsigned char *)linebuf, strlen(linebuf));
  sendto_realops_flags(UMODE_ALL, L_ALL,"\"%s\", line %d: %s: %s",
                       conffilebuf, lineno + 1, msg, newlinebuf);
  ilog(WARN, "\"%s\", line %d: %s: %s",
       conffilebuf, lineno + 1, msg, newlinebuf);
}

int
conf_fbgets(char *lbuf, unsigned int max_size, FBFILE *fb)
{
  char *buff;

  if ((buff = fbgets(lbuf, max_size, fb)) == NULL)
    return(0);

  return(strlen(lbuf));
}

int
conf_yy_fatal_error()
{
  return(0);
}
