/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * dbuf.c: Supports dynamic data buffers.
 * Copyright (C) 2004 MusIRCd Development
 * $Id: dbuf.c,v 1.16 2004/02/08 06:46:05 musirc Exp $
 */

#include "balloc.h"
#include "dbuf.h"
#include "list.h"
#include "memory.h"

static BlockHeap *dbuf_heap;

void
dbuf_init(void)
{
  dbuf_heap = BlockHeapCreate(sizeof(struct dbuf_block), DBUF_HEAP_SIZE);
}

static struct dbuf_block *
dbuf_alloc(struct dbuf_queue *qptr)
{
  struct dbuf_block *block = (struct dbuf_block *) BlockHeapAlloc(dbuf_heap);

  block->size = 0;
  dlinkAddTail(block, make_dlink_node(), &qptr->blocks);
  return block;
}

void
dbuf_put(struct dbuf_queue *qptr, char *data, size_t count)
{
  struct dbuf_block *last;
  size_t amount;

  if (qptr->blocks.tail == NULL)
    dbuf_alloc(qptr);

  do {
    last = qptr->blocks.tail->data;

    amount = DBUF_BLOCK_SIZE - last->size;
    if (!amount)
    {
      last = dbuf_alloc(qptr);
      amount = DBUF_BLOCK_SIZE;
    }
    if (amount > count)
      amount = count;

    memcpy((void *) &last->data[last->size], data, amount);
    count -= amount;
    last->size += amount;
    qptr->total_size += amount;
    data += amount;
  } while (count > 0);
}

void
dbuf_delete(struct dbuf_queue *qptr, size_t count)
{
  dlink_node *ptr;
  struct dbuf_block *first;

  if (count == 0)
    return;
  /* free whole blocks first.. */
  while (1)
  {
    if (!count)
      return;
    ptr = qptr->blocks.head;
    first = ptr->data;
    if (count < first->size)
      break;

    qptr->total_size -= first->size;
    count -= first->size;
    dlinkDelete(ptr, &qptr->blocks);
    free_dlink_node(ptr);
    BlockHeapFree(dbuf_heap, first);
  }

  /* ..then remove data from the beginning of the queue */
  first->size -= count;
  qptr->total_size -= count;
  memmove((void *) &first->data, (void *) &first->data[count], first->size);
}
