/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * event.c: Event functions.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: event.c,v 1.23 2004/02/08 06:46:05 musirc Exp $
 */

#include "ircd.h"
#include "event.h"
#include "client.h"
#include "send.h"
#include "log.h"
#include "numeric.h"

static const char *last_event_ran = NULL;
static struct ev_entry event_table[MAX_EVENTS];
static time_t event_time_min = -1;
static int eventFind(EVH *, void *);

/* Input: Name of event, function to call, arguments to pass, and frequency
 *	  of the event.
 * Side Effects: Adds the event to the event list.
 */
void
eventAdd(const char *name, EVH *func, void *arg, time_t when)
{
  int i;
  
  /* find first inactive index, or use next index */
  for (i = 0; i < MAX_EVENTS; i++)
  {
    if (event_table[i].active == 0)
    {
      event_table[i].func = func;
      event_table[i].name = name;
      event_table[i].arg = arg;
      event_table[i].when = CurrentTime + when;
      event_table[i].frequency = when; 
      event_table[i].active = 1;

      if ((event_table[i].when < event_time_min) || (event_time_min == -1))
	event_time_min = event_table[i].when;

      return;
    }
  }
  ilog(ERROR, "Event table is full! (%d)", i);
}

/* Input: Function handler, argument that was passed.
 * Side Effects: Removes the event from the event list
 */
void
eventDelete(EVH *func, void *arg)
{
  int i = eventFind(func, arg);

  if (i == -1)
    return;

  event_table[i].name = NULL;
  event_table[i].func = NULL;
  event_table[i].arg = NULL;
  event_table[i].active = 0;
}

/* Input: Name of event, function to call, arguments to pass, and frequency
 *	  of the event.
 * Side Effects: Adds the event to the event list within +- 1/3 of the
 *	         specified frequency.
 */
void
eventAddIsh(const char *name, EVH *func, void *arg, time_t delta_ish)
{
  if (delta_ish >= 3.0)
  {
    const time_t two_third = (2 * delta_ish) / 3;
    delta_ish = two_third + ((rand() % 1000) * two_third) / 1000;
  }
  eventAdd(name, func, arg, delta_ish);
}

/* Side Effects: Runs pending events in the event list */
void
eventRun(void)
{
  int i;

  for (i = 0; i < MAX_EVENTS; i++)
  {
    if (event_table[i].active && (event_table[i].when <= CurrentTime))
    {
      last_event_ran = event_table[i].name;
      event_table[i].func(event_table[i].arg);
      event_table[i].when = CurrentTime + event_table[i].frequency;
      event_time_min = -1;
    }
  }
}

/* Output: Specifies the next time eventRun() should be run */
time_t
eventNextTime(void)
{
  int i;

  if (event_time_min == -1)
  {
    for (i = 0; i < MAX_EVENTS; i++)
    {
      if (event_table[i].active && ((event_table[i].when < event_time_min) || (event_time_min == -1)))
        event_time_min = event_table[i].when;
    }
  }

  return(event_time_min);
}

/* Side Effects: Initializes the event system. */
void
eventInit(void)
{
  last_event_ran = NULL;
  memset(event_table, 0, sizeof(event_table));
}

/* Input: Event function and the argument passed to it
 * Output: Index to the slow in the event_table
 */
static int
eventFind(EVH *func, void *arg)
{
  int i;

  for (i = 0; i < MAX_EVENTS; i++)
  {
    if ((event_table[i].func == func) &&
        (event_table[i].arg == arg) &&
         event_table[i].active)
      return(i);
  }
  return(-1);
}

/* Input: Time to set back events by.
 * Side-effects: Sets back all events by "by" seconds.
 */
void
set_back_events(time_t by)
{
  int i;

  for (i = 0; i < MAX_EVENTS; i++)
  {
    if (event_table[i].when > by)
      event_table[i].when -= by;
    else
      event_table[i].when = 0;
  }
}
