/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * fileio.c: Provides a file input-output interface to ircd.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: fileio.c,v 1.22 2004/02/08 06:46:05 musirc Exp $
 */

#include "stdinc.h"
#include "fileio.h"
#include "istring.h"
#include "memory.h"
#include "log.h"
#include "fdlist.h"

/* Wrappers around open() / close() for fileio, since a whole bunch of
 * code that should be using the fbopen() / fbclose() code isn't.
 */
int
file_open(const char *filename, int mode, int fmode)
{
  int fd = open(filename, mode, fmode);

  if (fd == MASTER_MAX)
  {
    close(fd); /* Too many FDs! */
    errno = ENFILE;
    fd = -1;
  }
  else if (fd >= 0)
    fd_open(fd, FD_FILE, filename);

  return(fd);
}

void
file_close(int fd)
{
    fd_table[fd].type = FD_FILECLOSE;
    fd_close(fd);
}

FBFILE *
fbopen(const char *filename, const char *mode)
{
  int fd, openmode = 0, pmode = 0;
  FBFILE *fb = NULL;

  if (filename == NULL || mode == NULL)
  {
     errno = EINVAL;
     return NULL;
  }

  while (*mode)
  {
    switch (*mode)
    {
    case 'r':
      openmode = O_RDONLY;
      break;
    case 'w':
      openmode = O_WRONLY | O_CREAT | O_TRUNC;
      pmode = 0644;
      break;
    case 'a':
      openmode = O_WRONLY | O_CREAT | O_APPEND;
      pmode = 0644;
      break;
    case '+':
      openmode &= ~(O_RDONLY | O_WRONLY);
      openmode |= O_RDWR;
      break;
    default:
      break;
    }
    ++mode;
  }

  if ((fd = file_open(filename, openmode, pmode)) == -1)
    return(fb);

  if (NULL == (fb = fdbopen(fd, NULL)))
    file_close(fd);

  return(fb);
}

FBFILE *
fdbopen(int fd, const char *n)
{
  /* ignore mode, if file descriptor hasn't been opened with the
   * correct mode, the first use will fail
   */
  FBFILE *fb = MyMalloc(sizeof(FBFILE));

  fb->ptr = fb->endp = fb->buf;
  fb->fd = fd;
  fb->flags = 0;
  fb->pbptr = NULL;

  return(fb);
}

int
fbrewind(FBFILE *fb)
{
  if (fb != NULL)
  {
    fb->ptr   = fb->endp = fb->buf;
    fb->flags = 0;
    fb->pbptr = NULL;
  }

  lseek(fb->fd, 0l, SEEK_SET);
  return(0);
}

void
fbclose(FBFILE* fb)
{
  if (fb != NULL)
  {
    file_close(fb->fd);
    MyFree(fb);
  }
  else
    errno = EINVAL;
}

static int
fbfill(FBFILE *fb)
{
  int n;

  if (fb == NULL)
  {
    errno = EINVAL;
    return(-1);
  }

  if (fb->flags)
    return(-1);

  n = read(fb->fd, fb->buf, BUFSIZ);

  if (n > 0)
  {
    fb->ptr  = fb->buf;
    fb->endp = fb->buf + n;
  }
  else if (n < 0)
    fb->flags |= FB_FAIL;
  else
    fb->flags |= FB_EOF;

  return(n);
}

int
fbgetc(FBFILE *fb)
{
  if (fb == NULL)
  {
    errno = EINVAL;
    return(-1);
  }

  if (fb->pbptr != NULL)
  {
    if ((fb->pbptr == (fb->pbuf + BUFSIZ)) ||
        (!*fb->pbptr))
      fb->pbptr = NULL;
  }

  if (fb->ptr < fb->endp || fbfill(fb) > 0)
    return(*fb->ptr++);

  return(EOF);
}

void
fbungetc(char c, FBFILE *fb)
{
  if (fb == NULL)
  {
    errno = EINVAL;
    return;
  }

  if (!fb->pbptr)
  {
    fb->pbptr = fb->pbuf+BUFSIZ;
  }

  if (fb->pbptr != fb->pbuf)
  {
    fb->pbptr--;
    *fb->pbptr = c;
  }
}

char *
fbgets(char *buf, size_t len, FBFILE *fb)
{
  char *p = buf;

  if (fb == NULL || buf == NULL)
  {
    errno = EINVAL;
    return(NULL);
  }

  if(fb->pbptr != NULL)
  {
    strlcpy(buf,fb->pbptr,len);
    fb->pbptr = NULL;
    return(buf);
  }

  if (fb->ptr == fb->endp && fbfill(fb) < 1)
    return(NULL);

  --len;

  while (len--)
  {
    *p = *fb->ptr++;

    if ('\n' == *p)
    {
      ++p;
      break;
    }

    /* deal with CR's */
    else if ('\r' == *p)
    {
      if (fb->ptr < fb->endp || fbfill(fb) > 0)
      {
        if ('\n' == *fb->ptr)
          ++fb->ptr;
      }

      *p++ = '\n';
      break;
    }

    ++p;

    if (fb->ptr == fb->endp && fbfill(fb) < 1)
      break;
  }

  *p = '\0';
  return(buf);
}
 
int
fbputs(const char *str, FBFILE *fb, size_t nbytes)
{
  int n = -1;
  
  if (str == NULL || fb == NULL)
  {
    errno = EINVAL;
    return(-1);
  }

  if (0 == fb->flags)
  {
    n = write(fb->fd, str, nbytes);
    if (n == -1)
      fb->flags |= FB_FAIL;
  }

  return(n);
}
