/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * ircd.c: Starts up and runs the ircd.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: ircd.c,v 1.116.2.2 2004/07/09 04:56:45 musirc Exp $
 */

#include "user.h"
#include "ircd.h"
#include "channel.h"
#include "channel_mode.h"
#include "client.h"
#include "event.h"
#include "hash.h"
#include "istring.h"
#include "getopts.h"
#include "sprintf.h"
#include "list.h"
#include "hostmask.h"
#include "auth.h"
#include "restart.h"
#include "bsd.h"
#include "config.h"
#include "log.h"
#include "server.h"
#include "send.h"
#include "whowas.h"
#include "modules.h"

/* Try and find the correct name to use with getrlimit() for setting the max.
 * number of files allowed to be open by this process.

 * /quote set variables */
struct SetOptions GlobalSetOptions;
struct config_file_entry ConfigFileEntry;
struct server_info ServerInfo;
struct admin_info AdminInfo = {NULL, NULL, NULL};
struct Counter Count = {0, 0, 0, 0, 0, 0, 0};
struct ServerState_t server_state = { 0 };
struct logging_entry ConfigLoggingEntry = { 1, {0}, {0}, {0} };
struct timeval SystemTime;
int ServerRunning;           /* GLOBAL - server execution state */
struct Client me;               /* That's me */
struct LocalUser meLocalUser;   /* That's also part of me */
static unsigned long initialVMTop = 0;   /* top of virtual memory at init */
const char *logFileName = LPATH, *pidFileName = PPATH;
char **myargv;
int dorehash = 0, doremotd = 0, default_server_capabs = 0, splitmode, 
    splitchecking, split_users, rehashed_klines = 0, rehashed_xlines = 0;
time_t nextconnect = 1;        /* time for next try_connections call */
unsigned int split_servers;

/* get the operating systems notion of the resident set size */
static unsigned long
get_vm_top(void)
{
  /* NOTE: sbrk is not part of the ANSI C library or the POSIX.1 standard
   * however it seems that everyone defines it. Calling sbrk with a 0
   * argument will return a pointer to the top of the process virtual
   * memory without changing the process size, so this call should be
   * reasonably safe (sbrk returns the new value for the top of memory).
   * This code relies on the notion that the address returned will be an
   * offset from 0 (NULL), so the result of sbrk is cast to a size_t and
   * returned. We really shouldn't be using it here but...
   */
  void *vptr = sbrk(0);
  return((unsigned long)vptr);
}

/* get the operating systems notion of the resident set size */
unsigned long
get_maxrss(void)
{
 return(get_vm_top() - initialVMTop);
}

/* print startup information */
static void
print_startup(int pid)
{
  printf("%s running in %s from %s pid: %d\n", ircd_version,
	 !server_state.foreground ? "background" : "foreground",
	 ConfigFileEntry.dpath, pid);
}

/* side effects	- if boot_daemon flag is not set, don't daemonize */
static void 
init_sys(void)
{
#if defined(RLIMIT_FD_MAX) && defined(HAVE_SYS_RLIMIT_H)
  struct rlimit limit;

  if (!getrlimit(RLIMIT_FD_MAX, &limit))
  {
    if (limit.rlim_max < HARD_FDLIMIT)
    {
      fprintf(stderr, "ircd fd table too big\n");
      fprintf(stderr, "Hard Limit: %ld IRC max: %d\n",
              (long)limit.rlim_max, HARD_FDLIMIT);
      fprintf(stderr, "Fix HARD_FDLIMIT\n");
      exit(-1);
    }

    limit.rlim_cur = limit.rlim_max; /* make soft limit the max */

    if (setrlimit(RLIMIT_FD_MAX, &limit) == -1)
    {
      fprintf(stderr, "error setting max fd's to %ld\n",
              (long)limit.rlim_cur);
      exit(EXIT_FAILURE);
    }
  }
#endif
}

static void
make_daemon(void)
{
  int pid;

  if ((pid = fork()) < 0)
  {
    perror("fork");
    exit(EXIT_FAILURE);
  }
  else if (pid > 0)
  {
    print_startup(pid);
    exit(EXIT_SUCCESS);
  }

  setsid();
}

static int printVersion = 0;

struct lgetopt myopts[] = {
  {"dlinefile",  &ConfigFileEntry.dlinefile, 
   STRING, "File to use for dline.conf"},
  {"configfile", &ConfigFileEntry.configfile, 
   STRING, "File to use for ircd.conf"},
  {"klinefile",  &ConfigFileEntry.klinefile, 
   STRING, "File to use for kline.conf"},
  {"xlinefile",  &ConfigFileEntry.xlinefile,
  STRING, "File to use for xline.conf"},
  {"logfile",    &logFileName, 
   STRING, "File to use for ircd.log"},
  {"pidfile",    &pidFileName,
   STRING, "File to use for process ID"},
  {"foreground", &server_state.foreground, 
   YESNO, "Run in foreground (don't detach)"},
  {"version",    &printVersion, 
   YESNO, "Print version and exit"},
  {"help", NULL, USAGE, "Print this text"},
  {NULL, NULL, STRING, NULL},
};

void
set_time(void)
{
  static char to_send[200];
  struct timeval newtime;
  newtime.tv_sec = newtime.tv_usec = 0;

  if (gettimeofday(&newtime, NULL) == -1)
  {
    ilog(ERROR, "Clock Failure (%d)", errno);
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "Clock Failure (%d), TS can be corrupted", errno);
    restart("Clock Failure");
  }

  if (newtime.tv_sec < CurrentTime)
  {
    ircsprintf(to_send, "TS change: %lu to %lu",
               (unsigned long)CurrentTime, (unsigned long)newtime.tv_sec);
    report_error(L_ALL, to_send, me.name, 0);
    set_back_events(CurrentTime - newtime.tv_sec);
  }
  SystemTime.tv_sec = newtime.tv_sec;
  SystemTime.tv_usec = newtime.tv_usec;
}

static void
io_loop(void)
{
  while (ServerRunning)
  {
    if (rehashed_klines)
    {
      check_conf_klines();
      rehashed_klines = 0;
    }
    else if (rehashed_xlines)
    {
      check_xlines();
      rehashed_xlines = 0;
    }
    /* Run pending events, then get the number of seconds to the next event */
    while (eventNextTime() <= CurrentTime)
      eventRun();
  
    comm_select(500);
    exit_aborted_clients();
    free_exited_clients();
    send_queued_all();
    /* Check to see whether we have to rehash the configuration .. */
    if (dorehash)
    {
      rehash(1);
      dorehash = 0;
    }
    if (doremotd)
    {
      read_message_file(&ConfigFileEntry.motd);
      sendto_realops_flags(UMODE_ALL, L_ALL,
                          "Got signal SIGUSR1, reloading ircd motd file");
      doremotd = 0;
    }
  }
}

/* side effects - This sets all global set options needed */
static void
initialize_global_set_options(void)
{
  memset(&GlobalSetOptions, 0, sizeof(GlobalSetOptions));

  GlobalSetOptions.maxclients = MAXCONN;
  GlobalSetOptions.autoconn = 1;

  if (ConfigFileEntry.default_floodcount)
    GlobalSetOptions.floodcount = ConfigFileEntry.default_floodcount;
  else
    GlobalSetOptions.floodcount = 10;

  split_servers = ConfigChannel.default_split_server_count;
  split_users = ConfigChannel.default_split_user_count;

  if (split_users && split_servers && (ConfigChannel.no_create_on_split ||
     ConfigChannel.no_join_on_split))
  {
    splitmode = 1;
    splitchecking = 1;
  }
  GlobalSetOptions.ident_timeout = IDENT_TIMEOUT;
}

/* side effects - Set up all message files needed, motd etc. */
static void
initialize_message_files(void)
{
  init_message_file(USER_MOTD, MPATH, &ConfigFileEntry.motd);
  init_message_file(OPER_MOTD, OPATH, &ConfigFileEntry.opermotd);
  init_message_file(USER_LINKS, LIPATH, &ConfigFileEntry.linksfile);
  read_message_file(&ConfigFileEntry.motd);
  read_message_file(&ConfigFileEntry.opermotd);
  read_message_file(&ConfigFileEntry.linksfile);
}

static void
initialize_server_capabs(void)
{
  add_capability("QS", CAP_QS, 1);
}

/* inputs       - filename+path of pid file
 * side effects - write the pid of the ircd to filename
 */
static void
write_pidfile(const char *filename)
{
  FBFILE *fb;

  if ((fb = fbopen(filename, "w")))
  {
    char buff[32];
    unsigned int pid = (unsigned int)getpid();
    size_t nbytes = ircsprintf(buff, "%u\n", pid);

    if ((fbputs(buff, fb, nbytes) == -1))
      ilog(ERROR, "Error writing %u to pid file %s (%s)",
           pid, filename, strerror(errno));

    fbclose(fb);
    return;
  }
  else
  {
    ilog(ERROR, "Error opening pid file %s", filename);
  }
}

/* inputs       - filename+path of pid file
 * side effects - reads pid from pidfile and checks if ircd is in process
 *                list. if it is, gracefully exits
 */
static void
check_pidfile(const char *filename)
{
  FBFILE *fb;
  char buff[32];
  pid_t pidfromfile;

  /* Don't do logging here, since we don't have log() initialised */
  if ((fb = fbopen(filename, "r")))
  {
    if (fbgets(buff, 20, fb) != NULL)
    {
      pidfromfile = atoi(buff);

      if (!kill(pidfromfile, 0))
      {
        printf("ircd: daemon is already running\n");
        exit(-1);
      }
    }
    fbclose(fb);
  }
}

/* side effects - setups corefile to system limits. */
static void
setup_corefile(void)
{
#ifdef HAVE_SYS_RESOURCE_H
  struct rlimit rlim; /* resource limits */

  /* Set corefilesize to maximum */
  if (!getrlimit(RLIMIT_CORE, &rlim))
  {
    rlim.rlim_cur = rlim.rlim_max;
    setrlimit(RLIMIT_CORE, &rlim);
  }
#endif
}

void write_stats(void)
{
  int fd;
  char buff[20];

  if ((fd = open(MXPATH, O_CREAT|O_WRONLY, 0600))>=0)
  {
    ircsprintf(buff,"%d\n%d\n", Count.max_loc, Count.max_tot);
    if (write(fd, buff, strlen(buff)) == -1)
      ilog(ERROR,"Error writing to max clients file %s", MXPATH);
    close(fd);
    return;
  }
  else
    ilog(ERROR, "Error opening max clients file %s", MXPATH);
}

int
main(int argc, char *argv[])
{
  FBFILE *file;
  char mline[20];
  int m1;

  /* Check to see if the user is running us as root, which is a nono */
  if(geteuid() == 0)
  {
    fprintf(stderr, "Don't run ircd as root!!!\n");
    return(-1);
  }

  /* save server boot time right away, so getrusage works correctly */
  set_time();

  /* Setup corefile size immediately after boot -kre */
  setup_corefile();

  /* set initialVMTop before we allocate any memory */
  initialVMTop = get_vm_top();

  ServerRunning = 0;

  /* It ain't random, but it ought to be a little harder to guess */
  srand(SystemTime.tv_sec ^ (SystemTime.tv_usec | (getpid() << 20)));
  memset(&me, 0, sizeof(me));
  memset(&meLocalUser, 0, sizeof(meLocalUser));
  me.localClient = &meLocalUser;

  dlinkAdd(&me, &me.node, &global_client_list); /* Pointer to beginning
                                                   of Client list */
  memset(&ServerInfo, 0, sizeof(ServerInfo));

  ConfigFileEntry.dpath = DPATH;
  ConfigFileEntry.configfile = CPATH;   /* Server configuration file */
  ConfigFileEntry.klinefile = KPATH;    /* Server kline file */
  ConfigFileEntry.xlinefile  = XPATH;  /* Server xline file */
  ConfigFileEntry.dlinefile = DLPATH;   /* dline file */
  ConfigFileEntry.cjupefile  = CJUPEPATH; /* channel jupe file */
  ConfigFileEntry.njupefile  = NJUPEPATH; /* nick jupe file */
  myargv = argv;
  umask(077);                /* better safe than sorry --SRB */

  parseargs(&argc, &argv, myopts);

  if (printVersion)
  {
    printf("ircd: version %s\n", ircd_version);
    exit(EXIT_SUCCESS);
  }

  if (chdir(ConfigFileEntry.dpath))
  {
    perror("chdir");
    exit(EXIT_FAILURE);
  }

  if (!server_state.foreground)
    make_daemon();
  else
    print_startup(getpid());

  setup_signals();
  /* We need this to initialise the fd array before anything else */
  fdlist_init();
  if (!server_state.foreground)
    close_all_connections(); /* this needs to be before init_netio()! */
  else
    check_can_use_v6(); /* Done in close_all_connections normally */

  init_log(logFileName);
  init_netio();         /* This needs to be setup early ! -- adrian */
  /* Check if there is pidfile and daemon already running */
  check_pidfile(pidFileName);
  eventInit();
  init_sys();
  initBlockHeap();
  init_dlink_nodes();
  init_slink_nodes();
  initialize_message_files();
  dbuf_init();
  init_hash();
  init_ip_hash_table();     /* client host ip hash table */
  init_host_hash();             /* Host-hashtable. */
  clear_tree_parse();
  init_client();
  init_user();
  init_channels();
  init_class();
  init_whowas();
  read_conf_files(1);  /* cold start init conf files */
  init_auth();                  /* Initialise the auth code */
  init_resolver();      /* Needs to be setup before the io loop */
  initialize_server_capabs();   /* Set up default_server_capabs */
  initialize_global_set_options();

  if (ServerInfo.name == NULL)
  {
    fprintf(stderr, "ERROR: No server name specified in serverinfo block.\n");
    ilog(CRIT, "No server name specified in serverinfo block.");
    exit(EXIT_FAILURE);
  }
  strlcpy(me.name, ServerInfo.name, sizeof(me.name));

  /* serverinfo{} description must exist.  If not, error out.*/
  if (ServerInfo.description == NULL)
  {
    fprintf(stderr, "ERROR: No server description specified in serverinfo block.\n");
    ilog(CRIT, "ERROR: No server description specified in serverinfo block.");
    exit(EXIT_FAILURE);
  }
  strlcpy(me.info, ServerInfo.description, sizeof(me.info));

  if (!me.info[0])
    strlcpy(me.info, "No description", sizeof(me.info));

  me.servptr = me.from = &me;
  SetMe(&me);
  make_server(&me);
  me.lasttime = me.since = me.firsttime = CurrentTime;
  hash_add_client(&me);
  /* add ourselves to global_serv_list */
  dlinkAdd(&me, make_dlink_node(), &global_serv_list);
  check_class();

#ifndef STATIC_MODULES
  if (chdir(MODPATH))
  {
    ilog(CRIT, "Could not load core modules. Terminating!");
    exit(EXIT_FAILURE);
  }
  mod_set_base();
  load_all_modules(1);
  load_core_modules(1);
  chdir(ConfigFileEntry.dpath);
#else
  load_all_modules(1);
#endif

  write_pidfile(pidFileName);

  if ((file = fbopen(MXPATH, "r")))
  {
    if (fbgets(mline, sizeof(mline), file))
    {
      m1 = atoi(mline);
      if (fbgets(mline, sizeof(mline), file))
      {
        Count.max_loc = m1;
        Count.max_tot = atoi(mline);
      }
    }
    fbclose(file);
  }
  ilog(NOTICE, "Server Ready");
  
  eventAddIsh("cleanup_tklines", cleanup_tklines, NULL, CLEANUP_TKLINES_TIME);
  eventAddIsh("try_connections", try_connections, NULL, STARTUP_CONNECTIONS_TIME);
  eventAddIsh("comm_checktimeouts", comm_checktimeouts, NULL, 1);

  if (ConfigServerHide.links_delay > 0)
    eventAddIsh("write_links_file", write_links_file, NULL, ConfigServerHide.links_delay);
  else
    ConfigServerHide.links_disabled = 1;

  if (splitmode)
    eventAddIsh("check_splitmode", check_splitmode, NULL, 60);

  ServerRunning = 1;
  io_loop();
  return(0);
}
