/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * lexer.l: Scans the ircd configuration file for tokens.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: lexer.l,v 1.39.2.8 2004/07/18 21:43:54 musirc Exp $
 */

%option case-insensitive
%option noyywrap
%option nounput

%{
#define WE_ARE_MEMORY_C
#include "stdinc.h"
#include "istring.h"
#include "config.h"
#include "y.tab.h"
#include "log.h"

#undef YY_INPUT
#define YY_INPUT(buf, result, max_size) \
  if (!(result = conf_fbgets(buf, max_size, conf_fbfile_in))) \
    conf_yy_fatal_error();
#define MAX_INCLUDE_DEPTH 10

int lineno = 1, include_stack_ptr = 0, lineno_stack[MAX_INCLUDE_DEPTH];
char linebuf[BUFSIZE], conffilebuf[BUFSIZE], conffile_stack[MAX_INCLUDE_DEPTH][BUFSIZE];
YY_BUFFER_STATE include_stack[MAX_INCLUDE_DEPTH];
FBFILE* inc_fbfile_in[MAX_INCLUDE_DEPTH];
void ccomment(void);
void hashcomment(void);
void cinclude(void);
int ieof(void);
%}

ws        [ \t]*
digit     [0-9]
comment   #.*
qstring   \"[^\"\n]*[\"\n]
cppcomment "//"[^\n]*"\n"
include   \.include{ws}(\<.*\>|\".*\")

%%
{include}       { cinclude(); }
"/*"	        { ccomment(); }

\n.*	        { strcpy(linebuf, yytext+1); lineno++; yyless(1); }

{ws}            ;
{comment}       { hashcomment(); }

{digit}+        { yylval.number = atoi(yytext); return NUMBER; }

{qstring}	{ if(yytext[yyleng-2] == '\\')
		  {
		    yyless(yyleng-1); /* return last quote */
		    yymore();		/* append next string */
		  }
		  else
		  {
		    yylval.string = yytext+1;
		    if(yylval.string[yyleng-2] != '"')
		      ilog(ERROR, "Unterminated character string");
		    else
		    {
		      int i,j;

		      yylval.string[yyleng-2] = '\0'; /* remove close quote*/

		      for (j=i=0 ;yylval.string[i] != '\0'; i++,j++)
		      {
		        if (yylval.string[i] != '\\')
			{
			  yylval.string[j] = yylval.string[i];
			}
			else
			{
			  i++;
			  if (yylval.string[i] == '\0')
			  {
			    ilog(ERROR, "Unterminated character string");
			    break;
			  }
			  yylval.string[j] = yylval.string[i];
			}
		      }
		      yylval.string[j] = '\0';
		      return QSTRING;
		    }
		  }
		}

{cppcomment}	{ lineno++; }
accept_password { return ACCEPT_PASSWORD; }
admin		{ return ADMIN; }
aftype		{ return AFTYPE; }
all		{ return T_ALL; }
allow           { return T_ALLOW; }
auth		{ return AUTH; }
autoconn	{ return AUTOCONN; }
autodline_drones { return AUTODLINE_DRONES; }
can_flood	{ return CAN_FLOOD; }
ssignore_wait	{ return SSIGNORE_WAIT; }
channel		{ return CHANNEL; }
cjupe		{ return CJUPE; }
cjupe_dline	{ return CJUPE_DLINE; }
cjupe_dline_time { return CJUPE_DLINE_TIME; }
class		{ return CLASS; }
client_flood	{ return T_CLIENT_FLOOD; }
connect		{ return CONNECT; }
connectfreq     { return CONNECTFREQ; }
default_floodcount  { return DEFAULT_FLOODCOUNT; }
default_split_server_count { return DEFAULT_SPLIT_SERVER_COUNT; }
default_split_user_count   { return DEFAULT_SPLIT_USER_COUNT; }
deny		{ return DENY; }
description	{ return DESCRIPTION; }
drone_detect	{ return DRONE_DETECT; }
die		{ return DIE; }
disable_auth   { return DISABLE_AUTH; }
disable_hidden	{ return DISABLE_HIDDEN; }
disable_local_channels { return DISABLE_LOCAL_CHANNELS; }
disable_remote_commands	{ return DISABLE_REMOTE_COMMANDS; }
dots_in_ident   { return DOTS_IN_IDENT; }
dline		{ return DLINE;	}
drone_dline_time { return DRONE_DLINE_TIME; }
email		{ return EMAIL; }
encrypted       { return ENCRYPTED; }
exceed_limit    { return EXCEED_LIMIT; }
exempt		{ return EXEMPT; }
fakename	{ return FAKENAME; }
flatten_links	{ return FLATTEN_LINKS; }
ffailed_operlog	{ return FFAILED_OPERLOG; }
foperlog   	{ return FOPERLOG; }
fuserlog   	{ return FUSERLOG; }
general		{ return GENERAL; }
global_kill	{ return GLOBAL_KILL; }
have_ident	{ return HAVE_IDENT; }
havent_read_conf { return HAVENT_READ_CONF; }
hidden		{ return HIDDEN; }
hide_server_ips { return HIDE_SERVER_IPS; }
hide_servers	{ return HIDE_SERVERS; }
hide_spoof_ips  { return HIDE_SPOOF_IPS; }
host		{ return HOST; }
hub		{ return HUB; }
hub_mask	{ return HUB_MASK; }
idletime        { return IDLETIME; }
ip		{ return IP; }
ipv4            { return T_IPV4; }
ipv6            { return T_IPV6; }
jupe		{ return JUPE;; }
kill		{ return KILL; }
kill_chase_time_limit { return KILL_CHASE_TIME_LIMIT; }
kline		{ return KLINE; }
kline_exempt	{ return KLINE_EXEMPT; }
crit		{ return L_CRIT; }
error		{ return L_ERROR; }
info		{ return L_INFO; }
notice 		{ return L_NOTICE; }
trace		{ return L_TRACE; }
warn		{ return L_WARN; }
leaf_mask	{ return LEAF_MASK; }
listen		{ return LISTEN; }
log_level	{ return LOG_LEVEL; }
logging         { return LOGGING; }
logpath         { return T_LOGPATH; }
masked		{ return TMASKED; }
max_clients     { return T_MAX_CLIENTS; }
max_ident	{ return MAX_IDENT; }
max_local	{ return MAX_LOCAL; }
max_global	{ return MAX_GLOBAL; }
max_number	{ return MAX_NUMBER; }
maximum_links   { return MAXIMUM_LINKS; }
message_locale	{ return MESSAGE_LOCALE; }
min_nonwildcard { return MIN_NONWILDCARD; }
min_nonwildcard_simple  { return MIN_NONWILDCARD_SIMPLE; }
name		{ return NAME; }
need_password	{ return NEED_PASSWORD; }
network_name    { return NETWORK_NAME; }
nick		{ return NICK; }
nick_changes	{ return NICK_CHANGES; }
njupe		{ return NJUPE; }
no		{ yylval.number = 0; return TBOOL; }
no_create_on_split { return NO_CREATE_ON_SPLIT; }
no_join_on_split   { return NO_JOIN_ON_SPLIT; }
no_oper_flood   { return NO_OPER_FLOOD; }
no_tilde	{ return NO_TILDE; }
number_per_ip	{ return NUMBER_PER_IP; }
oper		{ return OPERATOR; }
oper_log        { return OPER_LOG; }
oper_pass_jupes	{ return OPER_PASS_JUPES; }
operator	{ return OPERATOR; }
password	{ return PASSWORD; }
ping_time	{ return PING_TIME; }
port		{ return PORT; }
quiet_on_ban    { return QUIET_ON_BAN; }
reason		{ return REASON; }
rehash		{ return REHASH; }
remote		{ return REMOTE; }
restricted      { return RESTRICTED; }
send_password	{ return SEND_PASSWORD; }
sendq		{ return SENDQ; }
serverhide	{ return SERVERHIDE; }
serverinfo	{ return SERVERINFO; }
shared          { return SHARED; }
spoof		{ return SPOOF; }
spoof_notice	{ return SPOOF_NOTICE; }
type		{ return TYPE; }
true_no_oper_flood { return TRUE_NO_OPER_FLOOD; }
use_logging   { return USE_LOGGING; }
use_services  { return USE_SERVICES; }
throttle_time { return THROTTLE_TIME; }
user		{ return USER;} 
vhost	{ return VHOST; }
vhost6	{ return VHOST6; }
xline	        { return XLINE; }
yes		{ yylval.number = 1; return TBOOL; }

max_accept              { return MAX_ACCEPT; }
max_nick_changes        { return MAX_NICK_CHANGES; }
max_chans_per_user      { return MAX_CHANS_PER_USER; }
max_nick_time           { return MAX_NICK_TIME; }
anti_nick_flood         { return ANTI_NICK_FLOOD; }
anti_spam_exit_message_time { return ANTI_SPAM_EXIT_MESSAGE_TIME; }
ts_max_delta            { return TS_MAX_DELTA; }
ts_warn_delta           { return TS_WARN_DELTA; }
links_delay		{ return LINKS_DELAY; }
hide_ban_reason		{ return HIDE_BAN_REASON; }
stats_o_oper_only       { return STATS_O_OPER_ONLY; }
stats_k_oper_only	{ return STATS_K_OPER_ONLY; }
stats_i_oper_only	{ return STATS_I_OPER_ONLY; }
stats_P_oper_only	{ return STATS_P_OPER_ONLY; }
pace_wait               { return PACE_WAIT; }
pace_wait_simple	{ return PACE_WAIT_SIMPLE; }
knock_delay             { return KNOCK_DELAY; }
knock_delay_channel	{ return KNOCK_DELAY_CHANNEL; }
max_bans                { return MAX_BANS; }
modules                 { return MODULES; }
module                  { return MODULE; }
path                    { return PATH; }
max_targets             { return MAX_TARGETS; }
oper_umodes             { return OPER_UMODES; }
crypt_oper_password   	{ return CRYPT_OPER_PASSWORD; }
flood                   { return T_FLOOD; }
cconn                   { return T_CCONN; }
nchange                 { return T_NCHANGE; }
unauth                  { return T_UNAUTH; }
spy                     { return T_SPY; }
external                { return T_EXTERNAL; }
servnotice              { return T_SERVNOTICE; }
invisible               { return T_INVISIBLE; }
jupes			{ return T_JUPES; }
wallop                  { return T_WALLOP; }
ssignore                { return T_SSIGNORE; }
drone                   { return T_DRONE; }
locops                  { return T_LOCOPS; }

weeks			{ return WEEKS; }
week			{ return WEEKS; }
days			{ return DAYS; }
day			{ return DAYS; }
hours			{ return HOURS; }
hour			{ return HOURS; }
minutes			{ return MINUTES; }
minute			{ return MINUTES; }
seconds			{ return SECONDS; }
second			{ return SECONDS; }

bytes			{ return BYTES; }
byte			{ return BYTES; }
kilobytes		{ return KBYTES; }
kilobyte		{ return KBYTES; }
kbytes			{ return KBYTES; }
kbyte			{ return KBYTES; }
kb			{ return KBYTES; }
megabytes		{ return MBYTES; }
megabyte		{ return MBYTES; }
mbytes			{ return MBYTES; }
mbyte			{ return MBYTES; }
mb			{ return MBYTES; }
gigabytes		{ return GBYTES; }
gigabyte		{ return GBYTES; }
gbytes			{ return GBYTES; }
gbyte			{ return GBYTES; }
gb			{ return GBYTES; }
terabytes		{ return TBYTES; }
terabyte		{ return TBYTES; }
tbytes			{ return TBYTES; }
tbyte			{ return TBYTES; }
tb			{ return TBYTES; }
\.\.                    { return TWODOTS; }

.		        { return yytext[0]; }
<<EOF>>     { if (ieof()) yyterminate(); }

%%

/* C-comment ignoring routine -kre*/
void ccomment(void)
{
  int c;

  while (1)
  {
    while ((c = input()) != '*' && c != EOF)
      if (c == '\n')
	++lineno;
    if (c == '*')
    {
      while ((c = input()) == '*');
        if (c == '/')
	  break;
    }
    if (c == EOF)
    {
      conf_yy_fatal_error();
      break;
    }
  }
}

/* #-comment style, look for #include */
#define INCLUDE "#include"
void hashcomment(void) 
{
  if (strlen(yytext) < strlen(INCLUDE))
    return;

  if (!strncasecmp(yytext, INCLUDE, strlen(INCLUDE)))
    yyerror("You probably meant '.include', skipping");
}

/* C-style .includes. This function will properly swap input conf buffers,
 * and lineno -kre */
void cinclude(void)
{
  char *c;
  if ((c = strchr(yytext, '<')) == NULL)
    *strchr(c = strchr(yytext, '"') + 1, '"') = 0;
  else
    *strchr(++c, '>') = 0;

  /* do stacking and co. */ 
  if (include_stack_ptr >= MAX_INCLUDE_DEPTH)
    ilog(ERROR, "Includes nested too deep in %s", c);
  else
  {
    FBFILE *tmp_fbfile_in;
    char filenamebuf[BUFSIZE];
    
    snprintf(filenamebuf, sizeof(filenamebuf), "%s/%s", ETCPATH, c);
    tmp_fbfile_in = fbopen(filenamebuf, "r");
    
    if (tmp_fbfile_in == NULL)
    {
      ilog(ERROR, "Cannot include %s", filenamebuf);
      return;
    }
    lineno_stack[include_stack_ptr] = lineno;
    lineno = 1;
    inc_fbfile_in[include_stack_ptr] = conf_fbfile_in;
    strlcpy(conffile_stack[include_stack_ptr], conffilebuf, BUFSIZE);
    include_stack[include_stack_ptr++] = YY_CURRENT_BUFFER;
    conf_fbfile_in = tmp_fbfile_in;
    snprintf(conffilebuf, sizeof(conffilebuf), "%s", c);
    yy_switch_to_buffer(yy_create_buffer(yyin, YY_BUF_SIZE));
  }
}

/* This is function that will be called on EOF in conf file. It will
 * apropriately close conf if it not main conf and swap input buffers -kre
 */
int ieof(void)
{
  if (include_stack_ptr)
    fbclose(conf_fbfile_in);
  if (--include_stack_ptr < 0)
  {
    /* We will now exit the lexer - restore init values if we get /rehash
     * later and reenter lexer -kre */
    include_stack_ptr = 0;
    lineno = 1;
    return 1;
  }
  /* switch buffer */
  yy_delete_buffer(YY_CURRENT_BUFFER);
  lineno = lineno_stack[include_stack_ptr];
  conf_fbfile_in = inc_fbfile_in[include_stack_ptr];
  strlcpy(conffilebuf, conffile_stack[include_stack_ptr], BUFSIZE); 
  yy_switch_to_buffer(include_stack[include_stack_ptr]);
  return 0;
}
