/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * modules.c: A module loader.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: modules.c,v 1.80.2.4 2004/08/03 04:54:37 musirc Exp $
 */

#include "log.h"
#include "ircd.h"
#include "client.h"
#include "send.h"
#include "config.h"
#include "handlers.h"
#include "numeric.h"
#include "istring.h"
#include "modules.h"

#ifndef STATIC_MODULES
dlink_list mod_list = { NULL, NULL, 0 };
static char *base_autoload = NULL;

static const char *core_module_table[] =
{
  "m_message",
  "m_nick",
  "m_channel",
  "m_server",
  NULL
};

int num_mods = 0;

static dlink_list mod_paths = { NULL, NULL, 0 };

static void mo_modload(struct Client *, struct Client *, int, char **);
static void mo_modlist(struct Client *, struct Client *, int, char **);
static void mo_modreload(struct Client *, struct Client *, int, char **);
static void mo_modunload(struct Client *, struct Client *, int, char **);
static void mo_modrestart(struct Client *, struct Client *, int, char **);

struct Message modload_msgtab = {
 "MODLOAD", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_modload}
};
struct Message modunload_msgtab = {
 "MODUNLOAD", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_modunload}
};
struct Message modreload_msgtab = {
  "MODRELOAD", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_modreload}
};
struct Message modlist_msgtab = {
 "MODLIST", 0, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_modlist}
};
struct Message modrestart_msgtab = {
 "MODRESTART", 0, MFLG_SLOW,
 {m_unregistered, m_not_oper, m_ignore, mo_modrestart}
};

void
modules_init(void)
{
  mod_add_cmd(&modload_msgtab);
  mod_add_cmd(&modunload_msgtab);
  mod_add_cmd(&modreload_msgtab);
  mod_add_cmd(&modlist_msgtab);
  mod_add_cmd(&modrestart_msgtab);
}

/* input	- path
 * side effects - returns a module path from path
 */
static struct module_path *
mod_find_path(const char *path)
{
  dlink_node *ptr;
  struct module_path *mpath;
  
  DLINK_FOREACH(ptr, mod_paths.head)
  {
    mpath = ptr->data;

    if (!strcmp(path, mpath->path))
      return(mpath);
  }
  return(NULL);
}

/* Defines the base path to find the official modules.
 * I.E., all core/ modules MUST be in this directory.
 * input - path
 * side effects - sets the base path to path
 */
void
mod_set_base(void)
{
  unsigned int len;

  MyFree (base_autoload);
  len = strlen(MODPATH) + 9; /* whatever MODPATH + "autoload/" */
  base_autoload = MyMalloc(len + 1);
  snprintf(base_autoload, len, "%sautoload/", MODPATH);
}

/* input        - path
 * side effects - adds path to list
 */
void
mod_add_path(const char *path)
{
  unsigned int len = strlen (path);
  struct module_path *pathst;

  if (mod_find_path(path))
    return;

  pathst = MyMalloc(sizeof(struct module_path));
  pathst->path = MyMalloc (len + 1);
  strlcpy(pathst->path, path, len + 1);
  dlinkAdd(pathst, &pathst->node, &mod_paths);
}

/* side effects - clear the lists of paths */
void
mod_clear_paths(void)
{
  struct module_path *pathst;
  dlink_node *ptr, *next_ptr;

  DLINK_FOREACH_SAFE(ptr, next_ptr, mod_paths.head)
  {
    pathst = ptr->data;
    dlinkDelete(&pathst->node, &mod_paths);
    MyFree(pathst->path);
    MyFree(pathst);
  }
}

dlink_node *
findmodule_byname(const char *name)
{
  dlink_node *ptr;
  struct module *modp;

  DLINK_FOREACH(ptr, mod_list.head)
  {
    modp = ptr->data;
    if (!irccmp(modp->name, name))
      return(ptr);
  }
  return(NULL);
}

void
load_all_modules(int warn)
{
  DIR *system_module_dir = NULL;
  struct dirent *ldirent = NULL;
  char *module_fq_name;
  int len;
  unsigned int mq_len;

  modules_init();
  system_module_dir = opendir(base_autoload);

  if (system_module_dir == NULL)
  {
    ilog(WARN, "Could not load modules from %s: %s",
         base_autoload, strerror (errno));
    return;
  }

  while ((ldirent = readdir(system_module_dir)) != NULL)
  {
    len = strlen(ldirent->d_name);

    /* On HPUX, we have *.sl as shared library extension
     * -TimeMr14C
     */
    if ((len > 3) && (ldirent->d_name[len-3] == '.') &&
        (ldirent->d_name[len-2] == 's') &&
        ((ldirent->d_name[len-1] == 'o') ||
        (ldirent->d_name[len-1] == 'l')))
    {
       if ((mq_len = strlen(base_autoload) + len + 1) > PATH_MAX)
       {
          ilog (ERROR, "Module path for %s truncated, module not loaded!",
                ldirent->d_name);
       }
       else /* Guaranteed the path fits into a string of PATH_MAX now */
       {
         module_fq_name = MyMalloc(mq_len + 1);
         snprintf(module_fq_name, mq_len + 1, "%s/%s", base_autoload, ldirent->d_name);
         load_a_module(module_fq_name, warn, 0);
         MyFree(module_fq_name);
       }
    }
  }

  closedir(system_module_dir);
}

/* side effects - core modules are loaded, if any fail, kill ircd */
void
load_core_modules(int warn)
{
  char *module_name;
  int i;
  unsigned int m_len;

  for (i = 0; core_module_table[i]; i++)
  {
    if ((m_len = (strlen(MODPATH) + strlen(core_module_table[i]) + strlen(SHARED_SUFFIX) + 2)) > PATH_MAX)
    {
      ilog (ERROR, "Path for %s%s was truncated, the module was not loaded.",
            core_module_table[i], SHARED_SUFFIX);
    }
    else
    {
      module_name = MyMalloc(m_len + 1);
      snprintf(module_name, m_len + 1, "%s%s%s", MODPATH, core_module_table[i], SHARED_SUFFIX);
      if (load_a_module(module_name, warn, 1) == -1)
      {
        ilog(CRIT, "Error loading core module %s%s: terminating ircd",
	     core_module_table[i], SHARED_SUFFIX);
        exit (EXIT_FAILURE);
      }
      MyFree (module_name);
    }
  }
}

int
load_one_module(char *path, int coremodule)
{
  char* modpath;
  dlink_node *ptr;
  struct module_path *mpath;
  struct stat statbuf;
  unsigned int m_len;

  DLINK_FOREACH(ptr, mod_paths.head)
  {
    mpath = ptr->data;

    if ((m_len = strlen(mpath->path) + strlen(path) + 1) > PATH_MAX)
    {
      ilog(ERROR, "Path for %s/%s was truncated, not loading module from there",
           mpath->path, path);
      continue;
    }
    else
    {
      modpath = MyMalloc (m_len + 1);
      snprintf (modpath, m_len + 1, "%s/%s", mpath->path, path);

      if ((strstr(modpath, "../") == NULL) &&
           (strstr(modpath, "/..") == NULL))
        if (stat(modpath, &statbuf) == 0)
          if (S_ISREG(statbuf.st_mode))
	    return(load_a_module(modpath, 1, coremodule));
    }
  }
  sendto_realops_flags(UMODE_ALL, L_ALL,
                       "Cannot locate module %s", path);
  ilog(WARN, "Cannot locate module %s", path);
  return(-1);
}

static void
mo_modload(struct Client *client_p, struct Client *source_p,
	   int parc, char *parv[])
{
  char *m_bn;

  if (!IsAdmin(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  m_bn = basename(parv[1]);

  if (findmodule_byname(m_bn) != NULL)
  {
    sendto_one(source_p, ":%s NOTICE %s :MODLOAD: %s already loaded",
               me.name, source_p->name, m_bn);
    return;
  }

  load_one_module(parv[1], 0);
}

static void
mo_modunload(struct Client *client_p, struct Client *source_p,
	     int parc, char *parv[])
{
  char *m_bn;
  dlink_node *ptr;
  struct module *modp;

  if (!IsAdmin(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  m_bn = basename(parv[1]);

  if ((ptr = findmodule_byname(m_bn)) == NULL)
  {
    sendto_one(source_p, ":%s NOTICE %s :MODUNLOAD: %s not loaded",
               me.name, source_p->name, m_bn);
    return;
  }

  modp = ptr->data;

  if (modp->core == 1)
  {
    sendto_one(source_p,
               ":%s NOTICE %s :MODUNLOAD: %s is a core module and can't be unloaded",
               me.name, source_p->name, m_bn);
    return;
  }

  if (unload_one_module(m_bn, 1) == -1)
  {
    sendto_one(source_p, ":%s NOTICE %s :MODUNLOAD: %s not loaded",
               me.name, source_p->name, m_bn);
  }
}

static void
mo_modreload(struct Client *client_p, struct Client *source_p,
	     int parc, char *parv[])
{
  char *m_bn;
  dlink_node *ptr;
  struct module *modp;
  int check_core;

  if (!IsAdmin(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  m_bn = basename(parv[1]);

  if ((ptr = findmodule_byname(m_bn)) == NULL)
  {
    sendto_one(source_p, ":%s NOTICE %s :MODRELOAD: %s not loaded",
               me.name, source_p->name, m_bn);
    return;
  }

  modp = ptr->data;
  check_core = modp->core;

  if (unload_one_module(m_bn, 1) == -1)
  {
    sendto_one(source_p, ":%s NOTICE %s :MODRELOAD: %s not loaded",
               me.name, source_p->name, m_bn);
    return;
  }

  if ((load_one_module(parv[1], check_core) == -1) && check_core)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "Error reloading core "
                         "module: %s: terminating ircd", parv[1]);
    ilog(CRIT, "Error loading core module %s: terminating ircd", parv[1]);
    exit(0);
  }
}

static void
mo_modlist(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  dlink_node *ptr;
  struct module *modp;

  if (!IsAdmin(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  DLINK_FOREACH(ptr, mod_list.head)
  {
    modp = ptr->data;
    if (parc > 1)
    {
      if (match(parv[1], modp->name))
        sendto_one(source_p, form_str(RPL_MODLIST), me.name, parv[0],
                   modp->name, modp->address,
                   modp->version, modp->core?"(core)":"");
    }
    else
      sendto_one(source_p, form_str(RPL_MODLIST), me.name, parv[0],
                 modp->name, modp->address,
                 modp->version, modp->core?"(core)":"");
  }
  sendto_one(source_p, form_str(RPL_ENDOFMODLIST),
             me.name, source_p->name);
}

static void
mo_modrestart(struct Client *client_p, struct Client *source_p, int parc, char *parv[])
{
  int modnum;
  dlink_node *ptr;
  dlink_node *tptr;
  struct module *modp;

  if (!IsAdmin(source_p))
  {
    sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
               me.name, source_p->name);
    return;
  }

  sendto_one(source_p, ":%s NOTICE %s :MODRESTART: Reloading all modules",
             me.name, source_p->name);

  modnum = num_mods;

  DLINK_FOREACH_SAFE(ptr, tptr, mod_paths.head)
  {
    modp = ptr->data;
    unload_one_module(modp->name, 0);
  }

  load_all_modules(0);
  load_core_modules(0);
  rehash(0);
  sendto_realops_flags(UMODE_ALL, L_ALL,
		       "Module Restart: %d modules unloaded, %d modules loaded",
                       modnum, num_mods);
  ilog(WARN, "Module Restart: %d modules unloaded, %d modules loaded",
       modnum, num_mods);
}

#else /* STATIC_MODULES */
#include "server.h"

/* side effects - all the msgtabs are added for static modules */
void
load_all_modules(int u)
{
  mod_add_cmd(&error_msgtab);
  mod_add_cmd(&accept_msgtab);
  mod_add_cmd(&admin_msgtab);
  mod_add_cmd(&ajoin_msgtab);
  mod_add_cmd(&away_msgtab);
  mod_add_cmd(&capab_msgtab);
  mod_add_cmd(&classlist_msgtab);
  mod_add_cmd(&close_msgtab);
  mod_add_cmd(&connect_msgtab);
  mod_add_cmd(&cjupe_msgtab);
  mod_add_cmd(&die_msgtab);
  mod_add_cmd(&flags_msgtab);
  mod_add_cmd(&help_msgtab);
  mod_add_cmd(&info_msgtab);
  mod_add_cmd(&invite_msgtab);
  mod_add_cmd(&ison_msgtab);
  mod_add_cmd(&join_msgtab);
  mod_add_cmd(&kick_msgtab);
  mod_add_cmd(&kill_msgtab);
  mod_add_cmd(&kline_msgtab);
  mod_add_cmd(&dline_msgtab);
  mod_add_cmd(&knock_msgtab);
  mod_add_cmd(&links_msgtab);
  mod_add_cmd(&list_msgtab);
  mod_add_cmd(&locops_msgtab);
  mod_add_cmd(&lusers_msgtab);
  mod_add_cmd(&privmsg_msgtab);
  mod_add_cmd(&notice_msgtab);
  mod_add_cmd(&map_msgtab);
  mod_add_cmd(&mode_msgtab);
  mod_add_cmd(&omode_msgtab);
  mod_add_cmd(&motd_msgtab);
  mod_add_cmd(&onames_msgtab);
  mod_add_cmd(&names_msgtab);
  mod_add_cmd(&nick_msgtab);
  mod_add_cmd(&njupe_msgtab);
  mod_add_cmd(&oper_msgtab);
  mod_add_cmd(&opme_msgtab);
  mod_add_cmd(&part_msgtab);
  mod_add_cmd(&pass_msgtab);
  mod_add_cmd(&ping_msgtab);
  mod_add_cmd(&pong_msgtab);
  mod_add_cmd(&post_msgtab);
  mod_add_cmd(&get_msgtab);
  mod_add_cmd(&put_msgtab);
  mod_add_cmd(&quit_msgtab);
  mod_add_cmd(&rehash_msgtab);
  mod_add_cmd(&restart_msgtab);
  mod_add_cmd(&server_msgtab);
  mod_add_cmd(&set_msgtab);
  mod_add_cmd(&sjoin_msgtab);
  mod_add_cmd(&sjupe_msgtab);
  mod_add_cmd(&squit_msgtab);
  mod_add_cmd(&stats_msgtab);
  mod_add_cmd(&svinfo_msgtab);
  mod_add_cmd(&svnick_msgtab);
  mod_add_cmd(&fixinvite_msgtab);
  mod_add_cmd(&fixts_msgtab);
  mod_add_cmd(&testline_msgtab);
  mod_add_cmd(&time_msgtab);
  mod_add_cmd(&otopic_msgtab);
  mod_add_cmd(&topic_msgtab);
  mod_add_cmd(&tburst_msgtab);
  mod_add_cmd(&ctrace_msgtab);
  mod_add_cmd(&trace_msgtab);
  mod_add_cmd(&uncjupe_msgtab);
  mod_add_cmd(&unkline_msgtab);
  mod_add_cmd(&undline_msgtab);
  mod_add_cmd(&unnjupe_msgtab);
  mod_add_cmd(&unsjupe_msgtab);
  mod_add_cmd(&unxline_msgtab);
  mod_add_cmd(&user_msgtab);
  mod_add_cmd(&userhost_msgtab);
  mod_add_cmd(&version_msgtab);
  mod_add_cmd(&wallops_msgtab);
  mod_add_cmd(&who_msgtab);
  mod_add_cmd(&owho_msgtab);
  mod_add_cmd(&owhois_msgtab);
  mod_add_cmd(&whois_msgtab);
  mod_add_cmd(&whowas_msgtab);
  mod_add_cmd(&xline_msgtab);
  mod_add_cmd(&nickserv_msgtab);
  mod_add_cmd(&chanserv_msgtab);
  mod_add_cmd(&memoserv_msgtab);
  mod_add_cmd(&operserv_msgtab);
  mod_add_cmd(&statserv_msgtab);
  mod_add_cmd(&helpserv_msgtab);
  mod_add_cmd(&global_msgtab);
  mod_add_cmd(&aim_msgtab);
  mod_add_cmd(&aimold_msgtab);
}
#endif /* STATIC_MODULES */
