/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * packet.c: Packet handlers.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: packet.c,v 1.36.2.1 2004/06/16 17:06:46 musirc Exp $
 */

#include "bsd.h"
#include "config.h"
#include "ircd.h"
#include "server.h"
#include "parse.h"
#include "istring.h"
#include "packet.h"

static char readBuf[READBUF_SIZE];
static void client_dopacket(struct Client *, char *, size_t);

/* inputs       - pointer to a dbuf queue
 *              - pointer to buffer to copy data to
 * output       - length of <buffer>
 * side effects - one line is copied and removed from the dbuf
 */
static int
extract_one_line(struct dbuf_queue *qptr, char *buffer)
{
  struct dbuf_block *block;
  int line_bytes = 0, empty_bytes = 0, phase = 0;
  unsigned int idx;
  char c;
  dlink_node *ptr;

  /* Phase 0: "empty" characters before the line
   * Phase 1: copying the line
   * Phase 2: "empty" characters after the line
   *          (delete them as well and free some space in the dbuf)
   * Empty characters are CR, LF and space (but, of course, not
   * in the middle of a line). We try to remove as much of them as we can,
   * since they simply eat server memory.
   */
  DLINK_FOREACH(ptr, qptr->blocks.head)
  {
    block = ptr->data;

    for (idx = 0; idx < block->size; idx++)
    {
      c = block->data[idx];
      if (IsEol(c) || (c == ' ' && phase != 1))
      {
        empty_bytes++;
        if (phase == 1)
          phase = 2;
      }
      else switch (phase)
      {
        case 0: phase = 1;
        case 1: if (line_bytes++ < BUFSIZE - 2)
                  *buffer++ = c;
                break;
        case 2: *buffer = '\0';
                dbuf_delete(qptr, line_bytes + empty_bytes);
                return IRCD_MIN(line_bytes, BUFSIZE - 2);
      }
    }
  }

  /* Now, if we haven't reached phase 2, ignore all line bytes
   * that we have read, since this is a partial line case.
   */
  if (phase != 2)
    line_bytes = 0;
  else
    *buffer = '\0';

  /* Remove what is now unnecessary */
  dbuf_delete(qptr, line_bytes + empty_bytes);
  return IRCD_MIN(line_bytes, BUFSIZE - 2);
}

/* parse_client_queued - parse client queued messages */
static void
parse_client_queued(struct Client *client_p)
{ 
  int dolen = 0;
  int checkflood = 1;
  struct LocalUser *lclient_p = client_p->localClient;

  if (IsUnknown(client_p))
  {
    int i = 0;

    for(;;)
    {
      if (IsDefunct(client_p))
	return;

      /* rate unknown clients at MAX_FLOOD per loop */
      if (i >= MAX_FLOOD)
        break;

      dolen = extract_one_line(&lclient_p->buf_recvq, readBuf);
      if (dolen == 0)
	break;

      client_dopacket(client_p, readBuf, dolen);
      i++;

      /* if they've dropped out of the unknown state, break and move
       * to the parsing for their appropriate status.  --fl
       */
      if(!IsUnknown(client_p))
        break;
    }
  }

  if (IsServer(client_p) || IsConnecting(client_p) || IsHandshake(client_p))
  {
    while (1)
    {
      if (IsDefunct(client_p))
        return;
      if ((dolen = extract_one_line(&lclient_p->buf_recvq,
                                    readBuf)) == 0)
        break;
      client_dopacket(client_p, readBuf, dolen);
    }
  }
  else if(IsClient(client_p))
  {
    if (ConfigFileEntry.no_oper_flood && (IsOper(client_p) || IsCanFlood(client_p)))
    {
      if (ConfigFileEntry.true_no_oper_flood)
        checkflood = -1;
      else
        checkflood = 0;
    }

    /* Handle flood protection here - if we exceed our flood limit on
     * messages in this loop, we simply drop out of the loop prematurely.
     */
    for (;;)
    {
      if (IsDefunct(client_p))
	break;

      /* This flood protection works as follows:
       * A client is given allow_read lines to send to the server.  Every
       * time a line is parsed, sent_parsed is increased.  sent_parsed
       * is decreased by 1 every time flood_recalc is called.
       * Thus a client can 'burst' allow_read lines to the server, any
       * excess lines will be parsed one per flood_recalc() call.
       * Therefore a client will be penalised more if they keep flooding,
       * as sent_parsed will always hover around the allow_read limit
       * and no 'bursts' will be permitted.
       */
      if (checkflood > 0)
      {
        if(lclient_p->sent_parsed >= lclient_p->allow_read)
          break;
      }
      
      /* allow opers 4 times the amount of messages as users. */
      else if (lclient_p->sent_parsed >= (4 * lclient_p->allow_read) &&
               checkflood != -1)
        break;

      dolen = extract_one_line(&lclient_p->buf_recvq, readBuf);
      if (dolen == 0)
        break;

      client_dopacket(client_p, readBuf, dolen);
      lclient_p->sent_parsed++;
    }
  }
}

/* marks the end of the clients grace period */
void
flood_endgrace(struct Client *client_p)
{
  SetFloodDone(client_p);

  /* Drop their flood limit back down */
  client_p->localClient->allow_read = MAX_FLOOD;

  /* sent_parsed could be way over MAX_FLOOD but under MAX_FLOOD_BURST,
   * so reset it.
   */
  client_p->localClient->sent_parsed = 0;
}
	    
/* recalculate the number of allowed flood lines. this should be called
 * once a second on any given client. We then attempt to flush some data.
 */
void
flood_recalc(int fd, void *data)
{
  struct Client *client_p = data;
  struct LocalUser *lclient_p = client_p->localClient;
 
  /* This can happen in the event that the client detached. */
  if (!lclient_p)
    return;

  /* allow a bursting client their allocation per second, allow
   * a client whos flooding an extra 2 per second
   */
  if(IsFloodDone(client_p))
    lclient_p->sent_parsed -= 2;
  else
    lclient_p->sent_parsed = 0;
  
  if(lclient_p->sent_parsed < 0)
    lclient_p->sent_parsed = 0;
  
  parse_client_queued(client_p);
  
  /* And now, try flushing .. */
  if (!IsDead(client_p))
  {
    /* and finally, reset the flood check */
    comm_setflush(fd, 1000, flood_recalc, client_p);
  }
}

void
read_packet(int fd, void *data)
{
  struct Client *client_p = data;
  int length = 0;

  if (IsDefunct(client_p))
    return;

  length = recv(client_p->localClient->fd, readBuf, READBUF_SIZE, 0);

  if (length <= 0)
  {
    if ((length == -1) && ignoreErrno(errno))
    {
      comm_setselect(client_p->localClient->fd, FDLIST_IDLECLIENT, COMM_SELECT_READ,
                     read_packet, client_p, 0);
      return;
    }
    dead_link_on_read(client_p, length);
    return;
  }

  if (client_p->lasttime < CurrentTime)
    client_p->lasttime = CurrentTime;
  if (client_p->lasttime > client_p->since)
    client_p->since = CurrentTime;
  ClearPingSent(client_p);

  dbuf_put(&client_p->localClient->buf_recvq, readBuf, length);

  /* Attempt to parse what we have */
  parse_client_queued(client_p);

  /* Check to make sure we're not flooding */
  if (!(IsServer(client_p) || IsHandshake(client_p) || IsConnecting(client_p)) &&
      (dbuf_length(&client_p->localClient->buf_recvq) >
       (unsigned int)ConfigFileEntry.client_flood))
  {
    if (!(ConfigFileEntry.no_oper_flood && IsOper(client_p)))
    {
      exit_client(client_p, client_p, client_p, "Excess Flood");
      return;
    }
  }

  if (!IsDefunct(client_p))
  {
    /* If we get here, we need to register for another COMM_SELECT_READ */
    if (PARSE_AS_SERVER(client_p))
      comm_setselect(client_p->localClient->fd, FDLIST_SERVER, COMM_SELECT_READ,
                     read_packet, client_p, 0);
    else
      comm_setselect(client_p->localClient->fd, FDLIST_IDLECLIENT, COMM_SELECT_READ,
                     read_packet, client_p, 0);
  }
}
/* client_dopacket - copy packet to client buf and parse it
 *      client_p - pointer to client structure for which the buffer data
 *             applies.
 *      buffer - pointr to the buffer containing the newly read data
 *      length - number of valid bytes of data in the buffer
 * Note:
 *      It is implicitly assumed that dopacket is called only
 *      with client_p of "local" variation, which contains all the
 *      necessary fields (buffer etc..)
 */
static void
client_dopacket(struct Client *client_p, char *buffer, size_t length)
{
  ++me.localClient->receiveM;
  ++client_p->localClient->receiveM;

  client_p->localClient->receiveB += length;

  if (client_p->localClient->receiveB > 1023)
  {
    client_p->localClient->receiveK += (client_p->localClient->receiveB >> 10);
    client_p->localClient->receiveB &= 0x03ff; /* 2^10 = 1024, 3ff = 1023 */
  }
  me.localClient->receiveB += length;

  if (me.localClient->receiveB > 1023)
  {
    me.localClient->receiveK += (me.localClient->receiveB >> 10);
    me.localClient->receiveB &= 0x03ff;
  }
  parse(client_p, buffer, buffer + length);
}
