/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * poll.c: POSIX poll() compatible network routines.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: poll.c,v 1.9 2004/02/08 06:46:05 musirc Exp $
 */

#include "stdinc.h"
#include <sys/poll.h>
#include "fdlist.h"
#include "bsd.h"
#include "client.h"
#include "istring.h"
#include "ircd.h"
#include "list.h"
#include "listener.h"
#include "numeric.h"
#include "packet.h"
#include "res.h"
#include "restart.h"
#include "auth.h"
#include "config.h"
#include "log.h"
#include "server.h"
#include "send.h"

#ifndef POLLRDNORM
#define POLLRDNORM POLLIN
#endif
#ifndef POLLWRNORM
#define POLLWRNORM POLLOUT
#endif

struct _pollfd_list {
    struct pollfd pollfds[HARD_FDLIMIT];
    int maxindex; /* highest FD number */
} pollfd_list;

static void poll_update_pollfds(int, short, PF *);

static inline int
poll_findslot(void)
{
  int i;

  for (i = 0; i < HARD_FDLIMIT; i++)
    if (pollfd_list.pollfds[i].fd == -1)
    {
      /* MATCH!!#$*&$ */
      return i;
    }
  return -1;
}

/* set and clear entries in the pollfds[] array. */ 
static void
poll_update_pollfds(int fd, short event, PF *handler)
{  
  fde_t *F = &fd_table[fd];
  int comm_index;

  if (F->comm_index < 0)
    F->comm_index = poll_findslot();
  comm_index = F->comm_index;

  /* Update the events */
  if (handler != NULL)
  {
    F->list = FDLIST_IDLECLIENT;
    pollfd_list.pollfds[comm_index].events |= event;
    pollfd_list.pollfds[comm_index].fd = fd;
    /* update maxindex here */
    if (comm_index > pollfd_list.maxindex)
      pollfd_list.maxindex = comm_index;
  }
  else {
    pollfd_list.pollfds[comm_index].events &= ~event;
    if (pollfd_list.pollfds[comm_index].events == 0)
    {
      pollfd_list.pollfds[comm_index].fd = -1;
      pollfd_list.pollfds[comm_index].revents = 0;
      F->comm_index = -1;
      F->list = FDLIST_NONE;

      /* update pollfd_list.maxindex here */
      if (comm_index == pollfd_list.maxindex)
        while (pollfd_list.maxindex >= 0 &&
               pollfd_list.pollfds[pollfd_list.maxindex].fd == -1)
          pollfd_list.maxindex--;
    }
  }
}

/* This is a needed exported function which will be called to initialise
 * the network loop code.
 */
void
init_netio(void)
{
  int fd;

  for (fd = 0; fd < HARD_FDLIMIT; fd++)
    pollfd_list.pollfds[fd].fd = -1;
  pollfd_list.maxindex = 0;
}

/* This is a needed exported function which will be called to register
 * and deregister interest in a pending IO state for a given FD.
 */
void
comm_setselect(int fd, fdlist_t list, unsigned int type, PF *handler,
               void *client_data, time_t timeout)
{  
  fde_t *F = &fd_table[fd];

  if (type & COMM_SELECT_READ)
  {
    F->read_handler = handler;
    F->read_data = client_data;
    poll_update_pollfds(fd, POLLRDNORM, handler);
  }
  if (type & COMM_SELECT_WRITE)
  {
    F->write_handler = handler;
    F->write_data = client_data;
    poll_update_pollfds(fd, POLLWRNORM, handler);
  }
  if (timeout)
    F->timeout = CurrentTime + (timeout / 1000);
}
 
/* Input: The maximum time to delay.
 * Side-effects: Deregisters future interest in IO and calls the handlers
 *               if an event occurs for an FD.
 * Comments: Check all connections for new connections and input data
 * that is to be processed. Also check for connections with data queued
 * and whether we can write it out.
 * Called to do the new-style IO, courtesy of of squid (like most of this
 * new IO code). This routine handles the stuff we've hidden in
 * comm_setselect and fd_table[] and calls callbacks for IO ready
 * events.
 */
void
comm_select(unsigned long delay)
{
  int num, fd, ci, revents;
  PF *hdl;
  fde_t *F;
  
  while ((num = poll(pollfd_list.pollfds, pollfd_list.maxindex + 1, delay)) < 0
         && ignoreErrno(errno))
    ;

  set_time();
 
  if (num == 0)
    return;

  for (ci = 0; ci < pollfd_list.maxindex + 1; ci++)
  {
    if (((revents = pollfd_list.pollfds[ci].revents) == 0) ||
        (pollfd_list.pollfds[ci].fd) == -1)
      continue;

    fd = pollfd_list.pollfds[ci].fd;
    F = &fd_table[fd];

    if (revents & (POLLRDNORM | POLLIN | POLLHUP | POLLERR))
    {
      hdl = F->read_handler;
      F->read_handler = NULL;
      poll_update_pollfds(fd, POLLRDNORM, NULL);
      if (hdl)
       hdl(fd, F->read_data);
    }
    if (revents & (POLLWRNORM | POLLOUT | POLLHUP | POLLERR))
    {
      hdl = F->write_handler;
      F->write_handler = NULL;
      poll_update_pollfds(fd, POLLWRNORM, NULL);
      if (hdl)
        hdl(fd, F->write_data);
    }
  }
}
