/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * server.c: Server related functions.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: server.c,v 1.42.2.8 2004/07/22 20:31:28 musirc Exp $
 */

#include "tools.h"
#include "channel.h"
#include "channel_mode.h"
#include "client.h"
#include "hash.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "bsd.h"
#include "numeric.h"
#include "packet.h"
#include "config.h"
#include "list.h"
#include "server.h"
#include "log.h"
#include "user.h"
#include "send.h"
#include "getinfo.h"

#define MIN_CONN_FREQ 300

dlink_list cap_list = { NULL, NULL, 0 };
static CNCB serv_connect_callback;

/* my_name_for_link - return wildcard name of my server name 
 * according to given config entry --Jto
 */
const char *
my_name_for_link(struct ConfItem *conf)
{
  struct AccessItem *aconf;

  aconf = (struct AccessItem *)map_to_conf(conf);
  if (aconf->fakename != NULL)
    return(aconf->fakename);
  else
    return(me.name);
}

void
write_links_file(void *n)
{
  MessageFileLine *next_mptr = 0, *mptr = 0, *currentMessageLine = 0, *newMessageLine = 0;
  MessageFile *MessageFileptr;
  const char *p;
  FBFILE* file;
  char buff[512];
  dlink_node *ptr;

  MessageFileptr = &ConfigFileEntry.linksfile;

  if ((file = fbopen(MessageFileptr->fileName, "w")) == NULL)
    return;

  for (mptr = MessageFileptr->contentsOfFile; mptr; mptr = next_mptr)
  {
    next_mptr = mptr->next;
    MyFree(mptr);
  }
  MessageFileptr->contentsOfFile = NULL;
  currentMessageLine = NULL;

  DLINK_FOREACH(ptr, global_serv_list.head)
  {
    size_t nbytes = 0;
    struct Client *target_p = ptr->data;

    /* skip ourselves, we send ourselves in /links */
    if (IsMe(target_p))
      continue;

    /* skip hidden servers */
    if (IsHidden(target_p) && !ConfigServerHide.disable_hidden)
      continue;

    if (target_p->info[0])
      p = target_p->info;
    else
      p = "No description";

    newMessageLine = MyMalloc(sizeof(MessageFileLine));

    ircsprintf(newMessageLine->line,"%s %s :1 %s",
               target_p->name, me.name, p);
    newMessageLine->next = NULL;

    if (MessageFileptr->contentsOfFile)
    {
      if (currentMessageLine)
        currentMessageLine->next = newMessageLine;
      currentMessageLine = newMessageLine;
    }
    else
    {
      MessageFileptr->contentsOfFile = newMessageLine;
      currentMessageLine = newMessageLine;
    }

    nbytes = ircsprintf(buff, "%s %s :1 %s\n", target_p->name, me.name, p);
    fbputs(buff, file, nbytes);
  }

  fbclose(file);
}

/* Do the basic thing in delivering the message (command)
 *      across the relays to the specific server (server) for
 *      actions.
 *      Note:   The command is a format string and *MUST* be
 *              of prefixed style (e.g. ":%s COMMAND %s ...").
 *              Command can have only max 8 parameters.
 *      server  parv[server] is the parameter identifying the
 *              target server.
 *              parv[server] is replaced with the pointer to the
 *              real servername from the matched client
 */
int
hunt_server(struct Client *client_p, struct Client *source_p, const char *command,
            int server, int parc, char *parv[])
{
  struct Client *target_p = NULL, *target_tmp = NULL;
  dlink_node *ptr;
  int wilds;

  /* Assume it's me, if no server */
  if (parc <= server || EmptyString(parv[server]) ||
      match(me.name, parv[server]) ||
      match(parv[server], me.name))
    return(HUNTED_ISME);

  /* These are to pickup matches that would cause the following
   * message to go in the wrong direction while doing quick fast
   * non-matching lookups.
   */
  if (MyClient(source_p))
    target_p = find_client(parv[server]);
  else
    target_p = find_person(parv[server]);

  if (target_p)
    if (target_p->from == source_p->from && !MyConnect(target_p))
      target_p = NULL;
  if (target_p == NULL && (target_p = find_server(parv[server])))
    if (target_p->from == source_p->from && !MyConnect(target_p))
      target_p = NULL;

  collapse(parv[server]);
  wilds = (strchr(parv[server], '?') || strchr(parv[server], '*'));

  /* Again, if there are no wild cards involved in the server
   * name, use the hash lookup
   */
  if (target_p == NULL)
  {
    if (!wilds)
    {
      if (!(target_p = find_server(parv[server])))
      {
        sendto_one(source_p, form_str(ERR_NOSUCHSERVER),
                   me.name, source_p->name, parv[server]);
        return(HUNTED_NOSUCH);
      }
    }
    else
    {
      DLINK_FOREACH(ptr, global_client_list.head)
      {
        target_tmp = ptr->data;

        if (match(parv[server], target_tmp->name))
        {
          if (target_tmp->from == source_p->from && !MyConnect(target_tmp)) 
            continue;
          target_p = ptr->data;

          if (IsRegistered(target_p) && (target_p != client_p))
            break;
        }
      }
    }
  }

  if (target_p != NULL)
  {
    if(!IsRegistered(target_p))
    {
      sendto_one(source_p, form_str(ERR_NOSUCHSERVER),
                 me.name, source_p->name, parv[server]);
      return HUNTED_NOSUCH;
    }

    if (IsMe(target_p) || MyClient(target_p))
      return HUNTED_ISME;

    if (!match(target_p->name, parv[server]))
      parv[server] = target_p->name;

    sendto_one(target_p, command, parv[0],
               parv[1], parv[2], parv[3], parv[4],
               parv[5], parv[6], parv[7], parv[8]);
    return(HUNTED_PASS);
  }

  sendto_one(source_p, form_str(ERR_NOSUCHSERVER),
             me.name, source_p->name, parv[server]);
  return(HUNTED_NOSUCH);
}

/* scan through configuration and try new connections.
 * Returns the calendar time when the next call to this
 * function should be made latest. (No harm done if this
 * is called earlier or later...)
 */
void
try_connections(void *n)
{
  struct Client *pending_connection;
  dlink_node *ptr;
  struct ConfItem *conf;
  struct AccessItem *aconf;
  struct ClassItem *cltmp;
  int confrq;

  if (GlobalSetOptions.autoconn == 0)
    return;

  DLINK_FOREACH(ptr, server_items.head)
  {
    conf = ptr->data;
    aconf = (struct AccessItem *)map_to_conf(conf);

    /* Also when already connecting! (update holdtimes) --SRB */
    if (!(aconf->status & CONF_SERVER) || aconf->port <= 0 ||
        !(IsConfAllowAutoConn(aconf)))
      continue;

    cltmp = (struct ClassItem *)map_to_conf(aconf->class_ptr);

    /* Skip this entry if the use of it is still on hold until
     * future. Otherwise handle this entry (and set it on hold
     * until next time). Will reset only hold times, if already
     * made one successfull connection... [this algorithm is
     * a bit fuzzy... -- msa >;) ]
     */
    if (aconf->hold > CurrentTime)
      continue;

    if ((confrq = get_con_freq(cltmp)) < MIN_CONN_FREQ )
      confrq = MIN_CONN_FREQ;

    aconf->hold = CurrentTime + confrq;

    /* Found a CONNECT config with port specified, scan clients
     * and see if this server is already connected?
     */
    if (find_server(conf->name) != NULL)
      continue;

    if (CurrUserCount(cltmp) < MaxTotal(cltmp))
    {
      /* Go to the end of the list, if not already last */
      if (ptr->next != NULL)
      {
        dlinkDelete(ptr, &server_items);
        dlinkAddTail(conf, &conf->node, &server_items);
      }

      DLINK_FOREACH(ptr, unknown_list.head)
      {
        pending_connection = ptr->data;

        if (pending_connection->name[0] && conf->name != NULL)
        {
          if (0 == irccmp(conf->name, pending_connection->name))
            return;
        }
      }

      /* We used to only print this if serv_connect() actually
       * succeeded, but since comm_tcp_connect() can call the callback
       * immediately if there is an error, we were getting error messages
       * in the wrong order. SO, we just print out the activated line,
       * and let serv_connect() / serv_connect_callback() print an
       * error afterwards if it fails.
       *   -- adrian
       */
      if (ConfigServerHide.hide_server_ips)
        sendto_realops_flags(UMODE_ALL, L_ALL, "CONNECT: %s",
                             conf->name);
      else
        sendto_realops_flags(UMODE_ALL, L_ALL, "CONNECT: %s[%s]",
                             conf->name, aconf->host);
      serv_connect(aconf, NULL);
      /* We connect only one at time... */
      return;
    }
  }
}

int
check_server(const char *name, struct Client *client_p)
{
  dlink_node *ptr;
  struct ConfItem *conf = NULL;
  struct ConfItem *server_conf = NULL;
  struct AccessItem *server_aconf = NULL;
  struct AccessItem *aconf = NULL;
  int error = -1;

  if (client_p == NULL)
    return(error);

  if (strlen(name) > HOSTLEN)
    return(-4);

  /* loop through looking for all possible connect items that might work */
  DLINK_FOREACH(ptr, server_items.head)
  {
    conf = ptr->data;
    aconf = (struct AccessItem *)map_to_conf(conf);

    if (!match(name, conf->name))
      continue;

    error = -3;

    if (match(aconf->host, client_p->host) ||
        match(aconf->host, client_p->localClient->sockhost))
    {
      error = -2;

      if (EmptyString(client_p->localClient->passwd) || aconf->passwd == NULL)
        return(-2);

      if (IsConfEncrypted(aconf))
      {
        if (!strcmp(aconf->passwd, (const char *)crypt(client_p->localClient->passwd,
             				               aconf->passwd)))
          server_conf = conf;
      }
      else
      {
        if (!strcmp(aconf->passwd, client_p->localClient->passwd))
          server_conf = conf;
      }
    }
  }

  if (server_conf == NULL)
    return(error);

  attach_conf(client_p, server_conf);
  /* Now find all leaf or hub config items for this server */
  DLINK_FOREACH(ptr, hub_items.head)
  {
    conf = ptr->data;

    if (!match(name, conf->name))
      continue;
    attach_conf(client_p, conf);
  }
  DLINK_FOREACH(ptr, leaf_items.head)
  {
    conf = ptr->data;
    if (!match(name, conf->name))
      continue;
    attach_conf(client_p, conf);
  }
  server_aconf = (struct AccessItem *)map_to_conf(server_conf);

  if (aconf != NULL)
  {
    struct sockaddr_in *v4;
#ifdef IPV6
    struct sockaddr_in6 *v6;
#endif
    switch (aconf->aftype)
    {
#ifdef IPV6
      case AF_INET6:
        v6 = (struct sockaddr_in6 *)&aconf->ipnum;

        if (IN6_IS_ADDR_UNSPECIFIED(&v6->sin6_addr))
          memcpy(&aconf->ipnum, &client_p->localClient->ip, sizeof(struct irc_ssaddr));
        break;
#endif
      case AF_INET:
        v4 = (struct sockaddr_in *)&aconf->ipnum;

        if (v4->sin_addr.s_addr == INADDR_NONE)
          memcpy(&aconf->ipnum, &client_p->localClient->ip, sizeof(struct irc_ssaddr));
        break;
    }
  }

  return(0);
}

/* inputs- string name of CAPAB
 * int flag of capability
 */
void
add_capability(const char *capab_name, int cap_flag, int add_to_default)
{
  struct Capability *cap;

  cap = (struct Capability *)MyMalloc(sizeof(*cap));
  DupString(cap->name, capab_name);
  cap->cap = cap_flag;
  dlinkAdd(cap, &cap->node, &cap_list);
  if (add_to_default)
    default_server_capabs |= cap_flag;
}

/* inputs- string name of capab to find
 * output- 0 if not found CAPAB otherwise
 */
int
find_capability(const char *capab)
{
  dlink_node *ptr;
  struct Capability *cap;

  DLINK_FOREACH(ptr, cap_list.head)
  {
    cap = ptr->data;

    if (cap->cap != 0)
    {
      if (!irccmp(cap->name, capab))
	return(cap->cap);
    }
  }
  return(0);
}

void
send_capabilities(struct Client *client_p)
{
  struct Capability *cap = NULL;
  char msgbuf[BUFSIZE], *t;
  int tl;
  dlink_node *ptr;

  t = msgbuf;
  DLINK_FOREACH(ptr, cap_list.head)
  {
    cap = ptr->data;

    if (cap->cap & (default_server_capabs))
    {
      tl = ircsprintf(t, "%s ", cap->name);
      t += tl;
    }
  }
  t--;
  *t = '\0';
  sendto_one(client_p, "CAPAB :%s", msgbuf);
}

/* inputs	- client (server) to send nick towards
 * 		- client to send nick for
 * side effects	- NICK message is sent towards given client_p
 */
void
sendnick_TS(struct Client *client_p, struct Client *target_p)
{
  static char ubuf[12];

  if (!IsPerson(target_p))
    return;
  
  send_umode(NULL, target_p, 0, SEND_UMODES, ubuf);
  if (!*ubuf)
  {
    ubuf[0] = '+';
    ubuf[1] = '\0';
  }
 
  sendto_one(client_p, "NICK %s %d %lu %s %s %s %s :%s",
             target_p->name, target_p->hopcount + 1,
             (unsigned long) target_p->tsinfo,
             ubuf, target_p->username, target_p->host,
             target_p->user->server->name, target_p->info);
}

/* show current server capabilities
 * inputs       - pointer to an struct Client
 * output       - pointer to static string
 * side effects - build up string representing capabilities of server listed
 */
const char *
show_capabilities(struct Client *target_p)
{
  static char msgbuf[BUFSIZE];
  struct Capability *cap;
  char *t;
  int tl;
  dlink_node *ptr;

  t = msgbuf;

  if (!target_p->localClient->caps)
  {
    *msgbuf = '\0';
    return(msgbuf);
  }

  DLINK_FOREACH(ptr, cap_list.head)
  {
    cap = ptr->data;

    if (cap->cap & target_p->localClient->caps)
    {
      tl = ircsprintf(t, "%s ", cap->name);
      t += tl;
    }
  }
  *t = '\0';

  return(msgbuf);
}

int
server_estab(struct Client *client_p)
{
  struct Membership *ms;
  struct Client *target_p;
  struct Channel *chptr;
  struct ConfItem *conf;
  struct AccessItem *aconf = NULL;
  char *host;
  const char *inpath;
  static char inpath_ip[HOSTLEN * 2 + USERLEN + 6];
  dlink_node *m, *ptr, *mptr;

  if (client_p == NULL)
    return(-1);

  strlcpy(inpath_ip, get_client_name(client_p, SHOW_IP), sizeof(inpath_ip));

  inpath = get_client_name(client_p, MASK_IP); /* "refresh" inpath with host */
  host = client_p->name;

  if ((conf = find_conf_name(&client_p->localClient->confs, host, SERVER_TYPE)) == NULL)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "Warning: Lost connect block "
                         "for server %s(this shouldn't happen)!", host);
    return(exit_client(client_p, client_p, client_p, "Lost connect block!"));
  }

  MyFree(client_p->localClient->passwd);
  client_p->localClient->passwd = NULL;
  SetGotId(client_p);

  /* If there is something in the serv_list, it might be this
   * connecting server..
   */
  if (!ServerInfo.hub && serv_list.head)
  {
    if (client_p != serv_list.head->data || serv_list.head->next)
    {
      sendto_one(client_p, "ERROR :I'm a leaf not a hub");
      return(exit_client(client_p, client_p, client_p, "I'm a leaf"));
    }
  }
  aconf = (struct AccessItem *)map_to_conf(conf);

  if (IsUnknown(client_p))
  {
    if (!EmptyString(aconf->spasswd))
      sendto_one(client_p, "PASS %s TS 5",
                 aconf->spasswd);
    send_capabilities(client_p);
    sendto_one(client_p, "SERVER %s 1 :%s%s",
               my_name_for_link(conf), ConfigServerHide.hidden ? "(H) " : "",
               me.info[0] ? me.info : "No description");
    send_queued_write(client_p);
  }
  if (!set_sock_buffers(client_p->localClient->fd, READBUF_SIZE))
    report_error(L_ALL, SETBUF_ERROR_MSG, get_client_name(client_p, SHOW_IP), errno);

  sendto_one(client_p, "SVINFO 5 3 0 :%lu", (unsigned long)CurrentTime);

  detach_conf(client_p, OPER_TYPE);
  client_p->servptr = &me;

  if (IsClosing(client_p))
    return(CLIENT_EXITED);

  SetServer(client_p);

  dlinkAdd(client_p, &client_p->lnode, &me.serv->servers);
  m = dlinkFind(&unknown_list, client_p);

  if (m != NULL)
  {
    dlinkDelete(m, &unknown_list);
    dlinkAdd(client_p, m, &serv_list);
  }
  else
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN, "Tried to re-register (%s) server", host);
    exit_client(client_p, client_p, client_p,
                "Tried to re-register server");
  }
  Count.myserver++;
  dlinkAdd(client_p, make_dlink_node(), &global_serv_list);
  hash_add_client(client_p);
  make_server(client_p);
  client_p->firsttime = CurrentTime;
  sendto_realops_flags(UMODE_ALL, L_ADMIN, "LINK->%s (%s) %s", inpath_ip, client_p->info,
		       show_capabilities(client_p));
  sendto_realops_flags(UMODE_ALL, L_OPER, "LINK->%s (%s) %s",
		       inpath, client_p->info, show_capabilities(client_p));
  ilog(NOTICE, "LINK->%s (%s) %s", inpath_ip, client_p->info,
       show_capabilities(client_p));
  client_p->serv->sconf = conf;

  DLINK_FOREACH(ptr, serv_list.head)
  {
    target_p = ptr->data;

    if (target_p == client_p)
      continue;

    if ((conf = target_p->serv->sconf) && match(my_name_for_link(conf), client_p->name))
      continue;

    sendto_one(target_p,":%s SERVER %s 2 :%s%s",
               me.name, client_p->name, IsHidden(client_p) ? "(H) " : "",
               client_p->info);
  }

  /* Pass on my client information to the new server
   * First, pass only servers (idea is that if the link gets
   * cancelled beacause the server was already there,
   * there are no NICK's to be cancelled...). Of course,
   * if cancellation occurs, all this info is sent anyway, 
   * and I guess the link dies when a read is attempted...? --msa
   *
   * Note: Link cancellation to occur at this point means
   * that at least two servers from my fragment are building
   * up connection this other fragment at the same time, it's
   * a race condition, not the normal way of operation...
   *
   * ALSO NOTE: using the get_client_name for server names--
   *    see previous *WARNING*!!! (Also, original inpath
   *    is destroyed...)
   */
  conf = client_p->serv->sconf;

  DLINK_FOREACH_PREV(ptr, global_serv_list.tail)
  {
    target_p = ptr->data;

    if (target_p->from == client_p)
      continue;

    if (match(my_name_for_link(conf), target_p->name))
      continue;

    if (IsJuped(target_p))
      sendto_one(client_p, ":%s SERVER %s 0 :%s",
                 target_p->servptr->name, target_p->name, target_p->info);
    else
      sendto_one(client_p, ":%s SERVER %s %d :%s%s",
                 target_p->servptr->name, target_p->name, target_p->hopcount+1,
                 IsHidden(target_p) ? "(H) " : "", target_p->info);
  }
  current_serial++;

  DLINK_FOREACH(ptr, global_channel_list.head)
  {
    chptr = ptr->data;

    if (dlink_list_length(&chptr->members) > 0)
    {
      DLINK_FOREACH(mptr, chptr->members.head)
      {
        ms = mptr->data;
        target_p = ms->client_p;

        if (target_p->serial != current_serial)
        {
          target_p->serial = current_serial;

          if (target_p->from != client_p)
            sendnick_TS(client_p, target_p);
        }
      }
      send_channel_modes(client_p, chptr);

      if (chptr->topic != NULL)
        send_tburst(client_p, chptr);
    }
  }

  /* also send out those that are not on any channel */
  DLINK_FOREACH(ptr, global_client_list.head)
  {
    target_p = ptr->data;

    if (target_p->serial != current_serial)
    {
      target_p->serial = current_serial;

      if (target_p->from != client_p)
        sendnick_TS(client_p, target_p);
    }
  }
  sendto_one(client_p, "PING :%s", me.name);
  return(0);
}

/* inputs       - struct Client pointer to oper requesting change
 * side effects - set autoconnect mode
 */
void
set_autoconn(struct Client *source_p, const char *name, int newval)
{
  struct ConfItem *conf;
  struct AccessItem *aconf;

  if (name != NULL)
  {
    conf = find_exact_name_conf(SERVER_TYPE, name, NULL, NULL);
    if (conf != NULL)
    {
      aconf = (struct AccessItem *)map_to_conf(conf);
      if (newval)
        SetConfAllowAutoConn(aconf);
      else
        ClearConfAllowAutoConn(aconf);

      sendto_realops_flags(UMODE_ALL, L_ALL, "%s changed AUTOCONN for %s to %i",
                           source_p->name, name, newval);
      sendto_one(source_p, ":%s NOTICE %s :AUTOCONN: %s currently %i",
                 me.name, source_p->name, name, newval);
    }
    else
    {
      sendto_one(source_p, ":%s NOTICE %s :AUTOCONN: %s not found",
                 me.name, source_p->name, name);
    }
  }
  else
    sendto_one(source_p, ":%s NOTICE %s :AUTOCONN: No servername",
               me.name, source_p->name);
}

/* initiate a server connection
 * inputs	- pointer to conf 
 *		- pointer to client doing the connet
 * This code initiates a connection to a server. It first checks to make
 * sure the given server exists. If this is the case, it creates a socket,
 * creates a client, saves the socket information in the client, and
 * initiates a connection to the server through comm_connect_tcp(). The
 * completion of this goes through serv_completed_connection().
 * We return 1 if the connection is attempted, since we don't know whether
 * it suceeded or not, and 0 if it fails in here somewhere.
 */
int
serv_connect(struct AccessItem *aconf, struct Client *by)
{
  struct ConfItem *conf;
  struct Client *client_p;
  int fd;
  char buf[HOSTIPLEN];

  /* conversion structs */
  struct sockaddr_in *v4;

  if(aconf == NULL)
    return (0);

  conf = unmap_conf_item(aconf);

  irc_getnameinfo((struct sockaddr*)&aconf->ipnum, aconf->ipnum.ss_len,
                  buf, HOSTIPLEN, NULL, 0, NI_NUMERICHOST);
  ilog(NOTICE, "CONNECT: %s[%s]@%s", aconf->user, aconf->host, buf);

  /* Still processing a DNS lookup? -> exit */
  if (aconf->dns_query != NULL)
  {
    sendto_realops_flags(UMODE_ALL, L_OPER, "DNS FAILED: %s", conf->name);
    return (0);
  }

  /* Make sure this server isn't already connected
   * Note: aconf should ALWAYS be a valid C: line
   */
  if ((client_p = find_server(conf->name)) != NULL)
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
                         "LINK: %s->%s already exists",
                         conf->name, get_client_name(client_p, SHOW_IP));
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "LINK: %s->%s already exists",
                         conf->name, get_client_name(client_p, MASK_IP));
    if (by && IsPerson(by) && !MyClient(by))
      sendto_one(by, ":%s NOTICE %s :LINK: %s->%s already exists",
                 me.name, by->name, conf->name,
                 get_client_name(client_p, MASK_IP));
    return (0);
  }

  /* create a socket for the server connection */
  if ((fd = comm_open(aconf->ipnum.ss.ss_family, SOCK_STREAM, 0, NULL)) < 0)
  {
    report_error(L_ALL, "opening stream socket to %s: %s", conf->name, errno);
    return (0);
  }

  /* Create a local client */
  client_p = make_client(NULL);

  /* Copy in the server, hostname, fd */
  strlcpy(client_p->name, conf->name, sizeof(client_p->name));
  strlcpy(client_p->host, aconf->host, sizeof(client_p->host));
  /* We already converted the ip once, so lets use it - stu */
  strlcpy(client_p->localClient->sockhost, buf, HOSTIPLEN);
  client_p->localClient->fd = fd;

  /* Set up the initial server evilness, ripped straight from
   * connect_server(), so don't blame me for it being evil.
   *   -- adrian
   */
  if (!set_non_blocking(client_p->localClient->fd))
  {
    report_error(L_ADMIN, NONB_ERROR_MSG, get_client_name(client_p, SHOW_IP), errno);
    report_error(L_OPER, NONB_ERROR_MSG, get_client_name(client_p, MASK_IP), errno);
  }

  if (!set_sock_buffers(client_p->localClient->fd, READBUF_SIZE))
  {
    report_error(L_ADMIN, SETBUF_ERROR_MSG, get_client_name(client_p, SHOW_IP), errno);
    report_error(L_OPER, SETBUF_ERROR_MSG, get_client_name(client_p, MASK_IP), errno);
  }

  /* Attach config entries to client here rather than in
   * serv_connect_callback(). This to avoid null pointer references.
   */
  if (!attach_connect_block(client_p, conf->name, aconf->host))
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "CONNECT: %s failed",
                         conf->name);
    if (by && IsPerson(by) && !MyClient(by))
      sendto_one(by, ":%s NOTICE %s :*** CONNECT: %s failed",
                 me.name, by->name, client_p->name);
    detach_all_confs(client_p);
    return (0);
  }
  /* at this point we have a connection in progress and C/N lines
   * attached to the client, the socket info should be saved in the
   * client and it should either be resolved or have a valid address.
   * The socket has been connected or connect is in progress.
   */
  make_server(client_p);

  if (by && IsPerson(by))
    strlcpy(client_p->serv->by, by->name, sizeof(client_p->serv->by));
  else
    strlcpy(client_p->serv->by, "AutoConn", sizeof(client_p->serv->by));

  SetConnecting(client_p);
  dlinkAdd(client_p, &client_p->node, &global_client_list);
  /* from def_fam */
  client_p->localClient->aftype = aconf->aftype;

  /* Now, initiate the connection */
  switch(aconf->aftype)
  {
    case AF_INET:
      v4 = (struct sockaddr_in*)&aconf->my_ipnum;
      if(v4->sin_addr.s_addr)
      {
        struct irc_ssaddr ipn;
        memset(&ipn, 0, sizeof(struct irc_ssaddr));
        ipn.ss.ss_family = AF_INET;
        ipn.ss_port = 0;
        memcpy(&ipn, &aconf->my_ipnum, sizeof(struct irc_ssaddr));
        comm_connect_tcp(client_p->localClient->fd, aconf->host, aconf->port,
                         (struct sockaddr *)&ipn, ipn.ss_len,
                         serv_connect_callback, client_p, aconf->aftype, CONNECTTIMEOUT);
      }
      else if(ServerInfo.specific_ipv4_vhost)
      {
        struct irc_ssaddr ipn;
        memset(&ipn, 0, sizeof(struct irc_ssaddr));
        ipn.ss.ss_family = AF_INET;
        ipn.ss_port = 0;
        memcpy(&ipn, &ServerInfo.ip, sizeof(struct irc_ssaddr));
        comm_connect_tcp(client_p->localClient->fd, aconf->host, aconf->port,
                         (struct sockaddr *)&ipn, ipn.ss_len,
                         serv_connect_callback, client_p, aconf->aftype, CONNECTTIMEOUT);
      }
      else
        comm_connect_tcp(client_p->localClient->fd, aconf->host, aconf->port,
                         NULL, 0, serv_connect_callback, client_p, aconf->aftype,
                         CONNECTTIMEOUT);
      break;
#ifdef IPV6
    case AF_INET6:
      if(ServerInfo.specific_ipv6_vhost)
      {
        struct irc_ssaddr ipn;
        memset(&ipn, 0, sizeof(struct irc_ssaddr));
        ipn.ss.ss_family = AF_INET6;
        ipn.ss_port = 0;
        memcpy(&ipn, &ServerInfo.ip6, sizeof(struct irc_ssaddr));
        comm_connect_tcp(client_p->localClient->fd, aconf->host, aconf->port,
                         (struct sockaddr *)&ipn, ipn.ss_len,
        serv_connect_callback, client_p, aconf->aftype, CONNECTTIMEOUT);
      }
      else
        comm_connect_tcp(client_p->localClient->fd, aconf->host, aconf->port,
                         NULL, 0, serv_connect_callback, client_p, aconf->aftype,
                         CONNECTTIMEOUT);
#endif
  }
  return (1);
}

/* complete a server connection.
 * This routine is called after the server connection attempt has
 * completed. If unsucessful, an error is sent to ops and the client
 * is closed. If sucessful, it goes through the initialisation/check
 * procedures, the capabilities are sent, and the socket is then
 * marked for reading.
 */
static void
serv_connect_callback(int fd, int status, void *data)
{
  struct Client *client_p = data;
  struct ConfItem *conf=NULL;
  struct AccessItem *aconf=NULL;

  set_no_delay(fd);

  /* Next, for backward purposes, record the ip of the server */
  memcpy(&client_p->localClient->ip, &fd_table[fd].connect.hostaddr,
         sizeof(struct irc_ssaddr));
  /* Check the status */
  if (status != COMM_OK)
  {
    /* We have an error, so report it and quit
     * Admins get to see any IP, mere opers don't *sigh*
     */
    if (ConfigServerHide.hide_server_ips)
      sendto_realops_flags(UMODE_ALL, L_ADMIN,
                           "%s: %s", comm_errstr(status), client_p->name);
    else
      sendto_realops_flags(UMODE_ALL, L_ADMIN,
                           "%s: %s[%s]", comm_errstr(status), client_p->name,
                           client_p->host);

    sendto_realops_flags(UMODE_ALL, L_OPER,
		         "%s: %s", comm_errstr(status), client_p->name);

    /* If a fd goes bad, call dead_link() the socket is no
     * longer valid for reading or writing.
     */
    dead_link_on_write(client_p);
    return;
  }

  /* COMM_OK, so continue the connection procedure
   * Get the C/N lines */
  conf = find_conf_name(&client_p->localClient->confs,
 		        client_p->name, SERVER_TYPE);
  if (conf == NULL)
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
		         "CONNECT{} LOST: %s", get_client_name(client_p, HIDE_IP));
    sendto_realops_flags(UMODE_ALL, L_OPER,
			 "CONNECT{} LOST %s", get_client_name(client_p, MASK_IP));

    exit_client(client_p, client_p, &me, "CONNECT{} Lost");
    return;
  }
  aconf = (struct AccessItem *)map_to_conf(conf);
  /* Next, send the initial handshake */
  SetHandshake(client_p);

  if (!EmptyString(aconf->spasswd))
    sendto_one(client_p, "PASS %s TS", aconf->spasswd);

  send_capabilities(client_p);
  sendto_one(client_p, "SERVER %s 1 :%s%s",
             my_name_for_link(conf),
             ConfigServerHide.hidden ? "(H) " : "",
             me.info);

  /* If we've been marked dead because a send failed, just exit
   * here now and save everyone the trouble of us ever existing.
   */
  if (IsDead(client_p)) 
  {
    sendto_realops_flags(UMODE_ALL, L_ADMIN,
			 "CONNECT DROPPED: %s[%s]",
                         client_p->name, client_p->host);
    sendto_realops_flags(UMODE_ALL, L_OPER,
                         "CONNECT DROPPED: %s",
                         client_p->name);
    return;
  }
  comm_setselect(fd, FDLIST_SERVER, COMM_SELECT_READ, read_packet,
                 client_p, 0);
}
