/* MusIRCd: an advanced Internet Relay Chat Daemon(ircd).
 * string.c: IRC string functions.
 * Copyright (C) 2004 by MusIRCd Development.
 * $Id: string.c,v 1.4 2004/02/08 06:46:05 musirc Exp $
 */

#include "stdinc.h"
#include "istring.h"
#include "sprintf.h"

#ifndef INADDRSZ 
#define INADDRSZ 4
#endif

#ifndef IN6ADDRSZ
#define IN6ADDRSZ 16
#endif

#ifndef INT16SZ
#define INT16SZ 2
#endif

/* This is like standard ctime()-function, but it zaps away
 *   the newline from the end of that string. Also, it takes
 *   the time value as parameter, instead of pointer to it.
 *   Note that it is necessary to copy the string to alternate
 *   buffer (who knows how ctime() implements it, maybe it statically
 *   has newline there and never 'refreshes' it -- zapping that
 *   might break things in other places...)
 * Thu Nov 24 18:22:48 1986 
 */
const char *
myctime(time_t value)
{
  static char buf[32];
  char *p;

  strcpy(buf, ctime(&value));
  if ((p = strchr(buf, '\n')) != NULL)
    *p = '\0';
  return(buf);
}

/* clean up a string possibly containing garbage */
char * 
clean_string(char* dest, const unsigned char* src, size_t len)
{
  char *d = dest; 

  if (dest == NULL || src == NULL)
    return NULL;
    
  len -= 3;  /* allow for worst case, '^A\0' */

  while (*src && (len > 0))
    {
      if(*src & 0x80)             /* if high bit is set */
        {
          *d++ = '.';
          --len;
        }
      else if(!IsPrint(*src))       /* if NOT printable */
        {
          *d++ = '^';
          --len;
          *d++ = 0x40 + *src;   /* turn it into a printable */
        }
      else
        *d++ = *src;
      ++src;
      --len;
    }
  *d = '\0';
  return dest;
}

/* Copies src to dst, while converting all \t (tabs) into spaces. */
char *
strip_tabs(char *dest, const unsigned char *src, size_t len)
{
  char *d = dest;
  /* Sanity check; we don't want anything nasty... */

  if (dest == NULL || src == NULL)
    return(NULL);
    
  while (*src && (len > 0))
  {
    if (*src == '\t')
      *d++ = ' '; /* Translate the tab into a space */
    else
      *d++ = *src; /* Copy src to dst */

    ++src;
    --len;
  }
  *d = '\0'; /* Null terminate, thanks and goodbye */
  return(dest);
}

/* walk through a string of tokens, using a set of separators */
#ifndef HAVE_STRTOK_R
char *
strtoken(char** save, char* str, const char* fs)
{
  char *pos = *save, *tmp;

  if (str)
    pos = str;    /* new string scan */

  while (pos && *pos && strchr(fs, *pos) != NULL)
    ++pos;        /* skip leading separators */

  if (!pos || !*pos)
    return (pos = *save = NULL);   /* string contains only sep's */

  tmp = pos;       /* now, keep position of the token */

  while (*pos && strchr(fs, *pos) == NULL)
    ++pos;       /* skip content of the token */

  if (*pos)
    *pos++ = '\0';    /* remove first sep after the token */
  else
    pos = NULL;    /* end of string */

  *save = pos;
  return tmp;
}
#endif /* !HAVE_STRTOK_R */

static const char *IpQuadTab[] =
{
    "0",   "1",   "2",   "3",   "4",   "5",   "6",   "7",   "8",   "9",
   "10",  "11",  "12",  "13",  "14",  "15",  "16",  "17",  "18",  "19",
   "20",  "21",  "22",  "23",  "24",  "25",  "26",  "27",  "28",  "29",
   "30",  "31",  "32",  "33",  "34",  "35",  "36",  "37",  "38",  "39",
   "40",  "41",  "42",  "43",  "44",  "45",  "46",  "47",  "48",  "49",
   "50",  "51",  "52",  "53",  "54",  "55",  "56",  "57",  "58",  "59",
   "60",  "61",  "62",  "63",  "64",  "65",  "66",  "67",  "68",  "69",
   "70",  "71",  "72",  "73",  "74",  "75",  "76",  "77",  "78",  "79",
   "80",  "81",  "82",  "83",  "84",  "85",  "86",  "87",  "88",  "89",
   "90",  "91",  "92",  "93",  "94",  "95",  "96",  "97",  "98",  "99",
  "100", "101", "102", "103", "104", "105", "106", "107", "108", "109",
  "110", "111", "112", "113", "114", "115", "116", "117", "118", "119",
  "120", "121", "122", "123", "124", "125", "126", "127", "128", "129",
  "130", "131", "132", "133", "134", "135", "136", "137", "138", "139",
  "140", "141", "142", "143", "144", "145", "146", "147", "148", "149",
  "150", "151", "152", "153", "154", "155", "156", "157", "158", "159",
  "160", "161", "162", "163", "164", "165", "166", "167", "168", "169",
  "170", "171", "172", "173", "174", "175", "176", "177", "178", "179",
  "180", "181", "182", "183", "184", "185", "186", "187", "188", "189",
  "190", "191", "192", "193", "194", "195", "196", "197", "198", "199",
  "200", "201", "202", "203", "204", "205", "206", "207", "208", "209",
  "210", "211", "212", "213", "214", "215", "216", "217", "218", "219",
  "220", "221", "222", "223", "224", "225", "226", "227", "228", "229",
  "230", "231", "232", "233", "234", "235", "236", "237", "238", "239",
  "240", "241", "242", "243", "244", "245", "246", "247", "248", "249",
  "250", "251", "252", "253", "254", "255"
};

/* in_addr to string
 *      changed name to remove collision possibility and
 *      so behaviour is guaranteed to take a pointer arg.
 *  returned the dotted notation of a given
 *      internet number
 */
const char *
inetntoa(const char *in)
{
  static char buf[16];
  char *bufptr = buf;
  const unsigned char *a = (const unsigned char *)in;
  const char *n;

  n = IpQuadTab[ *a++ ];
  while (*n)
    *bufptr++ = *n++;
  *bufptr++ = '.';
  n = IpQuadTab[ *a++ ];
  while ( *n )
    *bufptr++ = *n++;
  *bufptr++ = '.';
  n = IpQuadTab[ *a++ ];
  while ( *n )
    *bufptr++ = *n++;
  *bufptr++ = '.';
  n = IpQuadTab[ *a ];
  while ( *n )
    *bufptr++ = *n++;
  *bufptr = '\0';
  return(buf);
}

#ifndef HAVE_BASENAME
/* input        - i.e. "/usr/local/ircd/modules/m_whois.so"
 * output       - i.e. "m_whois.so"
 * side effects - this will be overwritten on subsequent calls
 */
char *
basename(char *path)
{
  char *s;

  if ((s = strrchr(path, '/')) == NULL)
    s = path;
  else
    s++;

  return (s);
}

#endif

/* Copyright (c) 1996-1999 by Internet Software Consortium. */

#define SPRINTF(x) ((size_t)ircsprintf x)

/* Don't even consider trying to compile this on a system where
 * sizeof(int) < 4.  sizeof(int) > 4 is fine; all the world's not a VAX.
 */
static const char *inet_ntop4(const unsigned char *src, char *dst, unsigned int size);
#ifdef IPV6
static const char *inet_ntop6(const unsigned char *src, char *dst, unsigned int size);
#endif

/*      format an IPv4 address
 * return:
 *      `dst' (as a const)
 * notes:
 *      (1) uses no statics
 *      (2) takes a unsigned char* not an in_addr as input
 */
static const char *
inet_ntop4(const unsigned char *src, char *dst, unsigned int size)
{
  if (size < 16)
    return(NULL);

  return(strcpy(dst, inetntoa((const char *)src)));
}

/* convert IPv6 binary address into presentation (printable) format */
#ifdef IPV6
static const char *
inet_ntop6(const unsigned char *src, char *dst, unsigned int size)
{
        /* Note that int32_t and int16_t need only be "at least" large enough
         * to contain a value of the specified size.  On some systems, like
         * Crays, there is no such thing as an integer variable with 16 bits.
         * Keep this in mind if you think this function should have been coded
         * to use pointer overlays.  All the world's not a VAX.
         */
        char tmp[sizeof "ffff:ffff:ffff:ffff:ffff:ffff:255.255.255.255"], *tp;
        struct { int base, len; } best, cur;
        unsigned int words[IN6ADDRSZ / INT16SZ];
        int i;

        /* Preprocess:
         *      Copy the input (bytewise) array into a wordwise array.
         *      Find the longest run of 0x00's in src[] for :: shorthanding.
         */
        memset(words, '\0', sizeof words);
        for (i = 0; i < IN6ADDRSZ; i += 2)
                words[i / 2] = (src[i] << 8) | src[i + 1];
        best.base = -1;
        cur.base = -1;
        for (i = 0; i < (IN6ADDRSZ / INT16SZ); i++) {
                if (words[i] == 0) {
                        if (cur.base == -1)
                                cur.base = i, cur.len = 1;
                        else
                                cur.len++;
                } else {
                        if (cur.base != -1) {
                                if (best.base == -1 || cur.len > best.len)
                                        best = cur;
                                cur.base = -1;
                        }
                }
        }
        if (cur.base != -1) {
                if (best.base == -1 || cur.len > best.len)
                        best = cur;
        }
        if (best.base != -1 && best.len < 2)
                best.base = -1;

        /*
         * Format the result.
         */
        tp = tmp;
        for (i = 0; i < (IN6ADDRSZ / INT16SZ); i++) {
                /* Are we inside the best run of 0x00's? */
                if (best.base != -1 && i >= best.base &&
                    i < (best.base + best.len)) {
                        if (i == best.base)
                                *tp++ = ':';
                        continue;
                }
                /* Are we following an initial run of 0x00s or any real hex? */
                if (i != 0)
                        *tp++ = ':';
                /* Is this address an encapsulated IPv4? */
                if (i == 6 && best.base == 0 &&
                    (best.len == 6 || (best.len == 5 && words[5] == 0xffff))) {
                        if (!inet_ntop4(src+12, tp, sizeof tmp - (tp - tmp)))
                                return (NULL);
                        tp += strlen(tp);
                        break;
                }
                tp += SPRINTF((tp, "%x", words[i]));
        }
        /* Was it a trailing run of 0x00's? */
        if (best.base != -1 && (best.base + best.len) ==
            (IN6ADDRSZ / INT16SZ))
                *tp++ = ':';
        *tp++ = '\0';

        if ((unsigned int)(tp - tmp) > size) {
                return (NULL);
        }
        return strcpy(dst, tmp);
}
#endif

/* convert a network format address to presentation format.
 * return:
 *      pointer to presentation format address (`dst'), or NULL (see errno).
 */
const char *
inetntop(int af, const void *src, char *dst, unsigned int size)
{
  switch (af)
  {
    case AF_INET:
      return(inet_ntop4(src, dst, size));
#ifdef IPV6
    case AF_INET6:
      if (IN6_IS_ADDR_V4MAPPED((const struct in6_addr *)src) ||
          IN6_IS_ADDR_V4COMPAT((const struct in6_addr *)src))
        return(inet_ntop4((unsigned char *)&((const struct in6_addr *)src)->s6_addr[12], dst, size));
      else
        return(inet_ntop6(src, dst, size));
#endif
    default:
      return(NULL);
  }
}

#ifndef HAVE_STRLCAT
size_t
strlcat(char *dst, const char *src, size_t siz)
{
  char *d = dst;
  const char *s = src;
  size_t n = siz, dlen;

  while (n-- != 0 && *d != '\0')
    d++;

  dlen = d - dst;
  n    = siz - dlen;

  if (n == 0)
    return(dlen + strlen(s));

  while (*s != '\0')
  {
    if (n != 1)
    {
      *d++ = *s;
      n--;
    }

    s++;
  }

  *d = '\0';
  return(dlen + (s - src)); /* count does not include NUL */
}
#endif

#ifndef HAVE_STRLCPY
size_t
strlcpy(char *dst, const char *src, size_t siz)
{
  char *d = dst;
  const char *s = src;
  size_t n = siz;

  /* Copy as many bytes as will fit */
  if (n != 0 && --n != 0)
  {
    do
    {
      if ((*d++ = *s++) == 0)
        break;
    } while (--n != 0);
  }

  /* Not enough room in dst, add NUL and traverse rest of src */
  if (n == 0)
  {
    if (siz != 0)
      *d = '\0'; /* NUL-terminate dst */
    while (*s++)
      ;
  }

  return(s - src - 1); /* count does not include NUL */
}
#endif

#ifndef HAVE_STRCASESTR
char *
strcasestr(const char* s, const char* find)
{
  char c, sc;
  size_t len;

  if ((c = *find++) != 0)
  {
    c = tolower((unsigned char)c);
    len = strlen(find);
    do
    {
      do
      {
        if ((sc = *s++) == 0)
          return (NULL);
      }
      while ((char)tolower((unsigned char)sc) != c);
    }
    while (strncasecmp(s, find, len) != 0);
      s--;
  }
  return ((char *)s);
}
#endif
