/* tools.h: Header for the various tool functions.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: tools.h,v 1.13 2005/01/27 11:48:01 musirc Exp $
 */

#ifndef __TOOLS_H__
#define __TOOLS_H__
#include "stdinc.h"

/* double-linked-list and single-linked-list stuff */
typedef struct _dlink_node dlink_node;
typedef struct _dlink_list dlink_list;
typedef struct _slink_node slink_node;
typedef struct _slink_list slink_list;

struct _dlink_node
{
  void *data;
  dlink_node *prev;
  dlink_node *next;
};
  
struct _dlink_list
{
  dlink_node *head;
  dlink_node *tail;
  unsigned long length;
};

struct _slink_node
{
  void *data;
  slink_node *next;
};

struct _slink_list
{
  slink_node *head;
  unsigned long length;
};

extern void dlinkAdd(void *, dlink_node *, dlink_list *);
extern void dlinkAddBefore(dlink_node *, void *, dlink_node *, dlink_list *);
extern void dlinkAddTail(void *, dlink_node *, dlink_list *);
extern void dlinkDelete(dlink_node *, dlink_list *);
extern void dlinkMoveList(dlink_list *, dlink_list *);
extern dlink_node *dlinkFind(dlink_list *, void *);
extern dlink_node *dlinkFindDelete(dlink_list *, void *);

extern void slink_add(void *, slink_node *, slink_list *);
extern void slink_delete(slink_node *, slink_list *);
extern slink_node *slink_find(slink_list *, void *);

#define DLINK_FOREACH(pos, head) for (pos = (head); pos != NULL; pos = pos->next)
   		
/* Walks forward of a list safely while removing nodes 
 * pos is your node
 * n is another list head for temporary storage
 * head is your list head
 */
#define DLINK_FOREACH_SAFE(pos, n, head) for (pos = (head), n = pos ? pos->next : NULL; pos != NULL; pos = n, n = pos ? pos->next : NULL)
	        
#define DLINK_FOREACH_PREV(pos, head) for (pos = (head); pos != NULL; pos = pos->prev)
              		       
#define SLINK_FOREACH(pos, head) for (pos = (head); pos != NULL; pos = pos->next)
#define SLINK_FOREACH_SAFE(pos, n, head) for (pos = (head), n = pos ? pos->next : NULL; pos != NULL; pos = n, n = pos ? pos->next : NULL)

/* Returns the list length */
#define dlink_list_length(list) (list)->length
#define slink_list_length(list) (list)->length

#ifdef __GNUC__

/* dlink_ routines are stolen from squid, except for dlinkAddBefore,
 * which is mine.
 *   -- adrian
 */
extern inline void
dlinkAdd(void *data, dlink_node * m, dlink_list * list)
{
 m->data = data;
 m->prev = NULL;
 m->next = list->head;
 /* Assumption: If list->tail != NULL, list->head != NULL */
 if (list->head != NULL)
   list->head->prev = m;
 else if (list->tail == NULL)
   list->tail = m;
 list->head = m;
 list->length++;
}

extern inline void
dlinkAddBefore(dlink_node *b, void *data, dlink_node *m, dlink_list *list)
{
    /* Shortcut - if its the first one, call dlinkAdd only */
    if (b == list->head)
        dlinkAdd(data, m, list);
    else {
        m->data = data;
        b->prev->next = m;
        m->prev = b->prev;
        b->prev = m; 
        m->next = b;
	list->length++;
    }
}

extern inline void
dlinkAddTail(void *data, dlink_node *m, dlink_list *list)
{
 m->data = data;
 m->next = NULL;
 m->prev = list->tail;
 /* Assumption: If list->tail != NULL, list->head != NULL */
 if (list->tail != NULL)
   list->tail->next = m;
 else if (list->head == NULL)
   list->head = m;
 list->tail = m;
 list->length++;
}

/* Execution profiles show that this function is called the most
 * often of all non-spontaneous functions. So it had better be
 * efficient. */
extern inline void
dlinkDelete(dlink_node *m, dlink_list *list)
{
 /* Assumption: If m->next == NULL, then list->tail == m
  *      and:   If m->prev == NULL, then list->head == m
  */
 if (m->next)
   m->next->prev = m->prev;
 else
   list->tail = m->prev;
 if (m->prev)
   m->prev->next = m->next;
 else
   list->head = m->next;

 /* Set this to NULL does matter */
  m->next = m->prev = NULL;
  list->length--;
}

/* dlinkFind()
 * inputs	- list to search 
 *		- data
 * output	- pointer to link or NULL if not found
 * side effects	- Look for ptr in the linked listed pointed to by link.
 */
extern inline dlink_node *
dlinkFind(dlink_list *list, void *data)
{
  dlink_node *ptr;

  DLINK_FOREACH(ptr, list->head)
  {
    if (ptr->data == data)
      return(ptr);
  }
  return(NULL);
}

extern inline void
dlinkMoveList(dlink_list *from, dlink_list *to)
{
  /* There are three cases */
  /* case one, nothing in from list */

    if (from->head == NULL)
      return;

  /* case two, nothing in to list
   * actually if to->head is NULL and to->tail isn't, thats a bug */
    if (to->head == NULL)
    {
       to->head = from->head;
       to->tail = from->tail;
       from->head = from->tail = NULL;
       to->length = from->length;
       from->length = 0;
       return;
    }

  /* third case play with the links */
    from->tail->next = to->head;
    from->head->prev = to->head->prev;
    to->head->prev = from->tail;
    to->head = from->head;
    from->head = from->tail = NULL;
    to->length += from->length;
    from->length = 0;
}

extern inline dlink_node *
dlinkFindDelete(dlink_list *list, void *data)
{
  dlink_node *m;
 
  DLINK_FOREACH(m, list->head)
  {
    if (m->data == data)
    {
      if (m->next)
        m->next->prev = m->prev;
      else
        list->tail = m->prev;
      if (m->prev)
        m->prev->next = m->next;
      else
        list->head = m->next;
      /* Set this to NULL does matter */
      m->next = m->prev = NULL;
      list->length--;

      return(m);
    }
  }
  return(NULL);
}

extern inline void
slink_add(void *data, slink_node *node, slink_list *list)
{
    if(list->head == NULL)
    {
        list->head = node;
        node->next = NULL;
    }
    else
    {
        node->next = list->head->next;
        list->head = node;
    }

    node->data = data;
    list->length++;
}

extern inline void
slink_delete(slink_node *node, slink_list *list)
{
    slink_node *ptr;

    if(list->head == NULL)
        return;

    if(list->head->next == NULL)
        list->head = NULL;
    else
    {
        SLINK_FOREACH(ptr, list->head)
        {
            if(ptr->next == node)
            {
                ptr->next = node->next;
                break;
            }
        }
    }
    list->length--;
}

extern inline slink_node*
slink_find(slink_list *list, void *data)
{
    slink_node *ptr;

    SLINK_FOREACH(ptr, list->head)
    {
        if(ptr->data == data)
            return ptr;
    }
    return NULL;
}
#endif /* __GNUC__ */

#endif
