/* m_flags.c: Implements mode flags.
 * Copyright 2005 by MusIRCd Development.
 * $Id: m_flags.c,v 1.37 2005/01/27 11:48:01 musirc Exp $
 */

#include "handlers.h"
#include "ircd.h"
#include "istring.h"
#include "sprintf.h"
#include "numeric.h"
#include "config.h"
#include "server.h"
#include "send.h"
#include "modules.h"
#include "user.h"

static void m_flags(struct Client *, struct Client *, int, char **);
static void mo_flags(struct Client *, struct Client *, int, char **);
static char *set_flags_to_string(struct Client *);
static char *unset_flags_to_string(struct Client *);

struct Message flags_msgtab = {
  "FLAGS", 0, MFLG_SLOW,
  {m_unregistered, m_flags, m_ignore, mo_flags}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&flags_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&flags_msgtab);
}

const char *_version = "$Revision: 1.37 $";
#endif

struct FlagTable
{
  const char *name;
  unsigned int mode;
  int oper;
};

static struct FlagTable flag_table[] =
{
  /* name		mode it represents	oper only? */
  { "CONNECT",          UMODE_CCONN,            1 },
  { "DEAF",		UMODE_DEAF,		0 },
  { "FLOOD",            UMODE_FLOOD,            1 },
  { "INVISIBLE",        UMODE_INVISIBLE,        0 },
  { "JUPES",		UMODE_JUPES,		1 },
  { "LOCOP",            UMODE_LOCOPS,           1 },
  { "NICKCHANGES",	UMODE_NCHANGE,		1 },
  { "REMOTE",           UMODE_EXTERNAL,         1 },
  { "SNOTICE",          UMODE_SERVNOTICE,       1 },
  { "SPY",	        UMODE_SPY,              1 },
  { "SSIGNORE",         UMODE_SSIGNORE,         0 },
  { "WALLOP",		UMODE_WALLOP,		1 },
  { "UNAUTH",		UMODE_UNAUTH,		1 },
  { NULL,		0,			0 }
};

#define USER_FLAGS (UMODE_WALLOP | UMODE_SERVNOTICE | UMODE_SSIGNORE |\
		    UMODE_INVISIBLE | UMODE_DEAF)
#define OPER_FLAGS (USER_FLAGS | UMODE_CCONN | UMODE_SPY |\
		    UMODE_FLOOD | UMODE_EXTERNAL | UMODE_UNAUTH |\
		    UMODE_LOCOPS | UMODE_NCHANGE | UMODE_JUPES)

/* parv[0] = sender prefix
 * parv[1] = parameter
 */
static void
m_flags(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  int i, j, isadd, isgood;
  unsigned int setflags;
  char *p, *flag;

  if (parc < 2)
  {
    /* Generate a list of what flags you have and what you are missing,
     * and send it to the user
     */
    sendto_one(source_p, ":%s NOTICE %s :FLAGS: Current:%s",
               me.name, source_p->name, set_flags_to_string(source_p));
    sendto_one(source_p, ":%s NOTICE %s :FLAGS: Missing:%s",
               me.name, source_p->name, unset_flags_to_string(source_p));
    return;
  }
  /* Preserve the current flags */
  setflags = source_p->umodes;

  for (i = 1; i < parc; i++)
  {
    for (flag = strtoken(&p, parv[i], " "); flag;
         flag = strtoken(&p, NULL, " "))
    {
      /* We default to being in ADD mode */
      isadd = 1;

      /* We default to being in BAD mode */
      isgood = 0;

      if (!isalpha(flag[0]))
      {
        if (flag[0] == '-')
          isadd = 0;
        else if (flag[0] == '+')
          isadd = 1;
        flag++;
      }

      /* support ALL here */
      if (!irccmp(flag, "ALL"))
      {
        if (isadd)
          source_p->umodes |= USER_FLAGS;
        else
          source_p->umodes &= ~USER_FLAGS;
        sendto_one(source_p, ":%s NOTICE %s :FLAGS: Current:%s",
                   me.name, source_p->name, set_flags_to_string(source_p));
        sendto_one(source_p, ":%s NOTICE %s :FLAGS: Missing:%s",
                   me.name, source_p->name, unset_flags_to_string(source_p));
        send_umode_out(client_p, source_p, setflags);
        return;
      }

      for (j = 0; flag_table[j].name; j++)
      {
        if (!flag_table[j].oper && !irccmp(flag, flag_table[j].name))
        {
          if (isadd)
            source_p->umodes |= flag_table[j].mode;
          else
            source_p->umodes &= ~ (flag_table[j].mode);
          isgood = 1;
          continue;
        }
      }
      /* This for ended without matching a valid FLAG, here is where
       * I want to operate differently than ircd-comstud, and just ignore
       * the invalid flag, send a warning and go on.
       */
      if (!isgood)
        sendto_one(source_p, ":%s NOTICE %s :FLAGS: %s invalid",
                   me.name, source_p->name, flag);
    }
  }

  /* All done setting the flags, print the notices out to the user
   * telling what flags they have and what flags they are missing
   */
  sendto_one(source_p, ":%s NOTICE %s :FLAGS: Current:%s",
             me.name, source_p->name, set_flags_to_string(source_p));
  sendto_one(source_p, ":%s NOTICE %s :FLAGS: Missing:%s",
             me.name, source_p->name, unset_flags_to_string(source_p));

  send_umode_out(client_p, source_p, setflags);
}

/* parv[0] = sender prefix
 * parv[1] = parameter
 */
static void
mo_flags(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{		 
  int i, j, isadd, isgood;
  unsigned int setflags;
  char *p, *flag;

  if (parc < 2)
  {
    /* Generate a list of what flags you have and what you are missing,
     * and send it to the user
     */
    sendto_one(source_p, ":%s NOTICE %s :FLAGS: Current:%s",
               me.name, source_p->name, set_flags_to_string(source_p));
    sendto_one(source_p, ":%s NOTICE %s :FLAGS: Missing:%s",
               me.name, source_p->name, unset_flags_to_string(source_p));
    return;
  }
  /* Preserve the current flags */
  setflags = source_p->umodes;

  for (i = 1; i < parc; i++)
  {
    for (flag = strtoken(&p, parv[i], " "); flag;
         flag = strtoken(&p, NULL, " "))
    {
      /* We default to being in ADD mode */
      isadd = 1;

      /* We default to being in BAD mode */
      isgood = 0;

      if (!isalpha(flag[0]))
      {
        if (flag[0] == '-')
          isadd = 0;
        else if (flag[0] == '+')
          isadd = 1;
        flag++;
      }

      /* support ALL here */
      if (!irccmp(flag, "ALL"))
      {
        if (isadd)
          source_p->umodes |= OPER_FLAGS;
        else
          source_p->umodes &= ~OPER_FLAGS;
        sendto_one(source_p, ":%s NOTICE %s :FLAGS: Current:%s",
                   me.name, source_p->name, set_flags_to_string(source_p));
        sendto_one(source_p, ":%s NOTICE %s :FLAGS: Missing:%s",
                   me.name, source_p->name, unset_flags_to_string(source_p));
        send_umode_out(client_p, source_p, setflags);
        return;
      }

      if (!irccmp(flag, "NICKCHANGES"))
      {
        if (!IsOperN(source_p))
        {
          sendto_one(source_p, ":%s NOTICE %s :FLAGS: You need nick_changes = yes;",
                     me.name, source_p->name);
          continue;
        }
        if (isadd)
          source_p->umodes |= UMODE_NCHANGE;
        else
          source_p->umodes &= ~UMODE_NCHANGE;
        isgood = 1;
        continue;
      }

      for (j = 0; flag_table[j].name; j++)
      {
        if (!irccmp(flag, flag_table[j].name))
        {
          if (isadd)
            source_p->umodes |= flag_table[j].mode;
          else
            source_p->umodes &= ~ (flag_table[j].mode);
          isgood = 1;
          continue;
        }
      }
      /* This for ended without matching a valid FLAG, here is where
       * I want to operate differently than ircd-comstud, and just ignore
       * the invalid flag, send a warning and go on.
       */
      if (!isgood)
        sendto_one(source_p, ":%s NOTICE %s :FLAGS: %s invalid",
                   me.name, source_p->name, flag);
    }
  }

  /* All done setting the flags, print the notices out to the user
   * telling what flags they have and what flags they are missing
   */
  sendto_one(source_p, ":%s NOTICE %s :FLAGS: Current:%s",
             me.name, source_p->name, set_flags_to_string(source_p));
  sendto_one(source_p, ":%s NOTICE %s :FLAGS: Missing:%s",
             me.name, source_p->name, unset_flags_to_string(source_p));

  send_umode_out(client_p, source_p, setflags);
}

static char
*set_flags_to_string(struct Client *client_p)
{
  static char setflags[BUFSIZE + 1];
  int i;

  /* Clear it to begin with, we'll be doing a lot of ircsprintf's */
  setflags[0] = '\0';

  /* Unlike unset_flags_to_string(), we don't have to care about oper
   * flags and not showing them
   */
  for (i = 0; flag_table[i].name; i++)
  {
    if (client_p->umodes & flag_table[i].mode)
      ircsprintf(setflags, "%s %s", setflags, flag_table[i].name);
  }

  return(setflags);
}

static char
*unset_flags_to_string(struct Client *client_p)
{
  static char setflags[BUFSIZE + 1];
  int i;

  /* Clear it to begin with, we'll be doing a lot of ircsprintf's */
  setflags[0] = '\0';

  for (i = 0; flag_table[i].name; i++)
  {
    if (!(client_p->umodes & flag_table[i].mode))
    {
      if (!IsOper(client_p) && flag_table[i].oper)
        continue;
      ircsprintf(setflags, "%s %s", setflags, flag_table[i].name);
    }
  }
  return(setflags);
}
