/* m_trace.c: Traces a path to a client/server.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: m_trace.c,v 1.69 2005/01/27 11:48:02 musirc Exp $
 */

#include "handlers.h"
#include "client.h"
#include "hash.h"
#include "istring.h"
#include "ircd.h"
#include "numeric.h"
#include "server.h"
#include "send.h"
#include "modules.h"
#include "config.h"
#include "getinfo.h"

static void m_trace(struct Client *, struct Client *, int, char **);
static void ms_trace(struct Client *, struct Client *, int, char **);
static void mo_trace(struct Client *, struct Client *, int, char **);
static void mo_ctrace(struct Client *, struct Client *, int, char **);
static void mo_classlist(struct Client *, struct Client *, int, char **);
static void do_actual_trace(const char *, struct Client *, int, char **);

struct Message trace_msgtab = {
  "TRACE", 0, MFLG_SLOW,
  {m_unregistered, m_trace, ms_trace, mo_trace}
};
struct Message ctrace_msgtab = {
  "CTRACE", 0, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_ctrace}
};
struct Message classlist_msgtab = {
  "CLASSLIST", 2, MFLG_SLOW,
  {m_unregistered, m_not_oper, m_ignore, mo_classlist}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&trace_msgtab);
  mod_add_cmd(&ctrace_msgtab);
  mod_add_cmd(&classlist_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&trace_msgtab);
  mod_del_cmd(&ctrace_msgtab);
  mod_del_cmd(&classlist_msgtab);
}

const char *_version = "$Revision: 1.69 $";
#endif
static int status(struct Client *, struct Client *, int, int, int);
static int cstatus(struct Client *, struct Client *);

/* parv[0] = sender prefix
 * parv[1] = target client/server to trace
 */
static void
m_trace(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  const char *tname;

  if (parc > 1)
    tname = parv[1];
  else
    tname = me.name;

  sendto_one(source_p, form_str(RPL_ENDOFTRACE),
	     me.name, source_p->name, tname);
  sendto_realops_flags(UMODE_SPY, L_ALL, "TRACE %s by %s (%s@%s) [%s]",
                       tname, source_p->name, source_p->username,
                       source_p->host, source_p->user->server->name);
}

/* parv[0] = sender prefix
 * parv[1] = servername
 */
static void
mo_trace(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  dlink_node *ptr;
  const char *tname;

  if (parc > 2)
    if (hunt_server(client_p, source_p, ":%s TRACE %s :%s", 2, parc, parv))
      return;

  if (parc > 1)
    tname = parv[1];
  else
    tname = me.name;

  switch (hunt_server(client_p, source_p, ":%s TRACE :%s", 1, parc, parv))
  {
    case HUNTED_PASS: /* note: gets here only if parv[1] exists */
    {
      struct Client *ac2ptr;

      if ((ac2ptr = find_client(tname)) == NULL)
      {
        DLINK_FOREACH(ptr, global_client_list.head)
        {
          ac2ptr = ptr->data;
          if (match(tname, ac2ptr->name) || match(ac2ptr->name, tname))
            break;
          else
            ac2ptr = NULL;
         }
      }
      if (ac2ptr != NULL)
        sendto_one(source_p, form_str(RPL_TRACELINK), me.name, source_p->name,
                   ircd_version, tname, ac2ptr->from->name);
      return;
    }
    case HUNTED_ISME:
      do_actual_trace(tname, source_p, parc, parv);
      break;
    default:
      return;
  }
  sendto_realops_flags(UMODE_SPY, L_ALL, "TRACE %s by %s (%s@%s) [%s]",
                       tname, source_p->name, source_p->username,
                       source_p->host, source_p->user->server->name);
}

static void
do_actual_trace(const char *tname, struct Client *source_p, int parc, char *parv[])
{
  struct Client *target_p = NULL;
  struct ConfItem *conf;
  struct ClassItem *cltmp;
  int doall = 0, link_s[HARD_FDLIMIT], link_u[HARD_FDLIMIT], cnt = 0, wilds, dow;
  dlink_node *gcptr, *ptr;

  if (!MyClient(source_p))
  {
    doall = 1;
    tname = me.name;
  }
  else if (match(tname, me.name))
    doall = 1;

  wilds = !parv[1] || strchr(tname, '*') || strchr(tname, '?');
  dow = wilds || doall;
  set_time();

  if (!IsOper(source_p) || !dow)
  {
    const char *name, *class_name;
    char ipaddr[HOSTIPLEN];

    target_p = find_client(tname);

    if (target_p && IsPerson(target_p))
    {
      name = get_client_name(target_p, HIDE_IP);
      irc_getnameinfo((struct sockaddr*)&target_p->localClient->ip,
		      target_p->localClient->ip.ss_len, ipaddr, HOSTIPLEN, NULL, 0,
		      NI_NUMERICHOST);

      class_name = get_client_class(target_p);

      if (IsOper(target_p))
        sendto_one(source_p, form_str(RPL_TRACEOPERATOR),
                   me.name, source_p->name, class_name, name,
                   IsIPSpoof(target_p) ? "masked" : ipaddr, target_p->info,
                   CurrentTime - target_p->lasttime,
                   (target_p->user) ? (CurrentTime - target_p->user->last) : 0);
      else
        sendto_one(source_p,form_str(RPL_TRACEUSER),
                   me.name, source_p->name, class_name, name,
                   IsIPSpoof(target_p) ? "masked" : ipaddr, target_p->info,
                   CurrentTime - target_p->lasttime,
                   (target_p->user)?(CurrentTime - target_p->user->last):0);
    }
    sendto_one(source_p, form_str(RPL_ENDOFTRACE),
	       me.name, source_p->name, tname);
    return;
  }
  memset((void *)link_s,0,sizeof(link_s));
  memset((void *)link_u,0,sizeof(link_u));

  /* Count up all the servers and clients in a downlink. */
  if (doall)
  {
    DLINK_FOREACH(gcptr, global_client_list.head)
    {
      target_p = gcptr->data;

      if (IsPerson(target_p))
        link_u[target_p->from->localClient->fd]++;
      else if (IsServer(target_p))
        link_s[target_p->from->localClient->fd]++;
    }
  }

  /* report all direct connections */
  DLINK_FOREACH(ptr, local_client_list.head)
  {
    target_p = ptr->data;

    if (IsInvisible(target_p) && dow &&
        !(MyConnect(source_p) && IsOper(source_p)) &&
        !IsOper(target_p) && (target_p != source_p))
      continue;
    if (!doall && wilds && !match(tname, target_p->name))
      continue;
    if (!dow && irccmp(tname, target_p->name))
      continue;

    cnt = status(source_p, target_p, dow, 0, 0);
  }

  DLINK_FOREACH(ptr, serv_list.head)
  {
    target_p = ptr->data;

    if (!doall && wilds && !match(tname, target_p->name))
      continue;
    if (!dow && irccmp(tname, target_p->name))
      continue;

    cnt = status(source_p, target_p, dow,
                             link_u[target_p->localClient->fd],
                             link_s[target_p->localClient->fd]);
  }

  /* This section is to report the unknowns */
  DLINK_FOREACH(ptr, unknown_list.head)
  {
    target_p = ptr->data;

    if (!doall && wilds && !match(tname, target_p->name))
      continue;
    if (!dow && irccmp(tname, target_p->name))
      continue;
    cnt = status(source_p, target_p, dow, 0, 0);
  }

  DLINK_FOREACH(ptr, class_items.head)
  {
    conf = ptr->data;
    cltmp = (struct ClassItem *)map_to_conf(conf);
    if (CurrUserCount(cltmp) > 0)
      sendto_one(source_p, form_str(RPL_TRACECLASS),
		 me.name, source_p->name, conf->name, CurrUserCount(cltmp));
  }
  sendto_one(source_p, form_str(RPL_ENDOFTRACE), me.name, source_p->name, tname);
}

/* parv[0] = sender prefix
 * parv[1] = servername
 */
static void
ms_trace(struct Client *client_p, struct Client *source_p,
	 int parc, char *parv[])
{
  if (!IsOper(source_p))
    return;

  if (hunt_server(client_p, source_p, ":%s TRACE %s :%s", 2, parc, parv))
    return;

  mo_trace(client_p, source_p, parc, parv);
}

/* parv[0] = sender prefix
 * parv[1] = classname
 */
static void
mo_ctrace(struct Client *client_p, struct Client *source_p,
          int parc, char *parv[])
{
  struct Client *target_p = NULL;
  dlink_node *ptr;
  char *class;
  const char *class_name;

  if (parc > 1)
    class = parv[1];
  else
    class = "*";

  set_time();

  /* report all direct connections */
  DLINK_FOREACH(ptr, local_client_list.head)
  {
    target_p = ptr->data;

    class_name = get_client_class(target_p);
    if ((class_name != NULL) && match(class, class_name))
      cstatus(source_p, target_p);
  }
  sendto_one(source_p, form_str(RPL_ENDOFTRACE),me.name,
             source_p->name, class);
  sendto_realops_flags(UMODE_SPY, L_ALL, "CTRACE '%s' by %s (%s@%s) [%s]",
                       class, source_p->name, source_p->username,
                       source_p->host, source_p->user->server->name);
}

/* inputs- pointer to client to report to
 * - pointer to client to report about
 * output- counter of number of hits
 */
static int
cstatus(struct Client *source_p, struct Client *target_p)
{
  const char *name, *class_name;
  char ip[HOSTIPLEN];
  int cnt = 0;

  irc_getnameinfo((struct sockaddr*)&target_p->localClient->ip,
        target_p->localClient->ip.ss_len, ip, HOSTIPLEN, NULL, 0,
        NI_NUMERICHOST);
  name = get_client_name(target_p, HIDE_IP);
  class_name = get_client_class(target_p);
  set_time();

  switch(target_p->status)
  {
    case STAT_CLIENT:

      if ((IsOper(source_p) &&
      (MyClient(source_p) || !IsInvisible(target_p)))
      || IsOper(target_p))
      {
        if (IsAdmin(target_p) && !ConfigFileEntry.hide_spoof_ips)
          sendto_one(source_p, form_str(RPL_TRACEOPERATOR),
                     me.name, source_p->name, class_name, name,
		     IsAdmin(source_p) ? ip : "masked", target_p->info,
                     CurrentTime - target_p->lasttime,
                     (target_p->user) ? (CurrentTime - target_p->user->last) : 0);      
        else if (IsOper(target_p))
        {
          if (ConfigFileEntry.hide_spoof_ips)
      	    sendto_one(source_p, form_str(RPL_TRACEOPERATOR),
		       me.name, source_p->name, class_name, name,
		       IsIPSpoof(target_p) ? "masked" : ip, target_p->info,
		       CurrentTime - target_p->lasttime,
		       (target_p->user)?(CurrentTime - target_p->user->last):0);
	  else  
            sendto_one(source_p, form_str(RPL_TRACEOPERATOR),
                       me.name, source_p->name, class_name, name,
                       (IsIPSpoof(target_p) ? "masked" : ip), target_p->info,
                       CurrentTime - target_p->lasttime,
                       (target_p->user)?(CurrentTime - target_p->user->last):0);
        }      
        else
        {
          if (ConfigFileEntry.hide_spoof_ips)
 	    sendto_one(source_p, form_str(RPL_TRACEUSER),
        	       me.name, source_p->name, class_name, name,
                       IsIPSpoof(target_p) ? "masked" : ip, target_p->info,
		       CurrentTime - target_p->lasttime,
		       (target_p->user)?(CurrentTime - target_p->user->last):0);
          else
            sendto_one(source_p, form_str(RPL_TRACEUSER),
                       me.name, source_p->name, class_name, name,
                       (IsIPSpoof(target_p) ? "masked" : ip), target_p->info,
                       CurrentTime - target_p->lasttime,
                       (target_p->user)?(CurrentTime - target_p->user->last):0);
        }
	cnt++;
      }
      break;
    case STAT_SERVER:
      if(!IsAdmin(source_p))
        name = get_client_name(target_p, MASK_IP);

      sendto_one(source_p, form_str(RPL_TRACESERVER),
 		 me.name, source_p->name, class_name, 0,
		 0, name, *(target_p->serv->by) ?
		 target_p->serv->by : "*", "*",
		 me.name, CurrentTime - target_p->lasttime);
      cnt++;
      break;
     
    default: /* ...we actually shouldn't come here... --msa */
      sendto_one(source_p, form_str(RPL_TRACENEWTYPE),
		 me.name, source_p->name, name);
      cnt++;
      break;
  }
  return(cnt);
}

/* inputs	- pointer to client to report to
 * 		- pointer to client to report about
 * output	- counter of number of hits
 */
static int
status(struct Client *source_p, struct Client *target_p,
       int dow, int link_u_p, int link_s_p)
{
  const char *name, *class_name;
  char ip[HOSTIPLEN];
  int cnt = 0;

  irc_getnameinfo((struct sockaddr*)&target_p->localClient->ip, 
        target_p->localClient->ip.ss_len, ip, HOSTIPLEN, NULL, 0, 
        NI_NUMERICHOST);
  name = get_client_name(target_p, HIDE_IP);
  class_name = get_client_class(target_p);
  set_time();

  switch(target_p->status)
  {
    case STAT_CONNECTING:
      sendto_one(source_p, form_str(RPL_TRACECONNECTING),
		 me.name, source_p->name, class_name, 
		 IsAdmin(source_p) ? name : target_p->name);
		   
      cnt++;
      break;
    case STAT_HANDSHAKE:
      sendto_one(source_p, form_str(RPL_TRACEHANDSHAKE),
                 me.name, source_p->name, class_name, 
		 IsAdmin(source_p) ? name : target_p->name);
		   
      cnt++;
      break;
    case STAT_ME:
      break;
    case STAT_UNKNOWN:
      sendto_one(source_p, form_str(RPL_TRACEUNKNOWN),
		 me.name, source_p->name, class_name, name, ip,
		 target_p->firsttime ? CurrentTime - target_p->firsttime : -1);
      cnt++;
      break;
    case STAT_CLIENT:
      if ((IsOper(source_p) &&
           (MyClient(source_p) || !(dow && IsInvisible(target_p))))
          || !dow || IsOper(target_p))
        {
          if (IsAdmin(target_p) && !ConfigFileEntry.hide_spoof_ips)
            sendto_one(source_p, form_str(RPL_TRACEOPERATOR),
                       me.name, source_p->name, class_name, name,
                       IsOperAdmin(source_p) ? ip : "masked", target_p->info,
                       CurrentTime - target_p->lasttime,
                       (target_p->user) ? (CurrentTime - target_p->user->last) : 0);

          else if (IsOper(target_p))
          {
            if (ConfigFileEntry.hide_spoof_ips)
              sendto_one(source_p, form_str(RPL_TRACEOPERATOR),
                         me.name, source_p->name, class_name, name,
                         IsIPSpoof(target_p) ? "masked" : ip, target_p->info,
                         CurrentTime - target_p->lasttime,
                         (target_p->user)?(CurrentTime - target_p->user->last):0);
            else
              sendto_one(source_p, form_str(RPL_TRACEOPERATOR),
                         me.name, source_p->name, class_name, name,
                         MyOper(source_p) ? ip :
                         (IsIPSpoof(target_p) ? "masked" : ip), target_p->info,
                         CurrentTime - target_p->lasttime,
                         (target_p->user)?(CurrentTime - target_p->user->last):0);
          }
          else
          {
            if (ConfigFileEntry.hide_spoof_ips)
              sendto_one(source_p, form_str(RPL_TRACEUSER),
                         me.name, source_p->name, class_name, name,
                         IsIPSpoof(target_p) ? "masked" : ip, target_p->info,
                         CurrentTime - target_p->lasttime,
                         (target_p->user)?(CurrentTime - target_p->user->last):0);
            else
              sendto_one(source_p, form_str(RPL_TRACEUSER),
                         me.name, source_p->name, class_name, name,
                         MyOper(source_p) ? ip :
                         (IsIPSpoof(target_p) ? "masked" : ip), target_p->info,
                         CurrentTime - target_p->lasttime,
                         (target_p->user)?(CurrentTime - target_p->user->last):0);
	  }
          cnt++;
        }
      break;
    case STAT_SERVER:
      if(!IsAdmin(source_p))
        name = get_client_name(target_p, MASK_IP);

      sendto_one(source_p, form_str(RPL_TRACESERVER),
		 me.name, source_p->name, class_name, link_s_p,
		 link_u_p, name, *(target_p->serv->by) ?
		 target_p->serv->by : "*", "*",
		 me.name, CurrentTime - target_p->lasttime);
      cnt++;
      break;
      
    default: /* ...we actually shouldn't come here... --msa */
      sendto_one(source_p, form_str(RPL_TRACENEWTYPE), me.name,
		 source_p->name, name);
      cnt++;
      break;
  }

  return(cnt);
}

/* parv[0] = sender prefix
 * parv[1] = classname
 */
static void
mo_classlist(struct Client *client_p, struct Client *source_p,
	     int parc, char *parv[])
{
  struct ClassItem *aclass;
  struct ConfItem *conf;
  char *classname;
  dlink_node *ptr;

  classname = parv[1];

  DLINK_FOREACH(ptr, class_items.head)
  {
    conf = ptr->data;

    if (conf == NULL)
      continue;

    if (match(classname, conf->name))
    {
      aclass = (struct ClassItem *)map_to_conf(conf);
      sendto_one(source_p, ":%s NOTICE %s :CLASSLIST: %s %d",
		 me.name, source_p->name, conf->name,
                 CurrUserCount(aclass));
      return;
    }
  }
  sendto_one(source_p, ":%s NOTICE %s :CLASSLIST: Class %s not found",
	     me.name, source_p->name, classname);
}
