/* m_who.c: Shows who is on a channel.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: m_who.c,v 1.76 2005/01/27 11:48:02 musirc Exp $
 */

#include "client.h"
#include "handlers.h"
#include "hash.h"
#include "channel.h"
#include "channel_mode.h"
#include "ircd.h"
#include "numeric.h"
#include "server.h"
#include "send.h"
#include "istring.h"
#include "sprintf.h"
#include "config.h"
#include "modules.h"

static void m_who(struct Client *, struct Client *, int, char **);
static void mo_owho(struct Client *, struct Client *, int, char **);

struct Message who_msgtab = {
  "WHO", 2, MFLG_SLOW,
  {m_unregistered, m_who, m_ignore, m_who}
};
struct Message owho_msgtab = {
  "OWHO", 2, MFLG_SLOW,
  {m_ignore, m_ignore, m_ignore, mo_owho}
};

#ifndef STATIC_MODULES
void
_modinit(void)
{
  mod_add_cmd(&who_msgtab);
  mod_add_cmd(&owho_msgtab);
}

void
_moddeinit(void)
{
  mod_del_cmd(&who_msgtab);
  mod_del_cmd(&owho_msgtab);
}

const char *_version = "$Revision: 1.76 $";
#endif
static void who_global(struct Client *, char *, int, int);
static void do_who(struct Client *, struct Client *, const char *, const char *, int);
static void do_who_on_channel(struct Client *, struct Channel *, const char *, int, int, int);

/* parv[0] = sender prefix
 * parv[1] = nickname mask list
 * parv[2] = additional selection flag, only 'o' for now.
 */
static void
m_who(struct Client *client_p, struct Client *source_p,
      int parc, char *parv[])
{
  struct Client *target_p;
  char *mask = parv[1];
  dlink_node *lp, *lp_next;
  struct Channel *chptr = NULL, *channel = NULL;
  int server_oper = parc > 2 ? (*parv[2] == 'o') : 0, member;

  /* See if mask is there, collapse it or return if not there */
  if (mask != NULL)
  {
    collapse(mask);

    if (*mask == '\0')
    {
      sendto_one(source_p, form_str(RPL_ENDOFWHO),
		 me.name, source_p->name, "*");
      return;
    }
  }
  else
  {
    who_global(source_p, mask, server_oper, 0);
    sendto_one(source_p, form_str(RPL_ENDOFWHO),
	       me.name, source_p->name, "*");
    return;
  }

  /* '/who *' */
  if ((*(mask + 1) == '\0') && (*mask == '*'))
  {
    if (source_p->user != NULL)
      if ((lp = source_p->user->channel.head) != NULL)
        channel = ((struct Membership *)lp->data)->chptr;

    if (channel == NULL)
    {
      sendto_one(source_p, form_str(RPL_ENDOFWHO), me.name, source_p->name, "*");
      return;
    }
    do_who_on_channel(source_p, channel, channel->name, 1, server_oper, 0);
    sendto_one(source_p, form_str(RPL_ENDOFWHO), me.name, source_p->name, "*");
    return;
  }

  /* '/who #channel' */
  if (IsChanPrefix(*mask))
  {
    /* List all users on a given channel */
    if ((chptr = hash_find_channel(mask)) != NULL)
    {
      if (IsMember(source_p, chptr))
        do_who_on_channel(source_p, chptr, chptr->name, 1, server_oper, 0);
      else if (!SecretChannel(chptr))
        do_who_on_channel(source_p, chptr, chptr->name, 0, server_oper, 0);
    }
    sendto_one(source_p, form_str(RPL_ENDOFWHO), me.name, source_p->name, mask);
    return;
  }

  /* '/who nick' */
  if (((target_p = find_client(mask)) != NULL) &&
      IsPerson(target_p) && (!server_oper || IsOper(target_p)))
  {
    int isinvis = 0;

    isinvis = IsInvisible(target_p);
    DLINK_FOREACH_SAFE(lp, lp_next, target_p->user->channel.head)
    {
      chptr = ((struct Membership *) lp->data)->chptr;
      member = IsMember(source_p, chptr);
      if (isinvis && !member)
        continue;
      if (member || (!isinvis && PubChannel(chptr)))
        break;
    }
    if (lp != NULL)
      do_who(source_p, target_p, ((struct Membership *) lp->data)->chptr->name,
   	     get_member_status(lp->data, 0), 0);
    else
      do_who(source_p, target_p, NULL, "", 0);

    sendto_one(source_p, form_str(RPL_ENDOFWHO), me.name, source_p->name, mask);
    return;
  }

  /* '/who 0' */
  if ((*(mask + 1) == '\0') && (*mask == '0'))
    who_global(source_p, NULL, server_oper, 0);
  else
    who_global(source_p, mask, server_oper, 0);

  /* Wasn't a nick, wasn't a channel, wasn't a '*' so ... */
  sendto_one(source_p, form_str(RPL_ENDOFWHO), me.name, source_p->name, mask);
}

/* parv[0] = sender prefix
 * parv[1] = nickname mask list
 * parv[2] = additional selection flag, only 'o' for now.
 */
static void
mo_owho(struct Client *client_p, struct Client *source_p,
        int parc, char *parv[])
{
  char *mask = parv[1];
  int server_oper = parc > 2 ? (*parv[2] == 'o') : 0;
  struct Channel *chptr;
  struct Client *target_p;

  if (mask != NULL)
  {
    collapse(mask);

    if (*mask == '\0')
    {
      sendto_one(source_p, form_str(RPL_ENDOFWHO),
		 me.name, source_p->name, "*");
      sendto_realops_flags(UMODE_SPY, L_ALL, "OWHO '*' by %s (%s@%s)",
                           source_p->name, source_p->username, source_p->host);
      return;
    }
  }
  else
  {
    who_global(source_p, NULL, server_oper, 1);
    sendto_one(source_p, form_str(RPL_ENDOFWHO),
	       me.name, source_p->name, "*");
    sendto_realops_flags(UMODE_SPY, L_ALL, "OWHO '*' by %s (%s@%s)",
                         source_p->name, source_p->username, source_p->host);
    return;
  }

  /* /owho #channel */
  if (IsChanPrefix(*mask))
  {
    if ((chptr = hash_find_channel(mask)) != NULL)
      do_who_on_channel(client_p, chptr, chptr->name, 1, server_oper, 1);
    sendto_realops_flags(UMODE_SPY, L_ALL, "OWHO '%s' by %s (%s@%s)",
                         mask, source_p->name, source_p->username, source_p->host);

    sendto_one(source_p, form_str(RPL_ENDOFWHO), me.name, source_p->name, mask);
    return;
  }

  /* /owho nick */
  if (((target_p = find_client(mask)) != NULL) &&
      IsPerson(target_p))
  {
    if (target_p->user->channel.head != NULL)
    {
      chptr = ((struct Membership *)target_p->user->channel.head->data)->chptr;

      do_who(client_p, target_p, chptr->name,
             get_member_status(target_p->user->channel.head->data, 0), 1);
    }
    else
    {
      do_who(source_p, target_p, NULL, "", 1);
    }
    sendto_one(source_p, form_str(RPL_ENDOFWHO), me.name, source_p->name, mask);
    sendto_realops_flags(UMODE_SPY, L_ALL, "OWHO '%s' by %s (%s@%s)",
                         mask, source_p->name, source_p->username, source_p->host);
    return;
  }

  /* /owho 0 */
  if ((*(mask+1) == '\0') && (*mask == '0'))
    who_global(source_p, NULL, server_oper, 1);
  else
    who_global(source_p, mask, server_oper, 1);

  sendto_one(source_p, form_str(RPL_ENDOFWHO), me.name, source_p->name, mask);

  sendto_realops_flags(UMODE_SPY, L_ALL, "OWHO '%s' by %s (%s@%s)",
                       mask, source_p->name, source_p->username,
		       source_p->host);
}

/* inputs	- pointer to client requesting who
 * 		- pointer to channel member chain.
 *		- char * mask to match
 *		- int if oper on a server or not
 *		- pointer to int maxmatches
 * side effects - lists matching clients on specified channel,
 * 		  marks matched clients.
 */
static void
who_common_channel(struct Client *source_p, struct Channel *chptr,
		   char *mask, int server_oper, int *maxmatches)
{
  dlink_node *ptr, *ptr_next;
  struct Client *target_p;

  DLINK_FOREACH_SAFE(ptr, ptr_next, chptr->members.head)
  {
    target_p = ((struct Membership *)ptr->data)->client_p;

    if (!IsInvisible(target_p) || IsMarked(target_p))
      continue;

    if (server_oper && !IsOper(target_p))
      continue;

    SetMark(target_p);

    if ((mask == NULL) ||
	match(mask, target_p->name) || match(mask, target_p->username) ||
	match(mask, target_p->host) || 
	(match(mask, target_p->user->server->name) &&
	 (IsOper(source_p) || !ConfigServerHide.hide_servers)) ||
	match(mask, target_p->info))
    {
      do_who(source_p, target_p, NULL, "", 0);

      if (*maxmatches > 0)
      {
        if (--(*maxmatches) == 0)
	  return;
      }
    }
  }
}

/* inputs	- pointer to client requesting who
 *		- char * mask to match
 *		- int if oper on a server or not
 * side effects - do a global scan of all clients looking for match
 */
static void
who_global(struct Client *source_p,char *mask, int server_oper, int type)
{
  struct Channel *chptr;
  struct Client *target_p;
  dlink_node *lp, *lp_next, *gcptr, *gcptr_next;
  int maxmatches = 5000;

  /* first, list all matching invisible clients on common channels */
  if (type == 0)
  {
    DLINK_FOREACH_SAFE(lp, lp_next, source_p->user->channel.head)
    {
      chptr = ((struct Membership *)lp->data)->chptr;
      who_common_channel(source_p, chptr, mask, server_oper, &maxmatches);
    }

    /* second, list all matching visible clients */
    DLINK_FOREACH_SAFE(gcptr, gcptr_next, global_client_list.head)
    {
      target_p = gcptr->data;

      if (!IsPerson(target_p))
        continue;

      if (IsInvisible(target_p))
      {
        ClearMark(target_p);
        continue;
      }

      if (server_oper && !IsOper(target_p))
        continue;

      if (!mask ||
          match(mask, target_p->name) || match(mask, target_p->username) ||
	  match(mask, target_p->host) || match(mask, target_p->user->server->name) ||
	  match(mask, target_p->info))
      {		
        do_who(source_p, target_p, NULL, "", 0);
          if (maxmatches > 0)
        {
          if (--maxmatches == 0)
  	    return;
        }
      }
    }
  }
  else
  {
    DLINK_FOREACH_SAFE(gcptr, gcptr_next, global_client_list.head)
    {
      target_p = gcptr->data;

      if (!IsPerson(target_p))
        continue;

      if (!mask ||
          match(mask, target_p->name) || match(mask, target_p->username) ||
          match(mask, target_p->host) || match(mask, target_p->user->server->name) ||
          match(mask, target_p->info) ||
          (MyClient(target_p) && match(mask, target_p->localClient->sockhost)))
      {
        if (dlink_list_length(&target_p->user->channel))
        {
          static char fl[5];

          chptr = ((struct Membership *)(target_p->user->channel.head->data))->chptr;
          snprintf(fl, sizeof(fl), "%s",
                   get_member_status((struct Membership *)(target_p->user->channel.head->data),
		   0));

          do_who(source_p, target_p, chptr->name, fl, 1);
        }
        else
          do_who(source_p, target_p, NULL, "", 1);

        if (maxmatches > 0)
        {
          if (--maxmatches == 0)
            return;
        }
      }
    }
  }
}

/* inputs	- pointer to client requesting who
 *		- pointer to channel to do who on
 *		- The "real name" of this channel
 *		- int if source_p is a server oper or not
 *		- int if client is member or not
 *		- int server_op flag
 * side effects - do a who on given channel
 */
static void
do_who_on_channel(struct Client *source_p, struct Channel *chptr,
		  const char *name, int member, int server_oper, int type)
{
  dlink_node *ptr, *ptr_next;
  struct Client *target_p;
  struct Membership *ms;

  DLINK_FOREACH_SAFE(ptr, ptr_next, chptr->members.head)
  {
    ms = ptr->data;

    if (type == 0)
    {
      target_p = ms->client_p;

      if (member || !IsInvisible(target_p))
      {
        if (server_oper && !IsOper(target_p))
          continue;
        do_who(source_p, target_p, name, get_member_status(ms, 0), 0);
      }
    }
    else
      do_who(source_p, ms->client_p, name, get_member_status(ms, 0), 1);
  }
}

/* inputs	- pointer to client requesting who
 *		- pointer to client to do who on
 *		- The reported name
 *		- channel flags
 * side effects - do a who on given person
 */
static void
do_who(struct Client *source_p, struct Client *target_p,
       const char *name, const char *op_flags, int type)
{
  char status[5];

  ircsprintf(status,"%c%s%s", target_p->user->away ? 'G' : 'H',
	     IsOper(target_p) ? "*" : "", op_flags );

  if ((type == 0) && ConfigServerHide.hide_servers)
    sendto_one(source_p, form_str(RPL_WHOREPLY), me.name, source_p->name,
	       (name) ? (name) : "*",
	       target_p->username,
	       target_p->host, IsOper(source_p) ? target_p->user->server->name : "*",
	       target_p->name,
	       status, 0, target_p->info);
  else
    sendto_one(source_p, form_str(RPL_WHOREPLY), me.name, source_p->name,
	       (name) ? (name) : "*",
	       target_p->username,
	       target_p->host, target_p->user->server->name, target_p->name,
	       status, target_p->hopcount, target_p->info);
}
