/* auth.c: Functions for querying a users ident.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: auth.c,v 1.17 2005/01/27 11:48:03 musirc Exp $
 */

#include "tools.h"
#include "auth.h"
#include "config.h"
#include "event.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "packet.h"
#include "res.h"
#include "bsd.h"
#include "log.h"
#include "send.h"

static dlink_list auth_doing_dns_list = { NULL, NULL, 0 };
static dlink_list auth_doing_ident_list = { NULL, NULL, 0 };
static EVH timeout_auth_queries_event;
static PF read_auth_reply;
static CNCB auth_connect_callback;

void
init_auth(void)
{
  eventAddIsh("timeout_auth_queries_event", timeout_auth_queries_event, NULL, 1);
}

static struct AuthRequest *
make_auth_request(struct Client *client)
{
  struct AuthRequest *request = 
    (struct AuthRequest *)MyMalloc(sizeof(struct AuthRequest));
  request->fd = -1;
  request->client  = client;
  request->timeout = CurrentTime + CONNECTTIMEOUT;
  return(request);
}

/* release auth client from auth system
 * this adds the client into the local client lists so it can be read by
 * the main io processing loop
 */
static void
release_auth_client(struct Client *client_p)
{
  if (client_p->localClient->fd > highest_fd)
    highest_fd = client_p->localClient->fd;

  client_p->localClient->allow_read = MAX_FLOOD;
  comm_setflush(client_p->localClient->fd, 1000, flood_recalc, client_p);
  set_no_delay(client_p->localClient->fd);
  dlinkAdd(client_p, &client_p->node, &global_client_list);
  read_packet(client_p->localClient->fd, client_p);
}
 
/* called when resolver query finishes
 * if the query resulted in a successful search, hp will contain
 * a non-null pointer, otherwise hp will be null.
 * set the client on it's way to a connection completion, regardless
 * of success of failure
 */
static void
auth_dns_callback(void *vptr, struct DNSReply *reply)
{
  struct AuthRequest *auth = (struct AuthRequest *)vptr;

  if (IsDNSPending(auth))
    dlinkDelete(&auth->dns_node, &auth_doing_dns_list);
  ClearDNSPending(auth);

  if (reply != NULL)
  {
    struct sockaddr_in *v4, *v4dns;
#ifdef IPV6
    struct sockaddr_in6 *v6, *v6dns;
#endif
    int good = 1;
    
#ifdef IPV6
    if(auth->client->localClient->ip.ss.ss_family == AF_INET6)
    {
      v6 = (struct sockaddr_in6 *)&auth->client->localClient->ip;
      v6dns = (struct sockaddr_in6 *)&reply->addr;
      if(memcmp(&v6->sin6_addr, &v6dns->sin6_addr, sizeof(struct in6_addr)) != 0)
        good = 0;
    }
    else
#endif
    {
      v4 = (struct sockaddr_in *)&auth->client->localClient->ip;
      v4dns = (struct sockaddr_in *)&reply->addr;
      if(v4->sin_addr.s_addr != v4dns->sin_addr.s_addr)
        good = 0;
    }
    if (good && strlen(reply->h_name) <= HOSTLEN)
      strlcpy(auth->client->host, reply->h_name,
	      sizeof(auth->client->host));
  }

  MyFree(auth->client->localClient->dns_query);
  auth->client->localClient->dns_query = NULL;

  if (!IsDoingAuth(auth))
  {
    struct Client *client_p = auth->client;
    MyFree(auth);
    release_auth_client(client_p);
  }
}

/* handle auth send errors */
static void
auth_error(struct AuthRequest *auth)
{
  fd_close(auth->fd);
  auth->fd = -1;

  if (IsAuthPending(auth))
    dlinkDelete(&auth->ident_node, &auth_doing_ident_list);
  ClearAuth(auth);

  if (!IsDNSPending(auth))
  {
    release_auth_client(auth->client);
    MyFree(auth);
  }
}

/* Flag the client to show that an attempt to 
 * contact the ident server on
 * the client's host.  The connect and subsequently the socket are all put
 * into 'non-blocking' mode.  Should the connect or any later phase of the
 * identifing process fail, it is aborted and the user is given a username
 * of "unknown".
 */
static int
start_auth_query(struct AuthRequest *auth)
{
  struct irc_ssaddr localaddr;
  socklen_t locallen = sizeof(struct irc_ssaddr);
  int fd;
#ifdef IPV6
  struct sockaddr_in6 *v6;
#else
  struct sockaddr_in *v4;
#endif

  /* open a socket of the same type as the client socket */
  if ((fd = comm_open(auth->client->localClient->ip.ss.ss_family, SOCK_STREAM,
          0, "ident")) == -1)
  {
    report_error(L_ALL, "creating auth stream socket %s:%s", 
        get_client_name(auth->client, SHOW_IP), errno);
    ilog(ERROR, "Unable to create auth socket for %s",
        get_client_name(auth->client, SHOW_IP));
    return 0;
  }
  if ((HARD_FDLIMIT - 10) < fd)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,"Can't allocate fd for auth on %s",
        get_client_name(auth->client, SHOW_IP));
    fd_close(fd);
    return 0;
  }

  if (!set_non_blocking(fd))
  {
    report_error(L_ALL, NONB_ERROR_MSG, get_client_name(auth->client, SHOW_IP), errno);
    fd_close(fd);
    return 0;
  }

  /* get the local address of the client and bind to that to
   * make the auth request.  This used to be done only for
   * ifdef VIRTUAL_HOST, but needs to be done for all clients
   * since the ident request must originate from that same address--
   * and machines with multiple IP addresses are common now
   */
  memset(&localaddr, 0, locallen);
  getsockname(auth->client->localClient->fd, (struct sockaddr*)&localaddr,
      &locallen);

#ifdef IPV6
  remove_ipv6_mapping(&localaddr);
  v6 = (struct sockaddr_in6 *)&localaddr;
  v6->sin6_port = htons(0);
#else
  localaddr.ss_len = locallen;
  v4 = (struct sockaddr_in *)&localaddr;
  v4->sin_port = htons(0);
#endif
  localaddr.ss_port = htons(0);

  auth->fd = fd;
  SetAuthConnect(auth);

  comm_connect_tcp(fd, auth->client->localClient->sockhost, 113, 
      (struct sockaddr *)&localaddr, localaddr.ss_len, auth_connect_callback, 
      auth, auth->client->localClient->ip.ss.ss_family, 
      GlobalSetOptions.ident_timeout);
  return 1;
}

/* parse ident query reply from identd server
 * Inputs        - pointer to ident buf
 * Output        - NULL if no valid ident found, otherwise pointer to name
 */
static char*
GetValidIdent(char *buf)
{
  int remp = 0, locp = 0;
  char *colon1Ptr, *colon2Ptr, *colon3Ptr, *commaPtr, *remotePortString;

  remotePortString = buf;
  
  if((colon1Ptr = strchr(remotePortString,':')) == NULL)
    return 0;

  *colon1Ptr = '\0';
  colon1Ptr++;

  if((colon2Ptr = strchr(colon1Ptr,':')) == NULL)
    return 0;

  *colon2Ptr = '\0';
  colon2Ptr++;
  
  if((commaPtr = strchr(remotePortString, ',')) == NULL)
    return 0;

  *commaPtr = '\0';
  commaPtr++;

  if ((remp = atoi(remotePortString)) == 0)
    return 0;
              
  if ((locp = atoi(commaPtr)) == 0)
    return 0;

  /* look for USERID bordered by first pair of colons */
  if (strstr(colon1Ptr, "USERID") == NULL)
    return 0;

  if((colon3Ptr = strchr(colon2Ptr,':')) == NULL)
    return 0;

  *colon3Ptr = '\0';
  colon3Ptr++;
  return(colon3Ptr);
}

/* starts auth (identd) and dns queries for a client */
void
start_auth(struct Client *client)
{
  struct AuthRequest *auth = NULL;

  if (client == NULL)
    return;

  auth = make_auth_request(client);
  client->localClient->dns_query = MyMalloc(sizeof(struct DNSQuery));
  client->localClient->dns_query->ptr = auth;
  client->localClient->dns_query->callback = auth_dns_callback;
  gethost_byaddr(&client->localClient->ip, client->localClient->dns_query);
  SetDNSPending(auth);
  dlinkAdd(auth, &auth->dns_node, &auth_doing_dns_list);

  if(ConfigFileEntry.disable_auth == 0)
    (void)start_auth_query(auth);
}

/* timeout resolver and identd requests
 * allow clients through if requests failed
 */
static void
timeout_auth_queries_event(void *n)
{
  dlink_node *ptr, *next_ptr;
  struct AuthRequest *auth;

  DLINK_FOREACH_SAFE(ptr, next_ptr, auth_doing_ident_list.head)
  {
    auth = ptr->data;

    if (auth->timeout < CurrentTime)
    {
      if (auth->fd >= 0)
	fd_close(auth->fd);

      if (IsDNSPending(auth))
      {
        struct Client *client_p=auth->client;
        ClearDNSPending(auth);
        dlinkDelete(&auth->dns_node, &auth_doing_dns_list);
        if (client_p->localClient->dns_query != NULL)
	{
          delete_resolver_queries(client_p->localClient->dns_query);
	  MyFree(client_p->localClient->dns_query);
	}
	auth->client->localClient->dns_query = NULL;
      }
      ilog(INFO, "DNS/AUTH timeout %s",
	   get_client_name(auth->client, SHOW_IP));

      auth->client->since = CurrentTime;
      if (IsAuthPending(auth))
        dlinkDelete(&auth->ident_node, &auth_doing_ident_list);
      release_auth_client(auth->client);
      MyFree(auth);
    }
  }
}

/* deal with the result of comm_connect_tcp()
 * If the connection failed, we simply close the auth fd and report
 * a failure. If the connection suceeded send the ident server a query
 * giving "theirport , ourport".
 */
static void
auth_connect_callback(int n, int error, void *data)
{
  struct AuthRequest *auth = data;
  struct irc_ssaddr us, them;
  char authbuf[32];
  socklen_t ulen = sizeof(struct irc_ssaddr);
  socklen_t tlen = sizeof(struct irc_ssaddr);
  uint16_t uport, tport;
#ifdef IPV6
  struct sockaddr_in6 *v6;
#else
  struct sockaddr_in *v4;
#endif

  if (error != COMM_OK)
  {
    auth_error(auth);
    return;
  }

  if (getsockname(auth->client->localClient->fd, (struct sockaddr *)&us,   (socklen_t*)&ulen) ||
      getpeername(auth->client->localClient->fd, (struct sockaddr *)&them, (socklen_t*)&tlen))
  {
    ilog(INFO, "auth get{sock,peer}name error for %s",
        get_client_name(auth->client, SHOW_IP));
    auth_error(auth);
    return;
  }

#ifdef IPV6
  v6 = (struct sockaddr_in6 *)&us;
  uport = ntohs(v6->sin6_port);
  v6 = (struct sockaddr_in6 *)&them;
  tport = ntohs(v6->sin6_port);
  remove_ipv6_mapping(&us);
  remove_ipv6_mapping(&them);
#else
  v4 = (struct sockaddr_in *)&us;
  uport = ntohs(v4->sin_port);
  v4 = (struct sockaddr_in *)&them;
  tport = ntohs(v4->sin_port);
  us.ss_len = ulen;
  them.ss_len = tlen;
#endif
  
  ircsprintf(authbuf, "%u , %u\r\n", tport, uport); 

  if (send(auth->fd, authbuf, strlen(authbuf), 0) == -1)
  {
    auth_error(auth);
    return;
  }
  ClearAuthConnect(auth);
  SetAuthPending(auth);
  dlinkAdd(auth, &auth->ident_node, &auth_doing_ident_list);
  read_auth_reply(auth->fd, auth);
}

/* read the reply (if any) from the ident server 
 * we connected to.
 * We only give it one shot, if the reply isn't good the first time
 * fail the authentication entirely.
 */
#define AUTH_BUFSIZ 128

static void
read_auth_reply(int fd, void *data)
{
  struct AuthRequest *auth = data;
  char *s = NULL, *t = NULL;
  int len, count;
  char buf[AUTH_BUFSIZ + 1]; /* buffer to read auth reply into */

  len = recv(auth->fd, buf, AUTH_BUFSIZ, 0);
  
  if (len < 0 && ignoreErrno(errno))
  {
    comm_setselect(fd, FDLIST_IDLECLIENT, COMM_SELECT_READ,
                   read_auth_reply, auth, 0);
    return;
  }

  if (len > 0)
  {
    buf[len] = '\0';

    if ((s = GetValidIdent(buf)))
    {
      t = auth->client->username;

      while(*s == '~' || *s == '^')
        s++;

      for (count = USERLEN; *s && count; s++)
      {
        if(*s == '@')
	  break;

        if (!IsSpace(*s) && *s != ':' && *s != '[')
	{
	  *t++ = *s;
	  count--;
	}
      }
      *t = '\0';
    }
  }

  fd_close(auth->fd);
  auth->fd = -1;

  if (IsAuthPending(auth))
    dlinkDelete(&auth->ident_node, &auth_doing_ident_list);  
  ClearAuth(auth);
  
  if (s == NULL)
    strcpy(auth->client->username, "unknown");
  else
    SetGotId(auth->client);

  if (!IsDNSPending(auth))
  {
    release_auth_client(auth->client);
    MyFree(auth);
  }
}

void 
delete_identd_queries(struct Client *target_p)
{
  dlink_node *ptr, *next_ptr;
  struct AuthRequest *auth;

  DLINK_FOREACH_SAFE(ptr, next_ptr, auth_doing_ident_list.head)
  {
    auth = ptr->data;

    if (auth->client == target_p)
    {
      if (auth->fd >= 0)
      {
        fd_close(auth->fd);
        auth->fd = -1;
      }

      if (IsAuthPending(auth))
        dlinkDelete(&auth->ident_node, &auth_doing_ident_list);

      MyFree(auth);
    }
  }
}
