/* channel.c: Controls channels.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: channel.c,v 1.92 2005/01/27 11:48:04 musirc Exp $
 */

#include "channel.h"
#include "channel_mode.h"
#include "client.h"
#include "hash.h"
#include "list.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "numeric.h"
#include "server.h"
#include "user.h"
#include "send.h"
#include "config.h"
#include "event.h"
#include "jupe.h"

struct config_channel_entry ConfigChannel;
dlink_list global_channel_list = { NULL, NULL, 0 };
BlockHeap *channel_heap;
BlockHeap *ban_heap;
BlockHeap *topic_heap;
BlockHeap *member_heap;

static char buf[BUFSIZE];
static char modebuf[MODEBUFLEN];
static char parabuf[MODEBUFLEN];
static void send_mode_list(struct Client *, struct Channel *, dlink_list *, char);
static int check_banned(struct Channel *, const char *, const char *);
static const char *channel_pub_or_secret(struct Channel *);

void
init_channels(void)
{
  channel_heap = BlockHeapCreate(sizeof(struct Channel), CHANNEL_HEAP_SIZE);
  ban_heap = BlockHeapCreate(sizeof(struct Ban), BAN_HEAP_SIZE);
  topic_heap = BlockHeapCreate(TOPICLEN+1 + USERHOST_REPLYLEN, TOPIC_HEAP_SIZE);
  member_heap = BlockHeapCreate(sizeof(struct Membership), CHANNEL_HEAP_SIZE);
}

/* inputs       - pointer to channel to add client to
 *              - pointer to client (who) to add
 *              - flags for chanops etc
 * side effects - adds a user to a channel by adding another link to the
 *                channels member chain.
 */
void
add_user_to_channel(struct Channel *chptr, struct Client *who,
                    unsigned int flags)
{
  struct Membership *ms;

  ms = BlockHeapAlloc(member_heap);
  memset(ms, 0, sizeof(struct Membership));
  ms->client_p = who;
  ms->chptr = chptr;
  ms->flags = flags;

  dlinkAdd(ms, &ms->channode, &chptr->members);
  if (MyConnect(who))
    dlinkAdd(ms, &ms->locchannode, &chptr->locmembers);
  dlinkAdd(ms, &ms->usernode, &who->user->channel);
}

/* inputs       - pointer to Membership struct
 * side effects - deletes a user from a channel by removing a link in the
 *                channels member chain.
 */
void
remove_user_from_channel(struct Membership *member)
{
  struct Client *client_p = member->client_p;
  struct Channel *chptr = member->chptr;

  dlinkDelete(&member->channode, &chptr->members);

  if (MyConnect(client_p))
    dlinkDelete(&member->locchannode, &chptr->locmembers);

  dlinkDelete(&member->usernode, &client_p->user->channel);
  BlockHeapFree(member_heap, member);

  if (dlink_list_length(&chptr->members) <= 0)
    destroy_channel(chptr);
}

static void
send_members(struct Client *client_p, struct Channel *chptr,
	     char *lmodebuf, char *lparabuf)
{
  struct Membership *ms;
  dlink_node *ptr;
  int tlen;              /* length of text to append */
  char *t, *start;       /* temp char pointer */

  start = t = buf + ircsprintf(buf, ":%s SJOIN %lu %s %s %s:",
                               me.name, (unsigned long)chptr->channelts,
                               chptr->name, lmodebuf, lparabuf);

  DLINK_FOREACH(ptr, chptr->members.head)
  {
    ms = ptr->data;

    tlen = strlen(ms->client_p->name) + 1;  /* nick + space */
    if (ms->flags & CHFL_CHANOP)
      tlen++;
    if (ms->flags & CHFL_HALFOP)
      tlen++;
    if (ms->flags & CHFL_VOICE)
      tlen++;

    if (t + tlen - buf > BUFSIZE)
    {
      *(t - 1) = '\0';  /* kill the space and terminate the string */
      sendto_one(client_p, "%s", buf);
      t = start;
    }
    strcpy(t, get_member_status(ms, 1));
    t += strlen(t);
    strcpy(t, ms->client_p->name);
    t += strlen(t);
    *t++ = ' ';
  }

  if (chptr->members.head != NULL)
    t--;  /* take the space out */
  *t = '\0';
  sendto_one(client_p, "%s", buf);
}

/* inputs       - pointer to client client_p
 *              - pointer to channel pointer
 * side effects - send "client_p" a full list of the modes for channel chptr.
 */
void
send_channel_modes(struct Client *client_p, struct Channel *chptr)
{
  if (*chptr->name != '#')
    return;

  *modebuf = *parabuf = '\0';
  channel_modes(chptr, client_p, modebuf, parabuf);
  send_members(client_p, chptr, modebuf, parabuf);

  send_mode_list(client_p, chptr, &chptr->banlist, 'b');
  send_mode_list(client_p, chptr, &chptr->exceptlist, 'e');
  send_mode_list(client_p, chptr, &chptr->invexlist, 'I');
}

/* inputs - client pointer to server
 * - pointer to channel
 * - pointer to top of mode link list to send
 * - char flag flagging type of mode i.e. 'b' 'e' etc.
 * - clear (remove all current modes, for ophiding, etc)
 * side effects - sends +b/+e/+I
 */
static void
send_mode_list(struct Client *client_p, struct Channel *chptr,
               dlink_list *top, char flag)
{
  dlink_node *lp;
  struct Ban *banptr;
  char mbuf[MODEBUFLEN];
  char pbuf[MODEBUFLEN];
  int tlen, mlen, cur_len;
  int count = 0;
  char *mp = mbuf;
  char *pp = pbuf;

  if (top == NULL || top->length == 0)
    return;

  ircsprintf(buf, ":%s MODE %s +", me.name, chptr->name);

  cur_len = mlen = (strlen(buf) + 2);
  *mp = *pp = '\0';

  DLINK_FOREACH(lp, top->head)
  {
    banptr = lp->data;
    tlen = strlen(banptr->banstr) + 1;

    if ((cur_len + tlen + 2) > MODEBUFLEN)
    {
      *(pp-1) = '\0'; /* get rid of trailing space on buffer */
      sendto_one(client_p, "%s%s %s", buf, mbuf, pbuf);

      mp = mbuf;
      pp = pbuf;
      cur_len = mlen;
      count = 0;
    }

    count++;
    *mp++ = flag;
    *mp = *pp = '\0';
    ircsprintf(pp, "%s ", banptr->banstr);
    pp += tlen;
    cur_len += tlen;
  }

  *(pp-1) = '\0'; /* get rid of trailing space on buffer */
  sendto_one(client_p, "%s%s %s", buf, mbuf, pbuf);
}

/* inputs       - channel name
 * output       - true  (1) if name ok,
 *              - false (0) otherwise
 * side effects - check channel name for
 *                invalid characters
 */
int
check_channel_name(const char *name)
{
  if (name == NULL)
    return(0);

  for (; *name; ++name)
  {
    if (!IsChanChar(*name))
      return(0);
  }

  return(1);
}

/* pointer to dlink_list */
void
free_channel_list(dlink_list *list)
{
  dlink_node *ptr, *next_ptr;

  DLINK_FOREACH_SAFE(ptr, next_ptr, list->head)
  {
    struct Ban *actualBan = ptr->data;

    MyFree(actualBan->banstr);
    MyFree(actualBan->who);
    dlinkDelete(&actualBan->node, list);
    BlockHeapFree(ban_heap, actualBan);
  }
}

/* inputs       - channel pointer
 * side effects - walk through this channel, and destroy it.
 */
void
destroy_channel(struct Channel *chptr)
{
  dlink_node *ptr, *ptr_next;

  DLINK_FOREACH_SAFE(ptr, ptr_next, chptr->invites.head)
    del_invite(chptr, ptr->data);

  free_channel_list(&chptr->banlist);
  free_channel_list(&chptr->exceptlist);
  free_channel_list(&chptr->invexlist);
  free_topic(chptr);
  chptr->banlist.head = chptr->exceptlist.head = chptr->invexlist.head = NULL;
  chptr->banlist.tail = chptr->exceptlist.tail = chptr->invexlist.tail = NULL;
  dlinkDelete(&chptr->node, &global_channel_list);
  hash_del_channel(chptr);
  BlockHeapFree(channel_heap, chptr);
}

/* inputs       - pointer to client struct requesting names
 *              - pointer to channel block
 *              - show ENDOFNAMES numeric or not
 *                (don't want it with /names with no params)
 * side effects - lists all names on given channel
 */
void
channel_member_names(struct Client *source_p, struct Channel *chptr,
                     int show_eon, int type)
{
  struct Client *target_p;
  struct Membership *ms;
  dlink_node *ptr;
  char lbuf[BUFSIZE + 1], *t, *start;
  int tlen, is_member;

  if (ShowChannel(source_p, chptr) || (type == 1))
  {
    is_member = IsMember(source_p, chptr);
    t = lbuf + ircsprintf(lbuf, form_str(RPL_NAMREPLY),
                          me.name, source_p->name,
			  channel_pub_or_secret(chptr),
			  chptr->name);
    start = t;
    tlen = 0;

    DLINK_FOREACH(ptr, chptr->members.head)
    {
      ms = ptr->data;
      target_p = ms->client_p;

      if (type == 0)
        if (IsInvisible(target_p) && !is_member)
          continue;

      tlen = strlen(target_p->name) + 1;  /* nick + space */
      if (ms->flags & (CHFL_CHANOP | CHFL_HALFOP | CHFL_VOICE))
        tlen++;
      if (t + tlen - lbuf > BUFSIZE)
      {
        *(t - 1) = '\0';
	sendto_one(source_p, "%s", lbuf);
	t = start;
      }

      t += ircsprintf(t, "%s%s ", get_member_status(ms, 0),
		      target_p->name);
    }

    if (tlen != 0)
      sendto_one(source_p, "%s", lbuf);
  }

  if (show_eon)
    sendto_one(source_p, form_str(RPL_ENDOFNAMES),
               me.name, source_p->name, chptr->name);
}

/* inputs 	- pointer to channel
 * output       - string pointer "=" if public, "@" if secret else "*"
 */
static const char *
channel_pub_or_secret(struct Channel *chptr)
{
  if (PubChannel(chptr))
    return("=");
  else if (SecretChannel(chptr))
    return("@");
  return("*");
}

/* inputs       - pointer to channel block
 *              - pointer to client to add invite to
 * side effects - adds client to invite list
 */
void
add_invite(struct Channel *chptr, struct Client *who)
{
  del_invite(chptr, who);
  /* delete last link in chain if the list is max length */
  if (dlink_list_length(&who->user->invited) >=
      ConfigChannel.max_chans_per_user)
    del_invite(chptr, who);

  /* add client to channel invite list */
  dlinkAdd(who, make_dlink_node(), &chptr->invites);

  /* add channel to the end of the client invite list */
  dlinkAdd(chptr, make_dlink_node(), &who->user->invited);
}

/* inputs       - pointer to Channel struct
 *              - pointer to client to remove invites from
 * side effects - Delete Invite block from channel invite list
 *                and client invite list
 */
void
del_invite(struct Channel *chptr, struct Client *who)
{
  dlink_node *ptr;

  DLINK_FOREACH(ptr, chptr->invites.head)
  {
    if (ptr->data == who)
    {
      dlinkDelete(ptr, &chptr->invites);
      free_dlink_node(ptr);
      break;
    }
  }

  DLINK_FOREACH(ptr, who->user->invited.head)
  {
    if (ptr->data == chptr)
    {
      dlinkDelete(ptr, &who->user->invited);
      free_dlink_node(ptr);
      break;
    }
  }
}

/* inputs       - pointer to struct Membership
 *              - YES if we can combine different flags
 * output       - string either @, +, % or "" depending on whether
 *                chanop, voiced or user
 */
const char *
get_member_status(struct Membership *ms, int combine)
{
  static char buffer[4];
  char *p;

  if (ms == NULL)
    return("");
  p = buffer;

  if (ms->flags & CHFL_CHANOP)
  {
    if (!combine)
      return "@";
    *p++ = '@';
  }
  if (ms->flags & CHFL_HALFOP)
  {
    if (!combine)
      return "%";
    *p++ = '%';
  }
  if (ms->flags & CHFL_VOICE)
    *p++ = '+';
  *p = '\0';

  return(buffer);
}

/* inputs       - pointer to channel block
 * output       - returns an int 0 if not banned,
 *                CHFL_BAN if banned
 */
int
is_banned(struct Channel *chptr, struct Client *who)
{
  char src_host[NICKLEN + USERLEN + HOSTLEN + 6];
  char src_iphost[NICKLEN + USERLEN + HOSTLEN + 6];

  if (!IsPerson(who))
    return(0);

  ircsprintf(src_host,"%s!%s@%s", who->name, who->username, who->host);
  ircsprintf(src_iphost,"%s!%s@%s", who->name, who->username,
	     who->localClient->sockhost);

  return(check_banned(chptr, src_host, src_iphost));
}

/* inputs       - pointer to channel block
 *              - pointer to client to check access fo
 *              - pointer to pre-formed nick!user@host
 *              - pointer to pre-formed nick!user@ip
 * output       - returns an int 0 if not banned,
 *                CHFL_BAN if banned
 */
static int
check_banned(struct Channel *chptr, const char *s, const char *s2)
{
  dlink_node *ban;
  dlink_node *except;
  struct Ban *actualBan = NULL;
  struct Ban *actualExcept = NULL;

  DLINK_FOREACH(ban, chptr->banlist.head)
  {
    actualBan = ban->data;

    if (match(actualBan->banstr,  s) || 
    	match(actualBan->banstr, s2) ||
        match_cidr(actualBan->banstr, s2))
      break;
    else
      actualBan = NULL;
  }

  if (actualBan != NULL)
  {
    DLINK_FOREACH(except, chptr->exceptlist.head)
    {
      actualExcept = except->data;

      if (match(actualExcept->banstr,  s) || 
          match(actualExcept->banstr, s2) ||
          match_cidr(actualExcept->banstr, s2))
      {
        return(CHFL_EXCEPTION);
      }
    }
  }

  return((actualBan ? CHFL_BAN : 0));
}

int
can_join(struct Client *source_p, struct Channel *chptr, const char *key)
{
  dlink_node *lp;
  dlink_node *ptr;
  struct Ban *invex = NULL;
  char src_host[NICKLEN + USERLEN + HOSTLEN + 6];
  char src_iphost[NICKLEN + USERLEN + HOSTLEN + 6];

  ircsprintf(src_host, "%s!%s@%s", source_p->name, source_p->username,
             source_p->host);
  ircsprintf(src_iphost, "%s!%s@%s", source_p->name, source_p->username,
	     source_p->localClient->sockhost);

  if ((check_banned(chptr, src_host, src_iphost)) == CHFL_BAN)
    return(ERR_BANNEDFROMCHAN);

  if (chptr->mode.mode & MODE_INVITEONLY)
  {
    DLINK_FOREACH(lp, source_p->user->invited.head)
      if (lp->data == chptr)
        break;

    if (lp == NULL)
    {
      DLINK_FOREACH(ptr, chptr->invexlist.head)
      {
        invex = ptr->data;

        if (match(invex->banstr, src_host) || match(invex->banstr, src_iphost) ||
            match_cidr(invex->banstr, src_iphost))
          break;
      }
      if (ptr == NULL)
        return(ERR_INVITEONLYCHAN);
    }
  }

  if (*chptr->mode.key && (EmptyString(key) || irccmp(chptr->mode.key, key)))
    return(ERR_BADCHANNELKEY);

  if (chptr->mode.limit && dlink_list_length(&chptr->members) >=
      chptr->mode.limit)
    return(ERR_CHANNELISFULL);

  return(0);
}

int
has_member_flags(struct Membership *ms, unsigned int flags)
{
  if (ms != NULL)
    return(ms->flags & flags);
  return(0);
}

struct Membership *
find_channel_link(struct Client *client_p, struct Channel *chptr)
{
  dlink_node *ptr;

  if (!IsClient(client_p))
    return(NULL);

  DLINK_FOREACH(ptr, client_p->user->channel.head)
    if (((struct Membership *)ptr->data)->chptr == chptr)
      return((struct Membership *)ptr->data);

  return(NULL);
}

/* inputs       - pointer to channel
 *              - pointer to client
 * outputs      - CAN_SEND_OPV if op or voiced on channel
 *              - CAN_SEND_NONOP if can send to channel but is not an op
 *                CAN_SEND_NO if they cannot send to channel
 *                Just means they can send to channel.
 */
int
can_send(struct Channel *chptr, struct Client *source_p)
{
  struct Membership *ms;

  if (IsServer(source_p))
    return(CAN_SEND_OPV);

  if (MyClient(source_p) && (find_cjupe(chptr->name) &&
      !(IsOper(source_p)) && ConfigFileEntry.oper_pass_jupes))
    return(CAN_SEND_NO);

  ms = find_channel_link(source_p, chptr);

  if ((ms != NULL) && ms->flags & (CHFL_CHANOP|CHFL_HALFOP|CHFL_VOICE))
    return(CAN_SEND_OPV);

  if (chptr->mode.mode & MODE_MODERATED)
    return(CAN_SEND_NO);

  if (ConfigChannel.quiet_on_ban && MyClient(source_p) &&
      (is_banned(chptr, source_p) == CHFL_BAN))
    return(CAN_SEND_NO);

  if (chptr->mode.mode & MODE_NOPRIVMSGS && ms == NULL)
    return(CAN_SEND_NO);

  if (chptr->mode.mode & MODE_NOCONTROL && ms == NULL)
    return(CAN_SEND_NO);

  if (chptr->mode.mode & MODE_NOCTCP && ms == NULL)
    return(CAN_SEND_NO);

  return(CAN_SEND_NONOP);
}

int
can_send_part(struct Membership *member, struct Channel *chptr,
              struct Client *source_p, const char *text)
{
  if (chptr->mode.mode & MODE_NOCONTROL)
    if (hascontrol(text))
      return(CAN_SEND_NO);

  if (has_member_flags(member, CHFL_CHANOP|CHFL_HALFOP))
    return(CAN_SEND_OPV);

  if (chptr->mode.mode & MODE_MODERATED)
    return(CAN_SEND_NO);

  if (ConfigChannel.quiet_on_ban && MyClient(source_p) &&
      (is_banned(chptr, source_p) == CHFL_BAN))
    return(CAN_SEND_NO);

  return(CAN_SEND_NONOP);
}

/* side effects - compares usercount and servercount against their split
 *                values and adjusts splitmode accordingly
 */
void
check_splitmode(void *n)
{
  if (splitchecking && (ConfigChannel.no_join_on_split ||
      ConfigChannel.no_create_on_split))
  {
    const unsigned int server = dlink_list_length(&global_serv_list);

    if (!splitmode && ((server < split_servers) || (Count.total < split_users)))
    {
      splitmode = 1;

      sendto_realops_flags(UMODE_ALL,L_ALL,
                         "Network split, activating splitmode");
      eventAddIsh("check_splitmode", check_splitmode, NULL, 10);
    }
    else if (splitmode && (server > split_servers) && (Count.total > split_users))
    {
      splitmode = 0;
    
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Network rejoined, deactivating splitmode");
      eventDelete(check_splitmode, NULL);
    }
  }
}

/* inputs       - Channel to allocate a new topic for
 * side effects - Allocates a new topic
 */
static void
allocate_topic(struct Channel *chptr)
{
  void *ptr;

  if (chptr == NULL)
    return;

  ptr = BlockHeapAlloc(topic_heap);  
  /* Basically we allocate one large block for the topic and
   * the topic info.  We then split it up into two and shove it
   * in the chptr 
   */
  chptr->topic       = ptr;
  chptr->topic_info  = (char *)ptr + TOPICLEN+1;
  *chptr->topic      = '\0';
  *chptr->topic_info = '\0';
}

void
free_topic(struct Channel *chptr)
{
  void *ptr;
  
  if (chptr == NULL || chptr->topic == NULL)
    return;

  /* This is safe for now - If you change allocate_topic you
   * MUST change this as well
   */
  ptr = chptr->topic; 
  BlockHeapFree(topic_heap, ptr);    
  chptr->topic      = NULL;
  chptr->topic_info = NULL;
}

/* Sets the channel topic */
void
set_channel_topic(struct Channel *chptr, const char *topic,
                  const char *topic_info, time_t topicts)
{
  if (strlen(topic) > 0)
  {
    if (chptr->topic == NULL)
      allocate_topic(chptr);

    strlcpy(chptr->topic, topic, TOPICLEN+1);
    strlcpy(chptr->topic_info, topic_info, USERHOST_REPLYLEN);
    chptr->topic_time = topicts; 
  }
  else
  {
    if (chptr->topic != NULL)
      free_topic(chptr);

    chptr->topic_time = 0;
  }
}

void
send_tburst(struct Client *client_p, struct Channel *chptr)
{
  sendto_one(client_p, ":%s TBURST %ld %s %ld %s :%s",
             me.name, (unsigned long)chptr->channelts, chptr->name,
             (unsigned long)chptr->topic_time, chptr->topic_info,
             chptr->topic);
}

int hascontrol(const char *text)
{
  if (text == NULL)
    return(0);

  if(strchr(text, '\002') || strchr(text, '\003') || strchr(text, '\022') || strchr(text, '\031'))
    return(1);

  return(0);
}
