/* dynlink.c: A module loader.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: dynlink.c,v 1.25 2005/03/28 04:18:16 musirc Exp $
 */

#include "tools.h"
#include "istring.h"
#include "client.h"
#include "modules.h"
#include "log.h"
#include "send.h"

#ifndef RTLD_NOW
#define RTLD_NOW RTLD_LAZY /* openbsd deficiency */
#endif

extern dlink_list mod_list;
extern int num_mods;
static char unknown_ver[] = "<unknown>";

#if !defined(HAVE_SHL_LOAD) && !defined(HAVE_DLFUNC)
typedef void (*__function_p)(void);

static __function_p
dlfunc(void *myHandle, const char *functionName)
{
  void *symbolp;

  symbolp = dlsym(myHandle, functionName);
  return((__function_p)(uintptr_t)symbolp);
}
#endif

#ifdef HAVE_MACH_O_DYLD_H
#include <mach-o/dyld.h>

#ifndef HAVE_DLOPEN
#ifndef	RTLD_LAZY
#define RTLD_LAZY 2185 /* built-in dl*(3) don't care */
#endif

void undefinedErrorHandler(const char *);
NSModule multipleErrorHandler(NSSymbol, NSModule, NSModule);
void linkEditErrorHandler(NSLinkEditErrors, int, const char *, const char *);
char *dlerror(void);
void *dlopen(char *, int);
int dlclose(void *);
void *dlsym(void *, char *);

static int firstLoad = TRUE;
static int myDlError;
static const char *myErrorTable[] =
{
  "Loading file as object failed\n",
  "Loading file as object succeeded\n",
  "Not a valid shared object\n",
  "Architecture of object invalid on this architecture\n",
  "Invalid or corrupt image\n",
  "Could not access object\n",
  "NSCreateObjectFileImageFromFile failed\n",
  NULL
};

void
undefinedErrorHandler(const char *symbolName)
{
  sendto_realops_flags(UMODE_ALL, L_ALL, "Undefined symbol: %s", symbolName);
  ilog(WARN, "Undefined symbol: %s", symbolName);
  return;
}

NSModule
multipleErrorHandler(NSSymbol s, NSModule old, NSModule new)
{
  sendto_realops_flags(UMODE_ALL, L_ALL, "Symbol `%s' found in `%s' and `%s'",
                       NSNameOfSymbol(s), NSNameOfModule(old), NSNameOfModule(new));
  ilog(WARN, "Symbol `%s' found in `%s' and `%s'", NSNameOfSymbol(s),
       NSNameOfModule(old), NSNameOfModule(new));
  /* We return which module should be considered valid, I believe */
  return(new);
}

void
linkEditErrorHandler(NSLinkEditErrors errorClass, int errnum,
                     const char *fileName, const char *errorString)
{
  sendto_realops_flags(UMODE_ALL, L_ALL, "Link editor error: %s for %s",
                       errorString, fileName);
  ilog(WARN, "Link editor error: %s for %s", errorString, fileName);
  return;
}

char
*dlerror(void)
{
  return(myDlError == NSObjectFileImageSuccess ? NULL : myErrorTable[myDlError % 7]);
}

void
*dlopen(char *filename, int unused)
{
  NSObjectFileImage myImage;
  NSModule myModule;

  if (firstLoad)
  {
    /* If we are loading our first symbol (huzzah!) we should go ahead
     * and install link editor error handling!
     */
    NSLinkEditErrorHandlers linkEditorErrorHandlers;

    linkEditorErrorHandlers.undefined = undefinedErrorHandler;
    linkEditorErrorHandlers.multiple  = multipleErrorHandler;
    linkEditorErrorHandlers.linkEdit  = linkEditErrorHandler;
    NSInstallLinkEditErrorHandlers(&linkEditorErrorHandlers);
    firstLoad = FALSE;
  }

  myDlError = NSCreateObjectFileImageFromFile(filename, &myImage);

  if (myDlError != NSObjectFileImageSuccess)
    return(NULL);

  myModule = NSLinkModule(myImage, filename, NSLINKMODULE_OPTION_PRIVATE);
  return((void *)myModule);
}

int
dlclose(void *myModule)
{
  NSUnLinkModule(myModule, FALSE);
  return(0);
}

void
*dlsym(void *myModule, char *mySymbolName)
{
  NSSymbol mySymbol;

  mySymbol = NSLookupSymbolInModule((NSModule)myModule, mySymbolName);
  return NSAddressOfSymbol(mySymbol);
}
#endif
#endif

/* inputs	- name of module to unload
 *		- 1 to say modules unloaded, 0 to not
 * output	- 0 if successful, -1 if error
 * side effects	- module is unloaded
 */
int
unload_one_module(char *name, int warn)
{
  dlink_node *ptr;
  struct module *modp;

  if ((ptr = findmodule_byname(name)) == NULL)
    return(-1);

  modp = ptr->data;

  if (modp->modremove) (*(modp->modremove))();

#ifdef HAVE_SHL_LOAD
    /* shl_* and friends have a slightly different format than dl*. But it does not
     * require creation of a totally new modules.c, instead proper usage of
     * defines solve this case. -TimeMr14C
     */
  shl_unload((shl_t) & (modp->address));
#else
  /* We use FreeBSD's dlfunc(3) interface, or fake it as we
   * used to here if it isn't there.  The interface should
   * be standardised some day, and so it eventually will be
   * providing something guaranteed to do the right thing here.
   *          -jmallett
   */
  dlclose(modp->address);
#endif
  dlinkDelete(ptr, &mod_list);
  MyFree(modp->name);
  MyFree(modp);

  if (num_mods != 0)
    num_mods--;

  if (warn == 1)
  {
    ilog(INFO, "Module %s unloaded", name);
    sendto_realops_flags(UMODE_ALL, L_ALL,"Module %s unloaded", name);
  }

  return(0);
}

/* inputs	- path name of module, int to notice, int of core
 * output	- -1 if error 0 if success
 * side effects - loads a module if successful
 */
int
load_a_module(char *path, int warn, int core)
{
#ifdef HAVE_SHL_LOAD
  shl_t tmpptr;
#else
  void *tmpptr = NULL;
#endif
  char *mod_basename;
  void (*initfunc)(void) = NULL;
  void (*mod_deinit)(void) = NULL;
  char **verp, *ver;
  struct module *modp;

  mod_basename = basename(path);

  if (findmodule_byname(mod_basename) != NULL)
    return(1);

#ifdef HAVE_SHL_LOAD
  tmpptr = shl_load(path, BIND_IMMEDIATE, NULL);
#else
  tmpptr = dlopen(path, RTLD_NOW);
#endif

  if (tmpptr == NULL)
  {
#ifdef HAVE_SHL_LOAD
    const char *err = strerror(errno);
#else
    const char *err = dlerror();
#endif
    sendto_realops_flags(UMODE_ALL, L_ALL, "Error loading module %s: %s",
                         mod_basename, err);
    ilog(WARN, "Error loading module %s: %s", mod_basename, err);
    return(-1);
  }

#ifdef HAVE_SHL_LOAD
  if (shl_findsym(&tmpptr, "_modinit", TYPE_UNDEFINED, (void *)&initfunc) == -1)
  {
    if (shl_findsym(&tmpptr, "__modinit", TYPE_UNDEFINED, (void *)&initfunc) == -1)
    {
      ilog(WARN, "Module %s has no _modinit() function", mod_basename);
      sendto_realops_flags(UMODE_ALL, L_ALL, "Module %s has no _modinit() function",
                           mod_basename);
      shl_unload(tmpptr);
      return(-1);
    }
  }

  if (shl_findsym(&tmpptr, "_moddeinit", TYPE_UNDEFINED, (void *)&mod_deinit) == -1)
  {
    if (shl_findsym(&tmpptr, "__moddeinit", TYPE_UNDEFINED, (void *)&mod_deinit) == -1)
    {
      ilog(WARN, "Module %s has no _moddeinit() function", mod_basename);
      sendto_realops_flags(UMODE_ALL, L_ALL, "Module %s has no _moddeinit() function",
                           mod_basename);
      /* this is a soft error.  we're allowed not to have one, i guess.
       * (judging by the code in unload_one_module() */
      mod_deinit = NULL;
    }
  }

  if (shl_findsym(&tmpptr, "_version", TYPE_UNDEFINED, &verp) == -1)
  {
    if (shl_findsym(&tmpptr, "__version", TYPE_UNDEFINED, &verp) == -1)
      ver = unknown_ver;
    else
      ver = *verp;
  }
  else
    ver = *verp;
#else
  if ((initfunc = (void(*)(void))dlfunc(tmpptr, "_modinit")) == NULL &&
      (initfunc = (void(*)(void))dlfunc(tmpptr, "__modinit")) == NULL)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "Module %s has no _modinit() function",
                         mod_basename);
    ilog(WARN, "Module %s has no _modinit() function", mod_basename);
    dlclose(tmpptr);
    return(-1);
  }

  mod_deinit = (void(*)(void))dlfunc(tmpptr, "_moddeinit");

  if (mod_deinit == NULL && (mod_deinit = (void(*)(void))dlfunc(tmpptr, "__moddeinit")) == NULL)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL, "Module %s has no _moddeinit() function",
                         mod_basename);
    ilog (WARN, "Module %s has no _moddeinit() function", mod_basename);
    /* blah blah soft error, see above. */
    mod_deinit = NULL;
  }
  verp = (char **)dlsym(tmpptr, "_version");

  if (verp == NULL && (verp = (char **)dlsym(tmpptr, "__version")) == NULL)
    ver = unknown_ver;
  else
    ver = *verp;
#endif

  modp = MyMalloc(sizeof(struct module));
  modp->address = tmpptr;
  modp->version = ver;
  modp->core = core;
  modp->modremove = mod_deinit;
  DupString(modp->name, mod_basename);
  dlinkAdd(modp, &modp->node, &mod_list);
  num_mods++;

  initfunc();

  if (warn == 1)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "Module %s [version: %s] loaded at 0x%lx",
                         mod_basename, ver, (unsigned long)tmpptr);
    ilog(WARN, "Module %s [version: %s] loaded at 0x%lx",
         mod_basename, ver, (unsigned long)tmpptr);
  }

  return(0);
}
