/* hash.c: Maintains hashtables.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: hash.c,v 1.85 2005/01/27 11:48:04 musirc Exp $
 */

#include "stdinc.h"
#include "config.h"
#include "client.h"
#include "list.h"
#include "hash.h"
#include "jupe.h"
#include "userhost.h"
#include "ircd.h"
#include "numeric.h"
#include "send.h"
#include "istring.h"
#include "channel.h"
#include "channel_mode.h"
#include "user.h"

extern BlockHeap *channel_heap;
static BlockHeap *userhost_heap, *namehost_heap;
static struct UserHost *find_or_add_userhost(const char *);

/* This hash function returns *exactly* N%HASHSIZE, where 'N'
 * is the string itself (included the trailing '\0') seen as
 * a baseHASHSHIFT number whose "digits" are the bytes of the
 * number mapped through a "weight" transformation that gives
 * the same "weight" to caseless-equal chars, example:
 *
 * Hashing the string "Nick\0" the result will be:
 * N  i  c  k \0
 * |  |  |  |  `--->  ( (hash_weight('\0') * (HASHSHIFT**0) +
 * |  |  |  `------>    (hash_weight('k')  * (HASHSHIFT**1) +
 * |  |  `--------->    (hash_weight('c')  * (HASHSHIFT**2) +
 * |  `------------>    (hash_weight('i')  * (HASHSHIFT**3) +
 * `--------------->    (hash_weight('N')  * (HASHSHIFT**4)   ) % HASHSIZE
 *
 * It's actually a lot similar to a base transformation of the
 * text representation of an integer.
 * Looking at it this way seems slow and requiring unlimited integer
 * precision, but we actually do it with a *very* fast loop, using only
 * short integer arithmetic and by means of two memory accesses and
 * 3 additions per each byte processed.. and nothing else, as a side
 * note the distribution of real nicks over the hash table of this
 * function is about 3 times better than the previous one, and the
 * hash function itself is about 25% faster with a "normal" HASHSIZE
 * (it gets slower with larger ones and faster for smallest ones
 * because the hash table size affect the size of some maps and thus
 * the effectiveness of RAM caches while accesing them).

 * The first prime bigger than the maximum character code (255) */
#define HASHSHIFT 257
#define HASHMAPSIZE (HASHSIZE + HASHSHIFT + 1)

/* We need a first function that, given an integer h between 0 and
 * HASHSIZE+HASHSHIFT, returns ( (h * HASHSHIFT) % HASHSIZE ) )
 * We'll map this function in this table
 */
static unsigned int hash_map[HASHMAPSIZE];

/* Then we need a second function that "maps" a char to its weitgh,
 * changed to a table this one too, with this macro we can use a char
 * as index and not care if it is signed or not, no.. this will not
 * cause an addition to take place at each access, trust me, the
 * optimizer takes it out of the actual code and passes "label+shift"
 * to the linker, and the linker does the addition :)
 */
static unsigned int hash_weight_table[CHAR_MAX - CHAR_MIN + 1];
#define hash_weight(ch) hash_weight_table[ch - CHAR_MIN]

/* The actual hash tables, both MUST be of the same HASHSIZE, variable
 * size tables could be supported but the rehash routine should also
 * rebuild the transformation maps, I kept the tables of equal size
 * so that I can use one hash function and one transformation map
 */
static struct Client *clientTable[HASHSIZE];
static struct Channel *channelTable[HASHSIZE];
static struct UserHost *userhostTable[HASHSIZE];
static struct JupeChannel *jupechannelTable[HASHSIZE];

/* side effects - Initialize the maps used by hash
 *                functions and clear the tables
 */
void
init_hash(void)
{
  int i, weight_key;
  unsigned long l, m;

  /* Default the userhost/namehost sizes to CLIENT_HEAP_SIZE for now,
   * should be a good close approximation anyway
   */
  userhost_heap = BlockHeapCreate(sizeof(struct UserHost), CLIENT_HEAP_SIZE);
  namehost_heap = BlockHeapCreate(sizeof(struct NameHost), CLIENT_HEAP_SIZE);
  weight_key = rand() % 256;  /* better than nothing --adx */

  /* Here is to what we "map" a char before working on it */
  for (i = CHAR_MIN; i <= CHAR_MAX; i++)
    hash_weight(i) = ((unsigned int) ToLower(i)) ^ weight_key;

  /* Clear the hash tables first */
  for (l = 0; l < HASHSIZE; l++)
  {
    clientTable[l] = NULL;
    channelTable[l] = NULL;
    userhostTable[l] = NULL;
    jupechannelTable[l] = NULL;
  }

  /* And this is our hash-loop "transformation" function,
   * basically it will be hash_map[x] == ((x*HASHSHIFT)%HASHSIZE)
   * defined for 0<=x<=(HASHSIZE+HASHSHIFT)
   */
  for (m = 0; m < (unsigned long) HASHMAPSIZE; m++)
  {
    l = m;
    l *= (unsigned long) HASHSHIFT;
    l &= HASHSIZE - 1;
    hash_map[m] = (unsigned int) l;
  }
}

/* These are the actual hash functions, since they are static
 * and very short any decent compiler at a good optimization level
 * WILL inline these in the following functions
 */
static unsigned int
strhash(const char *n)
{
  unsigned int hash = 0;

  while (*n != '\0')
    hash = hash_map[hash + hash_weight(*n++)];

  return(hash);
}

/* Optimization note: in these functions I supposed that the CSE optimization
 * (Common Subexpression Elimination) does its work decently, this means that
 * I avoided introducing new variables to do the work myself and I did let
 * the optimizer play with more free registers, actual tests proved this
 * solution to be faster than doing things like tmp2=tmp->hnext... and then
 * use tmp2 myself which would have given less freedom to the optimizer.
 * inputs       - pointer to client
 * side effects - Adds a client's name in the proper hash linked
 *                list, can't fail, client_p must have a non-null
 *                name or expect a coredump, the name is infact
 *                taken from client_p->name
 */
void
hash_add_client(struct Client *client_p)
{
  unsigned int hashv = strhash(client_p->name);

  client_p->hnext = clientTable[hashv];
  clientTable[hashv] = client_p;
}

/* hash_add_channel()
 * inputs       - pointer to channel
 * side effects - Adds a channel's name in the proper hash linked
 *                list, can't fail. chptr must have a non-null name
 *                or expect a coredump. As before the name is taken
 *                from chptr->name, we do hash its entire lenght
 *                since this proved to be statistically faster
 */
void
hash_add_channel(struct Channel *chptr)
{
  unsigned int hashv = strhash(chptr->name);

  chptr->hnextch = channelTable[hashv];
  channelTable[hashv] = chptr;
}

void
hash_add_jupe(struct JupeChannel *chptr)
{
  unsigned int hashv = strhash(chptr->name);

  chptr->hnext = jupechannelTable[hashv];
  jupechannelTable[hashv] = chptr;
}

void
hash_add_userhost(struct UserHost *userhost)
{
  unsigned int hashv = strhash(userhost->host);

  userhost->next = userhostTable[hashv];
  userhostTable[hashv] = userhost;
}

/* inputs       - pointer to client
 * side effects - Removes a Client's name from the hash linked list
 */
void
hash_del_client(struct Client *client_p)
{
  unsigned int hashv = strhash(client_p->name);
  struct Client *tmp = clientTable[hashv];

  if (tmp != NULL)
  {
    if (tmp == client_p)
    {
      clientTable[hashv] = client_p->hnext;
      client_p->hnext = client_p;
    }
    else
    {
      while (tmp->hnext != client_p)
      {
        if ((tmp = tmp->hnext) == NULL)
          return;
      }

      tmp->hnext = tmp->hnext->hnext;
      client_p->hnext = client_p;
    }
  }
}

/* inputs       - pointer to userhost
 * side effects - Removes a userhost from the hash linked list
 */
void
hash_del_userhost(struct UserHost *userhost)
{
  unsigned int hashv = strhash(userhost->host);
  struct UserHost *tmp = userhostTable[hashv];

  if (tmp != NULL)
  {
    if (tmp == userhost)
    {
      userhostTable[hashv] = userhost->next;
      userhost->next = userhost;
    }
    else
    {
      while (tmp->next != userhost)
      {
        if ((tmp = tmp->next) == NULL)
          return;
      }
      tmp->next = tmp->next->next;
      userhost->next = userhost;
    }
  }
}

/* inputs       - pointer to client
 * side effects - Removes the channel's name from the corresponding
 *                hash linked list
 */
void
hash_del_channel(struct Channel *chptr)
{
  unsigned int hashv = strhash(chptr->name);
  struct Channel *tmp = channelTable[hashv];

  if (tmp != NULL)
  {
    if (tmp == chptr)
    {
      channelTable[hashv] = chptr->hnextch;
      chptr->hnextch = chptr;
    }
    else
    {
      while (tmp->hnextch != chptr)
      {
        if ((tmp = tmp->hnextch) == NULL)
          return;
      }
      tmp->hnextch = tmp->hnextch->hnextch;
      chptr->hnextch = chptr;
    }
  }
}

void
hash_del_jupe(struct JupeChannel *chptr)
{
  unsigned int hashv = strhash(chptr->name);
  struct JupeChannel *tmp = jupechannelTable[hashv];

  if (tmp != NULL)
  {
    if (tmp == chptr)
    {
      jupechannelTable[hashv] = chptr->hnext;
      chptr->hnext = chptr;
    }
    else
    {
      while (tmp->hnext != chptr)
      {
        if ((tmp = tmp->hnext) == NULL)
          return;
      }
      tmp->hnext = tmp->hnext->hnext;
      chptr->hnext = chptr;
    }
  }
}

/* inputs       - pointer to name
 * side effects - New semantics: finds a client whose name is 'name'
 *                if can't find one returns NULL. If it finds one moves
 *                it to the top of the list and returns it.
 */
struct Client *
find_client(const char *name)
{
  unsigned int hashv = strhash(name);
  struct Client *client_p;

  if ((client_p = clientTable[hashv]) != NULL)
  {
    if (irccmp(name, client_p->name))
    {
      struct Client *prev;

      while (prev = client_p, (client_p = client_p->hnext) != NULL)
      {
        if (!irccmp(name, client_p->name))
        {
          prev->hnext = client_p->hnext;
          client_p->hnext = clientTable[hashv];
          clientTable[hashv] = client_p;
          break;
        }
      }
    }
  }
  return(client_p);
}

/* Whats happening in this next loop ? Well, it takes a name like
 * foo.bar.edu and proceeds to earch for *.edu and then *.bar.edu.
 * This is for checking full server names against masks although
 * it isnt often done this way in lieu of using matches().
 */
static struct Client *
hash_find_masked_server(const char *name)
{
  char buf[HOSTLEN + 1], *p = buf, *s;
  struct Client *server;

  if ('*' == *name || '.' == *name)
    return(NULL);

  strlcpy(buf, name, sizeof(buf));

  while ((s = strchr(p, '.')) != NULL)
  {
    *--s = '*';

    if ((server = find_client(s)) != NULL)
      return(server);
    p = s + 2;
  }

  return(NULL);
}

struct Client *
find_server(const char *name)
{
  unsigned int hashv = strhash(name);
  struct Client *client_p = NULL;

  if ((client_p == NULL) && (client_p = clientTable[hashv]) != NULL)
  {
    if ((!IsServer(client_p) && !IsMe(client_p)) ||
        irccmp(name, client_p->name))
    {
      struct Client *prev;

      while (prev = client_p, (client_p = client_p->hnext) != NULL)
      {
        if ((IsServer(client_p) || IsMe(client_p)) &&
            !irccmp(name, client_p->name))
        {
          prev->hnext = client_p->hnext;
          client_p->hnext = clientTable[hashv];
          clientTable[hashv] = client_p;
          break;
        }
      }
    }
  }
  return((client_p != NULL) ? client_p : hash_find_masked_server(name));
}

/* inputs       - pointer to name
 * side effects - New semantics: finds a channel whose name is 'name',
 *                if can't find one returns NULL, if can find it moves
 *                it to the top of the list and returns it.
 */
struct Channel *
hash_find_channel(const char *name)
{
  unsigned int hashv = strhash(name);
  struct Channel *chptr;

  if ((chptr = channelTable[hashv]) != NULL)
  {
    if (irccmp(name, chptr->name))
    {
      struct Channel *prev;

      while (prev = chptr, (chptr = chptr->hnextch) != NULL)
      {
        if (!irccmp(name, chptr->name))
        {
          prev->hnextch = chptr->hnextch;
          chptr->hnextch = channelTable[hashv];
          channelTable[hashv] = chptr;
          break;
        }
      }
    }
  }

  return(chptr);
}

/* inputs       - hash value (should be between 0 and HASHSIZE - 1)
 * returns      - pointer to first channel in channelTable[hashv]
 *                if that exists;
 *                NULL if there is no channel in that place;
 *                NULL if hashv is an invalid number.
 */
struct Channel *
hash_get_chptr(unsigned int hashv)
{
  if (hashv >= HASHSIZE)
    return NULL;

  return channelTable[hashv];
}

/* inputs       - pointer to name
 * side effects - New semantics: finds a juped channel whose name is 'name',
 *                if can't find one returns NULL, if can find it moves
 *                it to the top of the list and returns it.
 */
struct JupeChannel *
hash_find_jupe(const char *name)
{
  unsigned int hashv = strhash(name);
  struct JupeChannel *chptr;

  if ((chptr = jupechannelTable[hashv]) != NULL)
  {
    if (irccmp(name, chptr->name))
    {
      struct JupeChannel *prev;

      while (prev = chptr, (chptr = chptr->hnext) != NULL)
      {
	if (!irccmp(name, chptr->name))
        {
          prev->hnext = chptr->hnext;
          chptr->hnext = jupechannelTable[hashv];
          jupechannelTable[hashv] = chptr;
          break;
        }
      }
    }
  }
  return(chptr);
}

struct UserHost *
hash_find_userhost(const char *host)
{
  unsigned int hashv = strhash(host);
  struct UserHost *userhost;

  if ((userhost = userhostTable[hashv]))
  {
    if (irccmp(host, userhost->host))
    {
      struct UserHost *prev;

      while (prev = userhost, (userhost = userhost->next) != NULL)
      {
	if (!irccmp(host, userhost->host))
        {
          prev->next = userhost->next;
          userhost->next = userhostTable[hashv];
          userhostTable[hashv] = userhost;
          break;
        }
      }
    }
  }
  return(userhost);
}

/* inputs       - client pointer
 *              - channel name
 *              - pointer to int flag whether channel was newly created or not
 * output       - returns channel block or NULL if illegal name
 *              - also modifies *isnew
 * side effects - Get Channel block for name (and allocate a new channel
 *                block, if it didn't exist before).
 */
struct Channel *
get_or_create_channel(char *name, int *isnew)
{
  struct Channel *chptr;
  int len;

  if (EmptyString(name))
    return(NULL);

  len = strlen(name);

  if (len > CHANNELLEN)
    return(NULL);

  if ((chptr = hash_find_channel(name)) != NULL)
  {
    if (isnew != NULL)
      *isnew = 0;

    return(chptr);
  }

  if (isnew != NULL)
    *isnew = 1;

  chptr = BlockHeapAlloc(channel_heap);
  memset(chptr, 0, sizeof(struct Channel));
  strlcpy(chptr->name, name, sizeof(chptr->name));
  dlinkAdd(chptr, &chptr->node, &global_channel_list);
  chptr->channelts = CurrentTime; /* doesn't hurt to set it here */
  hash_add_channel(chptr);

  return(chptr);
}

/* inputs       - user name
 *              - hostname
 *              - int flag 1 if global, 0 if local
 *              - pointer to where global count should go
 *              - pointer to where local count should go
 *              - pointer to where identd count should go (local clients only)
 */
void
count_user_host(const char *user, const char *host, int *global_p,
                int *local_p, int *icount_p)
{
  dlink_node *ptr;
  struct UserHost *found_userhost;
  struct NameHost *nameh;

  if ((found_userhost = hash_find_userhost(host)) == NULL)
    return;

  DLINK_FOREACH(ptr, found_userhost->list.head)
  {
    nameh = ptr->data;

    if (!irccmp(user, nameh->name))
    {
      if (global_p != NULL)
        *global_p = nameh->gcount;
      if (local_p != NULL)
        *local_p  = nameh->lcount;
      if (icount_p != NULL)
        *icount_p = nameh->icount;
      return;
    }
  }
}

/* inputs       - user name
 *              - hostname
 *              - int flag 1 if global, 0 if local
 * side effects - add given user@host to hash tables
 */
void
add_user_host(char *user, const char *host, int global)
{
  dlink_node *ptr;
  struct UserHost *found_userhost;
  struct NameHost *nameh;
  int hasident = 1;

  if (*user == '~')
  {
    hasident = 0;
    user++;
  }

  if ((found_userhost = find_or_add_userhost(host)) == NULL)
    return;

  DLINK_FOREACH(ptr, found_userhost->list.head)
  {
    nameh = ptr->data;

    if (!irccmp(user, nameh->name))
    {
      if (global)
        nameh->gcount++;
      else
      {
        if (hasident)
          nameh->icount++;
        nameh->lcount++;
      }
      return;
    }
  }

  nameh = BlockHeapAlloc(namehost_heap);
  memset(nameh, 0, sizeof(struct NameHost));
  strlcpy(nameh->name, user, sizeof(nameh->name));

  if (global)
    nameh->gcount = 1;
  else
  {
    if (hasident)
      nameh->icount = 1;
    nameh->lcount = 1;
  }

  dlinkAdd(nameh, &nameh->node, &found_userhost->list);
}

/* inputs       - user name
 *              - hostname
 *              - int flag 1 if global, 0 if local
 * side effects - delete given user@host to hash tables
 */
void
delete_user_host(char *user, const char *host, int global)
{
  dlink_node *ptr, *next_ptr;
  struct UserHost *found_userhost;
  struct NameHost *nameh;
  int hasident = 1;

  if (*user == '~')
  {
    hasident = 0;
    user++;
  }

  if ((found_userhost = hash_find_userhost(host)) == NULL)
    return;

  DLINK_FOREACH_SAFE(ptr, next_ptr, found_userhost->list.head)
  {
    nameh = ptr->data;

    if (!irccmp(user, nameh->name))
    {
      if (global)
      {
        if (nameh->gcount > 0)
          nameh->gcount--;
      }
      else
      {
        if (nameh->lcount > 0)
          nameh->lcount--;
        if (hasident && (nameh->icount > 0))
          nameh->icount--;
      }
      if ((nameh->gcount == 0) && (nameh->lcount == 0))
      {
        dlinkDelete(&nameh->node, &found_userhost->list);
        BlockHeapFree(namehost_heap, nameh);
      }
      if (dlink_list_length(&found_userhost->list) == 0)
      {
        hash_del_userhost(found_userhost);
        BlockHeapFree(userhost_heap, found_userhost);
      }
      return;
    }
  }
}

/* inputs       - host name
 * side effects - find UserHost * for given host name
 */
static struct UserHost *
find_or_add_userhost(const char *host)
{
  struct UserHost *userhost;

  if ((userhost = hash_find_userhost(host)) != NULL)
    return(userhost);

  userhost = BlockHeapAlloc(userhost_heap);
  memset(userhost, 0, sizeof(struct UserHost));
  strlcpy(userhost->host, host, sizeof(userhost->host));
  hash_add_userhost(userhost);

  return(userhost);
}

/* inputs       - pointer to client to check
 * output        - 1 if client is in danger of blowing its sendq
 *                0 if it is not.
 */
static int
exceeding_sendq(struct Client *to)
{
  if (dbuf_length(&to->localClient->buf_sendq) > (get_sendq(to) / 2))
    return(1);
  else
    return(0);
}

void
free_list_task(struct ListTask *lt, struct Client *source_p)
{
  dlink_node *dl, *dln;

  DLINK_FOREACH_SAFE(dl, dln, lt->show_mask.head)
  {
    MyFree(dl->data);
    free_dlink_node(dl);
  }

  DLINK_FOREACH_SAFE(dl, dln, lt->hide_mask.head)
  {
    MyFree(dl->data);
    free_dlink_node(dl);
  }
  MyFree(lt);

  if (MyConnect(source_p))
    source_p->localClient->list_task = NULL;
}

/* inputs       - channel name
 *              - pointer to a list task
 * output       - 1 if the channel is to be displayed
 *                0 otherwise
 */
static int
list_allow_channel(const char *name, struct ListTask *lt)
{
  dlink_node *dl;

  DLINK_FOREACH(dl, lt->show_mask.head)
    if (!match(dl->data, name))
      return(0);

  DLINK_FOREACH(dl, lt->hide_mask.head)
    if (match(dl->data, name))
      return(0);

  return(1);
}

/* inputs       - client pointer to return result to
 *              - pointer to channel to list
 *              - pointer to ListTask structure
 */
static void
list_one_channel(struct Client *source_p, struct Channel *chptr,
                 struct ListTask *list_task, int type)
{
  char chmode[15] = "\0";

  if (type == 0 && (SecretChannel(chptr) && !IsMember(source_p, chptr)))
    return;

  if ((unsigned int)dlink_list_length(&chptr->members) < list_task->users_min ||
      (unsigned int)dlink_list_length(&chptr->members) > list_task->users_max ||
      (chptr->channelts != 0 && ((unsigned int)chptr->channelts < list_task->created_min ||
      (unsigned int)chptr->channelts > list_task->created_max)) ||
      (unsigned int) chptr->topic_time < list_task->topicts_min ||
      (chptr->topic_time ? (unsigned int)chptr->topic_time : UINT_MAX) >
      list_task->topicts_max)
    return;

  if (!list_allow_channel(chptr->name, list_task))
    return;

  listmode(chptr->mode.mode, chmode, (chptr->mode.limit > 0), (*chptr->mode.key));
  sendto_one(source_p, form_str(RPL_LIST), me.name, source_p->name,
             chptr->name, dlink_list_length(&chptr->members), chmode,
             chptr->topic == NULL ? "No topic" : chptr->topic);
}

/* inputs       - pointer to client requesting list
 * output       - 0/1
 * side effects - safely list all channels to source_p
 */
void
safe_list_channels(struct Client *source_p, struct ListTask *list_task,
                   int only_unmasked_channels, int type)
{
  struct Channel *chptr;

  if (!only_unmasked_channels)
  {
    int i;

    for (i = list_task->hash_index; i < HASHSIZE; i++)
    {
      if (exceeding_sendq(source_p))
      {
        list_task->hash_index = i;
        return; /* still more to do */
      }

      for (chptr = channelTable[i]; chptr; chptr = chptr->hnextch)
      {
	if (type == 0)
          list_one_channel(source_p, chptr, list_task, 0);
	else
          list_one_channel(source_p, chptr, list_task, 1);
      }
    }
  }
  else
  {
    dlink_node *dl;

    DLINK_FOREACH(dl, list_task->show_mask.head)
      if ((chptr = hash_find_channel(dl->data)) != NULL)
      {
	if (type == 0)
          list_one_channel(source_p, chptr, list_task, 0);
	else
          list_one_channel(source_p, chptr, list_task, 1);
      }
  }
  free_list_task(list_task, source_p);
  sendto_one(source_p, form_str(RPL_LISTEND), me.name, source_p->name);
}

void
listmode(unsigned int chanmodes, char modesStr[], char chlimit, char chkey)
{
  int outPos = 0;

  modesStr[outPos] = '[';
  outPos++;
  modesStr[outPos] = '+';
  outPos++;
  if (chanmodes & MODE_NOCTCP)
  {
    modesStr[outPos] = 'C';
    outPos++;
  }
  if (chanmodes & MODE_NOCONTROL)
  {
    modesStr[outPos] = 'c';
    outPos++;
  }
  if (chanmodes & MODE_INVITEONLY)
  {
    modesStr[outPos] = 'i';
    outPos++;
  }
  if (chkey)
  {
    modesStr[outPos] = 'k';
    outPos++;
  }
  if (chlimit)
  {
    modesStr[outPos] = 'l';
    outPos++;
  }
  if (chanmodes & MODE_MODERATED)
  {
    modesStr[outPos] = 'm';
    outPos++;
  }
  if (chanmodes & MODE_NOPRIVMSGS)
  {
    modesStr[outPos] = 'n';
    outPos++;
  }
  if (chanmodes & MODE_PRIVATE)
  {
    modesStr[outPos] = 'p';
    outPos++;
  }
  if (chanmodes & MODE_REG)
  {
    modesStr[outPos] = 'r';
    outPos++;
  }
  if (chanmodes & MODE_SECRET)
  {
    modesStr[outPos] = 's';
    outPos++;
  }
  if (chanmodes & MODE_TOPICLIMIT)
  {
    modesStr[outPos] = 't';
    outPos++;
  }
  modesStr[outPos] = ']';
  outPos++;
}
