/* jupe.c: Functions to jupe a nick/channel.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: jupe.c,v 1.4 2005/01/27 11:48:04 musirc Exp $
 */

#include "ircd.h"
#include "send.h"
#include "client.h"   
#include "numeric.h"
#include "jupe.h"
#include "hash.h"
#include "istring.h"
#include "config.h"

dlink_list cjupe_list = { NULL, NULL, 0 };

/* inputs	- name of channel to create jupe for
 *		- reason for jupe
 *		- flag, 1 for from ircd.conf 0 from elsehwere
 * output	- pointer to struct JupeChannel
 */
struct ConfItem *
create_cjupe(char *name, char *reason, int in_conf)
{
  struct ConfItem *conf;
  struct JupeChannel *jupe_p;

  if (name == NULL || reason == NULL)
    return(NULL);

  if (find_cjupe(name))
    return(NULL);

  if (strlen(reason) > TOPICLEN)
    reason[TOPICLEN] = '\0';

  conf = make_conf_item(CJUPE_TYPE);
  jupe_p = (struct JupeChannel *)map_to_conf(conf);
  strlcpy(jupe_p->name, name, sizeof(jupe_p->name));
  DupString(jupe_p->reason, reason);
  jupe_p->conf = in_conf;
  dlinkAdd(jupe_p, &jupe_p->node, &cjupe_list);
  hash_add_jupe(jupe_p);

  return(conf);
}

/* inputs	- name of nick to create jupe for
 *		- reason for jupe
 *		- 1 if from ircd.conf, 0 if from elsewhere
 * output	- pointer to struct JupeNick
 */
struct ConfItem *
create_njupe(char *name, char *reason, int in_conf)
{
  struct ConfItem *conf;
  struct MatchItem *jupe_p;

  if (name == NULL || reason == NULL)
    return(NULL);

  if (find_matching_name_conf(NJUPE_TYPE, name, NULL, NULL, 0))
    return(NULL);

  if (strlen(reason) > TOPICLEN)
    reason[TOPICLEN] = '\0';

  conf = make_conf_item(NJUPE_TYPE);
  jupe_p = (struct MatchItem *)map_to_conf(conf);
  DupString(conf->name, name);
  DupString(jupe_p->reason, reason);
  jupe_p->action = in_conf;

  return(conf);
}

/* side effects	- All jupes are cleared out */
void
clear_conf_jupe(void)
{
  dlink_node *ptr, *next_ptr;
  struct JupeChannel *jupe_cp;

  DLINK_FOREACH_SAFE(ptr, next_ptr, cjupe_list.head)
  {
    jupe_cp = ptr->data;
    delete_cjupe(jupe_cp);
  }
}

/* inputs	- pointer to channel jupe to delete
 * side effects	- given struct JupeChannel * is removed
 */
int
delete_cjupe(struct JupeChannel *jupe_p)
{
  struct ConfItem *conf;

  if (jupe_p == NULL)
    return(0);

  hash_del_jupe(jupe_p);
  dlinkDelete(&jupe_p->node, &cjupe_list);
  MyFree(jupe_p->reason);
  conf = unmap_conf_item(jupe_p);
  delete_conf_item(conf);

  return(1);
}

int
find_cjupe(const char *name)
{
  struct JupeChannel *jupe_p;

  if ((jupe_p = hash_find_jupe(name)) != NULL)
    return(1);

  return(0);
}

void
report_jupe(struct Client *source_p)
{
  dlink_node *ptr;
  struct ConfItem *conf;
  struct JupeChannel *jupe_cp;
  struct MatchItem *jupe_np;

  DLINK_FOREACH(ptr, cjupe_list.head)
  {
    jupe_cp = ptr->data;
    sendto_one(source_p, form_str(RPL_STATSQLINE),
               me.name, source_p->name, jupe_cp->conf ? 'Q' : 'q',
	       jupe_cp->name, jupe_cp->reason);
  }
  sendto_one(source_p, ":%s NOTICE %s :*** %lu matching cjupe%s found.",
             me.name, source_p->name, dlink_list_length(&cjupe_list),
            (dlink_list_length(&cjupe_list) == 1) ? "" : "s");

  DLINK_FOREACH(ptr, njupe_items.head)
  {
    conf = ptr->data;
    jupe_np = (struct MatchItem *)map_to_conf(conf);
    sendto_one(source_p, form_str(RPL_STATSQLINE),
               me.name, source_p->name, jupe_np->action ? 'Q' : 'q',
	       conf->name, jupe_np->reason);
  }
  sendto_one(source_p, ":%s NOTICE %s :*** %lu matching njupe%s found.",
             me.name, source_p->name, dlink_list_length(&njupe_items),
            (dlink_list_length(&njupe_items) == 1) ? "" : "s");
}

int
clean_jupe_nick(char *nick)
{
  char tmpch;
  int as = 0, q = 0, ch = 0;

  if (*nick == '-' || IsDigit(*nick))
    return(0);

  while ((tmpch = *nick++))
  {
    if (tmpch == '?')
      q++;
    else if (tmpch == '*')
      as++;
    else if (IsNickChar(tmpch))
      ch++;
    else
      return(0);
  }

  if (!ch && as)
    return(0);

  return(1);
}

/* inputs       - data to check for sufficient non-wildcard characters
 * outputs      - 1 if valid, else 0
 */
int
valid_wild_card_simple(char *data)
{
  char *p = data, tmpch;
  int nonwild = 0;

  while ((tmpch = *p++))
  {
    if (!IsMWildChar(tmpch))
      if (++nonwild >= ConfigFileEntry.min_nonwildcard_simple)
        break;
  }

  if (nonwild < ConfigFileEntry.min_nonwildcard_simple)
    return 0;
  else
    return 1;
}
