/* log.c: Logger functions.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: log.c,v 1.15 2005/01/27 11:48:05 musirc Exp $
 */

#include "stdinc.h"
#include "client.h"
#include "log.h"
#include "fileio.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "misc.h"
#include "event.h"
#include "config.h"

#define LOG_BUFSIZE 2000

static FBFILE *logFile;
static int logLevel = INIT_LOG_LEVEL;
static EVH user_log_resync;
static FBFILE *user_log_fb = NULL;

static const char *logLevelToString[] =
{
  "CRIT",
  "ERROR",
  "WARN",
  "NOTICE",
  "TRACE",
  "INFO"
};

/* open ircd logging file
 * returns true (1) if successful, false (0) otherwise
 */
static int 
open_log(const char *filename)
{
  if (!server_state.foreground)
    close(2); /* let the logfile grab fd 2 to catch stderr */

  logFile = fbopen(filename, "a");

  if (logFile == NULL)
    return(0);

  return(1);
}

static void 
write_log(const char *message)
{
  char buf[LOG_BUFSIZE];
  size_t nbytes = 0;

  if (logFile == NULL)
    return;

  nbytes = snprintf(buf, sizeof(buf), "[%s] %s\n",
           smalldate(CurrentTime), message);
  fbputs(buf, logFile, nbytes);
}
   
void
ilog(const int priority, const char *fmt, ...)
{
  char buf[LOG_BUFSIZE];
  va_list args;

  if (fmt == NULL)
    return;

  if (priority > logLevel)
    return;

  va_start(args, fmt);
  vsprintf(buf, fmt, args);
  va_end(args);

  if (ConfigLoggingEntry.use_logging)
    write_log(buf);
}
  
void
init_log(const char *filename)
{
  open_log(filename);
  eventAddIsh("user_log_resync", user_log_resync, NULL, 60);
}

void
reopen_log(const char *filename)
{
  fbclose(logFile);
  open_log(filename);
}

void
set_log_level(const int level)
{
  if (ERROR < level && level <= INFO)
    logLevel = level;
}

int
get_log_level(void)
{
  return(logLevel);
}

const char *
get_log_level_as_string(int level)
{
  if (level > INFO)
    level = INFO;
  else if (level < ERROR)
    level = ERROR;

  return(logLevelToString[level]);
}

/* inputs	- pointer to connecting client
 * side effects - Current exiting client is logged to file.
 */
void
log_user_exit(struct Client *source_p)
{
  time_t on_for = CurrentTime - source_p->firsttime;
  char linebuf[BUFSIZ];

  /* This conditional makes the logfile active only after
   * it's been created - thus logging can be turned off by
   * removing the file.
   */
  if (IsPerson(source_p))
  {
    if (user_log_fb == NULL)
    {
      if ((ConfigLoggingEntry.userlog[0] != '\0') &&
         (user_log_fb = fbopen(ConfigLoggingEntry.userlog, "r")) != NULL)
      {
        fbclose(user_log_fb);
        user_log_fb = fbopen(ConfigLoggingEntry.userlog, "a");
      }
    }
    if (user_log_fb != NULL)
    {
      size_t nbytes = ircsprintf(linebuf,
		   "%s (%3ld:%02ld:%02ld): %s!%s@%s %d/%d\n",
		   myctime(source_p->firsttime),
		   (signed long) on_for / 3600,
		   (signed long) (on_for % 3600)/60,
		   (signed long) on_for % 60,
		   source_p->name, source_p->username, source_p->host,
		   source_p->localClient->sendK,
		   source_p->localClient->receiveK);
      fbputs(linebuf, user_log_fb, nbytes);
    }
  }
}

static void
user_log_resync(void *n)
{
  if (user_log_fb != NULL)
  {
    fbclose(user_log_fb);
    user_log_fb = NULL;
  }
}

/* inputs	- pointer to client being opered up
 *              - oper name
 * side effects - foperlog is written to, if its present
 */
void
log_oper(struct Client *source_p, const char *name)
{
  if (ConfigLoggingEntry.operlog[0] == '\0')
    return;
  
  if (IsPerson(source_p))
  {
    FBFILE *oper_fb;

    if ((oper_fb = fbopen(ConfigLoggingEntry.operlog, "r")) != NULL)
    {
      fbclose(oper_fb);
      oper_fb = fbopen(ConfigLoggingEntry.operlog, "a");
    }

    if (oper_fb != NULL)
    {
      char linebuf[BUFSIZE];
      size_t nbytes = ircsprintf(linebuf,
                                 "%s OPER (%s) by (%s!%s@%s)\n",
		 myctime(CurrentTime), name, source_p->name,
                 source_p->username, source_p->host);
      fbputs(linebuf, oper_fb, nbytes);
      fbclose(oper_fb);
    }
  }
}

/* inputs       - pointer to client that failed to oper up	
 *              - oper name
 * side effects - ffailed_operlog is written to, if its present
 */
void
log_failed_oper(struct Client *source_p, const char *name)
{
  if (ConfigLoggingEntry.failed_operlog[0] == '\0')
    return;
  
  if (IsPerson(source_p))
  {
    FBFILE *oper_fb;
    if ((oper_fb = fbopen(ConfigLoggingEntry.failed_operlog, "r")) != NULL)
    {
      fbclose(oper_fb);
      oper_fb = fbopen(ConfigLoggingEntry.failed_operlog, "a");
    }

    if (oper_fb != NULL)
    {
      char linebuf[BUFSIZE];
      size_t nbytes = ircsprintf(linebuf,
                                 "%s FAILED OPER (%s) by (%s!%s@%s)\n",
		 myctime(CurrentTime), name, source_p->name,
                 source_p->username, source_p->host);
      fbputs(linebuf, oper_fb, nbytes);
      fbclose(oper_fb);
    }
  }
}
