/* parse.c: The message parser.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: parse.c,v 1.85 2005/02/03 08:18:25 musirc Exp $
 */

#include "parse.h"
#include "client.h"
#include "channel.h"
#include "handlers.h"
#include "hash.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "numeric.h"
#include "log.h"
#include "send.h"
#include "config.h"
#include "msg.h"
#include "server.h"
#include "user.h"

/* (based on orabidoo's parser code)
 * This has always just been a trie. Look at volume III of Knuth ACP
 * ok, you start out with an array of pointers, each one corresponds
 * to a letter at the current position in the command being examined.
 * so roughly you have this for matching 'trie' or 'tie'
 * 't' points -> [MessageTree *] 'r' -> [MessageTree *] -> 'i'
 *   -> [MessageTree *] -> [MessageTree *] -> 'e' and matches
 */
#define MAXPTRLEN	32
				/* Must be a power of 2, and
				 * larger than 26 [a-z]|[A-Z]
				 * its used to allocate the set
				 * of pointers at each node of the tree
				 * There are MAXPTRLEN pointers at each node.
				 * Obviously, there have to be more pointers
				 * Than ASCII letters. 32 is a nice number
				 * since there is then no need to shift
				 * 'A'/'a' to base 0 index, at the expense
				 * of a few never used pointers. For a small
				 * parser like this, this is a good compromise
				 * and does make it somewhat faster.
				 */

struct MessageTree
{
  int links; /* Count of all pointers (including msg) at this node
              * used as reference count for deletion of _this_ node. */
  struct Message *msg;
  struct MessageTree *pointers[MAXPTRLEN];
};

static struct MessageTree msg_tree;
static char *sender, *para[MAXPARA + 1], buffer[1024];
static void remove_unknown(struct Client *, char *, char *);
static void do_numeric(char[], struct Client *, struct Client *, int, char **);
static void handle_command(struct Message *, struct Client *, struct Client *, unsigned int, char **);
static void add_msg_element(struct MessageTree *, struct Message *, const char *);
static void del_msg_element(struct MessageTree *, const char *);

/* turn a string into a parc/parv pair */
static inline int
string_to_array(char *string, char *parv[MAXPARA])
{
  char *p, *buf = string;
  int x = 1;

  parv[x] = NULL;

  while (*buf == ' ') /* skip leading spaces */
    buf++;

  if (*buf == '\0') /* ignore all-space args */
    return(x);

  do
  {
    if (*buf == ':') /* Last parameter */
    {
      buf++;
      parv[x++] = buf;
      parv[x]   = NULL;
      return(x);
    }
    else
    {
      parv[x++] = buf;
      parv[x]   = NULL;

      if ((p = strchr(buf, ' ')) != NULL)
      {
        *p++ = '\0';
        buf  = p;
      }
      else
        return(x);
    }       

    while (*buf == ' ')
      buf++;

    if (*buf == '\0')
      return(x);
  } while (x < MAXPARA - 1);

  if (*p == ':')
    p++;

  parv[x++] = p;
  parv[x]   = NULL;
  return(x);
}

/* should not be called recusively by any other functions! */
void
parse(struct Client *client_p, char *pbuffer, char *bufend)
{
  struct Client *source_p = client_p;
  char *ch, *s, *numeric = 0;
  unsigned int i = 0;
  int paramcount;
  struct Message *mptr = NULL;

  if (IsDefunct(client_p))
    return;

  for (ch = pbuffer; *ch == ' '; ch++) /* skip spaces */
    /* null statement */ ;

  para[0] = client_p->name;

  if (*ch == ':')
  {
    ch++;

    /* Copy the prefix to 'sender' assuming it terminates
     * with SPACE (or NULL, which is an error, though).
     */
    sender = ch;

    if ((s = strchr(ch, ' ')) != NULL)
    {
      *s = '\0';
      s++;
      ch = s;
    }

    if (*sender && IsServer(client_p))
    {
      if ((source_p = find_person(sender)) == NULL)
        source_p = find_server(sender);

      /* Hmm! If the client corresponding to the
       * prefix is not found--what is the correct
       * action??? Now, I will ignore the message
       * (old IRC just let it through as if the
       * prefix just wasn't there...) --msa
       */
      if (source_p == NULL)
      {
        remove_unknown(client_p, sender, pbuffer);
        return;
      }

      para[0] = source_p->name;

      if (source_p->from != client_p)
      {
	exit_client(client_p, client_p, &me, "Fake prefix");
        return;
      }
    }

    while (*ch == ' ')
      ch++;
  }

  if (*ch == '\0')
    return;

  /* Extract the command code from the packet.  Point s to the end
   * of the command code and calculate the length using pointer
   * arithmetic. Note: only need length for numerics and *all*
   * numerics must have parameters and thus a space after the command
   * code. -avalon
   */
  if (*(ch + 3) == ' ' && IsDigit(*ch) && IsDigit(*(ch + 1)) && IsDigit(*(ch + 2)))
  {
    mptr = NULL;
    numeric = ch;
    paramcount = MAXPARA;
    s = ch + 3;
    *s++ = '\0';
  }
  else
  { 
    int ii = 0;

    if ((s = strchr(ch, ' ')) != NULL)
      *s++ = '\0';

    if ((mptr = find_command(ch)) == NULL)
    {
      /* Note: Give error message *only* to recognized
       * persons. It's a nightmare situation to have
       * two programs sending "Unknown command"'s or
       * equivalent to each other at full blast....
       * If it has got to person state, it at least
       * seems to be well behaving. Perhaps this message
       * should never be generated, though...  --msa
       * Hm, when is the buffer empty -- if a command
       * code has been found ?? -Armin
       */
      if (pbuffer[0] != '\0')
      {
        if (IsPerson(source_p))
          sendto_one(source_p, form_str(ERR_UNKNOWNCOMMAND),
                     me.name, source_p->name, ch);
      }
      return;
    }
    paramcount = mptr->parameters;
    ii = bufend - ((s) ? s : ch);
  }

  if (s != NULL)
    i = string_to_array(s, para);
  else
  {
    i = 0;
    para[1] = NULL;
  }

  if (mptr == NULL)
    do_numeric(numeric, client_p, source_p, i, para);
  else
    handle_command(mptr, client_p, source_p, i, para);
}

/* inputs	- pointer to message block
 *		- pointer to client
 *		- pointer to client message is from
 *		- count of number of args
 *		- pointer to argv[] array
 * output	- -1 if error from server
 */
static void
handle_command(struct Message *mptr, struct Client *client_p,
               struct Client *from, unsigned int i, char *hpara[MAXPARA])
{
  MessageHandler handler = 0;

  if (!IsRegistered(client_p))
  {
    /* if its from a possible server connection
     * ignore it.. more than likely its a header thats sneaked through
     */
    if ((IsHandshake(client_p) || IsConnecting(client_p) ||
        IsServer(client_p)) && !(mptr->flags & MFLG_UNREG))
      return;
  }

  handler = mptr->handlers[client_p->handler];

  /* check right amount of params is passed... --is */
  if (i < mptr->parameters)
  {
    if (!IsServer(client_p))
    {
      sendto_one(client_p, form_str(ERR_NEEDMOREPARAMS),
                   me.name, EmptyString(hpara[0]) ? "*" : hpara[0], mptr->cmd);
    }
    else
    {
      sendto_realops_flags(UMODE_ALL, L_ALL, 
                           "Dropping server %s due to (invalid) command '%s' "
                           "with only %d arguments (expecting %d).",
                           client_p->name, mptr->cmd, i, mptr->parameters);
      ilog(CRIT, "Insufficient parameters (%d) for command '%s' from %s.",
           i, mptr->cmd, client_p->name);
      exit_client(client_p, client_p, client_p,
                  "Not enough arguments to server command.");
    }
  }
  else
    (*handler)(client_p, from, i, hpara);
}

/* side effects - MUST MUST be called at startup ONCE before
 *                any other keyword routine is used.
 */
void
clear_tree_parse(void)
{
  memset(&msg_tree, 0, sizeof(msg_tree));
}

/* inputs	- pointer to MessageTree
 *		- pointer to Message to add for given command
 *		- pointer to current portion of command being added
 * side effects	- recursively build the Message Tree ;-)
 * The code first checks to see if its reached the end of the command
 * If so, that struct MessageTree has a msg pointer updated and the links
 * count incremented, since a msg pointer is a reference.
 * Then the code descends recursively, building the trie.
 * If a pointer index inside the struct MessageTree is NULL a new
 * child struct MessageTree has to be allocated.
 * The links (reference count) is incremented as they are created
 * in the parent.
 */
static void
add_msg_element(struct MessageTree *mtree_p, 
		struct Message *msg_p, const char *cmd)
{
  struct MessageTree *ntree_p;

  if (*cmd == '\0')
  {
    mtree_p->msg = msg_p;
    mtree_p->links++; /* Have msg pointer, so up ref count */
  }
  else
  {
    /* *cmd & (MAXPTRLEN-1) 
     * convert the char pointed to at *cmd from ASCII to an integer
     * between 0 and MAXPTRLEN.
     * Thus 'A' -> 0x1 'B' -> 0x2 'c' -> 0x3 etc.
     */
    if ((ntree_p = mtree_p->pointers[*cmd & (MAXPTRLEN-1)]) == NULL)
    {
      ntree_p = (struct MessageTree *)MyMalloc(sizeof(struct MessageTree));
      mtree_p->pointers[*cmd & (MAXPTRLEN-1)] = ntree_p;

      mtree_p->links++;/* Have new pointer, so up ref count */
    }
    add_msg_element(ntree_p, msg_p, cmd+1);
  }
}

/* inputs	- Pointer to MessageTree to delete from
 *		- pointer to command name to delete
 * side effects	- recursively deletes a token from the Message Tree ;-)
 * Well, first off, the code recursively descends into the trie
 * until it finds the terminating letter of the command being removed.
 * Once it has done that, it marks the msg pointer as NULL then
 * reduces the reference count on that allocated struct MessageTree
 * since a command counts as a reference.
 * Then it pops up the recurse stack. As it comes back up the recurse
 * The code checks to see if the child now has no pointers or msg
 * i.e. the links count has gone to zero. If its no longer used, the
 * child struct MessageTree can be deleted. The parent reference
 * to this child is then removed and the parents link count goes down.
 * Thus, we continue to go back up removing all unused MessageTree(s)
 */
static void
del_msg_element(struct MessageTree *mtree_p, const char *cmd)
{
  struct MessageTree *ntree_p;

  if ((*cmd == '\0') && (mtree_p->msg != NULL))
  {
    mtree_p->msg = NULL;
    mtree_p->links--;
  }
  else
  {
    if ((ntree_p = mtree_p->pointers[*cmd & (MAXPTRLEN-1)]) != NULL)
    {
      del_msg_element(ntree_p, cmd+1);
      if (ntree_p->links == 0)
      {
        mtree_p->pointers[*cmd & (MAXPTRLEN-1)] = NULL;
        mtree_p->links--;
        MyFree(ntree_p);
      }
    }
  }
}

/* inputs	- Pointer to command to find
 *		- Pointer to MessageTree root
 * output	- Find given command returning Message * if found NULL if not
 */
static struct Message *
msg_tree_parse(const char *cmd, struct MessageTree *root)
{
  struct MessageTree *mtree;

  for (mtree = root->pointers[(*cmd) & (MAXPTRLEN-1)]; mtree != NULL;
       mtree = mtree->pointers[(*++cmd) & (MAXPTRLEN-1)])
  {
    if (!IsAlpha(*cmd))
      return(NULL);
    if (*(cmd + 1) == '\0')
      return(mtree->msg);
  }
  return(NULL);
}

/* inputs       - pointer to struct Message
 * side effects - load this one command name
 */
void
mod_add_cmd(struct Message *msg)
{
  struct Message *found_msg;

  if (msg == NULL)
    return;

  /* command already added? */  
  if ((found_msg = msg_tree_parse(msg->cmd, &msg_tree)) != NULL)
    return;

  add_msg_element(&msg_tree, msg, msg->cmd);
}

/* inputs       - pointer to struct Message
 * side effects - unload this one command name
 */
void
mod_del_cmd(struct Message *msg)
{
  if (msg == NULL)
    return;

  del_msg_element(&msg_tree, msg->cmd);
}

/* inputs	- command name
 * output	- pointer to struct Message
 */
struct Message *
find_command(const char *cmd)
{
  return(msg_tree_parse(cmd, &msg_tree));
}

static void
remove_unknown(struct Client *client_p, char *lsender, char *lbuffer)
{
  /* Do kill if it came from a server because it means there is a ghost
   * user on the other server which needs to be removed. -avalon
   * Tell opers about this. -Taner
   * '[0-9]something'  is an ID      (KILL/SQUIT depending on its length)
   * 'nodots'          is a nickname (KILL)
   * 'no.dot.at.start' is a server   (SQUIT)
   */
  if (strchr(lsender, '.') != NULL)
    sendto_one(client_p, ":%s SQUIT %s :(Unknown prefix (%s) from %s)",
               me.name, lsender, lbuffer, client_p->name);
  else
    sendto_one(client_p, ":%s KILL %s :%s (Unknown Client)",
               me.name, lsender, me.name);
}

/*      parc    number of arguments ('sender' counted as one!)
 *      parv[0] pointer to 'sender' (may point to empty string) (not used)
 *      parv[1]..parv[parc-1]
 *              pointers to additional parameters, this is a NULL
 *              terminated list (parv[parc] == NULL).
 *      Numerics are mostly error reports. If there is something
 *      wrong with the message, just *DROP* it! Don't even think of
 *      sending back a neat error message -- big danger of creating
 *      a ping pong error message...
 */
static void
do_numeric(char numeric[], struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  struct Client *target_p;
  struct Channel *chptr;
  char *t;
  int i, tl;

  if (parc < 2 || !IsServer(source_p))
    return;

  /* Remap low number numerics. */
  if (numeric[0] == '0')
    numeric[0] = '1';

  /* Prepare the parameter portion of the message into 'buffer'.
   * (Because the buffer is twice as large as the message buffer
   * for the socket, no overflow can occur here... ...on current
   * assumptions--bets are off, if these are changed --msa)
   */
  t = buffer;

  for (i = 2; i < (parc - 1); i++)
  {
    tl = ircsprintf(t, " %s", parv[i]);
    t += tl;
  }
  ircsprintf(t," :%s", parv[parc-1]);

  if (((target_p = find_person(parv[1])) != NULL) ||
      ((target_p = find_server(parv[1])) != NULL))
  {
    if (IsMe(target_p)) 
    {
      if (atoi(numeric) != ERR_NOSUCHNICK)
        sendto_realops_flags(UMODE_ALL, L_ADMIN,
			     "*** %s(via %s) sent a %s numeric to me: %s",
			     source_p->name, client_p->name, numeric, buffer);
      return;
    }
    else if (target_p->from == client_p) 
    {
      /* This message changed direction (nick collision?)
       * ignore it.
       */
      return;
    }

    if ((atoi(numeric) == ERR_UMODEUNKNOWNFLAG) && MyClient(target_p))
      return;
    
    /* Fake it for server hiding, if its our client */
    if (ConfigServerHide.hide_servers && MyClient(target_p) && !IsOper(target_p))
      sendto_one(target_p, ":%s %s %s%s", me.name, numeric, target_p->name, buffer);
    else
      sendto_one(target_p, ":%s %s %s%s", source_p->name, numeric, target_p->name, buffer);
    return;
  }
  else if ((chptr = hash_find_channel(parv[1])) != NULL)
    sendto_channel_local(ALL_MEMBERS, chptr, ":%s %s %s %s",
			 source_p->name, numeric, chptr->name, buffer);
}

void
m_not_oper(struct Client *client_p, struct Client *source_p,
           int parc, char *parv[])
{
  sendto_one(source_p, form_str(ERR_NOPRIVILEGES),
             me.name, source_p->name);
}

void
m_unregistered(struct Client *client_p, struct Client *source_p,
               int parc, char *parv[])
{
  if (client_p->localClient->number_of_nick_changes == 0)
  {
    sendto_one(client_p, ":%s %d * %s :Register first.",
               me.name, ERR_NOTREGISTERED, source_p->name);
    client_p->localClient->number_of_nick_changes++;
  }
}

void
m_registered(struct Client *client_p, struct Client *source_p,
             int parc, char *parv[])
{
  sendto_one(client_p, form_str(ERR_ALREADYREGISTRED),   
             me.name, source_p->name); 
}

void
m_ignore(struct Client *client_p, struct Client *source_p,
         int parc, char *parv[])
{
  return;
}
