/* select.c: select() compatible network routines.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: select.c,v 1.9 2005/01/27 11:48:05 musirc Exp $
 */

#include "stdinc.h"
#include "fdlist.h"
#include "bsd.h"
#include "client.h"
#include "istring.h"
#include "ircd.h"
#include "list.h"
#include "listener.h"
#include "numeric.h"
#include "packet.h"
#include "res.h"
#include "restart.h"
#include "auth.h"
#include "config.h"
#include "log.h"
#include "server.h"
#include "send.h"

#if HARD_FDLIMIT >= FD_SETSIZE
#error HARD_FDLIMIT must be less than FD_SETSIZE (try using poll instead of select)
#endif

static fd_set select_readfds, select_writefds, tmpreadfds, tmpwritefds;
static void select_update_selectfds(int, short, PF *);

/* set and clear entries in the select array .. */ 
static void
select_update_selectfds(int fd, short event, PF *handler)
{  
  /* Update the read / write set */
  if (event & COMM_SELECT_READ)
  {
    if (handler)
      FD_SET(fd, &select_readfds);
    else
      FD_CLR(fd, &select_readfds);
  }
  if (event & COMM_SELECT_WRITE)
  {
    if (handler)
      FD_SET(fd, &select_writefds);
    else
      FD_CLR(fd, &select_writefds);
  }
}

/* This is a needed exported function which will be called to initialise
 * the network loop code.
 */
void
init_netio(void)
{
  FD_ZERO(&select_readfds);
  FD_ZERO(&select_writefds);
}

/* This is a needed exported function which will be called to register
 * and deregister interest in a pending IO state for a given FD.
 */
void
comm_setselect(int fd, fdlist_t list, unsigned int type, PF *handler,
               void *client_data, time_t timeout)
{
  fde_t *F = &fd_table[fd];

  if (type & COMM_SELECT_READ)
  {
    F->read_handler = handler;
    F->read_data = client_data;
    select_update_selectfds(fd, COMM_SELECT_READ, handler);
  }
  if (type & COMM_SELECT_WRITE)
  {
    F->write_handler = handler;
    F->write_data = client_data;
    select_update_selectfds(fd, COMM_SELECT_WRITE, handler);
  }

  if (timeout)
    F->timeout = CurrentTime + (timeout / 1000);
}
 
/* Check all connections for new connections and input data that is to be
 * processed. Also check for connections with data queued and whether we can
 * write it out.
 */
void
comm_select(unsigned long delay)
{
  int num, fd;
  PF *hdl;
  fde_t *F;
  struct timeval to;

  /* Copy over the read/write sets so we don't have to rebuild em */
  memcpy(&tmpreadfds, &select_readfds, sizeof(fd_set));
  memcpy(&tmpwritefds, &select_writefds, sizeof(fd_set));

  do {
    to.tv_sec = 0;
    to.tv_usec = delay * 1000;
  }
  while ((num = select(highest_fd + 1, &tmpreadfds, &tmpwritefds, NULL, &to)) < 0
         && ignoreErrno(errno));

  set_time();

  for (fd = 0; num > 0 && fd < highest_fd + 1; fd++)
  {
    F = &fd_table[fd];

    if (FD_ISSET(fd, &tmpreadfds))
    {
      hdl = F->read_handler;
      F->read_handler = NULL;
      select_update_selectfds(fd, COMM_SELECT_READ, NULL);
      if (hdl != NULL)
        hdl(fd, F->read_data);
    }
    if (FD_ISSET(fd, &tmpwritefds))
    {
      hdl = F->write_handler;
      F->write_handler = NULL;
      select_update_selectfds(fd, COMM_SELECT_WRITE, NULL);
      if (hdl != NULL)
        hdl(fd, F->write_data);
    }
  }
}
