/* send.c: Functions for sending messages.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: send.c,v 1.107 2005/01/27 11:48:05 musirc Exp $
 */

#include "send.h"
#include "channel.h"
#include "client.h"
#include "istring.h"
#include "sprintf.h"
#include "ircd.h"
#include "handlers.h"
#include "numeric.h"
#include "fdlist.h"
#include "bsd.h"
#include "server.h"
#include "config.h"
#include "log.h"

#define LOG_BUFSIZE 2048

static void send_message(struct Client *, char *, int);
static void send_message_remote(struct Client *, struct Client *, char *, int);
unsigned long current_serial = 0L;

/* inputs	- buffer to format into
 *              - size of the buffer
 *		- format pattern to use
 *		- var args
 * output	- number of bytes formatted output
 * side effects	- modifies sendbuf
 */
static inline int
send_format(char *lsendbuf, int bufsize, const char *pattern, va_list args)
{
  int len;

  /* from rfc1459
   * IRC messages are always lines of characters terminated with a CR-LF
   * (Carriage Return - Line Feed) pair, and these messages shall not
   * exceed 512 characters in length,  counting all characters 
   * including the trailing CR-LF.
   * Thus, there are 510 characters maximum allowed
   * for the command and its parameters.  There is no provision for
   * continuation message lines.  See section 7 for more details about
   * current implementations.
   */
  len = vsnprintf(lsendbuf, bufsize - 1, pattern, args);
  if (len > bufsize - 2)
    len = bufsize - 2;  /* required by some versions of vsnprintf */

  /* We have to get a \r\n\0 onto sendbuf[] somehow to satisfy
   * the rfc. We must assume sendbuf[] is defined to be 513
   * bytes - a maximum of 510 characters, the CR-LF pair, and
   * a trailing \0, as stated in the rfc. Now, if len is greater
   * than the third-to-last slot in the buffer, an overflow will
   * occur if we try to add three more bytes, if it has not
   * already occured. In that case, simply set the last three
   * bytes of the buffer to \r\n\0. Otherwise, we're ok. My goal
   * is to get some sort of vsnprintf() function operational
   * for this routine, so we never again have a possibility
   * of an overflow.
   * -wnder
   * Exactly, vsnprintf() does the job and we don't need to check
   * whether len > 510. We also don't need to terminate the buffer
   * with a '\0', since the dbuf code is raw-oriented. --adx
   */
  lsendbuf[len++] = '\r';
  lsendbuf[len++] = '\n';
  return (len);
}

/* Internal utility which appends given buffer to the sockets sendq. */
static void
send_message(struct Client *to, char *buf, int len)
{
  if (dbuf_length(&to->localClient->buf_sendq) + len > get_sendq(to))
  {
    if (IsServer(to))
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "SendQ limit exceeded for %s: %lu > %lu",
                           get_client_name(to, HIDE_IP),
                           (unsigned long)dbuf_length(&to->localClient->buf_sendq) + len,
                           get_sendq(to));
    if (IsClient(to))
      SetSendQExceeded(to);
    dead_link_on_write(to);
    return;
  }

  dbuf_put(&to->localClient->buf_sendq, buf, len);

  /* Update statistics. The following is slightly incorrect
   * because it counts messages even if queued, but bytes
   * only really sent. Queued bytes get updated in SendQueued.
   */
  ++to->localClient->sendM;
  ++me.localClient->sendM;

  if (dbuf_length(&to->localClient->buf_sendq) >
      (IsServer(to) ? (unsigned int) 1024 : (unsigned int) 4096))
    send_queued_write(to);
}

/* inputs	- pointer to client from message is being sent
 * 		- pointer to client to send to
 *		- pointer to preformatted buffer
 *		- length of input buffer
 * side effects	- Despite the function name, this only sends to directly
 *		  connected clients.
 */
static void
send_message_remote(struct Client *to, struct Client *from,
                    char *buf, int len)
{
  if (!MyConnect(to))
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,
			 "server send message to %s [%s] dropped from %s(Not local server)",
			 to->name, to->from->name, from->name);
    return;
  }

  /* Optimize by checking if (from && to) before everything
   * we set to->from up there.. */
  if (!MyClient(from) && IsPerson(to) && (to == from->from))
  {
    if (IsServer(from))
    {
      sendto_realops_flags(UMODE_ALL, L_ALL,
                           "Send message to %s [%s] dropped from %s(Fake Dir)",
                           to->name, to->from->name, from->name);
      return;
    }

    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "Ghosted: %s[%s@%s] from %s[%s@%s] (%s)",
                         to->name, to->username, to->host,
                         from->name, from->username, from->host,
                         to->from->name);

    sendto_server(NULL, NULL, ":%s KILL %s :%s (%s[%s@%s] Ghosted %s)",
                  me.name, to->name, me.name, to->name,
                  to->username, to->host, to->from->name);

    SetKilled(to);

    if (IsPerson(from))
      sendto_one(from, form_str(ERR_GHOSTEDCLIENT),
                 me.name, from->name, to->name, to->username,
                 to->host, to->from);

    exit_client(NULL, to, &me, "Ghosted client");

    return;
  } 

  send_message(to, buf, len);
}

/* Called when a socket is ready for writing. */
static void
sendq_unblocked(int n, struct Client *client_p)
{
  ClearSendqBlocked(client_p);
}

/* This is called when there is a chance that some output would
 * be possible. This attempts to empty the send queue as far as
 * possible, and then if any data is left, a write is rescheduled.
 */
void
send_queued_write(struct Client *to)
{
  int retlen;
  struct dbuf_block *first;
  
  /* Once socket is marked dead, we cannot start writing to it,
   * even if the error is removed...
   */
  if (IsDead(to) || IsSendqBlocked(to))
    return;  /* no use calling send() now */

  /* Next, lets try to write some data */
  if (dbuf_length(&to->localClient->buf_sendq))
  {
    do {
      first = to->localClient->buf_sendq.blocks.head->data;
      if ((retlen = send(to->localClient->fd, first->data,
                         first->size, 0)) <= 0)
        break;

      dbuf_delete(&to->localClient->buf_sendq, retlen);

      /* We have some data written .. update counters */
      to->localClient->sendB += retlen;
      me.localClient->sendB += retlen;
      if (to->localClient->sendB > 1023)
      { 
        to->localClient->sendK += (to->localClient->sendB >> 10);
        to->localClient->sendB &= 0x03ff;        /* 2^10 = 1024, 3ff = 1023 */
      }
      if (me.localClient->sendB > 1023)
      { 
        me.localClient->sendK += (me.localClient->sendB >> 10);
        me.localClient->sendB &= 0x03ff;
      }
    } while (dbuf_length(&to->localClient->buf_sendq));

    if ((retlen < 0) && (ignoreErrno(errno)))
    {
      /* we have a non-fatal error, reschedule a write */
      SetSendqBlocked(to);
      comm_setselect(to->localClient->fd, FDLIST_IDLECLIENT, COMM_SELECT_WRITE,
                     (PF *)sendq_unblocked, (void *)to, 0);
    }
    else if (retlen <= 0)
    {
      dead_link_on_write(to);
      return;
    }
  }
}

/* side effects - try to flush sendq of each client */
void
send_queued_all(void)
{
  dlink_node *ptr;

  /* Servers are processed first, mainly because this can generate
   * a notice to opers, which is to be delivered by this function.
   */
  DLINK_FOREACH(ptr, serv_list.head)
    send_queued_write((struct Client *) ptr->data);

  DLINK_FOREACH(ptr, unknown_list.head)
    send_queued_write((struct Client *) ptr->data);

  DLINK_FOREACH(ptr, local_client_list.head)
    send_queued_write((struct Client *) ptr->data);
}

/* inputs	- pointer to destination client
 *		- var args message
 * side effects	- send message to single client
 */
void
sendto_one(struct Client *to, const char *pattern, ...)
{
  va_list args;
  char buffer[BUFSIZE];
  int len;

  if (to->from != NULL)
    to = to->from;
  if (IsDead(to))
    return;

  va_start(args, pattern);
  len = send_format(buffer, BUFSIZE, pattern, args);
  va_end(args);

  send_message(to, buffer, len);
}

/* inputs	- pointer to client(server) to NOT send message to
 *		- pointer to client that is sending this message
 *		- pointer to channel being sent to
 *		- vargs message
 * side effects	- message as given is sent to given channel members.
 */
void
sendto_channel_butone(struct Client *one, struct Client *from,
                      struct Channel *chptr, const char *command,
                      const char *pattern, ...)
{
  va_list args;
  char local_buf[BUFSIZE], remote_buf[BUFSIZE];
  int local_len, remote_len;
  dlink_node *ptr, *ptr_next;
  struct Client *target_p;

  if (IsServer(from))
    local_len = ircsprintf(local_buf, ":%s %s %s ",
                           from->name, command, chptr->name);
  else
    local_len = ircsprintf(local_buf, ":%s!%s@%s %s %s ",
                           from->name, from->username, from->host,
                           command, chptr->name);
  remote_len = ircsprintf(remote_buf, ":%s %s %s ",
                          from->name, command, chptr->name);
  va_start(args, pattern);
  local_len += send_format(&local_buf[local_len], BUFSIZE - local_len,
                           pattern, args);
  remote_len += send_format(&remote_buf[remote_len], BUFSIZE - remote_len,
                            pattern, args);
  va_end(args);
  ++current_serial;

  DLINK_FOREACH_SAFE(ptr, ptr_next, chptr->members.head)
  {
    target_p = ((struct Membership *)ptr->data)->client_p;

    if (IsDefunct(target_p) || target_p->from == one || IsDeaf(target_p))
      continue;

    if (MyClient(target_p))
    {
      if (target_p->serial != current_serial)
      {
        send_message(target_p, local_buf, local_len);
        target_p->serial = current_serial;
      }
    }
    else
    {
      /* Now check whether a message has been sent to this
       * remote link already
       */
      if (target_p->from->serial != current_serial)
      {
        send_message_remote(target_p->from, from, remote_buf, remote_len);
        target_p->from->serial = current_serial;
      }
    }
  }
}

/* inputs       - pointer to client to NOT send to
 *              - printf style format string
 *              - args to format string
 * side effects - Send a message to all connected servers, except the
 *                client 'one' (if non-NULL), as long as the servers
 */
void 
sendto_server(struct Client *one, struct Channel *chptr, const char *format, ...)
{
  va_list args;
  struct Client *client_p;
  dlink_node *ptr;
  char buffer[BUFSIZE];
  int len;

  if (chptr != NULL)
  {
    if (chptr->name[0] != '#')
      return;
  }

  va_start(args, format);
  len = send_format(buffer, BUFSIZE, format, args);
  va_end(args);

  DLINK_FOREACH(ptr, serv_list.head)
  {
    client_p = ptr->data;

    if (IsDead(client_p))
      continue;
    if (one != NULL && (client_p == one->from))
      continue;
    send_message(client_p, buffer, len);
  }
}

/* inputs	- pointer to client
 *		- pattern to send
 * side effects	- Sends a message to all people on local server who are
 * 		  in same channel with user. 
 */
void
sendto_common_channels_local(struct Client *user, int touser,
                             const char *pattern, ...)
{
  va_list args;
  dlink_node *uptr, *cptr;
  struct Channel *chptr;
  struct Membership *ms;
  struct Client *target_p;
  char buffer[BUFSIZE];
  int len;

  va_start(args, pattern);
  len = send_format(buffer, BUFSIZE, pattern, args);
  va_end(args);
  ++current_serial;

  DLINK_FOREACH(cptr, user->user->channel.head)
  {
    chptr = ((struct Membership *) cptr->data)->chptr;

    DLINK_FOREACH(uptr, chptr->locmembers.head)
    {
      ms = uptr->data;
      target_p = ms->client_p;

      if (target_p == user || IsDefunct(target_p) ||
          target_p->serial == current_serial)
        continue;

      target_p->serial = current_serial;
      send_message(target_p, buffer, len);
    }
  }

  if (touser && MyConnect(user) && !IsDead(user) &&
      user->serial != current_serial)
    send_message(user, buffer, len);
}

/* inputs	- member status mask, e.g. CHFL_CHANOP | CHFL_VOICE
 *              - pointer to channel to send to
 *              - var args pattern
 * side effects - Send a message to all members of a channel that are
 *		  locally connected to this server.
 */
void
sendto_channel_local(int type, struct Channel *chptr, const char *pattern, ...)
{
  va_list args;
  char buffer[BUFSIZE];
  int len;
  dlink_node *ptr;
  struct Membership *ms;
  struct Client *target_p;

  va_start(args, pattern);
  len = send_format(buffer, BUFSIZE, pattern, args);
  va_end(args);
  /* Serial number checking isn't strictly necessary, but won't hurt */
  ++current_serial;

  DLINK_FOREACH(ptr, chptr->locmembers.head)
  {
    ms = ptr->data;
    target_p = ms->client_p;

    if (type != 0 && (ms->flags & type) == 0)
      continue;

    if (IsDefunct(target_p) || target_p->serial == current_serial)
      continue;

    target_p->serial = current_serial;

    send_message(target_p, buffer, len);
  }
}

/* inputs       - pointer to client to NOT send message to
 *              - member status mask, e.g. CHFL_CHANOP | CHFL_VOICE
 *              - pointer to channel to send to
 *              - var args pattern
 * side effects - Send a message to all members of a channel that are
 *                locally connected to this server except one.
 */
void       
sendto_channel_local_butone(struct Client *one, int type,
			    struct Channel *chptr, const char *pattern, ...)
{
  va_list args;
  char buffer[BUFSIZE];
  int len;
  struct Client *target_p;
  struct Membership *ms;
  dlink_node *ptr;

  va_start(args, pattern); 
  len = send_format(buffer, BUFSIZE, pattern, args);
  va_end(args);

  /* Serial number checking isn't strictly necessary, but won't hurt */
  ++current_serial;

  DLINK_FOREACH(ptr, chptr->locmembers.head)  
  {   
    ms = ptr->data;
    target_p = ms->client_p;

    if (type != 0 && (ms->flags & type) == 0)
      continue;

    if (target_p == one || IsDefunct(target_p) ||
        target_p->serial == current_serial)
      continue;

    target_p->serial = current_serial;

    send_message(target_p, buffer, len);
  }
}

/* inputs	- Client not to send towards
 *		- Client from whom message is from
 *		- member status mask, e.g. CHFL_CHANOP | CHFL_VOICE
 *              - pointer to channel to send to
 *              - var args pattern
 * side effects - Send a message to all members of a channel that are
 *		  remote to this server.
 */
void
sendto_channel_remote(struct Client *one, int type,
                      struct Channel *chptr, const char *pattern, ...)
{
  va_list args;
  char buffer[BUFSIZE];
  int len;
  dlink_node *ptr;
  struct Client *target_p;
  struct Membership *ms;

  va_start(args, pattern);
  len = send_format(buffer, BUFSIZE, pattern, args);
  va_end(args);

  /* Serial number checking isn't strictly necessary, but won't hurt */
  ++current_serial;

  DLINK_FOREACH(ptr, chptr->members.head)
  {
    ms = ptr->data;
    target_p = ms->client_p;

    if (type != 0 && (ms->flags & type) == 0)
      continue;

    if (MyConnect(target_p))
      continue;
    target_p = target_p->from;

    if (target_p == one->from || target_p->from->serial == current_serial)
      continue;

    target_p->serial = current_serial;

    send_message(target_p, buffer, len);
  } 
}

/* inputs	- client pointer to match on
 *		- actual mask to match
 *		- what to match on, HOST or SERVER
 * output	- 1 or 0 if match or not
 */
static int
match_it(const struct Client *one, const char *mask, int what)
{
  if (what == MATCH_HOST)
    return(match(mask, one->host));

  return(match(mask, one->user->server->name));
}

/* Send to all clients which match the mask in a way defined on 'what';
 * either by user hostname or user servername.
 */
void
sendto_match_butone(struct Client *one, struct Client *from, char *mask,
                    int what, const char *pattern, ...)
{
  va_list args;
  struct Client *client_p;
  dlink_node *ptr, *ptr_next;
  char local_buf[BUFSIZE], remote_buf[BUFSIZE];
  int local_len = ircsprintf(local_buf, ":%s!%s@%s ", from->name,
                             from->username, from->host);
  int remote_len = ircsprintf(remote_buf, ":%s ", from->name);

  va_start(args, pattern);
  local_len += send_format(&local_buf[local_len], BUFSIZE - local_len,
                           pattern, args);
  remote_len += send_format(&remote_buf[remote_len], BUFSIZE - remote_len,
                            pattern, args);
  va_end(args);

  /* scan the local clients */
  DLINK_FOREACH(ptr, local_client_list.head)
  {
    client_p = ptr->data;

    if (client_p != one && !IsDefunct(client_p) &&
        match_it(client_p, mask, what))
      send_message(client_p, local_buf, local_len);
  }

  /* Now scan servers */
  DLINK_FOREACH_SAFE(ptr, ptr_next, serv_list.head)
  {
    client_p = ptr->data;

    if (client_p != one && !IsDefunct(client_p))
      send_message_remote(client_p, from, remote_buf, remote_len);
  }
}

/* inputs       - source client
 *              - mask to send to
 *              - data
 * side effects - data sent to servers matching
 */
void
sendto_match_servs(struct Client *source_p, const char *mask, const char *pattern, ...)
{
  va_list args;
  struct Client *target_p;
  dlink_node *ptr;
  char buffer[BUFSIZE];

  va_start(args, pattern);
  vsnprintf((char *)&buffer, sizeof(buffer), pattern, args);
  va_end(args);
  current_serial++;

  DLINK_FOREACH(ptr, global_serv_list.head)
  {
    target_p = ptr->data;

    /* Do not attempt to send to ourselves, or the source */
    if (IsMe(target_p) || target_p->from == source_p->from)
      continue;

    if (target_p->from->serial == current_serial)
      continue;

    if (match(mask, target_p->name))
    {
      target_p->from->serial = current_serial;
      sendto_anywhere(target_p, source_p, "%s", buffer);
    }
  }
}

/* inputs	- pointer to dest client
 * 		- pointer to from client
 * 		- varags
 * side effects	- less efficient than sendto_remote and sendto_one
 * 		  but useful when one does not know where target "lives"
 */
void
sendto_anywhere(struct Client *to, struct Client *from,
                const char *pattern, ...)
{
  va_list args;
  char buffer[BUFSIZE];
  int len;
  struct Client *send_to = (to->from != NULL ? to->from : to);

  if (IsDead(send_to))
    return;

  if (MyClient(to))
  {
    if (IsServer(from))
      len = ircsprintf(buffer, ":%s ", from->name);
    else
      len = ircsprintf(buffer, ":%s!%s@%s ",
                       from->name, from->username, from->host);
  }
  else
    len = ircsprintf(buffer, ":%s ", from->name);

  va_start(args, pattern);
  len += send_format(&buffer[len], BUFSIZE - len, pattern, args);
  va_end(args);

  if(MyClient(to))
    send_message(send_to, buffer, len);
  else
    send_message_remote(send_to, from, buffer, len);
}

/* inputs	- flag types of messages to show to real opers
 *		- flag indicating opers/admins
 *		- var args input message
 * side effects	- Send to *local* ops only but NOT +s nonopers.
 */
void
sendto_realops_flags(unsigned int flags, int level, const char *pattern, ...)
{
  struct Client *client_p;
  char nbuf[BUFSIZE];
  dlink_node *ptr;
  va_list args;

  va_start(args, pattern);
  vsnprintf(nbuf, BUFSIZE, pattern, args);
  va_end(args);

  DLINK_FOREACH(ptr, oper_list.head)
  {
    client_p = ptr->data;
    /* If we're sending it to opers and theyre an admin, skip.
     * If we're sending it to admins, and theyre not, skip.
     */
    if (((level == L_ADMIN) && !IsAdmin(client_p)) ||
	((level == L_OPER) && IsAdmin(client_p)))
      continue;

    if (client_p->umodes & flags)
      sendto_one(client_p, ":%s NOTICE %s :%s", me.name, client_p->name, nbuf);
  }
}

/* inputs       - flag types of messages to show to real opers
 *              - client sending request
 *              - var args input message
 * side effects - Send a wallops to local opers
 */
void
sendto_wallops_flags(unsigned int flags, struct Client *source_p,
                     const char *pattern, ...)
{
  struct Client *client_p;
  dlink_node *ptr;
  va_list args;
  char buffer[BUFSIZE];
  int len;

  if (IsPerson(source_p))
    len = ircsprintf(buffer, ":%s!%s@%s WALLOPS :",
                     source_p->name, source_p->username, source_p->host);
  else
    len = ircsprintf(buffer, ":%s WALLOPS :", source_p->name);

  va_start(args, pattern);
  len += send_format(&buffer[len], BUFSIZE - len, pattern, args);
  va_end(args);

  DLINK_FOREACH(ptr, oper_list.head)
  {
    client_p = ptr->data;

    if ((client_p->umodes & flags) && !IsDefunct(client_p))
      send_message(client_p, buffer, len);
  }
}

/* inputs	- var args message
 * side effects	- Call sendto_realops_flags, with some flood checking
 *		  (at most 5 warnings every 5 seconds)
 */
void
ts_warn(const char *pattern, ...)
{
  va_list args;
  char buffer[LOG_BUFSIZE];
  static time_t last = 0;
  static int warnings = 0;

  if (CurrentTime - last < 5)
  {
    if (++warnings > 5)
      return;
  }
  else
  {
    last = CurrentTime;
    warnings = 0;
  }

  va_start(args, pattern);
  vsprintf_irc(buffer, pattern, args);
  va_end(args);
  sendto_realops_flags(UMODE_ALL, L_ALL, "%s", buffer);
  ilog(CRIT, "%s", buffer);
}

/* inputs	- client to send kill towards
 * 		- pointer to client to kill
 * 		- reason for kill
 */
void
kill_client(struct Client *client_p, struct Client *diedie,
            const char *pattern, ...)
{
  va_list args;
  char buffer[BUFSIZE];
  int len;

  if (client_p->from != NULL)
    client_p = client_p->from;
  if (IsDead(client_p))
    return;

  len = ircsprintf(buffer, ":%s KILL %s :", me.name, diedie->name);

  va_start(args, pattern);
  len += send_format(&buffer[len], BUFSIZE - len, pattern, args);
  va_end(args);

  send_message(client_p, buffer, len);
}
