/* user.c: User related functions.
 * Copyright (C) 2005 by MusIRCd Development.
 * $Id: user.c,v 1.64 2005/02/12 17:03:06 musirc Exp $
 */

#include "user.h"
#include "client.h"
#include "channel.h"
#include "channel_mode.h"
#include "hash.h"
#include "istring.h"
#include "sprintf.h"
#include "bsd.h"
#include "ircd.h"
#include "list.h"
#include "listener.h"
#include "log.h"
#include "numeric.h"
#include "config.h"
#include "supported.h"
#include "send.h"
#include "packet.h"
#include "userhost.h"
#include "getinfo.h"

static BlockHeap *user_heap;

static int valid_hostname(const char *);
static int valid_username(const char *);
static void user_welcome(struct Client *);
static void report_and_set_user_flags(struct Client *, struct AccessItem *);
static int check_x_line(struct Client *, struct Client *);
static int introduce_client(struct Client *, struct Client *);

/* table of ascii char letters to corresponding bitmask */
static const struct flag_item
{
  const unsigned int mode;
  const unsigned char letter;
} user_modes[] = {
  {UMODE_ADMIN,      'a'},
  {UMODE_CCONN,      'c'},
  {UMODE_FLOOD,      'f'},
  {UMODE_INVISIBLE,  'i'},
  {UMODE_JUPES,	     'j'},
  {UMODE_LOCOPS,     'l'},
  {UMODE_NCHANGE,    'n'},
  {UMODE_OPER,       'o'},
  {UMODE_SERVNOTICE, 's'},
  {UMODE_UNAUTH,     'u'},
  {UMODE_WALLOP,     'w'},
  {UMODE_EXTERNAL,   'x'},
  {UMODE_SPY,        'y'},
  {UMODE_AIM,	     'A'},
  {UMODE_DEAF,       'D'},
  {UMODE_SSIGNORE,   'I'},
  {UMODE_SERVICE,    'S'},
  {0, 		     '\0'}
};

const unsigned int user_modes_from_c_to_bitmask[] =
{ 
  /* 0x80 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0x8F */
  /* 0x10 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0x1F */
  /* 0x20 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0x2F */
  /* 0x30 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0x3F */
  0,            /* @ */
  UMODE_AIM,    /* A */
  0,            /* B */
  0,            /* C */
  UMODE_DEAF,   /* D */
  0,            /* E */
  0,            /* F */
  0,            /* G */
  0,            /* H */
  UMODE_SSIGNORE,  /* I */
  0,            /* J */
  0,            /* K */
  0,            /* L */
  0,            /* M */
  0,            /* N */
  0,            /* O */
  0,            /* P */
  0,            /* Q */
  0,            /* R */
  UMODE_SERVICE, /* S */
  0,            /* T */
  0,            /* U */
  0,            /* V */
  0,            /* W */
  0,            /* X */
  0,            /* Y */
  0,            /* Z 0x5A */
  0, 0, 0, 0, 0, /* 0x5F */ 
  0, 		/* 0x60 */
  UMODE_ADMIN,  /* a */
  0,		/* b */
  UMODE_CCONN,  /* c */
  0,		/* d */
  0,            /* e */
  UMODE_FLOOD,	/* f */
  0,            /* g */
  0,            /* h */
  UMODE_INVISIBLE, /* i */
  UMODE_JUPES,	/* j */
  0,		/* k */
  UMODE_LOCOPS, /* l */
  0,            /* m */
  UMODE_NCHANGE, /* n */
  UMODE_OPER,   /* o */
  0,            /* p */
  0,            /* q */
  0,		/* r */
  UMODE_SERVNOTICE, /* s */
  0,            /* t */
  UMODE_UNAUTH, /* u */
  0,            /* v */
  UMODE_WALLOP, /* w */
  UMODE_EXTERNAL, /* x */
  UMODE_SPY,    /* y */
  0, /* z */
  0,0,0,0,0,     /* 0x7B - 0x7F */

  /* 0x80 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0x9F */
  /* 0x90 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0x9F */
  /* 0xA0 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0xAF */
  /* 0xB0 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0xBF */
  /* 0xC0 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0xCF */
  /* 0xD0 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0xDF */
  /* 0xE0 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0xEF */
  /* 0xF0 */ 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0  /* 0xFF */
};

/* side effects - Creates a block heap for struct Users */
void
init_user(void)
{
  user_heap = BlockHeapCreate(sizeof(struct User), USER_HEAP_SIZE);
}

/* inputs	- pointer to Client struct
 * output       - pointer to struct User
 * side effects - add's an User information block to a client
 *                if it was not previously allocated.
 */
static struct User *
make_user(struct Client *client_p)
{
  if (client_p->user == NULL)
  {
    client_p->user = BlockHeapAlloc(user_heap);
    memset(client_p->user, 0, sizeof(struct User));
  }

  return(client_p->user);
}

/* inputs	- pointer to User struct
 *		- pointer to Client struct
 * side effects - free an User block
 */
void
free_user(struct User *user, struct Client *client_p)
{
  MyFree(user->away);

  /* sanity check */
  if (dlink_list_length(&user->channel) || user->invited.head || user->channel.head)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "* %#lx user (%s!%s@%s) %#lx %#lx %#lx %lu *",
                         (unsigned long)client_p, client_p ? client_p->name : "<noname>",
                         client_p->username, client_p->host, (unsigned long)user,
                         (unsigned long)user->invited.head,
                         (unsigned long)user->channel.head,
                         dlink_list_length(&user->channel));
  }

  BlockHeapFree(user_heap, user);
}

/* inputs	- pointer to client
 * side effects	- display to client user counts etc.
 */
void
show_lusers(struct Client *source_p)
{
  static time_t lastsave = 0;

  if ((CurrentTime - lastsave) > SAVE_TIME)
  {
    write_stats();
    lastsave = CurrentTime;
  }

  if (!ConfigServerHide.hide_servers || IsOper(source_p))
    sendto_one(source_p, form_str(RPL_LUSERCLIENT),
	       me.name, source_p->name, Count.total - Count.service,
	       Count.service, dlink_list_length(&global_serv_list));
  else
    sendto_one(source_p, form_str(RPL_LUSERCLIENT), me.name, source_p->name,
               Count.total - Count.service, Count.service, 1);

  if (Count.oper > 0)
    sendto_one(source_p, form_str(RPL_LUSEROP),
               me.name, source_p->name, Count.oper);

  if (dlink_list_length(&unknown_list) > 0)
    sendto_one(source_p, form_str(RPL_LUSERUNKNOWN),
               me.name, source_p->name, dlink_list_length(&unknown_list));

  if (dlink_list_length(&global_channel_list) > 0)
    sendto_one(source_p, form_str(RPL_LUSERCHANNELS),
               me.name, source_p->name,
               dlink_list_length(&global_channel_list));

  if (!ConfigServerHide.hide_servers || IsOper(source_p))
  {
    sendto_one(source_p, form_str(RPL_LUSERME),
               me.name, source_p->name,
               Count.local, Count.myserver);
    sendto_one(source_p, form_str(RPL_LOCALUSERS),
               me.name, source_p->name,
               Count.local, Count.max_loc);
  }
  else
  {
    sendto_one(source_p, form_str(RPL_LUSERME),
               me.name, source_p->name, Count.total - Count.service, 0);
    sendto_one(source_p, form_str(RPL_LOCALUSERS),
               me.name, source_p->name, Count.total - Count.service, Count.max_tot);
  }
  sendto_one(source_p, form_str(RPL_GLOBALUSERS),
             me.name, source_p->name, Count.total - Count.service, Count.max_tot);
}

/* inputs	- pointer to client
 * side effects	- display to client what we support (for them)
 */
void
show_isupport(struct Client *source_p) 
{
  char isupportbuffer[512];

  ircsprintf(isupportbuffer, FEATURES, FEATURESVALUES);
  sendto_one(source_p, form_str(RPL_ISUPPORT),
             me.name, source_p->name, isupportbuffer);
  ircsprintf(isupportbuffer, FEATURES2, FEATURESVALUES2);
  sendto_one(source_p, form_str(RPL_ISUPPORT),
             me.name, source_p->name, isupportbuffer);
}

/*      This function is called when both NICK and USER messages
 *      have been accepted for the client, in whatever order. Only
 *      after this, is the USER message propagated.
 *
 *      NICK's must be propagated at once when received, although
 *      it would be better to delay them too until full info is
 *      available. Doing it is not so simple though, would have
 *      to implement the following:
 *
 *      (actually it has been implemented already for a while) -orabidoo
 *
 *      1) user telnets in and gives only "NICK foobar" and waits
 *      2) another user far away logs in normally with the nick
 *         "foobar" (quite legal, as this server didn't propagate
 *         it).
 *      3) now this server gets nick "foobar" from outside, but
 *         has alread the same defined locally. Current server
 *         would just issue "KILL foobar" to clean out dups. But,
 *         this is not fair. It should actually request another
 *         nick from local user or kill him/her...
 */
int
register_local_user(struct Client *client_p, struct Client *source_p, 
	            const char *nick, const char *username)
{
  struct AccessItem *aconf;
  char ipaddr[HOSTIPLEN];
  int status;
  dlink_node *ptr, *m;

  if(source_p == NULL)
    return (-1);
  
  if(!MyConnect(source_p))
    return (-1);

  if (!IsPingSent(source_p) && source_p->localClient->random_ping == 0)
  {
    source_p->localClient->random_ping = (unsigned long)(rand() * rand()) << 1;
    sendto_one(source_p, "PING :%08lX", source_p->localClient->random_ping);
    SetPingSent(source_p);
    return(-1);
  }

  source_p->user->last = CurrentTime;
  /* Straight up the maximum rate of flooding... */
  source_p->localClient->allow_read = MAX_FLOOD_BURST;

  if ((status = check_client(client_p, source_p, username)) < 0)
    return(CLIENT_EXITED);

  if (!valid_hostname(source_p->host))
  {
    sendto_one(source_p, ":%s NOTICE %s :*** Illegal character in hostname",
	       me.name, source_p->name);
    strlcpy(source_p->host, source_p->localClient->sockhost,
            sizeof(source_p->host));
  }
  ptr = source_p->localClient->confs.head;
  aconf = map_to_conf((struct ConfItem *)ptr->data);

  if (!IsGotId(source_p))
  {
    const char *p;
    unsigned int i = 0;

    if (IsNeedIdentd(aconf) && !ConfigFileEntry.disable_auth)
    {	
      exit_client(client_p, source_p, &me, "Install identd");
      return(CLIENT_EXITED);
    }
    p = username;
      
    if (!IsNoTilde(aconf))
      source_p->username[i++] = '~';

    while (*p && i < USERLEN)
    {
      if (*p != '[')
        source_p->username[i++] = *p;
	p++;
    }
    source_p->username[i] = '\0';
  }

  /* password check */
  if (!EmptyString(aconf->passwd) &&
      (source_p->localClient->passwd == NULL ||
      strcmp(source_p->localClient->passwd, aconf->passwd)))
  {
    sendto_one(source_p, form_str(ERR_PASSWDMISMATCH),
	       me.name, source_p->name);
    exit_client(client_p, source_p, &me, "Bad Password");
    return(CLIENT_EXITED);
  }
  /* don't free source_p->localClient->passwd here - it can be required
   * by masked /stats I if there are auth blocks with need_password = no;
   * --adx
   */
  report_and_set_user_flags(source_p, aconf);
  
  /* Limit clients
   * We want to be able to have servers and F-line clients
   * connect, so save room for "buffer" connections.
   * Smaller servers may want to decrease this, and it should
   * probably be just a percentage of the MAXCLIENTS...
   *   -Taner
   */
  if ((((Count.local + 1) >= (GlobalSetOptions.maxclients + MAX_BUFFER))) ||
     ((Count.local >= ServerInfo.max_clients) && !IsExemptLimits(source_p)))
  {
    sendto_realops_flags(UMODE_CCONN, L_ALL,
                         "Too many clients, rejecting %s %s",
                         nick, source_p->host);
    exit_client(client_p, source_p, &me, "Server full");
    return(CLIENT_EXITED);
  }
  /* valid user name check */
  if (!valid_username(source_p->username))
  {
    char tmpstr2[BUFSIZE];
    ircsprintf(tmpstr2, "Invalid username %s", source_p->username);
    exit_client(client_p, source_p, &me, tmpstr2);
    return(CLIENT_EXITED);
  }

  /* end of valid user name check */
  if ((status = check_x_line(client_p, source_p)) < 0)
    return(status);

  if (IsDead(client_p))
    return(CLIENT_EXITED);

  irc_getnameinfo((struct sockaddr *)&source_p->localClient->ip,
                  source_p->localClient->ip.ss_len, ipaddr,
                  HOSTIPLEN, NULL, 0, NI_NUMERICHOST);
  sendto_realops_flags(UMODE_CCONN, L_ALL,
                       "Connect: %s (%s@%s) [ %s ] {%s} [%s]",
                       nick, source_p->username, source_p->host,
		       ConfigFileEntry.hide_spoof_ips && IsIPSpoof(source_p) ?
		       "masked" : ipaddr, get_client_class(source_p),
		       source_p->info);

  /* If they have died in send_* don't do anything. */
  if (IsDead(source_p))
    return(CLIENT_EXITED);
  
  if ((++Count.local) > Count.max_loc)
  {
    Count.max_loc = Count.local;

    if (!(Count.max_loc % 10))
      sendto_realops_flags(UMODE_ALL, L_ALL, "New Max Local Clients: %d",
                           Count.max_loc);
  }

  SetClient(source_p);

  source_p->servptr = &me;
  dlinkAdd(source_p, &source_p->lnode, &source_p->servptr->serv->users);

  if (++Count.total - Count.service> Count.max_tot)
    Count.max_tot = Count.total - Count.service;
  source_p->localClient->allow_read = MAX_FLOOD_BURST;

  if ((m = dlinkFindDelete(&unknown_list, source_p)) != NULL)
  {
    free_dlink_node(m);
     dlinkAdd(source_p, &source_p->localClient->lclient_node, &local_client_list);
  }
  else
  {
    exit_client(client_p, source_p, &me, "Client Exited");
    return(CLIENT_EXITED);
  }
  user_welcome(source_p);
  add_user_host(source_p->username, source_p->host, 0);
  SetUserHost(source_p);

  return(introduce_client(client_p, source_p));
}

/* side effects - This function is called when a remote client
 *                is introduced by a server.
 */
int
register_remote_user(struct Client *client_p, struct Client *source_p,
        	     const char *username, const char *host, const char *server,
		     const char *realname)
{
  struct Client *target_p;

  if (source_p == NULL)
    return(-1);

  source_p->user = make_user(source_p);
  /* coming from another server, take the servers word for it */
  source_p->user->server = find_server(server);
  strlcpy(source_p->host, host, sizeof(source_p->host));
  strlcpy(source_p->info, realname, sizeof(source_p->info));
  strlcpy(source_p->username, username, sizeof(source_p->username));

  SetClient(source_p);

  if (++Count.total - Count.service > Count.max_tot)
    Count.max_tot = Count.total - Count.service;

  source_p->user->last = CurrentTime;
  source_p->servptr = source_p->user->server;

  /* Super GhostDetect:
   * If we can't find the server the user is supposed to be on,
   * then simply blow the user away.        -Taner
   */
  if (source_p->servptr == NULL)
  {
    sendto_realops_flags(UMODE_ALL, L_ALL,
                         "No server %s for user %s %s@%s from %s",
	                 server, source_p->name, source_p->username,
                         source_p->host, source_p->from->name);
    kill_client(client_p, source_p, "%s (Server doesn't exist)", me.name);

    SetKilled(source_p);
    return(exit_client(NULL, source_p, &me, "Ghosted Client"));
  }

  dlinkAdd(source_p, &source_p->lnode, &source_p->servptr->serv->users);

  if ((target_p = source_p->servptr) && target_p->from != source_p->from)
  {
    kill_client(client_p, source_p,
                "%s (NICK from wrong direction (%s != %s))",
	         me.name, source_p->user->server->name, target_p->from->name);
    SetKilled(source_p);
    return(exit_client(source_p, source_p, &me, "Invalid direction"));
  }
  add_user_host(source_p->username, source_p->host, 1);
  SetUserHost(source_p);

  return(introduce_client(client_p, source_p));
}

/* side effects - This common function introduces a client to the rest
 *		  of the net, either from a local client connect or
 *		  from a remote connect.
 */
static int
introduce_client(struct Client *client_p, struct Client *source_p)
{
  dlink_node *server_node;
  static char ubuf[12];

  if (MyClient(source_p))
    send_umode(source_p, source_p, 0, SEND_UMODES, ubuf);
  else
    send_umode(NULL, source_p, 0, SEND_UMODES, ubuf);

  if (!*ubuf)
  {
    ubuf[0] = '+';
    ubuf[1] = '\0';
  }

  /* arghhh one could try not introducing new nicks to ll leafs
   * but then you have to introduce them "on the fly" in SJOIN
   * not fun.
   * Its not going to cost much more bandwidth to simply let new
   * nicks just ride on through.
   * We now introduce nicks "on the fly" in SJOIN anyway --
   * you _need_ to if you aren't going to burst everyone initially.
   */
  DLINK_FOREACH(server_node, serv_list.head)
  {
    struct Client *server = server_node->data;

    if (server == client_p)
      continue;

    sendto_one(server, "NICK %s %d %lu %s %s %s %s :%s",
               source_p->name, source_p->hopcount+1,
               (unsigned long)source_p->tsinfo,
               ubuf, source_p->username, source_p->host,
               source_p->user->server->name,
               source_p->info);
  }
  return(0);
}

/* Inputs       - pointer to hostname
 * Output	- 1 if valid, 0 if not
 * Side effects - check hostname for validity
 *
 * NOTE: this doesn't allow a hostname to begin with a dot and
 * will not allow more dots than chars.
 */
static int
valid_hostname(const char *hostname)
{
  const char *p = hostname;

  if (hostname == NULL)
    return(0);

  if (('.' == *p) || (':' == *p))
    return(0);

  while (*p)
  {
    if (!IsHostChar(*p))
      return(0);
    p++;
  }

  return(1);
}

/* Inputs       - pointer to user
 * Output	- 1 if valid, 0 if not
 * Side effects - check username for validity
 * 
 * Absolutely always reject any '*' '!' '?' '@' in an user name
 * reject any odd control characters names.
 * Allow '.' in username to allow for "first.last"
 * style of username
 */
static int
valid_username(const char *username)
{
  int dots = 0;
  const char *p = username;

  if (username == NULL)
    return(0);

  if ('~' == *p)
    ++p;

  /* reject usernames that don't start with an alphanum
   * i.e. reject jokers who have '-@somehost' or '.@somehost'
   * or "-hi-@somehost", "h-----@somehost" would still be accepted.
   */
  if (!IsAlNum(*p))
    return(0);

  while (*++p)
  {
    if ((*p == '.') && ConfigFileEntry.dots_in_ident)
    {
      dots++;

      if (dots > ConfigFileEntry.dots_in_ident)
        return(0);
      if (!IsUserChar(p[1]))
        return(0);
    }
    else if (!IsUserChar(*p))
      return(0);
  }

  return(1);
}

/* inputs       - pointer to source_p
 *              - pointer to aconf for this user
 * side effects - Report to user any special flags
 *		  they are getting, and set them.
 */
static void 
report_and_set_user_flags(struct Client *source_p, struct AccessItem *aconf)
{
  char buffer[BUFSIZE];
  int notice = 0;

  *buffer = '\0';

  if (IsConfDoSpoofIp(aconf))
  {
    struct ConfItem *conf;

    conf = unmap_conf_item(aconf);
    sendto_one(source_p,
               ":%s NOTICE %s :*** Spoofing [%s] to [%s]",
               me.name, source_p->name, source_p->localClient->sockhost, conf->name);
  }
  if (IsConfExemptKline(aconf))
  {
    SetExemptKline(source_p);
    strlcat(buffer, " [K-Lines]", sizeof(buffer));
    notice = 1;
  }
  if (IsConfExemptLimits(aconf))
  {
    SetExemptLimits(source_p);
    strlcat(buffer, " [user limits]", sizeof(buffer));
    notice = 1;
  }
  if (IsConfCanFlood(aconf))
  {
    SetCanFlood(source_p);
    strlcat(buffer, " [flood limits]", sizeof(buffer));
    notice = 1;
  }
  if (notice)
    sendto_one(source_p, ":%s NOTICE %s :*** You are exempt from%s",
	       me.name, source_p->name, buffer);
}

int
do_local_user(const char *nick, struct Client *client_p, struct Client *source_p,
	      const char *username, const char *host, const char *server,
	      const char *realname)
{
  if (source_p == NULL)
    return(0);

  if (!IsUnknown(source_p))
  {
    sendto_one(source_p, form_str(ERR_ALREADYREGISTRED),
               me.name, nick);
    return(0);
  }
  
  source_p->user = make_user(source_p);
  
  /* don't take the clients word for it, ever */
  source_p->user->server = &me;

  strlcpy(source_p->info, realname, sizeof(source_p->info));
  if (!IsGotId(source_p))
  {
    /* save the username in the client
     * If you move this you'll break ping cookies..you've been warned
     */
    strlcpy(source_p->username, username, sizeof(source_p->username));
  }
  if (source_p->name[0])
  { 
    /* NICK already received, now I have USER... */
    if (ConfigFileEntry.drone_detect)
    {
      if (!strcmp(host, "\"hotmail.com\"") && !strcmp(username, nick))
      {
        if (ConfigFileEntry.autodline_drones)
        {
          struct ConfItem *conf;
          struct AccessItem *aconf;

          conf = make_conf_item(DLINE_TYPE);
          aconf = (struct AccessItem *)map_to_conf(conf);
          DupString(aconf->host, source_p->localClient->sockhost);
          DupString(aconf->reason, "Drones");
	  DupString(aconf->oper_reason, "Auto D-Line for \"hotmail.com\" and matching nick/username");
          aconf->hold = CurrentTime + ConfigFileEntry.drone_dline_time;
          exit_client(client_p, source_p, source_p, "D-Lined");
          add_tdline(aconf);
          sendto_realops_flags(UMODE_ALL, L_ALL, "AUTODLINE: %d min. D-Line %s for \"hotmail.com\" and matching nick/username",
			       ConfigFileEntry.drone_dline_time/60,
                               source_p->localClient->sockhost);
          ilog(TRACE, "AUTODLINE: %d min. D-Line %s for \"hotmail.com\" and matching nick/username",
               ConfigFileEntry.drone_dline_time/60,
               source_p->localClient->sockhost);
        }
        exit_client(client_p, source_p, source_p, "Drone Autokill");
        return(CLIENT_EXITED);
      }
      if (!strcmp(nick, realname))
      {
        if (strchr(host, '0') && strchr(server,'0'))
        {
          if (!strchr(host, '.') || !strchr(server, '.'))
          {
	    if (ConfigFileEntry.autodline_drones)
 	    {
              struct ConfItem *conf;
              struct AccessItem *aconf;

              conf = make_conf_item(DLINE_TYPE);
              aconf = (struct AccessItem *)map_to_conf(conf);
	      DupString(aconf->host, source_p->localClient->sockhost);
	      DupString(aconf->reason, "Drones");
	      DupString(aconf->oper_reason, "Auto D-Line for 0 0 in USER and matching nick/gecos");
	      aconf->hold = CurrentTime + ConfigFileEntry.drone_dline_time;
	      exit_client(client_p, source_p, source_p, "D-Lined");
	      add_tdline(aconf);
	      sendto_realops_flags(UMODE_ALL, L_ALL, "AUTODLINE: %d min. D-Line %s for 0 0 in USER and matching nick/gecos",
                		   ConfigFileEntry.drone_dline_time/60,
				   source_p->localClient->sockhost);
              ilog(TRACE, "AUTODLINE: %d min. D-Line %s for 0 0 in USER and matching nick/gecos",
		   ConfigFileEntry.drone_dline_time/60,
		   source_p->localClient->sockhost);
            }
            exit_client(client_p, source_p, source_p, "Drone Autokill");
            return(CLIENT_EXITED);
          }
        }
      }
    }
    return(register_local_user(client_p, source_p, source_p->name, username));
  }
  return(0);
}

/* parv[0] - sender
 * parv[1] - username to change mode for
 * parv[2] - modes to change
 */
void
set_user_mode(struct Client *client_p, struct Client *source_p,
	      int parc, char *parv[])
{
  unsigned int i, flag, setflags;
  char **p, *m;
  struct Client *target_p;
  int what = MODE_ADD;
  int badflag = 0;
  char buf[BUFSIZE];

  if ((target_p = find_person(parv[1])) == NULL)
  {
    if (MyConnect(source_p))
      sendto_one(source_p, form_str(ERR_NOSUCHCHANNEL),
                 me.name, source_p->name, parv[1]);
    return;
  }

  if (source_p != target_p || target_p->from != source_p->from)
  {
     sendto_one(source_p, form_str(ERR_USERSDONTMATCH), me.name, source_p->name);
     return;
  }

  if (parc < 3)
  {
    m = buf;
    *m++ = '+';

    for (i = 0; user_modes[i].letter && (m - buf < BUFSIZE - 4); i++)
      if (source_p->umodes & user_modes[i].mode)
        *m++ = user_modes[i].letter;
    *m = '\0';
    sendto_one(source_p, form_str(RPL_UMODEIS), me.name, source_p->name, buf);
    return;
  }
  setflags = source_p->umodes;

  for (p = &parv[2]; p && *p; p++)
  {
    for (m = *p; *m; m++)
    {
      switch (*m)
      {
        case '+':
          what = MODE_ADD;
          break;
        case '-':
          what = MODE_DEL;
          break;
        case 'o':
          if (what == MODE_ADD)
          {
            if (IsServer(client_p) && !IsOper(source_p))
            {
              ++Count.oper;
              SetOper(source_p);
            }
          }
          else
          {
            if (!IsOper(source_p))
              break;

            ClearOper(source_p);
            source_p->umodes &= ~OUMODE;
            Count.oper--;
            if (MyConnect(source_p))
            {
              dlink_node *dm;

	      detach_conf(source_p, OPER_TYPE);
              ClearOperFlags(source_p);
              if ((dm = dlinkFindDelete(&oper_list, source_p)) != NULL)
              free_dlink_node(dm);
            }
          }
          break;
        case 'D' :
          if (what == MODE_ADD)
          {
            if (!IsDeaf(source_p))
              SetDeaf(source_p);
          }
          else
          {
            if (!IsDeaf(source_p))
              break;

            ClearDeaf(source_p);
          }
          break;
        case 'S' :
          if (what == MODE_ADD)
          {
            if (IsServer(client_p) && !IsService(source_p))
            {
              ++Count.service;
              SetService(source_p);
            }
          }
	  else
	  {
            if (!IsService(source_p))
              break;

	    ClearService(source_p);
	    Count.service--;
	  }
	  break;
        /* we may not get these,
         * but they shouldnt be in default
         */
        case ' ' :
        case '\n' :
        case '\r' :
        case '\t' :
          break;

        default:
          if ((flag = user_modes_from_c_to_bitmask[(unsigned char)*m]))
          {
            if (MyConnect(source_p) && !IsOper(source_p) && (flag & OUMODE))
            {
              badflag = 1;
            }
            else
            {
              if (what == MODE_ADD)
                source_p->umodes |= flag;
              else
                source_p->umodes &= ~flag;
            }
          }
          else
          {
            if (MyConnect(source_p))
              badflag = 1;
          }
          break;
      }
    }
  }

  if (badflag)
    sendto_one(source_p, form_str(ERR_UMODEUNKNOWNFLAG), me.name, source_p->name);

  if ((source_p->umodes & UMODE_NCHANGE) && !IsOperN(source_p))
  {
    sendto_one(source_p, ":%s NOTICE %s :*** You need nick_changes = yes;",
               me.name, source_p->name);
    source_p->umodes &= ~UMODE_NCHANGE;
  }

  if (MyConnect(source_p) && (source_p->umodes & UMODE_ADMIN) &&
     !IsOperAdmin(source_p))
  {
    sendto_one(source_p, ":%s NOTICE %s :*** You need admin = yes;",
               me.name, source_p->name);
    source_p->umodes &= ~UMODE_ADMIN;
  }

  /* compare new flags with old flags and send string which
   * will cause servers to update correctly.  */
  send_umode_out(client_p, source_p, setflags);
}
        
/* send the MODE string for user (user) to connection client_p */
void
send_umode(struct Client *client_p, struct Client *source_p,
           unsigned int old, unsigned int sendmask, char *umode_buf)
{
  int what = 0;
  unsigned int i, flag;
  char *m = umode_buf;

  /* build a string in umode_buf to represent the change in the user's
   * mode between the new (source_p->umodes) and 'old'.
   */
  for (i = 0; user_modes[i].letter; i++)
  {
    flag = user_modes[i].mode;

    if (MyClient(source_p) && !(flag & sendmask))
      continue;

    if ((flag & old) && !(source_p->umodes & flag))
    {
      if (what == MODE_DEL)
        *m++ = user_modes[i].letter;
      else
      {
        what = MODE_DEL;
        *m++ = '-';
        *m++ = user_modes[i].letter;
      }
    }
    else if (!(flag & old) && (source_p->umodes & flag))
    {
      if (what == MODE_ADD)
        *m++ = user_modes[i].letter;
      else
      {
        what = MODE_ADD;
        *m++ = '+';
        *m++ = user_modes[i].letter;
      }
    }
  }

  *m = '\0';
  if (*umode_buf && client_p)
    sendto_one(client_p, ":%s MODE %s :%s",
               source_p->name, source_p->name, umode_buf);
}

/* side effects - Only send ubuf out to servers that know about this client */
void
send_umode_out(struct Client *client_p, struct Client *source_p,
               unsigned int old)
{
  struct Client *target_p;
  char buf[BUFSIZE];
  dlink_node *ptr;

  send_umode(NULL, source_p, old, SEND_UMODES, buf);

  DLINK_FOREACH(ptr, serv_list.head)
  {
    target_p = ptr->data;

    if ((target_p != client_p) && (target_p != source_p) && (*buf))
      sendto_one(target_p, ":%s MODE %s :%s",
                 source_p->name, source_p->name, buf);
  }

  if (client_p && MyClient(client_p))
    send_umode(client_p, source_p, old, ALL_UMODES, buf);
}

/* inputs	- client pointer to client to welcome */
static void
user_welcome(struct Client *source_p)
{		   
  sendto_one(source_p, form_str(RPL_WELCOME), me.name, source_p->name, 
             ServerInfo.network_name, source_p->name);
  sendto_one(source_p, form_str(RPL_YOURHOST), me.name, source_p->name,
             get_listener_name(source_p->localClient->listener), ircd_version);
  sendto_one(source_p, form_str(RPL_CREATED),me.name,source_p->name, creation);
  sendto_one(source_p, form_str(RPL_MYINFO), me.name, source_p->name, me.name, ircd_version);
  show_isupport(source_p);
  show_lusers(source_p);
  send_message_file(source_p, &ConfigFileEntry.motd);

  if (IsRestricted(source_p))
    sendto_one(source_p, form_str(ERR_RESTRICTED),
               me.name, source_p->name);
}

/* inputs	- pointer to client to test
 * outupt	- -1 if exiting 0 if ok
 */
static int
check_x_line(struct Client *client_p, struct Client *source_p)
{
  struct ConfItem *conf;
  struct MatchItem *match_item;
  const char *reason;

  if ((conf = find_matching_name_conf(XLINE_TYPE, source_p->info,
      NULL, NULL, 0)) != NULL)
  {
    match_item = (struct MatchItem *)map_to_conf(conf);
    if (match_item->reason != NULL)
      reason = match_item->reason;
    else
      reason = "X-Lined";
              
    exit_client(client_p, source_p, &me, reason);
    return (CLIENT_EXITED);
  }
  return(0);
}

/* inputs	- pointer to given client to oper
 * side effects - Blindly opers up given source_p, using aconf info
 * all checks on passwords have already been done.
 */
void
oper_up(struct Client *source_p)
{
  unsigned int old = (source_p->umodes & ALL_UMODES);
  const char *operprivs = "";
  struct AccessItem *oconf;

  SetOper(source_p);

  if (ConfigFileEntry.oper_umodes)
    source_p->umodes |= ConfigFileEntry.oper_umodes & ALL_UMODES;
  else
    source_p->umodes |= (UMODE_SERVNOTICE|UMODE_WALLOP|UMODE_LOCOPS) & ALL_UMODES;
	
  Count.oper++;

  dlinkAdd(source_p, make_dlink_node(), &oper_list);
  oconf = map_to_conf((source_p->localClient->confs.head)->data);
  operprivs = oper_privs_as_string(oconf->port);
  SetOFlag(source_p, oconf->port);
  sendto_realops_flags(UMODE_ALL, L_ALL, "Oper %s (%s@%s)", source_p->name,
                       source_p->username, source_p->host);

  if (IsOperAdmin(source_p))
  {
    source_p->umodes |= UMODE_ADMIN;
    sendto_realops_flags(UMODE_ALL, L_ALL, "Admin %s (%s@%s)", source_p->name,
                         source_p->username, source_p->host);
    sendto_server(NULL, NULL, ":%s WALLOPS :Admin %s (%s@%s)",
                  me.name, source_p->name, source_p->username, source_p->host);
  }
  if (!IsOperN(source_p))
    source_p->umodes &= ~UMODE_NCHANGE;

  send_umode_out(source_p, source_p, old);
  sendto_one(source_p, form_str(RPL_YOUREOPER), me.name, source_p->name);
  sendto_one(source_p, ":%s NOTICE %s :*** Your oper privileges are: %s", me.name,
             source_p->name, operprivs);
  send_message_file(source_p, &ConfigFileEntry.opermotd);
}
