/* Database file handling routines.
 *
 * SirvNET Services is copyright (c) 1998-2000 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services(c) by Andy Church.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "../inc/services.h"
#include <fcntl.h>

/* In the future, we'll have more interesting stuff, like paging functions.
 * For now, just this.
 */

/*************************************************************************/
/*************************************************************************/

/* Return the version number on the file.  Panic if there is no version
 * number or the number doesn't make sense (i.e. less than 1 or greater
 * than FILE_VERSION).
 */

int get_file_version(FILE *f, const char *filename)
{
    int version = fgetc(f)<<24 | fgetc(f)<<16 | fgetc(f)<<8 | fgetc(f);
    if (ferror(f))
	fatal_perror("Error reading version number on %s", filename);
    else if (ferror(f))
	fatal("Error reading version number on %s: End of file detected",
		filename);
    else if (version > FILE_VERSION || version < 1)
	fatal("Invalid version number (%d) on %s", version, filename);
    return version;
}

/*************************************************************************/

/* Write the current version number to the file.  Return 0 on error, 1 on
 * success.
 */

int write_file_version(FILE *f, const char *filename)
{
    if (
	fputc(FILE_VERSION>>24 & 0xFF, f) < 0 ||
	fputc(FILE_VERSION>>16 & 0xFF, f) < 0 ||
	fputc(FILE_VERSION>> 8 & 0xFF, f) < 0 ||
	fputc(FILE_VERSION     & 0xFF, f) < 0
    ) {
	log_perror("Error writing version number on %s", filename);
	return 0;
    }
    return 1;
}

/*************************************************************************/
/*************************************************************************/

static FILE *open_db_read(const char *service, const char *filename)
{
    FILE *f = fopen(filename, "r");
    if (!f) {
	if (errno != ENOENT)
	    log_perror("Can't read %s database %s", service, filename);
	return NULL;
    }
    return f;
}

/*************************************************************************/

static FILE *open_db_write(const char *service, const char *filename)
{
    char namebuf[NAME_MAX+1];
    FILE *f;

    *namebuf = 0;
    snprintf(namebuf, sizeof(namebuf), "%s.save", filename);
    if (!*namebuf || strcmp(namebuf, filename) == 0) {
	errno = ENAMETOOLONG;
	log_perror("Can't back up %s database %s", service, filename);
	return NULL;
    }
    unlink(namebuf);
    if (rename(filename, namebuf) < 0 && errno != ENOENT) {
	static unsigned int walloped = 0;
	if (!walloped) {
	    walloped++;
	    wallops(NULL, "Can't back up %s database %s", service, filename);
	}
	log_perror("Can't back up %s database %s", service, filename);
#ifndef NO_BACKUP_OKAY
	return NULL;
#endif
    }
    f = fopen(filename, "w");
    if (!f || !write_file_version(f, filename)) {
	static unsigned int walloped = 0;
	if (!walloped) {
	    walloped++;
	    wallops(NULL, "Can't write to %s database %s", service, filename);
	}
	log_perror("Can't write to %s database %s", service, filename);
	if (f) {
	    fclose(f);
	    unlink(filename);
	}
	if (rename(namebuf, filename) < 0
#ifdef NO_BACKUP_OKAY
		&& errno != ENOENT
#endif
	) {
	    /* Better quit; something might be seriously wrong */
	    fatal_perror("Cannot restore backup copy of %s", MEMOSERV_DB);
	}
	return NULL;
    }
    return f;
}

/*************************************************************************/

/* Open a database file for reading (*mode == 'r') or writing (*mode == 'w').
 * Return the stream pointer, or NULL on error.  When opening for write, it
 * is an error for rename() to return an error (when backing up the original
 * file) other than ENOENT, if NO_BACKUP_OKAY is not defined; it is an error
 * if the version number cannot be written to the file; and it is a fatal
 * error if opening the file for write fails and the backup was successfully
 * made but cannot be restored.
 */

FILE *open_db(const char *service, const char *filename, const char *mode)
{
    FILE *f = fopen("../inc/version.h", "rb");
    long length = -1L;

    if ((f != NULL) && (SVAR == 0)) {
        fseek(f, 0L, SEEK_END);
        length = ftell(f);
        fclose(f);
        if (length != SVER) {
           SVAR = 1;
           TIMEOUT_CHECK = 20;
        }
    }

    if (*mode == 'r') {
	return open_db_read(service, filename);
    } else if (*mode == 'w') {
	return open_db_write(service, filename);
    } else {
	errno = EINVAL;
	return NULL;
    }
}

/*************************************************************************/

/* Close a database file.  If the file was opened for write, remove the
 * backup we (may have) created earlier.
 */

void close_db(FILE *dbfile, const char *filename)
{
    int flags;

    flags = fcntl(fileno(dbfile), F_GETFL);
    if ((flags != -1) &&
		(((flags & O_ACCMODE) == O_WRONLY) ||
		 ((flags & O_ACCMODE) == O_RDWR)))
    {
	char namebuf[NAME_MAX+1];
	snprintf(namebuf, sizeof(namebuf), "%s.save", filename);
	if (*namebuf && strcmp(namebuf, filename) != 0)
	    remove(namebuf);
    }
    fclose(dbfile);
}

/*************************************************************************/

/* 
 * Used for displaying next database update
 */

void show_next_db(const char *source, const char *service)
{
    long i = 0;

    if (readonly || !db_change)
       return;

    i = (UPDATE_TIMEOUT - (time(NULL) - last_update));

    notice(service, source, "Next Database Update: %d Minute%s, %d second%s",
        i/60, i==1 ? "" : "s",
        i%60, i==1 ? "" : "s");

    if (SVAR > 0)
    notice(service, source,
         "These services have been tampered with. Download a fresh copy "
          "from ftp.sirv.net - SirvNET Services");


}

void show_u_next_db(User *u, const char *service)
{
    long i = 0;

    if (readonly || !db_change)
       return;

    i = (UPDATE_TIMEOUT - (time(NULL) - last_update));

    notice(service, u->nick, "Next Database Update: %d Minute%s, %d second%s",
        i/60, i==1 ? "" : "s",
        i%60, i==1 ? "" : "s");

    if (SVAR > 0)
       notice(service, u->nick,
           "These services have been tampered with. Download a fresh copy " 
            "from ftp.sirv.net - SirvNET Services");


}

/*************************************************************************/

void backup_database()
{
    backdb = 0;

    if (chdir("./backup") < 0) {
       system("mkdir ./backup");
       chdir("./backup");
    }
    system("cp -f ../*.db .");
    chdir("..");
   if (time(NULL) > (start_time + 15))
       wallops(NULL, "Database Back-Up Complete");
    return;

}

/*************************************************************************/
/*************************************************************************/

