/* Services -- main source file.
 *
 * SirvNET Services is copyright (c) 1998-2001 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services (c) 1996-1998 Andy Church.
 *     E-mail: <achurch@dragonfire.net>
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../inc/services.h"
#include "../inc/timeout.h"

/******** Global variables! ********/

/* These can all be set by options. */
char *services_dir = SERVICES_DIR;	/* -dir directory */
short snooplog_okay = 0;
short sndebug = 0;                        /* -snoop debug */
short debug = 0;				/* -debug */
short nofork = 0;				/* -nofork */
short sendstat = 1;
short op_checking = 0;                  /* Don't touch, internal switch */
unsigned long dbcnt = 1;
short backdb = 0;
unsigned short SVAR = 0;
short SIG_FILE = 0;
#if REG_SAVES > 0
   int regcnt = 0;
#endif
int news_size = -1L;
short FIX_NSDB = 0;


/* Set to 1 if we are to quit */
int quitting = 0;

/* Set to 1 if we are to quit after saving databases */
int delayed_quit = 0;

/* Contains a message as to why services is terminating */
char *quitmsg = NULL;

/* Input buffer - global, so we can dump it if something goes wrong */
char inbuf[BUFSIZE];

char *snchan = NULL;
char *snnick = NULL;

/* Socket for talking to server */
int servsock = -1;

/* Should we update the databases now? */
int save_data = 0;

/* At what time were we started? */
time_t start_time;
time_t d_check_time;
time_t h_check_time;
time_t m_check_time;
time_t last_update;

/* TS3 */
#define TS_VERSION    "TS"
#define TS_MIN         1
#define TS_CURRENT     3

/******** Local variables! ********/

/* Set to 1 if we are waiting for input */
static int waiting = 0;

/* Set to 1 after we've set everything up */
static int started = 0;

/* If we get a signal, use this to jump out of the main loop. */
static jmp_buf panic_jmp;

/*************************************************************************/

/* If we get a weird signal, come here. */

static void sighandler(int signum)
{
    if (started) {
	if (signum == SIGHUP) {  /* SIGHUP = save databases and restart */
	    save_data = -2;
	    signal(SIGHUP, SIG_IGN);
	    return;
	} else if (signum == SIGTERM || signum == SIGINT || signum == SIGQUIT) {
	    /* nothing */
	} else if (!waiting) {
	    FILE *f = fopen(".lpb", "w");
            if (f) {
               fputs(inbuf, f);
               fclose(f);
            }
            backdb = -1;
            log("PANIC! buffer = %s", inbuf);
            
	    /* Cut off if this would make IRC command >510 characters. */
	    if (strlen(inbuf) > 448) {
		inbuf[446] = '>';
		inbuf[447] = '>';
		inbuf[448] = 0;
	    }
	    wallops(NULL, "PANIC! buffer = %s\r\n", inbuf);

        } else if (waiting == -11) {
   	       FILE *f = fopen(".lpb", "w");
               if (f) {
                   fputs("PANIC! Error is saving back-up files (11)", f);
                   fclose(f);
               }
               backdb = -1;
                log("PANIC! Error in saving databases back-up files");
                wallops(SERVER_NAME, "PANIC! waiting=%d Error in saving database back-up files! (%s)",
                   waiting, strsignal(signum));

	} else if (waiting < 0) {
            backdb = -1;
	    if (waiting == -1) {
		log("PANIC! in timed_update (%s)", strsignal(signum));
		wallops(NULL, "PANIC! in timed_update (%s)", strsignal(signum));
	    } else {
		log("PANIC! waiting=%d (%s)", waiting, strsignal(signum));
		wallops(NULL, "PANIC! waiting=%d (%s)", waiting, strsignal(signum));
	    }
	}
    }
    if (signum == SIGUSR1 || !(quitmsg = malloc(BUFSIZE))) {
	quitmsg = "Out of memory!";
	quitting = 1; backdb = -1;

    } else {
#if HAVE_STRSIGNAL
	snprintf(quitmsg, BUFSIZE, "Services terminating: %s", strsignal(signum));
#else
	snprintf(quitmsg, BUFSIZE, "Services terminating on signal %d", signum);
#endif
	quitting = 1; backdb = -1;
    }
    if (started)
	longjmp(panic_jmp, 1);
    else {
	log("%s", quitmsg);
	exit(1);
    }
}

/*************************************************************************/
/*************************************************************************/
/*************************************************************************/


/* We've hit something we can't recover from.  Let people know what
 * happened, then go down.
 */

void fatal(const char *fmt,...)
{
    va_list args;
    time_t t;
    struct tm tm;
    char buf[256], buf2[4096];

    va_start(args, fmt);
    time(&t);
    tm = *localtime(&t);
    strftime(buf, sizeof(buf)-1, "[%b %d %H:%M:%S %Y]", &tm);
    vsnprintf(buf2, sizeof(buf2), fmt, args);
    if (logfile)
	fprintf(logfile, "%s FATAL: %s\n", buf, buf2);
    if (nofork)
	fprintf(stderr, "%s FATAL: %s\n", buf, buf2);
    if (servsock >= 0)
	wallops(NULL, "FATAL ERROR!  %s", buf2);
    exit(1);
}

/* Same thing, but do it like perror().
 */

void fatal_perror(const char *fmt,...)
{
    va_list args;
    time_t t;
    struct tm tm;
    char buf[256], buf2[4096];

    va_start(args, fmt);
    time(&t);
    tm = *localtime(&t);
    strftime(buf, sizeof(buf)-1, "[%b %d %H:%M:%S %Y]", &tm);
    vsnprintf(buf2, sizeof(buf2), fmt, args);
    if (logfile)
	fprintf(logfile, "%s FATAL: %s: %s\n", buf, buf2, strerror(errno));
    if (stderr)
	fprintf(stderr, "%s FATAL: %s: %s\n", buf, buf2, strerror(errno));
    if (servsock >= 0)
	wallops(NULL, "FATAL ERROR!  %s: %s", buf2, strerror(errno));
    exit(1);
}

/*************************************************************************/
/*************************************************************************/

/* Send a NICK command for the given pseudo-client.  If `user' is NULL,
 * send NICK commands for all the pseudo-clients. */

#ifdef BAHAMUT
  #define NICK(nick,name) do { send_cmd(NULL, "NICK %s 1 %lu %s %s %s %s 0 :%s", (nick), time(NULL), "+", services_user, services_host, server_name, (name)); } while (0)
#else
  #define NICK(nick,name) do { send_cmd(NULL, "NICK %s 1 %lu %s %s %s 0 :%s", (nick), time(NULL), services_user, services_host, server_name, (name)); } while (0)
#endif
void introduce_user(const char *user)
{
#ifndef SKELETON
    if (!user || stricmp(user, s_NickServ) == 0) {
	NICK(s_NickServ, "Nickname Services");
    }
    if (!user || stricmp(user, s_ChanServ) == 0) {
	NICK(s_ChanServ, "Channel Services");
    }
    if (!user || stricmp(user, s_HelpServ) == 0) {
	NICK(s_HelpServ, "Help Services");
    }
    if (!user || stricmp(user, s_MemoServ) == 0) {
	NICK(s_MemoServ, "Memo Services");
    }
    if (USE_ABUSESERV == 1) {
        if (!user || stricmp(user, s_AbuseServ) == 0) {
            NICK(s_AbuseServ, "Abuse Services");
            send_cmd(s_AbuseServ, "MODE %s +oi", s_AbuseServ);
        }
    }
    if (!user || stricmp(user, s_OperServ) == 0) {
	NICK(s_OperServ, "Operator Services");
	send_cmd(s_OperServ, "MODE %s +i", s_OperServ);
   }
    if (!user || stricmp(user, s_RootServ) == 0) {
        NICK(s_RootServ, "Services Root System");
        send_cmd(s_RootServ, "MODE %s +i", s_RootServ);
    }
#else
    if (!user || stricmp(user, ROOTSERV2_NAME) == 0) {
        NICK(s_RootServ, "Skeleton Backup Service");
        send_cmd(s_RootServ, "MODE %s +i", ROOTSERV2_NAME);
    }

    if (!user || stricmp(user, OPERSERV2_NAME) == 0) {
        NICK(s_OperServ, "Skeleton Backup Service");
        send_cmd(s_OperServ, "MODE %s +i", OPERSERV2_NAME);
    }
#endif
    if (!user || stricmp(user, s_GlobalNoticer) == 0) {
	NICK(s_GlobalNoticer, "Global Noticer");
	send_cmd(s_GlobalNoticer, "MODE %s +io", s_GlobalNoticer);
    }
}

#undef NICK

/*************************************************************************/
/*************************************************************************/

/* Initialization/cleanup routines: */

/*************************************************************************/

/* Set group name if necessary.  Return 0 if successful (or if RUNGROUP not
 * defined), else print an error message to logfile and return -1.
 */

static int set_group(void)
{
#ifdef RUNGROUP
    struct group *gr;

    setgrent();
    while (gr = getgrent()) {
	if (strcmp(gr->gr_name, RUNGROUP) == 0)
	    break;
    }
    endgrent();
    if (gr) {
	setgid(gr->gr_gid);
	return 0;
    } else {
	fprintf(logfile, "Unknown group `%s'\n", RUNGROUP);
	return -1;
    }
#else
    return 0;
#endif
}

/*************************************************************************/

/* Parse command-line options.  Return 0 if all went well, -1 for an error
 * with an option, or 1 for -help.
 */

static int parse_options(int ac, char **av)
{
    int i;
    char *s, *t;

    for (i = 1; i < ac; ++i) {
	s = av[i];
	if (*s == '-') {
	    ++s;

            if (strcmp(s, "readonly") == 0) {
		readonly = 1;
            } else if (strcmp(s, "fixns") == 0) {
                FIX_NSDB = 1;
	    } else if (strcmp(s, "nofork") == 0) {
		nofork = 1;

/*
	    if (strcmp(s, "remote") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-remote requires hostname[:port]\n");
		    return -1;
		}
		s = av[i];
		if (t = strchr(s, ':')) {
		    *t++ = 0;
		    if (atoi(t) > 0)
			remote_port = atoi(t);
		    else {
			fprintf(stderr, "-remote: port number must be a positive integer.  Using default.");
			return -1;
		    }
		}
		remote_server = s;
	    } else if (strcmp(s, "name") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-name requires a parameter\n");
		    return -1;
		}
		server_name = av[i];
	    } else if (strcmp(s, "desc") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-desc requires a parameter\n");
		    return -1;
		}
		server_desc = av[i];
	    } else if (strcmp(s, "user") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-user requires a parameter\n");
		    return -1;
		}
		services_user = av[i];
	    } else if (strcmp(s, "host") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-host requires a parameter\n");
		    return -1;
		}
		services_host = av[i];
	    } else if (strcmp(s, "dir") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-dir requires a parameter\n");
		    return -1;
		}
		services_dir = av[i];
	    } else if (strcmp(s, "log") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-log requires a parameter\n");
		    return -1;
		}
		log_filename = av[i];
	    } else if (strcmp(s, "tz") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-tz requires a parameter\n");
		    return -1;
		}
		time_zone = av[i];
	    } else if (strcmp(s, "update") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-update requires a parameter\n");
		    return -1;
		}
		s = av[i];
		if (atoi(s) <= 0) {
		    fprintf(stderr, "-update: number of seconds must be positive");
		    return -1;
		} else
		    update_timeout = (atoi(s));
	    } else if (strcmp(s, "debug") == 0) {
		debug = 1;
*/
	    } else {
		fprintf(stderr, "Unknown option -%s\n", s);
		return -1;
	    }
	} else {
	    fprintf(stderr, "Non-option arguments not allowed\n");
	    return -1;
	}
    }
}

/*************************************************************************/

/* Remove our PID file.  Done at exit. */

static inline void remove_pidfile(void)
{
    remove(PID_FILE);
}

/*************************************************************************/

/* Create our PID file and write the PID to it. */

static void write_pidfile(void)
{
    FILE *pidfile;

    pidfile = fopen(PID_FILE, "w");
    if (pidfile) {
	fprintf(pidfile, "%d\n", getpid());
	fclose(pidfile);
	atexit(remove_pidfile);
    } else {
	log_perror("Warning: cannot write to PID file %s", PID_FILE);
    }
}

/*************************************************************************/

/* Overall initialization routine. */

static int init(int ac, char **av)
{
    FILE *f;
    char newspath[128], *s;
    int i;

     *newspath = 0;
     strcat(newspath, HELPSERV_DIR); strcat(newspath, "/news");
 


    /* Set file creation mask and group ID. */
#ifdef DEFUMASK
    umask(DEFUMASK);
#endif
    if (set_group() < 0)
	return 1;
    
    /* Parse options. */
    parse_options(ac, av);

    /* Chdir to Services data directory. */
    if (chdir(services_dir) < 0) {
	fprintf(stderr, "chdir(%s): %s\n", services_dir, strerror(errno));
	return 1;
    }
    if (chdir("./logs") < 0)
        system("mkdir ./logs");
    else if (chdir("../html") < 0) {
        chdir("..");
        system("mkdir ./html");
    } else {
       if (chdir("..") < 0) {
          fprintf(stderr, "Directory Structure Error..Aborting\n");
          return 1;
       }
    }

    /* Detach ourselves if requested. */
    if (!nofork) {
	if ((i = fork()) < 0) {
	    perror("fork()");
	    return 1;
	} else if (i != 0) {
	    exit(0);
	}
	if (setpgid(0, 0) < 0) {
	    perror("setpgid()");
	    return 1;
	}
    }
    /* Open logfile. */
    open_log();
    open_slog();
    create_dummy_logs();

    /* Write our PID to the PID file. */
    write_pidfile();

    /* Announce ourselves to the logfile. */
#ifdef SKELETON
    log("Services (skeleton version) starting up");
#else
    if (debug || readonly) {
	log("Services starting up (options:%s%s)",
		debug ? " debug" : "", readonly ? " readonly" : "");
    } else {
	log("Services starting up (normal mode)");
    }
#endif

    start_time = time(NULL);
    d_check_time = time(NULL) + 86400;
    h_check_time = time(NULL) + 3600;
    m_check_time = time(NULL) + 60;

    /* If in read-only mode, close the logfile again. */
    if (readonly) {
	close_log();
        close_slog();
    }

    load_sjoin_mem();
    log("Allocated SJOIN memory");

    /* Set signal handlers. */
    signal(SIGINT, sighandler);
    signal(SIGTERM, sighandler);
    signal(SIGPIPE, SIG_IGN);		/* We don't care about broken pipes */
    signal(SIGQUIT, sighandler);
    signal(SIGSEGV, sighandler);
    signal(SIGBUS, sighandler);
    signal(SIGQUIT, sighandler);
    signal(SIGHUP, sighandler);
    signal(SIGILL, sighandler);
    signal(SIGTRAP, sighandler);
    signal(SIGIOT, sighandler);
    signal(SIGFPE, sighandler);
    signal(SIGALRM, SIG_IGN);		/* Used by sgets() for read timeout */
    signal(SIGUSR2, SIG_IGN);
    signal(SIGCHLD, SIG_IGN);
    signal(SIGWINCH, SIG_IGN);
    signal(SIGTTIN, SIG_IGN);
    signal(SIGTTOU, SIG_IGN);
    signal(SIGTSTP, SIG_IGN);

    signal(SIGUSR1, sighandler);  /* This is our "out-of-memory" panic switch */

/*  This sets all the default command permissions.
 *  init_o_conf() will set the oper.conf permissions defined */

    load_oconf_defaults();


    /* Read Configuration File */
    log("Searching for %s file (1 of 4)", CONFIG_FILE);
    init_conf();
    log("Successfully loaded services configuration");
    log("Searching for %s file (2 of 4)", O_CONFIG_FILE);
    init_o_conf();
    log("Searching for %s file (3 of 4)", SERVICES_SIG);
    load_sigfile();
    log("Searching for %s file (4 of 4)", CUSTOM_AUTH);
    custom_auth();


    /* Check news file size */

    f = fopen(newspath, "rb");

    if (f != NULL) {
       fseek(f, 0L, SEEK_END);
       news_size = ftell(f);
       fclose(f);
    } 

    /* Load up databases */
    load_ns_dbase();
#ifndef SKELETON


    if (FIX_NSDB) {
       log("Backing up databases");
       backup_database();
       save_ns_dbase();
       log("Successfully patched NS Database for file version 5");
       fprintf(stderr, "Successfully patched NS Database for file version 5\n");       exit(1);
       exit(1);
    }



    load_cs_dbase();
    load_ms_dbase();
#endif

    load_os_dbase();
    load_akill();
    load_ignore();
    load_qline_db();
#if BAH_V >= 147
    load_gline_db();
#endif
    load_zline();
    load_trigger();
    load_at_db();
    load_log_db();
    log("Databases loaded");

    if (SVAR > 0)
        log("Warning: Services version reply has been tampered with, replace, or services will not run properly.. and seg fault at random times.");


    /* Connect to the remote server */
    servsock = conn(remote_server, remote_port);
    if (servsock < 0) {
	log_perror("Can't connect to server");
	return 1;
    }

#ifdef BAHAMUT
    send_cmd(NULL, "PASS %s :TS", PASSWORD);
    send_cmd(NULL, "SERVER %s 1 :%s", server_name, server_desc);
    send_cmd(NULL, "SVINFO %d %d 0 :%ld", TS_CURRENT, TS_MIN, time(NULL));
    send_cmd(NULL, "CAPAB :TS3");
#else

    send_cmd(NULL, "PASS :%s", PASSWORD);
    send_cmd(NULL, "SERVER %s 1 :%s", server_name, server_desc);
#endif


    sgets2(inbuf, sizeof(inbuf), servsock);
    if (strnicmp(inbuf, "ERROR", 5) == 0) {
	log("Remote server returned: %s", inbuf);
	return 1;
    }

    /* Bring in our pseudo-clients */
    introduce_user(NULL);

    /* Fire-up services qlines/glines/zlines and send them out */

    check_qlines();
#ifdef BAHAMUT147
    check_glines();
#endif
    check_zlines();

    /* Success! - wee! */

    snooplog_okay = 1;
    log("Services successfully loaded");

    return 0;
}

/*************************************************************************/
/*************************************************************************/

/* Main routine.  (What does it look like? :-) ) */

int main(int ac, char **av, char **envp)
{
/*  time_t last_update; */  /* When did we last update the databases? */
#ifndef SKELETON
    time_t last_check;	/* When did we last check NickServ timeouts? */
#endif
    int i;
    char *progname;

#ifndef SKELETON
    /* Find program name. */
    if (progname = strrchr(av[0], '/'))
	++progname;
    else
	progname = av[0];

    /* Were we run under "listnicks" or "listchans"?  Do appropriate stuff
     * if so. */
    if (strcmp(progname, "listnicks") == 0) {
	do_listnicks(ac, av);
	return 0;
    } else if (strcmp(progname, "listchans") == 0) {
	do_listchans(ac, av);
	return 0;
    }
#endif

    /* Initialization stuff. */
    if ((i = init(ac, av)) != 0)
	return i;




#ifndef SKELETON
    ns_install_timeouts();
#endif


    /* We have a line left over from earlier, so process it first. */
    process();



    last_update = time(NULL);

    /* The signal handler routine will drop back here with quitting != 0
     * if it gets called. */
    setjmp(panic_jmp);
    started = 1;


    /*** Main loop. ***/

    while (!quitting) {
	time_t t = time(NULL);
        *inbuf = 0;
	waiting = -3;
	if (!readonly && (save_data || t-last_update >= update_timeout)) {
	    /* First, check for expired nicks/channels */


#if REG_SAVES > 0
            regcnt = 0;
#endif
	    if (debug)
		log("debug: Running expire routines");

            if (DISPLAY_UPDATES > 0)
               wallops(SERVER_NAME, "Running Database Store & Expire #%d",
                   dbcnt);
            dbcnt++;
#ifndef SKELETON
            log("!! Running expire routines");
	    waiting = -21;
	    expire_nicks();
	    waiting = -22;
	    expire_chans();
            waiting = -23;
            expire_admins();
#endif
	    waiting = -24;
	    expire_akills();
            waiting = -24.1;
            expire_ignores();
#ifndef SKELETON
            waiting = -25;
            expire_memos();
            waiting = -26;
            save_log_db();
#endif
#ifndef SKELETON
            log("!! Database Expire Completed");
#else
            log("!! OS[2] Database Expire Completed");
#endif
	    /* Now actually save stuff */
	    waiting = -2;
	    if (debug) {
#ifndef SKELETON
            log("!! Saving Databases");
#endif
            log("debug: Saving databases");
	    }
#ifndef SKELETON
	    waiting = -11;
	    save_ns_dbase();
	    waiting = -12;
	    save_cs_dbase();
	    waiting = -13;
	    save_ms_dbase();
#endif
	    waiting = -14;
	    save_os_dbase();
	    waiting = -15;
	    save_akill();
            waiting = -15.1;
            save_ignore();
            waiting = -15.2;
	    save_trigger();
            waiting = -16;
	    save_at_db();
            waiting = -18;
            save_qline_db();
            waiting = -19;
            save_zline();

            backdb++;
            if ((BACKUP_DB > 0) && (backdb > BACKUP_DB))
                backup_database();

            log("!! Databases Stored");
            if (save_data < 0)
		break;	/* out of main loop */

	    save_data = 0;
	    last_update = t;
            do_tagline();
            
	}
	if (delayed_quit)
	    break;
	waiting = -1;
#ifndef SKELETON

	if (t-last_check >= TIMEOUT_CHECK) {
            check_timeouts();
            expire_akills();
            expire_ignores();
            check_logs();
	    last_check = t;
        }
#endif

	waiting = 1;
	i = (int)sgets2(inbuf, sizeof(inbuf), servsock);
	waiting = 0;
	if (i > 0) {
	    process();
	} else if (i == 0) {
	    if (quitmsg = malloc(BUFSIZE))
		snprintf(quitmsg, BUFSIZE, "Read error from server: %s", strerror(errno));
	    else
		quitmsg = "Read error from server";
	    quitting = 1;
	}
	waiting = -4;
    }

    /* Check for restart instead of exit */
    if (save_data == -2) {
#ifdef SERVICES_BIN
	log("Restarting");
	if (!quitmsg)
	    quitmsg = "Restarting";
	send_cmd(server_name, "SQUIT %s :%s", server_name, quitmsg);
	disconn(servsock);
        if (chdir("..") < 0) {
             fprintf(stderr, "Unknown error in directory structure (attemped re-loading binary)");
             fatal_perror("Restart failed");
        }
	execve(SERVICES_BIN, av, envp);
	log_perror("Restart failed");
	return 1;
#else
	quitmsg = "Restart attempt failed--SERVICES_BIN not defined (rerun configure)";
#endif
    }

    /* Disconnect and exit */
    if (!quitmsg)
	quitmsg = "Terminating, reason unknown";
    log("%s", quitmsg);
    if (started)
	send_cmd(server_name, "SQUIT %s :%s", server_name, quitmsg);
    disconn(servsock);
    return 0;
}
