/* OperServ functions.
 *
 * SirvNET Services is copyright (c) 1998-2001 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services(c) by Andy Church.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "../inc/services.h"
#include "../inc/version2.h"
#include "../inc/timeout.h"
#include "os-help.c"

/*************************************************************************/

/* SENDLOGS list */
char *sendlogs[MAX_SENDLOGS];

/* Services Q-LINED list */
char *s_qlines[64];

/* Services G-LINED list */
#if BAH_V >= 147
   char *s_glines[64];
#endif

#ifdef SKELETON
char *s_services_helpers[32];
char *s_services_opers[32];
char *s_services_admins[32];
char *s_services_roots[32];
#endif

static int nakill = 0;
static int akill_size = 0;

static int nignore = 0;
static int ignore_size = 0;

static int njupe = 0;
static int jupe_size = 0;

static int ntrigger = 0;
static int trigger_size = 0;

static int nzline = 0;
static int zline_size = 0;

static struct akill {
    char *mask;
    char *reason;
    char who[NICKMAX];
    time_t time;
    time_t expires;	/* or 0 for no expiry */
    long reserved[4];
} *akills = NULL;


static struct zline {
    char *host;
    char *reason;
    char *who;
} *zlines = NULL;

static struct ignore {
    char *mask;
    char who[NICKMAX];
    time_t time;
    time_t expires;	/* or 0 for no expiry */
    long reserved[4];
} *ignores = NULL;


static struct trigger {
    char *mask;
    long tvalue;
    char who[NICKMAX];
    long reserved[4];
} *trigger = NULL;

static struct jupe {
    char *jserver;
    char *reason;
    char who[NICKMAX];
    long reserved[4];
} *jupes = NULL;


struct clone {
    char *host;
    long time;
};

/*  List of most recent users - statically initialized to zeros */
static struct clone clonelist[CLONE_DETECT_SIZE];


/* Which hosts have we warned about, and when?  This is used to keep us
 * from sending out notices over and over for clones from the same host. */
static struct clone warnings[CLONE_DETECT_SIZE];


/*************************************************************************/

void load_os_sa_db(void);
void load_os_sra_db(void);
void load_os_sop_db(void);
void load_os_hop_db(void);
void save_os_sa_db(void);
void save_os_sra_db(void);
void save_os_sop_db(void);
void save_os_hop_db(void);
void load_qline_db(void);
void save_qline_db(void);
#if BAH_V >= 147
  void load_gline_db(void);
  void save_gline_db(void);
#endif
static void add_akill(const char *mask, const char *reason, const char *who,
		      const char *expiry, int akill_num);
static void add_trigger(const char *cmd, const char *tvalue);
static int del_trigger(const char *cmd);
static int del_akill(const char *mask);
static int del_ignore(const char *mask);
static int del_jupe(const char *jserver);
static void do_ohelp(const char *source);
static void do_clear(const char *source);
static void do_inject(const char *source);
static void do_sop(const char *source);
static void do_hop(const char *source);
static void do_akill(const char *source);
static void do_jupe(const char *source);
static void do_sqline(const char *source);
#if BAH_V >= 147
  static void do_sgline(const char *source);
#endif
static void do_zline(const char *source);
static void do_stats(const char *source);
static void do_settings(const char *source);
static void do_uptime(const char *source);
static void do_version(const char *source);
static void do_mem(const char *source);
static void do_listadm(const char *source);
static void do_uinfo(const char *source);
static void do_trigger(const char *source);
static void do_ignore(const char *source);
static void do_sendlog(const char *source);
static void send_clone_lists(const char *source);
#ifdef SKELETON
static void do_identify(const char *source);
#endif

/*************************************************************************/
/*************************************************************************/

/* Main OperServ routine. */

void operserv(const char *source, char *buf)
{
    NickInfo *ni;
    char *cmd;
    char *s, *chan, *nick, *host;
    int i;
    User *u = finduser(source);


#ifdef SKELETON
    ni = findnick(source);

    log("OS[2] (%s!%s@s) [%s]",source,u->username,u->host,buf);
    do_break_log("OS", "OS[2] (%s!%s@s) [%s]",source,u->username,u->host,buf);
#else
    log("OS (%s!%s@%s) [%s]",source,u->username,u->host,buf);
    do_break_log("OS", "OS (%s!%s@%s) [%s]",source,u->username,u->host,buf);
#endif

    cmd = strtok(buf, " ");

    if (!cmd) {
        notice(s_OperServ, source, "Type \"/msg OperServ OHELP\" for a listing OperServ commands.");
        return;
    } else if (stricmp(cmd, "\1PING") == 0) {

	if (!(s = strtok(NULL, "")))
	    s = "\1";
	notice(s_OperServ, source, "\1PING %s", s);

      } else if (stricmp(cmd, "HELP") == 0) {
        notice(s_OperServ, source, "OperServ permits IRC Operators, or services administrators");
        notice(s_OperServ, source, "to more easily take control of the network with powerful");
        notice(s_OperServ, source, "commands. Below are the index lists of all commands");
        notice(s_OperServ, source, "under each service.");
        notice(s_OperServ, source, "-");
        notice(s_OperServ, source, "IRC Operator Services Commands Index:");
        notice(s_OperServ, source, "For NickServ Help:");
        notice(s_OperServ, source, "\"\2/msg NickServ OHELP\2\"");
        notice(s_OperServ, source, "For ChanServ Help:");
        notice(s_OperServ, source, "\"\2/msg ChanServ OHELP\2\"");
        notice(s_OperServ, source, "For MemoServ Help:");
        notice(s_OperServ, source, "\"\2/msg MemoServ OHELP\2\"");
        notice(s_OperServ, source, "For OperServ Help:");
        notice(s_OperServ, source, "\"\2/msg OperServ OHELP\2\"");

    } else if (stricmp(cmd, "OHELP") == 0) {
        do_ohelp(source);

#ifdef SKELETON
    } else if (stricmp(cmd, "IDENTIFY") == 0) {
        do_identify(source);
#endif

    } else if (stricmp(cmd, "GLOBAL") == 0) {

        User *user;
        const char *target = strtok(NULL, " ");
	const char *msg = strtok(NULL, "");

	if (!msg) {
            notice(s_OperServ, source, "Syntax: \2GLOBAL\2 <target> <message>");
	    notice(s_OperServ, source, "\2/msg %s OHELP GLOBAL for more information.", s_OperServ);
	} else if (stricmp(target, "all") == 0 || stricmp(target, "*") == 0 
             || stricmp(target, "*.*") == 0) {
             wallops(SERVER_NAME, "%s used the Global Notice Command [Target: ALL]",
                source);
             for (user = userlist; user; user = user->next) {
                notice(s_GlobalNoticer, user->nick, "\2-Global Notice-\2 %s \2-No Replies Please-\2", msg);
             }
        } else {
             notice(s_GlobalNoticer, "#%s", "\2-Global Notice-\2 %s \2-No Replies Please-\2", target, msg);
             wallops(SERVER_NAME, "%s used the Global Notice Command [Target: %s]",
                source, target);

        }


    } else if (stricmp(cmd, "LISTADM") == 0) {
	do_listadm(source);

    } else if (stricmp(cmd, "STATS") == 0) {
        do_stats(source);

    } else if (stricmp(cmd, "SETTINGS") == 0) {
        do_settings(source);

    } else if (stricmp(cmd, "UPTIME") == 0) {
        do_uptime(source);

    } else if (stricmp(cmd, "VERSION") == 0) {
        do_version(source);

    } else if (stricmp(cmd, "MEM") == 0) {
        do_mem(source);


   /* These are only avail to opers for LISTing */

    } else if (stricmp(cmd, "AKILL") == 0) {
	do_akill(source);

    } else if (stricmp(cmd, "AUTOKILL") == 0) {
        do_akill(source);

    } else if (stricmp(cmd, "IGNORE") == 0) {
        do_ignore(source);

    } else if (stricmp(cmd, "TRIGGER") == 0) {
        do_trigger(source);

    } else if (stricmp(cmd, "SQLINE") == 0) {
        do_sqline(source);

#if BAH_V >= 147
    } else if (stricmp(cmd, "SGLINE") == 0) {
        do_sgline(source);
#endif

    } else if (stricmp(cmd, "HOP") == 0) {
        do_hop(source);

    } else if (stricmp(cmd, "SOP") == 0) {
        do_sop(source);



#ifdef SKELETON
    } else if (!ni || (!(ni->flags & NI_IDENTIFIED))) {
        notice(s_OperServ, source, "Permission denied.");
#endif


    } else if ((!check_o_access(source, cmd)) &&
              (stricmp(cmd, "AKILL") != 0) && 
              (stricmp(cmd, "AUTOKILL") != 0)) {
        notice(s_OperServ, source, "Permission Denied");


    } else if (stricmp(cmd, "UINFO") == 0) {
        do_uinfo(source);


    } else if (stricmp(cmd, "CLEAR") == 0) {
        do_clear(source);

    } else if (stricmp(cmd, "HELPOP") == 0) {
        do_hop(source);

    } else if (stricmp(cmd, "SENDLOGS") == 0) {
         do_sendlog(source);

    } else if (stricmp(cmd, "UPDATE") == 0) {
        notice(s_OperServ, source, OS_UPDATE);
        log("!! Updated Databases");
	save_data = 1;

    } else if (stricmp(cmd, "JUPE") == 0) {
	do_jupe(source);

    } else if (stricmp(cmd, "ZLINE") == 0) {
        do_zline(source);

    } else {
         notice(s_OperServ, source,
		"Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
		cmd, s_OperServ);

    }
}

/*************************************************************************/
/*************************************************************************/

/* Delete any expired nicks in the SRA/SA/SOP lists */

void expire_admins()
{

   NickInfo *ni;
   int i;

   for (i = 0; i < MAX_HELPOPS; i++) {
      if (services_helpers[i]) {
         ni = findnick(services_helpers[i]);
         if (!ni) {
            log("!! Deleted expired nick %s from HOP list",
                     services_helpers[i]);
            do_break_log("OS", "!! Deleted expired nick %s from HOP list",
                     services_helpers[i]);
            wallops(SERVER_NAME,
                  "%s has been expired from the HOP list",
                  services_helpers[i]);
            free(services_helpers[i]);
            services_helpers[i] = NULL;
         }
      }
   }

   for (i = 0; i < MAX_SERVOPERS; i++) {
      if (services_opers[i]) {
            ni = findnick(services_opers[i]);
            if (!ni) {
            log("!! Deleted expired nick %s from SOP list",
                     services_opers[i]);
            do_break_log("OS", "!! Deleted expired nick %s from SOP list",
                     services_opers[i]);
            wallops(SERVER_NAME,
                  "%s has been expired from the SOP list",
                  services_opers[i]);
            free(services_opers[i]);
            services_opers[i] = NULL;
         }
      }
   }

   for (i = 0; i < MAX_SERVADMINS; i++) {
      if (services_admins[i]) {
         ni = findnick(services_admins[i]);
         if (!ni) {
            log("!! Deleted expired nick %s from SA list",
                     services_admins[i]);
            do_break_log("OS", "!! Deleted expired nick %s from SA list",
                     services_admins[i]);
            wallops(SERVER_NAME,
                  "%s has been expired from the SA list",
                  services_admins[i]);
            free(services_admins[i]);
            services_admins[i] = NULL;
         }
      }
   }

   for (i = 0; i < MAX_SERVROOTS; i++) {
      if (services_roots[i]) {
         ni = findnick(services_roots[i]);
         if (!ni) {
            log("!! Deleted expired nick %s from SRA list",
                     services_roots[i]);
            do_break_log("OS", "!! Deleted expired nick %s from SRA list",
                     services_roots[i]);
            wallops(SERVER_NAME,
                  "%s has been expired from the SRA list",
                  services_roots[i]);
            free(services_roots[i]);
            services_roots[i] = NULL;
         }
      }
   }



}

/*************************************************************************/

/* Load OperServ access lists. */

void load_os_dbase(void)
{
    load_os_sa_db();
    load_os_sra_db();
    load_os_sop_db();
    load_os_hop_db();
    load_sendlogs();
}

void load_os_sra_db(void)
{
    FILE *f;
    int i;

    if (!(f = open_db(s_OperServ, OS_SRA_DB, "r")))
        return;
    switch (i = get_file_version(f, OS_SRA_DB)) {
      case 7:
      case 6:
      case 5:
        i = fgetc(f)<<8 | fgetc(f);
        while (--i >= 0)
            services_roots[i] = read_string(f, OS_SRA_DB);
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, OS_SRA_DB);
    } /* switch (version) */
    close_db(f, OS_SRA_DB);
}

void load_os_sa_db(void)
{
    FILE *f;
    int i;

    if (!(f = open_db(s_OperServ, OS_SA_DB, "r")))
	return;
    switch (i = get_file_version(f, OS_SA_DB)) {
      case 5:
      case 6:
      case 7:
	i = fgetc(f)<<8 | fgetc(f);
	while (--i >= 0)
	    services_admins[i] = read_string(f, OS_SA_DB);
	break;
      default:
	fatal("Unsupported version (%d) on %s", i, OS_SA_DB);
    } /* switch (version) */
    close_db(f, OS_SA_DB);
}

void load_os_sop_db(void)
{
    FILE *f;
    int i;

    if (!(f = open_db(s_OperServ, OS_SOP_DB, "r")))
        return;
    switch (i = get_file_version(f, OS_SOP_DB)) {
      case 5:
      case 6:
      case 7:
        i = fgetc(f)<<8 | fgetc(f);
        while (--i >= 0)
            services_opers[i] = read_string(f, OS_SOP_DB);
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, OS_SOP_DB);
    } /* switch (version) */
    close_db(f, OS_SOP_DB);
}

void load_os_hop_db(void)
{
    FILE *f;
    int i;

    if (!(f = open_db(s_OperServ, OS_HOP_DB, "r")))
        return;
    switch (i = get_file_version(f, OS_HOP_DB)) {
      case 5:
      case 6:
      case 7:
        i = fgetc(f)<<8 | fgetc(f);
        while (--i >= 0)
            services_helpers[i] = read_string(f, OS_HOP_DB);
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, OS_HOP_DB);
    } /* switch (version) */
    close_db(f, OS_HOP_DB);
}

/*************************************************************************/

void save_qline_db(void)
{
    FILE *f;
    unsigned int i, count = 0;

    if (!(f = open_db(s_OperServ, QLINE_DB, "w")))
        return;
    for (i = 0; i < 64; i++) {
        if (s_qlines[i])
            count++;
    }
    fputc(count>>8, f);
    fputc(count & 255, f);
    for (i = 0; i < 64; i++) {
        if (s_qlines[i])
            write_string(s_qlines[i], f, QLINE_DB);
    }
    close_db(f, QLINE_DB);
}

void load_qline_db(void)
{
    FILE *f;
    int i;

    if (!(f = open_db(s_OperServ, QLINE_DB, "r")))
        return;
    switch (i = get_file_version(f, QLINE_DB)) {
      case 5:
      case 6:
      case 7:
        i = fgetc(f)<<8 | fgetc(f);
        while (--i >= 0)
            s_qlines[i] = read_string(f, QLINE_DB);
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, QLINE_DB);
    } /* switch (version) */
    close_db(f, QLINE_DB);
}

/*************************************************************************/

#if BAH_V >= 147

void save_gline_db(void)
{
    FILE *f;
    unsigned int i, count = 0;

    if (!(f = open_db(s_OperServ, GLINE_DB, "w")))
        return;
    for (i = 0; i < 64; i++) {
        if (s_glines[i])
            count++;
    }
    fputc(count>>8, f);
    fputc(count & 255, f);
    for (i = 0; i < 64; i++) {
        if (s_glines[i])
            write_string(s_glines[i], f, GLINE_DB);
    }
    close_db(f, GLINE_DB);
}

void load_gline_db(void)
{
    FILE *f;
    int i;

    if (!(f = open_db(s_OperServ, GLINE_DB, "r")))
        return;
    switch (i = get_file_version(f, GLINE_DB)) {
      case 6:
      case 7:
        i = fgetc(f)<<8 | fgetc(f);
        while (--i >= 0)
            s_glines[i] = read_string(f, GLINE_DB);
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, GLINE_DB);
    } /* switch (version) */
    close_db(f, GLINE_DB);
}

#endif /* BAHAMUT147 */

/*************************************************************************/

void load_zline(void)
{
    FILE *f;
    int i;

    if (!(f = open_db("ZLINE", ZLINE_DB, "r")))
        return;

    switch (i = get_file_version(f, ZLINE_DB)) {
      case 5:
      case 6:
      case 7:
        nzline = fgetc(f) * 256 + fgetc(f);
        if (nzline < 8)
            zline_size = 16;
        else
            zline_size = 2*nzline;
        zlines = smalloc(sizeof(*zlines) * zline_size);
        if (!nzline) {
            fclose(f);
            return;
        }
        if (nzline != fread(zlines, sizeof(*zlines), nzline, f))
            fatal_perror("Read error on %s", ZLINE_DB);
        for (i = 0; i < nzline; i++) {
            zlines[i].host = read_string(f, ZLINE_DB);
            zlines[i].reason = read_string(f, ZLINE_DB);
            zlines[i].who = read_string(f, ZLINE_DB);

        }
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, ZLINE_DB);
    } /* switch (version) */

    close_db(f, ZLINE_DB);
}


void save_zline(void)
{
    FILE *f;
    int i;


    f = open_db("ZLINE", ZLINE_DB, "w");

    fputc(nzline/256, f);
    fputc(nzline & 255, f);
    if (fwrite(zlines, sizeof(*zlines), nzline, f) != nzline)
        fatal_perror("Write error on %s", ZLINE_DB);
    for (i = 0; i < nzline; i++) {
        write_string(zlines[i].host, f, ZLINE_DB);
        write_string(zlines[i].reason, f, ZLINE_DB);
        write_string(zlines[i].who, f, ZLINE_DB);
    }
    close_db(f, ZLINE_DB);
}

/*************************************************************************/


void load_trigger(void)
{
    FILE *f;
    int i;

    if (!(f = open_db("TRIGGER", TRIGGER_DB, "r")))
        return;

    switch (i = get_file_version(f, TRIGGER_DB)) {
      case 5:
      case 6:
      case 7:
        ntrigger = fgetc(f) * 256 + fgetc(f);
        if (ntrigger < 8)
            trigger_size = 16;
        else
            trigger_size = 2*ntrigger;
        trigger = smalloc(sizeof(*trigger) * trigger_size);
        if (!ntrigger) {
            fclose(f);
            return;
        }
        if (ntrigger != fread(trigger, sizeof(*trigger), ntrigger, f))
            fatal_perror("Read error on %s", TRIGGER_DB);
        for (i = 0; i < ntrigger; i++) {
            trigger[i].mask = read_string(f, TRIGGER_DB);
        }
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, TRIGGER_DB);
    } /* switch (version) */

    close_db(f, TRIGGER_DB);
}


void save_trigger(void)
{
    FILE *f;
    int i;


    f = open_db("TRIGGER", TRIGGER_DB, "w");

    fputc(ntrigger/256, f);
    fputc(ntrigger & 255, f);
    if (fwrite(trigger, sizeof(*trigger), ntrigger, f) != ntrigger)
        fatal_perror("Write error on %s", TRIGGER_DB);
    for (i = 0; i < ntrigger; i++) {
        write_string(trigger[i].mask, f, TRIGGER_DB);
    }
    close_db(f, TRIGGER_DB);
}

/*************************************************************************/

/* Save OperServ access lists. */

void save_os_dbase(void)
{
    save_os_sa_db();
    save_os_sra_db();
    save_os_sop_db();
    save_os_hop_db();
    save_sendlogs();
}


void save_os_sra_db(void)
{
    FILE *f;
    unsigned int i, count = 0;

    if (!(f = open_db(s_OperServ, OS_SRA_DB, "w")))
        return;
    for (i = 0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i])
            count++;
    }
    fputc(count>>8, f);
    fputc(count & 255, f);
    for (i = 0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i])
            write_string(services_roots[i], f, OS_SRA_DB);
    }
    close_db(f, OS_SRA_DB);
}

void save_os_sa_db(void)
{
    FILE *f;
    unsigned int i, count = 0;

    if (!(f = open_db(s_OperServ, OS_SA_DB, "w")))
	return;
    for (i = 0; i < MAX_SERVADMINS; i++) {
	if (services_admins[i])
	    count++;
    }
    fputc(count>>8, f);
    fputc(count & 255, f);
    for (i = 0; i < MAX_SERVADMINS; i++) {
	if (services_admins[i])
	    write_string(services_admins[i], f, OS_SA_DB);
    }
    close_db(f, OS_SA_DB);
}

void save_os_sop_db(void)
{
    FILE *f;
    unsigned int i, count = 0;

    if (!(f = open_db(s_OperServ, OS_SOP_DB, "w")))
        return;
    for (i = 0; i < MAX_SERVOPERS; i++) {
        if (services_opers[i])
            count++;
    }
    fputc(count>>8, f);
    fputc(count & 255, f);
    for (i = 0; i < MAX_SERVOPERS; i++) {
        if (services_opers[i])
            write_string(services_opers[i], f, OS_SOP_DB);
    }
    close_db(f, OS_SOP_DB);
}

void save_os_hop_db(void)
{
    FILE *f;
    unsigned int i, count = 0;

    if (!(f = open_db(s_OperServ, OS_HOP_DB, "w")))
        return;
    for (i = 0; i < MAX_HELPOPS; i++) {
        if (services_helpers[i])
            count++;
    }
    fputc(count>>8, f);
    fputc(count & 255, f);
    for (i = 0; i < MAX_HELPOPS; i++) {
        if (services_helpers[i])
            write_string(services_helpers[i], f, OS_HOP_DB);
    }
    close_db(f, OS_HOP_DB);
}


/*************************************************************************/

/* Does the given nick have Services root privileges? */

int is_services_root(const char *nick)
{
    NickInfo *ni;
    unsigned int i;

    if (is_services_coder(nick))
       return 1;
#ifndef SKELETON
    for (i = 0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i] && is_on_id_list(nick, services_roots[i]))
            return 1;
    }
#else
    for (i = 0; i < MAX_SERVROOTS; i++) {
       if (s_services_roots[i] && !stricmp(nick,s_services_roots[i]))
            return 1;
    }
#endif
    return 0;
}

int rec_services_root(const char *nick)
{
    NickInfo *ni;
    unsigned int i;

    if (is_services_root(nick))
       return 1;
    for (i = 0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i] && !stricmp(nick,services_roots[i]))
        {
#ifndef SKELETON
	    ni = findnick(nick);

	    if (ni->flags & NI_RECOGNIZED)
                return 1;
	    return 0;
#else
            for (i = 0; i < MAX_SERVROOTS; i++) {
                if (s_services_roots[i] && !stricmp(nick,s_services_roots[i]))
                return 1;
            }
#endif
        }
    }
    return 0;
}


/*************************************************************************/

/* Check to see if the person is one of the lovely coders */

int is_services_coder(const char *nick)
{
    NickInfo *ni;

#ifndef SKELETON
    if (!is_on_id_list(nick, SERVICES_MASTER))
            return 0;
#endif
}

int rec_services_coder(const char *nick)
{
    NickInfo *ni;

    if (is_services_coder(nick))
	return 1;
    if (stricmp(nick, SERVICES_MASTER))
            return 0;
#ifndef SKELETON
    ni = findnick(nick);
    if (ni->flags & NI_RECOGNIZED)
        return 1;
     return 0;
#endif
}

/*************************************************************************/

/* Does the given nick have Services admin privileges? */

int is_services_admin(const char *nick)
{
    NickInfo *ni;
    unsigned int i;

    if (is_services_root(nick))
	return 1;
    for (i = 0; i < MAX_SERVADMINS; i++) {
#ifndef SKELETON
        if (services_admins[i] && is_on_id_list(nick, services_admins[i])) {
            return 1;
	}
#else
            for (i = 0; i < MAX_SERVADMINS; i++) {
                if (s_services_admins[i] && !stricmp(nick,s_services_admins[i]))
                return 1;
            }
#endif

    }
    return 0;
}

int rec_services_admin(const char *nick)
{
    NickInfo *ni;
    unsigned int i;

    if (is_services_admin(nick) || rec_services_root(nick))
       return 1;
    for (i = 0; i < MAX_SERVADMINS; i++) {
        if (services_admins[i] && !stricmp(nick,services_admins[i]))
        {
#ifndef SKELETON
            ni = findnick(nick);

            if (ni->flags & NI_RECOGNIZED)
                return 1;
            return 0;
#else
            for (i = 0; i < MAX_SERVADMINS; i++) {
                if (s_services_admins[i] && !stricmp(nick,s_services_admins[i]))
                return 1;
            }
#endif
        }
    }
    return 0;
}


/*************************************************************************/

/* Does the given nick have Services Operator Access privileges? */

int is_services_oper(const char *nick)
{
    NickInfo *ni;
    unsigned int i;

    if (is_services_admin(nick))
	return 1;
    for (i = 0; i < MAX_SERVOPERS; i++) {
#ifndef SKELETON
        if (services_opers[i] && is_on_id_list(nick, services_opers[i])) {
	    return 1;
	}
#else
            for (i = 0; i < MAX_SERVOPERS; i++) {
                if (s_services_opers[i] && !stricmp(nick,s_services_opers[i]))
                return 1;
            }
#endif
    }
    return 0;
}

int rec_services_oper(const char *nick)
{
    NickInfo *ni;
    unsigned int i;

    if (is_services_oper(nick) || rec_services_admin(nick))
       return 1;
    for (i = 0; i < MAX_SERVOPERS; i++) {
        if (services_opers[i] && !stricmp(nick,services_opers[i]))
        {
#ifndef SKELETON
            ni = findnick(nick);

            if (ni->flags & NI_RECOGNIZED)
                return 1;
            return 0;
#else
            for (i = 0; i < MAX_SERVOPERS; i++) {
                if (s_services_opers[i] && !stricmp(nick,s_services_opers[i]))
                return 1;
            }
#endif
        }
    }
    return 0;
}

/*************************************************************************/

/* Does the given nick have Services Help Operator Access privileges? */

int is_services_helpop(const char *nick)
{
    NickInfo *ni;
    unsigned int i;

    if (is_services_oper(nick))
        return 1;
    for (i = 0; i < MAX_HELPOPS; i++) {
#ifndef SKELETON
        if (services_helpers[i] && is_on_id_list(nick, services_helpers[i])) {
	    return 1;
	}
#else
            for (i = 0; i < MAX_HELPOPS; i++) {
                if (s_services_helpers[i] && !stricmp(nick,s_services_helpers[i]))
                return 1;
            }
#endif
    }
    return 0;
}

/*************************************************************************/

void check_temp_os(User *u)
{
#ifdef SKELETON
  unsigned int i;

    for (i = 0; i < MAX_SERVOPERS; i++) {
        if (s_services_opers[i] && !stricmp(u->nick,s_services_opers[i])) {
             log("OS[2] ! -SOP %s (offline)", s_services_opers[i]);
             do_break_log("OS", "OS[2] ! -SOP %s (offline)", s_services_opers[i]);
             s_services_opers[i] = NULL;
             return;
        }
    }
    for (i = 0; i < MAX_SERVADMINS; i++) {
        if (s_services_admins[i] && !stricmp(u->nick,s_services_admins[i])) {
             log("OS[2] ! -SA %s (offline)", s_services_admins[i]);
             do_break_log("OS", "OS[2] ! -SA %s (offline)", s_services_opers[i]);
             s_services_admins[i] = NULL;
             return;
        }
    }
    for (i = 0; i < MAX_SERVROOTS; i++) {
        if (s_services_roots[i] && !stricmp(u->nick,s_services_roots[i])) {
             log("OS[2] ! -SRA %s (offline)", s_services_roots[i]);
             do_break_log("OS", "OS[2] ! -SRA %s (offline)", s_services_opers[i]);
             s_services_roots[i] = NULL;
             return;
        }
    }
    for (i = 0; i < MAX_HELPOPS; i++) {
        if (s_services_helpers[i] && !stricmp(u->nick,s_services_helpers[i])) {
             log("OS[2] ! -HOP %s (offline)", s_services_helpers[i]);
             do_break_log("OS", "OS[2] ! -HOP %s (offline)", s_services_opers[i]);
             s_services_helpers[i] = NULL;
             return;
        }
    }
#endif SKELETON
}
/*************************************************************************/


static void timeout_leave(Timeout *to)
{
    char *chan = to->data;
    send_cmd(s_ChanServ, "PART %s", chan);
    free(to->data);
}


static void timeout_unban(Timeout *to)
{
    char *chan = to->data;
    send_cmd(s_ChanServ, "MODE %s -b *!*@*", chan);
    free(to->data);
}


/*************************************************************************/

/* Expunge a deleted nick from the Services admin list. */

void os_remove_nick(const char *nick)
{
    unsigned int i;

    for (i = 0; i < MAX_SERVADMINS; i++) {
	if (services_admins[i] && stricmp(services_admins[i], nick) == 0)
	    services_admins[i] = NULL;
    }
}

/**************************************************************************/

static void do_sqline(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    char *qnick, *qreason;
    unsigned int i;

   if (!cmd) {
       notice(s_OperServ, source,
          "Syntax: \2SQLINE\2 [ADD|DEL|LIST] [<nick>] [<reason>]");


   } else if (stricmp(cmd, "LIST") == 0) {
           
        notice(s_OperServ, source, "\2Current Q-LINED Nicks\2");

        for (i = 0; i < 64; i++) {
             if (s_qlines[i])
                  notice(s_OperServ, source, "%s", s_qlines[i]);
        }

        notice(s_OperServ, source, "\2END OF LIST\2");


    } else if (!check_o_access(source, "SQLINE")) {
          notice(s_OperServ, source, "Permission denied.");
          return;


   } else if (stricmp(cmd, "ADD") == 0) {

       qnick = strtok(NULL, " ");
       qreason = strtok(NULL, "");

       if (!qreason) {
          notice(s_OperServ, source,
              "Syntax: \2SQLINE ADD\2 <nick> <reason>");
          return;
       }

       for (i = 0; i < 64; i++) {
            if (s_qlines[i] && !stricmp(qnick,s_qlines[i])) {
                notice(s_OperServ, source, "%s is already Q-LINED",
                    qnick);
                return;
            }
            if (!s_qlines[i])
                 break;
       }
       if ((i < MAX_QLINES) || (MAX_QLINES == 0)) {
          s_qlines[i] = sstrdup(qnick);
          send_cmd(NULL, "SQLINE %s :%s", qnick, qreason);
          log("!! Added \2%s\2 to Q-LINED List",qnick);
          do_break_log("OS", "!! Added \2%s\2 to Q-LINED List",qnick);
          wallops(SERVER_NAME, "%s Q-LINED nick \2%s\2 because: %s",
              source, qnick, qreason);
          notice(s_OperServ, source,
               "%s is now Q-LINED because: %s", qnick, qreason);
       } else {
           notice(s_OperServ, source,
                "Too many entries (%d) on Q-line list; cannot add more",
                     MAX_QLINES);
           wallops(s_OperServ,
                "\2Warning\2: Q-Line list is full! (%d)",
                     MAX_QLINES);
           
       }
       if (readonly) {
           notice(s_OperServ, source,
                "\2Notice:\2 Changes will not be saved!  Services is in readonly mode.");
       }
   } else if (stricmp(cmd, "DEL") == 0) {

        qnick = strtok(NULL, " ");

        if (qnick) {
            for (i = 0; i < 64; i++) {
                if (s_qlines[i] && stricmp(qnick,s_qlines[i]) == 0)
                    break;
            }
            if (i < 64) {
                send_cmd(NULL, "UNSQLINE %s", s_qlines[i]);
                free(s_qlines[i]);
                s_qlines[i] = NULL;
                log("!! Deleted \2%s\2 from the Q-LINED list",qnick);
                do_break_log("OS", "!! Deleted \2%s\2 from the Q-LINED list",qnick);
                wallops(SERVER_NAME, "%s had me UNSQLINE \2%s\2",source,qnick);
                notice(s_OperServ, source,
                        "%s removed as Q-LINED.", qnick);
                if (readonly) {
                    notice(s_OperServ, source,
                            "\2Notice:\2 Changes will not be saved! Services is in readonly mode.");
                }
            } else {
               send_cmd(NULL, "UNSQLINE %s", qnick);
            }
        } else {
            notice(s_OperServ, source, "Syntax: \2SQLINE DEL\2 <nick>");
        }
   }
}

/**************************************************************************/

#if BAH_V >= 147

static void do_sgline(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    unsigned int i;

   if (!cmd) {
       notice(s_OperServ, source,
          "Syntax: \2SGLINE\2 [ADD|DEL|LIST] [<realname>] [<reason>]");


   } else if (stricmp(cmd, "LIST") == 0) {
           
        notice(s_OperServ, source, "\2Current G-LINED Fields\2");

        for (i = 0; i < 64; i++) {
             if (s_glines[i])
                  notice(s_OperServ, source, "%s", s_glines[i]);
        }

        notice(s_OperServ, source, "\2END OF LIST\2");


    } else if (!check_o_access(source, "SGLINE")) {
          notice(s_OperServ, source, "Permission denied.");
          return;


   } else if (stricmp(cmd, "ADD") == 0) {

       char *gcos = strtok(NULL, ":");
       char *reason = strtok(NULL, "");

       if (stricmp(gcos+strlen(gcos)-1, " ")) {  // lets make sure it ends with space + :
           notice(s_OperServ, source,
              "Syntax: \2SGLINE ADD\2 [*]<realname>[*] : <reason>");
          return;
       }
       strscpy(gcos, gcos, strlen(gcos)); // strip whitespace from end

       if (!reason) {
          notice(s_OperServ, source,
              "Syntax: \2SGLINE ADD\2 [*]<realname>[*] : <reason>");
          return;
       }

       for (i = 0; i < 64; i++) {
            if (s_glines[i] && !stricmp(gcos,s_glines[i])) {
                notice(s_OperServ, source, "%s is already G-LINED",
                    gcos);
                return;
            }
            if (!s_glines[i])
                 break;
       }
       if ((i < MAX_GLINES) || (MAX_GLINES == 0)) {
          s_glines[i] = sstrdup(gcos);
          send_cmd(NULL, "SGLINE %d :%s:%s", strlen(gcos), gcos, reason);
          log("!! Added \2%s\2 to G-LINED List",reason);
          do_break_log("OS", "!! Added \2%s\2 to G-LINED List",gcos);
          wallops(SERVER_NAME, "%s G-LINED nick \2%s\2 because: %s",
              source, gcos, reason);
          notice(s_OperServ, source,
               "%s is now G-LINED because: %s", gcos, reason);
       } else {
           notice(s_OperServ, source,
                "Too many entries (%d) on G-LINE list; cannot add more",
                     MAX_GLINES);
           wallops(s_OperServ,
                "\2Warning\2: G-LINE list is full! (%d)",
                     MAX_GLINES);
           
       }
       if (readonly) {
           notice(s_OperServ, source,
                "\2Notice:\2 Changes will not be saved!  Services is in readonly mode.");
       }
   } else if (stricmp(cmd, "DEL") == 0) {

        char *gcos = strtok(NULL, " ");

        if (gcos) {
            for (i = 0; i < 64; i++) {
                if (s_glines[i] && stricmp(gcos,s_glines[i]) == 0)
                    break;
            }
            if (i < 64) {
                send_cmd(NULL, "UNSGLINE %s", s_glines[i]);
                free(s_glines[i]);
                s_glines[i] = NULL;
                log("!! Deleted \2%s\2 from the G-LINED list",gcos);
                do_break_log("OS", "!! Deleted \2%s\2 from the G-LINED list",gcos);
                wallops(SERVER_NAME, "%s had me UNSGLINE \2%s\2",source,gcos);
                notice(s_OperServ, source,
                        "%s removed as G-LINED.", gcos);
                if (readonly) {
                    notice(s_OperServ, source,
                            "\2Notice:\2 Changes will not be saved! Services is in readonly mode.");
                }
            } else {
               send_cmd(NULL, "UNSGLINE %s", gcos);
            }
        } else {
            notice(s_OperServ, source, "Syntax: \2SGLINE DEL\2 <realname>");
        }
   }
}

#endif /* BAHAMUT147 */

/**************************************************************************/
/******************************JUPE STUFF*********************************/
/**************************************************************************/

static void do_jupe(const char *source)
{
    const char *cmd = strtok(NULL," ");
    char *jserver, *reason;
    int i;
    char *s;

    if (!cmd) {
	goto err;
    }

    if (stricmp(cmd,"ADD") == 0) {
        jserver = strtok(NULL, " ");
        reason = strtok(NULL, "");
        if (!jserver || !reason) {
            notice(s_OperServ, source, "Syntax: \2JUPE ADD\2 <servername> <reason>");
	    return;
        }

        if (!stricmp(jserver, "*") || !stricmp(jserver, "*.*") ||
           !stricmp(jserver, "*.net") || !stricmp(jserver, "*.sirv.net")) {
                notice(s_OperServ, source,
                    "Bogus JUPE server. Smarten up!");
                 return;
        }

        for (i = 0; i < njupe; i++) {
            if (stricmp(jupes[i].jserver,jserver) == 0) {
                notice(s_OperServ,source,"%s is already present "
                    "on the JUPE list.",jserver);
                return;
            }
        }

        if (njupe >= jupe_size) {
            if (jupe_size < 8)
                jupe_size = 8;
            else
                jupe_size *= 2;
            jupes = srealloc(jupes, sizeof(*jupes) * jupe_size);
        }

        if (!njupe || (njupe < MAX_JUPES) || (MAX_JUPES == 0)) {
            jupes[njupe].jserver = sstrdup(jserver);
            jupes[njupe].reason = sstrdup(reason);
	    strscpy(jupes[njupe].who, source, NICKMAX);

            notice(s_OperServ, source, "Juping %s because: %s",
                    jserver, reason);
            wallops(SERVER_NAME, "Juping %s by request of %s for: %s",
                    jserver, source, reason);
            send_cmd(NULL, "SQUIT %s :Jupitered Server [%s]",
		jserver, reason);
            send_cmd(NULL, 
		"SERVER %s 2 :Jupitered server (Request of %s) [%s]",
                jserver, source, reason);
            ++njupe;
            if (readonly) {
                notice(s_OperServ, source,
                    "\2Notice:\2 Changes will not be saved! "
                    " Services is in read-only mode.");
            }
        }
	else {
	    notice(s_OperServ,source,"Sorry, JUPE list is full.");
            wallops(SERVER_NAME,"\2Warning\2: JUPE limit exceeded. (%d)",
                 MAX_JUPES);
	}

    } else if (stricmp(cmd, "INFO") == 0) {
        int i;
        const char *jserver = strtok(NULL, " ");
        const char *info = strtok(NULL, "");

        if (!is_services_root(source)) {
            notice(s_OperServ, source, "Permission denied.");
            return;
        }

        if ((jserver == NULL) || (info == NULL))
          return;

            if (strspn(jserver, "1234567890") == strlen(jserver) &&
                                (i = atoi(jserver)) > 0 && i <= njupe) {

                free(jupes[i-1].reason);
                jupes[i-1].reason = sstrdup(info);
                notice(s_OperServ, source, "JUPE reason for %s changed to: %s",
                             jupes[i-1].jserver, info);
                return;
            }

         for (i = 0; i < njupe; i++) {
               if (stricmp(jupes[i].jserver, jserver) == 0) {
                   free(jupes[i].reason);
                   jupes[i].reason = sstrdup(info);
                   notice(s_OperServ, source, "JUPE reason for %s changed to: %s",
                                jupes[i].jserver, jupes[i].reason);
               }
         }

    } else if (stricmp(cmd, "DEL") == 0) {
        int i;
        if (jserver = strtok(NULL, " ")) {
            if (strspn(jserver, "1234567890") == strlen(jserver) &&
                                (i = atoi(jserver)) > 0 && i <= njupe) {
                strcpy(jserver,jupes[i-1].jserver);
            }
            if (del_jupe(jserver)) {
                notice(s_OperServ, source, "%s removed from JUPE list.",jserver);
                wallops(SERVER_NAME,
                        "%s removed %s from the JUPE list.",
                        source, jserver);
		send_cmd(NULL,"SQUIT %s :Deleting JUPE",jserver);
                if (readonly) {
                    notice(s_OperServ, source,
                        "\2Notice:\2 Changes will not be saved!  Services is in readonly mode.");
                }
            } else {
                notice(s_OperServ, source, "%s not found on JUPE list.",
		    jserver);
            }
        } else {
            notice(s_OperServ, source, "Syntax: \2JUPE DEL\2 <server>");
        }
    } else if (stricmp(cmd, "LIST") == 0) {
        s = strtok(NULL, " ");
        if (!s)
            s = "*";
        notice(s_OperServ, source, "\2Current JUPE list:\2");
        for (i = 0; i < njupe; i++) {
            if (!s || match_wild(s, jupes[i].jserver)) {
                notice(s_OperServ, source, " %d) %-16s",
                                i+1,jupes[i].jserver);
		notice(s_OperServ, source, "Set by %s for: %s",
				jupes[i].who, jupes[i].reason);
            }
        }
        notice(s_OperServ,source,"\2End of List\2");
    } else {
err:
        notice(s_OperServ,source,"Syntax: \2JUPE\2 [ADD|DEL|LIST|INFO] [<server>]");
	notice(s_OperServ,source,"Type \2/msg %s OHELP JUPE\2 for more "
	    "information",s_OperServ);
	return;
    }
}

/*************************************************************************/

void check_zlines()
{
    unsigned int i;
        for (i = 0; i < nzline; i++)
            send_cmd(NULL, "ZLINE %s :%s (%s)",
                         zlines[i].host, zlines[i].reason, zlines[i].who);

}

/*************************************************************************/

void check_qlines()
{
    unsigned int i;
        for (i = 0; i < 64; i++) {
             if (s_qlines[i])
                  send_cmd(NULL, "SQLINE %s :Services Q-LINED", s_qlines[i]);
        }

}

/*************************************************************************/

#if BAH_V >= 147

void check_glines()
{
    unsigned int i;
        for (i = 0; i < 64; i++) {
             if (s_glines[i])
                  send_cmd(NULL, "SGLINE %d :%s:Services Q-LINED",
                     strlen(s_glines[i]), s_glines[i]);
        }

}

#endif /* BAHAMUT147 */

/*************************************************************************/

void check_jupes()
{
    unsigned int i;

    for (i = 0; i < njupe; i++) {
	send_cmd(NULL,"SQUIT %s :Jupitered Server",jupes[i].jserver);
	send_cmd(NULL,"SERVER %s 2 :Jupitered Server (Request of %s) [%s]",
	    jupes[i].jserver,jupes[i].who,jupes[i].reason);
    }
}

void check_jupe_squit(char *source, char *server)
{
    unsigned int i;
    User *u;

    if (!server || !source)
	return;

    u = finduser(source);

    for (i = 0; i < njupe && stricmp(jupes[i].jserver, server) != 0; i++)
        ;
    if (i < njupe && !stricmp(jupes[i].jserver, server)) {
        notice(s_OperServ,source,"Do not squit Jupitered Servers, "
            "use \2/msg %s JUPE DEL %s\2 to delete it.",
            s_OperServ,server);
        send_cmd(NULL,"SQUIT %s :Jupitered Server",server);
        send_cmd(NULL,"SERVER %s 2 :Jupitered Server (Requested by: %s) [%s]",
            server,jupes[i].who,jupes[i].reason);
    }
}

/*************************************************************************/

static int del_jupe(const char *jserver)
{
    unsigned int i;

    for (i = 0; i < njupe && stricmp(jupes[i].jserver, jserver) != 0; i++)
        ;
    if (i < njupe) {
        free(jupes[i].jserver);
	free(jupes[i].reason);
        njupe--;
        if (i < njupe)
            bcopy(jupes+i+1, jupes+i, sizeof(*jupes) * (njupe-i));
        return 1;
    } else {
        return 0;
    }
}

/*********************TRIGGER*********************/


static void add_trigger(const char *cmd, const char *tvalue)
{
/*    int tval = atoi(tvalue); */

    int amount = strtol(tvalue, (char **)&tvalue, 10);

    if (ntrigger >= trigger_size) {
        if (trigger_size < 8)
            trigger_size = 8;
	else
            trigger_size *= 2;
        trigger = srealloc(trigger, sizeof(*trigger) * trigger_size);
    }

    trigger[ntrigger].mask = sstrdup(cmd);
    ntrigger++;
}


static int del_trigger(const char *cmd)
{
    unsigned int i;

    for (i = 0; i < ntrigger && stricmp(trigger[i].mask, cmd) != 0; i++)
        ;
    if (i < ntrigger) {
        free(trigger[i].mask);
//        trigger[i].tvalue = NULL;
        ntrigger--;
        if (i < ntrigger)
            bcopy(trigger+i+1, trigger+i, sizeof(*trigger) * (ntrigger-i));
        return 1;
    } else {
        return 0;
    }
}

/*************************************************************************/
/****************************** AKILL stuff ******************************/
/*************************************************************************/

void load_akill(void)
{
    FILE *f;
    int i;

    if (!(f = open_db("AKILL", AKILL_DB, "r")))
	return;

    switch (i = get_file_version(f, AKILL_DB)) {
      case 5:
      case 6:
      case 7:
	nakill = fgetc(f) * 256 + fgetc(f);
	if (nakill < 8)
	    akill_size = 16;
	else
	    akill_size = 2*nakill;
	akills = smalloc(sizeof(*akills) * akill_size);
	if (!nakill) {
	    fclose(f);
	    return;
	}
	if (nakill != fread(akills, sizeof(*akills), nakill, f))
	    fatal_perror("Read error on %s", AKILL_DB);
	for (i = 0; i < nakill; i++) {
	    akills[i].mask = read_string(f, AKILL_DB);
	    akills[i].reason = read_string(f, AKILL_DB);
	}
	break;

      case 2: {
	struct {
	    char *mask;
	    char *reason;
	    char who[NICKMAX];
	    time_t time;
	} old_akill;
	nakill = fgetc(f) * 256 + fgetc(f);
	if (nakill < 8)
	    akill_size = 16;
	else
	    akill_size = 2*nakill;
	akills = smalloc(sizeof(*akills) * akill_size);
	if (!nakill) {
	    fclose(f);
	    return;
	}
	for (i = 0; i < nakill; i++) {
	    if (1 != fread(&old_akill, sizeof(old_akill), 1, f))
		fatal_perror("Read error on %s", AKILL_DB);
	    akills[i].time = old_akill.time;
	    strscpy(akills[i].who, old_akill.who, sizeof(akills[i].who));
	    akills[i].expires = 0;
	}
	for (i = 0; i < nakill; i++) {
	    akills[i].mask = read_string(f, AKILL_DB);
	    akills[i].reason = read_string(f, AKILL_DB);
	}
	break;
      } /* case 2 */

      case 1: {
	struct {
	    char *mask;
	    char *reason;
	    time_t time;
	} old_akill;
	nakill = fgetc(f) * 256 + fgetc(f);
	if (nakill < 8)
	    akill_size = 16;
	else
	    akill_size = 2*nakill;
	akills = smalloc(sizeof(*akills) * akill_size);
	if (!nakill) {
	    fclose(f);
	    return;
	}
	for (i = 0; i < nakill; i++) {
	    if (1 != fread(&old_akill, sizeof(old_akill), 1, f))
		fatal_perror("Read error on %s", AKILL_DB);
	    akills[i].time = old_akill.time;
	    akills[i].who[0] = 0;
	    akills[i].expires = 0;
	}
	for (i = 0; i < nakill; i++) {
	    akills[i].mask = read_string(f, AKILL_DB);
	    akills[i].reason = read_string(f, AKILL_DB);
	}
	break;
      } /* case 1 */

      default:
	fatal("Unsupported version (%d) on %s", i, AKILL_DB);
    } /* switch (version) */

    close_db(f, AKILL_DB);
}

/*************************************************************************/

void save_akill(void)
{
    FILE *f;
    int i;

    f = open_db("AKILL", AKILL_DB, "w");

    fputc(nakill/256, f);
    fputc(nakill & 255, f);
    if (fwrite(akills, sizeof(*akills), nakill, f) != nakill)
	fatal_perror("Write error on %s", AKILL_DB);
    for (i = 0; i < nakill; i++) {
	write_string(akills[i].mask, f, AKILL_DB);
	write_string(akills[i].reason, f, AKILL_DB);
    }
    close_db(f, AKILL_DB);
}

/*************************************************************************/

/* Does the user match any AKILLs? */

int check_akill(const char *nick, const char *username, const char *host)
{
    char buf[512];
    char *av[3];
    int i;
    char *host2, *username2;

    strscpy(buf, username, sizeof(buf)-2);
    i = strlen(buf);
    buf[i++] = '@';
    strlower(strscpy(buf+i, host, sizeof(buf)-i));
    for (i = 0; i < nakill; i++) {
	if (match_wild_nocase(akills[i].mask, buf)) {
	    time_t now = time(NULL);
	    /* Don't use kill_user(); that's for people who have already
	     * signed on.  This is called before the User structure is
	     * created.
	     */

             av[0] = sstrdup(nick);
             do_kill(SERVER_NAME, 3, av); free(av[0]);


             send_cmd(SERVER_NAME, "KILL %s :%s (%s)", nick, SERVER_NAME,
                   INCLUDE_AKILL_REASON==0 ? OS_AKILL : akills[i].reason); 

	    username2 = sstrdup(akills[i].mask);
	    host2 = strchr(username2, '@');

	    if (!host2) {
		/* Glurp... this oughtn't happen, but if it does, let's not
		 * play with null pointers.  Yell and bail out.
		 */
		wallops(NULL, "Missing @ in AKILL: %s", akills[i].mask);
		log("Missing @ in AKILL: %s", akills[i].mask);
		continue;
	    }
	    *host2++ = 0;
#ifndef BAHAMUT
       	        send_cmd(server_name,
		    "AKILL %s %s :%s",
  		    host2, username2,
                    INCLUDE_AKILL_REASON==0 ? OS_AKILL :
                                                       akills[i].reason);
#else

                send_cmd(server_name, "AKILL %s %s 0 %s %ld :%s",
                     host2, username2, akills[i].who, time(NULL), 
                     INCLUDE_AKILL_REASON==0 ? OS_AKILL :
                                                       akills[i].reason);
#endif
	    free(username2);
	    return 1;
	}
    }
    return 0;
}

/*************************************************************************/

/* Delete any expired autokills. */

void expire_akills(void)
{
    int i;
    time_t now = time(NULL);
    char *s;
    for (i = 0; i < nakill; i++) {
	if (akills[i].expires == 0 || akills[i].expires >= now)
	    continue;
        wallops(s_OperServ,  OS_RAKILL,
		akills[i].mask);

        if (s = strchr(akills[i].mask, '@')) {
            *s++ = 0;
            strlower(s);
            send_cmd(server_name, "RAKILL %s %s", s, akills[i].mask);
        }

        del_akill(akills[i].mask);
	i--;
    }
}

/*************************************************************************/

/* Note that all parameters except expiry are assumed to be non-NULL.  A
 * value of NULL for expiry indicates that the AKILL should not expire.
 */

static void add_akill(const char *mask, const char *reason, const char *who,
		      const char *expiry, int akill_num)
{
    float percent = akill_num + .0;

    if (nakill >= akill_size) {
	if (akill_size < 8)
	    akill_size = 8;
	else
	    akill_size *= 2;
	akills = srealloc(akills, sizeof(*akills) * akill_size);
    }
    akills[nakill].mask = sstrdup(mask);
    akills[nakill].reason = sstrdup(reason);
    akills[nakill].time = time(NULL);
    strscpy(akills[nakill].who, who, NICKMAX);
    percent *= 100.0 / usercnt;

    if (expiry) {
	int amount = strtol(expiry, (char **)&expiry, 10);
	if (amount == 0) {

            akills[nakill].expires = 0;
            wallops(s_OperServ,"%s added a permanent AKILL for %s because "
                 "%s [effects %d user%s (%.3f%s)]",
                 who, mask, reason, akill_num,
                 (akill_num == 1) ? "" : "s",
                  percent, "%");
                 
	} else {
            char expirebuf[64];
            time_t t = time(NULL);
	    switch (*expiry) {
                case 'd': amount *= 24;
                case 'h': amount *= 60;
		case 'm': amount *= 60; break;
		default : amount = AKILL_DEF_EXPIRY;
	    }
	    akills[nakill].expires = amount + akills[nakill].time;
            if (akills[nakill].expires == 0) {
                strcpy(expirebuf, "Does not expire");
            } else if (akills[nakill].expires <= t) {
                strcpy(expirebuf, "Expires at next database update");
            } else {
                time_t t2 = akills[nakill].expires - t;
                t2 += 59;
                if (t2 < 86400) {
                    t2 /= 60;
                    snprintf(expirebuf, sizeof(expirebuf),
                            "Expires in %d hour%s, %d minute%s",
                            t2/60, t2/60==1 ? "" : "s",
                            t2%60, t2%60==1 ? "" : "s");
                } else {
                    t2 /= 86400;
                    snprintf(expirebuf, sizeof(expirebuf),
                            "Expires in %d day%s", t2, t2==1 ? "" : "s");
                }
	    }
            wallops(s_OperServ,"%s added an AKILL for %s because "
                "\"%s\" [Timed: %s] [Affects %d user%s (%.3f%s)]",
                 who, mask, reason, expirebuf, akill_num,
                 (akill_num == 1) ? "" : "s",
                 percent, "%");
	}
    } else {

        akills[nakill].expires = AKILL_DEF_EXPIRY + akills[nakill].time;
        if (AKILL_DEF_EXPIRY < 3600)
            wallops(s_OperServ,"%s added an AKILL for %s because "
                "\"%s\" [Expires in %d minute(s)] [Affects %d user%s (%.3f%s)]",
                 who, mask, reason,
                (AKILL_DEF_EXPIRY / 60), akill_num,
                (akill_num == 1) ? "" : "s",
                percent, "%");
        else if (AKILL_DEF_EXPIRY < 86400)
            wallops(s_OperServ,"%s added an AKILL for %s because "
                "\"%s\" [Expires in %d hour(s)] [Affects %d user%s (%.3f%s)]",
                 who, mask, reason,
                (AKILL_DEF_EXPIRY / 60 / 60), akill_num,
                (akill_num == 1) ? "" : "s",
                percent, "%");
        else
            wallops(s_OperServ,"%s added an AKILL for %s because "
                "\"%s\" [Expires in %d day(s)] [Affects %d user%s (%.3f%s)]",
                 who, mask, reason,
                (AKILL_DEF_EXPIRY / 60 / 60 / 24), akill_num,
                (akill_num == 1) ? "" : "s",
                percent, "%");
    }
    nakill++;
}

/*************************************************************************/

/* Return whether the mask was found in the AKILL list. */

static int del_akill(const char *mask)
{
    int i;

    for (i = 0; i < nakill && stricmp(akills[i].mask, mask) != 0; i++)
	;
    if (i < nakill) {
	free(akills[i].mask);
	free(akills[i].reason);
	nakill--;
	if (i < nakill)
	    bcopy(akills+i+1, akills+i, sizeof(*akills) * (nakill-i));
	return 1;
    } else {
	return 0;
    }
}

/*************************************************************************/
/**************************** IGNORE STUFF *******************************/
/*************************************************************************/

void add_ignore(const char *mask, const char *who, const char *expiry)
{
    int ignore_num=0;

    if (nignore >= ignore_size) {
        if (ignore_size < 8)
            ignore_size = 8;
	else
            ignore_size *= 2;
        ignores = srealloc(ignores, sizeof(*ignores) * ignore_size);
    }
    ignores[nignore].mask = sstrdup(mask);
    ignores[nignore].time = time(NULL);
    strscpy(ignores[nignore].who, who, NICKMAX);
    if (expiry) {
	int amount = strtol(expiry, (char **)&expiry, 10);
	if (amount == 0) {

            ignores[nignore].expires = 0;
            ignore_num = check_ignore_num(mask);
            wallops(SERVER_NAME,"%s added a permanent IGNORE for %s. [Affects %d user%s]",
                 who, mask, ignore_num,
                 ignore_num==1 ? "" : "s");
                 
	} else {
            char expirebuf[64];
            time_t t = time(NULL);
	    switch (*expiry) {
                case 'd': amount *= 24;
                case 'h': amount *= 60;
		case 'm': amount *= 60; break;
                default : amount = IGNORE_DEF_EXPIRY;
	    }
            ignores[nignore].expires = amount + ignores[nignore].time;
            if (ignores[nignore].expires == 0) {
                strcpy(expirebuf, "Does not expire");
            } else if (ignores[nignore].expires <= t) {
                strcpy(expirebuf, "Expires at next database update");
            } else {
                time_t t2 = ignores[nignore].expires - t;
                t2 += 59;
                if (t2 < 86400) {
                    t2 /= 60;
                    snprintf(expirebuf, sizeof(expirebuf),
                            "Expires in %d hour%s, %d minute%s",
                            t2/60, t2/60==1 ? "" : "s",
                            t2%60, t2%60==1 ? "" : "s");
                } else {
                    t2 /= 86400;
                    snprintf(expirebuf, sizeof(expirebuf),
                            "Expires in %d day%s", t2, t2==1 ? "" : "s");
                }
	    }
            ignore_num = check_ignore_num(mask);
            wallops(SERVER_NAME,"%s added an IGNORE for %s "
                "[Timed: %s] [Affects %d user%s]",
                 who, mask, expirebuf, ignore_num,
                 ignore_num==1 ? "" : "s");
	}
    } else {
        ignores[nignore].expires = IGNORE_DEF_EXPIRY + ignores[nignore].time;
        ignore_num = check_ignore_num(mask);
        if (IGNORE_DEF_EXPIRY > 86400)
            wallops(SERVER_NAME,"%s added an IGNORE for %s "
                "[Expires in %d day%s] [Affects %d user%s]",
                 who, mask,
                (IGNORE_DEF_EXPIRY/86400),
                IGNORE_DEF_EXPIRY/86400==1 ? "" : "s",
                ignore_num, ignore_num==1 ? "" : "s");

        else if (IGNORE_DEF_EXPIRY > 3600)
            wallops(SERVER_NAME,"%s added an IGNORE for %s "
                "[Expires in %d hour%s] [Affects %d user%s]",
                 who, mask,
                (IGNORE_DEF_EXPIRY/3600),
                IGNORE_DEF_EXPIRY/3600==1 ? "" : "s",
                ignore_num, ignore_num==1 ? "" : "s");

        else if (IGNORE_DEF_EXPIRY >= 60)
            wallops(SERVER_NAME,"%s added an IGNORE for %s "
                "[Expires in %d min%s] [Affects %d user%s]",
                 who, mask,
                (IGNORE_DEF_EXPIRY/60),
                IGNORE_DEF_EXPIRY/60==1 ? "" : "s",
                ignore_num, ignore_num==1 ? "" : "s");
        else
           wallops(SERVER_NAME, "%s added an IGNORE for %s "
                "[Expires momentarily] [Affects %d user%s]",
                who, mask, ignore_num,
                ignore_num==1 ? "" : "s");

    }
    nignore++;
}

/*************************************************************************/

void load_ignore(void)
{
    FILE *f;
    int i;

    if (!(f = open_db("IGNORE", IGNORE_DB, "r")))
	return;

    switch (i = get_file_version(f, IGNORE_DB)) {
      case 5:
      case 6:
      case 7:
        nignore = fgetc(f) * 256 + fgetc(f);
        if (nignore < 8)
            ignore_size = 16;
	else
            ignore_size = 2*nignore;
        ignores = smalloc(sizeof(*ignores) * ignore_size);
        if (!nignore) {
	    fclose(f);
	    return;
	}
        if (nignore != fread(ignores, sizeof(*ignores), nignore, f))
            fatal_perror("Read error on %s", IGNORE_DB);
        for (i = 0; i < nignore; i++)
            ignores[i].mask = read_string(f, IGNORE_DB);
	break;

      case 2: {
	struct {
	    char *mask;
	    char who[NICKMAX];
	    time_t time;
        } old_ignore;
        nignore = fgetc(f) * 256 + fgetc(f);
        if (nignore < 8)
            ignore_size = 16;
	else
            ignore_size = 2*nignore;
        ignores = smalloc(sizeof(*ignores) * ignore_size);
        if (!nignore) {
	    fclose(f);
	    return;
	}
        for (i = 0; i < nignore; i++) {
            if (1 != fread(&old_ignore, sizeof(old_ignore), 1, f))
                fatal_perror("Read error on %s", IGNORE_DB);
            ignores[i].time = old_ignore.time;
            strscpy(ignores[i].who, old_ignore.who, sizeof(ignores[i].who));
            ignores[i].expires = 0;
	}
        for (i = 0; i < nignore; i++)
            ignores[i].mask = read_string(f, IGNORE_DB);
	break;
      } /* case 2 */

      case 1: {
	struct {
	    char *mask;
	    time_t time;
        } old_ignore;
        nignore = fgetc(f) * 256 + fgetc(f);
        if (nignore < 8)
            ignore_size = 16;
	else
            ignore_size = 2*nignore;
        ignores = smalloc(sizeof(*ignores) * ignore_size);
        if (!nignore) {
	    fclose(f);
	    return;
	}
        for (i = 0; i < nignore; i++) {
            if (1 != fread(&old_ignore, sizeof(old_ignore), 1, f))
                fatal_perror("Read error on %s", IGNORE_DB);
            ignores[i].time = old_ignore.time;
            ignores[i].who[0] = 0;
            ignores[i].expires = 0;
	}
        for (i = 0; i < nignore; i++)
            ignores[i].mask = read_string(f, IGNORE_DB);
	break;
      } /* case 1 */

      default:
        fatal("Unsupported version (%d) on %s", i, IGNORE_DB);
    } /* switch (version) */

    close_db(f, IGNORE_DB);
}

/*************************************************************************/

void save_ignore(void)
{
    FILE *f;
    int i;

    f = open_db("IGNORe", IGNORE_DB, "w");

    fputc(nignore/256, f);
    fputc(nignore & 255, f);
    if (fwrite(ignores, sizeof(*ignores), nignore, f) != nignore)
        fatal_perror("Write error on %s", IGNORE_DB);
    for (i = 0; i < nignore; i++)
        write_string(ignores[i].mask, f, IGNORE_DB);
    close_db(f, IGNORE_DB);
}

/*************************************************************************/

/* Does the user match any IGNOREs? */

int check_ignore(const char *nick, const char *username, const char *host)
{
    char buf[512];
    int i;

    strscpy(buf, nick, sizeof(buf)-2);
    i = strlen(buf);
    buf[i++] = '!';
    strscpy(buf+i, username, sizeof(buf)-i);
    i = strlen(buf);
    buf[i++] = '@';
    strlower(strscpy(buf+i, host, sizeof(buf)-i));
    for (i = 0; i < nignore; i++) {
        if (match_wild_nocase(ignores[i].mask, buf))
	    return 1;
    }
    return 0;
}

/* Note: 0 here? */

/*************************************************************************/

/* Delete any expired ignores. */

void expire_ignores(void)
{
    int i;
    time_t now = time(NULL);
    char *s;
    for (i = 0; i < nignore; i++) {
        if (ignores[i].expires == 0 || ignores[i].expires >= now)
	    continue;
        wallops(SERVER_NAME,"Removing IGNORE for %s [Time Expired]",
                ignores[i].mask);
        del_ignore(ignores[i].mask);
	i--;
    }
}

/*************************************************************************/

static int del_ignore(const char *mask)
{
    int i;

    for (i = 0; i < nignore && stricmp(ignores[i].mask, mask) != 0; i++)
	;
    if (i < nignore) {
        free(ignores[i].mask);
        nignore--;
        if (i < nignore)
            bcopy(ignores+i+1, ignores+i, sizeof(*ignores) * (nignore-i));
	return 1;
    } else {
	return 0;
    }
}


/*************************************************************************/
/**************************** Clone detection ****************************/
/*************************************************************************/

/* We just got a new user; does it look like a clone?  If so, send out a
 * wallops.
 */

void check_clones(User *user)
{
    char buf[512], ahost[512];
    int i, tfound=0, clone_count=0, uclone_count=0, u_match=0;
    int triggered=0;  /* to prevent automatic akill */
    time_t ctime = time(NULL);
    User *u, *u2;

    if (clone_detect == 0)
      return;

    strscpy(buf, user->username, sizeof(buf)-2);
    i = strlen(buf);
    buf[i++] = '@';
    strlower(strscpy(buf+i, user->host, sizeof(buf)-i));

    for (u = userlist; u; u = u->next) {
        if (match_wild_nocase(u->host, user->host))
              clone_count++;
        if ((match_wild_nocase(u->username, user->username)) &&
            (match_wild_nocase(u->host, user->host)))
              uclone_count++;
    }

    for (i = 0; i < ntrigger; i++) {
       if (match_wild_nocase(trigger[i].mask, buf)) {
          triggered=1;
          if (uclone_count >= trigger[i].tvalue)
              u_match = 1;
          if (clone_count >= trigger[i].tvalue)
              tfound = 1;
          else
              return;
       }
    }

    if ((clone_count >= CLONE_MIN_USERS) || (tfound == 1)) {

 /* I don't like this, but after overwhelming demand... */
        if (AKILL_CLONES) {
                                  /* blah... I REALLY don't like this */
             if ((clone_count >= AKILL_CLONES) && !triggered) {
                  *ahost=0;
                  strcpy(ahost, "*@");
                  strcat(ahost, user->host);
    	          for (i = 0; i < nakill; i++) {
		      if (stricmp(akills[i].mask,ahost) == 0)
       	 	          return;
    	          }

                  add_akill(ahost, CKILL_REASON, "Services",
                                            CKILL_TIME, clone_count);
                  mass_check_akill();
                  return;
             }
        }

	/* Okay, we have clones.  Check first to see if we already know
	 * about them. */

        if ((u_match == 1) || (uclone_count >= CLONE_MIN_USERS)) {

             for (i = CLONE_DETECT_SIZE-1; i >= 0; --i) {
                if (warnings[i].host == NULL)
                   ;
                else {
                      if (stricmp(warnings[i].host, user->username) == 0) {
                         if (warnings[i].time > (ctime - 10))
                             return;
                    }
                }
             }


            i = CLONE_DETECT_SIZE-1;
            if (warnings[0].host)
               free(warnings[0].host);
            bcopy(warnings+1, warnings, sizeof(struct clone) * i);

            warnings[i].host = sstrdup(user->username);
            warnings[i].time = time(NULL);

            wallops(SERVER_NAME,  D_CLONES,
                   user->username, user->host, uclone_count);
            log("%s: possible clones detected from %s@%s [%d]",
                 s_OperServ, user->username, user->host, uclone_count);

        }

        for (i = CLONE_DETECT_SIZE-1; i >= 0; --i) {
             if (warnings[i].host == NULL)
                ;
             else {
                if (stricmp(warnings[i].host, user->host) == 0) {
                     if (warnings[i].time > (ctime - 10))
                         return;
                }
            }
        }


        if (((tfound == 1) && (u_match == 1)
             && ((clone_count - uclone_count) >= CLONE_MIN_USERS))
             || ((tfound == 1) && (u_match == 0))
             || ((tfound == 0) && (u_match == 0)
             && (clone_count - uclone_count) >= CLONE_MIN_USERS)) {


            i = CLONE_DETECT_SIZE-1;
            if (warnings[0].host)
               free(warnings[0].host);
            bcopy(warnings+1, warnings, sizeof(struct clone) * i);

            warnings[i].host = sstrdup(user->host);
            warnings[i].time = time(NULL);

            wallops(SERVER_NAME, D_CLONES2,
                   user->host, clone_count);
            log("%s: possible clones detected from %s [%d]",
                 s_OperServ, user->host, clone_count);

        }
    }
}


/*************************************************************************/

void check_del_clones(User *user)
{
    int i;

    i = CLONE_DETECT_SIZE-1;
    while (--i >= 0) {
	if (stricmp(clonelist[i].host, user->host) == 0) {
            clonelist[i].host = NULL;
//            clonelist[i].time = NULL;
            free(clonelist[i].host);
            bcopy(clonelist+i+1, clonelist+i, sizeof(*clonelist) * (CLONE_DETECT_SIZE-1-i));
            memmove(clonelist+i, clonelist+i+1, sizeof(struct clone) * (CLONE_DETECT_SIZE-1-i));
            break;
        }
    }
}

/*************************************************************************/
/*********************** OperServ command functions **********************/
/*************************************************************************/

/* OHELP command. */

static void do_ohelp(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    char *access;

    if (!cmd) {
       log("OS - %s requested ohelp: (index)", source);
       serv_os_help(source, "OS", s_OperServ);
    } else if (stricmp(cmd, "MODE") == 0)
	notice_list(s_OperServ, source, mode_help);
    else if (stricmp(cmd, "UINFO") == 0)
        notice_list(s_OperServ, source, uinfo_help);
    else if (stricmp(cmd, "CLEAR") == 0)
        notice_list(s_OperServ, source, clear_help);
    else if (stricmp(cmd, "IGNORE") == 0)
        notice_list(s_OperServ, source, ignore_help);
    else if (stricmp(cmd, "AKILL") == 0)
	notice_list(s_OperServ, source, akill_help);
    else if (stricmp(cmd, "ZLINE") == 0)
        notice_list(s_OperServ, source, zline_help);
    else if (stricmp(cmd, "TRIGGER") == 0)
        notice_list(s_OperServ, source, trigger_help);
    else if (stricmp(cmd, "GLOBAL") == 0) {
	/* Information varies, so we need to do it manually. */
        notice(s_OperServ, source, "Syntax: GLOBAL <target> <message>");
	notice(s_OperServ, source, " ");
	notice(s_OperServ, source,
	    "Allows IRCops to send messages to all users on the Network.");
	notice(s_OperServ, source,
	    "The message will be sent from the nick %s.",
                  s_GlobalNoticer);
        notice(s_OperServ, source,
            "\2TARGET\2 is what domain of people you want the message to");
        notice(s_OperServ, source,
            "be sent. If \2TARGET\2 is \"ALL\" then every client on the");
        notice(s_OperServ, source,
            "network will be messaged");
        notice(s_OperServ, source,
            "Example: /MSG OperServ GLOBAL *.home.com <message here>");

    } else if (stricmp(cmd, "STATS") == 0)
	notice_list(s_OperServ, source, stats_help);
    else if (stricmp(cmd, "SETTINGS") == 0)
        notice_list(s_OperServ, source, settings_help);
    else if (stricmp(cmd, "SENDLOGS") == 0)
        notice_list(s_OperServ, source, sendlog_help);
    else if (stricmp(cmd, "JUPE") == 0)
	notice_list(s_OperServ, source, jupe_help);
    else if (stricmp(cmd, "SQLINE") == 0)
        notice_list(s_OperServ, source, sqline_help);
#if BAH_V >= 147
    else if (stricmp(cmd, "SGLINE") == 0)
        notice_list(s_OperServ, source, sgline_help);
#endif
    else if (stricmp(cmd, "AUTOKILL") == 0)
        notice_list(s_OperServ, source, akill_help);
    else if (stricmp(cmd, "HOP") == 0)
        notice_list(s_OperServ, source, os_acc_help);
    else if (stricmp(cmd, "SOP") == 0)
        notice_list(s_OperServ, source, os_acc_help);
    else {
	notice(s_OperServ, source,
			"No help topic for \2%s\2 found.", cmd);
        return;
    }

    if (cmd && (stricmp(cmd, "GLOBAL") != 0)) {
       if (access = find_level(cmd))
          notice(s_OperServ, source,
            "This command is limited to \2%s\2.", access);
    }



   notice(s_OperServ, source, "\2END OF OHELP\2");

}

/*************************************************************************/

static void do_clear(const char *source)
{
    char *chan = strtok(NULL, " ");
    const char *what = strtok(NULL, " ");
    const char *reason = strtok(NULL, "");
    User *u = finduser(source);
    Channel *c;
    ChannelInfo *ci;
    int i, x = 0;

    if (!chan || !what) {

        notice(s_OperServ, source,
                "Syntax: \2CLEAR\2 <#channel> [BANS|MODES|OPS|VOICES|USERS]");

    } else if (!(c = findchan(chan))) {
        notice(s_OperServ, source, "Channel %s does not exist.", chan);


    } else if (stricmp(what, "modes") == 0) {

	char *av[3];

	av[0] = chan;
	av[1] = sstrdup("-mintpslk");
	if (c->key)
	    av[2] = sstrdup(c->key);
	else
	    av[2] = sstrdup("");
        send_cmd(s_ChanServ, "MODE %s %s :%s", av[0], av[1], av[2]);
        do_cmode(s_ChanServ, 3, av);
	free(av[2]); free(av[1]);
        notice(s_OperServ, source, "All modes on %s have been reset.", chan);
        wallops(s_OperServ, "%s had me remove all modes for %s", source, chan);

    } else if (stricmp(what, "bans") == 0) {

	char *av[3];
        char buf[1024], buf2[128];
        int bcnt=0;

        *buf=0; *buf2=0;

	while (c->bancount > 0) {
	    av[0] = sstrdup(chan);
	    av[1] = sstrdup("-b");
	    av[2] = sstrdup(c->bans[0]);
//            send_cmd(s_ChanServ, "MODE %s %s :%s", av[0], av[1], av[2]);
	    do_cmode(s_ChanServ, 3, av);
               if (*buf)
                   strcat(buf, " ");
               else
                   strcat(buf2, "-");
               strcat(buf, av[2]);
               strcat(buf2, "b");
               bcnt++;
               if (bcnt > 5) {
                  bcnt = 0;
                  send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);
                  *buf=0; *buf2=0;
               }
            free(av[0]); free(av[1]); free(av[2]);
	}
        if (*buf)
          send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);
        notice(s_OperServ, source, "All bans on %s have been removed.", chan);
        wallops(s_OperServ, "%s had me remove all bans for %s", source, chan);


    } else if (stricmp(what, "ops") == 0) {

	char *av[3];
	struct c_userlist *cu, *next;
        char buf[1024], buf2[128];
        int dcnt=0;

        *buf=0; *buf2=0;

        i=0;
        for (cu = c->chanops; cu; cu = next) {
	    next = cu->next;
            if (stricmp(cu->user->nick, source) != 0) {

               i += 1;
   	       av[0] = sstrdup(chan);
	       av[1] = sstrdup("-o");
	       av[2] = sstrdup(cu->user->nick);
//               send_cmd(s_ChanServ, "MODE %s %s :%s", av[0], av[1], av[2]);
               do_cmode(s_ChanServ, 3, av);
               if (*buf)
                   strcat(buf, " ");
               else
                   strcat(buf2, "-");
               strcat(buf, av[2]);
               strcat(buf2, "o");
               dcnt++;
               if (dcnt > 5) {
                  dcnt = 0;
                  send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);
                  *buf=0; *buf2=0;
               }
               free(av[0]); free(av[1]); free(av[2]);
           }
	}
        if (*buf)
            send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);

        notice(s_OperServ, source, "Mode +o cleared from %s.", chan);
        wallops(s_OperServ,
             "%s had me remove all ops for %s. [Ops: %d]",
                    source, chan, i);

    } else if (stricmp(what, "voices") == 0) {

	char *av[3];
	struct c_userlist *cu, *next;
        char buf[1024], buf2[128];
        int vcnt=0;

        *buf=0; *buf2=0;

	for (cu = c->voices; cu; cu = next) {
	    next = cu->next;
	    av[0] = sstrdup(chan);
	    av[1] = sstrdup("-v");
	    av[2] = sstrdup(cu->user->nick);
//            send_cmd(s_ChanServ, "MODE %s %s :%s", av[0], av[1], av[2]);
            do_cmode(s_ChanServ, 3, av);
               if (*buf)
                   strcat(buf, " ");
               else
                   strcat(buf2, "-");
               strcat(buf, av[2]);
               strcat(buf2, "v");
               vcnt++;
               if (vcnt > 5) {
                  vcnt = 0;
                  send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);
                  *buf=0; *buf2=0;
               }
            free(av[0]); free(av[1]); free(av[2]);
	}
        if (*buf)
             send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);
        notice(s_OperServ, source, "Mode +v cleared from %s.", chan);

    } else if (stricmp(what, "users") == 0) {

	char *av[3];
	struct c_userlist *cu, *next_cu;
	char buf[256];
        Timeout *t;

	snprintf(buf, sizeof(buf), "CLEAR USERS command from %s", source);
        send_cmd(s_ChanServ, "JOIN %s", chan);
        send_cmd(s_ChanServ, "MODE %s +b *!*@", chan);

        i=0;
	for (cu = c->users; cu; cu = next_cu) {
	    next_cu = cu->next;
            if (stricmp(cu->user->nick, source) != 0) {
               i += 1;
	       av[0] = sstrdup(chan);
	       av[1] = sstrdup(cu->user->nick);
               if (!reason)
	          av[2] = sstrdup(buf);
               else
                  av[2] = sstrdup(reason);
	       send_cmd(s_ChanServ, "KICK %s %s :%s", av[0], av[1], av[2]);
	       do_kick(s_ChanServ, 3, av);
               free(av[0]); free(av[1]); free(av[2]);
            }
	}

        notice(s_OperServ, source, "All users kicked from %s.", chan);

        t = add_timeout(CHANNEL_INHABIT, timeout_unban, 0);
        t->data = sstrdup(chan);

        t = add_timeout(CHANNEL_INHABIT+1, timeout_leave, 0);
        t->data = sstrdup(chan);


        if (!reason)
           wallops(s_OperServ,
             "%s had me kick all user for %s. [Reason: Unspecified -=- Users: %d]",
                 source, chan, i);
        else
           wallops(s_OperServ,
             "%s had me kick all user for %s. [Reason: %s -=- Users: %d]",
                 source, chan, reason, i);

    }
}

/*************************************************************************/

static void do_inject(const char *source)
{
    const char *nick = strtok(NULL, " ");
    const char *snick = strtok(NULL, " ");
    char *cmd = strtok(NULL, "");
    char buf[BUFSIZE];

    if (!nick || !snick || !cmd)
	return;

    if (stricmp(snick, s_OperServ) == 0)
	operserv(nick,cmd);
    if (stricmp(snick, s_NickServ) == 0)
	nickserv(nick,cmd);
    if (stricmp(snick, s_ChanServ) == 0)
        chanserv(nick,cmd);
    if (stricmp(snick, s_MemoServ) == 0)
	memoserv(nick,cmd);
}

/*************************************************************************/

/* List admins */
static void do_listadm(const char *source)
{
    NickInfo *ni;
    int i;

    notice(s_OperServ, source, "\2Services Help Operators:\2");
    for (i = 0; i < MAX_HELPOPS; i++) {
            if (services_helpers[i]) {
              if (ni = findnick(services_helpers[i]))
                 notice(s_OperServ, source, "%s (%s)", services_helpers[i], ni->last_usermask);
            }
    }
    notice(s_OperServ, source, "\2Services Operators:\2");
    for (i = 0; i < MAX_SERVOPERS; i++) {
            if (services_opers[i]) {
              if (ni = findnick(services_opers[i]))
                 notice(s_OperServ, source, "%s (%s)", services_opers[i], ni->last_usermask);
            }
    }
    notice(s_OperServ, source, "\2Services Administrators:\2");
    for (i = 0; i < MAX_SERVADMINS; i++) {
            if (services_admins[i]) {
              if (ni = findnick(services_admins[i]))
                 notice(s_OperServ, source, "%s (%s)", services_admins[i], ni->last_usermask);
            }
    }
    notice(s_OperServ, source, "\2Services Root Administrators:\2");
    for (i = 0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i]) {
            if (stricmp(services_roots[i], SERVICES_MASTER) == 0) {
                   if (ni = findnick(services_roots[i]))
                      notice(s_OperServ, source, "%s (%s) [Master]", services_roots[i], ni->last_usermask);
            }
            else {
              if (ni = findnick(services_roots[i]))
                 notice(s_OperServ, source, "%s (%s)", services_roots[i], ni->last_usermask);
            }
        }
    }
   
    notice(s_OperServ, source, "\2End of List\2");
}


/*************************************************************************/

/* Services Opers list modification. */

static void do_sop(const char *source)
{
    char *cmd, *nick;
    NickInfo *ni;
    int i;

    cmd = strtok(NULL, " ");
    if (!cmd)
	cmd = "";

    if (stricmp(cmd, "ADD") == 0) {
        if (!check_o_access(source, "SOP")) {
	    notice(s_OperServ, source, "Permission denied.");
	    return;
	}
	nick = strtok(NULL, " ");
	if (nick) {
	    if (!findnick(nick)) {
		notice(s_OperServ, source, "Nick %s isn't registered!", nick);
		return;
	    }
            for (i = 0; i < MAX_SERVOPERS; i++) {
                if (services_opers[i] && !stricmp(nick,services_opers[i])) {
                    notice(s_OperServ, source, "%s is already a SOP",
                        nick);
                    return;
                }
                if (!services_opers[i])
                    break;
            }
            if ((i < MAX_SERVOPERS) || (MAX_SERVOPERS == 0)) {
                services_opers[i] = sstrdup(nick);
                log("!! Added \2%s\2 to SOP List", nick);
                do_break_log("OS", "!! Added %s to SOP List", nick);
                wallops(SERVER_NAME, "%s added %s to the SOP List", source, nick);
		notice(s_OperServ, source,
                        "%s added to Services Operator list.", nick);

                for (i = 0; i < MAX_HELPOPS; i++) {
                    if (services_helpers[i] && !stricmp(nick, services_helpers[i]))
                        services_helpers[i] = NULL;
                }

                for (i = 0; i < MAX_SERVADMINS; i++) {
                    if (services_admins[i] && !stricmp(nick, services_admins[i]))
                        services_admins[i] = NULL;
                }

                for (i = 0; i < MAX_SERVROOTS; i++) {
                    if (services_roots[i] && !stricmp(nick, services_roots[i]))
                        services_roots[i] = NULL;
                }

	    } else {
		notice(s_OperServ, source,
                        "Too many entries (%d) on Services Operator list; cannot add more.",
                        MAX_SERVOPERS);
                wallops(s_OperServ,
                        "\2Warning\2: Services Operators list is full! (%d)",
                        MAX_SERVOPERS);
	    }
	    if (readonly) {
		notice(s_OperServ, source,
		    "\2Notice:\2 Changes will not be saved!  Services is in read-only mode.");
	    }
	} else {
            notice(s_OperServ, source, "Syntax: \2SOP ADD\2 <nick>");
	}

    } else if (stricmp(cmd, "DEL") == 0) {
        if (!check_o_access(source, "SOP")) {
	    notice(s_OperServ, source, "Permission denied.");
	    return;
	}
	if (nick = strtok(NULL, " ")) {
            for (i = 0; i < MAX_SERVOPERS; i++) {
                if (services_opers[i] && stricmp(nick,services_opers[i]) == 0)
		    break;
	    }
            if ((i < MAX_SERVOPERS) || (MAX_SERVOPERS == 0)) {
                free(services_opers[i]);
                services_opers[i] = NULL;
                log("!! Deleted \2%s\2 from SOP List", nick);
                do_break_log("OS", "!! Deleted %s from SOP List", nick);
                wallops(SERVER_NAME, "%s deleted %s from the SOP List", source, nick);
		notice(s_OperServ, source,
                        "%s removed from Services Operator list.", nick);
		if (readonly) {
		    notice(s_OperServ, source,
			    "\2Notice:\2 Changes will not be saved!  Services is in read-only mode.");
		}
	    } else {
		notice(s_OperServ, source,
                        "%s not found on Services Operator list.", nick);
	    }
	} else {
            notice(s_OperServ, source, "Syntax: \2SOP DEL\2 <nick>");
	}

    } else if (stricmp(cmd, "LIST") == 0) {
           notice(s_OperServ, source, "\2Services Operators:\2");
           notice(s_OperServ, source, "");
        for (i = 0; i < MAX_SERVOPERS; i++) {
            if (services_opers[i])
              if (ni = findnick(services_opers[i]))
                 notice(s_OperServ, source, "%s (%s)", services_opers[i], ni->last_usermask);
            }
           notice(s_OperServ, source, "\2End of List\2");


    } else {
	notice(s_OperServ, source,
                "Syntax: \2SOP\2 [ADD|DEL|LIST] [<nick>]");
	notice(s_OperServ, source,
                "For help: \2/msg %s HELP SOP\2", s_OperServ);
    }
}

/*************************************************************************/

/* Services Help Operators list modification. */

static void do_hop(const char *source)
{
    char *cmd, *nick;
    NickInfo *ni;
    int i;

    cmd = strtok(NULL, " ");
    if (!cmd)
	cmd = "";


    if (stricmp(cmd, "ADD") == 0) {
        if (!check_o_access(source, "HOP")) {
	    notice(s_OperServ, source, "Permission denied.");
	    return;
	}
	nick = strtok(NULL, " ");
	if (nick) {
	    if (!findnick(nick)) {
		notice(s_OperServ, source, "Nick %s isn't registered!", nick);
		return;
	    }
            for (i = 0; i < MAX_HELPOPS; i++) {
               if (services_helpers[i] && !stricmp(nick,services_helpers[i])) {
                   notice(s_OperServ, source, "%s is already listed on the HOP list.",
                               nick);
                    return;
                }
                if (!services_helpers[i])
		    break;
	    }
            if ((i < MAX_HELPOPS) || (MAX_HELPOPS == 0)) {
                services_helpers[i] = sstrdup(nick);
                log("!! Added \2%s\2 to HOP List", nick);
                do_break_log("OS", "!! Added %s to HOP List", nick);
                wallops(SERVER_NAME, "%s added %s to the HOP List", source, nick);
		notice(s_OperServ, source,
                        "%s added to Services Help Operator list.", nick);

                for (i = 0; i < MAX_SERVOPERS; i++) {
                    if (services_opers[i] && !stricmp(nick, services_opers[i]))
                        services_opers[i] = NULL;
                }

                for (i = 0; i < MAX_SERVADMINS; i++) {
                    if (services_admins[i] && !stricmp(nick, services_admins[i]))
                        services_admins[i] = NULL;
                }

                for (i = 0; i < MAX_SERVROOTS; i++) {
                    if (services_roots[i] && !stricmp(nick, services_roots[i]))
                        services_roots[i] = NULL;
                }

	    } else {
		notice(s_OperServ, source,
                        "Too many entries (%d) on Services Help Operator list; cannot add more.",
                        MAX_HELPOPS);
                wallops(s_OperServ,
                        "\2Warning\2: Help Operator list is full! (%d)",
                        MAX_HELPOPS);
	    }
	    if (readonly) {
		notice(s_OperServ, source,
		    "\2Notice:\2 Changes will not be saved!  Services is in read-only mode.");
	    }
	} else {
            notice(s_OperServ, source, "Syntax: \2HOP ADD\2 <nick>");
	}

    } else if (stricmp(cmd, "DEL") == 0) {
        if (!check_o_access(source, "HOP")) {
	    notice(s_OperServ, source, "Permission denied.");
	    return;
	}
	if (nick = strtok(NULL, " ")) {
            for (i = 0; i < MAX_HELPOPS; i++) {
                if (services_helpers[i] && stricmp(nick,services_helpers[i]) == 0)
		    break;
	    }
            if (i < MAX_HELPOPS) {
                free(services_helpers[i]);
                services_helpers[i] = NULL;
                log("!! Deleted \2%s\2 from HOP List", nick);
                do_break_log("OS", "!! Deleted %s from HOP List", nick);
                wallops(SERVER_NAME, "%s deleted %s from the HOP List", source, nick);
		notice(s_OperServ, source,
                        "%s removed from Services Help Operator list.", nick);
		if (readonly) {
		    notice(s_OperServ, source,
			    "\2Notice:\2 Changes will not be saved!  Services is in read-only mode.");
		}
	    } else {
		notice(s_OperServ, source,
                        "%s not found on Services Help Operator list.", nick);
	    }
	} else {
            notice(s_OperServ, source, "Syntax: \2HOP DEL\2 <nick>");
	}

    } else if (stricmp(cmd, "LIST") == 0) {
           notice(s_OperServ, source, "\2Services Help Operators:\2");
           notice(s_OperServ, source, "");
        for (i = 0; i < MAX_HELPOPS; i++) {
            if (services_helpers[i]) {
              if (ni = findnick(services_helpers[i]))
                 notice(s_OperServ, source, "%s (%s)", services_helpers[i], ni->last_usermask);
            }
         }
           notice(s_OperServ, source, "\2End of List\2");

    } else {
	notice(s_OperServ, source,
                "Syntax: \2HOP\2 [ADD|DEL|LIST] [<nick>]");
	notice(s_OperServ, source,
                "For help: \2/msg %s HELP HOP\2", s_OperServ);
    }
}


/*************************************************************************/

/* AKILL list modification. */

static void do_akill(const char *source)
{
    User *user;
    char *cmd, *reason, *who, *s;

    cmd = strtok(NULL, " ");
    if (!cmd)
	cmd = "";

    if (stricmp(cmd, "LIST") == 0) {
        int i;
	s = strtok(NULL, " ");
	if (!s)
	    s = "*";
	if (strchr(s, '@'))
	    strlower(strchr(s, '@'));
	notice(s_OperServ, source, "\2Current AKILL list:\2");
	for (i = 0; i < nakill; i++) {
	    if (match_wild(s, akills[i].mask)) {
                char timebuf[32], expirebuf[64];
                struct tm tm;
                time_t t = time(NULL);

                tm = *localtime(akills[i].time ? &akills[i].time : &t);
                strftime(timebuf, sizeof(timebuf), "%d %b %Y", &tm);
                if (akills[i].expires == 0) {
                    strcpy(expirebuf, "Does not expire");
                } else if (akills[i].expires <= t) {
                    strcpy(expirebuf, "Expires at next database update");
                } else {
                    time_t t2 = akills[i].expires - t;
                    t2 += 59;
                    if (t2 < 86400) {
                        t2 /= 60;
                        snprintf(expirebuf, sizeof(expirebuf),
                                "Expires in %d hour%s, %d minute%s",
                                t2/60, t2<120 ? "" : "s",
                                t2%60, t2%60==1 ? "" : "s");
                    } else {
                        t2 /= 86400;
                        snprintf(expirebuf, sizeof(expirebuf),
                                "Expires in %d day%s", t2, t2==1 ? "" : "s");
                    }


                }

		notice(s_OperServ, source, " %d) %-32s  \"%s\"",
				i+1,akills[i].mask,akills[i].reason);
		notice(s_OperServ, source, "Set by %s on %s; %s",
				akills[i].who,timebuf,expirebuf);
         }
      }
      notice(s_OperServ,source,"\2End of List\2");


    } else if (!check_o_access(source, "AKILL")) {
        notice(s_OperServ, source, "Permission denied.");

    } else if (stricmp(cmd, "ADD") == 0) {

	char mask[256];
        char *expiry = NULL;
        const char *ident = strtok(NULL, "@");
        const char *host = strtok(NULL, " ");
        int i=0, x=0;
        float percent;

        if (!ident || !host) {
	    notice(s_OperServ, source,
                        "Syntax: \2AKILL ADD\2 <ident@host> <reason>");
            return;
        }         

        *mask = 0;
        strcat(mask, ident); strcat(mask, "@");
        strcat(mask, host);

    
        if (strchr(mask, '!')) {
             notice(s_OperServ, source,
                 "Akills may not contain '!' characters");
             return;
        }

        for (user = userlist; user; user = user->next) {
           if (match_wild_nocase(ident, user->username)
              && match_wild_nocase(host, user->host))
           x += 1;
        }
       percent = ((x + .0) * 100.0) / usercnt;
       if (percent > AKILL_PERCENT) {
           notice(s_OperServ, source, "AKILL rejected. Effected users would be "
              "greater than %.3f%s", AKILL_PERCENT, "%");
           wallops(s_OperServ, "%s tried to AKILL %.3f%s of the network! "
              "(Limit: %.3f%s)",
              source, percent, "%", AKILL_PERCENT, "%");
           return;
        }


#ifndef BAHAMUT
        if (strchr(mask, '!')) {
          notice(s_OperServ, source, "Mask must not contain an '!' -> *@* format only.");
          return;
        }

#endif
	if (mask && (reason = strtok(NULL, ""))) {
            if (!stricmp(mask,"*@*.*") || !stricmp(mask,"*@*") ||
                !stricmp(mask,"*@*.com") || !stricmp(mask,"*@*.net") ||
                !stricmp(mask,"*@*.org") || !stricmp(mask,"*@*.ca")) {
		notice(s_OperServ,source,"Hrmmm, What would your admin "
		    "think of that?");
                wallops(s_OperServ,"%s tried to AKILL %s",
		    source,mask);
		return;
	    }
	    for (i = 0; i < nakill; i++) {
		if (stricmp(akills[i].mask,mask) == 0) {
                    notice(s_OperServ,source,"%s is already present "
			"on the AKILL list.",mask);
		    return;
		}
	    }
	    if (s = strchr(mask, '@')) {
		strlower(s);
	    } else {
		notice(s_OperServ, source,
			"Hostmask must contain an `@' character.");
		return;
	    }
	    add_akill(mask, reason, source, expiry, x);
	    notice(s_OperServ, source, "%s added to AKILL list.", mask);
	    mass_check_akill();
	    if (readonly) {
		notice(s_OperServ, source,
		    "\2Notice:\2 Changes will not be saved! "
		    " Services is in read-only mode.");
	    }
	} else {
	    notice(s_OperServ, source,
                        "Syntax: \2AKILL ADD\2 <ident@host> <reason>");
	}

    } else if (stricmp(cmd, "TIME") == 0) {
	char mask[256];
	const char *expiry = strtok(NULL, " ");
        const char *ident = strtok(NULL, "@");
        const char *host = strtok(NULL, " ");
        int i=0, x=0;
        float percent;


        if (!ident || !host || !expiry) {
	    notice(s_OperServ, source,
                        "Syntax: \2AKILL TIME\2 <time> <ident@host> <reason>");
            return;
        }         

        *mask = 0;
        strcat(mask, ident); strcat(mask, "@");
        strcat(mask, host);


        if (strchr(mask, '!')) {
             notice(s_OperServ, source,
                 "Akills may not contain '!' characters");
             return;
        }

        for (user = userlist; user; user = user->next) {
           if (match_wild_nocase(ident, user->username)
              && match_wild_nocase(host, user->host))
            x += 1;
        }

       percent = ((x + .0) * 100.0) / usercnt;
       if (percent > AKILL_PERCENT) {
           notice(s_OperServ, source, "AKILL rejected. Effected users would be "
              "greater than %.3f%s", AKILL_PERCENT, "%");
           wallops(s_OperServ, "%s tried to AKILL %.3f%s of the network! "
              "(Limit: %.3f%s)",
              source, percent, "%", AKILL_PERCENT, "%");
           return;
        }

#ifndef BAHAMUT
        if (strchr(mask, '!')) {
          notice(s_OperServ, source, "Mask must not contain an '!' -> *@* format only.");
          return;
        }
#endif

        if (mask && (reason = strtok(NULL, ""))) {
            if (!stricmp(mask,"*@*.*") || !stricmp(mask,"*@*") ||
                !stricmp(mask,"*@*.com") || !stricmp(mask,"*@*.net") ||
                !stricmp(mask,"*@*.org") || !stricmp(mask,"*@*.ca")) {
                notice(s_OperServ,source,"Hrmmm, What would your admin "
                    "think of that?");
                wallops(s_OperServ,"%s tried to AKILL %s",
                    source,mask);
                return;
            }
            for (i = 0; i < nakill; i++) {
                if (stricmp(akills[i].mask,mask) == 0) {
                    notice(s_OperServ,source,"%s is already present "
                        "on the AKILL list.",mask);
                    return;
                }
            }
            if (s = strchr(mask, '@')) {
                strlower(s);
            } else {
                notice(s_OperServ, source,
                        "Hostmask must contain an `@' character.");
                return;
            }
            add_akill(mask, reason, source, expiry, x);
            notice(s_OperServ, source, "%s added to AKILL list.", mask);
            mass_check_akill();
            if (readonly) {
                notice(s_OperServ, source,
                    "\2Notice:\2 Changes will not be saved! "
                    " Services is in read-only mode.");
            }
        } else {
            notice(s_OperServ, source,
                        "Syntax: \2AKILL TIME\2 <expiry> <ident@host> <reason>");
        }

    } else if (stricmp(cmd, "INFO") == 0) {
        int i;
        const char *mask = strtok(NULL, " ");
        const char *info = strtok(NULL, "");

        if (!is_services_root(source)) {
            notice(s_OperServ, source, "Permission denied.");
            return;
        }

        if (!mask || !info)
            return;

        if (strspn(mask, "1234567890") == strlen(mask) &&
                            (i = atoi(mask)) > 0 && i <= nakill) {

            free(akills[i-1].reason);
            akills[i-1].reason = sstrdup(info);
            notice(s_OperServ, source, "AKILL reason for %s changed to: %s",
                         akills[i-1].mask, sstrdup(info));
            return;
        }

        for (i = 0; i < nakill; i++) {
              if (stricmp(akills[i].mask,mask) == 0) {
                  free(akills[i].reason);
                  akills[i].reason = sstrdup(info);
                  notice(s_OperServ, source, "AKILL reason for %s changed to: %s",
                               akills[i].mask, akills[i].reason);
		  return;
              }
        }
	notice(s_OperServ, source, "No match for %s", mask);

    } else if (stricmp(cmd, "DEL") == 0) {
        char *mask;
	int i;
	if (mask = strtok(NULL, " ")) {
	    if (strspn(mask, "1234567890") == strlen(mask) &&
                                (i = atoi(mask)) > 0 && i <= nakill) {

		strcpy(mask,akills[i-1].mask);
	    }
	    if (s = strchr(mask, '@'))
		strlower(s);
	    if (del_akill(mask)) {
		notice(s_OperServ, source, "%s removed from AKILL list.",mask);
		wallops(s_OperServ,
                        "%s removed %s from the AKILL list.",
			source, mask);
		if (s) {
		    *s++ = 0;
		    send_cmd(server_name, "RAKILL %s %s", s, mask);
		} else {
		    /* We lose... can't figure out what's a username and what's
		     * a hostname.  Ah well.
		     */
		}
		if (readonly) {
		    notice(s_OperServ, source,
			"\2Notice:\2 Changes will not be saved!  Services is in read-only mode.");
		}
	    } else {
		notice(s_OperServ, source, "%s not found on AKILL list.", mask);
	    }
	} else {
            notice(s_OperServ, source, "Syntax: \2AKILL DEL\2 <mask>");
	}

    } else {
	notice(s_OperServ, source,
                "Syntax: \2AKILL\2 [ADD|DEL|LIST|TIME] [<expiry>] [<mask>] [<reason>]");
	notice(s_OperServ, source,
		"For help: \2/msg %s HELP AKILL\2", s_OperServ);
    }
}

/*************************************************************************/

/* Set various Services runtime options. */

static void do_set(const char *source)
{
    const char *option = strtok(NULL, " ");
    const char *setting = strtok(NULL, " ");

    if (!option || !setting) {
	notice(s_OperServ, source,
                        "Syntax: \2SET <option> <value>");

    } else if (stricmp(option, "SNOOP") == 0) {
	if (stricmp(setting, "on") == 0) {
            if (snoop == 1) {
               notice(s_OperServ, source, "Snoop already enabled");
               return;
            }
            snoop = 1;
            notice(s_OperServ, source, OS_SET_SNOOP);
	} else if (stricmp(setting, "off") == 0) {
            if (snoop != 1) {
               notice(s_OperServ, source, "Snoop already disabled");
               return;
            }
            snoop = 0;
            notice(s_OperServ, source, OS_SET_SNOOP2);
	} else {
	    notice(s_OperServ, source,
                    "Setting for \2SNOOP\2 must be \2ON\2 or \2OFF\2.");
	}


    } else if (stricmp(option, "IGNORE") == 0) {
	if (stricmp(setting, "on") == 0) {
            if (allow_ignore == 1) {
               notice(s_OperServ, source, "Ignore code already enabled");
               return;
            }
	    allow_ignore = 1;
            notice(s_OperServ, source, OS_SET_IGNORE);
            wallops(SERVER_NAME, "Services Mode: Ignore Code Enabled (%s)",
                  source);
	} else if (stricmp(setting, "off") == 0) {
            if (allow_ignore != 1) {
               notice(s_OperServ, source, "Ignore code already disabled");
               return;
            }
	    allow_ignore = 0;
            notice(s_OperServ, source, OS_SET_IGNORE2);
            wallops(SERVER_NAME, "Services Mode: Ignore Code Disabled (%s)",
                  source);
	} else {
	    notice(s_OperServ, source,
		    "Setting for \2IGNORE\2 must be \2ON\2 or \2OFF\2.");
	}
    } else if (stricmp(option, "SDEBUG") == 0) {
        if (stricmp(setting, "on") == 0) {
            if (sndebug == 1) {
               notice(s_OperServ, source, "Snoop debug already enabled");
               return;
            }
           sndebug = 1;
           notice(s_OperServ, source, "Snoop Debug Activated");
           log("Snoop Debug Activated");
        }
        if (stricmp(setting, "off") == 0) {
            if (sndebug != 1) {
               notice(s_OperServ, source, "Snoop debug already disabled");
               return;
            }
           sndebug = 0;
           notice(s_OperServ, source, "Snoop Debug Deactivated");
           log("Snoop Debug Deactivated");
           log("Snoop Debug De-Activated");
        }

    } else if (stricmp(option, "DEBUG") == 0) {
        if (stricmp(setting, "on") == 0) {
            if (debug == 1) {
               notice(s_OperServ, source, "Debug already enabled");
               return;
            }
           debug = 1;
           notice(s_OperServ, source, "Debug Activated");
           log("Debug Activated");
        }
        if (stricmp(setting, "off") == 0) {
            if (debug != 1) {
               notice(s_OperServ, source, "Debug already disabled");
               return;
            }
           debug = 0;
           notice(s_OperServ, source, "Debug Deactivated");
           log("Debug De-Activated");
        }

    } else if (stricmp(option, "SENDLOGS") == 0) {
	if (stricmp(setting, "on") == 0) {
            if (sendstat == 1) {
               notice(s_OperServ, source, "Sending logs already enabled");
               return;
            }
           clone_detect = 0;
           log("Sending Logs Enabled");
           close_log();
            notice(s_OperServ, source, OS_SET_SENDLOG);
            wallops(SERVER_NAME, "Services Mode: Send Logs [ON] (%s)",
                  source);
	} else if (stricmp(setting, "off") == 0) {
            if (sendstat != 1) {
               notice(s_OperServ, source, "Send Logs already disabled");
               return;
            }
            sendstat = 0;
	    open_log();
            log("Send Logs Disabled");
            notice(s_OperServ, source, OS_SET_SENDLOG2);
            wallops(SERVER_NAME, "Services Mode: Send Logs [OFF] (%s)",
                  source);
	} else {
	    notice(s_OperServ, source,
                    "Setting for \2SENDLOGS\2 must be \2ON\2 or \2OFF\2.");
	}

    } else if (stricmp(option, "DUMPLOG") == 0) {
	if (stricmp(setting, "on") == 0) {
            if (dumplog == 1) {
               notice(s_OperServ, source, "Dumping Log already Active");
               return;
            }
           dumplog = 0;
           log("DumpLog Enabled");
           close_log();
            notice(s_OperServ, source, "Services will now dump logs");
	} else if (stricmp(setting, "off") == 0) {
            if (dumplog != 1) {
               notice(s_OperServ, source, "Dump Logs already disabled");
               return;
            }
            dumplog = 0;
	    open_log();
            log("Dump Logs Disabled");
            notice(s_OperServ, source, "Services will NOT dump logs.");
	} else {
	    notice(s_OperServ, source,
                    "Setting for \2DUMPLOG\2 must be \2ON\2 or \2OFF\2.");
	}


    } else if (stricmp(option, "CLONES") == 0) {
	if (stricmp(setting, "on") == 0) {
            if (clone_detect == 1) {
               notice(s_OperServ, source, "Clone detection already enabled");
               return;
            }
           clone_detect = 1;
           log("Clone Detection Enabled");
           close_log();
            notice(s_OperServ, source, OS_SET_CLONE);
            wallops(SERVER_NAME, "Services Mode: Clone Detection (%s)",
                  source);
	} else if (stricmp(setting, "off") == 0) {
            if (clone_detect != 1) {
               notice(s_OperServ, source, "Clone detection already disabled");
               return;
            }
            clone_detect = 0;
	    open_log();
            log("Clone Detect Deactivated");
            notice(s_OperServ, source, OS_SET_CLONE2);
            wallops(SERVER_NAME, "Services Mode: Clone Detection Disabled (%s)",
                  source);
	} else {
	    notice(s_OperServ, source,
                    "Setting for \2CLONES\2 must be \2ON\2 or \2OFF\2.");
	}

    } else if (stricmp(option, "FLOOD") == 0) {
	if (stricmp(setting, "on") == 0) {
            if (allow_flood == 1) {
               notice(s_OperServ, source, "Flood levels already enabled");
               return;
            }
           allow_flood = 1;
           log("Flood Levels Enabled");
           close_log();
            notice(s_OperServ, source, OS_SET_FLOOD);
            wallops(SERVER_NAME, "Services Mode: Flood Level Activated (%s)",
                  source);
	} else if (stricmp(setting, "off") == 0) {
            if (allow_flood != 1) {
               notice(s_OperServ, source, "Flood levels already disabled");
               return;
            }
            allow_flood = 0;
	    open_log();
            log("Flood levels Deactivated");
            notice(s_OperServ, source, OS_SET_FLOOD2);
            wallops(SERVER_NAME, "Services Mode: Flood Levels Disabled (%s)",
                  source);
	} else {
	    notice(s_OperServ, source,
                    "Setting for \2FLOOD\2 must be \2ON\2 or \2OFF\2.");
	}

    } else if (stricmp(option, "REALNAME") == 0) {
	if (stricmp(setting, "on") == 0) {
            if (realname == 1) {
               notice(s_OperServ, source, "Real Name Catching already enabled");
               return;
            }
           realname = 1;
           log("Real Name Catching Enabled");
           close_log();
            notice(s_OperServ, source, OS_SET_RNC);
            wallops(SERVER_NAME, "Services Mode: Real Name Catching (%s)",
                  source);
	} else if (stricmp(setting, "off") == 0) {
            if (realname != 1) {
               notice(s_OperServ, source, "Real Name Catching already disabled");
               return;
            }
            realname = 0;
	    open_log();
            log("Real Name Catching Deactivated");
            notice(s_OperServ, source, OS_SET_RNC2);
            wallops(SERVER_NAME, "Services Mode: Real Name Catching Disabled (%s)",
                  source);
	} else {
	    notice(s_OperServ, source,
                    "Setting for \2REALNAME\2 must be \2ON\2 or \2OFF\2.");
	}

    } else if (stricmp(option, "READONLY") == 0) {
	if (stricmp(setting, "on") == 0) {
            if (readonly == 1) {
               notice(s_OperServ, source, "Read only already enabled");
               return;
            }
	    readonly = 1;
	    log("Read-only mode activated");
	    close_log();
            notice(s_OperServ, source, "Services is now in read-only mode.");
            wallops(SERVER_NAME, "Services Mode: READ-ONLY (%s)",
                  source);
	} else if (stricmp(setting, "off") == 0) {
            if (readonly != 1) {
               notice(s_OperServ, source, "Read/Write already enabled");
               return;
            }
	    readonly = 0;
	    open_log();
	    log("Read-only mode deactivated");
            notice(s_OperServ, source, "Services is now in read/write mode.");
            wallops(SERVER_NAME, "Services Mode: READ/WRITE (%s)",
                  source);
	} else {
	    notice(s_OperServ, source,
		    "Setting for \2READONLY\2 must be \2ON\2 or \2OFF\2.");
	}
    } else {
	notice(s_OperServ, source, "Unknown option \2%s\2.", option);
    }
}

/*************************************************************************/

static void do_settings(const char *source)
{
    int timeout = AKILL_DEF_EXPIRY+59;
    int timeout2 = IGNORE_DEF_EXPIRY+59;
    char buf[512];

    if (timeout >= 86400)
        notice(s_OperServ, source,
                "Default AKILL expiry time: \2%d day%s\2",
                timeout/86400, timeout>=172800 ? "s" : "");
    else if (timeout >= 3600)
        notice(s_OperServ, source,
               "Default AKILL expiry time: \2%d hour%s\2",
               timeout/3600, timeout>=7200 ? "s" : "");
    else if (timeout >= 60)
        notice(s_OperServ, source,
                "Default AKILL expiry time: \2%d minute%s\2",
                timeout/60, timeout>=120 ? "s" : "");
    else
        notice(s_OperServ, source,
               "Default AKILL expiry time: \2None\2");

    if (timeout2 >= 86400)
        notice(s_OperServ, source,
                "Default IGNORE expiry time: \2%d day%s\2",
                timeout2/86400, timeout2>=172800 ? "s" : "");
    else if (timeout2 >= 3600)
        notice(s_OperServ, source,
               "Default IGNORE expiry time: \2%d hour%s\2",
               timeout2/3600, timeout2>=7200 ? "s" : "");
    else if (timeout2 >= 60)
        notice(s_OperServ, source,
                "Default IGNORE expiry time: \2%d minute%s\2",
                timeout2/60, timeout2>=120 ? "s" : "");
    else
        notice(s_OperServ, source,
               "Default IGNORE expiry time: \2None\2");

    notice(s_OperServ, source,
             "Clone Trigger: \2%d\2  Clone Wait Warnings: \2%d\2",
                 CLONE_MIN_USERS, CLONE_WARNING_DELAY);

    notice(s_OperServ, source,
             "Default Memo Limit: \2%d\2  Max Memo Length: \2%d\2  Memo Send Wait: \2%d\2",
                 DEF_MAX_MEMOS,
                 MAX_MEMO_LENGTH,
                 MEMO_SEND_WAIT);

    notice(s_OperServ, source,
             "Expiry Times:  Nicks: \2%d\2  Chans: \2%d\2  Memos: \2%d\2",
                   NICK_EXPIRE, CHANNEL_EXPIRE, MEMO_EXPIRE);

    notice(s_OperServ, source,
             "Maximums: Root: \2%d\2  SA: \2%d\2  SOP: \2%d\2  HOP: \2%d\2",
              MAX_SERVROOTS, MAX_SERVADMINS, MAX_SERVOPERS, MAX_HELPOPS);
#if BAH_V >= 147
    notice(s_OperServ, source,
             "Maximums: JUPES: \2%d\2  TRIGGERS: \2%d\2  GLINES: \2%d\2",
              MAX_JUPES, MAX_TRIGGERS, MAX_GLINES);
#else
    notice(s_OperServ, source,
             "Maximums: JUPES: \2%d\2  TRIGGERS: \2%d\2  NICK AKILLS: %s",
              MAX_JUPES, MAX_TRIGGERS, "N/A");
#endif

    notice(s_OperServ, source,
             "Maximums: AKICKS: %d  CHAN ACCESS NICKS: %d  QLINES: %d",
              AKICK_MAX, CHAN_ACCESS_MAX, MAX_QLINES);

    notice(s_OperServ, source,
             "Nick Release: \2%ds\2  Bad PW Limit: %d  MAX CHAN REGS: %d",
              RELEASE_TIMEOUT, BAD_PW_LIMIT, CHANNEL_MAXREG);

    notice(s_OperServ,source,
             "Flood Levels: 4/%d/%d [LEV:TLEV:RESET]", F_TLEV, F_RESET);

    if (UPDATE_TIMEOUT > 86400)
        notice(s_OperServ, source,
             "DataBase Update Freq: \2%d\2 day%s, \2%02d:%02d\2",
             UPDATE_TIMEOUT/86400, (UPDATE_TIMEOUT/86400 == 1) ? "" : "s",
             (UPDATE_TIMEOUT/3600) % 24, (UPDATE_TIMEOUT/60) % 60);
    else if (UPDATE_TIMEOUT > 3600)
        notice(s_OperServ, source,
            "DataBase Update Freq: \2%d hour%s, %d minute%s\2",
             UPDATE_TIMEOUT/3600, UPDATE_TIMEOUT/3600==1 ? "" : "s",
             (UPDATE_TIMEOUT/60) % 60, (UPDATE_TIMEOUT/60)%60==1 ? "" : "s");
    else
        notice(s_OperServ, source,
             "DataBase Update Freq: \2%d minute%s, %d second%s\2",
              UPDATE_TIMEOUT/60, UPDATE_TIMEOUT/60==1 ? "" : "s",
              UPDATE_TIMEOUT%60, UPDATE_TIMEOUT%60==1 ? "" : "s");


    *buf = 0;
    if (allow_flood == 1) {
     if (*buf)
       strcat(buf, ", ");
      strcat(buf, "Flood Levels");
    }
    if (debug == 1) {
     if (*buf)
       strcat(buf, ", ");
      strcat(buf, "Debug");
    }
    if (clone_detect == 1) {
     if (*buf)
       strcat(buf, ", ");
      strcat(buf, "Clone Detection");
    }
    if (allow_ignore == 1) {
     if (*buf)
       strcat(buf, ", ");
      strcat(buf, "Ignore Code");
    }
    if (realname == 1) {
     if (*buf)
       strcat(buf, ", ");
      strcat(buf, "RNC");
    }
    if (readonly == 1) {
     if (*buf)
       strcat(buf, ", ");
      strcat(buf, "Read Only");
    }
    if (readonly == 0) {
     if (*buf)
       strcat(buf, ", ");
      strcat(buf, "R/W Mode");
    }
    if (!*buf)
      strcpy(buf, "None");
    notice(s_OperServ, source, "Options: %s\n", buf);

    if (ENABLE_LOGS == 1)
        notice(s_OperServ, source, "* Services are emailing logs");
    else
        notice(s_OperServ, source, "* Services are not emailing logs");
    if (ALLOW_EMAIL == 1)
        notice(s_OperServ, source, "* Services allow emailing memos");
    else
        notice(s_OperServ, source, "* Services do not allow emailing memos");


    notice(s_OperServ, source, "** NOTE: '0' - Disabled");
    notice(s_OperServ, source, "\2END OF SETTINGS\2");

}

/*************************************************************************/

/* STATS command. */

static void do_stats(const char *source)
{
    time_t uptime = time(NULL) - start_time;
    char *extra = strtok(NULL, "");
    int timeout = AKILL_DEF_EXPIRY+59;
    int i, qlines = 0, temp_time = 0;
    long count, mem, count2, mem2;
    struct tm tm;
    char buf[512];

    notice(s_OperServ, source, "Current users: \2%d\2 (\2%d\2 ops)  Max: %d",
                        usercnt, opcnt, maxusercnt);
    get_nickserv_stats(&count, &mem);
    get_chanserv_stats(&count2, &mem2);

    for (i = 0; i < 64; i++) {
        if (s_qlines[i])
           ++qlines;
    }
 
    notice(s_OperServ,source,
               "Nicks: \2%d\2  Chans: \2%d\2  Akills: \2%d\2 Jupes: \2%d\2  Qlines: \2%d\2",
                count,count2,nakill,njupe,qlines);

    notice(s_OperServ, source, "Services Commands: A: \2%d\2, D: \2%d\2, H: \2%d\2,  M: \2%d\2",
              serv_com, day_serv_com, hour_serv_com, min_serv_com);

    if (uptime > 86400)
        notice(s_OperServ, source,
                        "Services up \2%d\2 day%s, \2%02d:%02d\2",
                        uptime/86400, (uptime/86400 == 1) ? "" : "s",
                        (uptime/3600) % 24, (uptime/60) % 60);
    else if (uptime > 3600)
        notice(s_OperServ, source,
                        "Services up \2%d hour%s, %d minute%s\2",
                        uptime/3600, uptime/3600==1 ? "" : "s",
                        (uptime/60) % 60, (uptime/60)%60==1 ? "" : "s");
    else
        notice(s_OperServ, source,
                        "Services up \2%d minute%s, %d second%s\2",
                        uptime/60, uptime/60==1 ? "" : "s",
                        uptime%60, uptime%60==1 ? "" : "s");

    show_next_db(source, s_OperServ);

    notice(s_OperServ, source, "/MSG OPERSERV OHELP STATS - for more info");
    notice(s_OperServ, source, "\2END OF STATS\2");
}

void static do_uptime(const char *source)
{
  time_t uptime = time(NULL) - start_time;
  
    if (uptime > 86400)
        notice(s_OperServ, source,
                        "Services up \2%d\2 day%s, \2%02d:%02d\2",
                        uptime/86400, (uptime/86400 == 1) ? "" : "s",
                        (uptime/3600) % 24, (uptime/60) % 60);
    else if (uptime > 3600)
        notice(s_OperServ, source,
                        "Services up \2%d hour%s, %d minute%s\2",
                        uptime/3600, uptime/3600==1 ? "" : "s",
                        (uptime/60) % 60, (uptime/60)%60==1 ? "" : "s");
    else
        notice(s_OperServ, source,
                        "Services up \2%d minute%s, %d second%s\2",
                        uptime/60, uptime/60==1 ? "" : "s",
                        uptime%60, uptime%60==1 ? "" : "s");
}

/*************************************************************************/

static void do_version(const char *source)
{
#ifndef SKELETON
    notice(s_OperServ, source, "Version: %s", version_number2);
#else
    notice(s_OperServ, source, "Version: %s [SKELETON]", version_number2);
#endif                                       
    notice(s_OperServ, source, "Build: %s", version_build2);
    notice(s_OperServ, source, "DataBase Version: \2%d\2", FILE_VERSION);
    notice(s_OperServ, source, "Coder(s): \2Talinon\2");
    notice(s_OperServ, source, "SirvNET Services copyright (c) 1998-2001 Trevor Klingbeil.");
    notice(s_OperServ, source, "\2END OF VERSION\2");
}

/*************************************************************************/

static void do_mem(const char *source)
{
   long count, mem;
   int i;

      get_user_stats(&count, &mem);
      notice(s_OperServ, source,
              "User    : \2%6d\2 records, \2%5d\2 kB",
               count, (mem+512) / 1024);
      get_channel_stats(&count, &mem);
      notice(s_OperServ, source,
              "Channel : \2%6d\2 records, \2%5d\2 kB",
               count, (mem+512) / 1024);
      get_nickserv_stats(&count, &mem);
      notice(s_OperServ, source,
              "NickServ: \2%6d\2 records, \2%5d\2 kB",
               count, (mem+512) / 1024);
      get_chanserv_stats(&count, &mem);
      notice(s_OperServ, source,
              "ChanServ: \2%6d\2 records, \2%5d\2 kB",
               count, (mem+512) / 1024);
      get_memoserv_stats(&count, &mem);
      notice(s_OperServ, source,
              "MemoServ: \2%6d\2 records, \2%5d\2 kB",
               count, (mem+512) / 1024);
      count = 0;
      mem = sizeof(struct clone) * CLONE_DETECT_SIZE * 2;
      for (i = 0; i < CLONE_DETECT_SIZE; i++) {
          if (warnings[i].host) {
                count++;
                mem += strlen(warnings[i].host)+1;
          }
	}
	for (i = 0; i < MAX_SERVADMINS; i++) {
	    if (services_admins[i]) {
		count++;
		mem += strlen(services_admins[i])+1;
	    }
	}
        for (i = 0; i < MAX_SERVROOTS; i++) {
            if (services_roots[i]) {
                count++;
                mem += strlen(services_roots[i])+1;
            }
        }
        for (i = 0; i < MAX_SERVOPERS; i++) {
            if (services_opers[i]) {
                count++;
                mem += strlen(services_opers[i])+1;
            }
        }
	mem += sizeof(struct akill) * akill_size;
	count += nakill;
	for (i = 0; i < nakill; i++)
	    mem += strlen(akills[i].mask)+1 + strlen(akills[i].reason)+1;
	notice(s_OperServ, source,
			"OperServ: \2%6d\2 records, \2%5d\2 kB",
			count, (mem+512) / 1024);
    notice(s_OperServ, source, "\2END OF MEM STATS\2");
}

/*************************************************************************/

/* UINFO command */

static void do_uinfo(const char *source)
{
    const char *nick = strtok(NULL, " ");
    NickInfo *ni;
    time_t tbuff = time(NULL);
    struct u_chanlist *c, *c2;
    struct u_chaninfolist *l, *l2;
    char s[BUFSIZE], *end;
    int i, i2, temp_time;

#ifdef SKELETON
  notice(s_OperServ, source, "\2UINFO\2 N/A in SKELETON version");
  return;
#else

    if (!nick) {
        notice(s_OperServ, source, "Syntax: \2UINFO\2 <nick>");

    } else {
        User *u = finduser(nick);
	struct tm tm;
	char buf[512];

        if (!(ni = findnick(nick)))
            goto skiped;

        if (ni->flags & NI_VERBOTEN) {
           notice(s_OperServ, source,
                "Nick \2%s\2 is set as FORBIDDEN", nick);
           return;
        }

skiped:
     if (u)
        if (!(ni = findnick(nick)))
           notice(s_OperServ, source, " \2Status\2: Online [UnRegistered]");
        else
           notice(s_OperServ, source, " \2Status\2: Online [Registered]");


     if (!u) {
        if (!(ni = findnick(nick))) {
           notice(s_OperServ, source, " \2Status\2: Offline [UnRegistered]");
           return;
        } else {
           notice(s_OperServ, source, " \2Status\2: Offline [Registered]");
        }
     }

     if (ni)
        notice(s_OperServ, source,
               " \2%s\2 is \"%s\"\n", nick, ni->last_realname);

     if (!u)
       return;

     if (u) {

        if (u->chans == NULL)
             notice(s_OperServ, source,
                  " \2Channels\2: None");
        else {
            char buf[512];
            *buf = 0;
            c = u->chans;
            while (c) {
                c2 = c->next;
                if (*buf)
                    strcat(buf, ", ");
                strcat(buf, c->chan->name);
                c = c2;
        }
           notice(s_OperServ,source," \2Channels\2: %s", buf);
     }

    if (u) {
         char buf[512];
         *buf = 0;
         for (i=0; i < MAX_IDS; i++) {
            if (u->id_nicks[i]) {
                 if (*buf)
                     strcat(buf, ", ");
                 strcat(buf, u->id_nicks[i]);
            }
         }
         if (!(*buf))
            strcat(buf, "None");
         notice(s_OperServ,source," \2Id Nicks\2: %s", buf);
       }

    if (u) {
          char buf[512];
          *buf = 0;
          l = u->founder_chans;
          while (l) {
             l2 = l->next;
             if (*buf)
                  strcat(buf, ", ");
             strcat(buf, l->chan->name);
             l = l2;
             }
          if (u->founder_chans == NULL)
            strcat(buf, "None");
          notice(s_OperServ,source," \2Id Chans\2: %s", buf);

       }

     }

   if (u && ni && ni->accesscount)
       notice(s_OperServ, source,
          " \2Access List Count:\2 %d", ni->accesscount);
   else
       notice(s_OperServ, source,
          " \2Access List Count:\2 0");

   if (u) {
       temp_time = u->flood_time - tbuff;
       if (u->flood_time < tbuff) {
           u->floodlev = 0;
           temp_time = 0;
       }
       notice(s_OperServ, source,
           " \2Flood Level\2: %d/%d/%d motd(%d) / version(%d)",
              u->floodlev2, u->floodlev, temp_time, u->motd, u->version);
   }

   if (u && u->lastmemosend) {
       temp_time = time(NULL) - u->lastmemosend;

       if (temp_time > 3600) {
          temp_time /= 60;

           notice(s_OperServ, source,
              " \2Last Memo Send\2: %d hour%s %d minute%s",
              temp_time/60, temp_time/60==1 ? "" : "s",
              temp_time%60, temp_time%60==1 ? "" : "s");

       } else {

       notice(s_OperServ, source,
        " \2Last Memo Send\2: %d minute%s, %d second%s",
        temp_time/60, temp_time/60==1 ? "" : "s",
        temp_time%60, temp_time%60==1 ? "" : "s");
     }

   }
   if (u && u->lastnickreg) {
       temp_time = time(NULL) - u->lastnickreg;

       if (temp_time > 3600) {
          temp_time /= 60;

           notice(s_OperServ, source,
              " \2Last nick reg:\2: %d hour%s %d minute%s",
              temp_time/60, temp_time/60==1 ? "" : "s",
              temp_time%60, temp_time%60==1 ? "" : "s");

       } else {

       notice(s_OperServ, source,
        " \2Last nick reg\2: %d minute%s, %d second%s",
        temp_time/60, temp_time/60==1 ? "" : "s",
        temp_time%60, temp_time%60==1 ? "" : "s");
     }

   }

   if (u && u->invalid_pw_count) {
       temp_time = time(NULL) - u->invalid_pw_time;

       notice(s_OperServ, source,
            " \2Invalid pass count: %d\n", u->invalid_pw_count);

       if (temp_time > 3600) {
          temp_time /= 60;

           notice(s_OperServ, source,
              " \2Last Invalid Time\2: %d hour%s %d minute%s",
              temp_time/60, temp_time/60==1 ? "" : "s",
              temp_time%60, temp_time%60==1 ? "" : "s");

       } else {

       notice(s_OperServ, source,
        " \2Last Invalid Time\2: %d minute%s, %d second%s",
        temp_time/60, temp_time/60==1 ? "" : "s",
        temp_time%60, temp_time%60==1 ? "" : "s");
     }

   }

   if (u) {
     notice(s_OperServ, source,
           " \2Current Server\2: %s", u->server);

     temp_time = 0;
     temp_time = time(NULL) - u->signon;

     if (temp_time > 3600) {
        temp_time /= 60;

        notice(s_OperServ, source,
           " \2Online Time\2: %d hour%s %d minute%s",
           temp_time/60, temp_time/60==1 ? "" : "s",
           temp_time%60, temp_time%60==1 ? "" : "s");

     } else {

       notice(s_OperServ, source,
        " \2Online Time\2: %d minute%s, %d second%s",
        temp_time/60, temp_time/60==1 ? "" : "s",
        temp_time%60, temp_time%60==1 ? "" : "s");
     }

   }

   if (u && ni) {
      if (ni->last_usermask)
        notice(s_OperServ, source,
               " \2Last seen address\2: %s\n", ni->last_usermask);
	tm = *localtime(&ni->time_registered);
     }

    if (u && ni) {
      for (i = 0; i < MAX_SENDLOGS; i++) {

        if (sendlogs[i]) {
              if (stricmp(sendlogs[i], ni->nick) == 0)
                  notice(s_OperServ, source,
                     " * Has logs being e-mailed to %s", ni->email);
        }
     }
    }
    notice(s_OperServ, source, "\2END OF STATS\2");

  }
#endif
}

/*************************************************************************/

static void do_ignore(const char *source)
{
    char *cmd, *mask, *who, *expiry, *s;
    int i;

    cmd = strtok(NULL, " ");
    if (!cmd)
	cmd = "";

    if (stricmp(cmd, "LIST") == 0) {
	s = strtok(NULL, " ");
	if (!s)
	    s = "*";
	if (strchr(s, '@'))
	    strlower(strchr(s, '@'));
        notice(s_OperServ, source, "\2Current IGNORE list:\2");
        for (i = 0; i < nignore; i++) {
            if (match_wild(s, ignores[i].mask)) {
                char timebuf[32], expirebuf[64];
                struct tm tm;
                time_t t = time(NULL);

                tm = *localtime(ignores[i].time ? &ignores[i].time : &t);
                strftime(timebuf, sizeof(timebuf), "%d %b %Y", &tm);
                if (ignores[i].expires == 0) {
                    strcpy(expirebuf, "Does not expire");
                } else if (ignores[i].expires <= t) {
                    strcpy(expirebuf, "Expires at next database update");
                } else {
                    time_t t2 = ignores[i].expires - t;
                    t2 += 59;
                    if (t2 < 86400) {
                        t2 /= 60;
                        snprintf(expirebuf, sizeof(expirebuf),
                                "Expires in %d hour%s, %d minute%s",
                                t2/60, t2<120 ? "" : "s",
                                t2%60, t2%60==1 ? "" : "s");
                    } else {
                        t2 /= 86400;
                        snprintf(expirebuf, sizeof(expirebuf),
                                "Expires in %d day%s", t2, t2==1 ? "" : "s");
                    }


                }

                notice(s_OperServ, source, " %d) %-32s",
                                i+1,ignores[i].mask);
		notice(s_OperServ, source, "Set by %s on %s; %s",
                                ignores[i].who,timebuf,expirebuf);
         }
      }
      notice(s_OperServ,source,"\2End of List\2");


    } else if (!check_o_access(source, "IGNORE")) {
        notice(s_OperServ, source, "Permission denied.");

    } else if (stricmp(cmd, "ADD") == 0) {
	mask = strtok(NULL, " ");
        expiry = NULL;
        if (mask && !match_wild_nocase("*!*@*", mask)) {
             notice(s_OperServ, source,
                "Syntax: \2IGNORE ADD\2 <nick!user@host>");
             return;
        }
        if (mask) {
            if (!stricmp(mask,"*@*.*") || !stricmp(mask,"*@*") ||
                !stricmp(mask,"*@*.com") || !stricmp(mask,"*@*.net") ||
                !stricmp(mask,"*@*.org") || !stricmp(mask,"*@*.ca")) {
		notice(s_OperServ,source,"Hrmmm, What would your admin "
		    "think of that?");
                wallops(s_OperServ,"%s tried to IGNORE %s",
		    source,mask);
		return;
	    }
            for (i = 0; i < nignore; i++) {
                if (stricmp(ignores[i].mask,mask) == 0) {
                    notice(s_OperServ,source,"%s is already present "
                        "on the IGNORE list.",mask);
		    return;
		}
	    }
	    if (s = strchr(mask, '@')) {
		strlower(s);
	    } else {
		notice(s_OperServ, source,
			"Hostmask must contain an `@' character.");
		return;
	    }
            add_ignore(mask, source, expiry);
            notice(s_OperServ, source, "%s added to IGNORE list.", mask);
	    if (readonly) {
		notice(s_OperServ, source,
		    "\2Notice:\2 Changes will not be saved! "
		    " Services is in read-only mode.");
	    }
	} else {
	    notice(s_OperServ, source,
                        "Syntax: \2IGNORE ADD\2 <mask>");
	}

    } else if (stricmp(cmd, "TIME") == 0) {
	expiry = strtok(NULL, " ");
        mask = strtok(NULL, " ");
        if (mask && !match_wild_nocase("*!*@*", mask) || !expiry) {
             notice(s_OperServ, source,
                "Syntax: \2IGNORE TIME\2 <expiry> <nick!user@host>");
        }
        if (mask) {
            if (!stricmp(mask,"*@*.*") || !stricmp(mask,"*@*") ||
                !stricmp(mask,"*@*.com") || !stricmp(mask,"*@*.net") ||
                !stricmp(mask,"*@*.org") || !stricmp(mask,"*@*.ca")) {
                notice(s_OperServ,source,"Hrmmm, What would your admin "
                    "think of that?");
                wallops(s_OperServ,"%s tried to IGNORE %s",
                    source,mask);
                return;
            }
            for (i = 0; i < nignore; i++) {
                if (stricmp(ignores[i].mask,mask) == 0) {
                    notice(s_OperServ,source,"%s is already present "
                        "on the IGNORE list.",mask);
                    return;
                }
            }
            if (s = strchr(mask, '@')) {
                strlower(s);
            } else {
                notice(s_OperServ, source,
                        "Hostmask must contain an `@' character.");
                return;
            }
            add_ignore(mask, source, expiry);
            notice(s_OperServ, source, "%s added to IGNORE list.", mask);
            if (readonly) {
                notice(s_OperServ, source,
                    "\2Notice:\2 Changes will not be saved! "
                    " Services is in read-only mode.");
            }
        } else {
            notice(s_OperServ, source,
                        "Syntax: \2IGNORE TIME\2 <expiry> <mask>");
        }

    } else if (stricmp(cmd, "DEL") == 0) {
	int i;
	if (mask = strtok(NULL, " ")) {
	    if (strspn(mask, "1234567890") == strlen(mask) &&
                                (i = atoi(mask)) > 0 && i <= nignore) {

                strcpy(mask,ignores[i-1].mask);
	    }
	    if (s = strchr(mask, '@'))
		strlower(s);
            if (del_ignore(mask)) {
                notice(s_OperServ, source, "%s removed from IGNORE list.",mask);
		wallops(s_OperServ,
                        "%s removed %s from the IGNORE list.",
			source, mask);
		if (readonly) {
		    notice(s_OperServ, source,
			"\2Notice:\2 Changes will not be saved!  Services is in read-only mode.");
		}
	    } else {
                notice(s_OperServ, source, "%s not found on IGNORE list.", mask);
	    }
	} else {
            notice(s_OperServ, source, "Syntax: \2IGNORE DEL\2 <mask>");
        }
 
    } else {
	notice(s_OperServ, source,
                "Syntax: \2IGNORE\2 [ADD|DEL|LIST|TIME] [<expiry>] [<mask>]");
	notice(s_OperServ, source,
                "For help: \2/msg %s HELP IGNORE\2", s_OperServ);
    }
}



/*************************************************************************/

static void do_trigger(const char *source)
{
  char *mask, *tvalue, *cmd, *s;
  int i;

  cmd = strtok(NULL, " ");

    if (!cmd)
        cmd = "";

  if (stricmp(cmd, "LIST") == 0) {
        s = strtok(NULL, " ");
        if (!s)
            s = "*";
        notice(s_OperServ, source, "\2Current Trigger list:\2");
        for (i = 0; i < ntrigger; i++) {
            if (!tvalue || match_wild(s, trigger[i].mask)) {
                notice(s_OperServ, source, " %d) %-16s (%ld)",
                              i+1,trigger[i].mask, trigger[i].tvalue);

            }
        }
        notice(s_OperServ, source, "\2End Of List\2");
        return;

   } else {

     char *numtvalue = strtok(NULL," ");
     int ntvalue;

     if (!check_o_access(source, "TRIGGER")) {
         notice(s_OperServ, source, "Permission Denied");
         return;
     }


   if (!cmd || !numtvalue) {
     notice(s_OperServ, source,
         "Syntax: \2TRIGGER\2 <[user@]host> <value>");
     return;

   } else {
        int tval = strtol(numtvalue, (char **)&numtvalue, 10);

      if (!strchr(cmd, '@')) {
          notice(s_OperServ, source, "Mask must contain an @");

          return;
      }

      for (i = 0; i < ntrigger; i++) {
            if (stricmp(trigger[i].mask,cmd) == 0) {
                if ((tval <= 0) || (tval == CLONE_MIN_USERS)) {
                    notice(s_OperServ, source, "Trigger for %s reset", cmd);
                    wallops(s_OperServ,"%s reset Trigger for %s",source,cmd);
                    del_trigger(cmd);
                  return;
                } else if (trigger[i].tvalue == tval) {
                    notice(s_OperServ, source, "Value of %s is already "
                            "set to %d", trigger[i].mask, tval);
                    return;
                } else {
                    notice(s_OperServ, source, "Re-triggered %s to %d->%d",
                              cmd, trigger[i].tvalue, tval);
                    wallops(s_OperServ,"%s re-triggered %s to %d->%d",
                              source, cmd, trigger[i].tvalue, tval);
                    trigger[i].tvalue = tval;

                   return;
                }

            }
      }

       if (tval < 1) {
          notice(s_OperServ, source, "Value must be > 0");
          return;
       } else if (tval == CLONE_MIN_USERS) {
          notice(s_OperServ, source, "Trigger value of %d is by default", tval);          return;
       }
       if ((MAX_TRIGGERS != 0) && (ntrigger > MAX_TRIGGERS)) {
          notice(s_OperServ, source, "Trigger List Full");
          wallops(SERVER_NAME, "\2WARNING\2: Trigger list full! (%d)",
               MAX_TRIGGERS);
          return;
       }
       notice(s_OperServ, source, "Trigger for %s set to %d",
                 cmd, tval);
       wallops(s_OperServ,"%s triggered %s to %d",
                 source, cmd, tval);

       if (ntrigger >= trigger_size) {
           if (trigger_size < 8)
               trigger_size = 8;
           else
               trigger_size *= 2;
           trigger = srealloc(trigger, sizeof(*trigger) * trigger_size);
       }

       trigger[ntrigger].mask = sstrdup(cmd);
       trigger[ntrigger].tvalue = tval;
       ntrigger++;
    }
  }
}






/*************************************************************************/

#define NICK(nick,name) do { send_cmd(NULL, "NICK %s 1 %lu %s %s %s 0 :%s", (nick), time(NULL), services_user, services_host, server_name, (name)); } while (0)


/*************************************************************************/

static void do_sendlog(const char *source)
{
#ifdef SKELETON

   notice(s_OperServ, source, "SENDLOGS Disabled in SKELETON version");
   return;

#else

  NickInfo *ni = findnick(source);
  const char *cmd = strtok(NULL, " ");
  int i;


  if (cmd && stricmp(cmd, "list") == 0) {
      notice(s_OperServ, source, "\2SENDLOGS LIST:\2");
      for (i = 0; i < MAX_SENDLOGS; i++) {
         if (sendlogs[i]) {
            if ((ni = findnick(sendlogs[i])) && (ni->email))
                 notice(s_OperServ, source, "%s -> %s",
                     sendlogs[i], ni->email);
         }
      }
      notice(s_OperServ, source, "\2END OF LIST");
      return;
  }

// This should never happen, but lets make sure
     if (!ni)
        return;

     if (!ni->email) {
          notice(s_OperServ, source,
             "You need to set your NickServ email field!");
          return;
     }

     for (i = 0; i < MAX_SENDLOGS; i++) {
         if (sendlogs[i] && stricmp(ni->nick,sendlogs[i]) == 0) {
              notice(s_OperServ, source, "You have been removed from the mailing list");
              sendlogs[i] = NULL;
              if (snoop == 1)
                 send_cmd(s_OperServ, "PRIVMSG %s :!! Removed %s from SENDLOGS list", snoopchan, source);
              return;
          }
          if (!sendlogs[i])
                break;
     }

     if ((i < MAX_SENDLOGS) || (MAX_SENDLOGS == 0)) {
         sendlogs[i] = sstrdup(ni->nick);
         if (snoop == 1)
            send_cmd(s_OperServ, "PRIVMSG %s :!! Added %s to SENDLOGS List",snoopchan, source);
         notice(s_OperServ, source, "You have been added to the SENDLOGS list. "
               "Logs will be sent daily to: %s", ni->email);

     } else {
          notice(s_OperServ, source,
                "Too many entries (%d) on SENDLOGS list; cannot add more",
                     MAX_SENDLOGS);
          wallops(s_OperServ,
                 "\2Warning\2: SENDLOGS list is full! (%d)",
                    MAX_SENDLOGS);

     }

     if (readonly) {
           notice(s_OperServ, source,
              "\2Notice:\2 Changes will not be saved!  Services is in readonly mode.");
     }
#endif
}

/*************************************************************************/


static void do_zline(const char *source) {

  const char *cmd = strtok(NULL, " ");
  const char *host = strtok(NULL, " ");
  int i;

  if (!cmd || !host) {
     notice(s_OperServ, source,
         "Syntax: \2ZLINE\2 [ADD|DEL] <ip> [<reason>] ");
     return;

  } else if (stricmp(cmd, "ADD") == 0) {
      const char *reason = strtok(NULL, "");
      
      if (!reason) {
        notice(s_OperServ, source,
              "Syntax: \2ZLINE\2 [ADD|DEL] <ip> [<reason>] ");
        return;
      }

      for (i = 0; i < nzline; i++) {
         if (stricmp(host, zlines[i].host) == 0) {
             notice(s_OperServ, source, "%s is already on the ZLINE list",
                  host);
             return;
         }
      }


      if (nzline >= zline_size) {
          if (zline_size < 8)
              zline_size = 8;
          else
              zline_size *= 2;
          zlines = srealloc(zlines, sizeof(*zlines) * zline_size);
      }

      send_cmd(SERVER_NAME, "ZLINE %s :%s (%s)", host, reason, source);
      wallops(SERVER_NAME, "Adding Global Z:LINE %s by request of %s for: %s",
             host, source, reason);
      notice(s_OperServ, source, "Added global Z:LINE for %s", host);
      log("%s added global z:line %s because: %s", source, host, reason);

      zlines[nzline].host = sstrdup(host);
      zlines[nzline].reason = sstrdup(reason);
      zlines[nzline].who = sstrdup(source);
      nzline++;

  } else if (stricmp(cmd, "DEL") == 0) {

      for (i = 0; i < nzline; i++) {
          if (stricmp(host, zlines[i].host) == 0) {
               wallops(SERVER_NAME, "Global Z:LINE %s delete by request of %s",
                       host, source);
               free(zlines[i].host); free(zlines[i].reason);
               nzline--;
               if (i < nzline)
                   bcopy(zlines+i+1, zlines+i, sizeof(*zlines) * (nzline-i));
          }
      }
      send_cmd(SERVER_NAME, "UNZLINE %s", host);

  }

}

/*************************************************************************/

static void send_clone_lists(const char *source)
{
    User *u;
    int i;

    u = finduser(source);
    if (!u)
      return;

    notice(s_RootServ, u->nick, "warnings[]");
    for (i = 0; i < CLONE_DETECT_SIZE; i++) {
        if (warnings[i].host)
            notice(s_RootServ, u->nick, "    %10ld  %s", warnings[i].time, warnings[i].host ? warnings[i].host : "(null)");
    }
}

/*************************************************************************/

#ifdef SKELETON

static void do_identify(const char *source) {

   NickInfo *ni = findnick(source);
   const char *pass = strtok(NULL, " ");
   char access[128];


   if (!ni || !pass)
      return;


   if (stricmp(ni->pass, pass) == 0) {

      if (!(ni->flags & NI_IDENTIFIED))
          ni->flags |= NI_IDENTIFIED;

      
     send_cmd(s_OperServ, "PRIVMSG %s :You are now authorized to use %s",
            source, s_OperServ);
   }

}

#endif 
/*************************************************************************/

