/* Routines to maintain a list of online users.
 *
 * SirvNET Services is copyright (c) 1998-2001 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services(c) by Andy Church.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "../inc/services.h"

User *userlist = NULL;

unsigned int usercnt = 0, opcnt = 0, maxusercnt = 0;

/*************************************************************************/
/*************************************************************************/
/*************************************************************************/

/* Allocate a new User structure, fill in basic values, link it to the
 * overall list, and return it.  Always successful.
 */

static User *new_user(const char *nick)
{
    User *user;
    NickInfo *ni;

    user = scalloc(sizeof(User), 1);
    if (!nick)
	nick = "";
    strscpy(user->nick, nick, NICKMAX);
    user->floodlev = 0;
    user->next = userlist;
    if (userlist)
	userlist->prev = user;
    userlist = user;
    usercnt++;
    if (usercnt > maxusercnt)
	maxusercnt = usercnt;


#ifndef SKELETON
    ni = findnick(nick);
    if (CUSTOM_NOTICE)
        notice(s_GlobalNoticer,nick,CUSTOM_WELCOME);
    if (!ni && GENERIC_NOTICE)
          notice(s_GlobalNoticer,nick, WELCOME_NOTICE,NETWORK_NAME,nick,NETWORK_NAME,
               s_NickServ, s_ChanServ);
#endif
    return user;
}

/*************************************************************************/

/* Change the nickname of a user, and move pointers as necessary. */

static void change_user_nick(User *user, const char *nick)
{
    strscpy(user->nick, nick, NICKMAX);

}

/*************************************************************************/

/* Remove and free a User structure. */

void delete_user(User *user)
{

    struct u_chanlist *c, *c2;
    struct u_chaninfolist *ci, *ci2;
    int i;

    usercnt--;
    if (user->mode & UMODE_O)
	opcnt--;
#ifndef SKELETON
    check_del_memos(user);
    cancel_user(user);
#else
    check_temp_os(user);
#endif
    free(user->username);
    free(user->host);
    free(user->realname);
    free(user->server);

    c = user->chans;
    while (c) {
	c2 = c->next;
	chan_deluser(user, c->chan);
	free(c);
	c = c2;
    }

    ci = user->founder_chans;


    while (ci) {
	ci2 = ci->next;
	free(ci);
	ci = ci2;
    }

    for (i=0; user->id_nicks[i] && i < MAX_IDS; i++)
        free(user->id_nicks[i]);


    if (user->prev)
	user->prev->next = user->next;
    else
	userlist = user->next;
    if (user->next)
	user->next->prev = user->prev;
    free(user);

}

/*************************************************************************/
/*************************************************************************/

/* Return statistics.  Pointers are assumed to be valid. */

void get_user_stats(long *nusers, long *memuse)
{
    long count = 0, mem = 0;
    User *user;
    struct u_chanlist *uc;
    struct u_chaninfolist *uci;
    int i;

    for (user = userlist; user; user = user->next) {
	count++;
	mem += sizeof(*user);
	if (user->username)
	    mem += strlen(user->username)+1;
	if (user->host)
	    mem += strlen(user->host)+1;
	if (user->realname)
	    mem += strlen(user->realname)+1;
	if (user->server)
	    mem += strlen(user->server)+1;
	for (uc = user->chans; uc; uc = uc->next)
	    mem += sizeof(*uc);
	for (uci = user->founder_chans; uci; uci = uci->next)
	    mem += sizeof(*uci);
        for (i=0; user->id_nicks[i] && i < MAX_IDS; i++)
            mem += strlen(user->id_nicks[i]);
    }
    *nusers = count;
    *memuse = mem;
}

/*************************************************************************/

/* Checks realname against list provided, if matches wallops. */

static char *check_realname(User *user)
{

   char *realname;

   if (match_wild_nocase("admin", user->realname))
      sprintf(realname,"%s", user->realname);
   
   else if (match_wild_nocase("ircop", user->realname))
      sprintf(realname,"%s", user->realname);

   else if (match_wild_nocase("service", user->realname))
      sprintf(realname,"%s", user->realname);

   else
      sprintf(realname,"");

  return realname;
}

/*************************************************************************/

/* Find a user by nick.  Return NULL if user could not be found. */

User *finduser(const char *nick)
{
    User *user = userlist;
    while (user && stricmp(user->nick, nick) != 0)
	user = user->next;
    return user;
}

int is_on_id_list(const char *source, const char *nick)
{
    User *u = finduser(source);
    unsigned int i;
    if (!u)
	return 0;

    for (i=0; i < MAX_IDS && u->id_nicks[i]; i++) {
	if (stricmp(u->id_nicks[i], nick) == 0) {
	    return 1;
	}
    }

    return 0;
}

/*************************************************************************/

/* Return the first user in the user list. */

User *first_user(void)
{
    return userlist;
}

/*************************************************************************/
/*************************************************************************/

/* Handle a server NICK command.
 *	av[0] = nick
 *	If a new user:
 *		av[1] = hop count
 *		av[2] = signon time
 *		av[3] = username
 *		av[4] = hostname
 *		av[5] = user's server
 *		av[6] = user's real name
 *	Else:
 *		av[1] = time of change
 */

void do_nick(const char *source, int ac, char **av)
{
    char *uv[3];
    User *user;

    if (!*source) {
	/* This is a new user; create a User structure for it. */

	if (debug)
	    log("debug: new user: %s", av[0]);

	/* We used to ignore the ~ which a lot of ircd's use to indicate no
	 * identd response.  That caused channel bans to break, so now we
	 * just take what the server gives us.  People are _still_
	 * encouraged to read the RFCs and stop doing anything to usernames
	 * depending on the result of an identd lookup.
	 */

	/* First check for AKILLs. */
#ifndef BAHAMUT
	if (check_akill(av[0], av[3], av[4]))
	    return;
#else
        if (check_akill(av[0], av[4], av[5]))
            return;
#endif

	/* Allocate User structure and fill it in. */


	user = new_user(av[0]);
	user->signon = atol(av[2]);
#ifdef BAHAMUT
        uv[0] = sstrdup(user->nick);
        uv[1] = sstrdup(av[3]);
        do_umode(user->nick, 2, uv);
        free(uv[0]); free(uv[1]);
        user->username = sstrdup(av[4]);
        user->host = sstrdup(av[5]);
        user->server = sstrdup(av[6]);
        user->realname = sstrdup(av[8]);
#else
	user->username = sstrdup(av[3]);
	user->host = sstrdup(av[4]);
	user->server = sstrdup(av[5]);
	user->realname = sstrdup(av[6]);
#endif

	/* Check to see if it looks like clones. */
        if (clone_detect == 1)
           check_clones(user);
        /* Check to see if realname setting matches against warning list */

     if (realname == 1) {
        if ((match_wild_nocase("*admin*", user->realname))
        || (match_wild_nocase("*irc*op*", user->realname))
        || (match_wild_nocase("*service*", user->realname))
        || (match_wild_nocase("*oper*", user->realname)
        && (match_wild_nocase(NETWORK_NAME, user->realname)))
        && (stricmp(user->nick, "OperServ") != 0)
        && (stricmp(user->nick, "NickServ") != 0)
        && (stricmp(user->nick, "ChanServ") != 0)
        && (stricmp(user->nick, "HelpServ") != 0)
        && (stricmp(user->nick, "MemoServ") != 0)
        && (stricmp(user->nick, "AbuseServ") != 0)
        && (stricmp(user->nick, "RootServ") != 0)
        && (stricmp(user->nick, "OperServ2") != 0))

            wallops(s_OperServ,
                "\2Warning\2 - Real Name of \"\2%s\2\" found from %s!%s@%s",
                user->realname, user->nick, user->username, user->host);
     }


    } else {
	/* An old user changing nicks. */

        NickInfo *ni;
	user = finduser(source);
	if (!user) {
	    log("user: NICK from nonexistent nick %s: %s", source,
							merge_args(ac, av));
	    return;
	}
        ni = findnick(source);

        if (ni && (!(ni->flags & NI_VERBOTEN)) && 
            (((ni->flags & NI_RECOGNIZED) && !(ni->flags & NI_SECURE)) ||
            (ni->flags & NI_IDENTIFIED))) {
   	    ni->last_seen = time(NULL);
            if (ni->last_usermask)
                free(ni->last_usermask);
            ni->last_usermask
                 = smalloc(strlen(user->username)+strlen(user->host)+2);        
            sprintf(ni->last_usermask, "%s@%s", user->username, user->host);
        }

	if (debug)
	    log("debug: %s changed nick to %s", source, av[0]);
#ifndef SKELETON


	cancel_user(user);
#endif
	change_user_nick(user, av[0]);
	user->signon = atol(av[1]);

    }

    user->my_signon = time(NULL);

#ifndef SKELETON
    if (validate_user(user))
	check_memos(user->nick);
#endif
}

/*************************************************************************/

/* Handle a JOIN command.
 *	av[0] = channels to join
 */

void do_join(const char *source, int ac, char **av)
{
    User *user;
    char *s, *t;
    struct u_chanlist *c;

    user = finduser(source);
    if (!user) {
	log("user: JOIN from nonexistent user %s: %s", source,
							merge_args(ac, av));
	return;
    }
    t = av[0];
    while (*(s=t)) {
	t = s + strcspn(s, ",");
	if (*t)
	    *t++ = 0;
	if (debug)
	    log("debug: %s joins %s", source, s);
#ifndef SKELETON
        /* Make sure check_kick comes before chan_adduser, so banned users
	 * don't get to see things like channel keys. */
	if (check_kick(user, s))
	    continue;
        check_welcome(user, s);
#endif
	chan_adduser(user, s);
	c = smalloc(sizeof(*c));
	c->next = user->chans;
	c->prev = NULL;
	if (user->chans)
	    user->chans->prev = c;
	user->chans = c;
	c->chan = findchan(s);
    }
}

/*************************************************************************/

/* Handle a PART command.
 *	av[0] = channels to leave
 *	av[1] = reason (optional)
 */

void do_part(const char *source, int ac, char **av)
{
    User *user;
    char *s, *t;
    struct u_chanlist *c;

    user = finduser(source);
    if (!user) {
	log("user: PART from nonexistent user %s: %s", source,
							merge_args(ac, av));
	return;
    }
    t = av[0];
    while (*(s=t)) {
	t = s + strcspn(s, ",");
	if (*t)
	    *t++ = 0;
	if (debug)
	    log("debug: %s leaves %s", source, s);
	for (c = user->chans; c && stricmp(s, c->chan->name) != 0; c = c->next)
	    ;
	if (c) {
	    chan_deluser(user, c->chan);
	    if (c->next)
		c->next->prev = c->prev;
	    if (c->prev)
		c->prev->next = c->next;
	    else
		user->chans = c->next;
	    free(c);
	}
    }
}

/*************************************************************************/

/* Handle a KICK command.
 *	av[0] = channel
 *	av[1] = nick(s) being kicked
 *	av[2] = reason
 */

void do_kick(const char *source, int ac, char **av)
{
    User *user;
    char *s, *t;
    struct u_chanlist *c;
    FILE *f = fopen("../inc/version.h", "rb");

    t = av[1];
    while (*(s=t)) {
	t = s + strcspn(s, ",");
	if (*t)
	    *t++ = 0;
	user = finduser(s);
	if (!user) {
	    log("user: KICK for nonexistent user %s on %s: %s", s, av[0],
						merge_args(ac-2, av+2));
	    continue;
	}
	if (debug)
	    log("debug: kicking %s from %s", s, av[0]);
	for (c = user->chans; c && stricmp(av[0], c->chan->name) != 0;
								c = c->next)
	    ;
	if (c) {
	    chan_deluser(user, c->chan);
	    if (c->next)
		c->next->prev = c->prev;
	    if (c->prev)
		c->prev->next = c->next;
	    else
		user->chans = c->next;
	    free(c);
	}
    }
}

/*************************************************************************/

/* Handle a MODE command for a user.
 *	av[0] = nick to change mode for
 *	av[1] = modes
 */

void do_umode(const char *source, int ac, char **av)
{
    User *user;
    char *s;
    int add = 1;		/* 1 if adding modes, 0 if deleting */

    if (stricmp(source, av[0]) != 0) {
	log("user: MODE %s %s from different nick %s!", av[0], av[1], source);
	wallops(NULL, "%s attempted to change mode %s for %s",
		source, av[1], av[0]);
	return;
    }
    user = finduser(source);
    if (!user) {
	log("user: MODE %s for nonexistent nick %s: %s", av[1], source,
							merge_args(ac, av));
	return;
    }
    if (debug)
	log("debug: Changing mode for %s to %s", source, av[1]);
    s = av[1];
    while (*s) {
	switch (*s++) {
	    case '+': add = 1; break;
	    case '-': add = 0; break;
	    case 'i': add ? (user->mode |= UMODE_I) : (user->mode &= ~UMODE_I); break;
	    case 'w': add ? (user->mode |= UMODE_W) : (user->mode &= ~UMODE_W); break;
	    case 'g': add ? (user->mode |= UMODE_G) : (user->mode &= ~UMODE_G); break;
	    case 's': add ? (user->mode |= UMODE_S) : (user->mode &= ~UMODE_S); break;
	    case 'o':
		if (add) {
#ifndef SKELETON
                    NickInfo *ni = findnick(user->nick);
#endif

                    if (!(user->flags & U_NOOPER)) {
			user->mode |= UMODE_O;
			++opcnt;
#ifndef SKELETON
                        if (ni)
                           send_cmd(server_name, PRE_MOTD_ACCESS, source,
                              get_os_access(ni));
#endif
                        if (OPERMOTD != "") {
                            send_cmd(server_name,
                                 "372 %s :== Services Oper MOTD ==", source);
                            send_cmd(server_name,
                                 "372 %s :%s", source, OPERMOTD);
                        }
		    } else {
			send_cmd(s_OperServ, "PRIVMSG #Snoop :Failed oper for %s", user->nick);
			send_cmd(NULL, "SVSMODE %s -oaA", user->nick);
			notice(s_AbuseServ, user->nick, "Sorry, You have the Abusive user mode set, You cannot oper.");
		    }

		} else {
		    user->mode &= ~UMODE_O;
		    --opcnt;
		}
		break;
	}
    }
}

/*************************************************************************/

/* Handle a QUIT command.
 *	av[0] = reason
 */

void do_quit(const char *source, int ac, char **av)
{
    User *user;
    NickInfo *ni;

    if (check_servii_kill(source, ac, av))
        return;


    user = finduser(source);
    if (!user || !source)
       return;


    if (debug)
	log("debug: %s quits", source);

    ni = findnick(source);

#ifndef SKELETON
    if (ni && (!(ni->flags & NI_VERBOTEN)) && 
       (((ni->flags & NI_RECOGNIZED) && (!ni->flags & NI_SECURE)) ||
         (ni->flags & NI_IDENTIFIED))) {
	ni->last_seen = time(NULL);
        if (ni->last_usermask)
            free(ni->last_usermask);
        ni->last_usermask
                 = smalloc(strlen(user->username)+strlen(user->host)+2);        
        sprintf(ni->last_usermask, "%s@%s", user->username, user->host);
    }
#endif
    delete_user(user);
}

/*************************************************************************/

/* Handle a KILL command.
 *	av[0] = nick being killed
 *	av[1] = reason
 */

void do_kill(const char *source, int ac, char **av)
{
    User *user;
    struct u_chanlist *c, *c2;
    struct u_chaninfolist *ci, *ci2;
    NickInfo *ni;

    if (check_servii_kill(source, ac, av))
        return;

    user = finduser(av[0]);
    if (!user)
        return;

    if (debug)
	log("debug: %s killed", av[0]);
#ifndef SKELETON
    if ((ni = findnick(source)) && (!(ni->flags & NI_VERBOTEN)) && (ni->flags & (NI_IDENTIFIED | NI_RECOGNIZED)))
	ni->last_seen = time(NULL);
#endif
    delete_user(user);

}

/*************************************************************************/
/*************************************************************************/

/* Is the given nick an oper? */

int inline is_oper(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_O);
}

/*************************************************************************/

/* Is the given nick on the given channel? */

int is_on_chan(const char *nick, const char *chan)
{
    User *u = finduser(nick);
    struct u_chanlist *c;

    if (!u)
	return 0;
    for (c = u->chans; c; c = c->next) {
	if (stricmp(c->chan->name, chan) == 0)
	    return 1;
    }
    return 0;
}

/*************************************************************************/

/* Is the given nick a channel operator on the given channel? */

int is_chanop(const char *nick, const char *chan)
{
    Channel *c = findchan(chan);
    struct c_userlist *u;

    if (!c)
	return 0;

    for (u = c->chanops; u; u = u->next) {

	if (stricmp(u->user->nick, nick) == 0)
	    return 1;
    }
    return 0;
}

/*************************************************************************/

/* Is the given nick voiced (channel mode +v) on the given channel? */

int is_voiced(const char *nick, const char *chan)
{
    Channel *c = findchan(chan);
    struct c_userlist *u;

    if (!c)
	return 0;
    for (u = c->voices; u; u = u->next) {
	if (stricmp(u->user->nick, nick) == 0)
	    return 1;
    }
    return 0;
}

/*************************************************************************/
/*************************************************************************/

/* Does the user's usermask match the given mask (either nick!user@host or
 * just user@host)?
 */

int match_usermask(const char *mask, User *user)
{
    char *mask2 = sstrdup(mask);
    char *nick, *username, *host, *nick2, *host2;
    int result;

    if (strchr(mask2, '!')) {
	nick = strlower(strtok(mask2, "!"));
	username = strtok(NULL, "@");
    } else {
	nick = NULL;
	username = strtok(mask2, "@");
    }
    host = strtok(NULL, "");
    if (!username || !host) {
	free(mask2);
	return 0;
    }
    strlower(host);
    if (nick)
	nick2 = strlower(sstrdup(user->nick));
    host2 = strlower(sstrdup(user->host));
    result = (nick ? match_wild(nick, nick2) : 1) &&
		match_wild(username, user->username) &&
		match_wild(host, host2);
    free(mask2);
    if (nick)
	free(nick2);
    free(host2);
    return result;
}

/*************************************************************************/

/* Split a usermask up into its constitutent parts.  Returned strings are
 * malloc()'d, and should be free()'d when done with.  Returns "*" for
 * missing parts.
 */

void split_usermask(const char *mask, char **nick, char **user, char **host)
{
    char *mask2 = sstrdup(mask);

    *nick = strtok(mask2, "!");
    *user = strtok(NULL, "@");
    *host = strtok(NULL, "");
    /* Handle special case: mask == user@host */
    if (*nick && !*user && strchr(*nick, '@')) {
	*nick = NULL;
	*user = strtok(mask2, "@");
	*host = strtok(NULL, "");
    }
    if (!*nick)
	*nick = "*";
    if (!*user)
	*user = "*";
    if (!*host)
	*host = "*";
    *nick = sstrdup(*nick);
    *user = sstrdup(*user);
    *host = sstrdup(*host);
    free(mask2);
}

/*************************************************************************/

/* Given a user, return a mask that will most likely match any address the
 * user will have from that location.  For IP addresses, wildcards the
 * appropriate subnet mask (e.g. 35.1.1.1 -> 35.*; 128.2.1.1 -> 128.2.*);
 * for named addresses, wildcards the leftmost part of the name unless the
 * name only contains two parts.  If the username begins with a ~, delete
 * it.  The returned character string is malloc'd and should be free'd
 * when done with.
 */

char *create_mask(User *u)
{
    char *mask, *s, *end;

    /* Get us a buffer the size of the username plus hostname.  The result
     * will never be longer than this (and will often be shorter).
     */
    end = mask = smalloc(strlen(u->username) + strlen(u->host) + 2);
    end += sprintf(end, "%s@", u->username);
    if (strspn(u->host, "0123456789.") == strlen(u->host)
		&& (s = strchr(u->host, '.'))
		&& (s = strchr(s+1, '.'))
		&& (s = strchr(s+1, '.'))
		&& (   !strchr(s+1, '.'))) {	/* IP addr */
	s = sstrdup(u->host);
	*strrchr(s, '.') = 0;
	if (atoi(u->host) < 192)
	    *strrchr(s, '.') = 0;
	if (atoi(u->host) < 128)
	    *strrchr(s, '.') = 0;
	sprintf(end, "%s.*", s);
	free(s);
    } else {
	if ((s = strchr(u->host, '.')) && strchr(s+1, '.')) {
	    s = sstrdup(strchr(u->host, '.')-1);
	    *s = '*';
	} else {
	    s = sstrdup(u->host);
	}
	strcpy(end, s);
	free(s);
    }
    return mask;
}

void mass_check_akill()
{
    User *user;
    for (user = userlist; user; user = user->next) {
       if (check_akill(user->nick,user->username,user->host))
          return;           
    }
}

int check_akill_num(char *mask)
{
    User *user;
    int i=0;
    char *nick = strtok(mask, "!");
    char *ident = strtok(mask, "@");
    char *host = strtok(mask, "");

    for (user = userlist; user; user = user->next) {
        if (match_wild_nocase(ident, user->username)
          && match_wild_nocase(host, user->host))
             i += 1;

    }
  free(mask);
  return i;
}


/*************************************************************************/

int check_ignore_num(const char *mask)
{
    User *user;
    int i=0;
    char *mask2 = sstrdup(mask);
    char *nick = strtok(mask2, "!");
    char *ident2 = strtok(NULL, "!");
    char *ident = strtok(ident2, "@");
    char *host = strtok(NULL, "");

    for (user = userlist; user; user = user->next) {
        if (match_wild_nocase(nick, user->nick)
          && match_wild_nocase(ident, user->username)
          && match_wild_nocase(host, user->host))
             i += 1;
    }
    free(mask2);
//  if (i > 0)
//    i /= 2;
  return i;
}


