/* AbuseServ functions.
 *
 * SirvNET Services is copyright (c) 1998-2002 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Most of AbuseServ functions created by Eric Andresen.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "../inc/services.h"
#include "../inc/timeout.h"
#include "as-help.c"

#define TO_UNABUSE 2
#define TO_NOSERV 3
#define TO_NOOPER 4

const char s_AbuseServ[] = ABUSESERV_NAME;

char *abuse_team[MAX_ABUSE];

void load_at_db(void);
static void add_as_timeout(User *u, int type, time_t delay);
static void del_as_timeout(User *u, int type);
static void do_at(const char *source);
static void do_ohelp(const char *source);

void abuseserv(const char *source, char *buf)
{
    char *cmd;
    char *s, *chan, *nick, *host;
    User *u = finduser(source);

    slog("AS (%s!%s@%s) [%s]",
                source,u->username,u->host,buf);
    do_break_log("AS", "AS (%s!%s@%s) [%s]",
                source,u->username,u->host,buf);
    log("%s: %s: %s", s_AbuseServ, source, buf);

    cmd = strtok(buf, " ");

    if (!cmd) {
        return;
    } else if (stricmp(cmd, "OHELP") == 0) {
        do_ohelp(source);

    } else if (stricmp(cmd, "\1PING") == 0) {

        if (!(s = strtok(NULL, "")))
            s = "\1";
        notice(s_AbuseServ, source, "\1PING %s", s);

    } else if ((stricmp(cmd, "AT") == 0) && is_services_coder(source)) {
	do_at(source);

    } else if (stricmp(cmd, "SVSMODE") == 0) {
	notice(s_AbuseServ, source, "Sorry, Please use UNMODE. Usage: \2UNMODE \37nick\37 \37flags\37\2");

    } else if (stricmp(cmd, "UNMODE") == 0) {
        const char *nick = strtok(NULL, " ");
        const char *mode = strtok(NULL, " ");

	if (!nick || !mode)
	    return;

	send_cmd(NULL, "SVSMODE %s -%s", nick, mode);
        wallops(s_AbuseServ, "%s used SVSMODE: %s -%s",
		source, nick, mode);

    } else if (stricmp(cmd, "WARN") == 0) {
	const char *nick = strtok(NULL, " ");
        User *us;

        if (!nick)
	    return;
        else
           us = finduser(nick);

        if (!us)
           return;

        if (is_services_coder(nick)) {
           notice (s_AbuseServ, source, "%s is a services master", nick);
           return;
        }

	wallops(s_AbuseServ, "\2WARNING-\2 %s is abusing his/her privledges.", nick);
	notice(s_AbuseServ, nick, "You have been marked as abusing priveldges.");
	notice(s_AbuseServ, source, "Warning sent and user marked.");
	if (!is_services_coder(nick)) {
	    add_as_timeout(us, TO_UNABUSE, 60);
	    us->flags |= U_ABUSIVE;
	}

    } else if (stricmp(cmd, "NOSERV") == 0) {
	const char *nick = strtok(NULL, " ");
        const char *numdelay = strtok(NULL, " ");
        int delay;
        User *u;

        notice(s_AbuseServ, source, "Replaced by OS IGNORE");
        return;

        if (!nick)
            return;
        else
           u = finduser(nick);

        if (!u)
           return;

        if (!numdelay)
           return;

        if (is_services_coder(nick)) {
           notice (s_AbuseServ, source, "%s is a services master", nick);
           return;
        }

        delay = atoi(numdelay);

        wallops(s_AbuseServ, "%s has been denied services access for %dsecond(s).", nick, delay);
	notice(s_AbuseServ, nick, "You are now being denied services access for %dsecond(s).", delay);
	if (!is_services_coder(nick)) {
	    u->flags |= U_NOSERV;
            add_as_timeout(u, TO_NOSERV, delay);
	}

    } else if (stricmp(cmd, "NOOPER") == 0) {
        const char *nick = strtok(NULL, " ");
        const char *numdelay = strtok(NULL, " ");
        int delay;
        User *u;

        if (!nick)
            return;
        else
           u = finduser(nick);

        if (!u)
           return;

        if (!numdelay)
           return;

        if (is_services_coder(nick)) {
           notice (s_AbuseServ, source, "%s is a services master", nick);
           return;
        }

        delay = atoi(numdelay);

        wallops(s_AbuseServ, "\2%s\2 is being denied operator status for %dsecond(s).", nick, delay);
	notice(s_AbuseServ, nick, "You are now being denied operator status for %dsecond(s).", delay);
	if (!is_services_coder(nick)) {
            u->flags |= U_NOOPER;
            add_as_timeout(u, TO_NOOPER, delay);
	    send_cmd(NULL, "SVSMODE %s -oaA", nick);
	}

    } else if (stricmp(cmd, "UNDENY") == 0) {
	const char *nick = strtok(NULL, " ");


        if (!nick)
	    return;
        else
           u = finduser(nick);

        if (!u)
           return;

	if (u->flags & U_ABUSIVE)
	    u->flags &= ~U_ABUSIVE;
	if (u->flags & U_NOOPER)
	    u->flags &= ~U_NOOPER;
	if (u->flags & U_NOSERV)
	    u->flags &= ~U_NOSERV;
        wallops(s_AbuseServ, "%s has been unmarked as an Abuser granted back privledges [Temp]",nick);
        notice(s_AbuseServ, nick, "You have been given back your privledges");

    } else if (stricmp(cmd, "DENY") == 0) {
	const char *nick = strtok(NULL, " ");
        const char *numdelay = strtok(NULL, " ");
        int delay;
        User *u;
	
        if (!nick)
            return;
        else
           u = finduser(nick);

        if (!u)
           return;

        if (!numdelay)
           return;

        if (is_services_coder(nick)) {
           notice (s_AbuseServ, source, "%s is a services master", nick);
           return;
        }

        delay = atoi(numdelay);

	send_cmd(s_AbuseServ, "SVSMODE %s -oaAwfgsb", nick);
        wallops(s_AbuseServ, "%s has been marked as an Abuser and Denied services and Operator status. [Temp]",nick);
	notice(s_AbuseServ, nick, "You have been abusing your privledges and they have been stripped from you.");
	if (!is_services_coder(nick)) {
	    u->flags |= U_ABUSIVE;
            u->flags |= U_NOSERV;
            u->flags |= U_NOOPER;

	    add_as_timeout(u, TO_UNABUSE, delay);
            add_as_timeout(u, TO_NOSERV, delay);
            add_as_timeout(u, TO_NOOPER, delay);
	}

    } else {
        notice(s_AbuseServ, source,
                "Unrecognized command \2%s\2.", cmd);
    }
}

/*************************************************************************/

/* OHELP command. */

static void do_ohelp(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    if (!cmd)
        notice_list(s_AbuseServ, source, as_ohelp);
    else if (stricmp(cmd, "WARN") == 0)
        notice_list(s_AbuseServ, source, warn_help);
    else if (stricmp(cmd, "UNMODE") == 0)
        notice_list(s_AbuseServ, source, unmode_help);
    else if (stricmp(cmd, "NOOPER") == 0)
        notice_list(s_AbuseServ, source, nooper_help);
    else if (stricmp(cmd, "DENY") == 0)
        notice_list(s_AbuseServ, source, deny_help);
    else if (stricmp(cmd, "UNDENY") == 0)
        notice_list(s_AbuseServ, source, deny_help);
    else {
        notice(s_AbuseServ, source,
			"No help topic for \2%s\2. found", cmd);
        return;
    }
    notice(s_AbuseServ, source, "\2END OF OHELP\2");
}
/*************************************************************************/

static struct my_timeout {
    struct my_timeout *next, *prev;
    User *u;
    Timeout *to;
    int type;
} *my_timeouts;

static void rem_as_timeout(User *u, int type)
{
    struct my_timeout *t, *t2;

    t = my_timeouts;
    while (t) {
        if (t->u == u && t->type == type) {
            t2 = t->next;
            if (t->next)
                t->next->prev = t->prev;
            if (t->prev)
                t->prev->next = t->next;
            else
                my_timeouts = t->next;
            free(t);
            t = t2;
        } else {
            t = t->next;
        }
    }
}

static void timeout_unabuse(Timeout *t)
{
    User *u = t->data;
    User *u2 = finduser(u->nick);

    rem_as_timeout(u, TO_UNABUSE);
    if (!u2)
        return;
    if (u->flags & U_ABUSIVE)
        u->flags &= ~U_ABUSIVE;
}

static void timeout_noserv(Timeout *t)
{
    User *u = t->data;
    User *u2 = finduser(u->nick);

    rem_as_timeout(u, TO_NOSERV);
    if (!u2)
        return;

    if (u->flags & U_NOSERV)
        u->flags &= ~U_NOSERV;
}

static void timeout_nooper(Timeout *t)
{
    User *u = t->data;
    User *u2 = finduser(u->nick);

    rem_as_timeout(u, TO_NOOPER);
    if (!u2)
        return;

    if (u->flags & U_NOOPER)
        u->flags &= ~U_NOOPER;
}

static void add_as_timeout(User *u, int type, time_t delay)
{
    Timeout *to;
    struct my_timeout *t;
    void (*timeout_routine)(Timeout *);

    if (type == TO_UNABUSE)
        timeout_routine = timeout_unabuse;
    else if (type == TO_NOOPER)
	timeout_routine = timeout_nooper;
    else if (type == TO_NOSERV)
	timeout_routine = timeout_noserv;
    else {
        log("AbuseServ: unknown timeout type %d!  u=%p (%s), delay=%d",
                type, u, u->nick, delay);
        return;
    }
    to = add_timeout(delay, timeout_routine, 0);
    to->data = u;
    t = smalloc(sizeof(*t));
    t->next = my_timeouts;
    my_timeouts = t;
    t->prev = NULL;
    t->to = to;
    t->type = type;
}

static void del_as_timeout(User *u, int type)
{
    struct my_timeout *t, *t2;

    t = my_timeouts;
    while (t) {
        if (t->u == u && t->type == type) {
            t2 = t->next;
            if (t->next)
                t->next->prev = t->prev;
            if (t->prev)
                t->prev->next = t->next;
            else
                my_timeouts = t->next;
            del_timeout(t->to);
            free(t);
            t = t2;
        } else {
            t = t->next;
        }
    }
}

void load_at_db(void)
{
    FILE *f;
    int i;

    if (!(f = open_db(s_AbuseServ, AT_DB, "r")))
        return;
    switch (i = get_file_version(f, AT_DB)) {
      case 5:
      case 6:
      case 7:
      case 8:
        i = fgetc(f)<<8 | fgetc(f);
        while (--i >= 0)
            abuse_team[i] = read_string(f, AT_DB);
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, AT_DB);
    } /* switch (version) */
    close_db(f, AT_DB);
}

void save_at_db(void)
{
    FILE *f;
    int i, count = 0;

    if (!(f = open_db(s_AbuseServ, AT_DB, "w")))
        return;
    for (i = 0; i < MAX_ABUSE; i++) {
        if (abuse_team[i])
            count++;
    }
    fputc(count>>8, f);
    fputc(count & 255, f);
    for (i = 0; i < MAX_ABUSE; i++) {
        if (abuse_team[i])
            write_string(abuse_team[i], f, AT_DB);
    }
    close_db(f, AT_DB);
}

int is_abuse(const char *nick)
{
    NickInfo *ni;
    unsigned int i;

    if (is_services_coder(nick))
       return 1;
    for (i = 0; i < MAX_ABUSE; i++) {
        if (abuse_team[i] && is_on_id_list(nick, abuse_team[i]))
        {
            return 1;
        }
    }
    return 0;
}

int rec_abuse(const char *nick)
{
#ifndef SKELETON
    NickInfo *ni = findnick(nick);
    unsigned int i;

    if (is_abuse(nick) || rec_services_coder(nick))
       return 1;
    for (i = 0; i < MAX_ABUSE; i++) {
        if (abuse_team[i] && !stricmp(nick,abuse_team[i]))
        {
            if (ni->flags & NI_RECOGNIZED)
		return 1;
	    return 0;
        }
    }
    return 0;
#endif
}


static void do_at(const char *source)
{
    char *cmd, *nick;
    NickInfo *ni;
    unsigned int i;

    cmd = strtok(NULL, " ");
    if (!cmd)
        cmd = "";

    if (stricmp(cmd, "ADD") == 0) {
        if (!is_services_coder(source)) {
            notice(s_AbuseServ, source, "Permission denied.");
            return;
        }
        nick = strtok(NULL, " ");
        if (nick) {
#ifndef SKELETON
            if (!findnick(nick)) {
                notice(s_AbuseServ, source, "Nick %s isn't registered!", nick);
                return;
            }
#endif
            for (i = 0; i < MAX_ABUSE; i++) {
                if (abuse_team[i] && !stricmp(nick,abuse_team[i])) {
                    notice(s_AbuseServ, source, "%s is already a ATM",
                        nick);
                    return;
                }
                if (!abuse_team[i])
                    break;
            }
            if (i < MAX_ABUSE) {
                abuse_team[i] = sstrdup(nick);
                log("!! Added \2%s\2 to AT List", nick);
                do_break_log("AS", "!! Added %s to AT List", nick);
                wallops(s_AbuseServ, "\2%s\2 added \2%s\2 to the \2AT\2 List", source, nick);
                notice(s_AbuseServ, source,
                        "%s added to Abuse Team list.", nick);
            } else {
                notice(s_AbuseServ, source,
                        "Too many entries (%d) on abuse team list; cannot add",
                        MAX_ABUSE);
            }
            if (readonly) {
                notice(s_AbuseServ, source,
                    "\2Notice:\2 Changes will not be saved!  Services is in read-only mode.");
            }
        } else {
            notice(s_AbuseServ, source, "Syntax: \2AT ADD\2 <nick>");
        }

    } else if (stricmp(cmd, "DEL") == 0) {
        if (!is_services_coder(source)) {
            notice(s_AbuseServ, source, "Permission denied.");
            return;
        }
        if (nick = strtok(NULL, " ")) {
            for (i = 0; i < MAX_ABUSE; i++) {
                if (abuse_team[i] && stricmp(nick,abuse_team[i]) == 0)
                    break;
            }
            if (i < MAX_ABUSE) {
                free(abuse_team[i]);
                abuse_team[i] = NULL;
                log("!! Deleted \2%s\2 from AT list", nick);
                do_break_log("AS", "!! Deleted %s from AT list", nick);
                wallops(s_AbuseServ, "%s deleted %s from the AT List.", source, nick);
                notice(s_AbuseServ, source,
                        "%s removed from the abuse team list.", nick);
                if (readonly) {
                    notice(s_AbuseServ, source,
                            "\2Notice:\2 Changes will not be saved! Services is in readonly mode");
                }
            } else {
                notice(s_AbuseServ, source,
                        "%s not found on abuse team list.", nick);
            }
        } else {
            notice(s_AbuseServ, source, "Syntax: \2AT DEL\2 <nick>");
        }

    } else if (stricmp(cmd, "LIST") == 0) {
           notice(s_AbuseServ, source, "\2Abuse Team:\2");
           for (i = 0; i < MAX_ABUSE; i++) {
               if (abuse_team[i]) {
#ifndef SKELETON
                   if (ni = findnick(abuse_team[i]))
                      notice(s_AbuseServ, source, "%s (%s)", abuse_team[i], ni->last_usermask);
                   else
#endif
                      notice(s_AbuseServ, source, "%s", abuse_team[i]);
              }
	   }

        notice(s_AbuseServ, source, "\2End of List\2");

    } else {
        notice(s_AbuseServ, source,
                "Syntax: \2AT\2 [ADD|DEL|LIST] [<nick>]");
    }
}

