/* ChanServ functions.
 *
 * SirvNET Services is copyright (c) 1998-2002 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services(c) by Andy Church.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

/*************************************************************************/

#include "../inc/services.h"
#include "../inc/timeout.h"
#include "../inc/encrypt.h"

# define MODE_SENDER s_ChanServ

/*************************************************************************/

static ChannelInfo *chanlists[256];


static int def_levels[][2] = {
    { CA_AUTOOP,             5 },
    { CA_AUTOVOICE,          3 },
    { CA_AUTODEOP,          -1 },
    { CA_NOJOIN,            -1 },
    { CA_INVITE,             5 },
    { CA_AKICK,             10 },
    { CA_SET,               13 },
    { CA_CLEAR,              5 },
    { CA_UNBAN,              5 },
    { CA_OPDEOP,             5 },
    { CA_ACCESS_LIST,        3 },
    { CA_ACCESS_CHANGE,      1 },
    { -1 }
};

typedef struct {
    int what;
    char *name;
    char *desc;
} LevelInfo;

static LevelInfo levelinfo[] = {
    { CA_AUTOOP,        "AUTOOP",     "Automatic channel operator status" },
    { CA_AUTOVOICE,     "AUTOVOICE",  "Automatic mode +v" },
    { CA_AUTODEOP,      "AUTODEOP",   "Channel operator status disallowed" },
    { CA_NOJOIN,        "NOJOIN",     "Not allowed to join channel if RESTRICTED" },
    { CA_INVITE,        "INVITE",     "Allowed to use INVITE command" },
    { CA_AKICK,         "AKICK",      "Allowed to use AKICK command" },
    { CA_SET,           "SET",        "Allowed to use SET command (not FOUNDER/PASSWORD)" },
    { CA_CLEAR,         "CLEAR",      "Allowed to use CLEAR command" },
    { CA_UNBAN,         "UNBAN",      "Allowed to use UNBAN command" },
    { CA_OPDEOP,        "OPDEOP",     "Allowed to use OP/DEOP commands" },
    { CA_ACCESS_LIST,   "ACC-LIST",   "Allowed to view the access list" },
    { CA_ACCESS_CHANGE, "ACC-CHANGE", "Allowed to modify the access list" },
    { -1 }
};
static int levelinfo_maxwidth = 0;

/*************************************************************************/

/* Help messages for IRC operators. */

#include "cs-help.c"

/*************************************************************************/

inline void check_welcome(User *user, const char *chan);

static void alpha_insert_chan(ChannelInfo *ci);
static ChannelInfo *makechan(const char *chan);
static int delchan(ChannelInfo *ci);
static void reset_levels(ChannelInfo *ci);
static int is_founder(User *user, NickInfo *ni, ChannelInfo *ci, int set);
static int is_identified(User *user, ChannelInfo *ci);
void reset_all_levels();
static void do_ohelp(const char *source);
static void do_ostopic(const char *source);
static void do_help(const char *source);
static void do_register(const char *source);
static void do_identify(const char *source);
static void do_delete(const char *source);
static void do_drop(const char *source);
static void do_set(const char *source);
static void do_set_founder(User *u, ChannelInfo *ci);
static void do_set_password(User *u, ChannelInfo *ci, const char *param);
static void do_set_desc(User *u, ChannelInfo *ci, const char *param);
static void do_set_welcome(User *u, ChannelInfo *ci, const char *param);
static void do_set_url(User *u, ChannelInfo *ci, const char *param);
static void do_memolev(User *u, ChannelInfo *ci, const char *param);
static void do_set_email(User *u, ChannelInfo *ci, const char *param);
static void do_set_topic(User *u, ChannelInfo *ci, const char *param);
static void do_set_mlock(User *u, ChannelInfo *ci, char *param);
static void do_set_keeptopic(User *u, ChannelInfo *ci, const char *param);
static void do_set_topiclock(User *u, ChannelInfo *ci, const char *param);
static void do_set_private(User *u, ChannelInfo *ci, const char *param);
static void do_set_opguard(User *u, ChannelInfo *ci, const char *param);
static void do_set_restricted(User *u, ChannelInfo *ci, const char *param);
static void do_set_protect(User *u, ChannelInfo *ci, const char *param);
static void do_set_ident(User *u, ChannelInfo *ci, const char *param);
static void do_set_unsecure(User *u, ChannelInfo *ci, const char *param);
static void do_cfounder(const char *source);
static void do_avoice(const char *source);
static void do_sop(const char *source);
static void do_aop(const char *source);
static void do_akick(const char *source);
static void do_info(const char *source);
static void do_list(const char *source);
static void do_flist(const char *source);
static void do_invite(const char *source);
static void do_op(const char *source);
static void do_deop(const char *source);
static void do_unban(const char *source);
static void do_clear(const char *source);
static void do_remove(const char *source);
static void do_wipe(const char *source);
static void do_count(const char *source);
static void do_getpass(const char *source);
static void do_sendpass(const char *source);
static void do_forbid(const char *source);
static void do_freeze(const char *source);
static void do_suspend(const char *source);
static void do_why(const char *source);
static void do_level(const char *source);
static void do_unhold(const char *source, const char *chan);
static void do_unmark(const char *source, const char *chan);
static void do_unforbid(const char *source, const char *chan);
static void do_unfreeze(const char *source, const char *chan);
static void do_reopen(const char *source, const char *chan);
static char *get_sponser(User *u, ChannelInfo *ci);
static void flush_auth(const char *source);
/*************************************************************************/
/*************************************************************************/

#ifndef SKELETON

/* Display total number of registered channels and info about each; or, if
 * a specific channel is given, display information about that channel
 * (like /msg ChanServ INFO <channel>).  If count_only != 0, then only
 * display the number of registered channels (the channel parameter is
 * ignored).
 */

void listchans(int count_only, const char *chan)
{
    long count = 0;
    ChannelInfo *ci;
    int i;

    if (count_only) {

	for (i = 33; i < 256; ++i)
	    for (ci = chanlists[i]; ci; ci = ci->next)
		++count;
	printf("%d channels registered.\n", count);

    } else if (chan) {

	struct tm tm;
	NickInfo *ni;
	char *t;

	if (!(ci = cs_findchan(chan))) {
	    printf("Channel %s not registered.\n", chan);
	    return;
	}
	if (ni = findnick(ci->founder))
	    t = ni->last_usermask;
	else
	    t = NULL;
	if (ci->flags & CI_VERBOTEN) {
	    printf("Channel %s is FORBIDden.\n");
	} else {
	    printf("Information about channel %s:\n", ci->name);
	    printf("        Founder: %s%s%s%s\n",
			ci->founder, t ? " (" : "", t ? t : "", t ? ")" : "");
	    printf("    Description: %s\n", ci->desc);
	    tm = *localtime(&ci->time_registered);
	    printf("     Registered: %s %2d %02d:%02d:%02d '%d\n",
			month_name(tm.tm_mon+1), tm.tm_mday, tm.tm_hour,
			tm.tm_min, tm.tm_sec, tm.tm_year);
	    tm = *localtime(&ci->last_used);
	    printf("      Last used: %s %2d %02d:%02d:%02d '%d\n",
			month_name(tm.tm_mon+1), tm.tm_mday, tm.tm_hour,
			tm.tm_min, tm.tm_sec, tm.tm_year);
	    if (ci->last_topic) {
		printf("     Last topic: %s\n", ci->last_topic);
		printf("   Topic set by: %s\n", ci->last_topic_setter);
	    }
	    if (ci->url)
		printf("            URL: %s\n", ci->url);
	    if (ci->email)
		printf(" E-mail address: %s\n", ci->email);
	    printf("        Options: ");
	    if (!ci->flags) {
		printf("None\n");
	    } else {
		int need_comma = 0;
		static const char commastr[] = ", ";
		if (ci->flags & CI_PRIVATE) {
		    printf("Private");
		    need_comma = 1;
		}
		if (ci->flags & CI_KEEPTOPIC) {
		    printf("%sTopic Retention", need_comma ? commastr : "");
		    need_comma = 1;
		}
		if (ci->flags & CI_TOPICLOCK) {
		    printf("%sTopic Lock", need_comma ? commastr : "");
		    need_comma = 1;
		}
		if (ci->flags & CI_OPGUARD) {
		    printf("%sOp Guard", need_comma ? commastr : "");
		    need_comma = 1;
		}
		if (ci->flags & CI_RESTRICTED) {
		    printf("%sRestricted Access", need_comma ? commastr : "");
		    need_comma = 1;
		}
		if (ci->flags & CI_IDENT) {
		    printf("%sIdent", need_comma ? commastr : "");
		    need_comma = 1;
		}
                if (ci->flags & CI_HELDCHAN) {
                    printf("%sHeld", need_comma ? commastr : "");
                    need_comma = 1;
                }
                if (ci->flags & CI_MARKCHAN) {
                    printf("%sMarked", need_comma ? commastr : "");
                    need_comma = 1;
                }
	    }
	    printf("      Mode lock: ");
	    if (ci->mlock_on || ci->mlock_key || ci->mlock_limit) {
                printf("+%s%s%s%s%s%s%s%s%s%s%s%s",
			(ci->mlock_on & CMODE_I) ? "i" : "",
			(ci->mlock_key         ) ? "k" : "",
			(ci->mlock_limit       ) ? "l" : "",
			(ci->mlock_on & CMODE_M) ? "m" : "",
			(ci->mlock_on & CMODE_N) ? "n" : "",
			(ci->mlock_on & CMODE_P) ? "p" : "",
			(ci->mlock_on & CMODE_S) ? "s" : "",
			(ci->mlock_on & CMODE_T) ? "t" : "",
                        (ci->mlock_on & CMODE_J) ? "j" : "",
                        (ci->mlock_on & CMODE_r) ? "R" : "",
                        (ci->mlock_on & CMODE_C) ? "c" : "",
                        (ci->mlock_on & CMODE_o) ? "O" : "",
                        (ci->mlock_on & CMODE_m) ? "M" : "");

	     }
	    if (ci->mlock_off)
		printf("-%s%s%s%s%s%s%s%s%s%s%s%s",
			(ci->mlock_off & CMODE_I) ? "i" : "",
			(ci->mlock_off & CMODE_K) ? "k" : "",
			(ci->mlock_off & CMODE_L) ? "l" : "",
			(ci->mlock_off & CMODE_M) ? "m" : "",
			(ci->mlock_off & CMODE_N) ? "n" : "",
			(ci->mlock_off & CMODE_P) ? "p" : "",
			(ci->mlock_off & CMODE_S) ? "s" : "",
			(ci->mlock_off & CMODE_T) ? "t" : "",
                        (ci->mlock_off & CMODE_J) ? "j" : "",
                        (ci->mlock_off & CMODE_r) ? "R" : "",
                        (ci->mlock_off & CMODE_C) ? "c" : "",
                        (ci->mlock_off & CMODE_o) ? "O" : "",
                        (ci->mlock_off & CMODE_m) ? "M" : "");
	    if (ci->mlock_key)
		printf(" %s", ci->mlock_key);
	    if (ci->mlock_limit)
		printf(" %ld", ci->mlock_limit);
	    printf("\n");
	}

    } else {

	for (i = 33; i < 256; ++i) {
	    for (ci = chanlists[i]; ci; ci = ci->next) {
		printf("%-20s  %s\n", ci->name,
			ci->flags & CI_VERBOTEN ? "Disallowed (FORBID)"
			                        : ci->desc);
		++count;
	    }
	}
	printf("%d channels registered.\n", count);

    }
}

#endif	/* SKELETON */

/*************************************************************************/

/* Return information on memory use.  Assumes pointers are valid. */

void get_chanserv_stats(long *nrec, long *memuse)
{
    long count = 0, mem = 0;
    unsigned int i, j;
    ChannelInfo *ci;

    for (i = 0; i < 256; i++) {
	for (ci = chanlists[i]; ci; ci = ci->next) {
	    count++;
	    mem += sizeof(*ci);
	    if (ci->desc)
		mem += strlen(ci->desc)+1;
	    mem += ci->accesscount * sizeof(ChanAccess);
	    mem += ci->akickcount * sizeof(AutoKick);
	    for (j = 0; j < ci->akickcount; j++) {
		if (ci->akick[j].name)
		    mem += strlen(ci->akick[j].name)+1;
		if (ci->akick[j].reason)
		    mem += strlen(ci->akick[j].reason)+1;
	    }
	    if (ci->mlock_key)
		mem += strlen(ci->mlock_key)+1;
	    if (ci->last_topic)
		mem += strlen(ci->last_topic)+1;
	    if (ci->levels)
		mem += sizeof(*ci->levels) * CA_SIZE;
            if (ci->url)
                mem += strlen(ci->url)+1;
            if (ci->email)
                mem += strlen(ci->email)+1;
            if (ci->welcome)
                mem += strlen(ci->welcome)+1;
            if (ci->hold)
                mem += strlen(ci->hold)+1;
            if (ci->mark)
                mem += strlen(ci->mark)+1;
            if (ci->forbid)
                mem += strlen(ci->forbid)+1;
            if (ci->freeze)
                mem += strlen(ci->freeze)+1;
	}
    }
    *nrec = count;
    *memuse = mem;
}

/*************************************************************************/
/*************************************************************************/

/* Main ChanServ routine. */

void chanserv(const char *source, char *buf)
{
    char *cmd, *s;

    cmd = strtok(buf, " ");

    if (!cmd) {
	return;

    } else if (stricmp(cmd, "\1PING") == 0) {
	if (!(s = strtok(NULL, "")))
	    s = "\1";
	notice(s_ChanServ, source, "\1PING %s", s);

#ifdef SKELETON

    } else {
	notice(s_ChanServ, source, "%s is currently offline.", s_ChanServ);

    }

#else

    } else if (stricmp(cmd, "HELP") == 0) {
	do_help(source);

    } else if (stricmp(cmd, "REGISTER") == 0) {
	do_register(source);

    } else if (stricmp(cmd, "IDENTIFY") == 0) {
	do_identify(source);

    } else if (stricmp(cmd, "ID") == 0) {
	do_identify(source);

    } else if (stricmp(cmd, "DROP") == 0) {
	do_drop(source);

    } else if (stricmp(cmd, "SET") == 0) {
	do_set(source);

    } else if (stricmp(cmd, "CFOUNDER") == 0) {
        do_cfounder(source);

    } else if (stricmp(cmd, "CF") == 0) {
        do_cfounder(source);

    } else if (stricmp(cmd, "AVOICE") == 0) {
        do_avoice(source);

    } else if (stricmp(cmd, "SOP") == 0) {
        do_sop(source);

    } else if (stricmp(cmd, "AOP") == 0) {
	do_aop(source);

    } else if (stricmp(cmd, "AKICK") == 0) {
	do_akick(source);

    } else if (stricmp(cmd, "INVITE") == 0) {
	do_invite(source);

    } else if (stricmp(cmd, "UNBAN") == 0) {
	do_unban(source);

    } else if (stricmp(cmd, "INFO") == 0) {
	do_info(source);

    } else if (stricmp(cmd, "WHY") == 0) {
        do_why(source);

    } else if (stricmp(cmd, "LIST") == 0) {
	do_list(source);

    } else if (stricmp(cmd, "OP") == 0) {
	do_op(source);

    } else if (stricmp(cmd, "DEOP") == 0) {
	do_deop(source);

    } else if (stricmp(cmd, "REMOVE") == 0) {
            do_remove(source);

    } else if (stricmp(cmd, "CLEAR") == 0) {
	do_clear(source);

    } else if (stricmp(cmd, "COUNT") == 0) {
        do_count(source);

    } else if (stricmp(cmd, "SENDPASS") == 0) {
	do_sendpass(source);


    } else if (!is_oper(source)) {
        notice(s_ChanServ, source,
           "Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
           cmd, s_ChanServ);
        return;

    } else if (stricmp(cmd, "FLIST") == 0) {
	do_flist(source);

    } else if (stricmp(cmd, "OHELP") == 0) {
	do_ohelp(source);

    } else if (stricmp(cmd, "RACCESS") == 0) {
        notice(s_ChanServ, source,
               "RACCESS has been replaced with LEVEL");
        do_level(source);

    } else if ((!check_o_access(source, cmd)) &&
               (stricmp(cmd, "SUSPEND") != 0)) {
         notice(s_ChanServ, source, "Permission Denied");

    } else if (stricmp(cmd, "LEVEL") == 0) {
            do_level(source);

    } else if (stricmp(cmd, "SUSPEND") == 0) {
            do_suspend(source);

    } else if (stricmp(cmd, "FREEZE") == 0) {
            do_freeze(source);

    } else if (stricmp(cmd, "TOPIC") == 0) {
            do_ostopic(source);

    } else if (stricmp(cmd, "FORBID") == 0) {
            do_forbid(source);

    } else if (stricmp(cmd, "WIPE") == 0) {
            do_wipe(source);

    } else if (stricmp(cmd, "HOLD") == 0) {
	const char *chan = strtok(NULL, " ");
	ChannelInfo *ci;
	User *u = finduser(source);

	if ((!chan) || (stricmp(chan, "-") == 0)) {
            notice(s_ChanServ,source,"Syntax: \2HOLD\2 [-]<#channel>");
	    notice(s_ChanServ,source,"\2/msg %s OHELP HOLD\2 for more information.",s_ChanServ);
            return;
        }
        if (*chan == '-') {
           *chan++;
           do_unhold(source, chan);

	} else if (!(ci = cs_findchan(chan))) {
	    notice(s_ChanServ, source, "Channel %s is not registered.", chan);
        } else if (!(ci->flags & CI_HELDCHAN)) {
            ci->flags |= CI_HELDCHAN;
            if (ci->hold)
                  free(ci->hold);
            ci->hold = sstrdup(u->nick);
            wallops(s_ChanServ,"%s set the HELD flag for \2%s\2",u->nick,chan);
            slog("CS +H %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
            do_break_log("CS_F", "CS +H %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);

	} else {
	    notice(s_ChanServ,source,"Channel \2%s\2 is already held",chan);
	}

    } else if (stricmp(cmd, "MARK") == 0) {
        const char *chan = strtok(NULL, " ");
        ChannelInfo *ci;
        User *u = finduser(source);

        if ((!chan) || (stricmp(chan, "-") == 0)) {	
            notice(s_ChanServ,source,"Syntax: \2MARK\2 [-]<#channel>");
            notice(s_ChanServ,source,
		"\2/msg %s OHELP MARK\2 for more information",s_ChanServ);
            return;
        }

        if (*chan == '-') {
           *chan++;
           do_unmark(source, chan);

        } else if (!(ci = cs_findchan(chan))) {
            notice(s_ChanServ, source, "Channel %s is not registered.",chan);
        } else if (!(ci->flags & CI_MARKCHAN)) {
            ci->flags |= CI_MARKCHAN;
            if (ci->mark)
                free(ci->mark);
            ci->mark = sstrdup(u->nick);
            slog("CS +M %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
            do_break_log("CS_F", "CS +M %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
	    wallops(s_ChanServ,
                "%s set the MARK flag for \2%s\2",u->nick,chan);

	} else {
	    notice(s_ChanServ,source,"Channel \2%s\2 already marked.",chan);
	}

    } else if (stricmp(cmd, "GETPASS") == 0) {
	do_getpass(source);

    } else if (stricmp(cmd, "FLUSH") == 0) {
        flush_auth(source);

/*
    } else if (stricmp(cmd, "RESETALLLEVELS") == 0) {
        reset_all_levels();
	notice(s_ChanServ, source, "All levels reset.");
        log("All ChanServ access levels reset.");
*/

    } else if (stricmp(cmd, "DELETE") == 0) {
        do_delete(source);


    } else {
        User *u = finduser(source);
	notice(s_ChanServ, source,
		"Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
		cmd, s_ChanServ);
        slog("CS *C %s (%s@%s) [%s]",
            source, u->username, u->host, cmd);

    }


#endif	/* SKELETON */

}

/*************************************************************************/

#ifndef SKELETON


/* Load/save data files. */

void load_cs_dbase(void)
{
    FILE *f = fopen(CHANSERV_DB, "r");
    int ver, i, j, len;
    ChannelInfo *ci;

    if (!(f = open_db(s_ChanServ, CHANSERV_DB, "r")))
	return;

    switch (ver = get_file_version(f, CHANSERV_DB)) {
      case 5:
      case 6:
      case 7:
        for (i = 33; i < 256; ++i) {


            while (fgetc(f) == 1) {
               ci = smalloc(sizeof(ChannelInfo));
                if (1 != fread(ci, sizeof(ChannelInfo), 1, f))
                    fatal_perror("Read error on %s", CHANSERV_DB);


#ifdef USE_ENCRYPTION
                if (!(ci->flags & (CI_ENCRYPTEDPW | CI_VERBOTEN))) {
                    if (debug)
                        log("debug: %s: encrypting password for %s on load",
                                s_ChanServ, ci->name);
                    if (encrypt_in_place(ci->founderpass, PASSMAX) < 0)
                        fatal("%s: load database: Can't encrypt %s password!",
                                s_ChanServ, ci->name);
                    ci->flags |= CI_ENCRYPTEDPW;
                }
#else
                if (ci->flags & CI_ENCRYPTEDPW) {
                    /* Bail: it makes no sense to continue with encrypted
                     * passwords, since we won't be able to verify them */
                    fatal("%s: load database: password for %s encrypted "
                          "but encryption disabled, aborting",
                          s_ChanServ, ci->name);
                }
#endif
                /* Can't guarantee the file is in a particular order...
                 * (Well, we can, but we don't have to depend on it.) */
                alpha_insert_chan(ci);
                if (ver < 3) {
                    NickInfo *ni = findnick(ci->founder);
                    if (ni && ni->channelcount+1 > ni->channelcount)
                        ni->channelcount++;
                }
                ci->desc = read_string(f, CHANSERV_DB);
                if (ci->url)
                    ci->url = read_string(f, CHANSERV_DB);
                if (ci->email)
                    ci->email = read_string(f, CHANSERV_DB);
                if (ci->mlock_key)
                    ci->mlock_key = read_string(f, CHANSERV_DB);
                if (ci->last_topic)
                    ci->last_topic = read_string(f, CHANSERV_DB);
		if (ci->welcome)
		    ci->welcome = read_string(f, CHANSERV_DB);
                if (ci->hold)
                    ci->hold = read_string(f, CHANSERV_DB);
                if (ci->mark)
                    ci->mark = read_string(f, CHANSERV_DB);
                if (ci->freeze)
                    ci->freeze = read_string(f, CHANSERV_DB);
                if (ci->forbid)
                    ci->forbid = read_string(f, CHANSERV_DB);

                if (ci->accesscount) {
                    ChanAccess *access;
                    ChanAccessOld *oaccess;
                    access = smalloc(sizeof(ChanAccess) * ci->accesscount);
                    oaccess = smalloc(sizeof(ChanAccessOld) * ci->accesscount);
                    ci->access = access;


                    if (ci->accesscount != fread(oaccess, sizeof(ChanAccessOld),
                                                        ci->accesscount, f))
                        fatal_perror("Read error on ", CHANSERV_DB);


                    for (j = 0; j < ci->accesscount; ++j, ++access, ++oaccess) {
                        access->name = read_string(f, CHANSERV_DB);
                        access->sponser = sstrdup(" ");
                        access->level = oaccess->level;
                        access->is_nick = oaccess->is_nick;
                    }
                    j = 0; access = ci->access;
                    /* Clear out unused entries */
                    while (j < ci->accesscount) {
#ifdef COMPATIBILITY_V2
                        if (access->is_nick < 0 ||
                            (access->is_nick == 0 && !findnick(access->name))
                        ) {

#else
                        if (!access->in_use) {
#endif
                            --ci->accesscount;
                            free(access->name);
                            free(access->sponser);
                            if (j < ci->accesscount)
                                bcopy(access+1, access, sizeof(*access) *
                                                        (ci->accesscount - j));
                        } else {
#ifdef COMPATIBILITY_V2
                            access->is_nick = 1;
#endif
                            ++j; ++access;
                        }
                    }
                    if (ci->accesscount)
                        ci->access = srealloc(ci->access,
                                        sizeof(ChanAccess) * ci->accesscount);
                    else {
                        free(ci->access);
                        ci->access = NULL;
                    }

                } /* if (ci->accesscount) */

                if (ci->akickcount) {
                    AutoKick *akick;

                    akick = smalloc(sizeof(AutoKick) * ci->akickcount);
                    ci->akick = akick;
                    if (ci->akickcount !=
                            fread(akick, sizeof(AutoKick), ci->akickcount, f))
                        fatal_perror("Read error on %s", CHANSERV_DB);
                    for (j = 0; j < ci->akickcount; ++j, ++akick) {
                        akick->name = read_string(f, CHANSERV_DB);
                        if (akick->reason)
                            akick->reason = read_string(f, CHANSERV_DB);
                    }
                    j = 0; akick = ci->akick;
                    while (j < ci->akickcount) {
                        if (akick->is_nick < 0) {
                            --ci->akickcount;
                            free(akick->name);
                            if (akick->reason)
                                free(akick->reason);
                            if (j < ci->akickcount)
                                bcopy(akick+1, akick, sizeof(*akick) *
                                                        (ci->akickcount - j));
                        } else {
                            ++j; ++akick;
                        }
                    }
                    if (ci->akickcount) {
                        ci->akick = srealloc(ci->akick,
                                        sizeof(AutoKick) * ci->akickcount);
                    } else {
                        free(ci->akick);
                        ci->akick = NULL;
                    }
                } /* if (ci->akickcount) */
                if (ci->levels) {
                    int n_entries;
                    ci->levels = NULL;
                    reset_levels(ci);
                    n_entries = fgetc(f)<<8 | fgetc(f);
                    if (n_entries < 0)
                        fatal_perror("Read error on %s", CHANSERV_DB);
#ifdef COMPATIBILITY_V2
                    /* Ignore earlier, incompatible levels list */
                    if (n_entries == 6) {
                        fseek(f, sizeof(short) * n_entries, SEEK_CUR);
                    } else
#endif
                    if (n_entries <= CA_SIZE) {
                        fread(ci->levels, sizeof(short), n_entries, f);
                    } else {
                        fread(ci->levels, sizeof(short), CA_SIZE, f);
                        fseek(f, sizeof(short) * (n_entries - CA_SIZE),
                                                                SEEK_CUR);
                    }
                } else {
                    reset_levels(ci);
                }

            } /* while (fgetc(f) == 1) */

        } /* for (i) */

        save_cs_dbase();
        fatal_perror("successfully converted database - restart services");
        exit(0);

        break; /* case 1, etc. */


      case 8:
        for (i = 33; i < 256; ++i) {

            while (fgetc(f) == 1) {
               ci = smalloc(sizeof(ChannelInfo));
                if (1 != fread(ci, sizeof(ChannelInfo), 1, f))
                    fatal_perror("Read error on %s", CHANSERV_DB);


#ifdef USE_ENCRYPTION
                if (!(ci->flags & (CI_ENCRYPTEDPW | CI_VERBOTEN))) {
                    if (debug)
                        log("debug: %s: encrypting password for %s on load",
                                s_ChanServ, ci->name);
                    if (encrypt_in_place(ci->founderpass, PASSMAX) < 0)
                        fatal("%s: load database: Can't encrypt %s password!",
                                s_ChanServ, ci->name);
                    ci->flags |= CI_ENCRYPTEDPW;
                }
#else
                if (ci->flags & CI_ENCRYPTEDPW) {
                    /* Bail: it makes no sense to continue with encrypted
                     * passwords, since we won't be able to verify them */
                    fatal("%s: load database: password for %s encrypted "
                          "but encryption disabled, aborting",
                          s_ChanServ, ci->name);
                }
#endif
                /* Can't guarantee the file is in a particular order...
                 * (Well, we can, but we don't have to depend on it.) */
                alpha_insert_chan(ci);
                if (ver < 3) {
                    NickInfo *ni = findnick(ci->founder);
                    if (ni && ni->channelcount+1 > ni->channelcount)
                        ni->channelcount++;
                }
                ci->desc = read_string(f, CHANSERV_DB);
                if (ci->url)
                    ci->url = read_string(f, CHANSERV_DB);
                if (ci->email)
                    ci->email = read_string(f, CHANSERV_DB);
                if (ci->mlock_key)
                    ci->mlock_key = read_string(f, CHANSERV_DB);
                if (ci->last_topic)
                    ci->last_topic = read_string(f, CHANSERV_DB);
		if (ci->welcome)
		    ci->welcome = read_string(f, CHANSERV_DB);
                if (ci->hold)
                    ci->hold = read_string(f, CHANSERV_DB);
                if (ci->mark)
                    ci->mark = read_string(f, CHANSERV_DB);
                if (ci->freeze)
                    ci->freeze = read_string(f, CHANSERV_DB);
                if (ci->forbid)
                    ci->forbid = read_string(f, CHANSERV_DB);

                if (ci->accesscount) {
                    ChanAccess *access;
                    access = smalloc(sizeof(ChanAccess) * ci->accesscount);
                    ci->access = access;


                    if (ci->accesscount != fread(access, sizeof(ChanAccess),
                                                        ci->accesscount, f))
                        fatal_perror("Read error on ", CHANSERV_DB);


                    for (j = 0; j < ci->accesscount; ++j, ++access) {
                        access->name = read_string(f, CHANSERV_DB);
                        access->sponser = read_string(f, CHANSERV_DB);
                    }
                    j = 0; access = ci->access;
                    /* Clear out unused entries */
                    while (j < ci->accesscount) {
#ifdef COMPATIBILITY_V2
                        if (access->is_nick < 0 ||
                            (access->is_nick == 0 && !findnick(access->name))
                        ) {

#else
                        if (!access->in_use) {
#endif
                            --ci->accesscount;
                            free(access->name);
                            if (access->sponser)
                                 free(access->sponser);
                            if (j < ci->accesscount)
                                bcopy(access+1, access, sizeof(*access) *
                                                        (ci->accesscount - j));
                        } else {
#ifdef COMPATIBILITY_V2
                            access->is_nick = 1;
#endif
                            ++j; ++access;
                        }
                    }
                    if (ci->accesscount)
                        ci->access = srealloc(ci->access,
                                        sizeof(ChanAccess) * ci->accesscount);
                    else {
                        free(ci->access);
                        ci->access = NULL;
                    }

                } /* if (ci->accesscount) */

                if (ci->akickcount) {
                    AutoKick *akick;

                    akick = smalloc(sizeof(AutoKick) * ci->akickcount);
                    ci->akick = akick;
                    if (ci->akickcount !=
                            fread(akick, sizeof(AutoKick), ci->akickcount, f))
                        fatal_perror("Read error on %s", CHANSERV_DB);
                    for (j = 0; j < ci->akickcount; ++j, ++akick) {
                        akick->name = read_string(f, CHANSERV_DB);
                        if (akick->reason)
                            akick->reason = read_string(f, CHANSERV_DB);
                    }
                    j = 0; akick = ci->akick;
                    while (j < ci->akickcount) {
                        if (akick->is_nick < 0) {
                            --ci->akickcount;
                            free(akick->name);
                            if (akick->reason)
                                free(akick->reason);
                            if (j < ci->akickcount)
                                bcopy(akick+1, akick, sizeof(*akick) *
                                                        (ci->akickcount - j));
                        } else {
                            ++j; ++akick;
                        }
                    }
                    if (ci->akickcount) {
                        ci->akick = srealloc(ci->akick,
                                        sizeof(AutoKick) * ci->akickcount);
                    } else {
                        free(ci->akick);
                        ci->akick = NULL;
                    }
                } /* if (ci->akickcount) */
                if (ci->levels) {
                    int n_entries;
                    ci->levels = NULL;
                    reset_levels(ci);
                    n_entries = fgetc(f)<<8 | fgetc(f);
                    if (n_entries < 0)
                        fatal_perror("Read error on %s", CHANSERV_DB);
#ifdef COMPATIBILITY_V2
                    /* Ignore earlier, incompatible levels list */
                    if (n_entries == 6) {
                        fseek(f, sizeof(short) * n_entries, SEEK_CUR);
                    } else
#endif
                    if (n_entries <= CA_SIZE) {
                        fread(ci->levels, sizeof(short), n_entries, f);
                    } else {
                        fread(ci->levels, sizeof(short), CA_SIZE, f);
                        fseek(f, sizeof(short) * (n_entries - CA_SIZE),
                                                                SEEK_CUR);
                    }
                } else {
                    reset_levels(ci);
                }

            } /* while (fgetc(f) == 1) */

        } /* for (i) */

        break; /* case 1, etc. */


      default:
	fatal("Unsupported version number (%d) on %s", i, CHANSERV_DB);

    } /* switch (version) */

    close_db(f, CHANSERV_DB);
}

/*************************************************************************/

void save_cs_dbase(void)
{
    FILE *f;
    int i, j, len;
    ChannelInfo *ci;

    if (!(f = open_db(s_ChanServ, CHANSERV_DB, "w")))
	return;

    for (i = 33; i < 256; ++i) {
	for (ci = chanlists[i]; ci; ci = ci->next) {
	    fputc(1, f);
	    if (1 != fwrite(ci, sizeof(ChannelInfo), 1, f))
		fatal_perror("Write error on %s", CHANSERV_DB);
	    write_string(ci->desc ? ci->desc : "", f, CHANSERV_DB);
	    if (ci->url)
		write_string(ci->url, f, CHANSERV_DB);
	    if (ci->email)
		write_string(ci->email, f, CHANSERV_DB);
	    if (ci->mlock_key)
		write_string(ci->mlock_key, f, CHANSERV_DB);
	    if (ci->last_topic)
		write_string(ci->last_topic, f, CHANSERV_DB);
	    if (ci->welcome)
		write_string(ci->welcome, f, CHANSERV_DB);
            if (ci->hold)
                write_string(ci->hold, f, CHANSERV_DB);
            if (ci->mark)
                write_string(ci->mark, f, CHANSERV_DB);
            if (ci->freeze)
                write_string(ci->freeze, f, CHANSERV_DB);
            if (ci->forbid)
                write_string(ci->forbid, f, CHANSERV_DB);

	    if (ci->accesscount) {
		ChanAccess *access = ci->access;
		if (ci->accesscount !=
			fwrite(access, sizeof(ChanAccess), ci->accesscount, f))
		    fatal_perror("Write error on %s", CHANSERV_DB);
		for (j = 0; j < ci->accesscount; ++j, ++access) {
		    if (access->in_use)
			write_string(access->name, f, CHANSERV_DB);
		    else
			write_string("", f, CHANSERV_DB);
                    write_string(access->sponser, f, CHANSERV_DB);
		}
	    }

	    if (ci->akickcount) {
		AutoKick *akick = ci->akick;
		if (ci->akickcount !=
			fwrite(akick, sizeof(AutoKick), ci->akickcount, f))
		    fatal_perror("Write error on %s", CHANSERV_DB);
		for (j = 0; j < ci->akickcount; ++j, ++akick) {
		    if (akick->is_nick >= 0) {
			write_string(akick->name, f, CHANSERV_DB);
			if (akick->reason)
			    write_string(akick->reason, f, CHANSERV_DB);
		    } else {
			write_string("", f, CHANSERV_DB);
			if (akick->reason)
			    write_string("", f, CHANSERV_DB);
		    }
		}
	    }

	    if (ci->levels) {
		fputc(CA_SIZE >> 8, f);
		fputc(CA_SIZE & 0xFF, f);
		fwrite(ci->levels, sizeof(short), CA_SIZE, f);
	    }

	} /* for (chanlists[i]) */

	fputc(0, f);

    } /* for (i) */

    close_db(f, CHANSERV_DB);
}

/*************************************************************************/

/* Check the current modes on a channel; if they conflict with a mode lock,
 * fix them. */

void check_modes(const char *chan)
{
    Channel *c = findchan(chan);
    ChannelInfo *ci = cs_findchan(chan);
    char newmodes[32], *newkey = NULL;
    long newlimit = 0;
    char *end = newmodes;
    int modes;
    int set_limit = 0, set_key = 0;

    if (!c || !ci)
	return;

    if (ci->flags & CI_FREEZECHAN)
        return;

    *end++ = '+';
    modes = ~c->mode & ci->mlock_on;

  if (ENABLE_R_MODES || ENABLE_R_MODES == 2) {
     if (modes & CMODE_R) {
        c->mode |= CMODE_R;
        *end++ = 'r';
     }
  }
#if BAH_V >= 147
    if (modes & CMODE_o) {
        *end++ = 'O';
        c->mode |= CMODE_o;
    }
    if (modes & CMODE_m) {
        *end++ = 'M';
        c->mode |= CMODE_m;
    }
#endif
    if (modes & CMODE_r) {
        *end++ = 'R';
	c->mode |= CMODE_r;
    }
    if (modes & CMODE_I) {
	*end++ = 'i';
	c->mode |= CMODE_I;
    }
    if (modes & CMODE_M) {
	*end++ = 'm';
	c->mode |= CMODE_M;
    }
    if (modes & CMODE_N) {
	*end++ = 'n';
	c->mode |= CMODE_N;
    }
    if (modes & CMODE_P) {
	*end++ = 'p';
	c->mode |= CMODE_P;
    }
    if (modes & CMODE_S) {
	*end++ = 's';
	c->mode |= CMODE_S;
    }
    if (modes & CMODE_T) {
	*end++ = 't';
	c->mode |= CMODE_T;
    }

    if (modes & CMODE_C) {
	*end++ = 'c';
	c->mode |= CMODE_C;

    }
    if (modes & CMODE_J) {
	*end++ = 'j';
	c->mode |= CMODE_J;
    }

    if (ci->mlock_limit && (ci->mlock_limit != c->limit)) {
	*end++ = 'l';
	newlimit = ci->mlock_limit;
	c->limit = newlimit;
	set_limit = 1;
    }
    if (!c->key && ci->mlock_key) {
	*end++ = 'k';
	newkey = ci->mlock_key;
	c->key = sstrdup(newkey);
	set_key = 1;
    } else if (c->key && ci->mlock_key && stricmp(c->key, ci->mlock_key) != 0) {
	char *av[3];
	send_cmd(MODE_SENDER, "MODE %s -k %s", c->name, c->key);
	free(c->key);
	*end++ = 'k';
	newkey = ci->mlock_key;
	c->key = sstrdup(newkey);
	set_key = 1;
    }

    if (end[-1] == '+')
	--end;

    *end++ = '-';
    modes = c->mode & ci->mlock_off;
    if (modes & CMODE_r) {
        *end++ = 'R';
        c->mode &= ~CMODE_r;
    }
#if BAH_V >= 147
    if (modes & CMODE_o) {
        *end++ = 'O';
        c->mode &= ~CMODE_o;
    }
    if (modes & CMODE_m) {
        *end++ = 'M';
        c->mode &= ~CMODE_m;
    }
#endif
    if (modes & CMODE_I) {
	*end++ = 'i';
	c->mode &= ~CMODE_I;
    }
    if (modes & CMODE_M) {
	*end++ = 'm';
	c->mode &= ~CMODE_M;
    }
    if (modes & CMODE_N) {
	*end++ = 'n';
	c->mode &= ~CMODE_N;
    }
    if (modes & CMODE_P) {
	*end++ = 'p';
	c->mode &= ~CMODE_P;
    }
    if (modes & CMODE_S) {
	*end++ = 's';
	c->mode &= ~CMODE_S;
    }
    if (modes & CMODE_T) {
	*end++ = 't';
	c->mode &= ~CMODE_T;
    }
    if (modes & CMODE_C) {
	*end++ = 'c';
	c->mode &= ~CMODE_C;
    }
    if (modes & CMODE_J) {
        *end++ = 'j';
        c->mode &= ~CMODE_J;
    }
    if (c->key && (ci->mlock_off & CMODE_K)) {
	*end++ = 'k';
	newkey = sstrdup(c->key);
	free(c->key);
	c->key = NULL;
	set_key = 1;
    }
    if (c->limit && (ci->mlock_off & CMODE_L)) {
	*end++ = 'l';
	c->limit = 0;
    }

    if (end[-1] == '-')
	--end;

    if (end == newmodes)
	return;
    *end = 0;

/*  Well it looks liked they didn't make ulined clients able to set
 *  cmode +O, so we'll fix that now.. 
 */

#if BAH_V >= 147
       if (strchr(newmodes, 'O'))
          send_cmd(s_ChanServ, "MODE %s +o", s_ChanServ);
#endif


    if (set_limit) {
	if (set_key)
	    send_cmd(MODE_SENDER, "MODE %s %s %d :%s", c->name,
				newmodes, newlimit, newkey ? newkey : "");
	else
	    send_cmd(MODE_SENDER, "MODE %s %s %d", c->name, newmodes, newlimit);
    } else if (set_key)
	send_cmd(MODE_SENDER, "MODE %s %s :%s", c->name,
				newmodes, newkey ? newkey : "");
    else
	send_cmd(MODE_SENDER, "MODE %s %s", c->name, newmodes);

    if (newkey && !c->key)
	free(newkey);

#if BAH_V >= 147
    if (strchr(newmodes, 'O'))
       send_cmd(s_ChanServ, "MODE %s -o", s_ChanServ);
#endif
}

/*************************************************************************/

/* Check whether a user is allowed to be opped on a channel; if they
 * aren't, deop them.  If serverop is 1, the +o was done by a server.
 * Return 1 if the user is allowed to be opped, 0 otherwise. */

int check_valid_op(User *user, const char *chan, int serverop)
{
    ChannelInfo *ci = cs_findchan(chan);
    NickInfo *ni;
    int root = 0;

    ni = findnick(user->nick);

    if (!ci)
      return 1;

    if (!ni || (ni && ni->flags & NI_NEVEROP)) {
       if (ci->flags & CI_LEAVEOPS)
           return 1;
    }

/* We used to have a different return for roots, but this way works better because
   of updating channel's "last used" fields. */

    if (rec_services_root(user->nick))
       root = 1;

    if (ci->flags & (CI_VERBOTEN | CI_CLOSED) && !root)
         return 0;
     /* check_kick() will get them out; we needn't explain. */

    if (ci->flags & CI_FREEZECHAN && !root)
        return 0;

    if (ni && (ni->flags & NI_NEVEROP) && !serverop)
        return 0;

    if ((!check_access(user, ci, CA_AUTOOP)) &&
           (!(ci->flags & CI_OPGUARD)) && !serverop)
       return 1;


    if (ci->flags & CI_OPGUARD) {
         if (!check_opguard(user, chan))
            return 0;
         else
            return 1;
    }

    if (!check_access(user, ci, CA_AUTOOP))
        return 0;

/* all else fails, let the user keep the ops */

    return 1;
}


/*************************************************************************/

inline void check_welcome(User *user, const char *chan)
{
    ChannelInfo *ci = cs_findchan(chan);

    if (ci && ci->welcome)
        notice(s_ChanServ, user->nick, "%s Welcome: %s", ci->name, ci->welcome);

}

/*************************************************************************/

/* Check whether a user should be opped on a channel, and if so, do it.
 * Return 1 if the user was opped, 0 otherwise.  (Updates the channel's
 * last used time if the user was opped.) */

int check_should_op(User *user, const char *chan)
{
    ChannelInfo *ci = cs_findchan(chan);
    NickInfo *ni;

    if (ci && ci->flags & CI_FREEZECHAN)
        return 0;

    
    if (!ci || (ci->flags & CI_VERBOTEN) || !ci->levels[CA_AUTOOP] ||
		*chan == '+')
	return 0;

    ni = findnick(user->nick);
    if (ci->flags & CI_IDENT) {
        if (!ni || !(ni->flags & NI_IDENTIFIED))
	    return 0;
    }
    

    if (!ni || !(ni->flags & NI_NEVEROP)) {
       if (check_access(user, ci, CA_AUTOOP)) {
           send_cmd(MODE_SENDER, "MODE %s +o %s", chan, user->nick);
           ci->last_used = time(NULL);
           if (ci->flags & CI_REMIND)
              ci->flags &= ~CI_REMIND;
           return 1;
       }
    }
    return 0;
}

/*************************************************************************/

/* Check whether a user should be voiced on a channel, and if so, do it.
 * Return 1 if the user was voiced, 0 otherwise. */

int check_should_voice(User *user, const char *chan)
{
    ChannelInfo *ci = cs_findchan(chan);
    NickInfo *ni;

    if (!ci || (ci->flags & CI_VERBOTEN) || !ci->levels[CA_AUTOVOICE] ||
		*chan == '+')
	return 0;

    if (ci->flags & CI_FREEZECHAN)
        return 0;

    ni = findnick(user->nick);
    if (ci->flags & CI_IDENT) {
        if (ni && !(ni->flags & NI_IDENTIFIED))
	    return 0;
    }

    if (ni && (ni->flags & NI_NEVEROP))
         return 0;

    if (get_access(user, ci) == 3) {
         send_cmd(MODE_SENDER, "MODE %s +v %s", chan, user->nick);
         return 1;
    }
    return 0;
}

/*************************************************************************/

/* Tiny helper routine to get ChanServ out of a channel after it went in. */

static inline void timeout_leave(Timeout *to)
{
    char *chan = to->data;
    send_cmd(s_ChanServ, "PART %s", chan);
    free(to->data);
}


static inline void timeout_unban(Timeout *to)
{
    char *chan = to->data;
    send_cmd(s_ChanServ, "MODE %s -b *!*@*", chan);
    free(to->data);
}


/* Check whether a user is permitted to be on a channel.  If so, return 0;
 * else, kickban the user with an appropriate message (could be either
 * AKICK or restricted access) and return 1.  Note that this is called
 * _before_ the user is added to internal channel lists (so do_kick() is
 * not called).
 */

int check_kick(User *user, const char *chan)
{
    ChannelInfo *ci = cs_findchan(chan);
    Channel *c = findchan(chan);
    C_SuspendData *c_spn = get_c_suspend(chan);
    struct c_userlist *u;
    AutoKick *akick;
    unsigned int i, id = 0, x=0;
    NickInfo *ni;
    char *av[3], *mask, *reason;
    int check_level;
    Timeout *t;

/*  Channel suspension check must come first because if the channel is
 *  not registered, services will not be able to kick because !ci returns 0.
 */
    if (c_spn && c_spn->time > time(NULL)) {
       mask = create_mask(user);
       reason = SUSPEND_REASON;
       goto kick;
    }

    if (!ci)
	return 0;

    if (ci->flags & CI_CLOSED) {
        mask = create_mask(user);
        reason = CLOSED_REASON;
        goto kick;
    }

    if (ci->flags & CI_VERBOTEN) {
	mask = create_mask(user);
	reason = FORBID_REASON;
	goto kick;
    }


    if (ci->flags & CI_FREEZECHAN)
        return 0;


    if (is_services_root(user->nick))
	return 0;


    if ((ci->flags & CI_CODERONLY) && !is_services_coder(user->nick)) {
        mask = create_mask(user);
        reason = "Only Services Masters are allowed to use this channel.";
        goto kick;
    }

    if ((ci->flags & CI_OPERONLY) && !is_oper(user->nick)) {
	mask = create_mask(user);
	reason = "Only IRC Operators are allowed to use this channel.";
	goto kick;
    }
    if ((ci->flags & CI_SOPONLY) && !rec_services_oper(user->nick)) {
	mask = create_mask(user);
	reason = "Only Services Opers are allowed to use this channel.";
	goto kick;
    }
    if ((ci->flags & CI_SAONLY) && !rec_services_admin(user->nick)) {
	mask = create_mask(user);
	reason = "Only Services Admins are allowed to use this channel.";
	goto kick;
    }
    if ((ci->flags & CI_SRAONLY) && !rec_services_root(user->nick)) {
	mask = create_mask(user);
	reason = "Only Services Roots are allowed to use this channel.";
	goto kick;
    }

    if ((ci->flags & CI_ABUSEONLY) && !rec_abuse(user->nick)) {
        mask = create_mask(user);
        reason = "Only Abuse Team Members are allowed to use this channel.";
        goto kick;
    }

    if ((ci->flags & CI_RESTRICTED) && !rec_services_root(user->nick)) {
         if (get_access(user, ci) < 3) {
             mask = create_mask(user);
             reason = "You are not permitted to be on this channel.";
	     goto kick;
         }
    }

    ni = findnick(user->nick);
    if (ni && ni->flags & (NI_IDENTIFIED | NI_RECOGNIZED))
        id = 1;


    if (get_access(user, ci) > 0)
        return 0;

    for (akick = ci->akick, i = 0; i < ci->akickcount; ++akick, ++i) {
	if ((akick->is_nick > 0
			&& id && stricmp(akick->name, user->nick) == 0) ||
			(!akick->is_nick && match_usermask(akick->name, user))
	) {
	    mask = akick->is_nick ? create_mask(user) : akick->name;
	    reason = akick->reason ? akick->reason : DEF_AKICK_REASON;
	    goto kick;
	}
    }

    return 0;

kick:

    if (debug) {
	log("debug: channel: AutoKicking %s!%s@%s",
		user->nick, user->username, user->host);
    }
    /* Remember that the user has not been added to our channel user list
     * yet, so we check whether the channel does not exist */

     if (c) {
        for (u = c->users; u; u = u->next)
          x++;
     }

    if (x < 2) {
       send_cmd(s_ChanServ, "JOIN %s", chan);
       t = add_timeout(CHANNEL_INHABIT, timeout_leave, 0);
       t->data = sstrdup(chan);
    }

/* 
   should be fine to leave out timeout_unban since the empty
   channel will remove the mode anyway.
*/

    av[0] = sstrdup(chan);
    strcpy(av[0], chan);
    av[1] = sstrdup("+b");
    av[2] = mask;
    send_cmd(s_ChanServ, "MODE %s +b %s  %lu", chan, av[2], time(NULL));
    do_cmode(s_ChanServ, 3, av);
    send_cmd(s_ChanServ, "KICK %s %s :%s", chan, user->nick, reason);
    av[1] = sstrdup(user->nick);
    av[2] = sstrdup(reason);
    do_kick(s_ChanServ, 3, av);
    free(av[0]); free(av[1]); free(av[2]);

    return 1;
}

/*************************************************************************/

/* Record the current channel topic in the ChannelInfo structure. */

void record_topic(const char *chan)
{
    Channel *c;
    ChannelInfo *ci;

    if (readonly)
	return;
    c = findchan(chan);
    ci = cs_findchan(chan);
    if (!c || !ci)
	return;
    if (ci->flags & CI_FREEZECHAN)
        return;
    if (ci->last_topic)
	free(ci->last_topic);
    if (c->topic)
	ci->last_topic = sstrdup(c->topic);
    else
	ci->last_topic = NULL;
    strscpy(ci->last_topic_setter, c->topic_setter, NICKMAX);
    ci->last_topic_time = c->topic_time;
}

/*************************************************************************/

/* Restore the topic in a channel when it's created, if we should. */

void restore_topic(const char *chan)
{
    Channel *c = findchan(chan);
    ChannelInfo *ci = cs_findchan(chan);


    if (!c || !ci)
	return;

    if (!(ci->flags & CI_KEEPTOPIC)) {
       return;
    }

    if (c->topic)
	free(c->topic);
    if (ci->last_topic) {
	c->topic = sstrdup(ci->last_topic);
	strscpy(c->topic_setter, ci->last_topic_setter, NICKMAX);
	c->topic_time = ci->last_topic_time;
    } else {
	c->topic = NULL;
	strscpy(c->topic_setter, s_ChanServ, NICKMAX);
    }
    if (c->topic != NULL)
      if (TOPIC_PROBS == 1)
       send_cmd(MODE_SENDER, "TOPIC %s %s %lu :%s", chan,
		c->topic_setter, c->topic_time, c->topic ? c->topic :"");
      else
       send_cmd(MODE_SENDER, "TOPIC %s %s %lu :%s (%s)", chan,
		c->topic_setter, c->topic_time, c->topic ? c->topic :"",
                ci->last_topic_setter);

}

/*************************************************************************/

/* See if the topic is locked on the given channel, and return 1 (and fix
 * the topic) if so. */

int check_topiclock(const char *source, const char *chan)
{
    Channel *c = findchan(chan);
    ChannelInfo *ci = cs_findchan(chan);
    User *u = finduser(source);
    int ulev;

    if (!c || !ci || (!(ci->flags & CI_TOPICLOCK)))
	return 0;


    if ((u && (ci->topic_allow <= (ulev = get_access(u, ci))))
        && (!(ci->flags & CI_FREEZECHAN))) 
        return 0;


    if (c->topic)
	free(c->topic);
    if (ci->last_topic)
	c->topic = sstrdup(ci->last_topic);
    else
	c->topic = NULL;
    strscpy(c->topic_setter, ci->last_topic_setter, NICKMAX);
    c->topic_time = ci->last_topic_time;
    send_cmd(MODE_SENDER, "TOPIC %s %s %lu :%s", chan, c->topic_setter,
          c->topic_time, c->topic ? c->topic : "");
    return 1;
}

/*************************************************************************/

/* Remove all channels which have expired. */

void expire_chans()
{
    ChannelInfo *ci, *next;
    NickInfo *ni;
    unsigned int i;
    const time_t expire_time = CHANNEL_EXPIRE*24*60*60;
    time_t now = time(NULL);
    long count = 0;
    long xcount = 0;


    for (i = 33; i < 256; ++i) {
	for (ci = chanlists[i]; ci; ci = next) {
            ni = findnick(ci->founder);
	    ++count;
	    next = ci->next;
            if ((now - ci->last_used >= expire_time && !(ci->flags & CI_VERBOTEN) && !(ci->flags & CI_HELDCHAN))
               || (!(ni) && (!(ci->flags & CI_VERBOTEN)))) {
		++xcount;
                slog("CS X %s", ci->name);
                do_break_log("CS_X", "CS X %s", ci->name);
		log("Expiring channel %s", ci->name);
                if (ni && ni->channelcount > 0)
                    ni->channelcount--;
		delchan(ci);
	    } else if (now - ci->last_used >= 
                              (expire_time - (24*60*60*REMINDER))
                              && (ni && ni->email)
                              && (ALLOW_EMAIL)
                              && (REMINDER > 0)
                              && !(ci->flags & CI_REMIND)
                              && !(ci->flags & CI_VERBOTEN)
                              && !(ci->flags & CI_HELDCHAN)) {

             struct tm tm;
             char timebuf[64];
             char cmdbuf[500]; char cmdbuf2[500];
             FILE *f = fopen("memo.txt", "w");
             FILE *fs = fopen(SERVICES_SIG, "r");

             tm = *localtime(&now);
             strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
             timebuf[sizeof(timebuf)-1] = 0;
               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->regemail);
               fprintf(f, "Subject: %s registration reminder\n\n",
                      NETWORK_NAME);
               fprintf(f, "Your chan (%s) will expire in %d days\n",
                      ci->name, REMINDER);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


            system(cmdbuf);
            system(cmdbuf2);

            ci->flags |= CI_REMIND;
            slog("CS X+ %s", ci->name);
            do_break_log("CS_X", "CS X+ %s", ci->name);

            }
	}
    }
  if (DISPLAY_UPDATES == 1)
     wallops(SERVER_NAME, "Completed Channel Expire (%d/%d)",xcount,count);
}

/*************************************************************************/

/* Remove a (deleted or expired) nickname from all channel access lists. */

void cs_remove_nick(const char *nick)
{
    unsigned int i;
    int j;
    ChannelInfo *ci;
    ChanAccess *ca;

    for (i = 33; i < 256; i++) {
	for (ci = chanlists[i]; ci; ci = ci->next) {
	    for (ca = ci->access, j = ci->accesscount; j > 0; ca++, j--) {
		if (ca->in_use && stricmp(ca->name, nick) == 0)
		    ca->in_use = 0;
	    }
	}
    }
}

/*************************************************************************/
/*********************** ChanServ private routines ***********************/
/*************************************************************************/

/* Return the ChannelInfo structure for the given channel, or NULL if the
 * channel isn't registered. */

ChannelInfo *cs_findchan(const char *chan)
{
    ChannelInfo *ci;

    for (ci = chanlists[tolower(chan[1])]; ci; ci = ci->next) {
	if (stricmp(ci->name, chan) == 0)
	    return ci;
    }
    return NULL;
}

/*************************************************************************/

/* Insert a channel alphabetically into the database. */

static void alpha_insert_chan(ChannelInfo *ci)
{
    ChannelInfo *ci2, *ci3;
    char *chan = ci->name;

    for (ci3 = NULL, ci2 = chanlists[tolower(chan[1])];
			ci2 && stricmp(ci2->name, chan) < 0;
			ci3 = ci2, ci2 = ci2->next)
	;
    ci->prev = ci3;
    ci->next = ci2;
    if (!ci3)
	chanlists[tolower(chan[1])] = ci;
    else
	ci3->next = ci;
    if (ci2)
	ci2->prev = ci;
}

/*************************************************************************/

/* Add a channel to the database.  Returns a pointer to the new ChannelInfo
 * structure if the channel was successfully registered, NULL otherwise.
 * Assumes channel does not already exist. */

static ChannelInfo *makechan(const char *chan)
{
    ChannelInfo *ci;
    Channel *c;
    char *s, *end;

    ci = scalloc(sizeof(ChannelInfo), 1);
    strscpy(ci->name, chan, CHANMAX);
    ci->time_registered = time(NULL);
    reset_levels(ci);
    alpha_insert_chan(ci);
    return ci;
}

/*************************************************************************/

/* Remove a channel from the ChanServ database.  Return 1 on success, 0
 * otherwise. */

static int delchan(ChannelInfo *ci)
{
    User *u;
    struct u_chaninfolist *c;
    unsigned int i;

   for (u = userlist; u; u = u->next) {
      for (c = u->founder_chans; c; c = c->next) {
          if (c->chan == ci) {
             if (c->prev)
                c->prev->next = c->next;
             else
                u->founder_chans = c->next;
             if (c->next)
                c->next->prev = c->prev;
             free(c);
             break;
          }
       }
   }


    if (ci->next)
	ci->next->prev = ci->prev;
    if (ci->prev)
	ci->prev->next = ci->next;
    else
	chanlists[tolower(ci->name[1])] = ci->next;
    if (ci->desc)
	free(ci->desc);
    if (ci->mlock_key)
	free(ci->mlock_key);
    if (ci->last_topic)
	free(ci->last_topic);
    for (i = 0; i < ci->accesscount; ++i) {
	free(ci->access[i].name);
        free(ci->access[i].sponser);
    }
    if (ci->access)
	free(ci->access);
    for (i = 0; i < ci->akickcount; ++i) {
	free(ci->akick[i].name);
	if (ci->akick[i].reason)
	    free(ci->akick[i].reason);
    }
    if (ci->akick)
	free(ci->akick);
    if (ci->levels)
	free(ci->levels);
    if (ci->welcome)
        free(ci->welcome);
    if (ci->hold)
        free(ci->hold);
    if (ci->mark)
        free(ci->mark);
    if (ci->freeze)
        free(ci->freeze);
    if (ci->forbid)
        free(ci->forbid);

    free(ci);
    return 1;
}

/*************************************************************************/

/* Reset defined access level values to their default state. */

static void reset_levels(ChannelInfo *ci)
{
    unsigned int i;

    if (ci->levels)
	free(ci->levels);
    ci->levels = smalloc(CA_SIZE * sizeof(*ci->levels));
    for (i = 0; def_levels[i][0] >= 0; i++)
	ci->levels[def_levels[i][0]] = def_levels[i][1];
}

/*************************************************************************/

/* Does the given user have founder access to the channel? */

static int is_founder(User *user, NickInfo *ni, ChannelInfo *ci, int set)
{
    struct u_chaninfolist *c;
    unsigned int i;

    if (is_identified(user, ci))
	return 1;

    ni = findnick(user->nick);
    if (set && (ci->flags & CI_UNSECURE)) {
       if (ni && (ni->flags & NI_RECOGNIZED)
              && (stricmp(ni->nick, ci->founder) == 0))
          return 1;
    }
    
    if (ni && ((ni->flags & NI_RECOGNIZED) && (!(ci->flags & CI_IDENT)))
           && !set && (stricmp(ni->nick, ci->founder) == 0))
         return 1;

    for (i=0; user->id_nicks[i] && i < MAX_IDS; i++) {
       ni = findnick(user->id_nicks[i]);

       if (ni && !set && ((ni->flags & NI_IDENTIFIED)
	   || ((ni->flags & NI_RECOGNIZED) && (!(ci->flags & CI_IDENT)))) 
	   && (stricmp(ni->nick, ci->founder) == 0))
           return 1;
       else if (ni && set && ((ni->flags & NI_IDENTIFIED)
           || ((ni->flags & NI_RECOGNIZED) && (ci->flags & CI_UNSECURE)))
              && (stricmp(ni->nick, ci->founder) == 0))
           return 1;
    }


    return 0;
}

/*************************************************************************/

/* Has the given user password-identified as founder for the channel? */

static int is_identified(User *user, ChannelInfo *ci)
{
    struct u_chaninfolist *c;

    for (c = user->founder_chans; c; c = c->next) {
	if (c->chan == ci)
	    return 1;
    }
    return 0;
}

/*************************************************************************/

/* Return the access level the given user has on the channel.  If the
 * channel doesn't exist, the user isn't on the access list, or the channel
 * is CS_IDENT and the user hasn't IDENTIFY'd with NickServ, return 0. */

int get_access(User *user, ChannelInfo *ci)
{

    NickInfo *ni, *ni2;
    ChanAccess *access;
    char **naccess;
    int i, i2, a=0, k=0, checked=0;

    if (!ci)
        return 0;

    ni = findnick(user->nick);

    for (access = ci->access, i2 = 0; i2 < ci->accesscount; ++access, ++i2) {
       if (ni && ((ni->flags & NI_RECOGNIZED) && (!(ci->flags & CI_IDENT)))
             && (!(ni->flags & NI_IDENTIFIED))
             && (stricmp(ni->nick, access->name) == 0)) {
                if (access->level > a)
                    a = access->level;
                if (access->level < 0)
                    k = access->level;
       } else if (!(ci->flags & CI_IDENT)) {
           if (match_usermask(access->name, user)) {
               if (access->level > a)
                  a = access->level;
               if (access->level < k)
                  k = access->level;
           } else {
               ni2 = findnick(access->name);
               if (ni2) {
                   for (naccess = ni2->access, i = 0; i < ni2->accesscount; ++naccess, ++i) {
                      if (is_on_access(user, ni2)) {
                          if (access->level > a)
                               a = access->level;
                      }
                   }
               }
           }
       }
    }

    if (is_founder(user, ni, ci, 0))
        return 15;

    for (i=0; i < MAX_IDS && user->id_nicks[i]; i++) {
      for (access = ci->access, i2 = 0; i2 < ci->accesscount; ++access, ++i2) {
          ni = findnick(user->id_nicks[i]);
          if (ni && (ni->flags & NI_IDENTIFIED)
              && stricmp(ni->nick, access->name) == 0) {
                   if (access->level > a)
                       a = access->level;
                   if (access->level < 0)
                       k = access->level;
          }
       }
    }

  if (a > 0)
     return a;
  else if ((a <= 0) && (k < 0))
     return k;
  else
     return 0;
}


/*************************************************************************/

/* Return 1 if the user's access level on the given channel falls into the
 * given category, 0 otherwise.  Note that this may seem slightly confusing
 * in some cases: for example, check_access(..., CA_NOJOIN) returns true if
 * the user does _not_ have access to the channel (i.e. matches the NOJOIN
 * criterion). */

int check_access(User *user, ChannelInfo *ci, int what)
{
    int level = get_access(user, ci);

    if (level == 15)
	return (what==CA_AUTODEOP || what==CA_NOJOIN) ? 0 : 1;
    if (ci->levels[what] == ACCESS_INVALID)
	return 0;
    if (what == CA_AUTODEOP || what == CA_NOJOIN)
	return level <= ci->levels[what];
    else
	return level >= ci->levels[what];
}

/*************************************************************************/
/*********************** ChanServ command routines ***********************/
/*************************************************************************/

static void do_ohelp(const char *source)
{
    const char *cmd = strtok(NULL, "");
    char buf[256];
    char *access;
    int is_servadmin = is_services_oper(source);


    if (!cmd)
       log("MS - %s requested ohelp: (index)",
             source);

    if (cmd) {

        log("CS - %s requested ohelp: %s",
              source, cmd);

        if (stricmp(cmd, "DELETE") == 0) {
            notice_list(s_ChanServ, source, oper_delete_help);
            return;
        } else if (stricmp(cmd, "FLIST") == 0) {
            notice_list(s_ChanServ, source, flist_help);
            return;
        } else if (stricmp(cmd, "TOPIC") == 0) {
            notice_list(s_ChanServ, source, ostopic_help);
            return;
        } else if (stricmp(cmd, "GETPASS") == 0) {
            notice_list(s_ChanServ, source, getpass_help);
            return;
        } else if (stricmp(cmd, "FREEZE") == 0) {
            notice_list(s_ChanServ, source, freeze_help);
            return;
        } else if (stricmp(cmd, "FORBID") == 0) {
            notice_list(s_ChanServ, source, forbid_help);
            return;
        } else if (stricmp(cmd, "MARK") == 0) {
            notice_list(s_ChanServ, source, mark_help);
            return;
        } else if (stricmp(cmd, "HOLD") == 0) {
            notice_list(s_ChanServ, source, hold_help);
            return;
        } else if (stricmp(cmd, "MARK") == 0) {
            notice_list(s_ChanServ, source, mark_help);
            return;
        } else if (stricmp(cmd, "HOLD") == 0) {
            notice_list(s_ChanServ, source, hold_help);
            return;
        } else if (stricmp(cmd, "SUSPEND") == 0) {
            notice_list(s_ChanServ, source, suspend_help);
            return;
        } else if (stricmp(cmd, "WIPE") == 0) {
            notice_list(s_ChanServ, source, wipe_help);
            return;
        } else if (stricmp(cmd, "LEVEL") == 0) {
            notice_list(s_ChanServ, source, level_help);
            return;
        } else if (stricmp(cmd, "RACCESS") == 0) {
            notice_list(s_ChanServ, source, level_help);
            return;
        } else if (stricmp(cmd, "FLUSH") == 0) {
            notice_list(s_ChanServ, source, flush_help);
            return;
        } else {
            notice(s_ChanServ, source, "No help topic for \2%s\2 found", cmd);
            return;
        }

    }
    if (!cmd)
       serv_os_help(source, "CS", s_ChanServ);
    else {
       if (access = find_level(cmd))
          notice(s_ChanServ, source,
            "This command is limited to \2%s\2.", access);
    }

    notice(s_ChanServ, source,
       "\2END OF OHELP\2");
}

static void do_help(const char *source)
{
    const char *cmd = strtok(NULL, "");
    char buf[256];

    snprintf(buf, sizeof(buf), "%s%s", s_ChanServ, cmd ? " " : "");
    strscpy(buf+strlen(buf), cmd ? cmd : "", sizeof(buf)-strlen(buf));
    helpserv(s_ChanServ, source, buf);
    if (!cmd || (cmd && stricmp(cmd, "REGISTER") == 0))
        notice(s_ChanServ, source,
           "Channels will expire after \2%d days\2 of inactivity",
            CHANNEL_EXPIRE);
}

/*************************************************************************/

static void do_register(const char *source)
{
    char *chan = strtok(NULL, " ");
    const char *pass = strtok(NULL, " ");
    const char *desc = strtok(NULL, "");
    char *keepmlocks;
    char *tmp;
    NickInfo *ni = findnick(source);
    ChannelInfo *ci;
    struct u_chaninfolist *c;
    User *u = finduser(source);
    int add = 1, x=0;


    if (readonly) {
	notice(s_ChanServ, source,
		"Sorry, channel registration is temporarily disabled.");
	return;
    }
    if (chan) {
      tmp = chan;
      *tmp++;
    }

    if (!desc) {

	notice(s_ChanServ, source,
                "Syntax: \2REGISTER\2 <#channel> <password> <description>");
	notice(s_ChanServ, source,
		"\2/msg %s HELP REGISTER\2 for more information.", s_ChanServ);
        return;

    } else if (*chan == '&') {

	notice(s_ChanServ, source, "Local channels cannot be registered.");

    } else if (chan[1] == 0) {

	notice(s_ChanServ, source, "\2%s\2 is not a valid channel name.", chan);

    } else if (!ni) {

	notice(s_ChanServ, source, "Your must register your nickname first.");
	notice(s_ChanServ, source,
		"\2/msg %s HELP\2 for information on registering nicknames.",
		s_NickServ);

    } else if (!(ni->flags & (NI_RECOGNIZED | NI_IDENTIFIED))) {

	notice(s_ChanServ, source,
		"Please identify with %s first, using the command:",
		s_NickServ);
	notice(s_ChanServ, source,
		"/msg %s IDENTIFY \37password\37", s_NickServ);

    } else if ((ni->flags & NI_AUTH || !ni->regemail) 
         && nsecure && auth_email) {
         notice(s_ChanServ, source,
            "You may not register a channel until you have authorized "
            "your nick. If you need assistance, find a %s representitive.",
             NETWORK_NAME);


    } else if (ci = cs_findchan(chan)) {

	if (ci->flags & CI_VERBOTEN) {
            slog("CS *R %s (%s!%s@%s) [FORBIDDEN]",
                chan, source, u->username, u->host);
            do_break_log("CS_R", "CS *R %s (%s!%s@%s) [FORBIDDEN]",
                chan, source, u->username, u->host);
	    log("%s: Attempt to register FORBIDden channel %s by %s!%s@%s",
			s_ChanServ, chan, source, u->username, u->host);
	    notice(s_ChanServ, source,
			"Channel \2%s\2 may not be registered.", chan);
	} else {
	    notice(s_ChanServ, source,
			"Channel \2%s\2 already registered!", chan);
	}

    } else if (!u) {

	log("%s: Attempt to register channel %s from nonexistent nick %s",
						s_ChanServ, chan, source);
	notice(s_ChanServ, source, "Sorry, registration failed.");

    } else if (!is_chanop(source, chan)) {

	notice(s_ChanServ, source,
		"You must be a channel operator to register the channel.");

    } else if ((CHANNEL_MAXREG > 0) && (ni->channelcount >= CHANNEL_MAXREG)
		&& !(is_services_admin(source))) {
	notice(s_ChanServ, source,
		"Sorry, you have already %s your limit of \2%d\2 channels.",
		ni->channelcount > CHANNEL_MAXREG ? "exceeded" : "reached",
		CHANNEL_MAXREG);


    } else if (match_wild_nocase(pass, tmp) || (strlen(pass) < 5)
            || (stricmp(chan, pass) == 0)) {

        notice(s_ChanServ, source,
                "Please try again with a more obscure password.  Passwords "
                "should be at least five characters long, should not be "
                "something easily guessed (e.g. your real name or your "
                "nick), and cannot contain the space or tab characters.  "
                "\2/msg %s HELP REGISTER\2 for more information.",
                s_ChanServ);


    } else if (strlen(pass) > PASSMAX) {
           notice(s_ChanServ, source,
            "Sorry, channel passwords may not exceed %d characters in length",
                 PASSMAX);
    } else {

	if (ci = makechan(chan)) {
	    Channel *c;
	    if (!(c = findchan(chan))) {
		log("%s: Channel %s not found for REGISTER", s_ChanServ, chan);
		notice(s_ChanServ, source, "Sorry, registration failed.");
		return;
	    }
            csreg += 1;
	    ci->flags = CI_KEEPTOPIC;
	    ci->last_used = ci->time_registered;
	    strscpy(ci->founder, source, NICKMAX);
	    strscpy(ci->founderpass, pass, PASSMAX);
	    ci->desc = sstrdup(desc);
            ci->welcome = NULL;
            ci->auth = 0;
            ci->flags |= CI_MEMO_AV;

            keepmlocks = sstrdup(DEF_MLOCKS);

            if (DEF_MLOCKS == "+nt") {
              ci->mlock_on |= CMODE_N; ci->mlock_on |= CMODE_T;
            } else {
                while (*DEF_MLOCKS) {
                   x++;
                   switch(*DEF_MLOCKS++) {

                   case '+':
                      add = 1; break;

                   case '-':
                      add = 0; break;

                   case 'n':
                     if (add)
                        ci->mlock_on |= CMODE_N;
                     else
                        ci->mlock_off |= CMODE_N;
                     break;

                   case 't':
                     if (add)
                        ci->mlock_on |= CMODE_T;
                     else
                        ci->mlock_off |= CMODE_T;
                     break;

                   case 'i':
                     if (add)
                        ci->mlock_on |= CMODE_I;
                     else
                        ci->mlock_off |= CMODE_I;
                     break;

                   case 'm':
                     if (add)
                        ci->mlock_on |= CMODE_M;
                     else
                        ci->mlock_off |= CMODE_M;
                     break;

                   case 'p':
                     if (add)
                        ci->mlock_on |= CMODE_P;
                     else
                        ci->mlock_off |= CMODE_P;
                     break;

                   case 's':
                     if (add)
                        ci->mlock_on |= CMODE_S;
                     else
                        ci->mlock_off |= CMODE_S;
                     break;

                   case 'c':
                     if (add)
                        ci->mlock_on |= CMODE_C;
                     else
                        ci->mlock_off |= CMODE_C;
                     break;

                   case 'R':
                     if (add)
                        ci->mlock_on |= CMODE_r;
                     else
                        ci->mlock_off |= CMODE_r;
                     break;

#if BAH_V >= 147
                   case 'O':
                     if (add)
                        ci->mlock_on |= CMODE_o;
                     else
                        ci->mlock_off |= CMODE_o;
                     break;
                   
                   case 'M':
                     if (add)
                        ci->mlock_on |= CMODE_m;
                     else
                        ci->mlock_off |= CMODE_m;
                     break;
#endif
                   }
                }
            }
            DEF_MLOCKS = sstrdup(keepmlocks);
            free(keepmlocks);

              if (ENABLE_R_MODES || ENABLE_R_MODES == 2)
                 ci->mlock_on |= CMODE_R;

/*
              ci->mlock_off |= CMODE_I; ci->mlock_off |= CMODE_M;
              ci->mlock_off |= CMODE_P; ci->mlock_off |= CMODE_S;
*/

	    if (c->topic) {
		ci->last_topic = sstrdup(c->topic);
		strscpy(ci->last_topic_setter, c->topic_setter, NICKMAX);
		ci->last_topic_time = c->topic_time;
	    }
	    if (ni->channelcount+1 > ni->channelcount)
		ni->channelcount++;
            slog("CS R %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
            do_break_log("CS_R", "CS R %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
	    log("%s: Channel %s registered by %s!%s@%s", s_ChanServ, chan,
			source, u->username, u->host);
            notice(s_ChanServ, source,  CS_REGISTER,  chan, source);
#ifndef USE_ENCRYPTION
            notice(s_ChanServ, source,  CS_REGISTER2,  ci->founderpass);
            if (x > 1)
              notice(s_ChanServ, source,  CS_REGISTER3,  chan, DEF_MLOCKS);

#endif
#if REG_SAVES > 0
            if (regcnt++ >= REG_SAVES) {
                wallops(SERVER_NAME,
                     "Automatic DataBase save on %d new registrations",
                            regcnt);
                save_data = 1;
            }
#endif
            show_next_db(source, s_ChanServ);
	} else {
	    notice(s_ChanServ, source, "Sorry, registration failed.");
	}

        c = smalloc(sizeof(*c));
        c->next = u->founder_chans;
        c->prev = NULL;
        if (u->founder_chans)
            u->founder_chans->prev = c;
        u->founder_chans = c;
        c->chan = ci;
    }
    check_modes(chan);
}

/*************************************************************************/

static void do_identify(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *pass = strtok(NULL, " ");
    ChannelInfo *ci;
    struct u_chaninfolist *c;
    User *u = finduser(source);

    if (!pass) {

	notice(s_ChanServ, source,
                "Syntax: \2IDENTIFY\2 <#channel> <password>");
	notice(s_ChanServ, source,
		"\2/msg %s HELP IDENTIFY\2 for more information.", s_ChanServ);

    } else if (!(ci = cs_findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!u) {

	log("%s: IDENTIFY from nonexistent nick %s for %s",
						s_ChanServ, source, chan);
	notice(s_ChanServ, source, "Sorry, identification failed.");

    } else {

	int res;
	int is_force = (is_services_root(source) && !stricmp(pass, "force"));

	if (is_force || (res = check_password(pass, ci->founderpass)) == 1) {
	    if (!is_identified(u, ci)) {
		c = smalloc(sizeof(*c));
		c->next = u->founder_chans;
		c->prev = NULL;
		if (u->founder_chans)
		    u->founder_chans->prev = c;
		u->founder_chans = c;
		c->chan = ci;
                if (ci->flags & CI_REMIND)
                     ci->flags &= ~CI_REMIND;
		log("%s: %s!%s@%s identified for %s %s", s_ChanServ,
			source, u->username, u->host, chan,
			is_force ? "[Force]" : "");
                if (EXTRA_SNOOP && !is_force) {
                     slog("CS I %s (%s!%s@%s) %s",
                           chan, u->nick, u->username, u->host,
      			   is_force ? "[Force]" : "");
                     do_break_log("CS_I", "CS I %s (%s!%s@%s) %s",
                           chan, u->nick, u->username, u->host,
   		           is_force ? "[Force]" : "");
                     csid += 1;
                }
	    }
            notice(s_ChanServ, source,  CS_IDENTIFY,  chan);
	} else if (res < 0) {
            if (EXTRA_SNOOP) {
                 slog("CS *I %s (%s!%s@%s)",
                     chan, u->nick, u->username, u->host);
                 do_break_log("CS_I", "CS *I %s (%s!%s@%s)",
                     chan, u->nick, u->username, u->host);
                 csidf += 1;
            }
	    log("%s: check_password failed for %s", s_ChanServ, chan);
	    notice(s_ChanServ, source, "Sorry, identification failed.");
	} else {
	    log("%s: Failed IDENTIFY for %s by %s!%s@%s",
			s_ChanServ, chan, source, u->username, u->host);
            notice(s_ChanServ, source, CS_BADPASS);
	    bad_password(u, chan);
            csidf += 1;
	}

    }
}
/*************************************************************************/

static void do_delete(const char *source)
{
    NickInfo *ni;
    const char *chan = strtok(NULL, " ");
    ChannelInfo *ci;
    User *u = finduser(source);
    int is_servop = is_services_root(source);

    if (readonly) {
        notice(s_ChanServ, source,
                "Sorry, channel deletion is temporarily disabled. (Read Only)");
	return;
    }

    if (!chan) {
       notice(s_ChanServ, source, "\2Delete\2: You MUST specify a channel");
       return;
    }

    if (!(ci = cs_findchan(chan))) {
        if (chan)
             notice(s_ChanServ, source, "Channel %s isn't registered.", chan);
        return;
    }
    ni = findnick(ci->founder);
    if (ni) {
       if (ni->channelcount > 0)
            ni->channelcount--;
    } else
       log("Could not find founder's nick in deleting channel %s", ci->name);

    delchan(ci);
    slog("CS +D %s (%s!%s@%s)", chan, source, u->username, u->host);
    do_break_log("CS_G", "CS +D %s (%s!%s@%s)", chan, source, u->username, u->host);
    log("%s: %s!%s@%s deleted channel %s", s_ChanServ,
            source, u->username, u->host, chan);
        notice(s_ChanServ, source, CS_DELETE, chan);
        wallops(s_ChanServ, "%s deleted channel \2%s\2", source, chan);
}

/*************************************************************************/

static void do_drop(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *auth = strtok(NULL, " ");
    C_SuspendData *c_spn;
    ChannelInfo *ci;
    NickInfo *ni;
    struct u_chaninfolist *c, *c2;
    User *u = finduser(source), *user;
    Timeout *t;


    if (readonly) {
	notice(s_ChanServ, source,
		"Sorry, channel de-registration is temporarily disabled.");
	return;
    }

    if (!chan) {
        notice(s_ChanServ, source, "Syntax: \2DROP\2 <#channel> %s",
             auth_email ? "[<drop code>]" : "");
	notice(s_ChanServ, source,
		"\2/msg %s HELP DROP\2 for more information.", s_ChanServ);
        return;
    }


    c_spn = get_c_suspend(chan);

    if (!(ci = cs_findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (ci->flags & CI_FREEZECHAN) {
         notice(s_ChanServ, source, "Channel %s is currently frozen", chan);

    } else if (c_spn && c_spn->time > time(NULL)) {
       char *av[3];
       send_cmd(s_ChanServ, "JOIN %s", chan);
       send_cmd(s_ChanServ, "MODE %s +b *!*@*", chan);
       send_cmd(s_ChanServ, "KICK %s %s :Channel is currently suspended",
              chan, u->nick);
       av[0] = sstrdup(chan);
       av[1] = sstrdup(u->nick);
       av[2] = sstrdup("Channel is currently Suspended");
       do_kick(s_ChanServ, 3, av);
      free(av[0]); free(av[1]); free(av[2]);
      t = add_timeout(CHANNEL_INHABIT, timeout_leave, 0);
      t->data = sstrdup(chan);


    } else if (!u || !is_identified(u, ci)) {

            slog("CS *D %s (%s!%s@%s) [UNIDENTIFIED]",
                chan, u->nick, u->username, u->host);
            do_break_log("CS_O", "CS *D %s (%s!%s@%s) [UNIDENTIFIED]",
                chan, u->nick, u->username, u->host);

	notice(s_ChanServ, source,
		"Password authentication required for that command.");
	notice(s_ChanServ, source,
		"Retry after typing \2/msg %s IDENTIFY %s <password>.",
		s_ChanServ, chan);

    } else if (!u) {

	log("%s: DROP %s from nonexistent oper %s", s_ChanServ, chan, source);
	notice(s_ChanServ, source, "Can't find your user record!");

    } else if (auth_email && !auth && !ci->auth) {

             struct tm tm;
             time_t now = time(NULL);
             char timebuf[64];
             char cmdbuf[500]; char cmdbuf2[500];
             FILE *f = fopen("memo.txt", "w");
             FILE *fs = fopen(SERVICES_SIG, "r");
             FILE *fs2 = fopen(SERVICES_SIG, "r");
             tm = *localtime(&now);
             strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
             timebuf[sizeof(timebuf)-1] = 0;

               if (!(ni = findnick(ci->founder))) {
                    notice(s_ChanServ, source,
                          "ERROR: Can't find channel founder information.");
                    return;
               }

               ci->auth = ci->time_registered + getrandom(1,99999) * getrandom(1,9999);

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->regemail);
               fprintf(f, "Subject: %s drop authorization\n\n",
                      NETWORK_NAME);
               fprintf(f, "You or someone has recently requested your channel to be dropped.\n");
               fprintf(f, "If you wish this action to be taken, follow these steps:\n");
               fprintf(f, "Your channel's drop code is %d. Log back onto %s and type - /MSG %s DROP %s %d\n",
                      ci->auth, NETWORK_NAME, s_ChanServ, ci->name, ci->auth);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


            system(cmdbuf);
            system(cmdbuf2);


         notice(s_ChanServ, source,
               "You will receive a drop authorization code in your email shortly. "
                "Please follow the instructions of the email to complete the action "
                "of dropping this channel");


    } else if (auth_email && !auth && ci->auth) {
          notice(s_ChanServ, source,
             "A drop authorization code has already been sent. "
             "Please contact a %s representitive for further assistance.", NETWORK_NAME);

    } else if ((auth_email && auth) && (atoi(auth) != ci->auth)) {
         notice(s_ChanServ, source,
              "Invalid drop authorization code.");


    } else {

	if (readonly) {
	    notice(s_ChanServ, source, "Warning: Services is in read-only "
	                               "mode; changes will not be saved.");
	}
	ni = findnick(ci->founder);

	if (ni) {
	    if (ni->channelcount > 0)
		ni->channelcount--;
	} else {
	    log("%s: Can't find founder nickname record for dropping %s",
			s_ChanServ, chan);
	}

        
        for (user = userlist; user; user = user->next) {
          for(c = user->founder_chans; c && stricmp(ci->name, c->chan->name) != 0; c = c->next)
             ;
          if (c) {
             if (c->next)
                c->next->prev = c->prev;
             if (c->prev)
                c->prev->next = c->next;
             else
                user->founder_chans = c->next;
             free(c);
          }
        }
             

	delchan(ci);

        csdrop += 1;
        slog("CS D %s (%s!%s@%s)",
              chan, u->nick, u->username, u->host);
        do_break_log("CS_O", "CS D %s (%s!%s@%s)",
              chan, u->nick, u->username, u->host);
	log("%s: Channel %s dropped by %s!%s@%s", s_ChanServ, chan,
			source, u->username, u->host);
        notice(s_ChanServ, source,  CS_DROP,  chan);
    }
}

/*************************************************************************/

/* Main SET routine.  Calls other routines as follows:
 *	do_set_command(User *command-sender, ChannelInfo *ci, char *param);
 * Additional parameters can be retrieved using strtok(NULL, toks).
 * (Exception is do_set_held(), which takes only source nick and channel
 * name.)
 */
static void do_set(const char *source)
{
    NickInfo *ni = findnick(source);
    const char *chan = strtok(NULL, " ");
    const char *cmd  = strtok(NULL, " ");
    char *param;
    ChannelInfo *ci;
    User *u = finduser(source);
    int i, access=0, x=0;

    if (readonly) {
	notice(s_ChanServ, source,
		"Sorry, channel option setting is temporarily disabled.");
	return;
    }

    if (cmd) {
	if (stricmp(cmd, "DESC") == 0 || stricmp(cmd, "TOPIC") == 0 ||
            stricmp(cmd,"WELCOME") == 0)
	    param = strtok(NULL, "");
        else if (stricmp(cmd, "FOUNDER") == 0) {
            param = sstrdup(source);
            x=1;
	} else
	    param = strtok(NULL, " ");

    } else
        param = NULL;

    csset += 1;

    if (!param) {

	notice(s_ChanServ, source,
                "Syntax: \2SET <#chan> <option> [<parameters>]");
		
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET\2 for more information.", s_ChanServ);

    } else if (!ni) {
        
        notice(s_ChanServ, source,
             "Your nick is not registered!");

    } else if (!(ci = cs_findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!u || (1 != (access = is_founder(u, ni, ci, 1))
                  && (get_access(u, ci) != 13))) {

	notice(s_ChanServ, source, "Access denied.");


    } else if (stricmp(cmd, "FOUNDER") == 0) {

        if (access != 1) {
	    notice(s_ChanServ, source,
		"Password authentication required for that command.");
	    notice(s_ChanServ, source,
		"Retry after typing \2/msg %s IDENTIFY %s \37password\37.",
		s_ChanServ, chan);
	} else {
	    do_set_founder(u, ci);
	}

    } else if (stricmp(cmd, "PASSWORD") == 0) {

        if (access != 1) {
	    notice(s_ChanServ, source,
		"Password authentication required for that command.");
	    notice(s_ChanServ, source,
		"Retry after typing \2/msg %s IDENTIFY %s \37password\37.",
		s_ChanServ, chan);
	} else {
	    do_set_password(u, ci, param);
	}

    } else if (stricmp(cmd, "WELCOME") == 0) {

        do_set_welcome(u, ci, param);

    } else if (stricmp(cmd, "DESC") == 0) {

	do_set_desc(u, ci, param);

    } else if (stricmp(cmd, "URL") == 0) {

	do_set_url(u, ci, param);

    } else if (stricmp(cmd, "EMAIL") == 0) {

	do_set_email(u, ci, param);

    } else if (stricmp(cmd, "TOPIC") == 0) {

	do_set_topic(u, ci, param);

    } else if (stricmp(cmd, "MEMO") == 0) {

        do_memolev(u, ci, param);

    } else if (stricmp(cmd, "MLOCK") == 0) {

	do_set_mlock(u, ci, param);

    } else if (stricmp(cmd, "KEEPTOPIC") == 0) {

	do_set_keeptopic(u, ci, param);

    } else if (stricmp(cmd, "TOPICLOCK") == 0) {

	do_set_topiclock(u, ci, param);

    } else if (stricmp(cmd, "PRIVATE") == 0) {

	do_set_private(u, ci, param);

    } else if (stricmp(cmd, "OPGUARD") == 0) {

        do_set_opguard(u, ci, param);

    } else if (stricmp(cmd, "RESTRICTED") == 0) {

	do_set_restricted(u, ci, param);

    } else if (stricmp(cmd, "PROTECT") == 0) {

	do_set_protect(u, ci, param);

    } else if (stricmp(cmd, "IDENT") == 0) {

	do_set_ident(u, ci, param);

    } else if (stricmp(cmd, "UNSECURE") == 0) {

	do_set_unsecure(u, ci, param);

    } else {

	notice(s_ChanServ, source,
			"Unknown SET option \2%s\2.", strupper((char *)cmd));

    }
   if (x)
     free(param);
}

/*************************************************************************/

static void do_set_founder(User *u, ChannelInfo *ci)
{
    NickInfo *ni = findnick(u->nick);

    if (!ni) {
       notice(s_ChanServ, u->nick,
           "You must be using a registered nick");
    } else if (!(ni->flags & NI_IDENTIFIED)) {
       notice(s_ChanServ, u->nick,
           "You must first identify to your nick before claiming  founder");
    } else {

        if (ni->channelcount > 0)
            ni->channelcount--;
        slog("CS F %s (%s!%s@%s) [%s -> %s]",
           ci->name, u->nick, u->username, u->host, ci->founder, u->nick);
        do_break_log("CS_O", "CS F %s (%s!%s@%s) [%s -> %s]",
           ci->name, u->nick, u->username, u->host, ci->founder, u->nick);
        strscpy(ci->founder, u->nick, NICKMAX);
        notice(s_ChanServ, u->nick,
		"You are now the Founder of %s.", ci->name);
        show_u_next_db(u, s_ChanServ);
    }
}

/*************************************************************************/

static void do_set_password(User *u, ChannelInfo *ci, const char *param)
{
    char *tmp = ci->name;

    *tmp++;

    if (match_wild_nocase(param, tmp) || (strlen(param) < 5)
            || (stricmp(param, ci->name) == 0)) {

        notice(s_ChanServ, u->nick,
                "Please try again with a more obscure password.  Passwords "
                "should be at least five characters long, should not be "
                "something easily guessed (e.g. your real name or your "
                "chan name), and cannot contain the space or tab characters.  "
                "\2/msg %s HELP REGISTER\2 for more information.",
                s_ChanServ);


    } else if (strlen(param) > PASSMAX) {
           notice(s_ChanServ, u->nick,
            "Sorry, channel passwords may not exceed %d characters in length",
                 PASSMAX);
    } else {


       strscpy(ci->founderpass, param, PASSMAX);
       slog("CS P %s (%s!%s@%s)",
          ci->name, u->nick, u->username, u->host);
       do_break_log("CS_O", "CS P %s (%s!%s@%s)",
          ci->name, u->nick, u->username, u->host);

       notice(s_ChanServ, u->nick,
		"%s password changed to \2%s\2.", ci->name, ci->founderpass);
       show_u_next_db(u, s_ChanServ);
   }
}

/*************************************************************************/

static void do_set_welcome(User *u, ChannelInfo *ci, const char *param)
{
    if (ci->welcome)
        free(ci->welcome);
    if (!stricmp(param,"none")) {
	ci->welcome = NULL;
        notice(s_ChanServ, u->nick, "Welcome message disabled for %s.",
            ci->name);
        slog("CS -W %s (%s!%s@%s)",
                ci->name, u->nick, u->username, u->host);
    } else {
        ci->welcome = sstrdup(param);
        notice(s_ChanServ, u->nick,
                    "Welcome Message for %s changed to \"\2%s\2\".", ci->name, ci->welcome);
        slog("CS W %s (%s!%s@%s) [%s]",
           ci->name, u->nick, u->username, u->host, ci->welcome);

        show_u_next_db(u, s_ChanServ);
    }
}

/*************************************************************************/

static void do_set_desc(User *u, ChannelInfo *ci, const char *param)
{
    free(ci->desc);
    ci->desc = sstrdup(param);
    notice(s_ChanServ, u->nick,
		"Description of %s changed to \2%s\2.", ci->name, param);
    show_u_next_db(u, s_ChanServ);
}

/*************************************************************************/

static void do_set_url(User *u, ChannelInfo *ci, const char *param)
{

    if (stricmp(param, "NONE") == 0) {
        if (ci->url)
           free(ci->url);
        notice(s_ChanServ, u->nick,
            "URL field for channel %s has been removed", ci->name);
        return;
    }


    if (ci->url)
	free(ci->url);
    ci->url = sstrdup(param);
    notice(s_ChanServ, u->nick,
		"URL for %s changed to \2%s\2.", ci->name, param);
    show_u_next_db(u, s_ChanServ);
}

/*************************************************************************/

static void do_set_email(User *u, ChannelInfo *ci, const char *param)
{
    if (stricmp(param, "NONE") == 0) {
        if (ci->email)
           free(ci->email);
        notice(s_ChanServ, u->nick,
            "E-mail field for channel %s has been removed", ci->name);
    } else if (!strchr(param, '@')) {
        notice(s_ChanServ, u->nick,
             "E-Mail addressed must contain an '@'");
        return;

    } else {

       if (ci->email)
          free(ci->email);
       ci->email = sstrdup(param);
       notice(s_ChanServ, u->nick,
		"E-mail address for %s changed to \2%s\2.", ci->name, param);
       show_u_next_db(u, s_ChanServ);
    }
}

/*************************************************************************/

static void do_set_topic(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick, *chan = ci->name;
    Channel *c = findchan(chan);
    int ulev2 = 0, clev = 0;

    if (!c) {
	log("%s: SET TOPIC for %s from %s: channel not found!",
		s_ChanServ, chan, source);
	notice(s_ChanServ, source, "Sorry, can't set topic.");
	return;
    }

    if (ci->topic_allow > 0) {
        if (ci->topic_allow > (ulev2 = get_access(u, ci))) {
             notice(s_ChanServ, source,
                 "Access Denied");

             return;
        }
    }


    if (ci->last_topic)
	free(ci->last_topic);
    if (*param)
	ci->last_topic = sstrdup(param);
    else
	ci->last_topic = NULL;

    if (c->topic) {
	free(c->topic);
	--c->topic_time;	/* to get around TS8 */
    } else
	c->topic_time = time(NULL);

    if (*param)
	c->topic = sstrdup(param);
    else
	c->topic = NULL;

    strscpy(ci->last_topic_setter, source, NICKMAX);
    strscpy(c->topic_setter, source, NICKMAX);
    ci->last_topic_time = c->topic_time;

    send_cmd(MODE_SENDER, "TOPIC %s %s %lu :%s",
                chan, source, c->topic_time, param);

}

/*************************************************************************/

static void do_set_mlock(User *u, ChannelInfo *ci, char *param)
{
    const char *source = u->nick, *chan = ci->name;
    char *s, modebuf[32], *end, c;
    int add = -1;  /* 1 if adding, 0 if deleting, -1 if neither */
    

    ci->mlock_on = ci->mlock_off = ci->mlock_limit = 0;
    if (ci->mlock_key) {
	free(ci->mlock_key);
	ci->mlock_key = NULL;
    }
    if (ENABLE_R_MODES || ENABLE_R_MODES == 2)
        ci->mlock_on |= CMODE_R;

    while (*param) {
        char *valid = param;
        if (*valid == '-' || *valid == '+') {
          *valid++; *valid++;
        } else
          *valid++;
	if (*param != '+' && *param != '-' && add < 0) {
	    ++param;
	    continue;
	}
        while (*valid++) {
          if (*valid == *param) {
             notice(s_ChanServ, source, 
               "Ignored duplicate mode: '%c'", *valid);
             *param++;
          }
        }

	switch (c = *param++) {
          
	  case '+':
	    add = 1;
	    break;
	  case '-':
	    add = 0;
	    break;
	  case 'i':
	    if (add)
		ci->mlock_on |= CMODE_I;
	    else
		ci->mlock_off |= CMODE_I;
	    break;
	  case 'k':
	    if (add) {
		if (!(s = strtok(NULL, " "))) {
		    notice(s_ChanServ, source,
				"Parameter required for MLOCK +k.");
		    break;
		}
		ci->mlock_key = sstrdup(s);
	    } else {
		if (ci->mlock_key) {
		    free(ci->mlock_key);
		    ci->mlock_key = NULL;
		}
		ci->mlock_off |= CMODE_K;
	    }
	    break;
	  case 'l':
	    if (add) {
		if (!(s = strtok(NULL, " "))) {
		    notice(s_ChanServ, source,
				"Parameter required for MLOCK +l.");
		    break;
		}
		if (atol(s) <= 0) {
		    notice(s_ChanServ, source,
			"Parameter for MLOCK +l must be a positive number.");
		    break;
		}
		ci->mlock_limit = atol(s);
	    } else
		ci->mlock_off |= CMODE_L;
	    break;
	  case 'm':
	    if (add)
		ci->mlock_on |= CMODE_M;
	    else
		ci->mlock_off |= CMODE_M;
	    break;
	  case 'n':
	    if (add)
		ci->mlock_on |= CMODE_N;
	    else
		ci->mlock_off |= CMODE_N;
	    break;
	  case 'p':
	    if (add)
		ci->mlock_on |= CMODE_P;
	    else
		ci->mlock_off |= CMODE_P;
	    break;
	  case 's':
	    if (add)
		ci->mlock_on |= CMODE_S;
	    else
		ci->mlock_off |= CMODE_S;
	    break;
	  case 't':
	    if (add)
		ci->mlock_on |= CMODE_T;
	    else
		ci->mlock_off |= CMODE_T;
	    break;
	  case 'c':
	    if (add)
		ci->mlock_on |= CMODE_C;
	    else
		ci->mlock_off |= CMODE_C;
	    break;
          case 'j':
            if (add)
                ci->mlock_on |= CMODE_J;
            else
                ci->mlock_off |= CMODE_J;
            break;
#if BAH_V >= 147
          case 'O':
            if (add)
                ci->mlock_on |= CMODE_o;
	    else
                ci->mlock_off |= CMODE_o;
	    break;
          case 'M':
            if (add)
                ci->mlock_on |= CMODE_m;
            else
                ci->mlock_off |= CMODE_m;
            break;
#endif
          case 'R':
	    if (add)
                ci->mlock_on |= CMODE_r;
	    else
                ci->mlock_off |= CMODE_r;
	    break;


	  default:
	    notice(s_ChanServ, source,
			"Unknown mode character \2%c\2 ignored.", c);
	    break;
	} /* switch */
    } /* while (*param) */

    end = modebuf;
    *end = 0;
    if (ci->mlock_on)
        end += sprintf(end, "+%s%s%s%s%s%s%s%s%s%s%s%sr",
				(ci->mlock_on & CMODE_I) ? "i" : "",
                                (ci->mlock_on & CMODE_J) ? "j" : "",
				(ci->mlock_key         ) ? "k" : "",
				(ci->mlock_limit       ) ? "l" : "",
				(ci->mlock_on & CMODE_M) ? "m" : "",
				(ci->mlock_on & CMODE_N) ? "n" : "",
				(ci->mlock_on & CMODE_P) ? "p" : "",
				(ci->mlock_on & CMODE_S) ? "s" : "",
				(ci->mlock_on & CMODE_T) ? "t" : "",
				(ci->mlock_on & CMODE_C) ? "c" : "",
                                (ci->mlock_on & CMODE_r) ? "R" : "",
                                (ci->mlock_on & CMODE_o) ? "O" : "",
                                (ci->mlock_on & CMODE_m) ? "M" : "");

    if (ci->mlock_off)
        end += sprintf(end, "-%s%s%s%s%s%s%s%s%s%s%s%s",
				(ci->mlock_off & CMODE_I) ? "i" : "",
                                (ci->mlock_off & CMODE_J) ? "j" : "",
				(ci->mlock_off & CMODE_K) ? "k" : "",
				(ci->mlock_off & CMODE_L) ? "l" : "",
				(ci->mlock_off & CMODE_M) ? "m" : "",
				(ci->mlock_off & CMODE_N) ? "n" : "",
				(ci->mlock_off & CMODE_P) ? "p" : "",
				(ci->mlock_off & CMODE_S) ? "s" : "",
				(ci->mlock_off & CMODE_T) ? "t" : "",
				(ci->mlock_off & CMODE_C) ? "c" : "",
                                (ci->mlock_off & CMODE_r) ? "R" : "",
                                (ci->mlock_off & CMODE_o) ? "O" : "",
                                (ci->mlock_off & CMODE_m) ? "M" : "");

    if (*modebuf) {
	notice(s_ChanServ, source,
		"Mode lock on channel %s changed to \2%s\2.", chan, modebuf);
        show_u_next_db(u, s_ChanServ);
    } else {
	notice(s_ChanServ, source,
		"Mode lock on channel %s removed.", chan);
    show_u_next_db(u, s_ChanServ);
    }
  check_modes(chan);
}

/*************************************************************************/

static void do_set_keeptopic(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick, *chan = ci->name;

    if (stricmp(param, "ON") == 0) {

	ci->flags |= CI_KEEPTOPIC;
	notice(s_ChanServ, source,
		"Topic retention option is now \2ON\2.");
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "OFF") == 0) {

	ci->flags &= ~CI_KEEPTOPIC;
	notice(s_ChanServ, source,
		"Topic retention option is now \2OFF\2.");
        show_u_next_db(u, s_ChanServ);

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2SET\2 %s KEEPTOPIC [ON|OFF]", chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET KEEPTOPIC\2 for more information.",
		s_ChanServ);
    }
}

/*************************************************************************/

static void do_set_topiclock(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick, *chan = ci->name;

    if (stricmp(param, "AVOICE") == 0) {

	ci->flags |= CI_TOPICLOCK;
        ci->topic_allow = 3;
	notice(s_ChanServ, source,
                "Topic lock option is now set to \2AVOICE\2.");
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "AOP") == 0) {

	ci->flags |= CI_TOPICLOCK;
        ci->topic_allow = 5;
	notice(s_ChanServ, source,
                "Topic lock option is now set to \2AOP\2.");
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "SOP") == 0) {

	ci->flags |= CI_TOPICLOCK;
        ci->topic_allow = 10;
	notice(s_ChanServ, source,
                "Topic lock option is now set to \2SOP\2.");
        show_u_next_db(u, s_ChanServ);

    } else if ((stricmp(param, "CFOUNDER") == 0) || (stricmp(param,"CF") == 0)) {

	ci->flags |= CI_TOPICLOCK;
        ci->topic_allow = 13;
	notice(s_ChanServ, source,
                "Topic lock option is now set to \2Co-Founder\2.");
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "FOUNDER") == 0) {

	ci->flags |= CI_TOPICLOCK;
        ci->topic_allow = 15;
	notice(s_ChanServ, source,
                "Topic lock option is now set to \2FOUNDER\2.");
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "OFF") == 0) {

	ci->flags &= ~CI_TOPICLOCK;
        ci->topic_allow = 0;
	notice(s_ChanServ, source,
		"Topic lock option is now \2OFF\2.");
        show_u_next_db(u, s_ChanServ);

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2SET\2 %s TOPICLOCK [AVOICE|AOP|SOP|CFOUNDER|FOUNDER|OFF]", chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET TOPICLOCK\2 for more information.",
		s_ChanServ);
    }
}

/*************************************************************************/

static void do_set_private(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick;
    const char *chan = ci->name;

    if (stricmp(param, "ON") == 0) {

	ci->flags |= CI_PRIVATE;
	notice(s_ChanServ, source, "Private option is now \2ON\2.");
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "OFF") == 0) {

	ci->flags &= ~CI_PRIVATE;
	notice(s_ChanServ, source, "Private option is now \2OFF\2.");
        show_u_next_db(u, s_ChanServ);

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2SET\2 %s PRIVATE [ON|OFF]", chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET PRIVATE\2 for more information.",
		s_ChanServ);
    }
}

/*************************************************************************/

static void do_set_opguard(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick, *chan = ci->name;

    if (!stricmp(param, "ON")) {

	ci->flags |= CI_OPGUARD;
	notice(s_ChanServ, source,
		"Secure ops option for %s is now \2ON\2.",chan);
        show_u_next_db(u, s_ChanServ);

    } else if (!stricmp(param, "OFF")) {

	ci->flags &= ~CI_OPGUARD;
	notice(s_ChanServ, source,
		"Op Guard option for %s is now \2OFF\2.",chan);
        show_u_next_db(u, s_ChanServ);

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2SET\2 %s OPGUARD [ON|OFF]", chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET OPGUARD\2 for more information.",
		s_ChanServ);
    }
}

/*************************************************************************/

static void do_set_restricted(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick, *chan = ci->name;

    if (stricmp(param, "ON") == 0) {

	ci->flags |= CI_RESTRICTED;
	notice(s_ChanServ, source,
		"Restricted access option is now \2ON\2.");
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "OFF") == 0) {

	ci->flags &= ~CI_RESTRICTED;
	notice(s_ChanServ, source,
		"Restricted access option is now \2OFF\2.");
        show_u_next_db(u, s_ChanServ);

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2SET\2 %s RESTRICTED [ON|OFF]", chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET RESTRICTED\2 for more information.",
		s_ChanServ);
    }
}


/*************************************************************************/

static void do_set_protect(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick, *chan = ci->name;

#ifndef CS_PROTECT
    notice(s_ChanServ, source,
        "Sorry, channel protection has been disabled by the network.");
    return;
#endif


    if (stricmp(param, "ON") == 0) {

	ci->flags |= CI_PROTECTED;
	notice(s_ChanServ, source,
		"Channel OP Protection is now \2ENABLED\2.");
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "OFF") == 0) {

	ci->flags &= ~CI_PROTECTED;
	notice(s_ChanServ, source,
		"Channel OP Proection is now \2OFF\2.");
        show_u_next_db(u, s_ChanServ);

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2SET\2 %s PROTECT [ON|OFF]", chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET PROTECT\2 for more information.",
		s_ChanServ);
    }
}



/*************************************************************************/

static void do_set_ident(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick, *chan = ci->name;

    if (stricmp(param, "ON") == 0) {

	ci->flags |= CI_IDENT;
	notice(s_ChanServ, source,
		"Ident option is now \2ON\2.");

        if (ci->flags & CI_UNSECURE) {
             ci->flags &= ~CI_UNSECURE;
             notice(s_ChanServ, source,
                 "UNSECURE setting automatically disabled when IDENT set to ON");
        }

        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "OFF") == 0) {

	ci->flags &= ~CI_IDENT;
	notice(s_ChanServ, source,
		"Ident option is now \2OFF\2.");
        show_u_next_db(u, s_ChanServ);

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2SET\2 %s IDENT [ON|OFF]", chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET IDENT\2 for more information.",
		s_ChanServ);
    }
}

/*************************************************************************/

static void do_set_unsecure(User *u, ChannelInfo *ci, const char *param)
{
    const char *source = u->nick, *chan = ci->name;

    if (stricmp(param, "ON") == 0) {

        if (ci->flags & CI_IDENT) {
            notice(s_ChanServ, source,
                 "Can't set UNSECURE when IDENT is already set!");
            return;
        }

	ci->flags |= CI_UNSECURE;
	notice(s_ChanServ, source,
		"Channel %s is now UNSECURE.", chan);
        show_u_next_db(u, s_ChanServ);

    } else if (stricmp(param, "OFF") == 0) {

	ci->flags &= ~CI_UNSECURE;
	notice(s_ChanServ, source,
		"Channel %s now has regular security options enabled",
                         chan);
        show_u_next_db(u, s_ChanServ);

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2SET\2 %s UNSECURE [ON|OFF]", chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP SET UNSECURE\2 for more information.",
		s_ChanServ);
    }
}

/*************************************************************************/
static void do_cfounder(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *cmd  = strtok(NULL, " ");
    const char *nick = strtok(NULL, " ");
    char *s    = "1";
    ChannelInfo *ci;
    NickInfo *ni;
    User *u;
    short ulev;
    int is_list = (cmd && !stricmp(cmd, "LIST"));
    ChanAccess *access;
    int i, f=0, c=0, z;

    if (!cmd || ((!stricmp(cmd, "LIST") == 0) && (!stricmp(cmd, "WIPE") == 0)
             && (!stricmp(cmd, "CLEAN") == 0) && !nick)) {
         notice(s_ChanServ, source,
           "Syntax: \2CFOUNDER\2 <#channel> [ADD|DEL|LIST|WIPE|CLEAN] [<nick>]");
         return;
    }

    if (!(ci = cs_findchan(chan))) {

        notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!(u = finduser(source))) {

        notice(s_ChanServ, source, "Access denied.");
        log("%s: ACCESS command from nonexistent user %s", s_ChanServ, source);

    } else if (stricmp(cmd, "LIST") == 0) {

        if (!(is_oper(source)) &&
             (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
             && !(is_list && is_services_admin(source)))) {

           notice(s_ChanServ, source, "Access Denied");

        } else {

	unsigned int j = 0;
        notice(s_ChanServ, source, "CFOUNDER list for \2%s\2:", chan);
        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if ((nick && !match_wild(nick, access->name)) ||
                                        (13 != access->level))
                continue;
            if (ni = findnick(access->name))
                s = ni->last_usermask;
            else
                s = NULL;
	    ++j;
            notice(s_ChanServ, source, " %d)  %s%s%s%s Added By: [%s]",
                        j, access->name,
                        s ? " (" : "", s ? s : "", s ? ")" : "",
                        access->sponser ? access->sponser : "");
        }
	notice(s_ChanServ,source,"End of List");
      }

    } else if (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
            && !(is_list && is_services_root(source)))
    {
        notice(s_ChanServ, source, "Access denied.");

    } else if (s && 13 >= (ulev = get_access(u, ci))) {

        notice(s_ChanServ, source, "Access denied.");

    } else if (stricmp(cmd, "ADD") == 0) {
        char *sponsern;
        char *mask = (char *)nick;(const char *)nick;

        if (readonly) {
            notice(s_ChanServ, source,
                "Sorry, channel access list modification is temporarily disabled");
            return;
        }
        ni = findnick(nick);
        if (!ni) {
           if (strlen(mask) > 60) {
               notice(s_ChanServ, source, "ChanServ access mask too large!");
               wallops(s_ChanServ, "%s tried to add a HUGE mask to the %s Co-Founder list", source, chan);
               return;

           } else if (mask && !strchr(mask, '@') || !strchr(mask, '!')) {
               notice(s_ChanServ, source, "ChanServ access mask must be in nick!user@host or nick format");
               notice(s_ChanServ, source,
                    "\2/msg %s HELP CFOUNDER\2 for more information.", s_ChanServ);
               return;

          } else if (ci->flags & CI_IDENT) {
              notice(s_ChanServ, source, "Channel %s has the security option set. You may not add unregistered nicks when this option is enabled", ci->name);
              return;
          }

        }
        if (ni && ni->forbid) {
           notice(s_ChanServ, source, "Forbidden nick %s not permitted to be added to channel access lists", ni->nick);
           return;
        }
        if (ni && (ni->flags & NI_AUTH || !ni->regemail)
           && accsecure && auth_email) {
            notice(s_ChanServ, source,
                 "Unauthorized nicks are not permitted to be added to "
                 "channel access lists. Talk to %s about authorizing "
                 "their nick.", ni->nick);
           return;
        }


        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (access->in_use && stricmp(access->name, nick) == 0) {
                if (access->level >= ulev) {
                    notice(s_ChanServ, source, "Permission denied.");
                    return;
                }
                if (access->level == 13) {
                    notice(s_ChanServ, source,
                        "\2%s\2 is already a CFOUNDER on \2%s\2.",
                        access->name, chan);
                    return;
                }
                access->level = 13;
                if (access->sponser)
                    free(access->sponser);
                if ((sponsern = get_sponser(u, ci)) != NULL)
                    access->sponser = sstrdup(sponsern);


                if (ni && (ni->flags & NI_NOOP))
                    notice(s_ChanServ, source, "Since %s is already on the %s access list, regardless of NOOP setting, adding to Co-Founder list",
                       ni->nick, chan);
                else
                   notice(s_ChanServ, source,
                        "\2%s\2 was changed to a Co-Founder on \2%s\2.",
                        access->name, chan);
                show_next_db(source, s_ChanServ);
                return;
            }
        }
        if ((CHAN_ACCESS_MAX > 0) && (ci->accesscount > CHAN_ACCESS_MAX-1)) {
	    notice(s_ChanServ, source,
                    "Sorry, you may only have %d channel access enteries.",
                    CHAN_ACCESS_MAX);
            wallops(s_ChanServ, "%s tried to add another nick to the access list on %s",
                   source, chan);
	    return;
	}
        if (ni && (ni->flags & NI_NOOP)) {
             notice(s_ChanServ, source, "Sorry, %s has the NOOP option set.", ni->nick);
             return;
        }
        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (!access->in_use)
                break;
        }
        if (i == ci->accesscount) {
            ++ci->accesscount;
               ci->access =
                srealloc(ci->access, sizeof(ChanAccess) * ci->accesscount);

            access = &ci->access[ci->accesscount-1];
        }
        if (ni)
           access->name = sstrdup(ni->nick);
        else
           access->name = sstrdup(mask);
           if ((sponsern = get_sponser(u, ci)) != NULL) 
           access->sponser = sstrdup(sponsern);
        access->in_use = 1;
        access->level = 13;
        notice(s_ChanServ, source,
                "\2%s\2 added to \2%s\2 as a Co-Founder.",
                access->name, chan);
        show_next_db(source, s_ChanServ);

    } else if (stricmp(cmd, "DEL") == 0) {
        if (readonly) {
            notice(s_ChanServ, source,
                "Sorry, channel access list modification is temporarily disabled");
            return;
        }
        /* Special case: is it a number?  Only do search if it isn't. */
        if (strspn(nick, "1234567890") == strlen(nick) &&
                                (i = atoi(nick)) > 0 && i <= ci->accesscount) {
            unsigned int j=0;
            for (access = ci->access, z = 0; z < ci->accesscount;
			++access, ++z) {
                if (!(13 == access->level))
                    continue;
	        j += 1;
                if (i == j) {
		    access = &ci->access[z];
		    break;
		}
            }
            if (!(i == j)) {
                notice(s_ChanServ, source,
                        "No such entry (#%d) on %s access list.", i, chan);
                return;
            }
        } else {
/*
            ni = findnick(nick);
            if (!ni) {
                notice(s_ChanServ, source,
                        "Nick \2%s\2 not registered.", nick);
                return;
            }
*/
            for (access = ci->access, z = 0; z < ci->accesscount;
                                                        ++access, ++z) {
                if ((stricmp(access->name, nick) == 0)
                        && (access->level == 13))
                    break;
            }
            if (z == ci->accesscount) {
                notice(s_ChanServ, source,
                        "\2%s\2 not found on %s Co-Founder list.", nick, chan);
                return;
            }
        }
        if (get_access(u, ci) <= access->level) {
            notice(s_ChanServ, source, "Permission denied.");
        } else {
            notice(s_ChanServ, source,
                "\2%s\2 deleted from %s access list.", access->name, chan);
            show_next_db(source, s_ChanServ);
            free(access->name);
            free(access->sponser);
            access->in_use = 0;
            access->name = NULL;
              --ci->accesscount;
            if (z < ci->accesscount)
                bcopy(access+1, access, sizeof(ChanAccess)
                                         * (ci->accesscount - z));
            if (!ci->accesscount) {
                free(ci->access);
                ci->access = NULL;
            }
        }

    } else if (stricmp(cmd, "CLEAN") == 0) {
        int check = 0;
        check = ci->accesscount;
        for (access = ci->access, i = 0; i < check; ++access, ++i) {
              if ((13 == access->level) && (!strchr(access->name, '@'))
                 && (!findnick(access->name))) {
                     free(access->name);
                     free(access->sponser);
                     access->name = NULL;
                     access->in_use = 0;
                     ++f;
                     --ci->accesscount;
                     if ((i-f) < ci->accesscount) {
                         bcopy(access+1, access, sizeof(ChanAccess)
                                        * (ci->accesscount - (i-f)));
                         --access;
                     }
              }
        }
        if (!(f > 0))
            notice(s_ChanServ, source,
               "No Expired nicknames found in the CFOUNDER list on \2%s\2",
                chan);
        else {
           if (!ci->accesscount) {
                 free(ci->access);
                 ci->access = NULL;
           }
           notice(s_ChanServ, source,
               "%d expired nicknames in the CFOUNDER list on \2%s\2 removed",
                f, chan);
           show_next_db(source, s_ChanServ);
       }
       f=0;


    } else if (stricmp(cmd, "WIPE") == 0) {
        int check = 0;
        check = ci->accesscount;
        for (access = ci->access, i = 0; i < check; ++access, ++i) {
              if (13 == access->level) {
                  free(access->name);
                  free(access->sponser);
                  access->name = NULL;
                  access->in_use = 0;
                  ++f;
              }
        }
        ci->accesscount = 0;
        if (!(f > 0))
            notice(s_ChanServ, source,
               "No enteries found in the CFOUNDER list on \2%s\2",
                chan);

        else {
           free(ci->access);
           ci->access = NULL;
           notice(s_ChanServ, source,
               "%d enter%s in the CFOUNDER list on \2%s\2 removed",
                f, f==1 ? "y" : "ies", chan);
           show_next_db(source, s_ChanServ);
       }
       f=0;


    } else {
        notice(s_ChanServ, source,
                "Syntax: \2CFOUNDER\2 %s [ADD|DEL|LIST|WIPE|CLEAN] [<nick>]\2",chan);
        notice(s_ChanServ, source,
                "\2/msg %s HELP SOP\2 for more information.", s_ChanServ);
    }
}

static void do_avoice(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *cmd  = strtok(NULL, " ");
    const char *nick = strtok(NULL, " ");
    char *s    = "1";
    ChannelInfo *ci;
    NickInfo *ni;
    User *u;
    short ulev;
    int is_list = (cmd && !stricmp(cmd, "LIST"));
    ChanAccess *access;
    int i, f=0, c=0, z;

    if (!cmd || ((!stricmp(cmd, "LIST") == 0) && (!stricmp(cmd, "WIPE") == 0)
             && (!stricmp(cmd, "CLEAN") == 0) && !nick)) {
         notice(s_ChanServ, source,
           "Syntax: \2AVOICE\2 <#channel> [ADD|DEL|LIST|WIPE|CLEAN] [<nick>]");
         return;
    }

    if (!(ci = cs_findchan(chan))) {

        notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!(u = finduser(source))) {

        notice(s_ChanServ, source, "Access denied.");
        log("%s: ACCESS command from nonexistent user %s", s_ChanServ, source);

    } else if (stricmp(cmd, "LIST") == 0) {

        if (!(is_oper(source)) &&
             (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
             && !(is_list && is_services_admin(source)))) {

           notice(s_ChanServ, source, "Access Denied");

        } else {

	unsigned int j=0;
        notice(s_ChanServ, source, "AVOICE list for \2%s\2:", chan);
        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if ((nick && !match_wild(nick, access->name)) ||
                                        (3 != access->level))
                continue;
            if (ni = findnick(access->name))
                s = ni->last_usermask;
            else
                s = NULL;
	    ++j;
            notice(s_ChanServ, source, " %d)  %s%s%s%s Added By: [%s]",
                        j, access->name,
                        s ? " (" : "", s ? s : "", s ? ")" : "",
                        access->sponser ? access->sponser : "");
        }
        notice(s_ChanServ,source,"End of List");
      }

    } else if (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
            && !(is_list && is_services_admin(source)))
    {
        notice(s_ChanServ, source, "Access denied.");

    } else if (s && 5 >= (ulev = get_access(u, ci))) {

        notice(s_ChanServ, source, "Access denied.");

    } else if (stricmp(cmd, "ADD") == 0) {
        char *sponsern;
        char *mask = (char *)nick; (const char *)nick;

        if (readonly) {
            notice(s_ChanServ, source,
                "Sorry, channel access list modification is temporarily disabled");
            return;
        }
        ni = findnick(nick);
        if (!ni) {
           if (strlen(mask) > 60) {
               notice(s_ChanServ, source, "ChanServ access mask too large!");
               wallops(s_ChanServ, "%s tried to add a HUGE mask to the %s AVOICE list", source, chan);
               return;

           } else if (mask && !strchr(mask, '@') || !strchr(mask, '!')) {
               notice(s_ChanServ, source, "ChanServ access mask must be in nick!user@host or nick format");
               notice(s_ChanServ, source,
                    "\2/msg %s HELP AVOICE\2 for more information.", s_ChanServ);
               return;

          } else if (ci->flags & CI_IDENT) {
              notice(s_ChanServ, source, "Channel %s has the security option set. You may not add unregistered nicks when this option is enabled", ci->name);
              return;
          }

        }
        if (ni && ni->forbid) {
           notice(s_ChanServ, source, "Forbidden nick %s not permitted to be added to channel access lists", ni->nick);
           return;
        }
        if (ni && (ni->flags & NI_AUTH || !ni->regemail)
           && accsecure && auth_email) {
            notice(s_ChanServ, source,
                 "Unauthorized nicks are not permitted to be added to "
                 "channel access lists. Talk to %s about authorizing "
                 "their nick.", ni->nick);
           return;
        }

        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (access->in_use && stricmp(access->name, nick) == 0) {
                if (access->level >= ulev) {
                    notice(s_ChanServ, source, "Permission denied.");
                    return;
                }
                if (access->level == 3) {
                    notice(s_ChanServ, source,
                        "\2%s\2 is already an Auto-Voice on \2%s\2.",
                        access->name, chan);
                    return;
                }
                access->level = 3;
                if (access->sponser)
                    free(access->sponser);
                if ((sponsern = get_sponser(u, ci)) != NULL)
                    access->sponser = sstrdup(sponsern);

                if (ni && (ni->flags & NI_NOOP))
                    notice(s_ChanServ, source, "Since %s is already on the %s access list, regardless of NOOP setting, adding to AVOICE list",
                       ni->nick, chan);
                else
                   notice(s_ChanServ, source,
                        "\2%s\2 was changed to an Auto-Voice on \2%s\2.",
                        access->name, chan);
                show_next_db(source, s_ChanServ);
                return;
            }
        }
        if ((CHAN_ACCESS_MAX > 0) && (ci->accesscount > CHAN_ACCESS_MAX-1)) {
	    notice(s_ChanServ, source,
                    "Sorry, you may only have %d channel access enteries.",
                    CHAN_ACCESS_MAX);
            wallops(s_ChanServ, "%s tried to add another nick to the access list on %s",
                   source, chan);
	    return;
	}
        if (ni && (ni->flags & NI_NOOP)) {
             notice(s_ChanServ, source, "Sorry, %s has the NOOP Option set",
                       ni->nick);
             return;
        }


        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (!access->in_use)
                break;
        }
        if (i == ci->accesscount) {
            ++ci->accesscount;
             ci->access =
                srealloc(ci->access, sizeof(ChanAccess) * ci->accesscount);

            access = &ci->access[ci->accesscount-1];
        }
        if (ni)
           access->name = sstrdup(ni->nick);
        else
           access->name = sstrdup(mask);
           if ((sponsern = get_sponser(u, ci)) != NULL)
               access->sponser = sstrdup(sponsern);

        access->in_use = 1;
        access->level = 3;
        notice(s_ChanServ, source,
                "\2%s\2 added to \2%s\2 as an Auto-Voice.",
                access->name, chan);
        show_next_db(source, s_ChanServ);

    } else if (stricmp(cmd, "DEL") == 0) {

        if (readonly) {
            notice(s_ChanServ, source,
                "Sorry, channel access list modification is temporarily disabled");
            return;
        }
        /* Special case: is it a number?  Only do search if it isn't. */
        if (strspn(nick, "1234567890") == strlen(nick) &&
                                (i = atoi(nick)) > 0 && i <= ci->accesscount) {
            unsigned int j=0;
            for (access = ci->access, z = 0; z < ci->accesscount;
                        ++access, ++z) {
                if (!(3 == access->level))
                    continue;
                j += 1;
                if (i == j) {
                    access = &ci->access[z];
                    break;
                }
            }
            if (!(i == j)) {
                notice(s_ChanServ, source,
                        "No such entry (#%d) on %s access list.", i, chan);
                return;
            }
        } else {
/*
            ni = findnick(nick);
            if (!ni) {
                notice(s_ChanServ, source,
                        "Nick \2%s\2 not registered.", nick);
                return;
            }
*/
            for (access = ci->access, z = 0; z < ci->accesscount;
                                                        ++access, ++z) {
                if ((stricmp(access->name, nick) == 0)
                        && (access->level == 3))
                    break;
            }
            if (z == ci->accesscount) {
                notice(s_ChanServ, source,
                        "\2%s\2 not found on %s AVOICE list.", nick, chan);
                return;
            }
        }
        if (get_access(u, ci) <= access->level) {
            notice(s_ChanServ, source, "Permission denied.");
        } else {
            notice(s_ChanServ, source,
                "\2%s\2 deleted from %s access list.", access->name, chan);
            show_next_db(source, s_ChanServ);
            free(access->name);
            free(access->sponser);
            access->in_use = 0;
            access->name = NULL;
              --ci->accesscount;
            if (z < ci->accesscount)
                bcopy(access+1, access, sizeof(ChanAccess)
                                         * (ci->accesscount-z));
            if (!ci->accesscount) {
                free(ci->access);
                ci->access = NULL;
            }

        }


    } else if (stricmp(cmd, "CLEAN") == 0) {
        int check = 0;
        check = ci->accesscount;
        for (access = ci->access, i = 0; i < check; ++access, ++i) {
              if ((3 == access->level) && (!strchr(access->name, '@'))
                 && (!findnick(access->name))) {
                     free(access->name);
                     free(access->sponser);
                     access->name = NULL;
                     access->in_use = 0;
                     ++f;
                     --ci->accesscount;
                     if ((i-f) < ci->accesscount) {
                         bcopy(access+1, access, sizeof(ChanAccess)
                                        * (ci->accesscount - (i-f)));
                         --access;
                      }
              }
        }
        if (!(f > 0))
            notice(s_ChanServ, source,
               "No Expired nicknames found in the AVOICE list on \2%s\2",
                chan);
        else {
           if (!ci->accesscount) {
                 free(ci->access);
                 ci->access = NULL;
           }
           notice(s_ChanServ, source,
               "%d expired nicknames in the AVOICE list on \2%s\2 removed",
                f, chan);
           show_next_db(source, s_ChanServ);
       }
       f=0;


    } else if (stricmp(cmd, "WIPE") == 0) {
        int check = 0;
        check = ci->accesscount;
        for (access = ci->access, i = 0; i < check; ++access, ++i) {
              if (3 == access->level) {
                  free(access->name);
                  free(access->sponser);
                  access->name = NULL;
                  access->in_use = 0;
                  ++f;
              }
        }
        ci->accesscount = 0;
        if (!(f > 0))
            notice(s_ChanServ, source,
               "No enteries found in the AVOICE list on \2%s\2",
                chan);

        else {
           free(ci->access);
           ci->access = NULL;
           notice(s_ChanServ, source,
               "%d entr%s in the AVOICE list on \2%s\2 removed",
                f, f==1 ? "y" : "ies", chan);
           show_next_db(source, s_ChanServ);
       }
       f=0;


    } else {

        notice(s_ChanServ, source,
                "Syntax: \2AVOICE\2 %s [ADD|DEL|LIST|WIPE|CLEAN] [<nick>]",chan);
        notice(s_ChanServ, source,
                "\2/msg %s HELP AVOICE\2 for more information.", s_ChanServ);
    }
}

static void do_sop(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *cmd  = strtok(NULL, " ");
    const char *nick = strtok(NULL, " ");
    char *s    = "1";
    ChannelInfo *ci;
    NickInfo *ni;
    User *u;
    short ulev;
    int is_list = (cmd && !stricmp(cmd, "LIST"));
    ChanAccess *access;
    int i, f=0, c=0, z;

    if (!cmd || ((!stricmp(cmd, "LIST") == 0) && (!stricmp(cmd, "WIPE") == 0)
             && (!stricmp(cmd, "CLEAN") == 0) && !nick)) {
         notice(s_ChanServ, source,
           "Syntax: \2SOP\2 <#channel> [ADD|DEL|LIST|WIPE|CLEAN] [<nick>]");
         return;
    }

    if (!(ci = cs_findchan(chan))) {

        notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!(u = finduser(source))) {

        notice(s_ChanServ, source, "Access denied.");
        log("%s: ACCESS command from nonexistent user %s", s_ChanServ, source);

    } else if (stricmp(cmd, "LIST") == 0) {

        if (!(is_oper(source)) &&
             (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
             && !(is_list && is_services_admin(source)))) {

           notice(s_ChanServ, source, "Access Denied");

        } else {


	unsigned int j=0;
        notice(s_ChanServ, source, "SOP list for \2%s\2:", chan);
        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if ((nick && !match_wild(nick, access->name)) ||
                                        (10 != access->level))
                continue;
            if (ni = findnick(access->name))
                s = ni->last_usermask;
            else
                s = NULL;
	    ++j;
            notice(s_ChanServ, source, " %d)  %s%s%s%s Added By: [%s]",
                        j, access->name,
                        s ? " (" : "", s ? s : "", s ? ")" : "",
                        access->sponser ? access->sponser : "");
        }
        notice(s_ChanServ,source,"End of List");

      }

    } else if (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
            && !(is_list && is_services_admin(source)))
    {
        notice(s_ChanServ, source, "Access denied.");

    } else if (s && 10 >= (ulev = get_access(u, ci))) {

        notice(s_ChanServ, source, "Access denied.");

    } else if (stricmp(cmd, "ADD") == 0) {
        char *sponsern;
        char *mask = (char *)nick;(const char *)nick;

        if (readonly) {
            notice(s_ChanServ, source,
                "Sorry, channel access list modification is temporarily disabled");
            return;
        }
        ni = findnick(nick);
        if (!ni) {
           if (strlen(mask) > 60) {
               notice(s_ChanServ, source, "ChanServ access mask too large!");
               wallops(s_ChanServ, "%s tried to add a HUGE mask to the %s SOP list", source, chan);
               return;

           } else if (mask && !strchr(mask, '@') || !strchr(mask, '!')) {
               notice(s_ChanServ, source, "ChanServ access mask must be in nick!user@host or nick format");
               notice(s_ChanServ, source,
                    "\2/msg %s HELP SOP\2 for more information.", s_ChanServ);
               return;
          } else if (ci->flags & CI_IDENT) {
              notice(s_ChanServ, source, "Channel %s has the security option set. You may not add unregistered nicks when this option is enabled", ci->name);
              return;
          }
        }

        if (ni && ni->forbid) {
           notice(s_ChanServ, source, "Forbidden nick %s not permitted to be added to channel access lists", ni->nick);
           return;
        }

        if (ni && (ni->flags & NI_AUTH || !ni->regemail)
           && accsecure && auth_email) {
            notice(s_ChanServ, source,
                 "Unauthorized nicks are not permitted to be added to "
                 "channel access lists. Talk to %s about authorizing "
                 "their nick.", ni->nick);
           return;
        }

        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (access->in_use && stricmp(access->name, nick) == 0) {
                if (access->level >= ulev) {
                    notice(s_ChanServ, source, "Permission denied.");
                    return;
                }
                if (access->level == 10) {
                    notice(s_ChanServ, source,
                        "\2%s\2 is already a SOP on \2%s\2.",
                        access->name, chan);
                    return;
                }

                access->level = 10;
                if (access->sponser)
                    free(access->sponser);
                if ((sponsern = get_sponser(u, ci)) != NULL)
                    access->sponser = sstrdup(sponsern);

                if (ni && (ni->flags & NI_NOOP))
                    notice(s_ChanServ, source, "Since %s is already on the %s access list, regardless of NOOP setting, adding to SOP list",
                       ni->nick, chan);
                else
                    notice(s_ChanServ, source,
                        "\2%s\2 was changed to a SOP on \2%s\2.",
                        access->name, chan);
                show_next_db(source, s_ChanServ);
                return;
            }
        }
        if ((CHAN_ACCESS_MAX > 0) && (ci->accesscount > CHAN_ACCESS_MAX-1)) {
	    notice(s_ChanServ, source,
                    "Sorry, you may only have %d channel access enteries.",
                    CHAN_ACCESS_MAX);
            wallops(s_ChanServ, "%s tried to add another nick to the access list on %s",
                   source, chan);
	    return;
	}
        if (ni && (ni->flags & NI_NOOP)) {
             notice(s_ChanServ, source, "Sorry, %s has the NOOP option set.", ni->nick);
             return;
        }
        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (!access->in_use)
                break;
        }
        if (i == ci->accesscount) {
            ++ci->accesscount;
            ci->access =
                srealloc(ci->access, sizeof(ChanAccess) * ci->accesscount);
            access = &ci->access[ci->accesscount-1];
        }

        if (ni)
           access->name = sstrdup(ni->nick);
        else
           access->name = sstrdup(mask);
           if ((sponsern = get_sponser(u, ci)) != NULL)
                 access->sponser = sstrdup(sponsern);


        access->in_use = 1;
        access->level = 10;

        notice(s_ChanServ, source,
                "\2%s\2 added to \2%s\2 as a SOP.",
                access->name, chan);
        show_next_db(source, s_ChanServ);

    } else if (stricmp(cmd, "DEL") == 0) {

        if (readonly) {
            notice(s_ChanServ, source,
                "Sorry, channel access list modification is temporarily disabled");
            return;
        }
        /* Special case: is it a number?  Only do search if it isn't. */
        if (strspn(nick, "1234567890") == strlen(nick) &&
                                (i = atoi(nick)) > 0 && i <= ci->accesscount) {
            unsigned int j=0;
            for (access = ci->access, z = 0; z < ci->accesscount;
                        ++access, ++z) {
                if (!(10 == access->level))
                    continue;
                j += 1;
                if (i == j) {
                    access = &ci->access[z];
                    break;
                }
            }
            if (!(i == j)) {
                notice(s_ChanServ, source,
                        "No such entry (#%d) on %s access list.", i, chan);
                return;
            }
        } else {
/*
            ni = findnick(nick);
            if (!ni) {
                notice(s_ChanServ, source,
                        "Nick \2%s\2 not registered.", nick);
                return;
            }
*/
            for (access = ci->access, z = 0; z < ci->accesscount;
                                                        ++access, ++z) {
                if ((stricmp(access->name, nick) == 0)
                        && (access->level == 10))
                    break;
            }
            if (z == ci->accesscount) {
                notice(s_ChanServ, source,
                        "\2%s\2 not found on %s SOP list.", nick, chan);
                return;
            }
        }
        if (get_access(u, ci) <= access->level) {
            notice(s_ChanServ, source, "Permission denied.");
        } else {
            notice(s_ChanServ, source,
                "\2%s\2 deleted from %s access list.", access->name, chan);
            show_next_db(source, s_ChanServ);
            free(access->name);
            free(access->sponser);
            access->in_use = 0;
            access->name = NULL;
              --ci->accesscount;
            if (z < ci->accesscount)
                bcopy(access+1, access, sizeof(ChanAccess)
                                      * (ci->accesscount - z));
            if (!ci->accesscount) {
                free(ci->access);
                ci->access = NULL;
            }

        }


    } else if (stricmp(cmd, "CLEAN") == 0) {
        int check = 0;
        check = ci->accesscount;
        for (access = ci->access, i = 0; i < check; ++access, ++i) {
              if ((10 == access->level) && (!strchr(access->name, '@'))
                 && (!findnick(access->name))) {
                     free(access->name);
                     free(access->sponser);
                     access->name = NULL;
                     access->in_use = 0;
                     ++f;
                     --ci->accesscount;
                     if ((i-f) < ci->accesscount) {
                         bcopy(access+1, access, sizeof(ChanAccess)
                                        * (ci->accesscount - (i-f)));
                         --access;
                     } 
             }
        }
        if (!(f > 0))
            notice(s_ChanServ, source,
               "No Expired nicknames found in the SOP list on \2%s\2",
                chan);
        else {
           if (!ci->accesscount) {
                 free(ci->access);
                 ci->access = NULL;
           }
           notice(s_ChanServ, source,
               "%d expired nicknames in the SOP list on \2%s\2 removed",
                f, chan);
           show_next_db(source, s_ChanServ);
       }
       f=0;



    } else if (stricmp(cmd, "WIPE") == 0) {
        int check = 0;
        check = ci->accesscount;
        for (access = ci->access, i = 0; i < check; ++access, ++i) {
              if (10 == access->level) {
                  free(access->name);
                  free(access->sponser);
                  access->name = NULL;
                  access->in_use = 0;
                  ++f;
              }
        }
        ci->accesscount = 0;
        if (!(f > 0))
            notice(s_ChanServ, source,
               "No enteries found in the SOP list on \2%s\2",
                chan);


        else {
           free(ci->access);
           ci->access = NULL;
           notice(s_ChanServ, source,
               "%d entr%s in the SOP list on \2%s\2 removed",
                f, f==1 ? "y" : "ies", chan);
           show_next_db(source, s_ChanServ);
       }
       f=0;





    } else {

        notice(s_ChanServ, source,
                "Syntax: \2SOP\2 %s [ADD|DEL|LIST|WIPE|CLEAN] [<nick>]",chan);
        notice(s_ChanServ, source,
                "\2/msg %s HELP SOP\2 for more information.", s_ChanServ);
    }
}


static void do_aop(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *cmd  = strtok(NULL, " ");
    const char *nick = strtok(NULL, " ");
    char *s    = "1";
    ChannelInfo *ci;
    NickInfo *ni;
    User *u;
    short ulev;
    int is_list = (cmd && !stricmp(cmd, "LIST"));
    ChanAccess *access;
    int i, f=0, c=0, z;

    if (!cmd || ((!stricmp(cmd, "LIST") == 0) && (!stricmp(cmd, "WIPE") == 0)
             && (!stricmp(cmd, "CLEAN") == 0) && !nick)) {
         notice(s_ChanServ, source,
           "Syntax: \2AOP <#channel> [ADD|DEL|LIST|WIPE|CLEAN] [<nick>]");
         return;
    }

    if (!(ci = cs_findchan(chan))) {

        notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!(u = finduser(source))) {

        notice(s_ChanServ, source, "Access denied.");
        log("%s: ACCESS command from nonexistent user %s", s_ChanServ, source);


    } else if (stricmp(cmd, "LIST") == 0) {

        if (!(is_oper(source)) &&
             (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
             && !(is_list && is_services_admin(source)))) {

           notice(s_ChanServ, source, "Access Denied");

        } else {

	unsigned int j=0;
        notice(s_ChanServ, source, "AOP list for \2%s\2:", chan);
        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if ((nick && !match_wild(nick, access->name)) ||
                                        (5 != access->level))
                continue;
            if (ni = findnick(access->name))
                s = ni->last_usermask;
            else
                s = NULL;
	    ++j;
            notice(s_ChanServ, source, " %d)  %s%s%s%s Added By: [%s]",
                        j, access->name,
                        s ? " (" : "", s ? s : "", s ? ")" : "",
                        access->sponser ? access->sponser : "");
        }

        notice(s_ChanServ,source,"End of List");

      }


    } else if (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
            && !(is_list && is_services_admin(source)))
    {
        notice(s_ChanServ, source, "Access denied.");

    } else if (s && 5 >= (ulev = get_access(u, ci))) {

        notice(s_ChanServ, source, "Access denied.");

    } else if (stricmp(cmd, "ADD") == 0) {
        char *sponsern;
        char *mask = (char *)nick;(const char *)nick;

        if (readonly) {
            notice(s_ChanServ, source,
                "Sorry, channel access list modification is temporarily disabled");
            return;
        }
        ni = findnick(nick);
        if (!ni) {
           if (strlen(mask) > 60) {
               notice(s_ChanServ, source, "ChanServ access mask too large!");
               wallops(s_ChanServ, "%s tried to add a HUGE mask to the %s AOP list", source, chan);
               return;

           } else if (mask && !strchr(mask, '@') || !strchr(mask, '!')) {
               notice(s_ChanServ, source, "ChanServ access mask must be in nick!user@host or nick format");
               notice(s_ChanServ, source,
                    "\2/msg %s HELP AOP\2 for more information.", s_ChanServ);
               return;
          } else if (ci->flags & CI_IDENT) {
              notice(s_ChanServ, source, "Channel %s has the security option set. You may not add unregistered nicks when this option is enabled", ci->name);
              return;
          }
      }
      if (ni && ni->forbid) {
           notice(s_ChanServ, source, "Forbidden nick %s not permitted to be added to channel access lists", ni->nick);
           return;
        }

      if (ni && (ni->flags & NI_AUTH || !ni->regemail)
         && accsecure && auth_email) {
          notice(s_ChanServ, source,
               "Unauthorized nicks are not permitted to be added to "
               "channel access lists. Talk to %s about authorizing "
               "their nick.", ni->nick);
         return;
      }

        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (access->in_use && stricmp(access->name, nick) == 0) {
                if (access->level >= ulev) {
                    notice(s_ChanServ, source, "Permission denied.");
                    return;
                }
                if (access->level == 5) {
                    notice(s_ChanServ, source,
                        "\2%s\2 is already an AOP on \2%s\2.",
                        access->name, chan);
                    return;
                }
                access->level = 5;
                if (access->sponser)
                     free(access->sponser);
                if ((sponsern = get_sponser(u, ci)) != NULL)
                    access->sponser = sstrdup(sponsern);
                if (ni && (ni->flags & NI_NOOP))
                    notice(s_ChanServ, source, "Since %s is already on the %s access list, regardless of NOOP setting, adding to AOP list",
                       ni->nick, chan);
                else
                    notice(s_ChanServ, source,
                        "\2%s\2 was changed to an AOP on \2%s\2.",
                        access->name, chan);
                show_next_db(source, s_ChanServ);
                return;
            }
        }
        if ((CHAN_ACCESS_MAX > 0) && (ci->accesscount > CHAN_ACCESS_MAX-1)) {
	    notice(s_ChanServ, source,
                    "Sorry, you may only have %d channel access enteries.",
                    CHAN_ACCESS_MAX);
            wallops(s_ChanServ, "%s tried to add another nick to the access list on %s",
                   source, chan);
	    return;
	}
        if (ni && (ni->flags & NI_NOOP)) {
             notice(s_ChanServ, source,
                 "Sorry, %s has the NOOP option set", ni->nick);
             return;
        }

        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (!access->in_use)
                break;
        }
        if (i == ci->accesscount) {
            ++ci->accesscount;
             ci->access =
               srealloc(ci->access, sizeof(ChanAccess) * ci->accesscount);

            access = &ci->access[ci->accesscount-1];
        }
        if (ni)
           access->name = sstrdup(ni->nick);
        else
           access->name = sstrdup(mask);
           if ((sponsern = get_sponser(u, ci)) != NULL)
               access->sponser = sstrdup(sponsern);
        access->in_use = 1;
        access->level = 5;
        notice(s_ChanServ, source,
                "\2%s\2 added to \2%s\2 as an AOP.",
                access->name, chan);
        show_next_db(source, s_ChanServ);

    } else if (stricmp(cmd, "DEL") == 0) {

        if (readonly) {
            notice(s_ChanServ, source,
                "Sorry, channel access list modification is temporarily disabled");
            return;
        }
        /* Special case: is it a number?  Only do search if it isn't. */
        if (strspn(nick, "1234567890") == strlen(nick) &&
                                (i = atoi(nick)) > 0 && i <= ci->accesscount) {
            unsigned int j=0;
            for (access = ci->access, z = 0; z < ci->accesscount;
                        ++access, ++z) {
                if (!(5 == access->level))
                    continue;
                j += 1;
                if (i == j) {
                    access = &ci->access[z];
                    break;
                }
            }
            if (!(i == j)) {
                notice(s_ChanServ, source,
                        "No such entry (#%d) on %s access list.", i, chan);
                return;
            }
        } else {
/*
            ni = findnick(nick);
            if (!ni) {
                notice(s_ChanServ, source,
                        "Nick \2%s\2 not registered.", nick);
                return;
            }
*/
            for (access = ci->access, z = 0; z < ci->accesscount;
                                                        ++access, ++z) {
                if ((stricmp(access->name, nick) == 0)
			&& (access->level == 5))
                    break;
            }
            if (z == ci->accesscount) {
                notice(s_ChanServ, source,
                        "\2%s\2 not found on %s AOP list.", nick, chan);
                return;
            }
        }
        if (get_access(u, ci) <= access->level) {
            notice(s_ChanServ, source, "Permission denied.");
        } else {
            notice(s_ChanServ, source,
                "\2%s\2 deleted from %s access list.", access->name, chan);
            show_next_db(source, s_ChanServ);
            free(access->name);
            free(access->sponser);
            access->in_use = 0;
            access->name = NULL;
              --ci->accesscount;
            if (z < ci->accesscount)
                bcopy(access+1, access, sizeof(ChanAccess)
                                      * (ci->accesscount - z));
            if (!ci->accesscount) {
                 free(ci->access);
                 ci->access = NULL;
            }
        }


    } else if (stricmp(cmd, "CLEAN") == 0) {
        int check = 0;
        check = ci->accesscount;
        for (access = ci->access, i = 0; i < check; ++access, ++i) {
              if ((5 == access->level) && (!strchr(access->name, '@'))
                 && (!findnick(access->name))) {
                     free(access->name);
                     free(access->sponser);
                     access->name = NULL;
                     access->in_use = 0;
                     ++f;
                     --ci->accesscount;
                     if ((i-f) < ci->accesscount) {
                         bcopy(access+1, access, sizeof(ChanAccess)
                                        * (ci->accesscount - (i-f)));
                         --access;
                     }
              }
        }
        if (!(f > 0))
            notice(s_ChanServ, source,
               "No Expired nicknames found in the AOP list on \2%s\2",
                chan);
        else {
           if (!ci->accesscount) {
                 free(ci->access);
                 ci->access = NULL;
           }
           notice(s_ChanServ, source,
               "%d expired nicknames in the AOP list on \2%s\2 removed",
                f, chan);
           show_next_db(source, s_ChanServ);
       }
       f=0;


    } else if (stricmp(cmd, "WIPE") == 0) {
        int check = 0;
        check = ci->accesscount;
        for (access = ci->access, i = 0; i < check; ++access, ++i) {
              if (5 == access->level) {
                  free(access->name);
                  free(access->sponser);
                  access->name = NULL;
                  access->in_use = 0;
                  ++f;
              }
        }
        ci->accesscount = 0;
        if (!(f > 0))
            notice(s_ChanServ, source,
               "No enteries found in the AOP list on \2%s\2",
                chan);

        else {
           free(ci->access);
           ci->access = NULL;
           notice(s_ChanServ, source,
               "%d entr%s in the AOP list on \2%s\2 removed",
                f, f==1 ? "y" : "ies", chan);
           show_next_db(source, s_ChanServ);
       }
       f=0;

    } else {

        notice(s_ChanServ, source,
                "Syntax: \2AOP\2 %s [ADD|DEL|LIST|WIPE|CLEAN] [<nick>]",chan);
        notice(s_ChanServ, source,
                "\2/msg %s HELP AOP\2 for more information.", s_ChanServ);
    }
}

/*************************************************************************/

static void do_akick(const char *source)
{
    const char *chan   = strtok(NULL, " ");
    const char *cmd    = strtok(NULL, " ");
    const char *mask   = strtok(NULL, " ");
    const char *reason = strtok(NULL, "");
    char *t;
    ChannelInfo *ci;
    NickInfo *ni;
    User *u;
    int i, k=0;
    AutoKick *akick;
    Channel *c;
    struct c_userlist *u2;

    if (!cmd || (stricmp(cmd, "LIST") != 0 &&
       stricmp(cmd, "WIPE") != 0 && !mask)) {

	notice(s_ChanServ, source,
                "Syntax: \2AKICK\2 %s [ADD|DEL|WIPE|LIST] [<nick> or <usermask>]",
                chan ? chan : "<#channel>");
	notice(s_ChanServ, source,
		"\2/msg %s HELP AKICK\2 for more information.", s_ChanServ);

    } else if (!(ci = cs_findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (stricmp(cmd, "LIST") == 0) {

       if (!(is_oper(source)) &&
        ((!(u = finduser(source)) || !check_access(u, ci, CA_AKICK))
             && !(stricmp(cmd, "LIST") == 0 && is_services_admin(source)))) {

           notice(s_ChanServ, source, "Access Denied");

        } else {
 
	notice(s_ChanServ, source, "AKICK list for %s:", chan);
	for (akick = ci->akick, i = 0; i < ci->akickcount; ++akick, ++i) {
	    if (mask && !match_wild(mask, akick->name))
		continue;
	    if (ni = findnick(akick->name))
		t = ni->last_usermask;
	    else
                t = NULL;
	    notice(s_ChanServ, source, " %d)  %s%s%s%s%s%s%s",
			i+1, akick->name,
			t ? " (" : "", t ? t : "", t ? ")" : "",
			akick->reason ? " (" : "",
			akick->reason ? akick->reason : "",
			akick->reason ? ")" : "");
	}
        notice(s_ChanServ,source,"End of List");
      }

    } else if ((!(u = finduser(source)) || !check_access(u, ci, CA_AKICK))
	       && !(stricmp(cmd, "LIST") == 0 && is_services_admin(source))) {

	notice(s_ChanServ, source, "Access denied.");

    } else if (stricmp(cmd, "WIPE") == 0) {

      User *u = finduser(source);

      if (get_access(u, ci) < 13)  {
          notice(s_ChanServ, source, "Access Denied");
          return;
      }

      for (akick = ci->akick, i = 0; ci->akickcount; ++akick, ++i) {
           free(akick->name);
           if (akick->reason)
               free(akick->reason);

             ++k;
             --ci->akickcount;
             if (!ci->akickcount) {
                 free(ci->akick);
                 ci->akick = NULL;
           }
      }

      notice(s_ChanServ, source, "%d enteries deleted from the %s AKICK list",
         k, chan);

    } else if (stricmp(cmd, "ADD") == 0) {

	NickInfo *ni = findnick(mask);
	char *nick, *user, *host;

	if (readonly) {
	    notice(s_ChanServ, source,
		"Sorry, channel AutoKick list modification is temporarily disabled.");
	    return;
	}

        if ((AKICK_MAX != 0) && (ci->akickcount > AKICK_MAX-1)) {
	    notice(s_ChanServ, source,
                    "Sorry, you may only have %d AutoKick masks for a channel.",
		    AKICK_MAX);
            wallops(s_ChanServ,
                 "%s tried to add another akick to %s", source, chan);
	    return;
	}

	if (!ni) {
	    split_usermask(mask, &nick, &user, &host);
	    mask = smalloc(strlen(nick) + strlen(user) + strlen(host) + 3);
	    sprintf((char *)mask, "%s!%s@%s", nick, user, host);
	    free(nick);
	    free(user);
	    free(host);
	}

	for (akick = ci->akick, i = 0; i < ci->akickcount; ++akick, ++i) {
	    if ((akick->is_nick ? stricmp(akick->name,mask) :
						strcmp(akick->name,mask)) == 0) {
		notice(s_ChanServ, source,
			"\2%s\2 already exists on %s AKICK list.",
			akick->name, chan);
                free((char *)mask);
		return;
	    }
	}

	++ci->akickcount;
	ci->akick = srealloc(ci->akick, sizeof(AutoKick) * ci->akickcount);
	akick = &ci->akick[ci->akickcount-1];
	akick->name = ni ? sstrdup(mask) : (char *)mask;
	akick->is_nick = (ni != NULL);
	akick->pad = 0;
	if (reason)
	    akick->reason = sstrdup(reason);
	else
	    akick->reason = NULL;
	notice(s_ChanServ, source,
		"\2%s\2 added to %s AKICK list.", akick->name, chan);
        show_next_db(source, s_ChanServ);


    } else if (stricmp(cmd, "DEL") == 0) {

	if (readonly) {
	    notice(s_ChanServ, source,
		"Sorry, channel AutoKick list modification is temporarily disabled.");
	    return;
	}

	/* Special case: is it a number?  Only do search if it isn't. */
	if (strspn(mask, "1234567890") == strlen(mask) &&
				(i = atoi(mask)) > 0 && i <= ci->akickcount) {
	    --i;
	    akick = &ci->akick[i];
	} else {
	    /* First try for an exact match; then, a case-insensitive one. */
	    for (akick = ci->akick, i = 0; i < ci->akickcount; ++akick, ++i) {
		if (strcmp(akick->name, mask) == 0)
		    break;
	    }
	    if (i == ci->akickcount) {
		for (akick = ci->akick, i = 0; i < ci->akickcount; ++akick, ++i) {
		    if (stricmp(akick->name, mask) == 0)
			break;
		}
	    }
	    if (i == ci->akickcount) {
		notice(s_ChanServ, source,
			"\2%s\2 not found on %s AKICK list.", mask, chan);
		return;
	    }
	}
	notice(s_ChanServ, source,
		"\2%s\2 deleted from %s AKICK list.", akick->name, chan);
        show_next_db(source, s_ChanServ);
        if (readonly)
             notice(s_ChanServ, source,
                   "Database changes temporarily disabled. Changes will not be saved!");

	free(akick->name);
	if (akick->reason)
	    free(akick->reason);
	--ci->akickcount;
	if (i < ci->akickcount)
	    bcopy(akick+1, akick, sizeof(AutoKick) * (ci->akickcount-i));
	if (ci->akickcount)
	    ci->akick = srealloc(ci->akick, sizeof(AutoKick) * ci->akickcount);
	else {
	    free(ci->akick);
	    ci->akick = NULL;
	}

    } else {

	notice(s_ChanServ, source,
                "Syntax: \2AKICK\2 %s [ADD|DEL|LIST|WIPE] [<nick> or <usermask>]",
		chan);
	notice(s_ChanServ, source,
		"\2/msg %s HELP AKICK\2 for more information.", s_ChanServ);

    }
}

/*************************************************************************/

static void do_info(const char *source)
{
    const char *chan = strtok(NULL, " ");
    ChannelInfo *ci;
    Channel *chan_s;
    NickInfo *ni;
    struct tm tm;
    char s[BUFSIZE], *t, *end, timebuf[64];
    time_t tbuff;   

    if (!chan) {

        notice(s_ChanServ, source, "Syntax: \2INFO\2 <#channel>");
	notice(s_ChanServ, source,
		"\2/msg %s HELP INFO\2 for more information.", s_ChanServ);

    } else if (!(ci = cs_findchan(chan))) {

	notice(s_ChanServ, source, "Channel \2%s\2 is not registered.", chan);

    } else if (ci->flags & CI_VERBOTEN) {

	notice(s_ChanServ, source,
		"Channel \2%s\2 may not be registered or used.", chan);

    } else {
        int need_comma = 0;
        int ltime = (time(NULL) - ci->last_used);

	notice(s_ChanServ, source, "Information for channel \2%s\2:", chan);
	if (ni = findnick(ci->founder))
	    t = ni->last_usermask;
	else
	    t = NULL;
	notice(s_ChanServ, source,
		"        Founder: %s%s%s%s",
			ci->founder, t ? " (" : "", t ? t : "", t ? ")" : "");
	notice(s_ChanServ, source,
		"    Description: %s", ci->desc);
        if (ci->welcome)
            notice(s_ChanServ, source,
                    "    Welcome MSG: %s", ci->welcome);

	tm = *localtime(&ci->time_registered);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;

	notice(s_ChanServ, source,
		"     Registered: %s", timebuf);

	tm = *localtime(&ci->last_used);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;

        if (ltime > 86400)
           notice(s_ChanServ, source,
                "      Last used: %s [over %d day%s]", timebuf,
                ltime/86400, (ltime/86400 == 1) ? "" : "s");
        else if (ltime > 3600)
           notice(s_ChanServ, source,
                "      Last used: %s [over %d hour%s]", timebuf,
                ltime/3600, (ltime/3600 == 1) ? "" : "s");
        else
           notice(s_ChanServ, source,
                "      Last used: %s [under 1 hour]", timebuf);

        tbuff = time(NULL);
	tm = *localtime(&tbuff);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;

	notice(s_ChanServ, source,
		"       Time Now: %s", timebuf);

	if (ci->last_topic) {
	    notice(s_ChanServ, source,
		"     Last topic: %s", ci->last_topic);
	    notice(s_ChanServ, source,
		"   Topic set by: %s\n", ci->last_topic_setter);
	}
        if (ci->flags & CI_MEMO_AV) {
            notice(s_ChanServ, source,
                "     Memo Level: AVOICE");
        } else if (ci->flags & CI_MEMO_NONE) {
            notice(s_ChanServ, source,
                "     Memo Level: Disabled by Founder");
        } else if (ci->flags & CI_MEMO_AOP) {
            notice(s_ChanServ, source,
                "     Memo Level: AOP");
        } else if (ci->flags & CI_MEMO_SOP) {
            notice(s_ChanServ, source,
                "     Memo Level: SOP");
        } else if (ci->flags & CI_MEMO_CF) {
            notice(s_ChanServ, source,
                "     Memo Level: CFOUNDER");
        } else if (ci->flags & CI_MEMO_FR) {
            notice(s_ChanServ, source,
                "     Memo Level: FOUNDER");
        } else
            notice(s_ChanServ, source,
                "     Memo Level: AVOICE");

/*  most likely is a case of an older channel that was created
    before the implementation of channel memos. Once a channel memo
    is sent, or memo level set, it will automatically be set anyways.
*/

	if (ci->url)
	    notice(s_ChanServ, source,
		"            URL: %s\n", ci->url);
	if (ci->email && !(ci->flags & CI_PRIVATE))
	    notice(s_ChanServ, source,
		" E-mail address: %s\n", ci->email);
	if (!ci->flags)
	    strcpy(s, "None\n");
	else {
	    static const char commastr[] = ", ";
	    end = s;
	    if (ci->flags & CI_PRIVATE) {
		end += snprintf(end, sizeof(s)-(end-s), "Private");
		need_comma = 1;
	    }
            if (ci->flags & CI_LEAVEOPS) {
		end += snprintf(end, sizeof(s)-(end-s), "%sLeave Ops", need_comma ? commastr : "");
		need_comma = 1;
	    }
	    if (ci->flags & CI_KEEPTOPIC) {
		end += snprintf(end, sizeof(s)-(end-s), "%sTopic Retention", need_comma ? commastr : "");
		need_comma = 1;
	    }
	    if (ci->flags & CI_TOPICLOCK) {
                if (ci->topic_allow == 3)
                    end += snprintf(end, sizeof(s)-(end-s), "%sTopic Lock(+v)", need_comma ? commastr : "");
                else if (ci->topic_allow == 5)
                    end += snprintf(end, sizeof(s)-(end-s), "%sTopic Lock(A)", need_comma ? commastr : "");
                else if (ci->topic_allow == 10)
                    end += snprintf(end, sizeof(s)-(end-s), "%sTopic Lock(S)", need_comma ? commastr : "");
                else if (ci->topic_allow == 13)
                    end += snprintf(end, sizeof(s)-(end-s), "%sTopic Lock(CF)", need_comma ? commastr : "");
                else if (ci->topic_allow == 15)
                    end += snprintf(end, sizeof(s)-(end-s), "%sTopic Lock(F)", need_comma ? commastr : "");
		need_comma = 1;
	    }
	    if (ci->flags & CI_OPGUARD) {
		end += snprintf(end, sizeof(s)-(end-s), "%sOp Guard", need_comma ? commastr : "");
		need_comma = 1;
	    }
	    if (ci->flags & CI_RESTRICTED) {
		end += snprintf(end, sizeof(s)-(end-s), "%sRestricted Access", need_comma ? commastr : "");
		need_comma = 1;
	    }
	    if (ci->flags & CI_PROTECTED) {
		end += snprintf(end, sizeof(s)-(end-s), "%sOp-Protect", need_comma ? commastr : "");
		need_comma = 1;
	    }
	    if (ci->flags & CI_IDENT) {
		end += snprintf(end, sizeof(s)-(end-s), "%sIdent", need_comma ? commastr : "");
		need_comma = 1;
	    }
	    if (ci->flags & CI_UNSECURE) {
		end += snprintf(end, sizeof(s)-(end-s), "%sUnsecure", need_comma ? commastr : "");
		need_comma = 1;
	    }
	}
        if (!need_comma)
  	   notice(s_ChanServ, source, "        Options: None");
        else
  	   notice(s_ChanServ, source, "        Options: %s", s);
	end = s;
	*end = 0;
	if (ci->mlock_on || ci->mlock_key || ci->mlock_limit)
            end += sprintf(end, "+%s%s%s%s%s%s%s%s%s%s%sr",
				(ci->mlock_on & CMODE_I) ? "i" : "",
				(ci->mlock_key         ) ? "k" : "",
				(ci->mlock_limit       ) ? "l" : "",
				(ci->mlock_on & CMODE_M) ? "m" : "",
				(ci->mlock_on & CMODE_N) ? "n" : "",
				(ci->mlock_on & CMODE_P) ? "p" : "",
				(ci->mlock_on & CMODE_S) ? "s" : "",
				(ci->mlock_on & CMODE_T) ? "t" : "",
				(ci->mlock_on & CMODE_C) ? "c" : "",
				(ci->mlock_on & CMODE_r) ? "R" : "",
                                (ci->mlock_on & CMODE_o) ? "O" : "",
                                (ci->mlock_on & CMODE_m) ? "M" : "");

	if (ci->mlock_off)
	    end += sprintf(end, "-%s%s%s%s%s%s%s%s%s%s%s",
				(ci->mlock_off & CMODE_I) ? "i" : "",
				(ci->mlock_off & CMODE_K) ? "k" : "",
				(ci->mlock_off & CMODE_L) ? "l" : "",
				(ci->mlock_off & CMODE_M) ? "m" : "",
				(ci->mlock_off & CMODE_N) ? "n" : "",
				(ci->mlock_off & CMODE_P) ? "p" : "",
				(ci->mlock_off & CMODE_S) ? "s" : "",
				(ci->mlock_off & CMODE_T) ? "t" : "",
				(ci->mlock_off & CMODE_C) ? "c" : "",
				(ci->mlock_off & CMODE_r) ? "R" : "",
                                (ci->mlock_off & CMODE_o) ? "O" : "",
                                (ci->mlock_off & CMODE_m) ? "M" : "");

	if (s)
	    notice(s_ChanServ, source, "      Mode lock: %s", s);
        if (ci->flags & CI_CLOSED)
            notice(s_ChanServ, source, "* Channel has been CLOSED by %s",
                NETWORK_NAME);
        chan_s = findchan(chan);
        if (is_services_admin(source)) {
            if (!chan_s && ci->mlock_key)
                notice(s_ChanServ, source, "\2SRA\2: Channel Key = %s",
                           ci->mlock_key);
            else {
                if (chan_s && chan_s->key)
                   notice(s_ChanServ, source, "\2SRA\2: Channel Key = %s",
                           chan_s->key);
            }
        }
        if (is_oper(source)) {

            if (ci->flags & CI_MARKCHAN)
                notice(s_ChanServ, source,
                   " * Channel has been marked by: %s", ci->mark);

            if (ci->flags & CI_HELDCHAN)
                notice(s_ChanServ, source,
                   " * Channel has been held by: %s", ci->hold);

            if (ci->flags & CI_FREEZECHAN)
                notice(s_ChanServ, source,
                   " * Channel has been frozen by: %s", ci->freeze);


        } else {

            if (ci->flags & CI_MARKCHAN)
                notice(s_ChanServ, source,
                   " * This Channel has been marked.");

            if (ci->flags & CI_HELDCHAN)
                notice(s_ChanServ, source,
                   " * This Channel has been held.");

            if (ci->flags & CI_FREEZECHAN)
                notice(s_ChanServ, source,
                   " * This Channel has been frozen.");
        }
        notice(s_ChanServ, source, "\2*** End of Info ***\2");

    }
}

/*************************************************************************/

void do_chanlist(const char *source)
{
   NickInfo *ni = findnick(source);
   ChannelInfo *ci;
   ChanAccess *access;
   int i, x, nchans=0, acc;


   if (!listchans_enable)
       notice(s_NickServ, source, "Sorry, CHANLIST has been disabled by the network");

   else if (!ni)
       notice(s_NickServ, source,
           "Your current nick is not registered");
   else if (!(ni->flags & NI_IDENTIFIED))
       notice(s_NickServ, source,
           "You need to identify to your nick before requesting the chanlist");
   else {
       for (i=33; i < 256; ++i) {
         for (ci = chanlists[i]; ci; ci = ci->next) {
             if (stricmp(ni->nick, ci->founder) == 0) {
                 if (!(ci->flags & CI_VERBOTEN)) {
                     ++nchans;
                     notice(s_NickServ, source, "%d) %s -- Access: Founder",
                            nchans, ci->name);
                 }
             } else {
                for (access = ci->access, x=0; x < ci->accesscount; ++access, ++x) {
                    if (match_wild(ni->nick, access->name)) {
                       ++nchans;
                       notice(s_NickServ, source,
                           "%d) %s -- Access: %s%s%s%s",
                           nchans, ci->name,
                           access->level==3 ? "Auto-Voice" : "",
                           access->level==5 ? "Auto-Op" : "",
                           access->level==10 ? "Super-Op" : "",
                           access->level==13 ? "Co-Founder" : "");
                    }
                }
                if (nchans > 50) {
                   notice(s_NickServ, source,
                      "Channel list returns greater than 50");
                   return;
                }
            }
         }
      }
      if (!nchans)
          notice(s_NickServ, source,
             "You currently have chanserv access to no channels");
      else
          notice(s_NickServ, source,
             "End of Channel List (%d)", nchans);
   }
}


/*************************************************************************/

static void do_list(const char *source)
{
    const char *pattern = strtok(NULL, " ");
    ChannelInfo *ci;
    int nchans, i;
    char buf[BUFSIZE];

    if (!switch_list && !is_oper(source)) {
        notice(s_ChanServ, source,
             "Sorry, the LIST command has been disabled by the network");

    } else if (!pattern) {

        notice(s_ChanServ, source, "Syntax: \2LIST\2 <pattern>");
	notice(s_ChanServ, source,
		"\2/msg %s HELP LIST\2 for more information.", s_ChanServ);

    } else {

	nchans = 0;
	notice(s_ChanServ, source, "List of entries matching \2%s\2:", pattern);
	for (i = 33; i < 256; ++i) {
	    for (ci = chanlists[i]; ci; ci = ci->next) {
		if (ci->flags & (CI_PRIVATE | CI_VERBOTEN))
		    continue;
		if (strlen(ci->name)+strlen(ci->desc) > sizeof(buf))
		    continue;
		snprintf(buf, sizeof(buf), "%-20s  %s", ci->name, ci->desc);
		if (stricmp(pattern, ci->name) == 0 ||
					match_wild_nocase(pattern, buf)) {
		    if (++nchans <= 50)
			notice(s_ChanServ, source, "    %s", buf);
		}
	    }
	}
	notice(s_ChanServ, source, "End of list - %d/%d matches shown.",
					nchans>50 ? 50 : nchans, nchans);
    }

}

/*************************************************************************/

static void do_invite(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *nick;
    User *u = finduser(source);
    ChannelInfo *ci;

    if (!chan) {

	notice(s_ChanServ, source,
                "Syntax: \2INVITE\2 <#channel>");
	notice(s_ChanServ, source,
		"\2/msg %s HELP INVITE\2 for more information.", s_ChanServ);

    } else if (!findchan(chan)) {

	notice(s_ChanServ, source, "Channel %s does not exist.", chan);

    } else if (!(ci = cs_findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!u || !check_access(u, ci, CA_INVITE)) {

	notice(s_ChanServ, source, "Access denied.");

    } else

           send_cmd(s_ChanServ, "INVITE %s %s", source, chan);
}

/*************************************************************************/

static void do_op(const char *source)
{
    char *chan = strtok(NULL, " ");
    char buf[196], buf2[24];
    User *u = finduser(source);
    User *u2, *u3;
    ChannelInfo *ci;
    Channel *c;
    NickInfo *ni;
    char *tmp, *tmp2[5];
    char *av[3];
    int p=0, i=0, acc=0;


     if (!u)
       return;

     while ((tmp = strtok(NULL, " ")) != NULL)
     {
        tmp2[p++] = tmp;
        if (p > 5)
           break;
     }
     if (p == 0 || !chan) {
        notice(s_ChanServ, source,
            "Syntax: \2OP\2 <#channel> <nick(s)>");
        notice(s_ChanServ, source,
            "\2/msg %s HELP OP\2 for more information.", s_ChanServ);
        return;

     } else if (!(c = findchan(chan))) {
        notice(s_ChanServ, source,
              "Channel %s does not currently exist (empty)", chan);
        return;
     }
     av[0] = chan;

     u3 = finduser(tmp2[i]);


     if (!(ci = cs_findchan(chan))) {
         notice(s_ChanServ, source, "Channel %s is not registered.", chan);
         return;
     }

//  checks u3 against u for multiple identified nicks

     acc = get_access(u, ci);

     if (acc < 3) {
         if (u3 && stricmp(u->nick, u3->nick) == 0) {
             if (acc < 3) {
                  notice(s_ChanServ, source, "Access Denied");
                  return;
             }
         } else {
             notice(s_ChanServ, source, "Access Denied");
             return;
         }
     }

     if (ci->flags & CI_FREEZECHAN) {
         notice(s_ChanServ, source, "Channel %s is FROZEN.", chan);
         return;

     } else if ((stricmp(tmp2[i], source) == 0) && (p < 2)) {
          ni = findnick(source);
          if (ni && (ni->flags & NI_VERBOTEN)) {
               notice(s_NickServ, source, "OP command denied - frozen nick");
               return;
          }
          if (acc == 3) {
             send_cmd(MODE_SENDER, "MODE %s +v %s", chan, source);
             av[1] = sstrdup("+v");
          } else {
             send_cmd(MODE_SENDER, "MODE %s +o %s", chan, source);
             av[1] = sstrdup("+o");
          }
          av[2] = sstrdup(source);
          do_cmode(s_ChanServ, 3, av);
          free(av[1]); free(av[2]);
          return;                    
     } else if ((acc < 13) && (ci->flags & CI_OPGUARD)) {

         notice(s_ChanServ, source,
             "Unless you are a founder, you may not use CS OP with other nicks (except your own: %s) when channel security is set", source);
         return;
     }

     *buf = 0;
     *buf2 = 0;

     for (i = 0; i < p; i++) {
         int bogus;
         for (bogus=0; bogus < i; bogus++) {
           if (stricmp(tmp2[bogus], tmp2[i]) == 0) {
                bogus = -1;
                break;
           }
         }
         if (bogus == -1)
             continue;

         ni = findnick(tmp2[i]);

         if (!(u2 = finduser(tmp2[i])))
             continue;
         else if (ni && (ni->flags & NI_NEVEROP))
             continue;
         else if (ni && (ni->flags & NI_VERBOTEN))
             continue;
         else {
             struct c_userlist *ul;
             if (ci->flags & CI_OPGUARD) {
                 if (get_access(u2, ci) < 3)
                      send_cmd(s_ChanServ, 
                        "PRIVMSG %s :You have been given op privledges regardless of the channel security. Please respect the founder's wishes.",
                         tmp2[i]);
             }         

             if (*buf)
                 strcat(buf, " ");
             strcat(buf, tmp2[i]);


             if (!*buf2)
                 strcat(buf2, "+");


             if ((get_access(u2, ci) == 3) 
                  || ((acc == 3) && (get_access(u2, ci) < 5))) {

                 strcat(buf2, "v");
                 av[1] = sstrdup("v");
             } else {
                 strcat(buf2, "o");
                 av[1] = sstrdup("o");
             }

             av[2] = sstrdup(u2->nick);
             do_cmode(s_ChanServ, 3, av);
             free(av[1]); free(av[2]);

         }

     }

     if (!*buf || !*buf2) {
        notice(s_ChanServ, source, "No valid nicks to be granted operator status or nick(s) have NEVEROP setting enabled.");
        return;
     }
     send_cmd(MODE_SENDER, "MODE %s %s %s", chan, buf2, buf);

     send_cmd(s_ChanServ, "NOTICE @+%s :%s!%s@%s used ChanServ OP: %s",
              chan, u->nick, u->username, u->host, buf);

}

/*************************************************************************/

static void do_deop(const char *source)
{
    ChannelInfo *ci;
    Channel *c;
    NickInfo *ni;
    User *u = finduser(source);
    User *u2, *u3;
    struct c_userlist *ul;
    char *chan = strtok(NULL, " ");
    char buf[256], buf2[32];
    char *tmp, *tmp2[5];
    char *av[3];
    int p=0, i=0;

     while ((tmp = strtok(NULL, " ")) != NULL)
     {
        tmp2[p++] = tmp;
        if (p > 5)
           break;
     }

     if (!chan || p == 0) {
        notice(s_ChanServ, source,
            "Syntax: \2DEOP\2 <#channel> <nick(s)>");
        notice(s_ChanServ, source,
            "\2/msg %s HELP DEOP\2 for more information.", s_ChanServ);
        return;

     } else if (!(c = findchan(chan))) {
           notice(s_ChanServ, source,
                "Channel %s does not currently exist (empty)", chan);
           return;
     }

     u3 = finduser(tmp2[i]);
     av[0] = chan;

     if (!(ci = cs_findchan(chan))) {
         notice(s_ChanServ, source, "Channel %s is not registered.", chan);
         return;
     }

     if (get_access(u, ci) < 3) {
         if (u3 && stricmp(u->nick, u3->nick) == 0) {
             if (get_access(u3, ci) < 3) {
                  notice(s_ChanServ, source, "Access Denied");
                  return;
             }
         } else {
             notice(s_ChanServ, source, "Access Denied");
             return;
         }
     }


     if (!u || ((get_access(u, ci) < 5)
                && !is_on_id_list(u->nick, tmp2[0]))) {
         notice(s_ChanServ, source, "Access denied.");
         return;

     } else if (ci->flags & CI_FREEZECHAN) {
         notice(s_ChanServ, source, "Channel %s is FROZEN.", chan);
         return;
     }

     if (p == 0) {
        notice(s_ChanServ, source,
            "Syntax: \2DEOP\2 <#channel> <nick(s)>");
        notice(s_ChanServ, source,
            "\2/msg %s HELP DEOP\2 for more information.", s_ChanServ);
        return;
     }

     *buf = 0;
     *buf2 = 0;

     for (i = 0; i < p; i++) {
         int bogus;
         for (bogus=0; bogus < i; bogus++) {
           if (stricmp(tmp2[bogus], tmp2[i]) == 0) {
                bogus = -1;
                break;
           }
         }
         if (bogus == -1)
             continue;

         ni = findnick(tmp2[i]);

         if (!(u2 = finduser(tmp2[i])))
             continue;
         else {

             if (*buf)
                 strcat(buf, " ");
             strcat(buf, tmp2[i]);


             if (!*buf2)
                 strcat(buf2, "-");

             if (get_access(u2, ci) == 3) {
                 strcat(buf2, "v");
                 av[1] = sstrdup("-v");
             } else {
                 strcat(buf2, "o");
                 av[1] = sstrdup("-o");
             }
         }
         av[2] = sstrdup(tmp2[i]);
         do_cmode(s_ChanServ, 3, av);
         free(av[1]); free(av[2]);

     }

     if (!*buf || !*buf2) {
        notice(s_ChanServ, source, "No valid nicks to be stripped of operator status");
        return;
     }

     send_cmd(MODE_SENDER, "MODE %s %s %s", chan, buf2, buf);

     send_cmd(s_ChanServ, "NOTICE %s :%s!%s@%s used ChanServ DEOP: %s",
              chan, u->nick, u->username, u->host, buf);

}


/*************************************************************************/

static void do_unban(const char *source)
{
    char *chan = strtok(NULL, " ");
    User *u = finduser(source);
    ChannelInfo *ci;
    Channel *c;
    int i;
    char *av[3];

    if (!u)
       return;

    if (!chan) {

        notice(s_ChanServ, source, "Syntax: \2UNBAN\2 <#channel>");
	notice(s_ChanServ, source,
		"\2/msg %s HELP UNBAN\2 for more information.", s_ChanServ);

    } else if (!(c = findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s does not exist.", chan);

    } else if (!(ci = cs_findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!u || !check_access(u, ci, CA_UNBAN)) {

	notice(s_ChanServ, source, "Access denied.");

    } else if (ci->flags & CI_FREEZECHAN) {
        notice(s_ChanServ, source, "Channel %s is FROZEN.", chan);

    } else {

	av[0] = chan;
	av[1] = sstrdup("-b");
	for (i = c->bancount-1; i > -1; --i) {
	    if (match_usermask(c->bans[i], u)) {
		send_cmd(MODE_SENDER, "MODE %s -b %s", chan, c->bans[i]);
		av[2] = sstrdup(c->bans[i]);
		do_cmode(s_ChanServ, 3, av);
		free(av[2]);
	    }
	}
        free(av[1]);
	notice(s_ChanServ, source, "You have been unbanned from %s.", chan);

    }
}

/*************************************************************************/

static void do_clear(const char *source)
{
    char *chan = strtok(NULL, " ");
    const char *what = strtok(NULL, " ");
    const char *reason = strtok(NULL, "");
    User *u = finduser(source);
    Channel *c;
    ChannelInfo *ci;

    if (!what) {

	notice(s_ChanServ, source,
                "Syntax: \2CLEAR\2 <#channel> <what> [<reason>]");
	notice(s_ChanServ, source,
		"\2/msg %s HELP CLEAR\2 for more information.", s_ChanServ);


    } else if (!(c = findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s does not exist.", chan);

    } else if (!(ci = cs_findchan(chan))) {

	notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (ci->flags & CI_FREEZECHAN) {
        notice(s_ChanServ, source, "Channel %s is Frozen.", chan);

    } else if (!u || !check_access(u, ci, CA_CLEAR)) {

	notice(s_ChanServ, source, "Access denied.");

    } else if (stricmp(what, "bans") == 0) {

	char *av[3];
        char buf[1024], buf2[128];
        int bcnt=0;

        *buf=0; *buf2=0;

	while (c->bancount > 0) {
	    av[0] = sstrdup(chan);
	    av[1] = sstrdup("-b");
	    av[2] = sstrdup(c->bans[0]);
//	    send_cmd(MODE_SENDER, "MODE %s %s :%s", av[0], av[1], av[2]);
	    do_cmode(s_ChanServ, 3, av);
                  if (*buf)
                     strcat(buf, " ");
                  else
                     strcat(buf2, "-");
                  strcat(buf, av[2]);
                  strcat(buf2, "b");
                  bcnt++;
                  if (bcnt > 5) {
                     bcnt = 0;
                     send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);
                     *buf=0; *buf2=0;
                  }

            free(av[0]); free(av[1]); free(av[2]);
	}
        if (*buf)
             send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);

	notice(s_ChanServ, source, "All bans on %s have been removed.", chan);
        slog("CS C %s (%s!%s@%s) [Bans]",
              chan, source, u->username, u->host);

    } else if (stricmp(what, "modes") == 0) {

	char *av[3];

	av[0] = chan;
	av[1] = sstrdup("-mintpslk");
	if (c->key)
	    av[2] = sstrdup(c->key);
	else
	    av[2] = sstrdup("");
	send_cmd(MODE_SENDER, "MODE %s %s :%s", av[0], av[1], av[2]);
	do_cmode(s_ChanServ, 3, av);
        free(av[1]); free(av[2]);
	notice(s_ChanServ, source, "All modes on %s have been reset.", chan);
        slog("CS C %s (%s!%s@%s) [Modes]",
              chan, source, u->username, u->host);

    } else if (stricmp(what, "ops") == 0) {

	char *av[3], superior[1024];
        char buf[1024], buf2[128];
	struct c_userlist *cu, *next;
        ChanAccess *access;
        int i, ulev=0, unick=0, dcnt=0;
        ulev = get_access(u, ci);
        *superior = 0, *buf = 0; *buf2 = 0;

	for (cu = c->chanops; cu; cu = next) {
            next = cu->next;
/* The founder doesn't have an entry in the access list.. */
            if ((get_access(finduser(cu->user->nick), ci) > 13)
                  && (stricmp(cu->user->nick, source) != 0)) {
                 if (*superior)
                      strcat(superior, " ");
                 strcat(superior, ci->founder);
                 unick = 15;
            }
            for (access = ci->access, i=0; i < ci->accesscount; 
                   ++access, ++i) {
                if (stricmp(cu->user->nick, access->name) == 0) {
                   unick = access->level;
                   if ((access->level >= ulev)
                        && (stricmp(access->name, source) != 0)) {
                        if (*superior)
                            strcat(superior, " ");
                        strcat(superior, access->name);
                   }
               }
            }
            if ((stricmp(cu->user->nick, source) != 0)
               && (unick < ulev)) {
                  av[0] = sstrdup(chan);
       	          av[1] = sstrdup("-o");
	          av[2] = sstrdup(cu->user->nick);
//	          send_cmd(MODE_SENDER, "MODE %s %s :%s", av[0], av[1], av[2]);
  	          do_cmode(s_ChanServ, 3, av);
                  if (*buf)
                     strcat(buf, " ");
                  else
                     strcat(buf2, "-");
                  strcat(buf, av[2]);
                  strcat(buf2, "o");
                  dcnt++;
                  if (dcnt > 5) {
                     dcnt = 0;
                     send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);
                     *buf=0; *buf2=0;
                  }
                  free(av[0]); free(av[1]); free(av[2]);
            }
	}
        if (*buf)
            send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);

        if (*superior)
           notice(s_ChanServ, source,
                "The following nicks have a superior access and were not "
                "effected by the CLEAR command: %s", superior);
	notice(s_ChanServ, source, "Mode +o cleared from %s.", chan);
        slog("CS C %s (%s!%s@%s) [Ops]",
              chan, source, u->username, u->host);

    } else if (stricmp(what, "voices") == 0) {

	char *av[3];
	struct c_userlist *cu, *next;
        char buf[1024], buf2[128];
        int vcnt=0;

        *buf=0; *buf2=0;

	for (cu = c->voices; cu; cu = next) {
	    next = cu->next;
	    av[0] = sstrdup(chan);
	    av[1] = sstrdup("-v");
	    av[2] = sstrdup(cu->user->nick);
//	    send_cmd(MODE_SENDER, "MODE %s %s :%s", av[0], av[1], av[2]);
	    do_cmode(s_ChanServ, 3, av);
                  if (*buf)
                     strcat(buf, " ");
                  else
                     strcat(buf2, "-");
                  strcat(buf, av[2]);
                  strcat(buf2, "v");
                  vcnt++;
                  if (vcnt > 5) {
                     vcnt = 0;
                     send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);
                     *buf=0; *buf2=0;
                  }
            free(av[0]); free(av[1]); free(av[2]);
	}
        if (*buf)
             send_cmd(s_ChanServ, "MODE %s %s %s", chan, buf2, buf);

	notice(s_ChanServ, source, "Mode +v cleared from %s.", chan);
        slog("CS C %s (%s!%s@%s) [Voices]",
              chan, source, u->username, u->host);

    } else if (stricmp(what, "users") == 0) {

	char *av[3], superior[1024];
	struct c_userlist *cu, *next_cu;
	char buf[256];
        ChanAccess *access;
        Timeout *t;
        int i, ulev=0, unick=0;

        ulev = get_access(u, ci);
        *superior = 0;


	snprintf(buf, sizeof(buf), "CLEAR USERS command from %s", source);
        send_cmd(s_ChanServ, "JOIN %s", chan);
        send_cmd(s_ChanServ, "MODE %s +b *!*@*", chan);

	for (cu = c->users; cu; cu = next_cu) {
            next_cu = cu->next;
/* The founder doesn't have an entry in the access list.. */
            if ((get_access(finduser(cu->user->nick), ci) > 13)
                  && (stricmp(cu->user->nick, source) != 0)) {
                 if (*superior)
                      strcat(superior, " ");
                 strcat(superior, ci->founder);
                 unick = 15;
            }
            for (access = ci->access, i=0; i < ci->accesscount;
                ++access, ++i) {
                if (stricmp(cu->user->nick, access->name) == 0) {
                   unick = access->level;
                   if ((access->level >= ulev)
                        && (stricmp(access->name, source) != 0)) {
                        if (*superior)
                            strcat(superior, " ");
                        strcat(superior, access->name);
                   }
                }
            }
            if ((stricmp(cu->user->nick, source) != 0)
               && (unick < ulev)) {
     	          av[0] = sstrdup(chan);
	          av[1] = sstrdup(cu->user->nick);
                  if (!reason)
	             av[2] = sstrdup(buf);
                  else
                     av[2] = sstrdup(reason);
	          send_cmd(s_ChanServ, "KICK %s %s :%s", av[0], av[1], av[2]);
  	          do_kick(s_ChanServ, 3, av);
                  free(av[0]); free(av[1]); free(av[2]);
            }
	}
        if (*superior)
           notice(s_ChanServ, source,
                "The following nicks have a superior access and were not "
                "effected by the CLEAR command: %s", superior);

	notice(s_ChanServ, source, "All Applicable users kicked from %s.", chan);
        slog("CS C %s (%s!%s@%s) [Users]",
              chan, source, u->username, u->host);

        t = add_timeout(CHANNEL_INHABIT, timeout_unban, 0);
        t->data = sstrdup(chan);

        t = add_timeout(CHANNEL_INHABIT+1, timeout_leave, 0);
        t->data = sstrdup(chan);


    } else {

	notice(s_ChanServ, source,
                "Syntax: \2CLEAR <#channel> <what> [<reason>]");
	notice(s_ChanServ, source,
		"\2/msg %s HELP CLEAR\2 for more information.", s_ChanServ);

    }
}

/*************************************************************************/

static void do_count(const char *source)
{
    const char *chan = strtok(NULL, " ");
    char *s    = "1";
    ChannelInfo *ci;
    NickInfo *ni;
    AutoKick *akick;
    User *u;
    short ulev;
    int is_list = (chan && !stricmp(chan, "LIST"));
    ChanAccess *access;
    int i;

    if (!chan) {
	notice(s_ChanServ, source, "Usage: \2/msg %s COUNT \37channel\37\2", s_ChanServ);
	return;
    }
    if (!(ci = cs_findchan(chan))) {

        notice(s_ChanServ, source, "Channel %s is not registered.", chan);

    } else if (!(u = finduser(source))) {

        notice(s_ChanServ, source, "Access denied.");
        log("%s: ACCESS command from nonexistent user %s", s_ChanServ, source);


    } else if (!(is_oper(source)) &&
             (((is_list && !check_access(u, ci, CA_ACCESS_LIST))
             || (!is_list && !check_access(u, ci, CA_ACCESS_CHANGE)))
             && !(is_list && is_services_admin(source)))) {

        notice(s_ChanServ, source, "Access denied.");

    } else if (s && 3 > (ulev = get_access(u, ci)) && !(is_oper(source))) {

        notice(s_ChanServ, source, "Access denied.");

    } else {
        int v=0;
        int a=0;
        int so=0;
        int f=0;
        for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
            if (3 == access->level)
               ++v;
            if (5 == access->level)
               ++a;
            if (10 == access->level)
               ++so;
            if (13 == access->level)
               ++f;
        }
        notice(s_ChanServ, source, "\2%s Count:\2 VOICE: %d, AOP: %d, SOP: %d, CF: %d AKICK: %d", chan, v, a, so, f, ci->akickcount);
        
    }
}


/*************************************************************************/

static void do_getpass(const char *source)
{
    const char *chan = strtok(NULL, " ");
    User *u = finduser(source);
    ChannelInfo *ci;


    if (!getpass_enable) {
         notice(s_ChanServ, source, "GETPASS has been disabled by the network");
         return;
    }

#ifdef USE_ENCRYPTION
    notice(s_ChanServ, source,
		"GETPASS command unavailable when encryption is in use.");
#else
    if (!chan) {
        notice(s_ChanServ, source, "Syntax: \2GETPASS\2 <#channel>");
    } else if (!(ci = cs_findchan(chan))) {
	notice(s_ChanServ, source, "Channel %s is not registered.", chan);
        slog("CS *G %s (%s!%s@%s) [UNREGISTERED]",
          chan, u->nick, u->username, u->host);
        do_break_log("CS_G", "CS *G %s (%s!%s@%s) [UNREGISTERED]",
          chan, u->nick, u->username, u->host);
        csgp += 1;
	return;
    } else if (!u) {
	notice(s_ChanServ, source, "Could not find your user record!");
    } else if (ci->flags & CI_MARKCHAN) {
	notice(s_ChanServ,source,"Channel \2%s\2 is marked!",chan);
        slog("CS *G %s (%s!%s@%s) [MARKED]",
          chan, u->nick, u->username, u->host);
        do_break_log("CS_G", "CS *G %s (%s!%s@%s) [MARKED]",
          chan, u->nick, u->username, u->host);
	wallops(s_ChanServ,
                "\2Warning:\2 %s attempted to use GETPASS on marked channel %s"
		,u->nick,chan);
	log("%s: %s!%s@%s attempted GETPASS on %s (Fail)"
		,s_ChanServ, source, u->username, u->host, chan);
    } else {
        slog("CS G %s (%s!%s@%s)",
          chan, u->nick, u->username, u->host);
        do_break_log("CS_G", "CS G %s (%s!%s@%s)",
          chan, u->nick, u->username, u->host);
	log("%s: %s!%s@%s used GETPASS on %s",
		s_ChanServ, source, u->username, u->host, chan);
        wallops(s_ChanServ, "%s used GETPASS on channel %s",
		source, chan);
	notice(s_ChanServ, source, "Password for %s is: \2%s\2",
		chan, ci->founderpass);
    }
#endif
}

/*************************************************************************/

static void do_sendpass(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *email = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);
    ChannelInfo *ci;


    if (!auth_email) {
         notice(s_ChanServ, source,
               "Email systems have been disabled by the network");

    } else if (!chan || !email) {
        notice(s_ChanServ, source, "Syntax: \2SENDPASS\2 <#channel> <e-mail>");
    } else if (!(ci = cs_findchan(chan))) {
	notice(s_ChanServ, source, "Channel %s is not registered.", chan);
        slog("CS *S %s (%s!%s@%s) [UNREGISTERED]",
          chan, u->nick, u->username, u->host);
        do_break_log("CS_G", "CS *S %s (%s!%s@%s) [UNREGISTERED]",
          chan, u->nick, u->username, u->host);
	return;
    } else if (!u) {
	notice(s_ChanServ, source, "Could not find your user record!");
    } else if (ci->flags & CI_MARKCHAN) {
	notice(s_ChanServ,source,"Channel \2%s\2 is marked!",chan);
        slog("CS *S %s (%s!%s@%s) [MARKED]",
          chan, u->nick, u->username, u->host);
        do_break_log("CS_G", "CS *S %s (%s!%s@%s) [MARKED]",
          chan, u->nick, u->username, u->host);
	log("%s: %s!%s@%s attempted SENDPASS on %s (Fail)"
		,s_ChanServ, source, u->username, u->host, chan);
        csgp += 1;
    } else if (!(ni = findnick(ci->founder))) {
          ;
    } else if (!ni->regemail) {
         notice(s_ChanServ, source, "SENDPASS failed: The founder has yet to authorize his nick");
    } else if (stricmp(ni->regemail, email) != 0) {
         notice(s_ChanServ, source, "The password you provided does not match with that of the channel founder e-mail");
    } else {
        sendpass_cs(source, ci);
    }
}


/*************************************************************************/

static void do_suspend(const char *source)
{
  C_SuspendData *c_spn, *prev, *id;
  struct c_userlist *cu, *next_cu;
  Timeout *t;
  char *av[3];
  char *cmd, *chan, *ndelta, *s;
  int i, delta, temp_time;

  cmd = strtok(NULL, " ");

  if (!cmd) {
     notice(s_ChanServ, source,
           "Syntax: \2SUSPEND\2 [ADD|DEL|LIST] [<#channel>] [<time>]");
     return;
  }

  if (stricmp(cmd, "LIST") == 0) {

        ChannelInfo *ci;
	int sent_header = 0;
        int temp_time, x=0;

        for (i = 0; i < 256; i++) {
           for (id = c_suspend[i]; id; id = id->next) {
              C_SuspendData *c_spn;
              if (c_spn && NULL != (c_spn = get_c_suspend(id->chan))) {
                  if (!sent_header) {
                      notice(s_ChanServ, source, "Channel Suspend list:");
                      sent_header = 1;
                  }
                  temp_time = id->time - time(NULL);
                  x++;
                  notice(s_ChanServ, source,
                      "%d) Chan: %s  Time: %d minute%s, %d second%s", x,
                      id->chan, temp_time/60, temp_time/60==1 ? "" : "s",
                      temp_time%60, temp_time%60==1 ? "" : "s");
              }
           }
            
	}
	if (!sent_header)
            notice(s_ChanServ, source, "Channel Suspend list is empty.");
        sent_header = 0; x=0;

        for (i = 33; i < 256; ++i) {
           for (ci = chanlists[i]; ci; ci = ci->next) {
               if (ci->flags & CI_CLOSED) {
                   if (!sent_header) {
                       sent_header = 1;
                       notice(s_ChanServ, source, "Closed Channel List:");
                    }
                    x++;
                    notice(s_ChanServ, source,
                       "%5d) %s", x, ci->name);
                }
            }
         }
         if (!sent_header)
             notice(s_ChanServ, source,
                    "Closed Channel List is empty.");

  } else if (!check_o_access(source, "SUSPEND")) {
       notice(s_ChanServ, source, "Access Denied");

  } else if (stricmp(cmd, "DEL") == 0) {

     ChannelInfo *ci;
     chan = strtok(NULL, " ");

     if (!chan) {
        notice(s_ChanServ, source, "Syntax: \2SUSPEND DEL\2 <#channel>");
        return;
     }

     for (i = 0; i < 256; i++) {
        for (id = c_suspend[i]; id; id = id->next) {
            if (stricmp(id->chan, chan) == 0) {
               temp_time = 0 - id->time;
               add_c_suspend(chan, temp_time);
               notice(s_ChanServ, source, "%s removed from SUSPEND list", chan);
               wallops(s_ChanServ, "%s had me no longer suspend channel %s",
                   source, chan);
               return;
            }
        }
     }
     for (i = 33; i < 256; ++i) {
        for (ci = chanlists[i]; ci; ci = ci->next) {
             if (stricmp(ci->name, chan) == 0) {
                   if (!ci->flags & CI_CLOSED) {
                       notice(s_ChanServ, source,
                          "ERROR: Channel %s is not closed");
                       return;
                    } else {
                        ci->flags &= ~CI_CLOSED;
                        notice(s_ChanServ, source,
                           "Channel %s has been re-opened", chan);
                        wallops(SERVER_NAME,
                           "%s has re-opened channel %s",
                                  source, chan);
                        return;
                    }
               }
          }
     }

  } else if (stricmp(cmd, "ADD") == 0) {

      Channel *c;
      chan = strtok(NULL, " ");
      ndelta = strtok(NULL, " ");

      if (!chan || !ndelta) {
         notice(s_ChanServ, source,
            "Syntax: \2SUSPEND ADD\2 <#channel> [<time>]");
         return;
      }

      c = findchan(chan);
      delta = atoi(ndelta);

      if (!delta) {
         ChannelInfo *ci = cs_findchan(chan);
         if (!ci) {
             notice(s_ChanServ, source,
               "Only registered channel may be permanetly suspended "
               "(Closed) - FORBID is suggested");
             return;
         } else {
             ci->flags |= CI_CLOSED;
             notice(s_ChanServ, source,
                  "Channel %s has been CLOSED", chan);
             wallops(SERVER_NAME, "%s has CLOSED channel %s",
                source, chan);
             goto close_chan;
             return;
         }

      } else if (!(delta > 0)) {
         notice(s_ChanServ, source, "IGNORE: Value must be greater than 0");
         return;
      }

      for (i = 0; c_suspend[i]; i++) {
          if (!c_suspend[i])
          break;
      }

      if ((i < MAX_SUSPEND) || (MAX_SUSPEND == 0)) {
          temp_time = delta * 60;
          add_c_suspend(chan, temp_time);
          notice(s_ChanServ, source, "%s added to SUSPEND list for %d minute%s",
                   chan, temp_time/60, temp_time/60==1 ? "" : "s");
          wallops(s_ChanServ, "%s had me suspend channel %s for +%d minute%s",
                   source, chan, temp_time/60, temp_time/60==1 ? "" : "s");


        send_cmd(s_ChanServ, "JOIN %s", chan);

close_chan:
        if (c) {

            for (cu = c->users; cu; cu = next_cu) {
               next_cu = cu->next;
               av[0] = sstrdup(chan);
               av[1] = sstrdup(cu->user->nick);
               av[2] = sstrdup("Channel Suspended");
               send_cmd(s_ChanServ, "MODE %s +b *!*@*", av[0], av[1]);
               send_cmd(s_ChanServ, "KICK %s %s :%s", av[0], av[1], av[2]);
               do_kick(s_ChanServ, 3, av);
               free(av[0]); free(av[1]); free(av[2]);
            }
        }

        t = add_timeout(CHANNEL_INHABIT, timeout_unban, 0);
        t->data = sstrdup(chan);

        t = add_timeout(CHANNEL_INHABIT+1, timeout_leave, 0);
        t->data = sstrdup(chan);

      } else {
        notice(s_ChanServ, source, "SUSPEND list full");
        wallops(SERVER_NAME, "\2WARNING\2: Suspend list full! (%d)",
           MAX_SUSPEND);
      }
  } else
     notice(s_ChanServ, source,
           "Syntax: \2SUSPEND\2 [ADD|DEL|LIST] [<#channel>] [<time>]");
      

}

/*************************************************************************/

static void do_freeze(const char *source)
{
    ChannelInfo *ci;
    const char *chan = strtok(NULL, " ");
    User *u = finduser(source);

        if ((!chan) || (stricmp(chan, "-") == 0)) {	
            notice(s_ChanServ,source,"Syntax: \2FREEZE\2 <#channel>");
            notice(s_ChanServ,source,
                "\2/msg %s OHELP FREEZE\2 for more information",s_ChanServ);
            return;
        }
        if (*chan == '-') {
           *chan++;
           do_unfreeze(source, chan);

        } else if (!(ci = cs_findchan(chan))) {
            notice(s_ChanServ, source, "Channel %s is not registered.",chan);
        } else if (!(ci->flags & CI_FREEZECHAN)) {
            ci->flags |= CI_FREEZECHAN;
            slog("CS +Z %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
            do_break_log("CS_F", "CS +Z %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
	    wallops(s_ChanServ,
                "%s set the FREEZE flag for \2%s\2",u->nick,chan);
           if (ci->freeze)
                free(ci->freeze);
           ci->freeze = sstrdup(u->nick);

	} else {
            notice(s_ChanServ,source,"Channel %s already FROZEN.",chan);
	}
}

static void do_unfreeze(const char *source, const char *chan)
{
    ChannelInfo *ci;
    User *u = finduser(source);

        if (!chan) {	
            notice(s_ChanServ,source,"Syntax: \2FREEZE\2 -<#channel>");
            notice(s_ChanServ,source,
                "\2/msg %s OHELP UNFREEZE\2 for more information",s_ChanServ);
        } else if (!(ci = cs_findchan(chan))) {
            notice(s_ChanServ, source, "Channel %s is not registered.",chan);
        } else if (ci->flags & CI_FREEZECHAN) {
            ci->flags &= ~CI_FREEZECHAN;
            slog("CS -Z %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
            do_break_log("CS_F", "CS -Z %s (%s!%s@%s)",
                chan, u->nick, u->username, u->host);
	    wallops(s_ChanServ,
                "%s UNFROZE channel \2%s\2",u->nick,chan);
           if (ci->freeze)
               free(ci->freeze);
           ci->freeze = NULL;

	} else {
            notice(s_ChanServ,source,"Channel %s is not FROZEN.",chan);
	}
}



/*************************************************************************/

static void do_forbid(const char *source)
{
    ChannelInfo *ci;
    const char *chan = strtok(NULL, " ");
    User *u = finduser(source);

    if (!chan) {
        notice(s_ChanServ, source, "Syntax: \2FORBID\2 [-]<#channel>");
        notice(s_ChanServ, source,
              "\2/MSG %s OHELP FORBID\2 fo more information.",
              s_ChanServ);
	return;
    }
    if (readonly) {
	notice(s_ChanServ, source,
	    "Warning: Services is in read-only mode.  Changes will not be saved.");
    }

    if (*chan == '-') {
        *chan++;
        if (chan != NULL)
           do_unforbid(source, chan);
        else
           notice(s_ChanServ, source, "Syntax: FORBID [-]<#channel>");
        return;
    }

    if (ci = cs_findchan(chan)) {
        if (ci->forbid) {
           notice(s_ChanServ, source, "Channel %s already forbidden", chan);
           return;
        }
	delchan(ci);
    }
    if (ci = makechan(chan)) {
        slog("CS +F %s (%s!%s@%s)",
          chan, u->nick, u->username, u->host);
        do_break_log("CS_F", "CS +F %s (%s!%s@%s)",
          chan, u->nick, u->username, u->host);
	log("%s: %s set FORBID for channel %s", s_ChanServ, source, chan);
	ci->flags |= CI_VERBOTEN;
        if (ci->forbid)
             free(ci->forbid);
        ci->forbid = sstrdup(u->nick);
        wallops(s_ChanServ, "%s has forbidden the channel \2%s\2",
             source, chan);
	notice(s_ChanServ, source,
		"Channel \2%s\2 has been marked FORBIDden.", chan);
    } else {
        slog("CS *F %s (%s!%s@%s) [Unknown, Possible Services Bug]",
          chan, u->nick, u->username, u->host);
	log("%s: Valid FORBID for %s by %s failed", s_ChanServ,
		chan, source);
	notice(s_ChanServ, source,
		"Couldn't FORBID channel \2%s\2!", chan);
    }
}



static void do_unforbid(const char *source, const char *chan)
{
    ChannelInfo *ci;
    User *u = finduser(source);

    if (readonly) {
	notice(s_ChanServ, source,
	    "Warning: Services is in read-only mode.  Changes will not be saved.");
    }

    if (ci = cs_findchan(chan)) {
       if (!(ci->flags & CI_VERBOTEN)) {
           slog("CS -F* %s (%s!%s@%s)",
                   chan,u->nick,u->username,u->host);
           log("%s: %s tried to set unforbidden flag for non forbidden chan %s",
                   s_ChanServ, source, chan);
           notice(s_ChanServ, source, "%s is not set as FORBIDDEN", chan);
           return;
        }
        slog("CS -F %s (%s!%s@%s) [Dropped]",
                chan,u->nick,u->username,u->host);
        do_break_log("CS_F", "CS -F %s (%s!%s@%s) [Dropped]",
            chan, u->nick, u->username, u->host);
        log("%s: %s set UNFORBID for chan %s", s_ChanServ, source, chan);
        if (ci->forbid)
            free(ci->forbid);
        ci->forbid = NULL;
        delchan(ci);
        wallops(s_ChanServ, "%s UNFORBID %s [Automatically Dropped]", source, chan);
        notice(s_ChanServ, source, "Channel \2%s\2 is now UNFORBIDDEN.", chan);

    } else {
        slog("CS -F* %s (%s!%s@%s) [Unknown, Possible Services Bug]",
          chan, u->nick, u->username, u->host);
        log("%s: Valid UNFORBID for %s by %s failed", s_ChanServ,
		chan, source);
	notice(s_ChanServ, source,
                "Couldn't UNFORBID channel \2%s\2!", chan);
    }
}


/*************************************************************************/

static char *get_sponser(User *u, ChannelInfo *ci)
{

    NickInfo *ni;
    ChanAccess *access;
    char *nick;
    int i, i2, acc=0;


    ni = findnick(u->nick);

    if (is_founder(u, ni, ci, 0))
         return ci->founder;

    for (access = ci->access, i = 0; i < ci->accesscount; ++access,
                                                                  ++i) {
       if (stricmp(access->name, u->nick) == 0) {
            if (ni && (ni->flags & NI_IDENTIFIED)) {
                 nick = access->name;
                 acc = access->level;
            } else if (ni && (ni->flags & NI_RECOGNIZED)) {
                 nick = access->name;
                 acc = access->level;;
            }
       }            

    }

    for (i=0; i < MAX_IDS && u->id_nicks[i]; i++) {
         for (access = ci->access, i2 = 0; i2 < ci->accesscount;
                                           ++access, ++i2) {
             ni = findnick(u->id_nicks[i]);
             if (ni && (ni->flags & NI_IDENTIFIED)
                && stricmp(ni->nick, access->name) == 0) {
                    if (access->level > acc) {
                         nick = access->name;
                         acc = access->level;
                    }
             }
         }
    }

   if (acc)
      return nick;
   else
      return NULL;
}

static void do_why(const char *source)
{
    NickInfo *ni, *ni2;
    ChanAccess *access;
    ChannelInfo *ci;
    User *u, *u2 = finduser(source);
    char *nick, *chan, *whom;
    char **naccess;
    int ulev = 0;
    int i, i2, x;
    int recognized=0, identified=0, acc=0;


    chan = strtok(NULL, " ");
    nick = strtok(NULL, " ");

    if (!nick || strtok(NULL, " ")) {
	notice(s_ChanServ, source,
                "Syntax: \2WHY\2 <#channel> <nickname>");
	notice(s_ChanServ, source,
                "\2/msg %s HELP WHY\2 for more information.", s_ChanServ);
	return;
    }

    if (!(ci = cs_findchan(chan))) {
        notice(s_ChanServ, source, "Channel %s is not registered!", chan);
        return;
    }

    if (u2 && !is_oper(u2->nick) && 3 > (ulev = get_access(u2, ci))) {
        notice(s_ChanServ, source, "Access Denied");
        return;
    }
    ni = findnick(nick);

    if (u = finduser(nick)) {

    for (access = ci->access, i = 0; i < ci->accesscount; ++access,
                                                                  ++i) {
       if (stricmp(access->name, nick) == 0) {
            if (ni && (ni->flags & NI_IDENTIFIED)) {
                 identified++;
                 whom = access->name;

            } else if (ni && (ni->flags & NI_RECOGNIZED)) {
                 recognized++;
                 whom = access->name;
                 acc = access->level;
            }
       } else {
             ni2 = findnick(access->name);
             if (ni2) {

                for (naccess = ni2->access, x=0; x < ni2->accesscount;
                                                ++naccess, ++x) {
                     if (is_on_access(u, ni2)) {
                         if (access->level > acc) {

                            whom = access->name;
                            identified = 0;
                            recognized++;
                         }
                     }
                  }
              }

          }

       }            


       for (i=0; i < MAX_IDS && u->id_nicks[i]; i++) {
         for (access = ci->access, i2 = 0; i2 < ci->accesscount;
                                           ++access, ++i2) {
             ni = findnick(u->id_nicks[i]);
             if (ni && (ni->flags & NI_IDENTIFIED)
                && stricmp(ni->nick, access->name) == 0) {
                    whom = access->name;
                    if (access->level > acc) {
                       identified = 1;
                       whom = access->name;
                    }
                       
             }
         }
       }

       ni = findnick(nick);
 
       ulev = get_access(u, ci);

        if (ulev < 0) {
            notice(s_ChanServ, source,
                   "%s does not have access to channel %s. "
                   "Reason: Banned", nick, chan);
        } else if (ulev == 0) {
            notice(s_ChanServ, source,
                   "%s does not have access to channel %s. "
                   "Reason: Basic User", nick, chan);
        } else if (ulev == 15) {
             if (is_identified(u, ci)) {
                  notice(s_ChanServ, source,
                      "%s has founder access to channel %s. "
                      "Reason: Identification to the channel.",
                       nick, chan);
             } else
                notice(s_ChanServ, source,
                       "%s has founder access to channel %s. "
                       "Reason: %s.",
                       nick, chan,
                       (ni && (ni->flags & NI_IDENTIFIED)) ?
                            "Identification to the founder's nick" :
                            "Recognition to the founder's nick");

        } else if (ulev > 0) {
            if (!whom) {
                notice(s_ChanServ, source,
                       "%s has %s%s%s%s access to channel %s. "
                       "Reason: User Host matches channel access mask.",
                       nick,
                        (ulev == 3) ? "AVOICE" : "",
                        (ulev == 5) ? "AOP" : "",
                        (ulev == 10) ? "SOP" : "",
                        (ulev == 13) ? "Co-Founder" : "",
                        chan);
            } else {
                notice(s_ChanServ, source, 
                       "%s has %s%s%s%s access to channel %s. "
                       "Reason: %s to the nick %s.",
                        nick,
                        (ulev == 3) ? "AVOICE" : "",
                        (ulev == 5) ? "AOP" : "",
                        (ulev == 10) ? "SOP" : "",
                        (ulev == 13) ? "Co-Founder" : "",
                        chan,
                        (identified)
                             ? "Identification" : "Recognition",
                        whom);
            }
        } else {
            notice(s_ChanServ, source, "Channel: \2%s\2 Nick: \2%s\2 (%s@%s) Access: Unknown", chan, nick, u->username, u->host);
        }
    } else {
       notice(s_ChanServ, source, "Nick %s not online", nick);
    }
}

static void do_flist(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    ChannelInfo *ci;
    unsigned int i;

    if (!cmd) {
        notice(s_ChanServ,source,"Syntax: \2FLIST\2 [HELD|MARKED|FORBIDDEN|FROZEN]");
	notice(s_ChanServ,source,
	    "\2/msg %s OHELP FLIST\2 for more information",s_ChanServ);

    } else if (stricmp(cmd, "HELD") == 0) {
	notice(s_ChanServ,source,"\2HELD\2 channel listing:");
        for (i = 0; i < 256; ++i) {
            for (ci = chanlists[i]; ci; ci = ci->next) {
              if (ci->hold) {
                 notice(s_ChanServ,source,"%-20s %s",ci->name,ci->desc);
                 notice(s_ChanServ, source,"Set by: %s",ci->hold);
              }
	    }
	}
	notice(s_ChanServ,source,"\2End of List\2");
    } else if (stricmp(cmd, "MARKED") == 0) {
        notice(s_ChanServ,source,"\2MARKED\2 channel listing:");
        for (i = 0; i < 256; ++i) {
            for (ci = chanlists[i]; ci; ci = ci->next) {
              if (ci->mark) {
                 notice(s_ChanServ,source,"%-20s  %s",ci->name,ci->desc);
                 notice(s_ChanServ, source,"Set by: %s",ci->mark);
              }
            }
        }
        notice(s_ChanServ,source,"\2End of List\2");
    } else if (stricmp(cmd, "FROZEN") == 0) {
        notice(s_ChanServ,source,"\2FROZEN\2 channel listing:");
        for (i = 0; i < 256; ++i) {
            for (ci = chanlists[i]; ci; ci = ci->next) {
               if (ci->freeze) {
                  notice(s_ChanServ,source,"%-20s  %s",ci->name, ci->desc);
                 notice(s_ChanServ, source,"Set by: %s",ci->freeze);
              }
            }
        }
        notice(s_ChanServ,source,"\2End of List\2");

    } else if (stricmp(cmd, "FORBIDDEN") == 0) {
        notice(s_ChanServ,source,"\2FORBIDDEN\2 channel listing:");
        for (i = 0; i < 256; ++i) {
            for (ci = chanlists[i]; ci; ci = ci->next) {
               if (ci->forbid) {
                 notice(s_ChanServ,source,"%s",ci->name);
                 notice(s_ChanServ, source,"Set by: %s",ci->forbid);
              }
            }
        }
        notice(s_ChanServ,source,"\2End of List\2");
    }
}

static void do_ostopic(const char *source)
{
    const char *chan = strtok(NULL, " ");
    const char *itopic = strtok(NULL, "");


    if (!chan || !itopic) {
       notice(s_ChanServ, source, "Syntax: \2TOPIC\2 <#channel> <topic>");

    } else if (!(findchan(chan))) {
       notice(s_ChanServ, source, "Channel %s doesn't exist!", chan);

    } else {

    send_cmd(s_ChanServ, "TOPIC %s %s 0 :%s (%s)", chan,
                source, itopic, source);
  }

}



static void do_level(const char *source)
{
    const char *chan = strtok(NULL," ");
    const char *who = strtok(NULL," ");
    ChannelInfo *ci;
    User *u = finduser(source);

    if (!chan || !who) {
        notice(s_ChanServ,source,"Syntax: \2LEVEL\2 <#channel> "
                "<to-whom>");
        notice (s_ChanServ,source,"Type: \2/msg %s HELP LEVEL\2 "
		"for more information.",s_ChanServ);
	return;
    }
    ci = cs_findchan(chan);
    if (!ci) {
	notice(s_ChanServ,source,"%s isnt registered.",chan);
	return;
    }
    if (!u)
	return;
    if (get_access(u,ci) < 13) {
	notice(s_ChanServ,source,"Permission Denied.");
	return;
    }
    if (stricmp(who,"OPERS") == 0) {
	if (ci->flags & CI_OPERONLY)
	    ci->flags &= ~CI_OPERONLY;
	if (ci->flags & CI_SOPONLY)
	    ci->flags &= ~CI_SOPONLY;
	if (ci->flags & CI_SAONLY)
	    ci->flags &= ~CI_SAONLY;
	if (ci->flags & CI_SRAONLY)
	    ci->flags &= ~CI_SRAONLY;
	if (ci->flags & CI_CODERONLY)
	    ci->flags &= ~CI_CODERONLY;
        if (ci->flags & CI_ABUSEONLY)
            ci->flags &= ~CI_ABUSEONLY;
	ci->flags |= CI_OPERONLY;
        notice(s_ChanServ,source,"LEVEL for \2%s\2 set to \2%s\2",
		chan,who);
	return;
    } else if (stricmp(who,"SOP") == 0) {
        if (ci->flags & CI_OPERONLY)
            ci->flags &= ~CI_OPERONLY;
        if (ci->flags & CI_SOPONLY)
            ci->flags &= ~CI_SOPONLY;
        if (ci->flags & CI_SAONLY)
            ci->flags &= ~CI_SAONLY;
        if (ci->flags & CI_SRAONLY)
            ci->flags &= ~CI_SRAONLY;
        if (ci->flags & CI_CODERONLY)
            ci->flags &= ~CI_CODERONLY;
        if (ci->flags & CI_ABUSEONLY)
            ci->flags &= ~CI_ABUSEONLY;
        ci->flags |= CI_SOPONLY;
        notice(s_ChanServ,source,"LEVEL for \2%s\2 set to \2%s\2",
                chan,who);
        return;
    } else if (stricmp(who,"SA") == 0) {
        if (ci->flags & CI_OPERONLY)
            ci->flags &= ~CI_OPERONLY;
        if (ci->flags & CI_SOPONLY)
            ci->flags &= ~CI_SOPONLY;
        if (ci->flags & CI_SAONLY)
            ci->flags &= ~CI_SAONLY;
        if (ci->flags & CI_SRAONLY)
            ci->flags &= ~CI_SRAONLY;
        if (ci->flags & CI_CODERONLY)
            ci->flags &= ~CI_CODERONLY;
        if (ci->flags & CI_ABUSEONLY)
            ci->flags &= ~CI_ABUSEONLY;
        ci->flags |= CI_SAONLY;
        notice(s_ChanServ,source,"LEVEL for \2%s\2 set to \2%s\2",
                chan,who);
        return;
    } else if (stricmp(who,"SRA") == 0) {
        if (ci->flags & CI_OPERONLY)
            ci->flags &= ~CI_OPERONLY;
        if (ci->flags & CI_SOPONLY)
            ci->flags &= ~CI_SOPONLY;
        if (ci->flags & CI_SAONLY)
            ci->flags &= ~CI_SAONLY;
        if (ci->flags & CI_SRAONLY)
            ci->flags &= ~CI_SRAONLY;
        if (ci->flags & CI_CODERONLY)
            ci->flags &= ~CI_CODERONLY;
        if (ci->flags & CI_ABUSEONLY)
            ci->flags &= ~CI_ABUSEONLY;
        ci->flags |= CI_SRAONLY;
        notice(s_ChanServ,source,"LEVEL for \2%s\2 set to \2%s\2",
                chan,who);
        return;
    } else if (stricmp(who,"MASTERS") == 0) {
        if (ci->flags & CI_OPERONLY)
            ci->flags &= ~CI_OPERONLY;
        if (ci->flags & CI_SOPONLY)
            ci->flags &= ~CI_SOPONLY;
        if (ci->flags & CI_SAONLY)
            ci->flags &= ~CI_SAONLY;
        if (ci->flags & CI_SRAONLY)
            ci->flags &= ~CI_SRAONLY;
        if (ci->flags & CI_CODERONLY)
            ci->flags &= ~CI_CODERONLY;
	if (ci->flags & CI_ABUSEONLY)
	    ci->flags &= ~CI_ABUSEONLY;
        ci->flags |= CI_CODERONLY;
        notice(s_ChanServ,source,"LEVEL for \2%s\2 set to \2%s\2",
                chan,who);
        return;
    } else if (stricmp(who,"ABUSE") == 0) {
        if (ci->flags & CI_OPERONLY)
            ci->flags &= ~CI_OPERONLY;
        if (ci->flags & CI_SOPONLY)
            ci->flags &= ~CI_SOPONLY;
        if (ci->flags & CI_SAONLY)
            ci->flags &= ~CI_SAONLY;
        if (ci->flags & CI_SRAONLY)
            ci->flags &= ~CI_SRAONLY;
        if (ci->flags & CI_CODERONLY)
            ci->flags &= ~CI_CODERONLY;
        if (ci->flags & CI_ABUSEONLY)
            ci->flags &= ~CI_ABUSEONLY;
        ci->flags |= CI_ABUSEONLY;
        notice(s_ChanServ,source,"LEVEL for \2%s\2 set to \2%s\2",
                chan,who);
        return;
    } else if (stricmp(who,"NONE") == 0) {
        if (ci->flags & CI_OPERONLY)
            ci->flags &= ~CI_OPERONLY;
        if (ci->flags & CI_SOPONLY)
            ci->flags &= ~CI_SOPONLY;
        if (ci->flags & CI_SAONLY)
            ci->flags &= ~CI_SAONLY;
        if (ci->flags & CI_SRAONLY)
            ci->flags &= ~CI_SRAONLY;
        if (ci->flags & CI_CODERONLY)
            ci->flags &= ~CI_CODERONLY;
        if (ci->flags & CI_ABUSEONLY)
            ci->flags &= ~CI_ABUSEONLY;
        notice(s_ChanServ,source,"LEVEL for \2%s\2 set to \2%s\2",
                chan,who);
        return;
    } else
        notice(s_ChanServ,source,"Unknown LEVEL who setting");
}

void reset_all_levels()
{
    unsigned int i;
    ChannelInfo *ci;

        for (i = 33; i < 256; ++i) {
            for (ci = chanlists[i]; ci; ci = ci->next) {
		reset_levels(ci);
	    }
	}
}

/*************************************************************************/

static void do_unhold(const char *source, const char *chan)
{
     User *u = finduser(source);
     ChannelInfo *ci;

     if (!(ci = cs_findchan(chan))) {
          notice(s_ChanServ, source, "Channel %s is not registered.",chan);
     } else if (ci->flags & CI_HELDCHAN) {
          ci->flags &= ~CI_HELDCHAN;
          if (ci->hold)
              free(ci->hold);
          ci->hold = NULL;

          slog("CS -H %s (%s!%s@%s)",
              chan,u->nick,u->username,u->host);
          do_break_log("CS_F", "CS -H %s (%s!%s@%s)",
              chan, u->nick, u->username, u->host);
          wallops(s_ChanServ,
              "%s unset the HELD flag for \2%s\2",u->nick,chan);
     } else {
          notice(s_ChanServ,source,"Channel \2%s\2 is not held.",chan);
     }
}

/*************************************************************************/

static void do_unmark(const char *source, const char *chan)
{
     User *u = finduser(source);
     ChannelInfo *ci;

     if (!(ci = cs_findchan(chan))) {
          notice(s_ChanServ, source, "Channel %s is not registered.",chan);
     } else if (ci->flags & CI_MARKCHAN) {
          ci->flags &= ~CI_MARKCHAN;
         if (ci->mark)
             free(ci->mark);
          ci->mark = NULL;
          slog("CS -M %s (%s!%s@%s)",
              chan,u->nick,u->username,u->host);
          do_break_log("CS_F", "CS -M %s (%s!%s@%s)",
              chan, u->nick, u->username, u->host);
          wallops(s_ChanServ,
              "%s unset the MARK flag for \2%s\2",u->nick,chan);

     } else {
          notice(s_ChanServ,source,"Channel \2%s\2 is not marked.",chan);
     }

}

/*************************************************************************/

static void do_memolev(User *u, ChannelInfo *ci, const char *param)
{
   if (!param) {
      notice(s_ChanServ, u->nick,
        "Syntax: \2MEMO\2 <#channel> <level>");
      notice(s_ChanServ, u->nick,
         "For Help, Type \"/msg %s HELP SET MEMO\" for help.",
           s_MemoServ);
      return;
   }

   if (get_access(u, ci) < 13) {
      notice(s_ChanServ, u->nick, "Access Denied");
      return;
   }

   if (!(stricmp(param, "NONE") == 0) && (stricmp(param, "AVOICE") == 0)
      && (stricmp(param, "AOP") == 0) && (stricmp(param, "SOP") == 0)
      && (stricmp(param, "CFOUNDER") == 0) && (stricmp(param, "CF") == 0)
      && (stricmp(param, "FOUNDER") == 0)) {
          notice(s_ChanServ, u->nick,
             "\2MEMO\2 options may only be NONE, AVOICE, AOP, SOP, CFOUNDER, or FOUNDER");
          return;
   }

   if (ci->flags & CI_MEMO_AV)
       ci->flags &= ~CI_MEMO_AV;
   if (ci->flags & CI_MEMO_AOP)
       ci->flags &= ~CI_MEMO_AOP;
  if (ci->flags & CI_MEMO_SOP)
      ci->flags &= ~CI_MEMO_SOP;
  if (ci->flags & CI_MEMO_CF)
      ci->flags &= ~CI_MEMO_CF;
  if (ci->flags & CI_MEMO_FR)
      ci->flags &= ~CI_MEMO_FR;
  if (ci->flags & CI_MEMO_NONE)
      ci->flags &= ~CI_MEMO_NONE;

   if (stricmp(param, "NONE") == 0) {
       ci->flags |= CI_MEMO_NONE;
       notice(s_ChanServ, u->nick,
          "Channel Memos for %s are now disabled", ci->name);

   } else if (stricmp(param, "AVOICE") == 0) {
       ci->flags |= CI_MEMO_AV;
       notice(s_ChanServ, u->nick,
          "Memo level for %s is now set to \2%s\2", ci->name, param);

   } else if (stricmp(param, "AOP") == 0) {
       ci->flags |= CI_MEMO_AOP;
       notice(s_ChanServ, u->nick,
          "Memo level for %s is now set to \2%s\2", ci->name, param);

   } else if (stricmp(param, "SOP") == 0) {
       ci->flags |= CI_MEMO_SOP;
       notice(s_ChanServ, u->nick,
          "Memo level for %s is now set to \2%s\2", ci->name, param);

   } else if (stricmp(param, "CFOUNDER") == 0 || stricmp(param, "CF") == 0) {
       ci->flags |= CI_MEMO_CF;
       notice(s_ChanServ, u->nick,
          "Memo level for %s is now set to \2%s\2", ci->name, param);

   } else if (stricmp(param, "FOUNDER") == 0) {
       ci->flags |= CI_MEMO_FR;
       notice(s_ChanServ, u->nick,
          "Memo level for %s is now set to \2%s\2", ci->name, param);
   }
}

/*************************************************************************/

static void do_wipe(const char *source)
{
   ChannelInfo *ci;
   const char *chan = strtok(NULL, " ");
   const char *what = strtok(NULL, " ");
   ChanAccess *access;
   AutoKick *akick;
   int i, c=0, c2=0, check=0, lev=0;

   if (!chan || !what) {
      notice(s_ChanServ, source,
         "Syntax: \2WIPE\2 <#channel> [CFOUNDER|SOP|AOP|AVOICE|AKICK|ALL]");
      return;

   }
   if (!(ci = cs_findchan(chan))) {
       notice(s_ChanServ, source,
          "Channel %s is not registered", chan);
       return;
   }

   if ((stricmp(what, "AKICK") == 0) || (stricmp(what, "ALL") == 0)) {
       for (akick = ci->akick, i = 0; ci->akickcount; ++akick, ++i) {
           free(akick->name);
           akick->name = NULL;
           if (akick->reason)
               free(akick->reason);
           ++c;
           --ci->akickcount;
          if (!ci->akickcount) {
              free(ci->akick);
              ci->akick = NULL;
          }
       }
   }

   if (stricmp(what, "CFOUNDER") == 0)
         (unsigned int)lev = 13;
   else if (stricmp(what, "SOP") == 0)
         (unsigned int)lev = 10;
   else if (stricmp(what, "AOP") == 0)
         (unsigned int)lev = 5;
   else if (stricmp(what, "AVOICE") == 0)
         (unsigned int)lev = 3;
   else if (stricmp(what, "ALL") == 0)
         (unsigned int)lev = 1;
   else if (stricmp(what, "AKICK") == 0)
         (unsigned int)lev = 0;
   else {
       notice(s_ChanServ, source,
           "Syntax: \2WIPE\2 <#channel> [CFOUNDER|SOP|AOP|AVOICE|AKICK|ALL]");
       return;
   }

   check = ci->accesscount;  

   for (access = ci->access, i = 0; i < check; ++access,++i) {
       if (!lev)
           break;

       if ((lev != access->level) && (lev != 1))
           continue;


       free(access->name);
       free(access->sponser);
       access->name = NULL;
       access->in_use = 0;
       ++c; ++c2;
       --ci->accesscount;
       if (!ci->accesscount) {
           free(ci->access);
           ci->access = NULL;
           break;
       }
       if ((i-c2) < ci->accesscount) {
            bcopy(access+1, access, sizeof(ChanAccess)
                                      * (ci->accesscount - (i-c2)));
            --access;
       }
   }

   switch(lev) {

      case 13:
         wallops(s_ChanServ,
            "%s had me wipe the CFOUNDER list on %s [%d deleted]",
            source, chan, c);
         return;
      case 10:
         wallops(s_ChanServ,
            "%s had me wipe the SOP list on %s [%d deleted]",
            source, chan, c);
         return;
      case 5:
         wallops(s_ChanServ,
            "%s had me wipe the AOP list on %s [%d deleted]",
            source, chan, c);
         return;
      case 3:
         wallops(s_ChanServ,
            "%s had me wipe the AVOICE list on %s [%d deleted]",
            source, chan, c);
         return;
      case 1:
         wallops(s_ChanServ,
            "%s had me wipe ALL access lists on %s [%d deleted]",
            source, chan, c);
         return;
      case 0:
         wallops(s_ChanServ,
            "%s had me wipe the akick list on %s [%d deleted]",
            source, chan, c);
         return;
   }
}

/*************************************************************************/

static void do_remove(const char *source)
{
   NickInfo *ni = findnick(source);
   ChannelInfo *ci;
   ChanAccess *access;
   const char *chan = strtok(NULL, " ");
   int i, x=0, check=0, z=0;
   User *u = finduser(source);

   if (!u)
     return;

   if (!chan) {
      notice(s_ChanServ, source,
         "Syntax: \2REMOVE\2 <#channel>");
      return;
   }

   if (!(ci = cs_findchan(chan))) {
       notice(s_ChanServ, source,
          "Channel %s is not registered", chan);
       return;
   }

   if (!ni) {
     notice(s_ChanServ, source, "Your nick (%s) is not registered!", source);
     return;
   }

   if (!(ni->flags & NI_IDENTIFIED)) {
     notice(s_ChanServ, source, "You have not identified to your nick!");
     return;
   }

   if (!cremove) {
        notice(s_ChanServ, source,
           "Sorry, the REMOVE command has been disabled by the network");
        return;
   }

   if ((x = get_access(u, ci)) > 0) {
       if (x==15) {
          notice(s_ChanServ, source,
             "The REMOVE command may not be used by the founder");
          return;
       }
       check = ci->accesscount;
       for (access = ci->access, i = 0; i < check; ++access,++i) {
           if (stricmp(access->name, source) == 0) {
              csremove += 1;
              free(access->name);
              free(access->sponser);
              access->in_use = 0;
              access->name = NULL;
              --ci->accesscount;
              if (i < ci->accesscount)
                  bcopy(access+1, access, sizeof(ChanAccess)
                                        * (ci->accesscount - i));

/* we need z incase the nick matches an access mask pulled from
   get_access(), which would return stupid replies.. */
           z=1;
           break;
           }
       }
   }
   if (z)
   notice(s_ChanServ, source,
      "Your nick (%s) has been removed from the %s %s%s%s%s list",
         source, chan,
         x==3 ? "AVOICE" : "", x==5 ? "AOP" : "",
         x==10 ? "SOP" : "", x==13 ? "CFOUNDER" : "");
   else
   notice(s_ChanServ, source,
       "Your nick (%s) was not found on any %s access list", 
        source, chan);
}

/*************************************************************************/

int check_opguard(User *user, const char *chan)
{
   ChannelInfo *ci;

    if (!(ci = cs_findchan(chan)))
        return 0;

    if (ci->flags & CI_OPGUARD) {
         if (!is_on_id_list(user->nick, user->nick))
              return 0;
         else if (check_access(user, ci, CA_AUTOVOICE))
              return 1;
         else
              return 0;
    } else

       return 1;
}

/*************************************************************************/

int check_invite(User *user, const char *chan)
{
   ChannelInfo *ci;
   struct u_chanlist *c;
   char *av[3];
   Timeout *t;

    if (!(ci = cs_findchan(chan)))
        return 0;

    if (get_access(user, ci) > 0)
        return 1;

    if (ci->mlock_on & CMODE_I) {
        send_cmd(s_ChanServ, "JOIN %s", chan);
        send_cmd(s_ChanServ, "MODE %s +b *!*@*", chan);
        send_cmd(s_ChanServ, "KICK %s %s :Sorry, Channel is invite only", 
                chan, user->nick);
        t = add_timeout(CHANNEL_INHABIT, timeout_leave, 0);
        t->data = sstrdup(chan); 

        av[0] = sstrdup(chan);
        av[1] = sstrdup(user->nick);
        do_kick(s_ChanServ, 3, av);
        for (c = user->chans; c && stricmp(chan, c->chan->name) != 0;
                                                               c = c->next)
            ;
        if (c)
           chan_deluser(user, c->chan);
        free(av[0]); free(av[1]);
        return 0;

    }
   return 1;

}
/*************************************************************************/

static void flush_auth(const char *source)
{
   ChannelInfo *ci;
   const char *chan = strtok(NULL, " ");

   if (!chan)
       notice(s_ChanServ, source,
           "Syntax: FLUSH <#chan>");
   else if (!(ci = cs_findchan(chan)))
       notice(s_ChanServ, source,
           "Channel %s not found in %s database", chan, s_ChanServ);
   else if (!ci->auth)
       notice(s_ChanServ, source,
           "Channel %s doesn't have an authorization code set!", chan);
   else {
      ci->auth = 0;
      notice(s_ChanServ, source,
           "Authorization code for channel %s has been cleared.");
   }


}

/*************************************************************************/



#endif	/* !SKELETON */

