/* Compatibility routines.
 *
 * Services is copyright (c) 1996-1998 Andy Church.
 *     E-mail: <achurch@dragonfire.net>
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "../inc/services.h"

/*************************************************************************/

#if !HAVE_SNPRINTF

/* [v]snprintf: Like [v]sprintf, but don't write more than len bytes
 *              (including null terminator).  Return the number of bytes
 *              written.
 */

int vsnprintf(char *buf, size_t len, const char *fmt, va_list args)
{
    if (len <= 0)
	return 0;
#if BAD_SNPRINTF
# undef vsnprintf
    *buf = 0;
    vsnprintf(buf, len, fmt, args);
# define vsnprintf my_vsnprintf
    return strlen(buf);
#else
    /* DANGER WILL ROBINSON!  There's no real portable way to implement
     * vsnprintf(), so we cheat and call vsprintf.  Buffer overflows
     * galore! */
    return vsprintf(buf, fmt, args);
#endif
}

int snprintf(char *buf, size_t len, const char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    return vsnprintf(buf, len, fmt, args);
}

#endif /* HAVE_SNPRINTF */

/*************************************************************************/

#if !HAVE_STRICMP && !HAVE_STRCASECMP

/* stricmp, strnicmp:  Case-insensitive versions of strcmp() and
 *                     strncmp().
 */

int stricmp(const char *s1, const char *s2)
{
    register int c;

    while ((c = tolower(*s1)) == tolower(*s2)) {
	if (c == 0) return 0;
	s1++; s2++;
    }
    if (c < tolower(*s2))
	return -1;
    return 1;
}

int strnicmp(const char *s1, const char *s2, size_t len)
{
    register int c;

    if (!len) return 0;
    while ((c = tolower(*s1)) == tolower(*s2) && len > 0) {
	if (c == 0 || --len == 0) return 0;
	s1++; s2++;
    }
    if (c < tolower(*s2))
	return -1;
    return 1;
}
#endif

/*************************************************************************/

#if !HAVE_STRDUP
char *strdup(const char *s)
{
    char *new = malloc(strlen(s)+1);
    if (new)
	strcpy(new, s);
    return new;
}
#endif

/*************************************************************************/

#if !HAVE_STRSPN
size_t strspn(const char *s, const char *accept)
{
    size_t i = 0;

    while (*s && strchr(accept, *s))
	++i, ++s;
    return i;
}
#endif

/*************************************************************************/

#if !HAVE_STRERROR
# if HAVE_SYS_ERRLIST
extern char *sys_errlist[];
# endif

char *strerror(int errnum)
{
# if HAVE_SYS_ERRLIST
    return sys_errlist[errnum];
# else
    static char buf[32];
    sprintf(buf, "Error %d", errnum);
    return buf;
# endif
}
#endif

/*************************************************************************/

#if !HAVE_STRSIGNAL
char *strsignal(int signum)
{
    static char buf[32];
    switch (signum) {
	case SIGHUP:	strcpy(buf, "Hangup"); break;
	case SIGINT:	strcpy(buf, "Interrupt"); break;
	case SIGQUIT:	strcpy(buf, "Quit"); break;
#ifdef SIGILL
	case SIGILL:	strcpy(buf, "Illegal instruction"); break;
#endif
#ifdef SIGABRT
	case SIGABRT:	strcpy(buf, "Abort"); break;
#endif
#if defined(SIGIOT) && (!defined(SIGABRT) || SIGIOT != SIGABRT)
	case SIGIOT:	strcpy(buf, "IOT trap"); break;
#endif
#ifdef SIGBUS
	case SIGBUS:	strcpy(buf, "Bus error"); break;
#endif
	case SIGFPE:	strcpy(buf, "Floating point exception"); break;
	case SIGKILL:	strcpy(buf, "Killed"); break;
	case SIGUSR1:	strcpy(buf, "User signal 1"); break;
	case SIGSEGV:	strcpy(buf, "Segmentation fault"); break;
	case SIGUSR2:	strcpy(buf, "User signal 2"); break;
	case SIGPIPE:	strcpy(buf, "Broken pipe"); break;
	case SIGALRM:	strcpy(buf, "Alarm clock"); break;
	case SIGTERM:	strcpy(buf, "Terminated"); break;
	case SIGSTOP:	strcpy(buf, "Suspended (signal)"); break;
	case SIGTSTP:	strcpy(buf, "Suspended"); break;
	case SIGIO:	strcpy(buf, "I/O error"); break;
	default:	sprintf(buf, "Signal %d\n", signum); break;
    }
    return buf;
}
#endif

/*************************************************************************/
