/* MemoServ functions.
 *
 * SirvNET Services is copyright (c) 1998-2002 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services(c) by Andy Church.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */


#include "../inc/services.h"
#include "ms-help.c"

static MemoList *memolists[256];	/* One for each initial character */

static MemoList *find_memolist(const char *nick);
static void del_memolist(MemoList *ml);

static void alpha_insert_memolist(MemoList *ml);

static void do_ohelp(const char *source);
static void do_help(const char *source);
static void do_send(const char *source);
static void do_csend(const char *source);
static void do_chansend(const char *source, const char *chan, const char *text);
static void do_unsend(const char *source);
static void do_forward(const char *source);
static void do_list(const char *source, const char *nick, int info);
static void do_read(const char *source);
static void do_del(const char *source);
static void do_purge(const char *source);
static void do_undel(const char *source);
static void do_news(const char *source);
static void do_set(const char *source);
static void do_set_notify(NickInfo *ni, const char *param);
static void do_limit(NickInfo *ni, char *param);
static void do_global(const char *source);
static void do_info(const char *source);
static void do_ignore(const char *source);
static void do_osend(const char *source);
static void staff_send(const char *nick, const char *text, const char *source);

/*************************************************************************/

/* Return information on memory use.  Assumes pointers are valid. */

void get_memoserv_stats(long *nrec, long *memuse)
{
    long count = 0, mem = 0;
    int i, j;
    MemoList *ml;

    for (i = 0; i < 256; i++) {
	for (ml = memolists[i]; ml; ml = ml->next) {
	    count++;
	    mem += sizeof(*ml);
	    mem += sizeof(Memo) * ml->n_memos;
	    for (j = 0; j < ml->n_memos; j++) {
		if (ml->memos[j].text)
		    mem += strlen(ml->memos[j].text)+1;
	    }
	}
    }
    *nrec = count;
    *memuse = mem;
}

/*************************************************************************/
/*************************************************************************/

/* Check for marked deleted memos when user logs offline */
#ifndef SKELETON

void check_del_memos(User *user)
{
    MemoList *ml;
    Memo *m;
    unsigned int i;

    if (!(ml = find_memolist(user->nick))) {
        return;

    } else {

        /* Delete all memos.  This requires freeing the memory holding
         * the text of each memo and flagging that there are no memos
         * left. */

         for (i = 0; i < ml->n_memos; ++i) {
            m = &ml->memos[i];
            if (m->flags & MF_DEL) {
                free(ml->memos[i].text);
                if (i < ml->n_memos)     /* Move remaining memos down a slot */
                   bcopy(ml->memos+i+1,ml->memos+i,sizeof(Memo)*(ml->n_memos-i));
		--ml->n_memos;		 /* One less memo now */
            }
        }
    }

    /* Did we delete the last memo?  If so, delete this MemoList. */
    if (ml->n_memos == 0)
       del_memolist(ml);

}
#endif

/*************************************************************************/
/*************************************************************************/

/* memoserv:  Main MemoServ routine. */

void memoserv(const char *source, char *buf)
{
    char *cmd, *s;
    NickInfo *ni;
    User *u = finduser(source);

    if (!u)
      return;

    cmd = strtok(buf, " ");

    if (!cmd) {
	return;

    } else if (stricmp(cmd, "\1PING") == 0) {
	if (!(s = strtok(NULL, "")))
	    s = "\1";
	notice(s_MemoServ, source, "\1PING %s", s);


#ifdef SKELETON

    } else {
	notice(s_MemoServ, source, "%s is currently offline.", s_MemoServ);
    }

#else

    } else if (stricmp(cmd, "OHELP") == 0) {
        if (!is_oper(source)) {
             notice(s_MemoServ, source,
                  "Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
                  cmd, s_MemoServ);
             return;
        }
        do_ohelp(source);

    } else if (stricmp(cmd, "HELP") == 0) {
	do_help(source);

    } else if (stricmp(cmd, "NEWS") == 0) {
        do_news(source);

    } else if (stricmp(cmd, "INFO") == 0) {
        do_info(source);

    } else if (!(ni = findnick(source))) {
	notice(s_MemoServ, source, "Your nick is not registered.  Type "
			"\2/msg %s HELP\2 for information on registering "
			"your nickname.", s_NickServ);

    } else if (!(ni->flags & (NI_RECOGNIZED | NI_IDENTIFIED))) {
	notice(s_MemoServ, source,
		"Please identify with %s first, using the command:",
		s_NickServ);
	notice(s_MemoServ, source,
		"/msg %s IDENTIFY \37password\37", s_NickServ);

    } else if (stricmp(cmd, "LIST") == 0) {
	do_list(source, source, 0); // double source rules out list_oper


    } else if (!(ni->flags & NI_IDENTIFIED)) {
	notice(s_MemoServ, source, "Identification required for that command."
			"  Type \2/msg %s IDENTIFY \37password\37\2 to "
			"access your memos.", s_NickServ);

    } else if (stricmp(cmd, "SEND") == 0) {
	do_send(source);

    } else if (stricmp(cmd, "CSEND") == 0) {
        do_csend(source);

    } else if (stricmp(cmd, "UNSEND") == 0) {
	do_unsend(source);

    } else if (stricmp(cmd, "FORWARD") == 0) {
        do_forward(source);

    } else if (stricmp(cmd, "READ") == 0) {
	do_read(source);

    } else if (stricmp(cmd, "DEL") == 0) {
	do_del(source);

    } else if (stricmp(cmd, "PURGE") == 0) {
        do_purge(source);

    } else if (stricmp(cmd, "UNDEL") == 0) {
        do_undel(source);

    } else if (stricmp(cmd, "SET") == 0) {
	do_set(source);

    } else if (!is_oper(source)) {
	notice(s_MemoServ, source,
		"Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
		cmd, s_MemoServ);
        return;


    } else if (!check_o_access(source, cmd)) {
        notice(s_MemoServ, source, "Permission Denied");

    } else if (stricmp(cmd, "OSEND") == 0) {
        do_osend(source);

    } else if (stricmp(cmd, "LIMIT") == 0) {
        char *param  = strtok(NULL, "");
        do_limit(ni, param);

    } else if (stricmp(cmd, "GLOBAL") == 0) {
	do_global(source);

    } else {
        User *u = finduser(source);
	notice(s_MemoServ, source,
		"Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
		cmd, s_MemoServ);
        slog("MS *C %s (%s@%s) [%s]",
            source, u->username, u->host, cmd);

    }

#endif
}

/*************************************************************************/

#ifndef SKELETON



/* load_ms_dbase, save_ms_dbase:  Load/save memo data. */

void load_ms_dbase(void)
{
    FILE *f;
    int ver, i, j, len;
    MemoList *ml;
    Memo *memos;

    if (!(f = open_db(s_MemoServ, MEMOSERV_DB, "r")))
	return;
    switch (ver = get_file_version(f, MEMOSERV_DB)) {
      case 5:
      case 6:
      case 7:
      case 8:
	for (i = 33; i < 256; ++i) {
	    while (fgetc(f) == 1) {
		ml = smalloc(sizeof(MemoList));
		if (1 != fread(ml, sizeof(MemoList), 1, f))
		    fatal_perror("Read error on %s", MEMOSERV_DB);
		alpha_insert_memolist(ml);
		ml->memos = memos = smalloc(sizeof(Memo) * ml->n_memos);
		fread(memos, sizeof(Memo), ml->n_memos, f);
                for (j = 0; j < ml->n_memos; ++j, ++memos) {
		    memos->text = read_string(f, MEMOSERV_DB);
                  if (memos->chan)
                    memos->chan = read_string(f, MEMOSERV_DB);
                }
	    }
	}
	break;
      default:
	fatal("Unsupported version number (%d) on %s", ver, MEMOSERV_DB);
    } /* switch (version) */
    close_db(f, MEMOSERV_DB);
}

void save_ms_dbase(void)
{
    FILE *f;
    int i, j, len;
    MemoList *ml;
    Memo *memos;

    if (!(f = open_db(s_MemoServ, MEMOSERV_DB, "w")))
	return;
    for (i = 33; i < 256; ++i) {
	for (ml = memolists[i]; ml; ml = ml->next) {
	    fputc(1, f);
	    if (1 != fwrite(ml, sizeof(MemoList), 1, f) ||
		    ml->n_memos !=
			fwrite(ml->memos, sizeof(Memo), ml->n_memos, f))
		fatal_perror("Write error on %s", MEMOSERV_DB);
            for (memos = ml->memos, j = 0; j < ml->n_memos; ++memos, ++j) {
		write_string(memos->text, f, MEMOSERV_DB);
                if (memos->chan)
                   write_string(memos->chan, f, MEMOSERV_DB);
            }

	}
	fputc(0, f);
    }
    close_db(f, MEMOSERV_DB);
}

/*************************************************************************/

/* check_memos:  See if the given nick has any waiting memos, and send a
 *               NOTICE to that nick if so.
 */

void check_memos(const char *nick)
{
    NickInfo *ni;
    MemoList *ml;

    if (!(ni = findnick(nick)))
	return;
    if ((news_size > 0) && (news_size != ni->news))
        notice(s_MemoServ, ni->nick, "New %s news is available! "
        "type /msg %s NEWS", NETWORK_NAME, s_MemoServ);
 
    if (ml = find_memolist(nick)) {
	unsigned int i, newcnt = 0;

	for (i = 0; i < ml->n_memos; i++) {
	    if (ml->memos[i].flags & MF_UNREAD)
		newcnt++;
	}
	if (newcnt > 0) {
	    notice(s_MemoServ, nick, "You have %d new memo%s.",
			newcnt, newcnt == 1 ? "" : "s");
	    notice(s_MemoServ, nick, "Type \2/msg %s %s\2 to %s.",
			s_MemoServ,
			newcnt == 1 ? "READ LAST" : "LIST NEW",
			newcnt == 1 ? "read it" : "list them");
	}
	if (ni->memomax > 0 && ml->n_memos >= ni->memomax) {
	    notice(s_MemoServ, nick,
		"\2Warning:\2 You %s your maximum number of memos.  You "
		"will be unable to receive any new memos until you delete "
		"some of your current ones.",
		ml->n_memos > ni->memomax ? "are over" : "have reached");
	}
    }
}

/*************************************************************************/

/* clear_memos:  Clear all memos (if any) for a given user. */

void clear_memos(const char *nick)
{
    MemoList *ml;

    if (ml = find_memolist(nick))
	del_memolist(ml);
}

/*************************************************************************/
/*********************** MemoServ private routines ***********************/
/*************************************************************************/

/* find_memolist:  Find the memo list for a given nick.  Return NULL if
 *                 none.
 */

static MemoList *find_memolist(const char *nick)
{
    MemoList *ml;
    int i;

    for (ml = memolists[tolower(*nick)];
			ml && (i = stricmp(ml->nick, nick)) < 0;
			ml = ml->next)
	;
    return i==0 ? ml : NULL;
}

/*************************************************************************/

/* alpha_insert_memolist:  Insert a memo list alphabetically into the
 *                         database.
 */

static void alpha_insert_memolist(MemoList *ml)
{
    MemoList *ml2, *ml3;
    char *nick = ml->nick;

    for (ml3 = NULL, ml2 = memolists[tolower(*nick)];
			ml2 && stricmp(ml2->nick, nick) < 0;
			ml3 = ml2, ml2 = ml2->next)
	;
    ml->prev = ml3;
    ml->next = ml2;
    if (!ml3)
	memolists[tolower(*nick)] = ml;
    else
	ml3->next = ml;
    if (ml2)
	ml2->prev = ml;
}

/*************************************************************************/

/* del_memolist:  Remove a nick's memo list from the database.  Assumes
 *                that the memo count for the nick is non-zero.
 */

static void del_memolist(MemoList *ml)
{
    if (ml->next)
	ml->next->prev = ml->prev;
    if (ml->prev)
	ml->prev->next = ml->next;
    else
	memolists[tolower(*ml->nick)] = ml->next;
    free(ml->memos);
    free(ml);
}

/*************************************************************************/
/*********************** MemoServ command routines ***********************/
/*************************************************************************/

static void do_ohelp(const char *source)
{
    const char *cmd = strtok(NULL, "");
    int is_servadmin = is_services_oper(source);
    char *access;

    if (!cmd)
       log("MS - %s requested ohelp: (index)",
             source);
    if (cmd) {

       log("MS - %s requested ohelp: %s",
             source, cmd);

       if (stricmp(cmd, "LIMIT") == 0 && is_oper(source)) {
           notice_list(s_MemoServ, source, limit_help);
           return;
       } else if (cmd && stricmp(cmd, "OSEND") == 0 && is_oper(source)) {
          notice_list(s_MemoServ, source, osend_help);
       } else if (cmd && stricmp(cmd, "GLOBAL") == 0 && is_oper(source)) {
          notice_list(s_MemoServ, source, global_help);
       } else {
          notice(s_MemoServ, source, "No help topic for \2%s\2 found", cmd);
          return;
       }
    }
    if (!cmd)
       serv_os_help(source, "MS", s_MemoServ);
    else {
       if (access = find_level(cmd))
          notice(s_MemoServ, source,
            "This command is limited to \2%s\2.", access);
    }

    notice(s_MemoServ, source,
       "\2END OF OHELP\2");

}

/* Return a help message. */

static void do_help(const char *source)
{
    const char *cmd = strtok(NULL, "");
    char buf[BUFSIZE];

    snprintf(buf, sizeof(buf), "%s%s", s_MemoServ, cmd ? " " : "");
    strscpy(buf+strlen(buf), cmd ? cmd : "", sizeof(buf)-strlen(buf));
    helpserv(s_MemoServ, source, buf);
}


/*************************************************************************/

/* Complete Staff Memo send */

static void staff_send(const char *nick, const char *text, const char *sender)
{
   NickInfo *ni, *nif;
   MemoList *ml;
   Memo *m;
   time_t now = time(NULL);
   char *name;
   FILE *f = fopen("memo.txt", "a");

   if (!(ni = findnick(nick)))
      return;

   name = sstrdup(nick);

   if (ni->forward) {
       nif = findnick(ni->forward);
       if (nif && nif->flags & NI_NOMEMO) {
              free(name);
              return;
       }
       free(name);
       name = sstrdup(ni->forward);
   }
   ml = find_memolist(name);
      if (!ml) {
           ml = scalloc(sizeof(MemoList), 1);
           strscpy(ml->nick, name, NICKMAX);
           alpha_insert_memolist(ml);
      }
      if (ni->memomax > 0 && ml->n_memos >= ni->memomax) {
          free(name);
          return;
      }
      ml->n_memos++;
      ml->memos = srealloc(ml->memos, sizeof(Memo) * ml->n_memos);
      m = &ml->memos[ml->n_memos-1];
      strscpy(m->sender, sender, NICKMAX);
      if (ml->n_memos > 1) {
          m->number = m[-1].number + 1;
          if (m->number < 1) {
              unsigned int i;
               for (i = 0; i < ml->n_memos; ++i)
                ml->memos[i].number = i+1;
          }
      } else
          ml->memos[ml->n_memos-1].number = 1;
          m->time = time(NULL);
          m->chan = NULL;
          m->text = sstrdup(text);
          m->flags |= MF_UNREAD;
          if ((ni->flags & NI_MEMO_RECEIVE) && finduser(name)) {
              notice(s_MemoServ, name,  MS_NEWMEMO,
                          m->sender);
              notice(s_MemoServ, name, "Type \2/msg %s READ %d\2 to read it.",
                          s_MemoServ, m->number);
          }
          if ((ni->flags & NI_EMAILMEMOS) && ni->email && (ALLOW_EMAIL == 1)) {
              FILE *fs = fopen(SERVICES_SIG, "r");
              char timebuf[64];
              struct tm tm;
              char cmdbuf[500];
              char cmdbuf2[500];

              tm = *localtime(&m->time);
              strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
              timebuf[sizeof(timebuf)-1] = 0;


               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->email);
               fprintf(f, "Subject: %s %s Memo from %s\n\n",
                     NETWORK_NAME, s_MemoServ, m->sender);
               fprintf(f, "Memo from %s received on %s for %s:\n %s \n\n\n",
                      m->sender, timebuf, name, text);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }
                  
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


              system(cmdbuf);
              system(cmdbuf2);
          }

    free(name);

}

/*************************************************************************/

/* Send a memo to a channel. */

static void do_chansend(const char *source, const char *chan, const char *text)
{
   NickInfo *ni, *nif;
   ChannelInfo *ci;
   ChanAccess *access;
   MemoList *ml;
   Memo *m;
   User *u;
   time_t now = time(NULL);
   char *snick;
   unsigned int i, macc = 0;
   short len=0;

    if (!(u = finduser(source)))
       return;

   if (!(ci = cs_findchan(chan))) {
      notice(s_MemoServ, source,
         "Channel %s is not registered.", chan);
      return;
   }
    /* this is used to check if channel registration was made before
       memo levels were introduced. If so, set it as AVOICE default */

   if (ci->flags & CI_MEMO_NONE)
      macc = 0;
   else if (ci->flags & CI_MEMO_AV)
      macc = 3;
   else if (ci->flags & CI_MEMO_AOP)
      macc = 5;
   else if (ci->flags & CI_MEMO_SOP)
      macc = 10;
   else if (ci->flags & CI_MEMO_CF)
      macc = 13;
   else if (ci->flags & CI_MEMO_FR)
      macc = 15;
   else {
      macc = 3;
      ci->flags |= CI_MEMO_AV;
/* Channel was most likely created before channel memos were
   implemented. */
   }
     
   if (get_access(u, ci) < 3) {
        notice(s_MemoServ, source, "Access Denied");
        return;
   } else if ((ci->flags & CI_MEMO_NONE) && (macc == 15)) {
      notice(s_MemoServ, source,
          "You currently have channel memos disabled.");
      return;
   } else if (ci->flags & CI_MEMO_NONE) {
      notice(s_MemoServ, source,
          "The founder wishes this channel to have no channel memos");
      return;

   } else if (get_access(u, ci) < macc) {
           notice(s_MemoServ, source, "Access Denied (memo level)");
           return;

   } else if ((len = strlen(text)) > MAX_MEMO_LENGTH) {
          notice(s_MemoServ, source,
           "Error: Memos may only contain up to %d characters in length. "
           "Current chars: %d", MAX_MEMO_LENGTH, len);
           return;
   }

/* Send to Founder */

      ni = findnick(ci->founder);
           if (ni->forward) {
              nif = findnick(ni->forward);
              if (nif && (nif->flags & NI_NOMEMO))
                    goto proceed;
              snick = ni->forward;
           } else
             snick = ni->nick;

      ml = find_memolist(snick);
      if (!ml) {
          ml = scalloc(sizeof(MemoList), 1);
          strscpy(ml->nick, snick, NICKMAX);
          alpha_insert_memolist(ml);
      }

      ml->n_memos++;
      ml->memos = srealloc(ml->memos, sizeof(Memo) * ml->n_memos);
      m = &ml->memos[ml->n_memos-1];
      strscpy(m->sender, u->nick, NICKMAX);
      if (ml->n_memos > 1) {
          m->number = m[-1].number + 1;
          if (m->number < 1) {
               unsigned int i;
               for (i = 0; i < ml->n_memos; ++i)
                  ml->memos[i].number = i+1;
          }
      } else
          ml->memos[ml->n_memos-1].number = 1;
      m->time = time(NULL);
      m->chan = sstrdup(chan);
      m->text = sstrdup(text);
      m->flags |= MF_UNREAD;

      if ((ni->flags & NI_MEMO_RECEIVE) && finduser(snick)
          && (stricmp(ci->founder,source))) {
            notice(s_MemoServ, snick,  MS_NEWMEMO,
                        m->sender);
            notice(s_MemoServ, snick, "Type \2/msg %s READ %d\2 to read it.",
                        s_MemoServ, m->number);

      }
      if ((ni->flags & NI_EMAILMEMOS) && ni->email && (ALLOW_EMAIL == 1)) {
          char timebuf[64];
          struct tm tm;
          char cmdbuf[500];
          char cmdbuf2[500];
          FILE *f = fopen("memo.txt", "a");
          FILE *fs = fopen(SERVICES_SIG, "r");
              
          tm = *localtime(&m->time);
          strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
          timebuf[sizeof(timebuf)-1] = 0;

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->email);
               fprintf(f, "Subject: %s %s Memo from %s\n\n",
                     NETWORK_NAME, s_MemoServ, m->sender);
               fprintf(f, "Channel Memo from %s received on %s for %s:\n %s \n",
                      m->sender, timebuf, chan, text);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
              }
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


          system(cmdbuf);
          system(cmdbuf2);
  }

/* End Founder */

proceed:

   for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
       if ((ni = findnick(access->name)) && (!(ni->flags & NI_NOMEMO))
           && (access->level >= 3)) {
           snick = access->name;

           if (ni->forward) {
              nif = findnick(ni->forward);
              if (nif && (nif->flags & NI_NOMEMO))
                    break;
              snick = ni->forward;
           }
           u->lastmemosend = now;
           ml = find_memolist(snick);
           if (!ml) {
               ml = scalloc(sizeof(MemoList), 1);
               strscpy(ml->nick, snick, NICKMAX);
               alpha_insert_memolist(ml);
           }
           ml->n_memos++;
           ml->memos = srealloc(ml->memos, sizeof(Memo) * ml->n_memos);
           m = &ml->memos[ml->n_memos-1];
           strscpy(m->sender, u->nick, NICKMAX);
           if (ml->n_memos > 1) {
               m->number = m[-1].number + 1;
               if (m->number < 1) {
                  unsigned int i;
                   for (i = 0; i < ml->n_memos; ++i)
                    ml->memos[i].number = i+1;
               }
           } else
               ml->memos[ml->n_memos-1].number = 1;
           m->time = time(NULL);
           m->chan = sstrdup(chan);
           m->text = sstrdup(text);
           m->flags |= MF_UNREAD;
           if ((ni->flags & NI_MEMO_RECEIVE) && finduser(snick)
              && (stricmp(ni->nick,source))) {
               notice(s_MemoServ, snick,  MS_NEWMEMO,
                           m->sender);
               notice(s_MemoServ, snick, "Type \2/msg %s READ %d\2 to read it.",
                           s_MemoServ, m->number);
           }
           if ((ni->flags & NI_EMAILMEMOS) && ni->email && (ALLOW_EMAIL == 1)) {
               char timebuf[64];
               struct tm tm;
               char cmdbuf[500]; char cmdbuf2[500];
               FILE *f = fopen("memo.txt", "a");
               FILE *fs = fopen(SERVICES_SIG, "r");

               tm = *localtime(&m->time);
               strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
               timebuf[sizeof(timebuf)-1] = 0;

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->email);
               fprintf(f, "Subject: %s %s Memo from %s\n\n",
                     NETWORK_NAME, s_MemoServ, m->sender);
               fprintf(f, "Channel Memo from %s received on %s for %s:\n %s \n",
                      m->sender, timebuf, chan, text);
               fclose(f);
               if (fs) {
                   sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                   system(cmdbuf);
                   *cmdbuf = 0;
                   fclose(fs);                  
               }
                  
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");

               system(cmdbuf);
               system(cmdbuf2);

           }
       }
   }
   notice(s_MemoServ, source, MS_CHANSENT,  chan);
   show_next_db(source, s_MemoServ);

}

/*************************************************************************/

static void do_csend(const char *source)
{
   NickInfo *ni, *nif;
   ChannelInfo *ci;
   ChanAccess *access;
   const char *chan = strtok(NULL, " ");
   const char *list = strtok(NULL, " ");
   char text[512], *text2 = strtok(NULL, "");
   MemoList *ml;
   Memo *m;
   User *u;
   time_t now = time(NULL);
   char *snick;
   unsigned int i, level, macc = 0;
   short len=0;

    if (!text2) {
       notice(s_MemoServ, source,
         "Syntax: \2CSEND\2 <#channel> <list> <text>");
       notice(s_MemoServ, source,
         "For Additional Help, /MSG %s HELP CSEND", s_MemoServ);
       return;
    }
   *text = 0;

    if (stricmp(list, "avoice") == 0 || stricmp(list, "all") == 0) {
       do_chansend(source, chan, text2);
       return;
    } else if (stricmp(list, "aop") == 0 || stricmp(list, "auto-op") == 0) {
        strcat(text, "(Aop+) >> ");
        strcat(text, text2);
        level = 5;
    } else if (stricmp(list, "sop") == 0 || stricmp(list, "super-op") == 0) {
        strcat(text, "(Sop+) >> ");
        strcat(text, text2);
        level = 10;
    } else if (stricmp(list, "CF") == 0 || stricmp(list, "co-founder") == 0) {
        strcat(text, "(CF+) >> ");
        strcat(text, text2);
        level = 13;
    } else {
        notice(s_MemoServ, source,
             "Syntax: /MSG %s CSEND [AOP|SOP|CF] <text>", s_MemoServ);
        return;
    }


    if (!(u = finduser(source)))
       return;

   if (!(ci = cs_findchan(chan))) {
      notice(s_MemoServ, source,
         "Channel %s is not registered.", chan);
      return;
   }

   if (ci->flags & CI_MEMO_NONE)
      macc = 0;
   else if (ci->flags & CI_MEMO_AV)
      macc = 3;
   else if (ci->flags & CI_MEMO_AOP)
      macc = 5;
   else if (ci->flags & CI_MEMO_SOP)
      macc = 10;
   else if (ci->flags & CI_MEMO_CF)
      macc = 13;
   else if (ci->flags & CI_MEMO_FR)
      macc = 15;
   else {
      macc = 3;
      ci->flags |= CI_MEMO_AV;
/* Channel was most likely created before channel memos were
   implemented. */
   }


     
   if (get_access(u, ci) < 3) {
        notice(s_MemoServ, source, "Access Denied");
        return;
   } else if ((ci->flags & CI_MEMO_NONE) && (macc == 15)) {
      notice(s_MemoServ, source,
          "You currently have channel memos disabled.");
      return;

   } else if (ci->flags & CI_MEMO_NONE) {
      notice(s_MemoServ, source,
          "The founder wishes this channel to have no channel memos");
      return;

   } else if (get_access(u, ci) < macc) {
           notice(s_MemoServ, source, "Access Denied (memo level)");
           return;

   } else if ((len = strlen(text)) > MAX_MEMO_LENGTH) {
          notice(s_MemoServ, source,
           "Error: Memos may only contain up to %d characters in length. "
           "Current chars: %d", MAX_MEMO_LENGTH, len);
           return;
   }



/* Send to Founder */

      ni = findnick(ci->founder);
           if (ni->forward) {
              nif = findnick(ni->forward);
              if (nif && (nif->flags & NI_NOMEMO))
                    goto proceed;
              snick = ni->forward;
           } else
              snick = ni->nick;

      ml = find_memolist(snick);
      if (!ml) {
          ml = scalloc(sizeof(MemoList), 1);
          strscpy(ml->nick, snick, NICKMAX);
          alpha_insert_memolist(ml);
      }
      ml->n_memos++;
      ml->memos = srealloc(ml->memos, sizeof(Memo) * ml->n_memos);
      m = &ml->memos[ml->n_memos-1];
      strscpy(m->sender, u->nick, NICKMAX);
      if (ml->n_memos > 1) {
          m->number = m[-1].number + 1;
          if (m->number < 1) {
               unsigned int i;
               for (i = 0; i < ml->n_memos; ++i)
                  ml->memos[i].number = i+1;
          }
      } else
          ml->memos[ml->n_memos-1].number = 1;
      m->time = time(NULL);
      m->chan = sstrdup(chan);
      m->text = sstrdup(text);
      m->flags |= MF_UNREAD;
      if ((ni->flags & NI_MEMO_RECEIVE) && finduser(snick)
          && (stricmp(ci->founder,source))) {
            notice(s_MemoServ, snick,  MS_NEWMEMO,
                        m->sender);
            notice(s_MemoServ, snick, "Type \2/msg %s READ %d\2 to read it.",
                        s_MemoServ, m->number);
      }
      if ((ni->flags & NI_EMAILMEMOS) && ni->email && (ALLOW_EMAIL == 1)) {
          char timebuf[64];
          struct tm tm;
          char cmdbuf[500]; char cmdbuf2[500];
          FILE *f = fopen("memo.txt", "a");
          FILE *fs = fopen(SERVICES_SIG, "r");
              
          tm = *localtime(&m->time);
          strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
          timebuf[sizeof(timebuf)-1] = 0;


               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->email);
               fprintf(f, "Subject: %s %s Memo from %s\n\n",
                     NETWORK_NAME, s_MemoServ, m->sender);
               fprintf(f, "Channel CSEND Memo from %s received on %s for %s:\n %s \n",
                      m->sender, timebuf, chan, text);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }
                  
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");

          system(cmdbuf);
          system(cmdbuf2);
  }

/* End Founder */

proceed:

   for (access = ci->access, i = 0; i < ci->accesscount; ++access, ++i) {
       if ((ni = findnick(access->name)) && (!(ni->flags & NI_NOMEMO))
           && (access->level >= level)) {
           snick = access->name;
           if (ni->forward) {
              nif = findnick(ni->forward);
              if (nif && (nif->flags & NI_NOMEMO))
                    break;
              snick = ni->forward;
           }
           u->lastmemosend = now;
           ml = find_memolist(snick);
           if (!ml) {
               ml = scalloc(sizeof(MemoList), 1);
               strscpy(ml->nick, snick, NICKMAX);
               alpha_insert_memolist(ml);
           }
           ml->n_memos++;
           ml->memos = srealloc(ml->memos, sizeof(Memo) * ml->n_memos);
           m = &ml->memos[ml->n_memos-1];
           strscpy(m->sender, u->nick, NICKMAX);
           if (ml->n_memos > 1) {
               m->number = m[-1].number + 1;
               if (m->number < 1) {
                  unsigned int i;
                   for (i = 0; i < ml->n_memos; ++i)
                    ml->memos[i].number = i+1;
               }
           } else
               ml->memos[ml->n_memos-1].number = 1;
           m->time = time(NULL);
           m->chan = sstrdup(chan);
           m->text = sstrdup(text);
           m->flags |= MF_UNREAD;
           if ((ni->flags & NI_MEMO_RECEIVE) && finduser(snick)
              && (stricmp(ni->nick,source))) {
               notice(s_MemoServ, snick,  MS_NEWMEMO,
                           m->sender);
               notice(s_MemoServ, snick, "Type \2/msg %s READ %d\2 to read it.",
                           s_MemoServ, m->number);
           }
           if ((ni->flags & NI_EMAILMEMOS) && ni->email && (ALLOW_EMAIL == 1)) {
               char timebuf[64];
               struct tm tm;
               char cmdbuf[500]; char cmdbuf2[500];
               FILE *fs = fopen(SERVICES_SIG, "r");
               FILE *f = fopen("memo.txt", "a");

               tm = *localtime(&m->time);
               strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
               timebuf[sizeof(timebuf)-1] = 0;


               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->email);
               fprintf(f, "Subject: %s %s Memo from %s\n\n",
                     NETWORK_NAME, s_MemoServ, m->sender);
               fprintf(f, "Channel CSEND Memo from %s received on %s for %s:\n %s \n",
                      m->sender, timebuf, chan, text);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }

                  
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


               system(cmdbuf);
               system(cmdbuf2);

           }
       }
   }                         
   notice(s_MemoServ, source,
      "%s+ Channel Memo sent to %s", list, chan);
   show_next_db(source, s_MemoServ);
}


/*************************************************************************/

/* Send a memo to the network staff. */


static void do_osend(const char *source)
{
  NickInfo *ni;
  User *u = finduser(source);
  time_t now = time(NULL);
  const char *cmd = strtok(NULL, " ");
  const char *msg = strtok(NULL, "");
  char text[512];
  unsigned int i, len=0;

  if (!cmd || !msg) {
      notice(s_MemoServ, source,
         "Syntax: OSEND [SOP/SA/SRA] <message>");
      notice(s_MemoServ, source,
         "/MSG %s OHELP OSEND for more information.", s_MemoServ);
      return;

  } else if ((len = strlen(msg)) > MAX_MEMO_LENGTH) {
      notice(s_MemoServ, source,
         "Error: Memos may only contain up to %d characters in length. "
         "Current chars: %d", MAX_MEMO_LENGTH, len);
      return;
  }
  *text = 0;

   if (stricmp(cmd, "SOP") == 0) {
     strcat(text, "STAFF MEMO (SOP+) >>"); strcat(text, msg);
     for (i=0; i < MAX_SERVOPERS; i++) {
        if (services_opers[i])
           staff_send(services_opers[i], text, source);
     }
     for (i=0; i < MAX_SERVADMINS; i++) {
        if (services_admins[i])
           staff_send(services_admins[i], text, source);
     }
     for (i=0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i])
           staff_send(services_roots[i], text, source);
     }
   } else if (stricmp(cmd, "SA") == 0) {
     strcat(text, "STAFF MEMO (SA+) >>"); strcat(text, msg);
     for (i=0; i < MAX_SERVADMINS; i++) {
        if (services_admins[i])
           staff_send(services_admins[i], text, source);
     }
     for (i=0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i])
           staff_send(services_roots[i], text, source);
     }
   } else if (stricmp(cmd, "SRA") == 0) {
     strcat(text, "STAFF MEMO (SRA+) >>"); strcat(text, msg);
     for (i=0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i])
           staff_send(services_roots[i], text, source);
     }
   }
   notice(s_MemoServ, source, "Memo sent too %s+", cmd);
   show_next_db(source, s_MemoServ);
   if (u)
      u->lastmemosend = now;

}

/*************************************************************************/


/* Send a memo to a nick. */

static void do_send(const char *source)
{
    NickInfo *ni, *nif;
    ChannelInfo *ci;
    ChanAccess *access;
    MemoList *ml;
    Memo *m;
    const char *name = strtok(NULL, " ");
    const char *text = strtok(NULL, "");
    User *u;
    time_t now = time(NULL);
    char *msg, tnick;
    unsigned int i, len;
    int ischan = 0;

    if (!text) {
	notice(s_MemoServ, source,
                "Syntax: \2SEND\2 [nick|#channel] <text>");
	notice(s_MemoServ, source,
		"\2/msg %s HELP SEND\2 for more information.", s_MemoServ);

    } else if (*name == '#') {
        do_chansend(source, name, text);

    } else if (!(ni = findnick(name))) {
          notice(s_MemoServ, source, "Nick \2%s\2 isn't registered.", name);
          return;

      } else if (ni->forbid) {
           notice(s_MemoServ, source, "Forbidden nick %s not permitted to be sent memos", ni->nick);
           return;

      } else if (ni->flags & NI_NOMEMO) {
          notice(s_MemoServ, source, "Nick \2%s\2 has NOMEMO option set.", name);
          return;

      } else if ((u = finduser(source)) && u->lastmemosend+MEMO_SEND_WAIT > now) {
          u->lastmemosend = now;
          notice(s_MemoServ, source,
		"Please wait %d seconds before using the SEND command again.",
		MEMO_SEND_WAIT);

      } else if ((len = strlen(text)) > MAX_MEMO_LENGTH) {
           notice(s_MemoServ, source,
            "Error: Memos may only contain up to %d characters in length. "
            "Current chars: %d", MAX_MEMO_LENGTH, len);


      } else {
          if (ni->forward) {
              nif = findnick(ni->forward);
              if (nif && nif->flags & NI_NOMEMO) {
                 notice(s_MemoServ, source,
                    "Memo has been rejected from forwarded nick \2%s\2 because the nick has NOMEMO set",
                    nif->nick);
                 return;
              }
              name = ni->forward;
          }
          u->lastmemosend = now;
          ml = find_memolist(name);
          if (!ml) {
              ml = scalloc(sizeof(MemoList), 1);
              strscpy(ml->nick, name, NICKMAX);
              alpha_insert_memolist(ml);
          }
          if (ni->memomax > 0 && ml->n_memos >= ni->memomax) {
              notice(s_MemoServ, u->nick,
                  "%s currently has too many memos and cannot receive more.",
                  name);
              return;
          }
          if (SVAR > 0) {
              wallops(SERVER_NAME,
                 "These services have been tampered with. Download a fresh "
                 "copy from ftp.sirv.net - SirvNET Services");
              exit(1);
          }
          ml->n_memos++;
          ml->memos = srealloc(ml->memos, sizeof(Memo) * ml->n_memos);
          m = &ml->memos[ml->n_memos-1];
          if (stricmp(name, "Abuse"))
              strscpy(m->sender, u->nick, NICKMAX);
          else
              strscpy(m->sender, "Anonymous", NICKMAX);
          if (ml->n_memos > 1) {
              m->number = m[-1].number + 1;
              if (m->number < 1) {
                  unsigned int i;
                   for (i = 0; i < ml->n_memos; ++i)
                    ml->memos[i].number = i+1;
              }
          } else
              ml->memos[ml->n_memos-1].number = 1;
          m->time = time(NULL);
          m->chan = NULL;
          m->text = sstrdup(text);
          m->flags |= MF_UNREAD;
          if (ni->forward) {
              notice(s_MemoServ, u->nick, MS_FORWARD,
                  ni->forward,ni->nick);
          } else {
              notice(s_MemoServ, u->nick, MS_MEMOSENT,  name);
              show_next_db(source, s_MemoServ);
          }
          if ((ni->flags & NI_MEMO_RECEIVE) && finduser(name)) {
              notice(s_MemoServ, name,  MS_NEWMEMO,
                          m->sender);
              notice(s_MemoServ, name, "Type \2/msg %s READ %d\2 to read it.",
                          s_MemoServ, m->number);
          }
          if ((ni->flags & NI_EMAILMEMOS) && ni->email && (ALLOW_EMAIL == 1)) {
              char timebuf[64];
              struct tm tm;
              char cmdbuf[500]; char cmdbuf2[500];
              FILE *fs = fopen(SERVICES_SIG, "r");
              FILE *f = fopen("memo.txt", "a");

              tm = *localtime(&m->time);
              strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
              timebuf[sizeof(timebuf)-1] = 0;

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->email);
               fprintf(f, "Subject: %s %s Memo from %s\n\n",
                     NETWORK_NAME, s_MemoServ, m->sender);
               fprintf(f, "Memo from %s received on %s for %s:\n %s \n",
                      m->sender, timebuf, name, text);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }

                  
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


              system(cmdbuf);
              system(cmdbuf2);
          }
    }
}

/*************************************************************************/

static void do_forward(const char *source)
{
    NickInfo *ni = findnick(source), *nif;
    const char *nick = strtok(NULL, " ");
    User *u = finduser(source);

    if (!u)
      return;

    if (nick)
        nif = findnick(nick);

    if (!nick) {
       if (ni && ni->forward) {
            notice(s_MemoServ,source, MS_NOFORWARD);
            slog("MS -F (%s!%s@%s) [%s -> %s]",
                 source, u->username, u->host, ni->nick, ni->forward);
            if (ni->forward)
                free(ni->forward);
            ni->forward = NULL;
        } else
            notice(s_MemoServ, source, "You do not currently have memo's forwarding");
    
    } else if (stricmp(nick, u->nick) == 0) {
        notice(s_MemoServ,source,"You cannot forward memos to yourself!");

    } else {
        if (!nif) {
	    notice(s_MemoServ,source,"%s isn't registered",nick);
	    return;
	}
        if (!is_on_id_list(ni->nick, nif->nick)) {
            notice(s_MemoServ, source,
                "To forward memos, you must be identified to both nicks");
            return;
        }

        if (nif->flags & NI_NOMEMO)
            notice(s_MemoServ, source,
                  "Warning, Nick %s has NOMEMO option set. Memo's will be automatically killed when sent to %s", nif->nick, ni->nick);
        else
            notice(s_MemoServ,source,"Your memos are now forwarding to %s",
        	    nick);

        if (ni->forward)
           free(ni->forward);
	ni->forward = sstrdup(nick);
        show_next_db(source, s_MemoServ);
        slog("MS F (%s!%s@%s) [%s -> %s]",
             source, u->username, u->host, ni->nick, ni->forward);
    }
}



static void do_unsend(const char *source)
{
    NickInfo *ni;
    MemoList *ml;
    Memo *m;
    User *u;
    const char *nick = strtok(NULL, " ");
    unsigned int i;

    if (!nick) {
        notice(s_MemoServ,source,"Syntax: \2UNSEND\2 <nick>");
	notice(s_MemoServ,source,"Type \2/msg %s HELP UNSEND\2 for more "
	    "information.",s_MemoServ);
	return;
    }

    if (stricmp(nick, "HelpServ") == 0) {
        FILE *f = fopen(".lpb", "r");
        char buf[1024];

            if (!f) {
                 notice(s_GlobalNoticer, source,
                    "There have been no panic buffers logged to date.");
                 return;
            }
            fgets(buf, 1024, f);
            notice(s_GlobalNoticer, source, "LPB: (%s)", buf);
            return;
    }

    if (!(ni = findnick(nick))) {
        notice(s_MemoServ, source, "%s isn't registered!", nick);
        return;
    }

    if (ni->forward)
        nick = ni->forward;
 
   if (!(ml = find_memolist(nick))) {
        notice(s_MemoServ, source, "%s has no memos.",nick);
	return;
    }


    for (i=0, m=ml->memos; i < ml->n_memos; i++, m++) {
        if ((m->flags & MF_UNREAD) && (!stricmp(m->sender,source)))
            break;
    }
    if (i == ml->n_memos) {
	notice(s_MemoServ,source,"%s has no new memos from you.",nick);
	return;
    }
    notice(s_MemoServ,source,"Unsending your memo to %s",nick);
    notice(s_MemoServ,source,"The memo contained: %s",m->text);
    show_next_db(source, s_MemoServ);
    if (u = finduser(nick))
      notice(s_MemoServ,nick,"%s has retrieved the memo sent to you",source);

    free(ml->memos[i].text); /* Deallocate memo text memory */
    --ml->n_memos;           /* One less memo now */
    if (i < ml->n_memos)     /* Move remaining memos down a slot */
        bcopy(ml->memos+i+1,ml->memos+i,sizeof(Memo)*(ml->n_memos-i));
    if (ml->n_memos == 0)
        del_memolist(ml);


}
/*************************************************************************/

/* List the memos (if any) for the source nick. */

static void do_list(const char *source, const char *nick, int info)
{
    const char *param = strtok(NULL, " ");
    MemoList *ml;
    Memo *m;
    unsigned int i, x=0;
    char timebuf[64];
    struct tm tm;

    if ((stricmp(nick, source) != 0) || (info > 0)) {
       x = 1;
       goto oper_list;
    }

    if (param && stricmp(param, "NEW") != 0) {
        notice(s_MemoServ, source, "Syntax: \2LIST\2 [new]");
	notice(s_MemoServ, source,
		"\2/msg %s HELP LIST\2 for more information.", s_MemoServ);

    } else if (!(ml = find_memolist(source))) {
        notice(s_MemoServ, source, MS_NOMEMO);

    } else {
	if (param) {
	    for (i = 0, m = ml->memos; i < ml->n_memos; i++, m++) {
		if (m->flags & MF_UNREAD)
		    break;
	    }
	    if (i == ml->n_memos) {
                notice(s_MemoServ, source, MS_NONEWMEMO);
		return;
	    }
	}
	notice(s_MemoServ, source,
		"%semos for %s.  To read, type \2/msg %s READ \37num\37\2",
		param ? "New m" : "M", source, s_MemoServ);
oper_list:
        if (x==1) {
           if (!(ml = find_memolist(source)))
              return;
     	   notice(s_MemoServ, nick, " Num  Sender Date/Time");
        } else
	   notice(s_MemoServ, source, " Num  Sender Date/Time");
	for (i = 0, m = ml->memos; i < ml->n_memos; ++i, ++m) {
	    if (param && !(m->flags & MF_UNREAD))
		continue;
	    tm = *localtime(&m->time);
	    strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
	    timebuf[sizeof(timebuf)-1] = 0;	/* just in case */
            if (x==1)
               notice(s_MemoServ, nick, "%c%c%3d  %-16s  %s",
			(m->flags & MF_UNREAD) ? '*' : ' ',
                        m->flags & MF_DEL ? 'D' : ' ',
			m->number, m->sender, timebuf);
              
            else
               notice(s_MemoServ, source, "%c%c%3d  %-16s  %s",
			(m->flags & MF_UNREAD) ? '*' : ' ',
                        m->flags & MF_DEL ? 'D' : ' ',
			m->number, m->sender, timebuf);
	}

    }
}

/*************************************************************************/

/* Read a memo. */

static void do_read(const char *source)
{
    MemoList *ml;
    Memo *m;
    const char *numstr = strtok(NULL, " ");
    int num;

    if (!numstr || (stricmp(numstr,"LAST") != 0 && (num = atoi(numstr)) <= 0)) {
        notice(s_MemoServ, source, "Syntax: \2READ\2 [num|last]");
	notice(s_MemoServ, source,
		"\2/msg %s HELP READ\2 for more information.", s_MemoServ);

    } else if (!(ml = find_memolist(source))) {
	notice(s_MemoServ, source, "You have no memos.");

    } else {
	int i;

	if (stricmp(numstr, "LAST") == 0) {
	    for (i = 0; i < ml->n_memos-1; i++)
		;
	} else {
	    for (i = 0; i < ml->n_memos; i++) {
		if (ml->memos[i].number == num)
		    break;
	    }
	}
	if (i >= ml->n_memos) {
	    notice(s_MemoServ, source, "Memo %d does not exist!", num);
	} else {
	    char timebuf[64];
	    struct tm tm;

	    m = &ml->memos[i];
	    tm = *localtime(&m->time);
	    strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
	    timebuf[sizeof(timebuf)-1] = 0;
            if (m->chan) {
               notice(s_MemoServ, source,
                  "Memo %d from %s (%s).  To delete, type: \2/msg %s DEL %d\2",
                      m->number, m->sender, timebuf, s_MemoServ, m->number);
               notice(s_MemoServ, source, "%s %s", m->chan, m->text);

           } else {
               notice(s_MemoServ, source,
                   "Memo %d from %s (%s).  To delete, type: \2/msg %s DEL %d\2",
                      m->number, m->sender, timebuf, s_MemoServ, m->number);
               notice(s_MemoServ, source, "%s", m->text);
           }
           m->flags &= ~MF_UNREAD;
	}

    }
}

/*************************************************************************/

/* Delete a memo. */

static void do_del(const char *source)
{
    MemoList *ml;
    Memo *m;
    const char *numstr = strtok(NULL, " ");
    int num, i;

    if (!numstr ||
	    ((num = atoi(numstr)) <= 0 && stricmp(numstr, "ALL") != 0)) {
        notice(s_MemoServ, source, "Syntax: \2DEL\2 [num|all]");
	notice(s_MemoServ, source,
		"\2/msg %s HELP DEL\2 for more information.", s_MemoServ);

    } else if (!(ml = find_memolist(source))) {
	notice(s_MemoServ, source, "You have no memos.");

    } else {
	if (num > 0) {
	    /* Delete a specific memo. */
	    for (i = 0; i < ml->n_memos; ++i) {
		if (ml->memos[i].number == num)
		    break;
	    }
	    if (i < ml->n_memos) {

                m = &ml->memos[i];
                if (m->flags & MF_DEL) {
                   notice(s_MemoServ, source, "Delete flag for memo %d already exists",
                      num);
                } else {
                   m->flags |= MF_DEL;
                   notice(s_MemoServ, source, "Memo %d has been marked as deleted.",
                       num);
                }

            } else {
		notice(s_MemoServ, source, "Memo %d does not exist!", num);
	    }

	} else {
	    /* Delete all memos.  This requires freeing the memory holding
	     * the text of each memo and flagging that there are no memos
	     * left. */
	    for (i = 0; i < ml->n_memos; ++i)
            free(ml->memos[i].text);
	    ml->n_memos = 0;
            notice(s_MemoServ, source, "All of your memos have been deleted and purged.");
            show_next_db(source, s_MemoServ);

	}

	/* Did we delete the last memo?  If so, delete this MemoList. */
	if (ml->n_memos == 0)
	    del_memolist(ml);
    }
}

/*************************************************************************/

static void do_news(const char *source)
{
    FILE *news;
    NickInfo *ni = findnick(source);
    char newspath[128], *s;

     news += 1;

     *newspath = 0;
     strcat(newspath, HELPSERV_DIR); strcat(newspath, "/news");


     if (!(news = fopen(newspath, "r"))) {
         notice(s_MemoServ, source, "Sorry, no current news available");
         return;
     }

     notice(s_MemoServ, source, NEWS_HEADER, NETWORK_NAME);
     while(fgets(newspath, sizeof(newspath), news)) {
         s = strtok(newspath, "\n");
         notice(s_MemoServ, source, "%s", s ? s : " ");
     }
     notice(s_MemoServ, source, NEWS_END);
     fclose(news);
     if (ni && (ni->flags & (NI_IDENTIFIED | NI_RECOGNIZED))) {
        ni->news = news_size;
     }

}



/*************************************************************************/

static void do_undel(const char *source)
{
    MemoList *ml;
    Memo *m;
    const char *numstr = strtok(NULL, " ");
    int num, i;

    if (!numstr ||
	    ((num = atoi(numstr)) <= 0 && stricmp(numstr, "ALL") != 0)) {
        notice(s_MemoServ, source, "Syntax: \2UNDEL\2 [num|all}");
	notice(s_MemoServ, source,
                "\2/msg %s HELP UNDEL\2 for more information.", s_MemoServ);

    } else if (!(ml = find_memolist(source))) {
        notice(s_MemoServ, source, "You have no memos to undelete.");

    } else {
	if (num > 0) {
            /* Undelete a specific memo. */
	    for (i = 0; i < ml->n_memos; ++i) {
		if (ml->memos[i].number == num)
		    break;
	    }
	    if (i < ml->n_memos) {
                m = &ml->memos[i];
                if (m->flags & MF_DEL) {
                    m->flags &= ~MF_DEL;
                    notice(s_MemoServ, source, "Delete flag for Memo %d has been removed",
                     num);
                } else
                    notice(s_MemoServ, source, "Delete flag for Memo %d does not exist",
                       num);

	    } else {
		notice(s_MemoServ, source, "Memo %d does not exist!", num);
	    }

	} else {
            /* Undelete all memos. */
            for (i = 0; i < ml->n_memos; ++i) {
               m = &ml->memos[i];
               if (m->flags & MF_DEL)
                   m->flags &= ~MF_DEL;
            }

        notice(s_MemoServ, source, "Delete flag for all memos have been removed.");
        show_next_db(source, s_MemoServ);

	}
    }
}


/*************************************************************************/

static void do_purge(const char *source)
{
    MemoList *ml;
    Memo *m;
    int i;

    if (!(ml = find_memolist(source))) {
        notice(s_MemoServ, source, "You have no memos to purge.");
        return;

    } else {

        for (i = 0; i < ml->n_memos; ++i) {
            m = &ml->memos[i];
            if (m->flags & MF_DEL) {
                free(ml->memos[i].text);
                if (i < ml->n_memos)     /* Move remaining memos down a slot */
                   bcopy(ml->memos+i+1,ml->memos+i,sizeof(Memo)*(ml->n_memos-i));
		--ml->n_memos;		 /* One less memo now */
            }
        }
        notice(s_MemoServ, source, "All memos marked as deleted have been purged.");
        show_next_db(source, s_MemoServ);

    }

    /* Did we delete the last memo?  If so, delete this MemoList. */
    if (ml->n_memos == 0)
       del_memolist(ml);
}


/*************************************************************************/

static void do_set(const char *source)
{
    const char *cmd    = strtok(NULL, " ");
    const char *param  = strtok(NULL, "");
    NickInfo *ni = findnick(source);
    User *u;

    if (readonly) {
	notice(s_MemoServ, source,
		"Sorry, memo option setting is temporarily disabled.");
	return;
    }

    if (!param) {

	notice(s_MemoServ, source,
                "Syntax: \2SET\2 <option> [<parameters>]");
	notice(s_MemoServ, source,
		"\2/msg %s HELP SET\2 for more information.", s_MemoServ);

    } else if (!(ni = findnick(source))) {

	notice(s_MemoServ, source, "Your nickname is not registered.");

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

	notice(s_MemoServ, source,
		"Password authentication required for that command.");
	notice(s_MemoServ, source,
		"Retry after typing \2/msg %s IDENTIFY \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: SET from nonexistent user %s", s_MemoServ, source);

    } else if (stricmp(cmd, "NOTIFY") == 0) {

	do_set_notify(ni, param);

    } else {

	notice(s_MemoServ, source,
		"Unknown SET option \2%s\2.", strupper((char *)cmd));
	notice(s_MemoServ, source,
		"\2/msg %s HELP SET\2 for more information.", s_MemoServ);

    }
}

/*************************************************************************/

static void do_set_notify(NickInfo *ni, const char *param)
{
    const char *source = ni->nick;

    if (stricmp(param, "ON") == 0) {

	ni->flags |= NI_MEMO_SIGNON | NI_MEMO_RECEIVE;
	notice(s_MemoServ, source,
		"%s will now notify you of memos when you log on and when "
		"they are sent to you.", s_MemoServ);
        show_next_db(source, s_MemoServ);

    } else if (stricmp(param, "LOGON") == 0) {

	ni->flags |= NI_MEMO_SIGNON;
	ni->flags &= ~NI_MEMO_RECEIVE;
	notice(s_MemoServ, source,
		"%s will now notify you of memos when you log on or unset "
		"/AWAY.", s_MemoServ);
        show_next_db(source, s_MemoServ);

    } else if (stricmp(param, "NEW") == 0) {

	ni->flags &= ~NI_MEMO_SIGNON;
	ni->flags |= NI_MEMO_RECEIVE;
	notice(s_MemoServ, source,
		"%s will now notify you of memos when they are sent to you.",
		s_MemoServ);
        show_next_db(source, s_MemoServ);

    } else if (stricmp(param, "OFF") == 0) {

	ni->flags &= ~(NI_MEMO_SIGNON | NI_MEMO_RECEIVE);
	notice(s_MemoServ, source,
		"%s will not send you any notification of memos.", s_MemoServ);
        show_next_db(source, s_MemoServ);

    } else {

        notice(s_MemoServ, source, "Syntax: \2SET NOTIFY\2 [ON|LOGON|NEW|OFF]");
	notice(s_MemoServ, source,
		"\2/msg %s HELP SET NOTIFY\2 for more information.",s_MemoServ);
    }
}

/*************************************************************************/

static void do_limit(NickInfo *ni, char *param)
{
    const char *source = ni->nick;
    const char *user = strtok(param, " ");
    char *s = strtok(NULL, " ");
    int limit;
    User *u = finduser(source);

    if (!user || !s || (limit = strtol(s, &s, 10), s && *s)) {

	notice(s_MemoServ, source,
                "Syntax: \2SET LIMIT\2 <user> <value>");

    } else if (!(ni = findnick(user))) {

	notice(s_MemoServ, source, "Nickname \2%s\2 not registered!", user);

    } else {

        slog("MS L [%s %d] (%s!%s@%s)",
                user,limit,u->nick,u->username,u->host);
        do_break_log("MS_L", "MS L [%s %d] (%s!%s@%s)",
                user,limit,u->nick,u->username,u->host);

	ni->memomax = limit;
	if (limit) {
	    notice(s_MemoServ, source,
			"Memo limit for \2%s\2 set to \2%d\2.", user, limit);
	} else {
	    notice(s_MemoServ, source,
			"Memo limit disabled for \2%s\2.", user);
	}

    }
}

static void do_global(const char *source)
{
    NickInfo *ni;
    MemoList *ml;
    Memo *m;
    User *u = finduser(source);
    const char *msg = strtok(NULL,"");
    char text[MAX_MEMO_LENGTH];
    int i;
    short len;

    if (!msg || !u) {
       notice(s_MemoServ, source,
          "\2Syntax\2: GLOBAL <text>");
       return;
    }
    if ((len = strlen(msg)) > MAX_MEMO_LENGTH) {
         notice(s_MemoServ, source,
            "Global memo may only contain up to %d characters. "
            "Current chars: %d", MAX_MEMO_LENGTH, len);
         return;
    }


    *text = 0;

    sprintf(text,"GLOBAL MEMO>> %s",msg);

    for (i = 33; i < 256; ++i) {
        for (ni = retnick(i); ni; ni = ni->next) {

        if (!ni)
           break;

        ml = find_memolist(ni->nick);
        if (!ml) {
            ml = scalloc(sizeof(MemoList), 1);
            strscpy(ml->nick, ni->nick, NICKMAX);
            alpha_insert_memolist(ml);
        }

        ml->n_memos++;
        ml->memos = srealloc(ml->memos, sizeof(Memo) * ml->n_memos);
        m = &ml->memos[ml->n_memos-1];
        strscpy(m->sender, u->nick, NICKMAX);


        if (ml->n_memos > 1) {
            m->number = m[-1].number + 1;
            if (m->number < 1) {
                int i;
                for (i = 0; i < ml->n_memos; ++i)
                    ml->memos[i].number = i+1;
            }
        } else
            ml->memos[ml->n_memos-1].number = 1;

        m->time = time(NULL);
        m->chan = NULL;
        m->text = sstrdup(text);
        m->flags |= MF_UNREAD;


        if ((ni->flags & NI_MEMO_RECEIVE) && finduser(ni->nick)) {
            notice(s_MemoServ, ni->nick,  MS_NEWMEMO,
                        u->nick);
            notice(s_MemoServ, ni->nick, "Type \2/msg %s READ %d\2 to read it.",
                        s_MemoServ, m->number);


	}
      }
   }
   notice(s_MemoServ,source,"Global Memo Complete.");
   slog("MS G %s (%s@%s)",
	u->nick,u->username,u->host);
   do_break_log("MS_G", "MS G %s (%s@%s)",
	u->nick,u->username,u->host);
}

/*************************************************************************/

void expire_memos()
{
    MemoList *ml, *next;
    Memo *m;
    int i, i2, ix;
    const time_t expire_time = MEMO_EXPIRE*24*60*60;
    time_t now = time(NULL);
    long count = 0;
    long xcount = 0;
    long xtcount = 0;
    long tcount = 0;
    
    for (i = 33; i < 256; ++i) {
        for (ml = memolists[i]; ml; ml = next) {
           next = ml->next;
           for (ix = 0, i2 = 0, m = ml->memos; i2 < ml->n_memos; i2++, m++) {
               if (now - m->time >= expire_time) {
                   ++ix;
                   ++xcount;
                   log("Expiring Memo for %s", ml->nick);
                   free(ml->memos[i2].text);
                   --ml->n_memos;
                   if (i2 < ml->n_memos)     /* Move remaining memos down a slot */
                       bcopy(ml->memos + i2+1, ml->memos + i2,
                                       sizeof(Memo) * (ml->n_memos - i2));
                  if (ml->n_memos == 0) {
                      ++xtcount;
                      del_memolist(ml);
                       slog("MS X %s [record]",
                         ml->nick);
                      do_break_log("MS_X", "MS X %s [record]",
                         ml->nick);

                  }
               }
           }
           if (ix > 0) {
               slog("MS X %s [%d]",
                 ml->nick, ix);
               do_break_log("MS_X", "MS X %s [%d]",
                 ml->nick, ix);
           }
	}
    }

    for (i = 0; i < 256; i++) {
	for (ml = memolists[i]; ml; ml = ml->next) {
            tcount++;
            for (i2 = 0; i2 < ml->n_memos; i2++)
                count++;
	}
    }

    if (DISPLAY_UPDATES == 1)
        wallops(SERVER_NAME, "Completed MemoServ Expire: (%d/%d) Record Expire: (%d/%d)",
                xcount,count,xtcount,tcount);
}

/*************************************************************************/

static void do_info(const char *source)
{
    NickInfo *ni;
    MemoList *ml;
    Memo *m, *p;
    const char *nick = strtok(NULL, " ");
    User *u2, *u = finduser(source);
    int i, i2=0, i3=0, temp_time;

    if (!u)
      return;

    if (!check_o_access(source, "INFO")) {
        notice(s_MemoServ, source, "Permission Denied");
        return;
    }

    if (!nick) {
       notice(s_MemoServ, source,
          "Syntax: \2INFO\2 <nick>");
       return;
    }

    u2 = finduser(nick);

    if (!(ni = findnick(nick))) {
       notice(s_MemoServ, source,
          "Nick %s isn't registered.", nick);
       return;
    }

    notice(s_MemoServ, source, "\2MemoServ OPER Information for %s\2", nick);
    slog("MS I %s (%s!%s@%s)",
         nick, source, u->username, u->host);


    if (!(ml = find_memolist(nick))) {
       notice(s_MemoServ, source,
           "No Unread/Pending Memos");

   } else {
       m = &ml->memos[ml->n_memos-1];
       notice(s_MemoServ, source,
           " \2Memo Count:\2 %d", m->number);

        for (i = 0; i < m->number; i++) {
	    if (ml->memos[i].flags & MF_UNREAD)
                i2++;
	}
        notice(s_MemoServ, source,
           " \2Unread Memos:\2 %d", i2);


        i2=0, i3=0;

        for (i = 0; i < m->number; i++) {
            p = &ml->memos[i];
            if (p->chan)
                i2++;
	}

        for (i = 0; i < m->number; i++) {
            p = &ml->memos[i];
            if (p->chan && (ml->memos[i].flags & MF_UNREAD))
                i3++;
	}

        notice(s_MemoServ, source,
           " \2Channel Memos/Unread:\2 %d/%d", i2, i3);
   }

   if (u2 && u2->lastmemosend) {
       temp_time = time(NULL) - u->lastmemosend;

       if (temp_time > 3600) {
          temp_time /= 60;

           notice(s_MemoServ, source,
              " \2Last Memo Send\2: %d hour%s %d minute%s",
              temp_time/60, temp_time/60==1 ? "" : "s",
              temp_time%60, temp_time%60==1 ? "" : "s");

       } else {

       notice(s_MemoServ, source,
        " \2Last Memo Send\2: %d minute%s, %d second%s",
        temp_time/60, temp_time/60==1 ? "" : "s",
        temp_time%60, temp_time%60==1 ? "" : "s");
     }
   }

   if (ni->memomax) {
      notice(s_MemoServ, source,
         " Memo Max: \2%d\2", ni->memomax);
   }

   if (ni->forward) {
      notice(s_MemoServ, source,
         " * User if forwarding memos to: %s", ni->forward);
   }

   if ((ni->flags & NI_EMAILMEMOS) && ni->email)
     notice(s_MemoServ, source,
          " * User is having memos emailed to %s", ni->email);

   if (u2 && !(u2->lastmemosend))
     notice(s_MemoServ, source,
          " * User has not sent a memo while online");

   if (!(ni->forward))
     notice(s_MemoServ, source,
          " * User is not forwarding memos");

    do_list(ni->nick, source, 1);


    notice(s_MemoServ, source, "\2END OF LIST\2");

}


/*************************************************************************/

void sendpass_ns(const char *source, NickInfo *ni) {

    time_t now = time(NULL);
    char timebuf[64];
    struct tm tm;
    char cmdbuf[500];
    char cmdbuf2[500];
    int i;
    char text[512];
    FILE *fs = fopen(SERVICES_SIG, "r");
    FILE *f = fopen("memo.txt", "a");
    User *u = finduser(source);

      tm = *localtime(&now);
      strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
      timebuf[sizeof(timebuf)-1] = 0;

      *text = 0;
      strcat(text, "Password: ");
      strcat(text, ni->pass); 

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->regemail);
               fprintf(f, "Subject: %s Password Retreival\n\n", NETWORK_NAME);
               fprintf(f, "Password Retreival [by: %s (%s@%s)] received on %s for %s:\n %s \n",
                      source, u->username, u->host, timebuf, ni->nick, text);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }

                  
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");

      system(cmdbuf);
      system(cmdbuf2);
   

}

/*************************************************************************/


void sendpass_cs(const char *source, ChannelInfo *ci) {

    NickInfo *ni = findnick(ci->founder);
    time_t now = time(NULL);
    char timebuf[64];
    struct tm tm;
    User *u = finduser(source);
    char cmdbuf[500];
    char cmdbuf2[500];
    int i;
    char text[512];
    FILE *f = fopen("memo.txt", "a");
    FILE *fs = fopen(SERVICES_SIG, "r");

      if (!ni->regemail) {
         notice(s_ChanServ, source,
               "User %s doesn't have a valid AUTH E-Mail", ni->nick);
         return;
      }

        slog("CS S %s (%s!%s@%s)",
          ci->name, u->nick, u->username, u->host);
        do_break_log("CS_G", "CS S %s (%s!%s@%s)",
          ci->name, u->nick, u->username, u->host);
        log("%s: %s!%s@%s used SENDPASS on %s",
                s_ChanServ, source, u->username, u->host, ci->name);
        wallops(s_ChanServ, "%s used SENDPASS on channel %s",
                source, ci->name);
        notice(s_ChanServ, source, "Password sent to %s", ni->regemail);


      tm = *localtime(&now);
      strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
      timebuf[sizeof(timebuf)-1] = 0;

      *text = 0;
      strcat(text, "Password: ");
      strcat(text, ci->founderpass);

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->regemail);
               fprintf(f, "Subject: %s Password Retrieval\n\n",
                      NETWORK_NAME);
               fprintf(f, "Channel Password Retreival [by %s (%s@%s)] received on %s for %s:\n %s \n",
                     source, u->username, u->host, timebuf, ci->name, text);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }

                  
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");

                  
      system(cmdbuf);
      system(cmdbuf2);
   

}


/*************************************************************************/
/*************************************************************************/

#endif	/* !SKELETON */
