/* Miscellaneous routines.
 *
 * SirvNET Services is copyright (c) 1998-2002 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services(c) by Andy Church.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "../inc/services.h"


/*************************************************************************/

/* toupper/tolower:  Like the ANSI functions, but make sure we return an
 *                   int instead of a (signed) char.
 */

int toupper(char c)
{
    if (islower(c))
	return (unsigned char)c - ('a'-'A');
    else
	return (unsigned char)c;
}

int tolower(char c)
{
    if (isupper(c))
	return (unsigned char)c + ('a'-'A');
    else
	return (unsigned char)c;
}

/*************************************************************************/

/* sgets2:  Read a line of text from a socket, and strip newline and
 *          carriage return characters from the end of the line.
 */

char *sgets2(char *buf, long size, int sock)
{
    char *s = sgets(buf, size, sock);
    if (!s || s == (char *)-1)
	return s;
    if (buf[strlen(buf)-1] == '\n')
	buf[strlen(buf)-1] = 0;
    if (buf[strlen(buf)-1] == '\r')
	buf[strlen(buf)-1] = 0;
    return buf;
}

/*************************************************************************/

/* strscpy:  Copy at most len-1 characters from a string to a buffer, and
 *           add a null terminator after the last character copied.
 */

char *strscpy(char *d, const char *s, size_t len)
{
    char *d_orig = d;

    if (!len)
	return d;
    while (--len && (*d++ = *s++))
	;
    *d = 0;
    return d_orig;
}

/*************************************************************************/

/* stristr:  Search case-insensitively for string s2 within string s1,
 *           returning the first occurrence of s2 or NULL if s2 was not
 *           found.
 */

char *stristr(char *s1, char *s2)
{
    register char *s = s1, *d = s2;

    while (*s1) {
	if (tolower(*s1) == tolower(*d)) {
	    s1++; d++;
	    if (*d == 0)
		return s;
	} else {
	    s = ++s1;
	    d = s2;
	}
    }
    return NULL;
}

/*************************************************************************/

/* merge_args:  Take an argument count and argument vector and merge them
 *              into a single string in which each argument is separated by
 *              a space.
 */

char *merge_args(int argc, char **argv)
{
    int i;
    static char s[4096];
    char *t;

    t = s;
    for (i = 0; i < argc; ++i)
	t += snprintf(t, sizeof(s)-(t-s), "%s%s", *argv++, (i<argc-1) ? " " : "");
    return s;
}

/*************************************************************************/

/* match_wild:  Attempt to match a string to a pattern which might contain
 *              '*' or '?' wildcards.  Return 1 if the string matches the
 *              pattern, 0 if not.
 */

static int do_match_wild(const char *pattern, const char *str, int docase)
{
    char c;
    const char *s;

    /* This WILL eventually terminate: either by *pattern == 0, or by a
     * trailing '*'. */

    for (;;) {
	switch (c = *pattern++) {
	  case 0:
	    if (!*str)
		return 1;
	    return 0;
	  case '?':
	    if (!*str)
		return 0;
	    ++str;
	    break;
	  case '*':
	    if (!*pattern)
		return 1;	/* trailing '*' matches everything else */
	    s = str;
	    while (*s) {
		if ((docase ? (*s==*pattern) : (tolower(*s)==tolower(*pattern)))
					&& do_match_wild(pattern, s, docase))
		    return 1;
		++s;
	    }
	    break;
	  default:
	    if (docase ? (*str++ != c) : (tolower(*str++) != tolower(c)))
		return 0;
	    break;
	} /* switch */
    }
}

int match_wild(const char *pattern, const char *str)
{
    return do_match_wild(pattern, str, 1);
}

int match_wild_nocase(const char *pattern, const char *str)
{
    return do_match_wild(pattern, str, 0);
}

/*************************************************************************/

/* month_name:  Return the three-character month name corresponding to the
 *              given month number, 1-12.
 */

static char months[12][4] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

char *month_name(int month)
{
    return months[month-1];
}

/*************************************************************************/

/* strupper, strlower:  Convert a string to upper or lower case.
 */

char *strupper(char *s)
{
    char *t = s;
    while (*t)
	*t++ = toupper(*t);
    return s;
}

char *strlower(char *s)
{
    char *t = s;
    while (*t)
	*t++ = tolower(*t);
    return s;
}

/*************************************************************************/

/* read_string, write_string:
 *	Read a string from a file, or write a string to a file, with the
 *	string length prefixed as a two-byte big-endian integer.  The
 *	filename is passed in so that it can be reported in the log file
 *	(and possibly with wallops) if an error occurs.
 */

char *read_string(FILE *f, const char *filename)
{
    char *s;
    int len;

    len = fgetc(f) * 256 + fgetc(f);
    s = smalloc(len);
    if (len != fread(s, 1, len, f)) {
        if (FIX_NSDB) {
           fprintf(stderr, "Read error on file: %s (File either already "
             "patched or corrupted.\n",filename);
           exit(1);
        } else
           fprintf(stderr, "Read error on file: %s",filename);
           fprintf(stderr, "Possible Fix -- TYPE: \"./services -fixns\"\n");
           exit(1);
    }
    return s;

}

char *write_string(const char *s, FILE *f, const char *filename)
{
    int i;

    i = strlen(s) + 1;        /* Include trailing null */
    fputc(i / 256, f);
    fputc(i & 255, f);
    if (i != fwrite(s, 1, i, f))
        fatal_perror("Write error on file: %s", filename);
}

/*************************************************************************/

int check_servii_kill (const char *source, int ac, char **av)
{
    User *user;
    int server = 0;

    user = finduser(source);
    if (!user)
        server = 1;


    if (!stricmp(av[0],s_ChanServ) || !stricmp(av[0],s_NickServ)
        || !stricmp(av[0],s_OperServ) || !stricmp(av[0],s_MemoServ)
        || !stricmp(av[0],s_GlobalNoticer) || !stricmp(av[0],s_AbuseServ)
        || !stricmp(av[0],s_HelpServ)) {

        if (!stricmp(av[0],s_OperServ))
            introduce_user(s_OperServ);
        if (!stricmp(av[0],s_NickServ))
            introduce_user(s_NickServ);
        if (!stricmp(av[0],s_ChanServ))
            introduce_user(s_ChanServ);
        if (!stricmp(av[0],s_GlobalNoticer))
            introduce_user(s_GlobalNoticer);
        if (!stricmp(av[0],s_MemoServ))
            introduce_user(s_MemoServ);
        if (!stricmp(av[0],s_AbuseServ))
            introduce_user(s_AbuseServ);
        if (!stricmp(av[0],s_HelpServ))
            introduce_user(s_HelpServ);
        if (!stricmp(av[0],s_RootServ))
            introduce_user(s_RootServ);


        if (!server) {
            notice(s_GlobalNoticer,source,"Do \2*NOT*\2 kill services!");
            wallops(SERVER_NAME,"%s Tried to kill a services client!", source);
            send_cmd(NULL, "SVSMODE %s -oaA", source);
            return 1;
        } else {
            wallops(SERVER_NAME,"%s revived after server kill", av[0]);
        }

    }
    return 0;
}


/*************************************************************************/

/* A useless little function, but hey, it looks neat. */

void load_sigfile()
{
   FILE *f = fopen(SERVICES_SIG, "r");

   if (!f)
      log("Could not find Services Signature file... continuing");
   else {
      fseek(f, 0L, SEEK_END);
      log("Successfully loaded Services Signature file (%d bytes)",
           ftell(f));
      fclose(f);

   }
}


void custom_auth()
{
   FILE *f = fopen(CUSTOM_AUTH, "r");

   if (!f)
      log("Could not find email customization file %s... continuing",
          CUSTOM_AUTH);
   else {
      fseek(f, 0L, SEEK_END);
      log("Successfully loaded customization e-mail file %s (%d bytes)",
           CUSTOM_AUTH, ftell(f));
      fclose(f);

   }
}

/*************************************************************************/


