/* NickServ functions.
 *
 * SirvNET Services is copyright (c) 1998-2002 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services(c) by Andy Church.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "../inc/services.h"
#include "../inc/timeout.h"
#include "../inc/encrypt.h"

static NickInfo *nicklists[256];	/* One for each initial character */

#include "ns-help.c"


/* For local timeout use: */
#define TO_COLLIDE	0	/* Collide the user with this nick */
#define TO_RELEASE	1	/* Release a collided nick */
#define TO_COLLIDE_TL   2       /* Time left warning to collide */


static void alpha_insert_nick(NickInfo *ni);
static NickInfo *makenick(const char *nick);
static int delnick(NickInfo *ni);
static void do_unhold(const char *source, const char *nick);
static void do_unmark(const char *source, const char *nick);

static void collide(NickInfo *ni, int from_timeout);
static void release(NickInfo *ni, int from_timeout);
static inline void enforce(NickInfo *ni, int from_timeout);
static void add_ns_timeout(NickInfo *ni, int type, time_t delay);
static void del_ns_timeout(NickInfo *ni, int type);

static void update_last_seen_time(Timeout *to);

static void do_ohelp(const char *source);
static void do_help(const char *source);
static void do_register(const char *source);
static void do_identify(const char *source);
static void do_delete(const char *source);
static void do_drop(const char *source);
static void do_set(const char *source);
static void do_set_password(NickInfo *ni, const char *param);
static void do_set_url(NickInfo *ni, const char *param);
static void do_set_email(NickInfo *ni, const char *param);
static void do_set_icq(NickInfo *ni, const char *param);
static void do_set_enforce(NickInfo *ni, const char *param);
static void do_set_secure(NickInfo *ni, const char *param);
static void do_set_private(NickInfo *ni, const char *param);
static void do_set_emailmemos(NickInfo *ni, const char *param);
static void do_set_neverop(NickInfo *ni, const char *param);
static void do_set_noop(NickInfo *ni, const char *param);
static void do_set_nomemo(NickInfo *ni, const char *param);
static void do_set_nohost(NickInfo *ni, const char *param);
static void do_access(const char *source);
// static void do_link(const char *source);
static void do_info(const char *source);
static void do_acc(const char *source);
static void do_flist(const char *source);
static void do_list(const char *source);
static void do_recover(const char *source);
static void do_release(const char *source);
static void do_ghost(const char *source);
static void do_getpass(const char *source);
static void do_sendpass(const char *source);
static void do_forbid(const char *source);
static void do_unforbid(const char *source, const char *nick);
static void do_auth(const char *source);
static void do_regemail(const char *source);
static void flush_regemail(const char *source);
char *get_os_access(NickInfo *ni);

/*************************************************************************/


#ifndef SKELETON

/* Display total number of registered nicks and info about each; or, if
 * a specific nick is given, display information about that nick (like
 * /msg NickServ INFO <nick>).  If count_only != 0, then only display the
 * number of registered nicks (the nick parameter is ignored).
 */

void listnicks(int count_only, const char *nick)
{
    long count = 0;
    NickInfo *ni;
    int i;

    if (count_only) {

	for (i = 33; i < 256; ++i) {
	    for (ni = nicklists[i]; ni; ni = ni->next)
		++count;
	}
	printf("%d nicknames registered.\n", count);

    } else if (nick) {

	struct tm tm;
	char buf[512];

	if (!(ni = findnick(nick))) {
	    printf("%s not registered.\n");
	    return;
	} else if (ni->flags & NI_VERBOTEN) {
	    printf("%s is FORBIDden.\n");
	    return;
	}
	printf("%s is %s\n", nick, ni->last_realname);
	printf("Last seen address: %s\n", ni->last_usermask);
	tm = *localtime(&ni->time_registered);

	printf("  Time registered: %s %2d %02d:%02d:%02d %d\n",
			month_name(tm.tm_mon+1), tm.tm_mday, tm.tm_hour,
			tm.tm_min, tm.tm_sec, tm.tm_year);
	tm = *localtime(&ni->last_seen);
	printf("   Last seen time: %s %2d %02d:%02d:%02d %d\n",
			month_name(tm.tm_mon+1), tm.tm_mday, tm.tm_hour,
			tm.tm_min, tm.tm_sec, tm.tm_year);
	if (ni->url)
	    printf("              URL: %s\n", ni->url);
        if (ni->email && (ni->flags & NI_HIDE_EMAIL))
            printf("   E-mail address: Hidden\n");
        else
	    printf("   E-mail address: %s\n", ni->email);
	*buf = 0;
	if (ni->flags & NI_KILLPROTECT)
	    strcat(buf, "Enforced");
	if (ni->flags & NI_SECURE) {
	    if (*buf)
		strcat(buf, ", ");
	    strcat(buf, "Security");
	}
	if (ni->flags & NI_PRIVATE) {
	    if (*buf)
		strcat(buf, ", ");
	    strcat(buf, "Private");
	}
        if (ni->flags & NI_MARK) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "Marked");
        }
        if (ni->flags & NI_HOLD) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "Held");
        }
	if (!*buf)
	    strcpy(buf, "None");
	printf("          Options: %s\n", buf);

    } else {

	for (i = 33; i < 256; ++i) {
	    for (ni = nicklists[i]; ni; ni = ni->next) {
		printf("    %-20s  %s\n", ni->nick, ni->last_usermask);
		++count;
	    }
	}
	printf("%d nicknames registered.\n", count);

    }
}

#endif	/* !SKELETON */

/*************************************************************************/

NickInfo *retnick(int i)
{
    return nicklists[i];
}

/* Return information on memory use.  Assumes pointers are valid. */

void get_nickserv_stats(long *nrec, long *memuse)
{
    long count = 0, mem = 0;
    int i, j;
    NickInfo *ni;
    char **accptr;


    for (i = 0; i < 256; i++) {
	for (ni = nicklists[i]; ni; ni = ni->next) {
	    count++;
	    mem += sizeof(*ni);
	    if (ni->last_usermask)
		mem += strlen(ni->last_usermask)+1;
	    if (ni->last_realname)
		mem += strlen(ni->last_realname)+1;
            if (ni->url)
                mem += strlen(ni->url)+1;
            if (ni->email)
                mem += strlen(ni->email)+1;
            if (ni->forward)
                mem += strlen(ni->forward)+1;
            if (ni->mark)
                mem += strlen(ni->mark)+1;
            if (ni->hold)
                mem += strlen(ni->hold)+1;
            if (ni->forbid)
                mem += strlen(ni->forbid)+1;

	    mem += sizeof(char *) * ni->accesscount;
	    for (accptr=ni->access, j=0; j < ni->accesscount; accptr++, j++) {
		if (*accptr)
		    mem += strlen(*accptr)+1;
	    }
	}
    }
    *nrec = count;
    *memuse = mem;
}

/*************************************************************************/
/*************************************************************************/

/* Main NickServ routine. */

void nickserv(const char *source, char *buf)
{
    const char *cmd = strtok(buf, " ");
    char *s;


    if (!cmd) {
	return;

    } else if (stricmp(cmd, "\1PING") == 0) {
	if (!(s = strtok(NULL, "")))
	    s = "\1";
	notice(s_NickServ, source, "\1PING %s", s);

#ifdef SKELETON

    } else {
	notice(s_NickServ, source, "%s is currently offline.", s_NickServ);
        return;

    }

#else

    } else if (stricmp(cmd, "HELP") == 0) {
	do_help(source);

    } else if (stricmp(cmd, "REGISTER") == 0) {
	do_register(source);

    } else if (stricmp(cmd, "AUTH") == 0) {
	do_auth(source);

    } else if (stricmp(cmd, "REGEMAIL") == 0) {
        do_regemail(source);
 
    } else if (stricmp(cmd, "IDENTIFY") == 0) {
	do_identify(source);

    } else if (stricmp(cmd, "SIDENTIFY") == 0) {
	do_identify(source);

    } else if (stricmp(cmd, "ID") == 0) {
        do_identify(source);
/*
    } else if (stricmp(cmd, "LINK") == 0) {
	do_link(source);
*/

    } else if (stricmp(cmd, "DROP") == 0) {
	do_drop(source);

    } else if (stricmp(cmd, "SET") == 0) {
	do_set(source);

    } else if (stricmp(cmd, "ACCESS") == 0) {
	do_access(source);

    } else if (stricmp(cmd, "INFO") == 0) {
	do_info(source);

    } else if (stricmp(cmd, "ACC") == 0) {
        do_acc(source);

    } else if (stricmp(cmd, "LIST") == 0) {
	do_list(source);

    } else if (stricmp(cmd, "RECOVER") == 0) {
	do_recover(source);

    } else if (stricmp(cmd, "RELEASE") == 0) {
	do_release(source);

    } else if (stricmp(cmd, "GHOST") == 0) {
	do_ghost(source);

    } else if (stricmp(cmd, "SENDPASS") == 0) {
          do_sendpass(source);

    } else if (stricmp(cmd, "LISTCHANS") == 0) {
          do_chanlist(source);

    } else if (!is_oper(source)) {
	notice(s_NickServ, source,
		"Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
		cmd, s_NickServ);
        return;

    } else if (stricmp(cmd, "OHELP") == 0) {
	do_ohelp(source);

    } else if (stricmp(cmd, "FLIST") == 0) {
        do_flist(source);

    } else if (!check_o_access(source, cmd)) {
          notice(s_NickServ, source, "Permission Denied");

    } else if (stricmp(cmd, "FLUSH") == 0) {
          flush_regemail(source);

    } else if (stricmp(cmd, "FORBID") == 0) {
          do_forbid(source);

    } else if (stricmp(cmd, "GETPASS") == 0) {
          do_getpass(source);

    } else if (stricmp(cmd, "MARK") == 0) {

        const char *nick = strtok(NULL, " ");
        NickInfo *ni;
        User *u = finduser(source);


        if ((!nick) || (stricmp(nick, "-") == 0)) {
            notice(s_NickServ,source,"Syntax: \2MARK\2 [-]<nick>");
            notice(s_NickServ,source,
		"\2/msg %s OHELP MARK\2 for more information",s_NickServ);
            return;
        }
        if (*nick == '-') {
           *nick++;
           do_unmark(source, nick);

	} else if (!(ni = findnick(nick))) {
            notice(s_NickServ, source, "Nickname %s is not registered.",nick);
        } else if (!(ni->flags & NI_MARK)) {
            ni->flags |= NI_MARK;
            if (ni->mark)
                  free(ni->mark);
            ni->mark = sstrdup(u->nick);
            slog("NS +M %s (%s!%s@%s)",
                nick,u->nick,u->username,u->host);
            do_break_log("NS_F", "NS +M %s (%s!%s@%s)",
                nick,u->nick,u->username,u->host);
            wallops(s_NickServ,
                "%s set the MARK flag for \2%s\2",u->nick,nick);
        } else {
            notice(s_NickServ,source,"Nickname %s already marked.",nick);
        }

    } else if (stricmp(cmd, "HOLD") == 0) {
        const char *nick = strtok(NULL, " ");
        NickInfo *ni;
        User *u = finduser(source);

        if ((!nick) || (stricmp(nick, "-") == 0)) {
            notice(s_NickServ,source,"Syntax: \2HOLD\2 [-]<nick>");
            notice(s_NickServ,source,
                "\2/msg %s OHELP HOLD\2 for more information",s_NickServ);
            return;
        }
        if (*nick == '-') {
           *nick++;
           do_unhold(source, nick);

        } else if (!(ni = findnick(nick))) {
            notice(s_NickServ, source, "Nickname %s is not registered.",nick);
        } else if (!(ni->flags & NI_HOLD)) {
            ni->flags |= NI_HOLD;
            if (ni->hold)
                  free(ni->hold);
            ni->hold = sstrdup(u->nick);
            slog("NS +H %s (%s!%s@%s)",
                nick,u->nick,u->username,u->host);
            do_break_log("NS_F", "NS +H %s (%s!%s@%s)",
                nick,u->nick,u->username,u->host);
            wallops(s_NickServ,
                "%s set the HELD flag for \2%s\2",u->nick,nick);
        } else {
            notice(s_NickServ,source,"Nickname %s already held.",nick);
        }

    } else if (stricmp(cmd, "DELETE") == 0) {
          do_delete(source);


    } else {
        User *u = finduser(source);
	notice(s_NickServ, source,
		"Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
		cmd, s_NickServ);

        slog("NS *C %s (%s@%s) [%s]",
            source, u->username, u->host, cmd);
    }

#endif	/* SKELETON */

}

/*************************************************************************/


/* Load/save data files. */

void load_ns_dbase(void)
{
    FILE *f;
    int ver, i, j, len;
    NickInfo *ni;

    if (!(f = open_db(s_NickServ, NICKSERV_DB, "r")))
	return;

    switch (ver = get_file_version(f, NICKSERV_DB)) {
      case 5:
        for (i = 33; i < 256; ++i) {
            while (fgetc(f) == 1) {
                if (FIX_NSDB) {
                   ni = smalloc((sizeof(NickInfo) -24));
                   if (1 != fread(ni, (sizeof(NickInfo)-24), 1, f))
                       fatal_perror("Read error on %s", NICKSERV_DB);
                } else {
                   ni = smalloc(sizeof(NickInfo));
                   if (1 != fread(ni, sizeof(NickInfo), 1, f))
                       fatal_perror("Read error on %s", NICKSERV_DB);
                }

                ni->flags &= ~(NI_IDENTIFIED | NI_RECOGNIZED);
                alpha_insert_nick(ni);
                if (ni->url)
                    ni->url = read_string(f, NICKSERV_DB);
                if (ni->email)
                    ni->email = read_string(f, NICKSERV_DB);
		if (ni->forward)
		    ni->forward = read_string(f, NICKSERV_DB);
                if (ni->hold)
                    ni->hold = read_string(f, NICKSERV_DB);
                if (ni->mark)
                    ni->mark = read_string(f, NICKSERV_DB);
                if (ni->forbid)
                    ni->forbid = read_string(f, NICKSERV_DB);
                ni->last_usermask = read_string(f, NICKSERV_DB);
                ni->last_realname = read_string(f, NICKSERV_DB);
                if (ni->accesscount) {
                    char **access;
                    access = smalloc(sizeof(char *) * ni->accesscount);
                    ni->access = access;
                    for (j = 0; j < ni->accesscount; j++, access++)
                        *access = read_string(f, NICKSERV_DB);
                }
                ni->id_timestamp = 0;
            } /* while (fgetc(f) == 1) */
        } /* for (i) */
        break;

      case 6:
      case 7:
      case 8:
        for (i = 33; i < 256; ++i) {
            while (fgetc(f) == 1) {
                   ni = smalloc(sizeof(NickInfo));
                   if (1 != fread(ni, sizeof(NickInfo), 1, f))
                       fatal_perror("Read error on %s", NICKSERV_DB);

                ni->flags &= ~(NI_IDENTIFIED | NI_RECOGNIZED);
                alpha_insert_nick(ni);
                if (ni->url)
                    ni->url = read_string(f, NICKSERV_DB);
                if (ni->email)
                    ni->email = read_string(f, NICKSERV_DB);
		if (ni->forward)
		    ni->forward = read_string(f, NICKSERV_DB);
                if (ni->hold)
                    ni->hold = read_string(f, NICKSERV_DB);
                if (ni->mark)
                    ni->mark = read_string(f, NICKSERV_DB);
                if (ni->forbid)
                    ni->forbid = read_string(f, NICKSERV_DB);
                if (ni->regemail)
                    ni->regemail = read_string(f, NICKSERV_DB);
                ni->last_usermask = read_string(f, NICKSERV_DB);
                ni->last_realname = read_string(f, NICKSERV_DB);
                if (ni->accesscount) {
                    char **access;
                    access = smalloc(sizeof(char *) * ni->accesscount);
                    ni->access = access;
                    for (j = 0; j < ni->accesscount; j++, access++)
                        *access = read_string(f, NICKSERV_DB);
                }
                ni->id_timestamp = 0;
            } /* while (fgetc(f) == 1) */
        } /* for (i) */
        break;

      default:
	fatal("Unsupported version number (%d) on %s", i, NICKSERV_DB);
    } /* switch (version) */

    close_db(f, NICKSERV_DB);
}

#ifndef SKELETON


void save_ns_dbase(void)
{
    FILE *f;
    int i, j, len;
    NickInfo *ni;
    char **access;

    if (!(f = open_db(s_NickServ, NICKSERV_DB, "w")))
	return;
    for (i = 33; i < 256; ++i) {
	for (ni = nicklists[i]; ni; ni = ni->next) {
	    fputc(1, f);
	    if (1 != fwrite(ni, sizeof(NickInfo), 1, f))
		fatal_perror("Write error on %s", NICKSERV_DB);
	    if (ni->url)
		write_string(ni->url, f, NICKSERV_DB);
	    if (ni->email)
		write_string(ni->email, f, NICKSERV_DB);
	    if (ni->forward)
		write_string(ni->forward, f, NICKSERV_DB);
            if (ni->hold)
                write_string(ni->hold, f, NICKSERV_DB);
            if (ni->mark)
                write_string(ni->mark, f, NICKSERV_DB);
            if (ni->forbid)
                write_string(ni->forbid, f, NICKSERV_DB);
            if (ni->regemail)
                write_string(ni->regemail, f, NICKSERV_DB);

	    write_string(ni->last_usermask ? ni->last_usermask : "",
							f, NICKSERV_DB);
	    write_string(ni->last_realname ? ni->last_realname : "",
							f, NICKSERV_DB);
	    for (access = ni->access, j = 0; j < ni->accesscount; ++access, ++j)
		write_string(*access, f, NICKSERV_DB);
	}
	fputc(0, f);
    }
    close_db(f, NICKSERV_DB);
}

/*************************************************************************/

/* Install any timeouts we need at startup. */

void ns_install_timeouts(void)
{
#if NS_LAST_SEEN_DELAY > 0
    add_timeout(NS_LAST_SEEN_DELAY, update_last_seen_time, 1);
#endif
}

/*************************************************************************/

/* Check whether a user is on the access list of the nick they're using.
 * If not, send warnings as appropriate.  If so (and not NI_SECURE), update
 * last seen info.  Return 1 if the user is valid and recognized, 0
 * otherwise (note that this means an NI_SECURE nick will always return 0
 * from here). */

int validate_user(User *u)
{
    NickInfo *ni;
    int on_access;

    if (!(ni = findnick(u->nick)))
	return;

    if (ni->flags & NI_VERBOTEN) {
	notice(s_NickServ, u->nick,
		"This nickname may not be used.  Please choose another one.");
	notice(s_NickServ, u->nick,
		"If you do not change within one minute, your nick will "
		"be changed.");
        add_ns_timeout(ni, TO_COLLIDE_TL, 30);
//        add_ns_timeout(ni, TO_COLLIDE, 60);
	return 0;
    }

    if (ni->id_timestamp != 0 && u->signon == ni->id_timestamp)
	;

    on_access = is_on_access(u, ni);

    if (!(ni->flags & NI_SECURE) &&
		(on_access || is_on_id_list(u->nick, u->nick))) {
	ni->flags |= NI_RECOGNIZED;
    	    ni->last_seen = time(NULL);
	    if (ni->last_usermask)
	        free(ni->last_usermask);
	    ni->last_usermask = smalloc(strlen(u->username)+strlen(u->host)+2);
	    sprintf(ni->last_usermask, "%s@%s", u->username, u->host);
	    if (ni->last_realname)
 	        free(ni->last_realname);
	    ni->last_realname = sstrdup(u->realname);
	if (is_on_id_list(u->nick, u->nick)) {
	    ni->flags |= NI_IDENTIFIED;
        }
	return 1;
    }

    if (ni->flags & NI_SECURE) {
	notice(s_NickServ, u->nick,
		"This nickname is registered and protected.  If it is your");
	notice(s_NickServ, u->nick,
		"nick, type \2/msg %s IDENTIFY \37password\37\2.  Otherwise,",
		s_NickServ);
	notice(s_NickServ, u->nick,
		"please choose a different nick.");
    } else {
	notice(s_NickServ, u->nick,
		"This nick is owned by someone else.  Please choose another.");
	notice(s_NickServ, u->nick,
		"(If this is your nick, type \2/msg %s IDENTIFY \37password\37\2.)",
		s_NickServ);
    }

    if ((ni->flags & NI_KILLPROTECT)
		&& !((ni->flags & NI_SECURE) && (on_access || is_on_id_list(u->nick, u->nick)))) {
	notice(s_NickServ, u->nick,
		"If you do not change within one minute, your nick will be "
		"changed.");
        add_ns_timeout(ni, TO_COLLIDE_TL, 30);
//        add_ns_timeout(ni, TO_COLLIDE, 60);

    }

    return 0;
}

/*************************************************************************/

/* Cancel validation flags for a nick (i.e. when the user with that nick
 * signs off or changes nicks).  Also cancels any impending collide. */

void cancel_user(User *u)
{
    NickInfo *ni;

    if (ni = findnick(u->nick)) {
	ni->flags &= ~(NI_IDENTIFIED | NI_RECOGNIZED);
	del_ns_timeout(ni, TO_COLLIDE);
    }
}

/*************************************************************************/

/* Remove all nicks which have expired. */

void expire_nicks()
{
    NickInfo *ni, *next;
    int i;
    const time_t expire_time = NICK_EXPIRE*24*60*60;
    time_t now = time(NULL);
    long count = 0;
    long xcount = 0;

    for (i = 33; i < 256; ++i) {
	for (ni = nicklists[i]; ni; ni = next) {
	    next = ni->next;
            ++count;
	    if ((now - ni->last_seen >= expire_time
					&& !(ni->flags & NI_VERBOTEN)
					&& !(ni->flags & NI_HOLD)) 
                      || ((authdel > 0) && auth_email && (ni->flags & NI_AUTH)
                        && (now > (ni->time_registered + (86400 * authdel))))) {

		++xcount;
                slog("NS X %s", ni->nick);
                do_break_log("NS_X", "NS X %s", ni->nick);

		log("Expiring nickname %s", ni->nick);
		delnick(ni);
	    } else if (now - ni->last_seen >= 
                                  (expire_time - (24*60*60*REMINDER))
                                        && (ni->email)
                                        && (ALLOW_EMAIL)
                                        && (REMINDER > 0)
                                        && !(ni->flags & NI_REMIND)
                                        && !(ni->flags & NI_HOLD)
                                        && !(ni->flags & NI_VERBOTEN)) {


              struct tm tm;
              char timebuf[64];
              char cmdbuf[500]; char cmdbuf2[500];
              FILE *f = fopen("memo.txt", "w");
              FILE *fs = fopen(SERVICES_SIG, "r");

               tm = *localtime(&now);
               strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
               timebuf[sizeof(timebuf)-1] = 0;
               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->regemail);
               fprintf(f, "Subject: %s registration reminder\n\n",
                      NETWORK_NAME);
               fprintf(f, "Your nick (%s) will expire in %d days\n",
                      ni->nick, REMINDER);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }

               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");
               system(cmdbuf);
               system(cmdbuf2);

               ni->flags |= NI_REMIND;
                slog("NS X+ %s", ni->nick);
               do_break_log("NS_X", "NS X+ %s", ni->nick);

            }
	}
    }
    if (DISPLAY_UPDATES == 1)
        wallops(SERVER_NAME, "Completed NickName Expire (%d/%d)",
        	xcount,count);
}

#endif /* Skeleton */

/*************************************************************************/

/* Return the NickInfo structure for the given nick, or NULL if the nick
 * isn't registered. */

NickInfo *findnick(const char *nick)
{
    NickInfo *ni;

    for (ni = nicklists[tolower(*nick)]; ni; ni = ni->next) {
	if (stricmp(ni->nick, nick) == 0)
	    return ni;
    }
    return NULL;
}

#ifndef SKELETON

/*************************************************************************/
/*********************** NickServ private routines ***********************/
/*************************************************************************/

/* Is the given user's address on the given nick's access list?  Return 1
 * if so, 0 if not. */

int is_on_access(User *u, NickInfo *ni)
{
    int i;
    char *buf;

    i = strlen(u->username);
    buf = smalloc(i + strlen(u->host) + 2);
    sprintf(buf, "%s@%s", u->username, u->host);
    strlower(buf+i+1);
    for (i = 0; i < ni->accesscount; ++i) {
	if (match_wild(ni->access[i], buf)) {
            free(buf);
	    return 1;
        }
    }
    free(buf);
    return 0;
}

/*************************************************************************/

#endif /* SKELETON */

/* Insert a nick alphabetically into the database. */

static void alpha_insert_nick(NickInfo *ni)
{
    NickInfo *ni2, *ni3;
    char *nick = ni->nick;

    for (ni3 = NULL, ni2 = nicklists[tolower(*nick)];
			ni2 && stricmp(ni2->nick, nick) < 0;
			ni3 = ni2, ni2 = ni2->next)
	;
    ni->prev = ni3;
    ni->next = ni2;
    if (!ni3)
	nicklists[tolower(*nick)] = ni;
    else
	ni3->next = ni;
    if (ni2)
	ni2->prev = ni;
}

/*************************************************************************/

#ifndef SKELETON

/* Add a nick to the database.  Returns a pointer to the new NickInfo
 * structure if the nick was successfully registered, NULL otherwise.
 * Assumes nick does not already exist. */

static NickInfo *makenick(const char *nick)
{
    NickInfo *ni;
    User *u;

    ni = scalloc(sizeof(NickInfo), 1);
    strscpy(ni->nick, nick, NICKMAX);
    alpha_insert_nick(ni);
    return ni;
}

/*************************************************************************/

/* Remove a nick from the NickServ database.  Return 1 on success, 0
 * otherwise.  Also deletes the nick from any channel access lists it is
 * on. */

static int delnick(NickInfo *ni)
{
    User *u;
    int i;

    for (u = userlist; u; u = u->next) {
       for (i=0; u->id_nicks[i] && i < MAX_IDS; i++) {
          if (stricmp(u->id_nicks[i], ni->nick) == 0)
              free(u->id_nicks[i]);
       }
    }
    for (i=0; i < MAX_SENDLOGS; i++) {
       if (sendlogs[i]) {
          if (ni && stricmp(ni->nick, sendlogs[i]) == 0) {
              free(sendlogs[i]);
              sendlogs[i] = NULL;
          }
       }
    }

    if (ni->next)
	ni->next->prev = ni->prev;
    if (ni->prev)
	ni->prev->next = ni->next;
    else
	nicklists[tolower(*ni->nick)] = ni->next;
    if (ni->last_usermask)
	free(ni->last_usermask);
    if (ni->last_realname)
	free(ni->last_realname);
    if (ni->access) {
	for (i = 0; i < ni->accesscount; ++i)
	    free(ni->access[i]);
	free(ni->access);
    }
    if (ni->url)
      free(ni->url);
    if (ni->email)
      free(ni->email);
    if (ni->forward)
      free(ni->forward);
    if (ni->hold)
      free(ni->hold);
    if (ni->mark)
      free(ni->mark);
    if (ni->forbid)
      free(ni->forbid);
    cs_remove_nick(ni->nick);
    clear_memos(ni->nick);
    os_remove_nick(ni->nick);
    free(ni);
    return 1;
}


static void collide(NickInfo *ni, int from_timeout)
{
    unsigned long grn = getrandom(1000,300000);
    char *newnick = (char *)malloc(sizeof(char *));
    User *user = finduser(ni->nick);

    if (!from_timeout)
        del_ns_timeout(ni, TO_COLLIDE);

    sprintf(newnick, "Guest%d", grn);

    send_cmd(NULL, "SVSNICK %s %s %ld", ni->nick, newnick, time(NULL));
    if (user)
       strscpy(user->nick, newnick, NICKMAX);

#ifdef BAHAMUT
   send_cmd(NULL, "NICK %s 1 %lu %s enforcer %s %s 0 :Nick Protection Enforcement",
         ni->nick, time(NULL), "+", services_host, server_name);
#else
   send_cmd(NULL, "NICK %s 1 %lu enforcer %s %s 0 :Nick Protection Enforcement",
         ni->nick, time(NULL), services_host, server_name);
#endif
   ni->flags |= NI_KILL_HELD;
   add_ns_timeout(ni, TO_RELEASE, RELEASE_TIMEOUT);

   free(newnick);
}




/*************************************************************************/

/* Release hold on a nick. */

static void release(NickInfo *ni, int from_timeout)
{
    if (!from_timeout)
	del_ns_timeout(ni, TO_RELEASE);
    send_cmd(ni->nick, "QUIT");
    ni->flags &= ~NI_KILL_HELD;
}

/*************************************************************************/
/*************************************************************************/

static struct my_timeout {
    struct my_timeout *next, *prev;
    NickInfo *ni;
    Timeout *to;
    int type;
} *my_timeouts;

/*************************************************************************/

/* Remove a collide/release timeout from our private list. */

static void rem_ns_timeout(NickInfo *ni, int type)
{
    struct my_timeout *t, *t2;

    t = my_timeouts;
    while (t) {
	if (t->ni == ni && t->type == type) {
	    t2 = t->next;
	    if (t->next)
		t->next->prev = t->prev;
	    if (t->prev)
		t->prev->next = t->next;
	    else
		my_timeouts = t->next;
	    free(t);
	    t = t2;
	} else {
	    t = t->next;
	}
    }
}

/*************************************************************************/

/* Collide a nick on timeout. */

static void timeout_collide(Timeout *t)
{
    NickInfo *ni = t->data;
    User *u;

    rem_ns_timeout(ni, TO_COLLIDE);
    /* If they identified or don't exist anymore, don't kill them. */
    if (ni->flags & NI_IDENTIFIED
		|| !(u = finduser(ni->nick))
		|| u->my_signon > t->settime)
	return;
    /* The RELEASE timeout will always add to the beginning of the
     * list, so we won't see it.  Which is fine because it can't be
     * triggered yet anyway. */
    collide(ni, 1);
}

static void timeout_collide_tl(Timeout *t)
{
   NickInfo *ni = t->data;
   User *u;

   rem_ns_timeout(ni, TO_COLLIDE_TL);
   if (ni->flags & NI_IDENTIFIED
		|| !(u = finduser(ni->nick))
		|| u->my_signon > t->settime)
	return;
   add_ns_timeout(ni, TO_COLLIDE, 30);
   notice(s_NickServ, ni->nick,
          "If you do not change within 30 seconds, your nick will "
          "be changed.");
}

/*************************************************************************/

/* Release a nick on timeout. */

static void timeout_release(Timeout *t)
{
    NickInfo *ni = t->data;

    rem_ns_timeout(ni, TO_RELEASE);
    release(ni, 1);
}

/*************************************************************************/


/* Add a collide/release timeout. */

static void add_ns_timeout(NickInfo *ni, int type, time_t delay)
{
    Timeout *to;
    struct my_timeout *t;
    void (*timeout_routine)(Timeout *);

    if (type == TO_COLLIDE_TL)
        timeout_routine = timeout_collide_tl;
    else if (type == TO_COLLIDE)
	timeout_routine = timeout_collide;
    else if (type == TO_RELEASE)
	timeout_routine = timeout_release;
    else {
	log("NickServ: unknown timeout type %d!  ni=%p (%s), delay=%ld",
		type, ni, ni->nick, delay);
	return;
    }
    to = add_timeout(delay, timeout_routine, 0);
    to->data = ni;
    t = smalloc(sizeof(*t));
    t->next = my_timeouts;
    my_timeouts = t;
    t->prev = NULL;
    t->ni = ni;
    t->to = to;
    t->type = type;
}

/*************************************************************************/

/* Delete a collide/release timeout. */

static void del_ns_timeout(NickInfo *ni, int type)
{
    struct my_timeout *t, *t2;

    t = my_timeouts;
    while (t) {
	if (t->ni == ni && t->type == type) {
	    t2 = t->next;
	    if (t->next)
		t->next->prev = t->prev;
	    if (t->prev)
		t->prev->next = t->next;
	    else
		my_timeouts = t->next;
	    del_timeout(t->to);
	    free(t);
	    t = t2;
	} else {
	    t = t->next;
	}
    }
}

/*************************************************************************/
/*************************************************************************/

/* Update last-seen time for all users currently online. */

static void update_last_seen_time(Timeout *to_unused)
{
    User *u;
    NickInfo *ni;

    /* This is O(n^2) at the moment, because the user list isn't known to
     * be sorted.
     */
    for (u = first_user(); u; u = u->next) {
	if (ni = findnick(u->nick))
            if (ni && (ni->flags & (NI_IDENTIFIED | NI_RECOGNIZED)))
        	 ni->last_seen = time(NULL);
    }
}

/*************************************************************************/
/*********************** NickServ command routines ***********************/
/*************************************************************************/

/* Return a help message. */

static void do_ohelp(const char *source)
{
    const char *cmd = strtok(NULL, "");
    char buf[BUFSIZE];
    char *access;
    int is_servadmin = is_services_oper(source);

    if (!cmd)
        log("NS - %s requested ohelp: (index)", source);

    if (cmd) {
        log("NS - %s requested ohelp: %s", source, cmd);

        if (stricmp(cmd, "DELETE") == 0) {
            notice_list(s_NickServ, source, oper_delete_help);
        } else if (stricmp(cmd, "LIST") == 0) {
            notice_list(s_NickServ, source, oper_list_help);
        } else if (stricmp(cmd, "FLIST") == 0) {
            notice_list(s_NickServ, source, flist_help);
        } else if (stricmp(cmd, "FLUSH") == 0) {
            notice_list(s_NickServ, source, flush_help);
        } else if (stricmp(cmd, "GETPASS") == 0) {
            notice_list(s_NickServ, source, getpass_help);
        } else if (stricmp(cmd, "FORBID") == 0) {
            notice_list(s_NickServ, source, forbid_help);
        } else if (stricmp(cmd, "MARK") == 0) {
            notice_list(s_NickServ, source, mark_help);
        } else if (stricmp(cmd, "HOLD") == 0) {
            notice_list(s_NickServ, source, hold_help);
        } else {
            notice(s_NickServ, source, "No help topic for \2%s\2 found.", cmd);
            return;
        }

    }

    if (!cmd)
       serv_os_help(source, "NS", s_NickServ);
    else {
       if (access = find_level(cmd))
          notice(s_NickServ, source,
            "This command is limited to \2%s\2.", access);
    }
    

    notice(s_NickServ, source,
       "\2END OF OHELP\2");

}

static void do_help(const char *source)
{
    const char *cmd = strtok(NULL, "");
    char buf[BUFSIZE];

/* If we have email disabled display a different help file */
    if (cmd && stricmp(cmd, "REGISTER") == 0) {
      if (!ALLOW_EMAIL) {
        (char *)cmd = 0; (const char *)cmd = "register2";
        snprintf(buf, sizeof(buf), "%s%s", s_NickServ, cmd ? " " : "");
        strscpy(buf+strlen(buf), cmd ? cmd : "", sizeof(buf)-strlen(buf));
        helpserv(s_NickServ, source, buf);
        return;
      }
    }
    snprintf(buf, sizeof(buf), "%s%s", s_NickServ, cmd ? " " : "");
    strscpy(buf+strlen(buf), cmd ? cmd : "", sizeof(buf)-strlen(buf));
    helpserv(s_NickServ, source, buf);
    if (!cmd || (cmd && stricmp(cmd, "REGISTER") == 0))
       notice(s_NickServ, source,
             "Nicknames will expire after \2%d days\2 of inactivity",
              NICK_EXPIRE);
}

/*************************************************************************/

/* Register a nick. */

static void do_register(const char *source)
{
    NickInfo *ni;
    User *u;
    char *gender;
    const char *pass = strtok(NULL, " ");
    const char *email = strtok(NULL, " ");
    int i;

    if (readonly) {
	notice(s_NickServ, source,
		"Sorry, nickname registration is temporarily disabled.");
	return;
    }


    if (!pass || (!email && auth_email)) {
        notice(s_NickServ, source, "Syntax: \2REGISTER\2 <password> %s",
              auth_email==1 ? "<e-mail>" : "");

	notice(s_NickServ, source,
			"\2/msg %s HELP REGISTER\2 for more information.",
			s_NickServ);


    } else if (!(u = finduser(source))) {

	log("%s: Can't register nick %s: nick not online", s_NickServ, source);
	notice(s_NickServ, source, "Sorry, registration failed.");

    } else if (time(NULL) < u->lastnickreg + NS_REG_DELAY) {

	notice(s_NickServ, source,
	    "Please wait %d seconds before using the REGISTER command again.",
	    NS_REG_DELAY);

    } else if (ni = findnick(source)) {

	if (ni->flags & NI_VERBOTEN) {
	    log("%s: %s@%s tried to register FORBIDden nick %s", s_NickServ,
			u->username, u->host, source);
	    notice(s_NickServ, source,
			"Nickname \2%s\2 may not be registered.", source);
	} else {
	    notice(s_NickServ, source,
			"Nickname \2%s\2 is already registered!", source);
	}

    } else if (stricmp(source, pass) == 0 || strlen(pass) < 5) {

	notice(s_NickServ, source,
		"Please try again with a more obscure password.  Passwords "
		"should be at least five characters long, should not be "
		"something easily guessed (e.g. your real name or your "
		"nick), and cannot contain the space or tab characters.  "
		"\2/msg %s HELP REGISTER\2 for more information.", s_NickServ);

    } else if (strlen(pass) > PASSMAX) {
         notice(s_NickServ, source,
             "Sorry, nick passwords may only be up to %d characters long",
                 PASSMAX);

    } else if (match_wild_nocase("guest*", source)) {
         notice(s_NickServ, source,
                "You may not register nicks starting with \"Guest\". "
                "\2/msg %s HELP REGISTER\2 for more information.", s_NickServ);

    } else if (email && (!strchr(email, '@')) && auth_email) {
         notice(s_NickServ, source,
             "ERROR: Email must contain an '@'");
    } else if (email && ((!strchr(email, '.')) || strchr(email, '\\')
               || strchr(email, '&') || strchr(email, '/')) &&
               auth_email) {
         notice(s_NickServ, source, "ERROR: Invalid email address");

    } else {

      struct tm tm;
      char timebuf[64];
      char cmdbuf[500]; char cmdbuf2[500];
      time_t now = time(NULL);
      FILE *f = fopen("memo.txt", "w");

	if (ni = makenick(source)) {
	    strscpy(ni->pass, pass, PASSMAX);
	    ni->flags = NI_IDENTIFIED | NI_RECOGNIZED;
            if (!is_on_id_list(source, source)) {
                for (i=0; u->id_nicks[i] && i < MAX_IDS; i++)
                    ;
                if (i < MAX_IDS)
                     u->id_nicks[i] = sstrdup(source);
            }
            nsreg += 1;
            if (enforce_default)
   	        ni->flags |= NI_KILLPROTECT;
	    ni->flags |= NI_MEMO_SIGNON | NI_MEMO_RECEIVE;
	    ni->last_usermask = smalloc(strlen(u->username)+strlen(u->host)+2);
            if (AUTO_NOHOST > 0)
                ni->flags |= NI_NOHOST;
	    sprintf(ni->last_usermask, "%s@%s", u->username, u->host);
            if (auth_email)
               ni->regemail = sstrdup(email);
            ni->flags |= NI_AUTH;
	    ni->last_realname = sstrdup(u->realname);
	    ni->time_registered = ni->last_seen = time(NULL);
	    ni->accesscount = 1;
	    ni->access = smalloc(sizeof(char *));
	    ni->access[0] = create_mask(u);
            ni->memomax = DEF_MAX_MEMOS;
	    clear_memos(ni->nick);
	    log("%s: `%s' registered by %s@%s", s_NickServ,
		    source, u->username, u->host);
            slog("NS R %s (%s@%s)", source, u->username, u->host);

            do_break_log("NS_R", "NS R %s (%s@%s)",
                 source, u->username, u->host);

          if (auth_email) {

               FILE *fs2 = fopen(CUSTOM_AUTH, "r");
               FILE *fs = fopen(SERVICES_SIG, "r");
               tm = *localtime(&now);
               strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
               timebuf[sizeof(timebuf)-1] = 0;

               ni->auth = ni->time_registered + getrandom(1,99999) * getrandom(1,9999);

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->regemail);
               fprintf(f, "Subject: %s registration AUTH\n\n",
                      NETWORK_NAME);
               fprintf(f, "Your nick's AUTH code is %d. Log back onto %s using the nick %s and type - /MSG %s AUTH %d\n",
                      ni->auth, NETWORK_NAME, ni->nick, s_NickServ, ni->auth);
               fprintf(f, "--------------------------------\n\n");
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }
               if (fs2) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", CUSTOM_AUTH);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs2);
               }
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


               system(cmdbuf);
               system(cmdbuf2);

         }


            notice(s_NickServ, source, NS_REGISTER, source, ni->access[0]);
#ifndef USE_ENCRYPTION
            notice(s_NickServ, source, NS_REGISTER2, pass);
#endif
         if (enforce_default)
            notice(s_NickServ, source, "By default, your nick has ENFORCE protection enabled.");
         if (auth_email)
            notice(s_NickServ, source, NS_REGISTER3, pass);
         if (nsecure)
            notice(s_NickServ, source, NS_REGISTER4);
         
#if REG_SAVES > 0
            if (regcnt++ >= REG_SAVES) {
                wallops(SERVER_NAME,
                     "Automatic DataBase save on %d new registrations",
                          regcnt);
                save_data = 1;
                regcnt = 0;
            }
#endif

            show_next_db(source, s_NickServ);
	    u->lastnickreg = time(NULL);
            if (ENABLE_R_MODES && (ENABLE_R_MODES != 2))
               send_cmd(s_NickServ, "SVSMODE %s +r 0", source);
	} else {
	    log("%s: makenick(%s) failed", s_NickServ, source);
	    notice(s_NickServ, source,
		"Sorry, couldn't register your nickname.");
	}

    }

}

/*************************************************************************/

static void do_identify(const char *source)
{
    char *nick = strtok(NULL, " ");
    char *pass = strtok(NULL, " ");
    char *tmp;
    NickInfo *ni;
    User *u;
    int res;
    char access[128];
//    char **link;
    int i, x=0;
    int j;
    int is_force;

    *access = 0;

    if (!nick) {

        notice(s_NickServ, source, "Syntax: \2IDENTIFY\2 [nick] <password>");
	notice(s_NickServ, source,
		"\2/msg %s HELP IDENTIFY\2 for more information.",
		s_NickServ);
	return;
    }

    if (!pass) {
	pass = sstrdup(nick);
        nick = sstrdup(source);
        x=1;
    }
    ni = findnick(nick);
    is_force = (is_services_root(source) && !stricmp(pass, "force"));
    if (!ni) {
	notice(s_NickServ, source, "Your nick (%s) isn't registered.",nick);
    } else if (!(u = finduser(source))) {

	log("%s: IDENTIFY from nonexistent nick %s", s_NickServ, source);
	notice(s_NickServ, source, "Sorry, identification failed.");

    } else if (u->flags & U_ABUSIVE) {

	log("%s: Failed identify from abusive user %s", s_NickServ, source);
	notice(s_NickServ, source, "Sorry, You have the abusive flag set");
      if (EXTRA_SNOOP > 0) {
          slog("NS *I %s (%s!%s@%s) [Abusive]",
		nick, source, u->username, u->host);
          do_break_log("NS_I", "NS *I %s (%s!%s@%s)",
              nick, source, u->username, u->host);
          nsidf += 1;
      }


    } else if (!is_force && !(res = check_password(pass, ni->pass))) {

	log("%s: Failed IDENTIFY for %s (%s!%s@%s)",
		s_NickServ, nick, source, u->username, u->host);
        slog("NS *I %s (%s!%s@%s)",
             nick, source, u->username, u->host);
        do_break_log("NS_I", "NS *I %s (%s!%s@%s)",
                nick, source, u->username, u->host);
        notice(s_NickServ, source, NS_BADPASS);
	bad_password(u, nick);
        nsidf += 1;

    } else if (res == -1) {

	notice(s_NickServ, source, "Sorry, identification failed.");
        nsidf += 1;

    } else {

      if (!(ni->flags & NI_IDENTIFIED)) {
        if (EXTRA_SNOOP) {
          slog("NS I %s (%s!%s@%s) %s",
		nick, source, u->username, u->host,
                is_force ? "[Forced]" : "");
          do_break_log("NS_I", "NS I %s (%s!%s@%s) %s",
             nick, source, u->username, u->host,
             is_force ? "[Forced]" : "");
        }
          nsid += 1;
      }

        nsid += 1;
	ni->flags |= NI_IDENTIFIED;
	ni->id_timestamp = u->signon;


	if (!is_on_id_list(u->nick, nick)) {
	    for (i=0; u->id_nicks[i] && i < MAX_IDS; i++)
		;
	    if (i < MAX_IDS)
	         u->id_nicks[i] = sstrdup(nick);
	}

	if (!(ni->flags & NI_SECURE)) {
	    ni->last_seen = time(NULL);
	    if (ni->last_usermask)
		free(ni->last_usermask);
	    ni->last_usermask = smalloc(strlen(u->username)+strlen(u->host)+2);
	    sprintf(ni->last_usermask, "%s@%s", u->username, u->host);
	    if (ni->last_realname)
		free(ni->last_realname);
	    ni->last_realname = sstrdup(u->realname);
                  
	    log("%s: %s!%s@%s identified for nick %s %s", s_NickServ,
			source, u->username, u->host, source,
                        is_force ? "[Forced]" : "");
	}
        if (ni->flags & NI_REMIND)
             ni->flags &= ~NI_REMIND;
        if (!ni->regemail && auth_email)
             notice(s_NickServ, source,
                "Your nick was registered before the implementation of "
                "the AUTH security feature. Take the extra step in "
                "securing your nick, type - /MSG %s REGEMAIL "
                "<your@email.com>", s_NickServ);

	if (is_services_coder(source)) {
                if (ENABLE_UMODE_a)
                    send_cmd(s_NickServ,"SVSMODE %s +a 0",source);
                strcpy(access, "as a Services Master.");
	} else if (is_services_root(source)) {
                if (ENABLE_UMODE_a)
                    send_cmd(s_NickServ,"SVSMODE %s +a 0",source);
                strcpy(access, "as a Services Root.");
	} else if (is_services_admin(source)) {
                if (ENABLE_UMODE_a)
                    send_cmd(s_NickServ,"SVSMODE %s +a 0",source);
                strcpy(access, "as a Services Admin.");
	} else if (is_services_oper(source)) {
                strcpy(access, "as a Services Operator.");
	} else if (is_services_helpop(source)) {
                send_cmd(s_NickServ,"SVSMODE %s +hw 0",source);
                strcpy(access, "as a Services HelpOp.");
	} else {
                strcpy(access, "");
	}
        notice(s_NickServ, source, NS_IDENTIFY,ni->nick,access);
        if (ENABLE_R_MODES && (ENABLE_R_MODES != 2))
            send_cmd(s_NickServ, "SVSMODE %s +r 0", source);

	if (!(ni->flags & NI_RECOGNIZED))
	    check_memos(source);
    }
    if (x) {
      free(nick);
      free(pass);
    }
}

/*************************************************************************/

static void do_delete(const char *source)
{

    const char *nick = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source), *user;
    int is_root = is_services_root(source), i;

    if (readonly) {
	notice(s_NickServ, source,
                "Sorry, nickname deletion is temporarily disabled. (Read Only)");
	return;
    }
    if (!nick) {
       notice(s_NickServ, source, "\2Delete\2: You MUST specify a nick");
       return;
    }

    if (!(ni = findnick(nick))) {
        if (nick)
           notice(s_NickServ, source, "Nick %s isn't registered.", nick);
        return;
    }
    if (stricmp(nick, SERVICES_MASTER) == 0 && !is_services_coder(source)) {
        notice(s_NickServ, source,
            "You are not permitted to delete a Master's account");
        return;
    }
    for (user = userlist; user; user = user->next) {
       for (i=0; user->id_nicks[i] && i < MAX_IDS; i++) {
          if (stricmp(user->id_nicks[i], ni->nick) == 0) {
              free(user->id_nicks[i]);
              user->id_nicks[i] = NULL;
          }
       }
    }
    delnick(ni);
    slog("NS +D %s (%s!%s@%s)", nick, source, u->username, u->host);
    do_break_log("NS_G", "NS +D %s (%s!%s@%s)",
       nick, source, u->username, u->host);
    log("%s: %s!%s@%s deleted nickname %s", s_NickServ,
            source, u->username, u->host, nick);
        notice(s_NickServ, source, NS_DELETE, nick);
        wallops(s_NickServ, "%s deleted nickname \2%s\2", source, nick);
}



/*************************************************************************/

static void do_drop(const char *source)
{
    const char *nick = strtok(NULL, " ");
    const char *auth = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source), *user;
    int i;

    if (readonly) {
	notice(s_NickServ, source,
		"Sorry, nickname de-registration is temporarily disabled.");
	return;
    }

    if (!nick) {
       notice(s_NickServ, source, "Syntax: \2DROP\2 <nick> %s",
              auth_email ? "[<drop code>]" : "");
       return;
    }

    if (!(ni = findnick(nick))) {
	    notice(s_NickServ, source, "Your nick isn't registered.");

    } else if (!is_on_id_list(source, nick)) {

        slog("NS *D %s (%s!%s@%s) [User Hasn't Identified]", source, source, u->username, u->host);
        do_break_log("NS_O", "NS *D %s (%s!%s@%s) [User Hasn't Identified]", source, source, u->username, u->host);
	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY [\37nick\37] \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: DROP from nonexistent user %s", s_NickServ, source);

    } else if (!u) {

	log("%s: DROP %s from nonexistent oper %s", s_NickServ, nick, source);
	notice(s_NickServ, source, "Can't find your user record!");

    } else if (auth_email && (ni->flags & NI_AUTH)) {
          notice(s_NickServ, source,
                "You may not drop your nick until you have properly authorized this nick.");

    } else if (auth_email && !auth && !ni->auth) {

               struct tm tm;
               time_t now = time(NULL);
               FILE *f = fopen("memo.txt", "w");
               FILE *fs2 = fopen(CUSTOM_AUTH, "r");
               FILE *fs = fopen(SERVICES_SIG, "r");
               char cmdbuf[500], cmdbuf2[500], timebuf[64];
               tm = *localtime(&now);
               strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
               timebuf[sizeof(timebuf)-1] = 0;

               ni->auth = ni->time_registered + getrandom(1,99999) * getrandom(1,9999);

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->regemail);
               fprintf(f, "Subject: %s nick drop AUTH\n\n",
                      NETWORK_NAME);
               fprintf(f, "You or someone has recently requested your nickname to be dropped.\n");
               fprintf(f, "If you wish this action to be taken, follow these steps:\n");
               fprintf(f, "Your nick's DROP code is %d. Log back onto %s using the nick %s and type - /MSG %s DROP %d\n",
                      ni->auth, NETWORK_NAME, ni->nick, s_NickServ, ni->auth);
               fprintf(f, "--------------------------------\n\n");
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }
               if (fs2) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", CUSTOM_AUTH);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs2);
               }
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


               system(cmdbuf);
               system(cmdbuf2);

         notice(s_NickServ, source,
               "You will receive a drop authorization code in your email shortly. "
                "Please follow the instructions of the email to complete the action "
                "of dropping this nick");

    } else if (auth_email && !auth && ni->auth) {
          notice(s_NickServ, source,
                "A drop authorization code has already been sent. "
                "Please contact a %s representitive for further assistance.", NETWORK_NAME);

    } else if ((auth_email && auth) && (atoi(auth) != ni->auth)) {
          notice(s_NickServ, source,
               "Invalid drop authorization code.");

    } else {

	if (readonly) {
	    notice(s_NickServ, source,
		"Warning: Services is in read-only mode.  Changes will not be saved.");
	}
        if (stricmp(nick, SERVICES_MASTER) == 0 && !is_services_coder(source)) {
            notice(s_NickServ, source,
               "You are not permitted to delete a Master's account");
            return;
        }
        for (user = userlist; user; user = user->next) {
           for (i=0; user->id_nicks[i] && i < MAX_IDS; i++) {
              if (stricmp(user->id_nicks[i], ni->nick) == 0) {
                  free(user->id_nicks[i]);
                  user->id_nicks[i] = NULL;
              }
           }
        }
	delnick(ni);

        nsdrop += 1;
        slog("NS D %s (%s!%s@%s)", nick, source, u->username, u->host);
        do_break_log("NS_O", "NS D %s (%s!%s@%s)",
             nick, source, u->username, u->host);
	log("%s: %s!%s@%s dropped nickname %s", s_NickServ,
		source, u->username, u->host, nick ? nick : source);
        notice(s_NickServ, source, NS_DROP);
    }
}

/*************************************************************************/

static void do_set(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    const char *param  = strtok(NULL, " ");
    NickInfo *ni = findnick(source);
    User *u;

    if (readonly) {
	notice(s_NickServ, source,
		"Sorry, nickname option setting is temporarily disabled.");
	return;
    }

    if (!param) {

	notice(s_NickServ, source,
                "Syntax: \2SET\2 <command> [<parameters>]");
	notice(s_NickServ, source,
		"\2/msg %s HELP SET\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(source))) {

	notice(s_NickServ, source, "Your nickname is not registered.");

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: SET from nonexistent user %s", s_NickServ, source);


    } else if ((ni->flags & NI_AUTH || !ni->regemail)
        && nsecure && auth_email) {
        notice(s_NickServ, source,
            "You may not change any settings until you have authorized "
            "your nick. If you need assistance, find a %s representitive.",
             NETWORK_NAME);

    } else if (stricmp(cmd, "PASSWORD") == 0) {

	do_set_password(ni, param);

    } else if (stricmp(cmd, "URL") == 0) {

	do_set_url(ni, param);

    } else if (stricmp(cmd, "EMAIL") == 0) {

	do_set_email(ni, param);

    } else if (stricmp(cmd, "ICQ") == 0) {

	do_set_icq(ni, param);

    } else if (stricmp(cmd, "ENFORCE") == 0 || stricmp(cmd, "KILL") == 0) {

	do_set_enforce(ni, param);

    } else if (stricmp(cmd, "SECURE") == 0) {

	do_set_secure(ni, param);

    } else if (stricmp(cmd, "PRIVATE") == 0) {

	do_set_private(ni, param);

    } else if (stricmp(cmd, "EMAILMEMOS") == 0) {
	do_set_emailmemos(ni, param);

    } else if (stricmp(cmd, "NEVEROP") == 0) {
        do_set_neverop(ni, param);

    } else if (stricmp(cmd, "NOOP") == 0) {
        do_set_noop(ni, param);

    } else if (stricmp(cmd, "NOMEMO") == 0) {
        do_set_nomemo(ni, param);

    } else if (stricmp(cmd, "NOHOST") == 0) {
        do_set_nohost(ni, param);

    } else {

	notice(s_NickServ, source,
			"Unknown SET option \2%s\2.", strupper((char *)cmd));

    }
    nsset += 1;
}

/*************************************************************************/

static void do_set_password(NickInfo *ni, const char *param)
{
    const char *source = ni->nick;
    User *u = finduser(source);

    if (stricmp(source, param) == 0 || strlen(param) < 5) {

	notice(s_NickServ, source,
		"Please try again with a more obscure password.  Passwords "
		"should be at least five characters long, should not be "
		"something easily guessed (e.g. your real name or your "
		"nick), and cannot contain the space or tab characters.  "
		"\2/msg %s HELP REGISTER\2 for more information.", s_NickServ);

    } else if (strlen(param) > PASSMAX) {
         notice(s_NickServ, source,
             "Sorry, nick passwords may only be up to %d characters long",
                 PASSMAX);

    } else {

       strscpy(ni->pass, param, PASSMAX);
       notice(s_NickServ, ni->nick, "Password changed to \2%s\2.", param);
       show_next_db(source, s_NickServ);
       if (!u)
         return;
       slog("NS P %s (%s!%s@%s)",
         ni->nick, u->nick, u->username, u->host);
       do_break_log("NS_O", "NS P %s (%s!%s@%s)",
         ni->nick, u->nick, u->username, u->host);

    }
}

/*************************************************************************/

static void do_set_url(NickInfo *ni, const char *param)
{
    const char *source = ni->nick;
    if (ni->url)
	free(ni->url);

    if (stricmp(param,"none")) {
        ni->url = sstrdup(param);
        notice(s_NickServ, ni->nick,
		"URL for %s changed to \2%s\2.", ni->nick, param);
    } else {
       if (!ni->url) {
            notice(s_NickServ, ni->nick,
                "You have no url set to disable!");
            return;
       }

       ni->url = NULL;
       notice(s_NickServ, source, "URL disabled for nick \2%s\2", ni->nick);
    }
    show_next_db(source, s_NickServ);
}

/*************************************************************************/

static void do_set_email(NickInfo *ni, const char *param)
{
    const char *source = ni->nick;
    int i;

    if ((stricmp(param,"none")) && (stricmp(param,"hide"))
          && (stricmp(param,"-hide"))) {

       if (!strchr(param, '@')) {
            notice(s_NickServ, source,
               "ERROR: E-Mail Address must contain an '@'");
            return;
       }
       if (strchr(param, '\\') || strchr(param, '&') ||
           strchr(param, '/')) {
           notice(s_NickServ, source,
               "ERROR: invalid email characters.");
           return;
       }

       if (ni->email)
           free(ni->email);

       ni->email = sstrdup(param);
       notice(s_NickServ, ni->nick,
		"E-mail address for %s changed to \2%s\2.", ni->nick, param);
       slog("NS E %s [%s]", ni->nick, param);

     } else if (!stricmp(param,"hide")) {
         if (!ni->email) {
            notice(s_NickServ, ni->nick,
                "You have no email address set to hide!");
            return;
         }
         ni->flags |= NI_HIDE_EMAIL;
         notice(s_NickServ, ni->nick,
                  "Your email address will no longer be shown");
         show_next_db(source, s_NickServ);
         return;

     } else if (!stricmp(param,"-hide")) {
         if (!ni->email) {
            notice(s_NickServ, ni->nick,
                "You have no email address set!");
            return;
         }
         ni->flags &= ~NI_HIDE_EMAIL;
         notice(s_NickServ, ni->nick,
                  "Your email address will now be visible to all");
         show_next_db(source, s_NickServ);
         return;

    } else {
       if (!ni->email) {
           notice(s_NickServ, ni->nick,
               "You have no email address set to disable!");
           return;
       }

       for (i=0; i < MAX_SENDLOGS; i++) {
         if (sendlogs[i]) {
           if (ni && stricmp(ni->nick, sendlogs[i])) {
              notice(s_NickServ, source,
                 "ERROR: Can't remove email when in sendlogs list");
              return;
            }
          }
       }

       ni->flags &= ~NI_HIDE_EMAIL;
       free(ni->email);
       ni->email = NULL;
       notice(s_NickServ, source, "EMAIL disabled for nick \2%s\2", ni->nick);
    }
    show_next_db(source, s_NickServ);
}

/*************************************************************************/

static void do_set_icq(NickInfo *ni, const char *param)
{
   const char *source = ni->nick;

   if (stricmp(param, "NONE") == 0) {
       if (!ni->icq) {
         notice(s_NickServ, source,
             "You don't have an icq uin to disable!");
         return;
       } else {
          ni->icq = 0;
          notice(s_NickServ, source,
              "Your ICQ uin is now disabled");
          return;
       }
    } else {
       long uin = strtol(param, (char **)&param, 10);
       if (!(uin > 0))
           notice(s_NickServ, source, "Invalid uin number");
       else {
           ni->icq = uin;
           notice(s_NickServ, source,
                "Your ICQ uin is now set to: %ld", ni->icq);
       }
    }
}


/*************************************************************************/

static void do_set_enforce(NickInfo *ni, const char *param)
{
    const char *source = ni->nick;

    if (stricmp(param, "ON") == 0) {

	ni->flags |= NI_KILLPROTECT;
	notice(s_NickServ, source, "Nick Enforcement is now \2ON\2.");
        show_next_db(source, s_NickServ);

    } else if (stricmp(param, "OFF") == 0) {

	ni->flags &= ~NI_KILLPROTECT;
	notice(s_NickServ, source, "Nick Enforcement is now \2OFF\2.");
        show_next_db(source, s_NickServ);

    } else {

        notice(s_NickServ, source, "Syntax: \2SET ENFORCE\2 [\37ON\37|\37OFF\37]");
	notice(s_NickServ, source,
		"\2/msg %s HELP SET ENFORCE\2 for more information.", s_NickServ);
    }
}

/*************************************************************************/

static void do_set_secure(NickInfo *ni, const char *param)
{
    const char *source = ni->nick;

    if (stricmp(param, "ON") == 0) {

	ni->flags |= NI_SECURE;
	notice(s_NickServ, source, "Secure option is now \2ON\2.");
        show_next_db(source, s_NickServ);

    } else if (stricmp(param, "OFF") == 0) {

	ni->flags &= ~NI_SECURE;
	notice(s_NickServ, source, "Secure option is now \2OFF\2.");
        show_next_db(source, s_NickServ);

    } else {

        notice(s_NickServ, source, "Syntax: \2SET SECURE\2 [\37ON\37|\37OFF\37]");
	notice(s_NickServ, source,
		"\2/msg %s HELP SET SECURE\2 for more information.",
		s_NickServ);
    }
}

/*************************************************************************/

static void do_set_private(NickInfo *ni, const char *param)
{
    const char *source = ni->nick;

    if (stricmp(param, "ON") == 0) {

	ni->flags |= NI_PRIVATE;
	notice(s_NickServ, source, "Private option is now \2ON\2.");
        show_next_db(source, s_NickServ);

    } else if (stricmp(param, "OFF") == 0) {

	ni->flags &= ~NI_PRIVATE;
	notice(s_NickServ, source, "Private option is now \2OFF\2.");
        show_next_db(source, s_NickServ);

    } else {

        notice(s_NickServ, source, "Syntax: \2SET PRIVATE\2 [\37ON\37|\37OFF\37]");
	notice(s_NickServ, source,
		"\2/msg %s HELP SET PRIVATE\2 for more information.",
		s_NickServ);
    }
}

static void do_set_emailmemos(NickInfo *ni, const char *param)
{
    const char *source = ni->nick;


    if (stricmp(param, "ON") == 0) {
	if (!ni->email) {
	    notice(s_NickServ,source,"You must set your e-mail address.");
	    notice(s_NickServ,source,"Do do this type: \2/msg %s set email \37email\37\2",s_NickServ);
	    return;
        } else if (ALLOW_EMAIL == 0) {
              notice(s_NickServ, source,
                    "Sorry, emailing memos have been disabled");
              return;
        }
        ni->flags |= NI_EMAILMEMOS;
        notice(s_NickServ, source, "Your memos will now be emailed to %s.", ni->email);
        show_next_db(source, s_NickServ);

    } else if (stricmp(param, "OFF") == 0) {

        ni->flags &= ~NI_EMAILMEMOS;
        notice(s_NickServ, source, "Your memos will no longer be emailed.");
        show_next_db(source, s_NickServ);

    } else {

        notice(s_NickServ, source, "Syntax: \2SET EMAILMEMOS\2 [\37ON\37|\37OFF\37]");
        notice(s_NickServ, source,
                "\2/msg %s HELP SET EMAILMEMOS\2 for more information.",
                s_NickServ);
    }
}


/*************************************************************************/

static void do_set_neverop(NickInfo *ni, const char *param)
{
   const char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "\2Never Op\2 setting enabled");
       ni->flags |= NI_NEVEROP;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "\2Never Op\2 setting disabled");
       ni->flags &= ~NI_NEVEROP;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET NEVEROP\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET NEVEROP\2 for more information.",
              s_NickServ);

  }
}



/*************************************************************************/

static void do_set_noop(NickInfo *ni, const char *param)
{
   const char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "\2No Op\2 setting enabled");
       ni->flags |= NI_NOOP;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "\2No Op\2 setting disabled");
       ni->flags &= ~NI_NOOP;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET NOOP\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET NOOP\2 for more information.",
              s_NickServ);

  }
}

/*************************************************************************/

static void do_set_nomemo(NickInfo *ni, const char *param)
{
   const char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "\2No Memo\2 setting enabled");
       ni->flags |= NI_NOMEMO;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "\2No Memo\2 setting disabled");
       ni->flags &= ~NI_NOMEMO;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET NOMEMO\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET NOMEMO\2 for more information.",
              s_NickServ);

  }
}

static void do_set_nohost(NickInfo *ni, const char *param)
{
   const char *source = ni->nick;

   if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "\2No Host Display\2 setting disabled");
       ni->flags &= ~NI_NOHOST;
       show_next_db(source, s_NickServ);
       return;

    } else if (!NOHOST_CONF) {
       notice(s_NickServ, source, 
          "The \2NOHOST\2 command has been disabled by the network.");
       return;


   } else if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "\2No Host Display\2 setting enabled");
       ni->flags |= NI_NOHOST;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET NOHOST\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET NOHOST\2 for more information.",
              s_NickServ);
   }
}

/*************************************************************************/

/*

static void do_link(const char *source)
{
    char *cmd = strtok(NULL, " ");
    char *nick = strtok(NULL, " ");
    NickInfo *ni;
    User *u;
    int i;
    char **link;
    if (cmd && stricmp(cmd, "LIST") == 0 && nick && is_services_admin(source)) {
        if (!(ni = findnick(nick))) {
            notice(s_NickServ, source, "Nick \2%s\2 not registered!", nick);
            return;
        }
        notice(s_NickServ, source, "Link list for \2%s\2:", nick);
        for (link = ni->link, i = 0; i < ni->linkcount; ++link, ++i)
            notice(s_NickServ, source, "    %s", *link);
	notice(s_NickServ, source, "\2End of List\2");

    } else if (!cmd || (((stricmp(cmd,"LIST")==0) || (stricmp(cmd,"WIPE")==0)) ? !!nick : !nick)) {
        notice(s_NickServ, source,
                "Syntax: \2LINK {ADD|DEL|LIST|WIPE} [\37mask\37]\2");
        notice(s_NickServ, source,
                "\2/msg %s HELP LINK\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(source))) {
       notice(s_NickServ, source, "Your nick isn't registered.");

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

        notice(s_NickServ, source,
                "Password authentication required for that command.");
        notice(s_NickServ, source,
                "Retry after typing \2/msg %s IDENTIFY \37password\37.",
                s_NickServ);
        if (!u)
            log("%s: SET from nonexistent user %s", s_NickServ, source);

    } else if (stricmp(cmd, "ADD") == 0) {

	if (stricmp(source, nick) == 0) {
	    notice(s_NickServ,source, "There is no reason to add your "
		"nick to your own link list.");
	    return;
	}

        for (link = ni->link, i = 0; i < ni->linkcount; ++link, ++i) {
            if (strcmp(*link, nick) == 0) {
                notice(s_NickServ, source,
                        "Nick \2%s\2 already present on your link list.",
                        *link);
                return;
            }
        }
	if (!is_on_id_list(source, nick)) {
	    notice(s_NickServ,source,"Please identify to that nickname "
		"before adding it to your link list.");
	    notice(s_NickServ,source,"/msg %s IDENTIFY %s \37password\37",
		s_NickServ, nick);
	    return;
	}
        ++ni->linkcount;
        ni->link = srealloc(ni->link, sizeof(char *) * ni->linkcount);
        ni->link[ni->linkcount-1] = sstrdup(nick);
        notice(s_NickServ, source, "\2%s\2 added to your link list.", nick);

    } else if (stricmp(cmd, "DEL") == 0) {

        for (link = ni->link, i = 0; i < ni->linkcount; ++link, ++i) {
            if (strcmp(*link, nick) == 0)
                break;
        }
        if (i == ni->linkcount) {
            for (link = ni->link, i = 0; i < ni->linkcount;
                                                        ++link, ++i) {
                if (stricmp(*link, nick) == 0)
                    break;
            }
        }
        if (i == ni->linkcount) {
            notice(s_NickServ, source,
                        "\2%s\2 not found on your link list.", nick);
            return;
        }
        notice(s_NickServ, source,
                "\2%s\2 deleted from your link list.", *link);
        free(*link);
        --ni->linkcount;
        if (i < ni->linkcount)
            bcopy(link+1, link, (ni->linkcount-i) * sizeof(char *));
        if (ni->linkcount)
            ni->link = srealloc(ni->link, ni->linkcount * sizeof(char *));
        else {
            free(ni->link);
            ni->link = NULL;
        }

    } else if (stricmp(cmd, "LIST") == 0) {

        notice(s_NickServ, source, "\2Link list:\2");
        for (link = ni->link, i = 0; i < ni->linkcount; ++link, ++i) {
            notice(s_NickServ, source, "    %s", *link);
        }
        notice(s_NickServ, source, "\2End of List\2");

    } else if (stricmp(cmd, "WIPE") == 0) {
        for (link = ni->link, i = 0; i < ni->linkcount; ++link, ++i) {
              free(*link);
        }
        ni->linkcount=0;
        free(ni->link);
        ni->link = NULL;
        notice(s_NickServ,source,"Your link list has been wiped.");

    } else {

        notice(s_NickServ, source,
                "Syntax: \2LINK {ADD|DEL|LIST|WIPE} [\37mask\37]\2");
        notice(s_NickServ, source,
                "\2/msg %s HELP LINK\2 for more information.", s_NickServ);
    }
}

*/

static void do_access(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    const char *mask = strtok(NULL, " ");
    NickInfo *ni;
    User *u;
    int i;
    char **access;

    if (cmd && stricmp(cmd, "LIST") == 0 && mask && is_services_admin(source)) {

	if (!(ni = findnick(mask))) {
	    notice(s_NickServ, source, "Nick \2%s\2 not registered!", mask);
	    return;
	}
	notice(s_NickServ, source, "Access list for \2%s\2:", mask);
	for (access = ni->access, i = 0; i < ni->accesscount; ++access, ++i)
	    notice(s_NickServ, source, "    %s", *access);

    } else if (!cmd || (((stricmp(cmd,"LIST")==0) || (stricmp(cmd,"WIPE")==0)) ? !!mask : !mask)) {

	notice(s_NickServ, source,
                "Syntax: \2ACCESS\2 [ADD|DEL|LIST|WIPE] [mask]");
	notice(s_NickServ, source,
		"\2/msg %s HELP ACCESS\2 for more information.", s_NickServ);

    } else if (mask && ((!strchr(mask, '@') || strchr(mask, '!')) &&
               (stricmp(cmd, "ADD") == 0))) {

	notice(s_NickServ, source,
		"Mask must be in the form \37user\37@\37host\37.");
	notice(s_NickServ, source,
		"\2/msg %s HELP ACCESS\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(source))) {

	notice(s_NickServ, source, "Your nick isn't registered.");

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: SET from nonexistent user %s", s_NickServ, source);

    } else if (stricmp(cmd, "ADD") == 0) {

       if (strlen(mask) > 60) {
           notice(s_NickServ, source, "NickServ access mask too large!");
           wallops(s_NickServ, "%s tried to add a HUGE mask to the access list", source);
           return;
       }
       if (stricmp(mask, "*@*") == 0 || stricmp(mask, "*@*.*") == 0) {
           notice(s_NickServ, source,
               "Please try to add a more secure access mask");
           return;
        }

	for (access = ni->access, i = 0; i < ni->accesscount; ++access, ++i) {
	    if (strcmp(*access, mask) == 0) {
		notice(s_NickServ, source,
			"Mask \2%s\2 already present on your access list.",
			*access);
		return;
	    }
	}
        if ((ni->accesscount > USER_ACCESS_MAX-1) && (USER_ACCESS_MAX > 0)) {
            notice(s_NickServ, source,
                 "Sorry, access list full for nick %s", ni->nick);
            wallops(SERVER_NAME, "%s tried to add another access mask", ni->nick);
            return;
        }
            
	++ni->accesscount;
	ni->access = srealloc(ni->access, sizeof(char *) * ni->accesscount);
	ni->access[ni->accesscount-1] = sstrdup(mask);
	notice(s_NickServ, source, "\2%s\2 added to your access list.", mask);
        show_next_db(source, s_NickServ);

        if (readonly)
            notice(s_NickServ, source,
               "Warning: Services is in read-only mode; changes will not be saved.");

    } else if (stricmp(cmd, "DEL") == 0) {
        int z, j=0;

        if (strspn(mask, "1234567890") == strlen(mask) &&
                            (i = atoi(mask)) > 0 && i <= ni->accesscount) {

            for (access = ni->access, z = 0; z < ni->accesscount;
                        ++access, ++z) {

               j += 1;
               if (i == j) {
                   access = &ni->access[z];
                   break;
               }
               
            }
            if (!(i == j)) {
                notice(s_NickServ, source,
                        "No such entry (#%d) on %s access list.", i, mask);
                return;
            }
            i = z;

        } else {

           for (access = ni->access, i = 0; i < ni->accesscount; ++access, ++i) {
               if (strcmp(*access, mask) == 0)
                    break;
           }
           if (i == ni->accesscount) {
               for (access = ni->access, i = 0; i < ni->accesscount;
                                                     ++access, ++i) {
               if (stricmp(*access, mask) == 0)
		    break;
               }
           }
           if (i == ni->accesscount) {
               notice(s_NickServ, source,
			"\2%s\2 not found on your access list.", mask);
               return;
           }
        }

        notice(s_NickServ, source,
             "\2%s\2 deleted from your access list.", *access);
        show_next_db(source, s_NickServ);
        if (readonly)
            notice(s_NickServ, source,
               "Warning: Services is in read-only mode; changes will not be saved.");

	free(*access);
	--ni->accesscount;
	if (i < ni->accesscount)	/* if it wasn't the last entry... */
	    bcopy(access+1, access, (ni->accesscount-i) * sizeof(char *));
	if (ni->accesscount)		/* if there are any entries left... */
	    ni->access = srealloc(ni->access, ni->accesscount * sizeof(char *));
	else {
	    free(ni->access);
	    ni->access = NULL;
	}

    } else if (stricmp(cmd, "LIST") == 0) {

	notice(s_NickServ, source, "\2Access list:\2");
	for (access = ni->access, i = 0; i < ni->accesscount; ++access, ++i) {
	    if (mask && !match_wild(mask, *access))
		continue;
            notice(s_NickServ, source, " %d)  %s", i+1, *access);
	}
	notice(s_NickServ, source, "\2End of List\2");

    } else if (stricmp(cmd, "WIPE") == 0) {
        for (access = ni->access, i = 0; i < ni->accesscount; ++access, ++i) {
	      free(*access);
	}
        ni->accesscount=0;
        free(ni->access);
        ni->access = NULL;
	notice(s_NickServ,source,"Your access list has been wiped.");

    } else {

	notice(s_NickServ, source,
                "Syntax: \2ACCESS\2 [ADD|DEL|LIST|WIPE] [mask]");
	notice(s_NickServ, source,
		"\2/msg %s HELP ACCESS\2 for more information.", s_NickServ);

    }
}

/*************************************************************************/

static void do_acc(const char *source)
{
 NickInfo *ni;
 const char *nick = strtok(NULL, " ");

 if (!nick) {
     notice(s_NickServ, source, "Syntax: \2ACC\2 <nick>");
     notice(s_NickServ, source,
         "\2/msg %s HELP ACC\2 for more information.", s_NickServ);
          return;

    } else if ((ni = findnick(nick)) && (ni->flags & NI_VERBOTEN)) {
	notice(s_NickServ, source,
                "Nick \2%s\2: 5 [FORBIDDEN]", nick);
           return;

    } else if (!(ni = findnick(nick))) {
        notice(s_NickServ, source, "Nick \2%s\2: 0 [UnRegistered]", nick);
          return;

    } else if (!finduser(nick)) {
        notice(s_NickServ, source, "Nick \2%s\2: 1 [OffLine]", nick);
         return;

    } else if ((ni = findnick(nick)) && (ni->flags & NI_IDENTIFIED)) {
         notice(s_NickServ, source,
                 "Nick \2%s\2: 3 [Identified]", nick); 
           return;

    } else if ((ni = findnick(nick)) && (ni->flags & NI_RECOGNIZED)) {
        notice(s_NickServ, source, "\2%s\2: 4 Recognized [Access List]", nick);
           return;

    } else if ((ni = findnick(nick)) && !(ni->flags & NI_IDENTIFIED)) {
	notice(s_NickServ, source,
                "Nick \2%s\2: 2 [UnIdentified]", nick);
           return;


  } else {

     notice(s_NickServ, source, "Nick \2%s\2: Unable to Retreive Data", nick);
   }
}



/*************************************************************************/

static void do_info(const char *source)
{
    const char *nick = strtok(NULL, " ");
    char *tmp, timebuf[64];
    NickInfo *ni;
    time_t tbuff;
    User *u;

    if (!nick) {
        notice(s_NickServ, source, "Syntax: \2INFO\2 <nick>");
	notice(s_NickServ, source,
		"\2/msg %s HELP INFO\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(nick))) {
	notice(s_NickServ, source, "Nick \2%s\2 isn't registered.", nick);

    } else if (ni->flags & NI_VERBOTEN) {
	notice(s_NickServ, source,
		"Nick \2%s\2 may not be registered or used.", nick);

    } else {
	struct tm tm;
	char buf[512];
        int ltime = (time(NULL) - ni->last_seen);
        u = finduser(nick);

        if (u && (ni->flags & NI_IDENTIFIED))
            notice(s_NickServ, source,
               "(\2Currently on IRC\2 [Identified]) - \"/whois %s\" for more information", nick);
        else if (u && (ni->flags & NI_RECOGNIZED))
            notice(s_NickServ, source,
               "(\2Currently on IRC\2 [Recognized]) - \"/whois %s\" for more information", nick);

        if (ni->last_realname)
            notice(s_NickServ, source,
	    	"\2%s\2 is \"%s\"\n", nick, ni->last_realname);
	notice(s_NickServ, source,
		"           Access: %s\n", get_os_access(ni));
	if (!(ni->flags & NI_NOHOST) && ni->last_usermask)
	    notice(s_NickServ, source,
		    "Last seen address: %s\n", ni->last_usermask);

	tm = *localtime(&ni->time_registered);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;
        notice(s_NickServ, source,
                "  Time registered: %s", timebuf);

	tm = *localtime(&ni->last_seen);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;
        if (ltime > 86400)
  	   notice(s_NickServ, source,
		"   Last seen time: %s [over %d day%s]", timebuf,
                   ltime/86400, (ltime/86400 == 1) ? "" : "s");
        else if (ltime > 3600)
  	   notice(s_NickServ, source,
		"   Last seen time: %s [over %d hour%s]", timebuf,
                   ltime/3600, (ltime/3600 == 1) ? "" : "s");
        else
  	   notice(s_NickServ, source,
		"   Last seen time: %s [under 1 hour]", timebuf);


        tbuff = time(NULL);
        tm = *localtime(&tbuff);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;
	notice(s_NickServ, source,
		"         Time Now: %s", timebuf);

	if (ni->url)
	    notice(s_NickServ, source,
		"              URL: %s\n", ni->url);
        if (ni->email && (!(ni->flags & NI_HIDE_EMAIL)))
	    notice(s_NickServ, source,
		"   E-mail address: %s\n", ni->email);

        if (ni->icq > 0)
             notice(s_NickServ, source,
                "          ICQ uin: %ld", ni->icq);

	*buf = 0;
	if (ni->flags & NI_KILLPROTECT)
	    strcat(buf, "Enforced");
	if (ni->flags & NI_SECURE) {
	    if (*buf)
		strcat(buf, ", ");
	    strcat(buf, "Security");
	}
        if (ni->flags & NI_NEVEROP) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "NEVER OP");
        }
        if (ni->flags & NI_NOOP) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "NO OP");
        }
        if (ni->flags & NI_NOMEMO) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "NO MEMO");
        }
        if (ni->flags & NI_NOHOST) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "NO HOST");
        }
        if (ni->flags & NI_PRIVATE) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "Private");
        }

	if (!*buf)
	    strcpy(buf, "None");
	notice(s_NickServ, source, "          Options: %s", buf);

        if (ni->forward)
           notice(s_NickServ, source,
               " * Nick is forwarding memos to: %s", ni->forward);

        if (is_oper(source)) {
            if (ni->flags & NI_MARK)
                notice(s_NickServ, source,
                   " * Nick marked by: %s", ni->mark);
            if (ni->flags & NI_HOLD)
                notice(s_NickServ, source,
                   " * Nick Held by: %s", ni->hold);
            if (ALLOW_EMAIL) {
              if (ni->flags & NI_AUTH || !ni->regemail)
                  notice(s_NickServ, source,
                     " * This nick has not been authorized");
              else
                  notice(s_NickServ, source,
                     " * This nick has been authorized via: %s",
                        ni->regemail);
            }

        } else {
           if (ni->flags & NI_MARK)
               notice(s_NickServ, source,
                  " * This nick has been marked");
           if (ni->flags & NI_HOLD)
               notice(s_NickServ, source,
                  " * This nick has been held");
       }        
       notice(s_NickServ, source, "\2*** End of Info ***\2");

    }
}

/*************************************************************************/

static void do_list(const char *source)
{
    const char *pattern = strtok(NULL, " ");
    NickInfo *ni;
    int nnicks, i;
    char buf[BUFSIZE];
    int is_servadmin = is_services_admin(source);

    if (!switch_list && !is_oper(source)) {

        notice(s_NickServ, source,
            "Sorry, the LIST command has been disabled by the network");

    } else if (!pattern) {

        notice(s_NickServ, source, "Syntax: \2LIST\2 [pattern]");
	notice(s_NickServ, source,
		"\2/msg %s HELP LIST\2 for more information.", s_NickServ);

    } else {

	nnicks = 0;
	notice(s_NickServ, source, "List of entries matching \2%s\2:",
		pattern);
	for (i = 33; i < 256; ++i) {
	    for (ni = nicklists[i]; ni; ni = ni->next) {
		if (ni->flags & NI_VERBOTEN)
		    continue;
		if (!is_servadmin && (ni->flags & NI_PRIVATE))
		    continue;
		snprintf(buf, sizeof(buf), "%-20s  %s", ni->nick, ni->last_usermask);
		if (stricmp(pattern, ni->nick) == 0 ||
					match_wild_nocase(pattern, buf)) {
		    if (++nnicks <= 50)
			notice(s_NickServ, source, "    %s", buf);
		}
	    }
	}
	notice(s_NickServ, source, "End of list - %d/%d matches shown.",
					nnicks>50 ? 50 : nnicks, nnicks);
    }

}

/*************************************************************************/

static void do_recover(const char *source)
{
    const char *nick = strtok(NULL, " ");
    const char *pass = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (!nick) {

	notice(s_NickServ, source,
                "Syntax: \2RECOVER\2 <nickname> <password>");
	notice(s_NickServ, source,
		"\2/msg %s HELP RECOVER\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(nick))) {

	notice(s_NickServ, source, "Nick %s isn't registered.", nick);

    } else if (!finduser(nick)) {

	notice(s_NickServ, source, "Nick %s isn't currently in use.");

    } else if (stricmp(nick, source) == 0) {

	notice(s_NickServ, source, "You can't recover yourself.");

    } else if (u &&
	    (pass ? check_password(pass,ni->pass)==1 :
	     !(ni->flags & NI_SECURE) && is_on_access(u,ni))) {

	collide(ni, 0);
	notice(s_NickServ, source, "User claiming your nick has been killed.");
	notice(s_NickServ, source,
		"\2/msg %s RELEASE\2 to get it back before the one-minute "
		"timeout.", s_NickServ);

    } else {

	notice(s_NickServ, source, "Access denied.");
	if (!u)
	    log("%s: RECOVER: source user %s not found!", s_NickServ, source);
	else if (pass) {
	    log("%s: RECOVER: invalid password for %s by %s!%s@%s",
                        s_NickServ, nick, source, u->username, u->host);
           if (EXTRA_SNOOP) {
                slog("NS *R %s (%s!%@%s)",
                     nick, source, u->username, u->host);
                 do_break_log("NS_O", "NS *R %s (%s!%@%s)",
                     nick, source, u->username, u->host);
           }
	   bad_password(u, nick);
	}

    }
}

/*************************************************************************/

static void do_release(const char *source)
{
    const char *nick = strtok(NULL, " ");
    const char *pass = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (!nick) {

	notice(s_NickServ, source,
                "Syntax: \2RELEASE\2 <nickname> <password>");
	notice(s_NickServ, source,
		"\2/msg %s HELP RELEASE\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(nick))) {

	notice(s_NickServ, source, "Nick %s isn't registered.", nick);

    } else if (!(ni->flags & NI_KILL_HELD)) {

	notice(s_NickServ, source, "Nick %s isn't being held.", nick);

    } else if (u &&
	    (pass ? check_password(pass,ni->pass)==1 :
	     !(ni->flags & NI_SECURE) && is_on_access(u,ni))) {

	release(ni, 0);
	notice(s_NickServ, source,
		"Services' hold on your nick has been released.");

    } else {

	notice(s_NickServ, source, "Access denied.");
	if (!u)
	    log("%s: RELEASE: source user %s not found!", s_NickServ, source);
	else if (pass) {
	    log("%s: RELEASE: invalid password for %s by %s!%s@%s",
                        s_NickServ, nick, source, u->username, u->host);
            if (EXTRA_SNOOP) {
                slog("NS *Re %s (%s!%s@%s)",
                    nick, source, u->username, u->host);
                do_break_log("NS_O", "NS *Re %s (%s!%s@%s)",
                    nick, source, u->username, u->host);
            }
	    bad_password(u, nick);
	}

    }
}

/*************************************************************************/

static void do_ghost(const char *source)
{
    const char *nick = strtok(NULL, " ");
    const char *pass = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (!nick) {

	notice(s_NickServ, source,
                "Syntax: \2GHOST\2 <nickname> <password>");
	notice(s_NickServ, source,
		"\2/msg %s HELP GHOST\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(nick))) {

	notice(s_NickServ, source, "Nick %s isn't registered.", nick);

    } else if (!finduser(nick)) {

	notice(s_NickServ, source, "Nick %s isn't currently in use.", nick);

    } else if (stricmp(nick, source) == 0) {

	notice(s_NickServ, source, "You can't ghost yourself!");

    } else if (u &&
	    (pass ? check_password(pass,ni->pass)==1 :
	     !(ni->flags & NI_SECURE) && is_on_access(u,ni))) {

	char buf[NICKMAX+32];

	snprintf(buf, sizeof(buf), "Ghost command used by %s", source);
        kill_user(s_NickServ, nick, buf);
	notice(s_NickServ, source, "Ghost with your nick has been killed.");
        ghost += 1;

    } else {

	notice(s_NickServ, source, "Access denied.");
	if (!u)
	    log("%s: GHOST: source user %s not found!", s_NickServ, source);
	else if (pass) {
	    log("%s: GHOST: invalid password for %s by %s!%s@%s",
                        s_NickServ, nick, source, u->username, u->host);
            if (EXTRA_SNOOP) {
                slog("NS *K %s (%s!%s@%s)",
                     nick, source, u->username, u->host);
                do_break_log("NS_O", "NS *K %s (%s!%s@%s)",
                     nick, source, u->username, u->host);
            }
	    bad_password(u, nick);
	}

    }
}

/*************************************************************************/

static void do_sendpass(const char *source)
{
    const char *nick = strtok(NULL, " ");
    const char *email = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (!auth_email) {
        notice(s_NickServ, source, "Email systems have been disabled by the network");
    } else if (!nick || !email) {
        notice(s_NickServ, source, "Syntax: \2SENDPASS\2 <nickname> <e-mail>");
    } else if (!u) {
	notice(s_NickServ, source, "Couldn't get your user info!");
    } else if (!(ni = findnick(nick))) {
	notice(s_NickServ, source, "Nick %s isn't registered.", nick);
    } else if (ni->flags & NI_AUTH || !ni->regemail) {
          notice(s_NickServ, source,
               "User %s has not authorized via email", ni->nick);
    } else if (stricmp(email, ni->regemail) != 0) {
        notice(s_NickServ, source, "The email you specified does not match with the one on file");

    } else if (ni->flags & NI_MARK) {
        slog("NS *S %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        do_break_log("NS_G", "NS *S %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        log("%s: %s!%s@%s attempted SENDPASS on %s (Fail)"
                ,s_NickServ, source, u->username, u->host, nick);

    } else {
        nsgp += 1;
	log("%s: %s!%s@%s used SENDPASS on %s",
		s_NickServ, source, u->username, u->host, nick);
        slog("NS S %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        wallops(s_NickServ, "%s used SENDPASS on %s", source, nick);
        notice(s_NickServ, source,
              "Password sent to %s", ni->regemail);
        sendpass_ns(source, ni);
    }
}


/*************************************************************************/

static void do_getpass(const char *source)
{
    const char *nick = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);


    if (!getpass_enable) {
        notice(s_NickServ, source, "GETPASS has been disabled by the network");
        return;
    }

#ifdef USE_ENCRYPTION
    notice(s_NickServ, source,
		"GETPASS command unavailable when encryption is in use.");
#else
    if (!nick) {
        notice(s_NickServ, source, "Syntax: \2GETPASS\2 <nickname>");
    } else if (!u) {
	notice(s_NickServ, source, "Couldn't get your user info!");
    } else if (!(ni = findnick(nick))) {
	notice(s_NickServ, source, "Nick %s isn't registered.", nick);
    } else if (is_services_admin(nick) && !is_services_coder(source)
          && !stricmp(source, nick) == 0) {
        notice(s_NickServ, source,
           "You may not getpass a services administrator");

    } else if (ni->flags & NI_MARK) {
        slog("NS *G %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        do_break_log("NS_G", "NS *G %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
            notice(s_NickServ,source,"Nick \2%s\2 is marked!",nick);
        log("%s: %s!%s@%s attempted GETPASS on %s (Fail)"
                ,s_NickServ, source, u->username, u->host, nick);
    } else {
        nsgp += 1;
	log("%s: %s!%s@%s used GETPASS on %s",
		s_NickServ, source, u->username, u->host, nick);
        slog("NS G %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        slog("NS G %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        wallops(s_NickServ, "%s used GETPASS on %s", source, nick);
	notice(s_NickServ, source, "Password for %s is \2%s\2.",
		nick, ni->pass);
    }
#endif
}

/*************************************************************************/

static void do_forbid(const char *source)
{
    NickInfo *ni;
    User *u = finduser(source), *user;
    const char *nick = strtok(NULL, " ");
    int i;

    if (!nick) {
        notice(s_NickServ, source, "Syntax: \2FORBID\2 [-]<nickname>");
        notice(s_NickServ,source,
             "\2/msg %s OHELP FORBID\2 for more information",s_NickServ);
	return;
    }
    if (*nick == '-') {
       *nick++;
       do_unforbid(source, nick);
       return;
    }
    if (readonly) {
	notice(s_NickServ, source,
	    "Warning: Services is in read-only mode; changes will not be saved.");
    }
    if (is_services_oper(nick)) {
           notice(s_NickServ, source, "Error: Nick is a services admin");
           wallops(s_NickServ, "%s tried to forbid \2%s\2", source, nick);
           return;
    }
    if (ni = findnick(nick)) {
        if (ni->forbid) {
           notice(s_NickServ, source, "Nick %s already forbidden", nick);
           return;
        }
        for (user = userlist; user; user = user->next) {
           for (i=0; user->id_nicks[i] && i < MAX_IDS; i++) {
              if (stricmp(user->id_nicks[i], ni->nick) == 0) {
                  free(user->id_nicks[i]);
                  user->id_nicks[i] = NULL;
              }
           }
        }
	delnick(ni);
    }
    if (ni = makenick(nick)) {
	ni->flags |= NI_VERBOTEN;
        if (ni->forbid)
             free(ni->forbid);
        ni->forbid = sstrdup(u->nick);
        slog("NS +F %s (%s!%s@%s)",
             nick,u->nick,u->username,u->host);
        do_break_log("NS_F", "NS +F %s (%s!%s@%s)",
             nick,u->nick,u->username,u->host);
	log("%s: %s set FORBID for nick %s", s_NickServ, source, nick);
        wallops(s_NickServ, "%s set \2%s\2 as FORBIDDEN ", source, nick);
	notice(s_NickServ, source, "Nick \2%s\2 is now FORBIDden.", nick);
    } else {
	log("%s: Valid FORBID for %s by %s failed", s_NickServ,
		nick, source);
	notice(s_NickServ, source, "Couldn't FORBID nick \2%s\2!", nick);
    }
}

static void do_unforbid(const char *source, const char *nick)
{
    NickInfo *ni;
    User *u = finduser(source);

    if (!nick) {
        notice(s_NickServ, source, "Syntax: \2FORBID\2 [-]<nick>");
        notice(s_NickServ, source,
            "\2/MSG %s OHELP FORBID\2 for more information", s_NickServ);
	return;
    }
    if (readonly) {
	notice(s_NickServ, source,
	    "Warning: Services is in read-only mode; changes will not be saved.");
    }

   if (ni = findnick(nick)) {
       if (!(ni->flags & NI_VERBOTEN)) {
           slog("NS -F* %s (%s!%s@%s)",
                nick,u->nick,u->username,u->host);
           do_break_log("NS_F", "NS -F* %s (%s!%s@%s)",
                nick,u->nick,u->username,u->host);

           log("%s: %s tried to set unforbidden flag for non forbidden nick %s",
                   s_NickServ, source, nick);
           if (ni->forbid)
               free(ni->forbid);
           ni->forbid = NULL;

           notice(s_NickServ, source, "%s is not set as FORBIDDEN", nick);
           return;
        }

        slog("NS -F %s (%s!%s@%s) [Dropped]",
             nick,u->nick,u->username,u->host);
        do_break_log("NS_F", "NS -F %s (%s!%s@%s) [Dropped]",
             nick,u->nick,u->username,u->host);
        log("%s: %s set FORBID for nick %s", s_NickServ, source, nick);
        delnick(ni);
        wallops(s_NickServ, "%s removed \2%s\2 from the FORBIDDEN list [Automatically Dropped]", source, nick);
        notice(s_NickServ, source, "Nick %s is now UNFORBIDDEN.", nick);
    } else {
        log("%s: Valid UNFORBID for %s by %s failed", s_NickServ,
		nick, source);
        notice(s_NickServ, source, "Couldn't UNFORBID nick \2%s\2!", nick);
    }
}

char *get_os_access(NickInfo *ni)
{
    char *access, buf[128];
    int i=0, end=0;
   

    for (i = 0; i < MAX_SERVROOTS; i++) {
        if (services_roots[i]) {
           if (stricmp(services_roots[i], ni->nick) == 0) {
              end++;
              if (stricmp(services_roots[i], SERVICES_MASTER) == 0)
                   snprintf(buf, sizeof(buf), "Services Root [Master], IRC Operator");
              else
                   snprintf(buf, sizeof(buf), "Services Root, IRC Operator");
           }
        }
    }

    for (i = 0; i < MAX_SERVADMINS; i++) {
        if (services_admins[i]) {
           if (stricmp(services_admins[i], ni->nick) == 0) {
                   if (!end) {
                      snprintf(buf, sizeof(buf), "Services Admin, IRC Operator");
                      end++;
                   }
           }
        }
    }

    for (i = 0; i < MAX_SERVOPERS; i++) {
        if (services_opers[i]) {
           if (stricmp(services_opers[i], ni->nick) == 0) {
                   if (!end) {
                      snprintf(buf, sizeof(buf), "Services Operator, IRC Operator");
                      end++;
                   }
           }
        }
    }

    for (i = 0; i < MAX_HELPOPS; i++) {
        if (services_helpers[i]) {
           if (stricmp(services_helpers[i], ni->nick) == 0) {
                   if (!end) {
                      if (is_oper(ni->nick))
                          snprintf(buf, sizeof(buf), "IRC Operator");
                      else
                          snprintf(buf, sizeof(buf), "Help Operator");
                      end++;
                   }
           }
        }
    }
    if (is_oper(ni->nick) && !end) {
         snprintf(buf, sizeof(buf), "IRC Operator");
         end++;
    }

    if (!end)
       snprintf(buf, sizeof(buf), "User");

    access = buf;
    return access;
}

static void do_flist(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    NickInfo *ni;
    int i;

    if (!cmd) {
        notice(s_NickServ,source,"Syntax: \2FLIST\2 [HELD|MARKED|FORBIDDEN]");
        notice(s_NickServ,source,
            "\2/msg %s OHELP FLIST\2 for more information",s_NickServ);

    } else if (stricmp(cmd, "HELD") == 0) {
        notice(s_NickServ,source,"\2HELD\2 nickname listing:");
        for  (i = 0; i < 256; ++i) {
          for (ni = nicklists[i]; ni; ni = ni->next) {
              if (ni->hold) {
                notice(s_NickServ,source,"%-20s  %s",
                   ni->nick,ni->last_usermask);
                notice(s_NickServ,source,
                     "Set By: %s",ni->hold);
              }
          }
        }
        notice(s_NickServ,source,"\2End of List\2");
    } else if (stricmp(cmd, "MARKED") == 0) {
        notice(s_NickServ,source,"\2MARKED\2 nickname listing:");
        for (i = 0; i < 256; ++i) {
            for (ni = nicklists[i]; ni; ni = ni->next) {
                if (ni->mark) {
                  notice(s_NickServ,source,"%-20s  %s",
                        ni->nick,ni->last_usermask);
                notice(s_NickServ,source,
                     "Set By: %s",ni->mark);
                }
            }
        }
        notice(s_NickServ,source,"\2End of List\2");
    } else if (stricmp(cmd, "FORBIDDEN") == 0) {
        notice(s_NickServ,source,"\2FORBIDDEN\2 nickname listing:");
        for (i = 0; i < 256; ++i) {
            for (ni = nicklists[i]; ni; ni = ni->next) {
                if (ni->forbid) {
                  notice(s_NickServ,source,"%-20s  Set By: %s",
                        ni->nick,ni->forbid);
                }
            }
        }
        notice(s_NickServ,source,"\2End of List\2");
    }
}

/*************************************************************************/

static void do_unhold(const char *source, const char *nick)
{
     User *u = finduser(source);
     NickInfo *ni;

     if (!(ni = findnick(nick))) {
          notice(s_NickServ, source, "Nickname %s is not registered.",nick);
     } else if (ni->flags & NI_HOLD) {
          ni->flags &= ~NI_HOLD;
         if (ni->hold)
             free(ni->hold);
          ni->hold = NULL;
          slog("NS -H %s (%s!%s@%s)",
              nick,u->nick,u->username,u->host);
          do_break_log("NS_F", "NS -H %s (%s!%s@%s)",
               nick,u->nick,u->username,u->host);
          wallops(s_NickServ,
              "%s unset the HELD flag for \2%s\2",u->nick,nick);
     } else {
          notice(s_NickServ,source,"Nickname \2%s\2 is not held.",nick);
     }
}

/*************************************************************************/

static void do_unmark(const char *source, const char *nick)
{
     User *u = finduser(source);
     NickInfo *ni;

     if (!(ni = findnick(nick))) {
          notice(s_NickServ, source, "Nickname %s is not registered.",nick);
     } else if (ni->flags & NI_MARK) {
          ni->flags &= ~NI_MARK;
         if (ni->mark)
             free(ni->mark);
          ni->mark = NULL;
          slog("NS -M %s (%s!%s@%s)",
              nick,u->nick,u->username,u->host);
          do_break_log("NS_F", "NS -M %s (%s!%s@%s)",
               nick,u->nick,u->username,u->host);
            wallops(s_NickServ,
              "%s unset the MARK flag for \2%s\2",u->nick,nick);
     } else {
          notice(s_NickServ,source,"Nickname \2%s\2 is not marked.",nick);
     }

}

/*************************************************************************/

static void do_regemail(const char *source)
{
   NickInfo *ni = findnick(source);
   const char *email = strtok(NULL, " ");

   if (!email)
       notice(s_NickServ, source,
           "Syntax: REGEMAIL <your@email.com>");
   else if (!ni)
       notice(s_NickServ, source,
           "Your nick is not registered.");

   else if (!strchr(email, '@'))
         notice(s_NickServ, source,
             "ERROR: Email must contain an '@'");

   else if ((!strchr(email, '.')) || strchr(email, '\\'))
         notice(s_NickServ, source, "ERROR: Invalid email address");


   else if (ni->regemail && (!(ni->flags & NI_AUTH)))
         notice(s_NickServ, source, "Your nick is already secured");


   else if (!ALLOW_EMAIL)
        notice(s_NickServ, source,
           "Sorry, EMAIL features have been disabled.");

   else {

      struct tm tm;
      char timebuf[64];
      char cmdbuf[500]; char cmdbuf2[500];
      time_t now = time(NULL);
      FILE *f = fopen("memo.txt", "w");
      FILE *fs = fopen(SERVICES_SIG, "r");

               ni->flags |= NI_AUTH;
               ni->regemail = sstrdup(email);
               tm = *localtime(&now);
               strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
               timebuf[sizeof(timebuf)-1] = 0;

               ni->auth = ni->time_registered + getrandom(1,99999) * getrandom(1,9999);

               fprintf(f, "Date: %s\n", timebuf);
               fprintf(f, "From: %s <%s>\n", NETWORK_NAME, RETURN_EMAIL);
               fprintf(f, "To: %s <%s>\n", ni->nick, ni->regemail);
               fprintf(f, "Subject: %s registration AUTH\n\n",
                      NETWORK_NAME);
               fprintf(f, "Your nick's AUTH code is %d. Log back onto %s using the nick %s and type - /MSG %s AUTH %d\n",
                      ni->auth, NETWORK_NAME, ni->nick, s_NickServ, ni->auth);
               fclose(f);
               if (fs) {
                  sprintf(cmdbuf, "cat %s >> memo.txt", SERVICES_SIG);
                  system(cmdbuf);
                  *cmdbuf = 0;
                  fclose(fs);
               }
               sprintf(cmdbuf, "%s -t %s < memo.txt",
                      SENDMAIL_PATH, ni->regemail);
               sprintf(cmdbuf2, "rm -f memo.txt");


               system(cmdbuf);
               system(cmdbuf2);

               notice(s_NickServ, source, NS_REGISTER3);

   }

}
/*************************************************************************/


static void do_auth(const char *source)
{
    NickInfo *ni = findnick(source);
    const char *auth = strtok(NULL, " ");

    if (!auth)
        notice(s_NickServ, source,
            "ERROR: Sytnax: /MSG %s AUTH <number>", s_NickServ);
    else if (!ni)
        notice(s_NickServ, source,
            "ERROR: Your nick is not registered");
    else if (!(ni->flags & NI_AUTH))
         notice(s_NickServ, source,
            "ERROR: your nick is already authorized");
    else {

        if (atoi(auth) == ni->auth) {
             ni->flags &= ~NI_AUTH;
             ni->auth = 0;
             notice(s_NickServ, source,
                   "AUTH code accepted, your nick is now authorized.");
         } else {
             notice(s_NickServ, source,
                   "ERROR: invalid AUTH code");
         }

    }
}



/*************************************************************************/


void flush_regemail(const char *source)
{
   NickInfo *ni;
   const char *nick = strtok(NULL, " ");

   if (!nick)
       notice(s_NickServ, source,
           "Syntax: FLUSH <nick>");
   else if (!(ni = findnick(nick)))
       notice(s_NickServ, source,
           "Nick %s not found in %s databases", nick, s_NickServ);
   else if (!ni->regemail)
       notice(s_NickServ, source,
           "Nick %s doesn't have a registration E-Mail set!", nick);

   else {
       ni->flags |= NI_AUTH;
       ni->regemail = NULL;
       ni->auth = 0;
       notice(s_NickServ, source,
           "Registration E-Mail information for nick %s has been flushed",
                 nick);
   }

}

/*************************************************************************/
/*************************************************************************/


#endif	/* !SKELETON */
