/*
 * atheme-services: A collection of minimalist IRC services
 * users.c: User management.
 *
 * Copyright (c) 2005-2007 Atheme Project (http://www.atheme.org)
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "atheme.h"

static BlockHeap *user_heap;

mowgli_dictionary_t *userlist;
mowgli_dictionary_t *uidlist;

/*
 * init_users()
 *
 * Initializes the users heap and DTree.
 *
 * Inputs:
 *     - none
 *
 * Outputs:
 *     - none
 *
 * Side Effects:
 *     - the users heap and DTree are initialized.
 */
void init_users(void)
{
	user_heap = BlockHeapCreate(sizeof(user_t), HEAP_USER);

	if (user_heap == NULL)
	{
		slog(LG_DEBUG, "init_users(): block allocator failure.");
		exit(EXIT_FAILURE);
	}

	userlist = mowgli_dictionary_create(irccasecmp);
	uidlist = mowgli_dictionary_create(strcmp);
}

/*
 * user_add(const char *nick, const char *user, const char *host, const char *vhost, const char *ip,
 *          const char *uid, const char *gecos, server_t *server, time_t ts);
 *
 * User object factory.
 *
 * Inputs:
 *     - nickname of new user
 *     - username of new user
 *     - hostname of new user
 *     - virtual hostname of new user if applicable otherwise NULL
 *     - ip of user if applicable otherwise NULL
 *     - unique identifier (UID) of user if appliable otherwise NULL
 *     - gecos of new user
 *     - pointer to server new user is on
 *     - user's timestamp
 *
 * Outputs:
 *     - on success, a new user
 *     - on failure, NULL
 *
 * Side Effects:
 *     - if successful, a user is created and added to the users DTree.
 *     - if unsuccessful, a kill has been sent if necessary
 */
user_t *user_add(const char *nick, const char *user, const char *host, 
	const char *vhost, const char *ip, const char *uid, const char *gecos, 
	server_t *server, time_t ts)
{
	user_t *u, *u2;

	slog(LG_DEBUG, "user_add(): %s (%s@%s) -> %s", nick, user, host, server->name);

	u2 = user_find_named(nick);
	if (u2 != NULL)
	{
		if (server == me.me)
		{
			/* caller should not let this happen */
			slog(LG_ERROR, "user_add(): tried to add local nick %s which already exists", nick);
			return NULL;
		}
		slog(LG_INFO, "user_add(): nick collision on %s", nick);
		if (u2->server == me.me)
		{
			if (uid != NULL)
			{
				/* If the new client has a UID, our
				 * client will have a UID too and the
				 * remote server will send us a kill
				 * if it kills our client.  So just kill
				 * their client and continue.
				 */
				kill_id_sts(NULL, uid, "Nick collision with services (new)");
				return NULL;
			}
			if (ts == u2->ts || (ts < u2->ts ^ (!irccasecmp(user, u2->user) && !irccasecmp(host, u2->host))))
			{
				/* If the TSes are equal, or if their TS
				 * is less than our TS and the u@h differs,
				 * or if our TS is less than their TS and
				 * the u@h is equal, our client will be
				 * killed.
				 *
				 * Hope that a kill has arrived just before
				 * for our client; we will have reintroduced
				 * it.
				 */
				return NULL;
			}
			else /* Our client will not be killed. */
				return NULL;
		}
		else
		{
			wallops("Server %s is introducing nick %s which already exists on %s",
					server->name, nick, u2->server->name);
			if (uid != NULL && *u2->uid != '\0')
			{
				kill_id_sts(NULL, uid, "Ghost detected via nick collision (new)");
				kill_id_sts(NULL, u2->uid, "Ghost detected via nick collision (old)");
				user_delete(u2);
			}
			else
			{
				/* There is no way we can do this properly. */
				kill_id_sts(NULL, nick, "Ghost detected via nick collision");
				user_delete(u2);
			}
			return NULL;
		}
	}

	u = BlockHeapAlloc(user_heap);

	if (uid != NULL)
	{
		strlcpy(u->uid, uid, IDLEN);
		mowgli_dictionary_add(uidlist, u->uid, u);
	}

	strlcpy(u->nick, nick, NICKLEN);
	strlcpy(u->user, user, USERLEN);
	strlcpy(u->host, host, HOSTLEN);
	strlcpy(u->gecos, gecos, GECOSLEN);

	if (vhost)
		strlcpy(u->vhost, vhost, HOSTLEN);
	else
		strlcpy(u->vhost, host, HOSTLEN);

	if (ip && strcmp(ip, "0") && strcmp(ip, "0.0.0.0") && strcmp(ip, "255.255.255.255"))
		strlcpy(u->ip, ip, HOSTIPLEN);

	u->server = server;
	u->server->users++;
	node_add(u, node_create(), &u->server->userlist);

	if (ts)
		u->ts = ts;
	else
		u->ts = CURRTIME;

	mowgli_dictionary_add(userlist, u->nick, u);

	cnt.user++;

	hook_call_event("user_add", u);

	return u;
}

/*
 * user_delete(user_t *u)
 *
 * Destroys a user object and deletes the object from the users DTree.
 *
 * Inputs:
 *     - user object to delete
 *
 * Outputs:
 *     - nothing
 *
 * Side Effects:
 *     - on success, a user is deleted from the users DTree.
 */
void user_delete(user_t *u)
{
	node_t *n, *tn;
	chanuser_t *cu;
	mynick_t *mn;

	if (!u)
	{
		slog(LG_DEBUG, "user_delete(): called for NULL user");
		return;
	}

	slog(LG_DEBUG, "user_delete(): removing user: %s -> %s", u->nick, u->server->name);

	hook_call_event("user_delete", u);

	u->server->users--;
	if (is_ircop(u))
		u->server->opers--;
	if (u->flags & UF_INVIS)
		u->server->invis--;

	/* remove the user from each channel */
	LIST_FOREACH_SAFE(n, tn, u->channels.head)
	{
		cu = (chanuser_t *)n->data;

		chanuser_delete(cu->chan, u);
	}

	mowgli_dictionary_delete(userlist, u->nick);

	if (*u->uid)
		mowgli_dictionary_delete(uidlist, u->uid);

	n = node_find(u, &u->server->userlist);
	node_del(n, &u->server->userlist);
	node_free(n);

	if (u->myuser)
	{
		LIST_FOREACH_SAFE(n, tn, u->myuser->logins.head)
		{
			if (n->data == u)
			{
				node_del(n, &u->myuser->logins);
				node_free(n);
				break;
			}
		}
		u->myuser->lastlogin = CURRTIME;
		if ((mn = mynick_find(u->nick)) != NULL &&
				mn->owner == u->myuser)
			mn->lastseen = CURRTIME;
		u->myuser = NULL;
	}

	BlockHeapFree(user_heap, u);

	cnt.user--;
}

/*
 * user_find(const char *nick)
 *
 * Finds a user by UID or nickname.
 *
 * Inputs:
 *     - nickname or UID to look up
 *
 * Outputs:
 *     - on success, the user object that was requested
 *     - on failure, NULL
 *
 * Side Effects:
 *     - none
 */
user_t *user_find(const char *nick)
{
	user_t *u;

	if (nick == NULL)
		return NULL;

	if (ircd->uses_uid == TRUE)
	{
		u = mowgli_dictionary_retrieve(uidlist, nick);

		if (u != NULL)
			return u;
	}

	u = mowgli_dictionary_retrieve(userlist, nick);

	if (u != NULL)
	{
		if (ircd->uses_p10)
			wallops(_("user_find(): found user %s by nick!"), nick);
		return u;
	}

	return NULL;
}

/*
 * user_find_named(const char *nick)
 *
 * Finds a user by nickname. Prevents chasing users by their UID.
 *
 * Inputs:
 *     - nickname to look up
 *
 * Outputs:
 *     - on success, the user object that was requested
 *     - on failure, NULL
 *
 * Side Effects:
 *     - none
 */
user_t *user_find_named(const char *nick)
{
	return mowgli_dictionary_retrieve(userlist, nick);
}

/*
 * user_changeuid(user_t *u, const char *uid)
 *
 * Changes a user object's UID.
 *
 * Inputs:
 *     - user object to change UID
 *     - new UID
 *
 * Outputs:
 *     - nothing
 *
 * Side Effects:
 *     - a user object's UID is changed.
 */
void user_changeuid(user_t *u, const char *uid)
{
	if (*u->uid)
		mowgli_dictionary_delete(uidlist, u->uid);

	strlcpy(u->uid, uid ? uid : "", IDLEN);

	if (*u->uid)
		mowgli_dictionary_add(uidlist, u->uid, u);
}

/*
 * user_changenick(user_t *u, const char *uid)
 *
 * Changes a user object's nick and TS.
 *
 * Inputs:
 *     - user object to change
 *     - new nick
 *     - new TS
 *
 * Outputs:
 *     - whether the user was killed as result of the nick change
 *
 * Side Effects:
 *     - a user object's nick and TS is changed.
 *     - in event of a collision, the user may be killed
 */
boolean_t user_changenick(user_t *u, const char *nick, time_t ts)
{
	mynick_t *mn;
	user_t *u2;

	u2 = user_find_named(nick);
	if (u2 != NULL)
	{
		if (u->server == me.me)
		{
			/* caller should not let this happen */
			slog(LG_ERROR, "user_changenick(): tried to change local nick %s to %s which already exists", u->nick, nick);
			return FALSE;
		}
		slog(LG_INFO, "user_changenick(): nick collision on %s", nick);
		if (u2->server == me.me)
		{
			if (*u->uid != '\0')
			{
				/* If the changing client has a UID, our
				 * client will have a UID too and the
				 * remote server will send us a kill
				 * if it kills our client.  So just kill
				 * their client and continue.
				 */
				kill_id_sts(NULL, u->uid, "Nick change collision with services");
				user_delete(u);
				return TRUE;
			}
			if (ts == u2->ts || (ts < u2->ts ^ (!irccasecmp(u->user, u2->user) && !irccasecmp(u->host, u2->host))))
			{
				/* If the TSes are equal, or if their TS
				 * is less than our TS and the u@h differs,
				 * or if our TS is less than their TS and
				 * the u@h is equal, our client will be
				 * killed.
				 *
				 * Hope that a kill has arrived just before
				 * for our client; we will have reintroduced
				 * it.
				 * But kill the changing client using its
				 * old nick.
				 */
				kill_id_sts(NULL, u->nick, "Nick change collision with services");
				user_delete(u);
				return TRUE;
			}
			else
			{
				/* Our client will not be killed.
				 * But kill the changing client using its
				 * old nick.
				 */
				kill_id_sts(NULL, u->nick, "Nick change collision with services");
				user_delete(u);
				return TRUE;
			}
		}
		else
		{
			wallops("Server %s is sending nick change from %s to %s which already exists on %s",
					u->server->name, u->nick, nick,
					u2->server->name);
			if (*u->uid != '\0' && *u2->uid != '\0')
			{
				kill_id_sts(NULL, u->uid, "Ghost detected via nick change collision (new)");
				kill_id_sts(NULL, u2->uid, "Ghost detected via nick change collision (old)");
				user_delete(u);
				user_delete(u2);
			}
			else
			{
				/* There is no way we can do this properly. */
				kill_id_sts(NULL, u->nick, "Ghost detected via nick change collision");
				kill_id_sts(NULL, nick, "Ghost detected via nick change collision");
				user_delete(u);
				user_delete(u2);
			}
			return TRUE;
		}
	}
	if (u->myuser != NULL && (mn = mynick_find(u->nick)) != NULL &&
			mn->owner == u->myuser)
		mn->lastseen = CURRTIME;
	mowgli_dictionary_delete(userlist, u->nick);

	strlcpy(u->nick, nick, NICKLEN);
	u->ts = ts;

	mowgli_dictionary_add(userlist, u->nick, u);
	return FALSE;
}

/*
 * user_mode(user_t *user, const char *modes)
 *
 * Changes a user object's modes.
 *
 * Inputs:
 *     - user object to change modes on
 *     - modestring describing the usermode change
 *
 * Outputs:
 *     - nothing
 *
 * Side Effects:
 *     - on success, a user's modes are adjusted.
 *
 * Bugs:
 *     - this routine only tracks +i and +o usermode changes.
 */
void user_mode(user_t *user, const char *modes)
{
	int dir = MTYPE_ADD;

	if (!user)
	{
		slog(LG_DEBUG, "user_mode(): called for nonexistant user");
		return;
	}

	while (*modes != '\0')
	{
		switch (*modes)
		{
		  case '+':
			  dir = MTYPE_ADD;
			  break;
		  case '-':
			  dir = MTYPE_DEL;
			  break;
		  case 'i':
			  if (dir == MTYPE_ADD)
			  {
				  if (!(user->flags & UF_INVIS))
					  user->server->invis++;
				  user->flags |= UF_INVIS;
			  }
			  else if ((dir = MTYPE_DEL))
			  {
				  if (user->flags & UF_INVIS)
					  user->server->invis--;
				  user->flags &= ~UF_INVIS;
			  }
			  break;
		  case 'o':
			  if (dir == MTYPE_ADD)
			  {
				  if (!is_ircop(user))
				  {
					  user->flags |= UF_IRCOP;
					  slog(LG_DEBUG, "user_mode(): %s is now an IRCop", user->nick);
					  snoop("OPER: %s (%s)", user->nick, user->server->name);
					  user->server->opers++;
					  hook_call_event("user_oper", user);
				  }
			  }
			  else if ((dir = MTYPE_DEL))
			  {
				  if (is_ircop(user))
				  {
					  user->flags &= ~UF_IRCOP;
					  slog(LG_DEBUG, "user_mode(): %s is no longer an IRCop", user->nick);
					  snoop("DEOPER: %s (%s)", user->nick, user->server->name);
					  user->server->opers--;
					  hook_call_event("user_deoper", user);
				  }
			  }
		  default:
			  break;
		}
		modes++;
	}
}

/* vim:cinoptions=>s,e0,n0,f0,{0,}0,^0,=s,ps,t0,c3,+s,(2s,us,)20,*30,gs,hs
 * vim:ts=8
 * vim:sw=8
 * vim:noexpandtab
 */
